/** @file cyber.h
 *  Main CyOS header file.
 */

#ifndef __CYBER_H_INCLUDE__
#define __CYBER_H_INCLUDE__

/**  Seek to the absolute position.
 *   @ingroup filesystem_overview
*/
#define  SEEK_SET    0

/**  Seek from the current position (a positive offset indicates towards end of file).
 *   @ingroup filesystem_overview
*/
#define  SEEK_CUR    1

/**  Seek from the end. A negative offset indicates closer to the beginning of the file; zero offset means set after the last byte of the file.
 *   @ingroup filesystem_overview
*/
#define  SEEK_END    2

/** Stream's state flags. Stream state is represented by setting and clearing following flags:
 *  @ingroup filesystem_overview
 */
enum Flags
{
   /**
    *  is set if the stream object is invalid
    */
  FLAG_BAD = 1,        
   /**
    *  is set when the stream reaches the end of the file
    */
  FLAG_EOF = 2,       
   /**
    *  indicates that implementation must throw exceptions on errors
    */
  FLAG_EXCEPTIONS = 4   
};

//============================
//=                          =
//=    DLink destructor.     =
//=    Helper functions.     =
//=                          =
//============================
#ifndef _SKIP_DOC
__cyos 125
bool Dispatcher_in_critical;

struct DLink
{
  struct DLink* next;
  struct DLink* prev;
};

struct NameBlock : public DLink
{
  char* name;
  int kind;
  int PADDING;
  void* vtable;
};

#define DLink_dtor(ptr_dlink, memory_flag) {      \
    bool old_state = Dispatcher_in_critical;      \
    Dispatcher_in_critical = TRUE;                \
    (ptr_dlink)->prev->next = (ptr_dlink)->next;  \
    (ptr_dlink)->next->prev = (ptr_dlink)->prev;  \
    Dispatcher_in_critical = old_state;           \
    if((memory_flag) == FREE_MEMORY) free(ptr_dlink);} 

#endif

//============================
//=                          =
//=     object Buffer.       =
//=  Dynamic memory buffer.  =
//=                          =
//============================

/** @brief a dynamic memory buffer
 *  @ingroup messaging_group
 *  This object provides a movable memory buffer that can be dynamically
 *  created, resized, and used as a plain memory block or as storage with access
 *  helpers.
 *  <br>
 *  <br>
 *  For CyOS, buffers are better than plain dynamic memory - allocated chunks because
 *  they can be locked and unlocked. While they are unlocked, the OS may move them in order to
 *  organize memory more efficiently, and may even drop the buffer to the flash so it may be used as a kind of
 *  virtual memory.
 *  <br>
 *  <br>
 *  The best way to work with this object is to keep the buffer unlocked and use access helpers methods
 *  Buffer_load, Buffer_load_string, Buffer_store, Buffer_store_string,
 *  Buffer_get_byte, Buffer_set_byte, Buffer_get_char, Buffer_set_char,
 *  Buffer_get_int, Buffer_set_int, Buffer_get_long, Buffer_set_long and
 *  so on.
 *  <br>
 *  <br>
 *  The difference between store_ and set_ methods is that store_* automatically
 *  resizes the buffer to fit the size required, while set_ methods check and treat
 *  " out of bounds " as an error.
 *  <br>
 *  <br>
 *  Please note that the offsets for all helpers get_* and set_* are ALWAYS IN BYTES.
 *  The Buffers have a size/increment/capacity space strategy. This means that when
 *  a Buffer needs to expand its storage, it uses an increment and stores both
 *  the actual data size and the current capacity to minimize the load on the dynamic
 *  memory subsystem.
 *  <br>
 *  <br>
 *  Use Buffer_set_size, Buffer_ensure_size and Buffer_compact methods
 *  to manage the buffer's capacity.
 *  @see messaging_group
 */
struct Buffer
{
#ifndef _SKIP_DOC
  char Buffer_Members_STUB[40];
};
#endif

/** @fn struct Buffer* Buffer_ctor(struct Buffer* ptr_buffer, size_t size, size_t increment)
 *   Create an empty buffer of a specified data_size.
 *  @param ptr_buffer A pointer to the Buffer structure
 *  @param size The initial buffer size
 *  @param increment An increment value
 *  @return A pointer to the initialized Buffer object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       int index;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       for( index = 0; strlen( sz_string_array[index] ); index++ )
 *       {
 *         Buffer_store_string( &buffer, sz_string_array[index],
 *                              Buffer_get_size( &buffer ) );
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_ctor_Ex.
 */
#ifndef _SKIP_DOC
__cyos 222
#endif
struct Buffer* Buffer_ctor(struct Buffer* ptr_buffer, size_t size, size_t increment);

/** @fn struct Buffer* Buffer_ctor_Ex(struct Buffer* ptr_buffer, struct Buffer* templ)
 *   Creates a copy of the existing buffer.
 *  @param ptr_buffer A pointer to the Buffer structure
 *  @param templ A pointer to the template Buffer object
 *  @return A pointer to the initialized Buffer object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       struct Buffer buffer_tmpl;
 *       int index;
 *       ...
 *       Buffer_ctor( &buffer_tmpl, 1024, 64 );
 *       ...
 *       for( index = 0; strlen( sz_string_array[index] ); index++ )
 *       {
 *         Buffer_store_string( &buffer_templ, sz_string_array[index],
                                Buffer_get_size( &buffer_templ ) );
 *       }
 *       ...
 *       Buffer_ctor_Ex( &buffer, &buffer_templ );
 *       ...
 *       Buffer_dtor( &buffer_templ, LEAVE_MEMORY );
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_ctor.
 */
#ifndef _SKIP_DOC
__cyos 221
#endif
struct Buffer* Buffer_ctor_Ex(struct Buffer* ptr_buffer, struct Buffer* templ);

/** @fn void Buffer_store(struct Buffer* ptr_buffer, void* ptr_data , size_t offset, size_t length)
 *  Stores some data at the specified offset. If needed, the buffer will expand it's storage to fit the specified 'length' of the data.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param ptr_data A pointer to the data to store
 *  @param offset The offset in the buffer
 *  @param length The data's length; if it sets to 0 then the current size of the buffer will be used
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       int index;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       for( index = 0; strlen( sz_string_array[index]); index++ )
 *       {
 *         Buffer_store( &buffer,
 *                       sz_string_array[index],
 *                       Buffer_get_size( &buffer ),
 *                       strlen( sz_string_array[index] ) + 1 );
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1196
#endif
void Buffer_store(struct Buffer* ptr_buffer, void* ptr_data , size_t offset, size_t length);

/** @fn void Buffer_store_string(struct Buffer* ptr_buffer, char* str, size_t offset)
 *  Stores a null-terminated string into the buffer. If needed, buffer will extend it's storage to fit the data's specified length.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param str The string to store
 *  @param offset The offset in the buffer
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       int index;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       for( index = 0; strlen( sz_string_array[index] ); index++ )
 *       {
 *         Buffer_store_string( &buffer, sz_string_array[index],
 *                              Buffer_get_size( &buffer ) );
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1201
#endif
void Buffer_store_string(struct Buffer* ptr_buffer, char* str, size_t offset);

/** @fn void* Buffer_load(struct Buffer* ptr_buffer, void* ptr_data, size_t offset, size_t length)
 *  Loads 'length' bytes from the Buffer, starting at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param ptr_data Where to store the loaded bytes
 *  @param offset The point at which to start copying bytes
 *  @param length How many bytes to copy; if 0 then all data size is supposed
 *  @return A pointer for convenience and style reasons
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char* some_data[120];
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       //  Stores data;
 *       Buffer_store( &buffer, &some_data, 0, 120 );
 *       ...
 *       //  Loads data;
 *       Buffer_load( &buffer, &some_data, 0, 120 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_load_string.
 */
#ifndef _SKIP_DOC
__cyos 851
#endif
void* Buffer_load(struct Buffer* ptr_buffer, void* ptr_data, size_t offset, size_t length);

/** @fn char* Buffer_load_string(struct Buffer* ptr_buffer, char* str, size_t offset, size_t length)
 *  Loads null terminated string.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param str Where to store the string
 *  @param offset The offset in the buffer
 *  @param length The maximum length of the string; if 0 then all data size is supposed
 *  @return A pointer to the string
 *  @code
 *       #include <cywin.h>
 *       #define MAX_STRING_LENGTH 64
 *       ...
 *       struct Buffer buffer;
 *       int string_number;
 *       int index;
 *       int offset = 0;
 *       char sz_text_buffer[MAX_STRING_LENGTH];
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       //  Stores strings.
 *       for( string_number = 0; strlen( sz_string_array[string_number] ); string_number++ )
 *       {
 *         Buffer_store_string( &buffer, sz_string_array[string_number],
 *                              Buffer_get_size( &buffer ) );
 *       }
 *       ...
 *       //  Trace strings.
 *       for( index = 0; index < string_number; index++ )
 *       {
 *         Buffer_load_string( &buffer, sz_text_buffer, offset, 0 );
 *         offset += strlen() + 1;
 *         TRACE( "%i string is %s", index, sz_text_buffer );
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_load.
 */
#ifndef _SKIP_DOC
__cyos 862
#endif
char* Buffer_load_string(struct Buffer* ptr_buffer, char* str, size_t offset, size_t length);

/** @fn size_t Buffer_get_size(struct Buffer* ptr_buffer)
 *  Returns the size of the data in the buffer.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @return The size of the data in the buffer
 *  @code
 *       #include <cywin.h>
 *       #define MAX_STRING_LENGTH 64
 *       ...
 *       struct Buffer buffer;
 *       int string_number;
 *       int index;
 *       int offset = 0;
 *       char sz_text_buffer[MAX_STRING_LENGTH];
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       //  Stores strings.
 *       for( string_number = 0; strlen( sz_string_array[string_number] ); string_number++ )
 *       {
 *         Buffer_store_string( &buffer, sz_string_array[string_number],
                                Buffer_get_size( &buffer ) );
 *       }
 *       ...
 *       //  Trace strings.
 *       for( index = 0; index < string_number; index++ )
 *       {
 *         Buffer_load_string( &buffer, sz_text_buffer, offset, 0 );
 *         offset += strlen() + 1;
 *         TRACE( "%i string is %s", index, sz_text_buffer );
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_get_allocated_size, Buffer_set_size.
 */
#ifndef _SKIP_DOC
__cyos 728
#endif
size_t Buffer_get_size(struct Buffer* ptr_buffer);

#ifndef _SKIP_DOC
__cyos 1165
size_t Buffer_size(struct Buffer* ptr_buffer);
#endif

/** @fn size_t Buffer_get_allocated_size(struct Buffer* ptr_buffer)
 *  Returns the size of the allocated space (current capacity that may be bigger than Buffer_get_size()).
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @return Size of the allocated space
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char* some_data[120];
 *       int real_size;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       //  Stores data;
 *       Buffer_store( &buffer, &some_data, 0, 120 );
 *       ...
 *       // The real_size value will be equal to 1024.
 *       real_size = Buffer_get_allocated_size( &buffer );
 *       ...
 *       //  Loads data;
 *       Buffer_load( &buffer, &some_data, 0, 120 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_get_size.
 */
#ifndef _SKIP_DOC
__cyos 628
#endif
size_t Buffer_get_allocated_size(struct Buffer* ptr_buffer);

/** @fn void Buffer_compact(struct Buffer* ptr_buffer)
 *  Resizes storage to the current data size.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char* some_data[120];
 *       int real_size;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       //  Stores data;
 *       Buffer_store( &buffer, &some_data, 0, 120 );
 *       ...
 *       Buffer_compact( &buffer );
 *       // The real_size value will be equal to 120.
 *       // Without calling the Buffer_compact function
 *       // the real_size value will be equal to 1024.
 *       real_size = Buffer_get_allocated_size( &buffer );
 *       ...
 *       //  Loads data;
 *       Buffer_load( &buffer, &some_data, 0, 120 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC  
__cyos 467
#endif
void Buffer_compact(struct Buffer* ptr_buffer);

/** @fn bool Buffer_set_size(struct Buffer* ptr_buffer, size_t size)
 *  Sets the size. If the size requested is less than the current capacity, extra space might be freed for the OS.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param size The new buffer size
 *  @return TRUE if the buffer size is changed successfully
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char* some_data[120];
 *       int real_size;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       //  Stores data;
 *       Buffer_store( &buffer, &some_data, 0, 120 );
 *       ...
 *       Buffer_set_size( &buffer, 120 );
 *       // The real_size value will be equal to 120.
 *       real_size = Buffer_get_allocated_size( &buffer );
 *       ...
 *       //  Loads data;
 *       Buffer_load( &buffer, &some_data, 0, 120 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_get_size.
 */
#ifndef _SKIP_DOC
__cyos 1154
#endif
bool Buffer_set_size(struct Buffer* ptr_buffer, size_t size);

/** @fn void Buffer_free(struct Buffer* ptr_buffer)
 *  Clears the buffer and frees its storage.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Frees the buffer and allocates a new buffer size.
 *       Buffer_free( &buffer );
 *       Buffer_set_size( &buffer, 120 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 612
#endif
void Buffer_free(struct Buffer* ptr_buffer);

/** @fn bool Buffer_ensure_size(struct Buffer* ptr_buffer, size_t test_size)
 *  Reallocates and copies the buffer's contents, if needed. Make sure the buffer's storage is not less than test_size
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param test_size The size of the Buffer to test
 *  @return TRUE if the buffer has suffiicient capacity, FALSE if enough space can not be allocated to fit the specified size
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Frees the buffer and allocates a new buffer size.
 *       Buffer_free( &buffer );
 *       if ( Buffer_ensure_size( &buffer, 10000 ) )
 *       {
 *          Buffer_set_size( &buffer, 10000 );
 *          ...
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 548
#endif
bool Buffer_ensure_size(struct Buffer* ptr_buffer, size_t test_size);

/** @fn char Buffer_get_byte(struct Buffer* ptr_buffer, size_t offset);
 *  Reads byte at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param offset The offset in the buffer
 *  @return Byte was read
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char   read_byte;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Writes the byte 'y ' at the 12th position in the buffer.
 *       Buffer_set_byte( &buffer, 12, 'y ' );
 *       ...
 *       // Reads a byte at the 12th position in the buffer.
 *       read_byte = Buffer_get_byte( &buffer, 12 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_set_byte.
 */
#ifndef _SKIP_DOC
__cyos 642
char Buffer_get_byte(struct Buffer* ptr_buffer, size_t offset);
#endif

/** @fn void Buffer_set_byte(struct Buffer* ptr_buffer, size_t offset, char data);
 *  Reads byte at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param offset The offset in the buffer
 *  @param data The character to set
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char   read_byte;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Writes the byte 'y ' at the 12th position in the buffer.
 *       Buffer_set_byte( &buffer, 12, 'y ' );
 *       ...
 *       // Reads a byte at the 12th position in the buffer.
 *       read_byte = Buffer_get_byte( &buffer, 12 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_set_byte.
 */
#ifndef _SKIP_DOC
__cyos 1099
void Buffer_set_byte(struct Buffer* ptr_buffer, size_t offset, char data);
#endif

/** @fn char Buffer_get_char(struct Buffer* ptr_buffer, size_t offset)
 *  Reads char at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param offset The offset in the buffer
 *  @return Char was read
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char   read_byte;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Writes the byte 'y ' at the 12th position in the buffer.
 *       Buffer_set_byte( &buffer, 12, 'y ' );
 *       ...
 *       // Reads a byte at the 12th position in the buffer.
 *       read_byte = Buffer_get_byte( &buffer, 12 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_set_char.
 */
#ifndef _SKIP_DOC
__cyos 648
#endif
char Buffer_get_char(struct Buffer* ptr_buffer, size_t offset);

/** @fn void Buffer_set_char(struct Buffer* ptr_buffer, size_t offset, char data)
 *  Writes char value 'data' at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param offset The offset in the buffer
 *  @param data The character to set
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char   read_byte;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Writes the byte 'y' at the 12th position in the buffer.
 *       Buffer_set_byte( &buffer, 12, 'y' );
 *       ...
 *       // Reads a byte at the 12th position in the buffer.
 *       read_byte = Buffer_get_byte( &buffer, 12 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_get_char.
 */
#ifndef _SKIP_DOC
__cyos 1100
#endif
void Buffer_set_char(struct Buffer* ptr_buffer, size_t offset, char data);

/** @fn int Buffer_get_int(struct Buffer* ptr_buffer, size_t offset)
 *  Reads the int value at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param offset The offset in the buffer
 *  @return Int value was read
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       int    read_int;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Writes a value at the 12th position in the buffer.
 *       Buffer_set_int( &buffer, 12, 120 );
 *       ...
 *       // Reads a value at the 12th position in the buffer.
 *       read_int = Buffer_get_int( &buffer, 12 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_set_int.
 */
#ifndef _SKIP_DOC
__cyos 682
#endif
int Buffer_get_int(struct Buffer* ptr_buffer, size_t offset);

/** @fn void Buffer_set_int(struct Buffer* ptr_buffer, size_t offset, int data)
 *  Writes the int 'data' at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param offset The offset in the buffer
 *  @param data The int to set
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       int    read_int;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Writes a value at the 12th position in the buffer.
 *       Buffer_set_int( &buffer, 12, 120 );
 *       ...
 *       // Reads a value at the 12th position in the buffer.
 *       read_int = Buffer_get_int( &buffer, 12 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_get_int.
 */
#ifndef _SKIP_DOC
__cyos 1129
#endif
void Buffer_set_int(struct Buffer* ptr_buffer, size_t offset, int data);

/** @fn long Buffer_get_long(struct Buffer* ptr_buffer, size_t offset)
 *  Reads the long value at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param offset The offset in the buffer
 *  @return Long value was read
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       long   read_long;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Writes a value at the 12th position in the buffer.
 *       Buffer_set_long( &buffer, 12, 0x7FFFFFFF );
 *       ...
 *       // Reads a value at the 12th position in the buffer.
 *       read_long = Buffer_get_long( &buffer, 12 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_set_long.
 */
#ifndef _SKIP_DOC
__cyos 685
#endif
long Buffer_get_long(struct Buffer* ptr_buffer, size_t offset);

/** @fn void Buffer_set_long(struct Buffer* ptr_buffer, size_t offset, long data)
 *  Writes the long value 'data' at the specified 'offset'.
 *  @param ptr_buffer A pointer to the initialized Buffer object.
 *  @param offset The offset in the buffer
 *  @param data The long to set
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       long   read_long;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       // Writes a value at the 12th position in the buffer.
 *       Buffer_set_long( &buffer, 12, 0x7FFFFFFF );
 *       ...
 *       // Reads a value at the 12th position in the buffer.
 *       read_long = Buffer_get_long( &buffer, 12 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_get_long.
 */
#ifndef _SKIP_DOC
__cyos 1133
#endif
void Buffer_set_long(struct Buffer* ptr_buffer, size_t offset, long data);

/** @fn bool Buffer_is_locked(struct Buffer* ptr_buffer)
 *  Returns lock state (locked or unlocked).
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @return TRUE, if the buffer is locked, FALSE otherwise
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char* ptr_pure_data;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       ptr_pure_data = Buffer_lock( &buffer );
 *       ptr_pure_data[12] = 'y ';
 *       ...
 *       if( Buffer_is_locked( &buffer ) )
 *       {
 *         Buffer_unlock( &buffer );
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_lock.
 */
#ifndef _SKIP_DOC
__cyos 815
#endif
bool Buffer_is_locked(struct Buffer* ptr_buffer);

/** @fn void* Buffer_lock(struct Buffer* ptr_buffer)
 *  Locks storage and returns the pointer to it's area.
 *  Note that a locked buffer can not be trashed. Be very careful with
 *  this pointer and do not access more data than the number of Buffer_get_size() bytes. Do not use this
 *  pointer after Buffer_unlock() was called.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @return Address of the locked buffer's data area
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char* ptr_pure_data;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       ptr_pure_data = Buffer_lock( &buffer );
 *       ptr_pure_data[12] = 'y';
 *       ...
 *       if( Buffer_is_locked( &buffer ) )
 *       {
 *         Buffer_unlock( &buffer );
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_unlock, Buffer_is_locked.
 */
#ifndef _SKIP_DOC
__cyos 866
#endif
void* Buffer_lock(struct Buffer* ptr_buffer);

/** @fn void Buffer_unlock(struct Buffer* ptr_buffer)
 *  Unlocks the buffer's storage.
 *   Note that all references to the buffer area returned by Buffer_lock() become
 *   invalid and very dangerous to use; you can destroy the system very easily
 *   by using this pointer after Buffer_unlock().
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char* ptr_pure_data;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       ptr_pure_data = Buffer_lock( &buffer );
 *       ptr_pure_data[12] = 'y';
 *       ...
 *       if( Buffer_is_locked( &buffer ) )
 *       {
 *         Buffer_unlock( &buffer );
 *       }
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Buffer_lock.
 */
#ifndef _SKIP_DOC
__cyos 1262
#endif
void Buffer_unlock(struct Buffer* ptr_buffer);

/** @fn void Buffer_dtor(struct Buffer* ptr_buffer, int memory_flag)
 *  Unlocks and frees a previously locked buffer.
 *  @param ptr_buffer A pointer to the initialized Buffer object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in stack 
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Buffer buffer;
 *       char* some_data[120];
 *       int real_size;
 *       ...
 *       Buffer_ctor( &buffer, 1024, 64 );
 *       ...
 *       //  Stores data;
 *       Buffer_store( &buffer, &some_data, 0, 120 );
 *       ...
 *       Buffer_compact( &buffer );
 *       // The real_size value will be equal to 120.
 *       // Without calling the Buffer_compact function
 *       // the real_size value will be equal to 1024.
 *       real_size = Buffer_get_allocated_size( &buffer );
 *       ...
 *       //  Loads data;
 *       Buffer_load( &buffer, &some_data, 0, 120 );
 *       ...
 *       Buffer_dtor( &buffer, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos 107
#endif
void Buffer_dtor(struct Buffer* ptr_buffer, int);

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=     object Input.        =
//=  Abstract input stream.  =
//=                          =
//============================

/**
 *  @brief an abstract object for sequential data access objects.
 *  @ingroup filesystem_overview 
 *  The CyOS stream modes have 2 basic streams that are abstractions for the sequential data access objects:
 *  Input and Output. Those streams can pump data to and from virtually any source, including disk files,
 *  network connections, console terminals and more. Those basic streams are used, in turn,
 *  with filter streams that apply specific features to some abstract streams' filters. 
 *  <br>
 *  <br>
 *  Typical stream implementations include the file streams FileInput and FileOutput, console streams,
 *  and various filters. Filter streams process the data from other streams.
 *  <br>
 *  <br>
 *  They take the other streams' objects while constructing, and are just like regular streams themselves.
 *  Typical filters are UNICODE/ASCII conversion, text formatting streams Reader and Writer,
 *  and compression streams Compressor and Decompressor.
 *  <br>
 *  <br>
 *  Filters are intended to combine the desired stream implementation with the selected filter object,
 *  to combine the features that best suit your needs. This object is just an interface for an input stream.
 *  @see filesystem_overview
 */
struct Input
{
#ifndef _SKIP_DOC
  char Input_Members_STUB[20]; /* CANNOT be instantiated! */
};
#endif

#ifndef DISABLE_STANDARD_RTL

// for the "outer world", most of the following functions look just like
// those of CyOS, although they're actually implemented by the rtl; note
// that there may exist custom rtls with different import sets;
// application using such rtl must #define DISABLE_STANDARD_RTL right
// before #include <cybiko.h>

/** @fn long Input_seekg(struct Input* ptr_input, long pos, seek_t mode)
 *  Seeks an input stream. If the stream supports a seek operation,
 *  it seeks to the specified position in the specified mode.
 *  If the requested seek operation cannot be done (not supported
 *  or wrong parameters), -1 will be returned. Seeking prior to the beginning
 *  of the stream sets the pointer to the stream's first byte.
 *  Seeking after the end of the stream sets the pointer to the end of the
 *  stream (after the stream's last byte).
 *  @param ptr_input A pointer to the initialized Input object
 *  @param pos The seek offset
 *  @param mode The seek mode
 *  @return The new position or -1 if an error occurred
 *  @code
 *       #include <cywin.h>
 *       ...
 *       char tenth_byte;
 *       struct Input* ptr_input;
 *       ...
 *       ptr_input = open_resource_Ex( "data.txt" );
 *       ...
 *       Input_seekg( ptr_input, 10, SEEK_SET );
 *       tenth_byte = Input_read_byte( ptr_input );
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
import 3
#endif
long Input_seekg(struct Input* ptr_input, long pos, seek_t mode);

/** @fn long Input_tellg(struct Input* ptr_input)
 *  Returns the stream's position.
 *  @param ptr_input A pointer to the initialized Input object
 *  @return The stream's position, or -1 of that operation is not supported for this stream
 *  @code
 *       #include <cywin.h>
 *       ...
 *       char tenth_byte;
 *       long current_pos;
 *       struct Input* ptr_input;
 *       ...
 *       ptr_input = open_resource_Ex( "data.txt" );
 *       ...
 *       Input_seekg( ptr_input, 10, SEEK_SET );
 *       tenth_byte = Input_read_byte( ptr_input );
 *       // The current_pos value will be equal to 11.
 *       current_pos = Input_tellg( ptr_input );
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
import 4
#endif
long Input_tellg(struct Input* ptr_input);

/** @fn long Input_read(struct Input* ptr_input, void* buffer, long length)
 *  Reads the 'length' bytes from the stream.
 *  @param ptr_input A pointer to the initialized Input object
 *  @param buffer A pointer to the allocated buffer
 *  @param length The number of bytes to read
 *  @return The exact number of bytes read, or 0 if no bytes were read
 *  @code
 *       #include <cywin.h>
 *       #include "score.h"
 *       ...
 *       struct Input* ptr_input;
 *       struct score_t high_scores[10];
 *       ...
 *       ptr_input = open_resource_Ex( "score.inf" );
 *       ...
 *       Input_read( ptr_input, high_scores, sizeof(high_scores) );
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see Input_read_byte.
 */
#ifndef _SKIP_DOC
import 22
#endif
long Input_read(struct Input* ptr_input, void* buffer, long length);

/** @fn int Input_read_byte(struct Input* ptr_input)
 *  Reads the next byte from the stream.
 *  @param ptr_input A pointer to the initialized Input object
 *  @return The next byte from the stream
 *  @code
 *       #include <cywin.h>
 *       ...
 *       char tenth_byte;
 *       struct Input* ptr_input;
 *       ...
 *       ptr_input = open_resource_Ex( "data.txt" );
 *       ...
 *       Input_seekg( ptr_input, 10, SEEK_SET );
 *       tenth_byte = Input_read_byte( ptr_input );
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see Input_read.
 */
#ifndef _SKIP_DOC
import 5
#endif
int Input_read_byte(struct Input* ptr_input);

/** @fn int Input_get_flags(struct Input* ptr_input)
 *  Returns the current flags.
 *  @param ptr_input A pointer to the initialized Input object
 *  @return The current flags
 *  @code
 *       #include <cywin.h>
 *       ...
 *       long sum = 0;
 *       struct Input* ptr_input;
 *       ...
 *       ptr_input = open_resource_Ex( "values.txt" );
 *       ...
 *       while( ! ( Input_get_flags(ptr_input) & FLAG_EOF ) )
 *       {
 *         sum += (long)Input_read_byte( ptr_input );
 *       }
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
import 21
#endif
int Input_get_flags(struct Input* ptr_input);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Input_is_eof(ptr_input)     (Input_get_flags(ptr_input)&FLAG_EOF)
#else
/** @fn bool Input_is_eof (struct Input* ptr_input)
 *  Returns TRUE if the EOF flag is set (stream reached end-of-file).
 *  @param ptr_input A pointer to the initialized Input object
 *  @return TRUE if the EOF flag is set (stream reached end-of-file)
 *  @code
 *       #include <cywin.h>
 *          ...
 *          long sum = 0;
 *          struct Input* ptr_input;
 *          ...
 *          ptr_input = open_resource_Ex( "values.txt" );
 *          ...
 *          while( ! ( Input_is_eof(ptr_input) )
 *          {
 *            sum += (long)Input_read_byte( ptr_input );
 *          }
 *          ...
 *          Input_dtor( ptr_input, FREE_MEMORY );
 *  @endcode
 *  @return Stream size, or -1 if the operation can't be done on this stream.
 *  @see Input_is_bad, Input_is_good.
 */
bool Input_is_eof (struct Input* ptr_input);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Input_is_bad(ptr_input) (Input_get_flags(ptr_input)&FLAG_BAD) 
#else
/** @fn bool Input_is_bad (struct Input* ptr_input)
 *  Returns TRUE if the BAD flag is set (stream is bad). 
 *  @param ptr_input A pointer to the initialized Input object
 *  @return TRUE if the BAD flag is set (stream is bad)
 *  @code
 *       #include <cywin.h>
 *          ...
 *          long sum = 0;
 *          struct Input* ptr_input;
 *          ...
 *          ptr_input = open_resource_Ex( "values.txt" );
 *          if (!Input_is_bad(ptr_input))
 *          {
 *             //success
 *          }
 *           ...
 *          Input_dtor( ptr_input, FREE_MEMORY );
 *  @endcode
 *  @see Input_is_good, Input_is_eof.
 */
bool Input_is_bad (struct Input* ptr_input);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Input_is_good(ptr_input)     (!(Input_get_flags(ptr_input)&FLAG_BAD))  
#else
/** @fn bool Input_is_good (struct Input* ptr_input)
 *  Returns TRUE if the BAD flag is not set (stream is good).
 *  @param ptr_input A pointer to the initialized Input object
 *  @return TRUE if the BAD flag is not set (stream is good)
 *  @code
 *       #include <cywin.h>
 *          ...
 *          long sum = 0;
 *          struct Input* ptr_input;
 *          ...
 *          ptr_input = open_resource_Ex( "values.txt" );
 *          if (Input_is_good (ptr_input))
 *          {
 *             //success
 *          }
 *           ...
 *          Input_dtor( ptr_input, FREE_MEMORY );
 *  @endcode
 *  @see Input_is_bad, Input_is_eof.
 */
bool Input_is_good (struct Input* ptr_input);
#endif


/** @fn long Input_get_size(struct Input* ptr_input)
 *  Returns the stream size (if applicable).
 *  @param ptr_input A pointer to the initialized Input object
 *  @return Stream size, or -1 if the operation can't be done on this stream
 *  @code
 *       #include <cywin.h>
 *       ...
 *       long size;
 *       long index;
 *       long sum = 0;
 *       struct Input* ptr_input;
 *       ...
 *       ptr_input = open_resource_Ex( "values.txt" );
 *       ...
 *       size = Input_get_size( ptr_input );
 *       for( index = 0; index < size; index++ )
 *       {
 *         sum += (long)Input_read_byte( ptr_input );
 *       }
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
import 1
#endif
long Input_get_size(struct Input* ptr_input);

/** @fn void Input_dtor(struct Input* ptr_input, int memory_flag)
 *  Destructor.
 *  @param ptr_input A pointer to the initialized Input object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in a stack
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       char tenth_byte;
 *       struct Input* ptr_input;
 *       ...
 *       ptr_input = open_resource_Ex( "data.txt" );
 *       ...
 *       Input_seekg( ptr_input, 10, SEEK_SET );
 *       tenth_byte = Input_read_byte( ptr_input );
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
import 2
#endif
void Input_dtor(struct Input* ptr_input, int);

#endif // DISABLE_STANDARD_RTL

/** @fn short Input_read_word(struct Input* ptr_input)
 *  Reads a 16-bit word, independent of hardware.
 *  @param ptr_input A pointer to the initialized Input object
 *  @return The next 2 bytes from the stream
 *  @code
 *       #include <cywin.h>
 *       ...
 *       long size;
 *       long index;
 *       long sum = 0;
 *       struct Input* ptr_input;
 *       ...
 *       ptr_input = open_resource_Ex( "values.txt" );
 *       ...
 *       size = Input_get_size( ptr_input )/(sizeof (int));
 *       for( index = 0; index < size; index++ )
 *       {
 *         sum += (long)Input_read_word( ptr_input );
 *       }
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see Input_read_dword.
 */
#ifndef _SKIP_DOC
__cyos 1007
#endif
short Input_read_word(struct Input* ptr_input);

/** @fn long Input_read_dword(struct Input* ptr_input)
 *  Reads a 32-bit word, independent of hardware.
 *  @param ptr_input A pointer to the initialized Input object
 *  @return The next 4 bytes from the stream
  *  @code
 *       #include <cywin.h>
 *       ...
 *       long size;
 *       long index;
 *       long sum = 0;
 *       struct Input* ptr_input;
 *       ...
 *       ptr_input = open_resource_Ex( "values.txt" );
 *       ...
 *       size = Input_get_size( ptr_input )/(sizeof (long));
 *       for( index = 0; index < size; index++ )
 *       {
 *         sum += (long)Input_read_dword( ptr_input );
 *       }
 *       ...
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see Input_read_word.
 */
#ifndef _SKIP_DOC
__cyos 1001
#endif
long Input_read_dword(struct Input* ptr_input);

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=      object Output.      =
//=  Abstract output stream. =
//=                          =
//============================

/**
 *  @brief an abstract object for the sequential data access objects.
 *  @ingroup filesystem_overview
 *  The CyOS's stream modes have two basic streams that are abstractions for the sequential data access objects: Input and Output.
 *  Those streams can pump data to and from virtually any source, including disk file, network
 *  connection, console terminals and more.
 *  Those basic streams are used, in turn, with filter streams, that apply
 *  specific features to some abstract streams' filters.
 *  <br>
 *  <br>
 *  Typical stream implementations include the file streams FileInput and FileOutput,
 *  console streams and various filters.
 *  Filter streams process the data from other streams.
 *  They take the other stream's objects while constructing, and are just like regular streams
 *  themselves. Typical filters are UNICODE/ASCII conversion, text formatting
 *  streams Reader and Writer and compressing streams Compressor and Decompressor.
 *  <br>
 *  <br>
 *  Filters are intended to combine the desired stream implementation with the selected
 *  filter object, to combine the features that best suit your needs.
 *  This object is just an interface for an input stream.
 *  @see filesystem_overview
 */
struct Output
{
#ifndef _SKIP_DOC
  char Output_Members_STUB[20]; /* CANNOT be instantiated! */
};
#endif

#ifndef DISABLE_STANDARD_RTL

// for the "outer world", most of the following functions look just like
// those of CyOS, although they're actually implemented by the rtl; note
// that there may exist custom rtls with different import sets;
// application using such rtl must #define DISABLE_STANDARD_RTL right
// before #include <cybiko.h>

/** @fn long Output_seekp(struct Output* ptr_output, long pos, seek_t mode)
 *  Seeks an output stream. If the stream supports a seek operation, it seeks to the specified 
 *  position in the specified mode. If the requested seek operation cannot be done (not 
 *  supported or wrong parameters), -1 will be returned. Seeking prior to the beginning of the 
 *  stream sets the pointer to the stream's first byte. Seeking after the end of the stream sets
 *  the pointer to the end of the stream; then the stream will be extended with garbage bytes 
 *  to the specified size.
 *  @param ptr_output A pointer to the initialized Output object
 *  @param pos The seek offset
 *  @param mode The seek mode
 *  @return The new position, or -1 if an error occurred
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       char tenth_byte = 100;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       Output_seekp( ptr_output, 10, SEEK_SET );
 *       Output_write_byte( ptr_output, tenth_byte );
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
import 6
#endif
long Output_seekp(struct Output* ptr_output, long pos, seek_t mode);

/** @fn long Output_tellp(struct Output* ptr_output)
 *  Returns the stream's position.
 *  @param ptr_output A pointer to the initialized Output object
 *  @return The stream's position, or -1 if that operation is not supported for this stream
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       char tenth_byte = 100;
 *       long cur_pos;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       Output_seekp( ptr_output, 10, SEEK_SET );
 *       Output_write_byte( ptr_output, tenth_byte );
 *       // cur_pos will be equal to 11.
 *       cur_pos = Output_tellp( ptr_output );
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
import 7
#endif
long Output_tellp(struct Output* ptr_output);

/** @fn long Output_write(struct Output* ptr_output, void* buffer, long length)
 *  Writes the 'length' bytes to the stream.
 *  @param ptr_output A pointer to the initialized Output object
 *  @param buffer Where to get the data
 *  @param length The number of bytes to be written
 *  @return The actual number of written bytes; may be less that than 'length' or even zero
 *  @code
 *       #include <cywin.h>
 *       #include "score.h"
 *       ...
 *       struct Input* ptr_input;
 *       struct score_t high_scores[10];
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       char tenth_byte = 100;
 *       long cur_pos;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       Output_write( ptr_output, high_scores, sizeof( high_scores ) );
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see Output_write_byte.
 */
#ifndef _SKIP_DOC
import 26
#endif
long Output_write(struct Output* ptr_output, void* buffer, long length);

/** @fn int Output_write_byte(struct Output* ptr_output, int byte)
 *  Writes a byte to the stream.
 *  @param ptr_output A pointer to the initialized Output object
 *  @param byte Determines which byte to write
 *  @return -1 if the process fails, the parameter's value if it succeeds
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       char tenth_byte = 100;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       Output_seekp( ptr_output, 10, SEEK_SET );
 *       Output_write_byte( ptr_output, tenth_byte );
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see Output_write.
 */
#ifndef _SKIP_DOC
import 8
#endif
int Output_write_byte(struct Output* ptr_output, int byte);

/** @fn int Output_get_flags(struct Output* ptr_output)
 *  Returns the current flags.
 *  @param ptr_output A pointer to the initialized Output object
 *  @return Current flags
 *  @code
 *       #include <cywin.h>
 *       ...
 *       char obtain_new_value();
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       char current_value;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       while( !( Output_get_flags(ptr_output) & FLAG_EOF ) )
 *       {
 *         current_value = obtain_new_value();
 *         Output_write_byte( ptr_output, current_value );
 *       }
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
import 25
#endif
int Output_get_flags(struct Output* ptr_output);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Output_is_eof(ptr_output)          (Output_get_flags(ptr_output)&FLAG_EOF) 
#else
/** @fn bool Output_is_eof (struct Output* ptr_output) 
 *  Returns TRUE if the EOF flag is set (stream reached end-of-file).
 *  @param ptr_output A pointer to the initialized Output object
 *  @return TRUE if the EOF flag is set (stream reached end-of-file)
 *  @code
 *   #include <cywin.h>
 *   ...
 *   char obtain_new_value();
 *   ...
 *   struct module_t main_module;
 *   struct Output* ptr_output;
 *   char current_value;
 *   ...
 *   ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                       "data.txt" );
 *   ...
 *   while( !( Output_is_eof(ptr_output))
 *   {
 *     current_value = obtain_new_value();
 *     Output_write_byte( ptr_output, current_value );
 *   }
 *   ...
 *   Output_dtor( ptr_output, FREE_MEMORY );
 *   ...
 *  @endcode
 *  @see Output_is_good, Output_is_bad.
 */
bool Output_is_eof (struct Output* ptr_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Output_is_bad(ptr_output)          (Output_get_flags(ptr_output)&FLAG_BAD)
#else
/** @fn bool Output_is_bad (struct Output* ptr_output) 
 *  Returns TRUE if the BAD flag is set (stream is bad).
 *  @param ptr_output A pointer to the initialized Output object
 *  @return TRUE if the BAD flag is set (stream is bad)
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Output *ptr_output;
 *       ...
 *       Output_ctor_Ex(ptr_output, "game.save");
 *       if (!FileOutput_is_bad(ptr_output))
 *       {
 *         //success
 *       }
 *       ...
 *       Output_dtor( ptr_output, LEAVE_MEMORY );
 *  @endcode
 *  @see Output_is_good, Output_is_eof.
 */
bool Output_is_bad (struct Output* ptr_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Output_is_good(ptr_output)         (!(Output_get_flags(ptr_output)&FLAG_BAD))  
#else
/** @fn  bool Output_is_good (struct Output* ptr_output)
 *  Returns TRUE if the BAD flag is not set (stream is good).
 *  @param ptr_output A pointer to the initialized Output object
 *  @return TRUE if the BAD flag is not set (stream is good)
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Output *file_output;
 *       ...
 *       Output_ctor_Ex(file_output, "game.save");
 *       if (FileOutput_is_good(file_output))
 *       {
 *         //success
 *       }
 *       ...
 *       Output_dtor( &file_output, LEAVE_MEMORY );
 *  @endcode
 *  @see Output_is_eof, Output_is_bad.
 */
bool Output_is_good (struct Output* ptr_output);
#endif

/** @fn long Output_get_size(struct Output* ptr_output)
 *  Returns the stream size (if applicable).
 *  @param ptr_output A pointer to the initialized Output object
 *  @return Stream size, or -1 if the operation can't be done on this stream
 *  @code
 *       #include <cywin.h>
 *       ...
 *       char obtain_new_value();
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       char current_value;
 *       long size;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       size = Output_get_size( ptr_output );
 *       for( index = 0; index < size; index++ )
 *       {
 *         current_value = obtain_new_value();
 *         Output_write_byte( ptr_output, current_value );
 *       }
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
import 23
#endif
long Output_get_size(struct Output* ptr_output);

/** @fn void Output_dtor(struct Output* ptr_output, int memory_flag)
 *  Destructor.
 *  @param ptr_output A pointer to the initialized Output object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in a stack
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       char tenth_byte = 100;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       Output_seekp( ptr_output, 10, SEEK_SET );
 *       Output_write_byte( ptr_output, tenth_byte );
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
import 24
#endif
void Output_dtor(struct Output* ptr_output, int);

#endif // DISABLE_STANDARD_RTL

/** @fn short Output_write_word(struct Output* ptr_output, short word)
 *  Writes a 16-bit word, independent of hardware.
 *  @param ptr_output A pointer to the initialized Output object
 *  @param word A word to be written
 *  @return -1 if the process fails, the parameter's value if it succeeds
 *  @code
 *       #include <cywin.h>
 *       ...
 *       int obtain_new_value();
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       int current_value;
 *       long size;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       size = Output_get_size( ptr_output )/sizeof(int);
 *       for( index = 0; index < size; index++ )
 *       {
 *         current_value = obtain_new_value();
 *         Output_write_word( ptr_output, current_value );
 *       }
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see Output_write_dword.
 */
#ifndef _SKIP_DOC
__cyos 1298
#endif
short Output_write_word(struct Output* ptr_output, short word);

/** @fn long Output_write_dword(struct Output* ptr_output, long dword)
 *  Writes a 32-bit word, independent of hardware.
 *  @param ptr_output A pointer to the initialized Output object
 *  @param dword A dword to be written
 *  @return -1 if the process fails, the parameter's value if it succeeds 
 *  @code
 *       #include <cywin.h>
 *       ...
 *       long obtain_new_value();
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       long current_value;
 *       long size;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       size = Output_get_size( ptr_output )/sizeof(long);
 *       for( index = 0; index < size; index++ )
 *       {
 *         current_value = obtain_new_value();
 *         Output_write_dword( ptr_output, current_value );
 *       }
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see Output_write_word.
 */
#ifndef _SKIP_DOC
__cyos 1293
#endif
long Output_write_dword(struct Output* ptr_output, long dword);

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=      object Flag.        =
//=  Synchronization object. =
//=                          =
//============================

/**
 *  @brief the flag synchronization object (aka Semaphore).
 *
 *  @ingroup syncronization_group
 *  This synchronization object has two states: set and clear. Threads may wait
 *  for the flag be in the specific state and change it.
 *  @see syncronization_group
 */

#ifndef _SKIP_DOC
struct Flag : public NameBlock
#else
struct Flag
#endif
{
#ifndef _SKIP_DOC
  char Flag_Members_STUB[12];
};
#endif

/** @fn struct Flag* Flag_ctor(struct Flag* ptr_flag, char* name, bool set)
 *  Creates a named flag.
 *  @param ptr_flag A pointer to the Flag structure
 *  @param name The name of the flag
 *  @param set if TRUE, then flag state will be set; otherwise it will clear
 *  @return A pointer to the initialized Flag object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait_clear( &data_busy, 1000 ) )
 *       {
 *         Flag_set( &data_busy );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_clear( &data_busy );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 209
#endif
struct Flag* Flag_ctor(struct Flag* ptr_flag, char* name, bool set);

/** @fn void Flag_set(struct Flag* ptr_flag)
 *  Sets a flag, or changes the flag to the set state.
 *  If there are threads waiting for the flag object's state to become set,
 *  all of them will become active. If there were higher priority threads
 *  already waiting before this one, the context switch may happen immediately.
 *  Does nothing if the flag is already set.
 *  @param ptr_flag A pointer to the Flag structure
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait_clear( &data_busy, 1000 ) )
 *       {
 *         Flag_set( &data_busy );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_clear( &data_busy );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Flag_set_Ex.
 */
#ifndef _SKIP_DOC
__cyos 1087
#endif
void Flag_set(struct Flag* ptr_flag);

/** @fn void Flag_set_Ex(struct Flag* ptr_flag, bool state)
 *  Sets the flag to a specified state, or changes the flag's state to set or clear.
 *  If there are threads waiting for the flag object's state to change this way,
 *  all of them will become active. If there were higher priority threads already
 *  waiting before this one, the context switch may happen immediately. 
 *  @param ptr_flag A pointer to the initialized Flag object
 *  @param state The desired state; "TRUE" for set, "FALSE" for clear
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait_clear( &data_busy, 1000 ) )
 *       {
 *         Flag_set_Ex( &data_busy, TRUE );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_set_Ex( &data_busy, FALSE );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Flag_set.
 */
#ifndef _SKIP_DOC
__cyos 1088
#endif
void Flag_set_Ex(struct Flag* ptr_flag, bool state);

/** @fn void Flag_clear(struct Flag* ptr_flag)
 *  Sets a flag in a clear state, or changes the flag's state to clear.
 *  If there are threads waiting for the flag object's state to become clear,
 *  all of them will become active. If there were higher priority threads
 *  already waiting before this one, the context switch may happen immediately. 
 *  @param ptr_flag A pointer to the initialized Flag object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait_clear( &data_busy, 1000 ) )
 *       {
 *         Flag_set( &data_busy );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_clear( &data_busy );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 439
#endif
void Flag_clear(struct Flag* ptr_flag);

/** @fn bool Flag_is_set(struct Flag* ptr_flag)
 *  Returns TRUE if a flag is in the set state.
 *  @param ptr_flag A pointer to the initialized Flag object
 *  @return TRUE if a flag is in the set state
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       while( Flag_is_set( &data_busy ) )
 *       {
 *         cWinApp( main_module.m_process, 250 );
 *       }
 *       Flag_set( &data_busy );
 *       ...
 *       // Uses shared data.
 *       ...
 *       Flag_clear( &data_busy );
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 821
#endif
bool Flag_is_set(struct Flag* ptr_flag);

/** @fn void Flag_flip(struct Flag* ptr_flag)
 *  Flips a flag's state from set to clear or vice-versa.
 *  @param ptr_flag A pointer to the initialized Flag object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait_clear( &data_busy, 1000 ) )
 *       {
 *         Flag_set( &data_busy );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_flip( &data_busy );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 600
#endif
void Flag_flip(struct Flag* ptr_flag);

/** @fn bool Flag_wait_set(struct Flag* ptr_flag, clock_t timeout)
 *  Wait the flag's state to change to set.
 *  If flag is already set, it returns immediately. If the flag is clear,
 *  it waits until the timeout expires or flag's state changes. 
 *  @param ptr_flag A pointer to the initialized Flag object
 *  @param timeout Time to wait, in milliseconds (0 means forever) 
 *  @return TRUE on success, FALSE on timeout
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_not_busy;
 *       ...
 *       Flag_ctor( &data_not_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait_set( &data_not_busy, 1000 ) )
 *       {
 *         Flag_clear( &data_not_busy );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_flip( &data_not_busy );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_not_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1279
#endif
bool Flag_wait_set(struct Flag* ptr_flag, clock_t timeout);

/** @fn bool Flag_wait_clear(struct Flag* ptr_flag, clock_t timeout)
 *  Waits for the flag's state to become clear. If the flag is already clear,
 *  it returns immediately. If the flag is set, it waits until the timeout
 *  expires or the flag state changes.
 *  @param ptr_flag A pointer to the initialized Flag object
 *  @param timeout Time to wait, in milliseconds. (0 means forever)
 *  @return TRUE on success, FALSE on timeout
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait_clear( &data_busy, 1000 ) )
 *       {
 *         Flag_set( &data_busy );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_flip( &data_busy );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1275
#endif
bool Flag_wait_clear(struct Flag* ptr_flag, clock_t timeout);

/** @fn bool Flag_wait(struct Flag* ptr_flag, bool set, clock_t timeout)
 *  Waits for a specified state. If the flag is already in the desired state,
 *  it returns; otherwise it waits for a specified timeout. 
 *  @param ptr_flag A pointer to the initialized Flag object
 *  @param set 'true' if you need to wait for the set state, 'false' for the clear state
 *  @param timeout Time to wait, in milliseconds. (0 means forever)
 *  @return TRUE on success, FALSE on timeout
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait( &data_busy, FALSE, 1000 ) )
 *       {
 *         Flag_set( &data_busy );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_flip( &data_busy );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1273
#endif
bool Flag_wait(struct Flag* ptr_flag, bool set, clock_t timeout);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Flag_dtor(ptr_flag, memory_flag) DLink_dtor(ptr_flag, memory_flag)
#else
/** @fn void Flag_dtor(struct Flag* ptr_flag, int memory_flag)
 *  Destructor.
 *  @param ptr_flag A pointer to the initialized Flag object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in a stack
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flag data_busy;
 *       ...
 *       Flag_ctor( &data_busy, "shared_data", FALSE );
 *       ...
 *       if( Flag_wait_clear( &data_busy, 1000 ) )
 *       {
 *         Flag_set( &data_busy );
 *         ...
 *         // Uses shared data.
 *         ...
 *         Flag_clear( &data_busy );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Flag_dtor( &data_busy, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
void Flag_dtor(struct Flag* ptr_flag, int memory_flag);
#endif

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=      object Mutex.       =
//=  Synchronization object. =
//=                          =
//============================

/**
 *  @brief Mutex a kind of a synchronization object that provides MUTually EXclusive (MUTEX) access to some part of the code (in some systems this is also known as Resource).
 *  @ingroup syncronization_group
 *  Mutex is a kind of a synchronization object that provides MUTually EXclusive (MUTEX)
 *  access to some part of the code (in some systems this is also known as Resource).
 *  Mutex can be locked by a Thread using the Mutex_lock
 *  method, then released. Only one thread at a time owns the lock on any given Mutex
 *  when the lock is released, the next thread (if any) waiting for the Mutex is activated.
 *  <br>
 *  <br>
 *  If a thread tries to lock a Mutex that is already locked, the thread will be set
 *  to wait until a timeout expires or until the Mutex object is unlocked.
 *  Take care to prevent deadlocks: for example, when one thread owns
 *  Mutex m1 and tries to lock Mutex m2,while a second thread already owns Mutex m2 and is
 *  trying to lock Mutex m1. Deadlocks can occur not only with Mutexes, but practically with any set of synchronization
 *  objects. Since the OS can not yet detect all deadlocks, it is up to programmers
 *  to design their systems accordingly.
 *  <br>
 *  <br>
 *  It doesn't matter how long you've locked a Mutex object-it unlocks the first time you
 *  call Mutex_unlock(). The system does not allow unowned
 *  Mutex objects to be unlocked. 
 *  @see syncronization_group
 */
#ifndef _SKIP_DOC
struct Mutex : public NameBlock
#else
struct Mutex
#endif
{
#ifndef _SKIP_DOC
  char Mutex_Members_STUB[8];
};
#endif

/** @fn struct Mutex* Mutex_ctor(struct Mutex* ptr_mutex, char* name)
 *  Constructs the named Mutex object.
 *  @param ptr_mutex A pointer to the Mutex structure
 *  @param name The name of the Mutex object
 *  @return A pointer to the initialized Mutex object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Mutex data_mutex;
 *       ...
 *       Mutex_ctor( &data_mutex, "shared_data" );
 *       ...
 *       if( Mutex_lock( &data_mutex, 1000 ) )
 *       {
 *         // Uses shared data.
 *         ...
 *         Mutex_unlock( &data_mutex );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Mutex_dtor( &data_mutex, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 214
#endif
struct Mutex* Mutex_ctor(struct Mutex* ptr_mutex, char* name);

/** @fn bool Mutex_is_locked(struct Mutex* ptr_mutex)
 *  Checks whether the Mutex is locked.
 *  @param ptr_mutex A pointer to the Mutex structure
 *  @return TRUE if the Mutex object is locked
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Mutex data_mutex;
 *       ...
 *       Mutex_ctor( &data_mutex, "shared_data" );
 *       ...
 *       while( Mutex_is_locked( &data_mutex ) )
 *       {
 *         cWinApp( main_module.m_process, 250 );
 *       }
 *       Mutex_lock( &data_mutex, 0 );
 *       ...
 *       // Uses shared data.
 *       Mutex_unlock( &data_mutex );
 *       ...
 *       Mutex_dtor( &data_mutex, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 814
#endif
bool Mutex_is_locked(struct Mutex* ptr_mutex);

/** @fn bool Mutex_lock(struct Mutex* ptr_mutex, clock_t timeout)
 *  Returns a lock.
 *  Attempts to get the lock for specified time. Note that if you already
 *  locked an object, another Mutex_lock() operations does absolutely \a nothing, and
 *  the first Mutex_unlock() call unlocks it immediately no matter how many
 *  times you've called the Mutex_lock() method. Be careful when designing your
 *  Mutex logic!
 *  @param ptr_mutex A pointer to the initialized Mutex object
 *  @param timeout Determines how long wait if a lock can not be acquired (0 means forever)
 *  @return TRUE if a lock was acquired, FLASE if timeout has expired
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Mutex data_mutex;
 *       ...
 *       Mutex_ctor( &data_mutex, "shared_data" );
 *       ...
 *       if( Mutex_lock( &data_mutex, 1000 ) )
 *       {
 *         // Uses shared data.
 *         ...
 *         Mutex_unlock( &data_mutex );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Mutex_dtor( &data_mutex, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Mutex_unlock.
 */
#ifndef _SKIP_DOC
__cyos 865
#endif
bool Mutex_lock(struct Mutex* ptr_mutex, clock_t timeout);

#ifndef _SKIP_DOC
__cyos 1022
void Mutex_release(struct Mutex* ptr_mutex);
#endif

/** @fn void Mutex_unlock(struct Mutex* ptr_mutex)
 *  Unlocks a Mutex object.
 *  Note that if you did not Mutex_lock() it before, the system ignores the unlock command
 *  and prints a warning on the console. Also be warned: Mutex_unlock()
 *  releases the lock immediately. No matter how many times Mutex_lock()
 *  was called, only the first Mutex_unlock() works. any repeared commands will be ignored.
 *  @param ptr_mutex A pointer to the initialized Mutex object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Mutex data_mutex;
 *       ...
 *       Mutex_ctor( &data_mutex, "shared_data" );
 *       ...
 *       if( Mutex_lock( &data_mutex, 1000 ) )
 *       {
 *         // Uses shared data.
 *         ...
 *         Mutex_unlock( &data_mutex );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Mutex_dtor( &data_mutex, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Mutex_lock.
 */
#ifndef _SKIP_DOC
__cyos 1261
#endif
void Mutex_unlock(struct Mutex* ptr_mutex);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Mutex_dtor(ptr_mutex, memory_flag) DLink_dtor(ptr_mutex, memory_flag)
#else
/** @fn void Mutex_dtor(struct Mutex* ptr_mutex, int memory_flag)
 *  Destructor.
 *  @param ptr_mutex A pointer to the initialized Mutex object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in a stack
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Mutex data_mutex;
 *       ...
 *       Mutex_ctor( &data_mutex, "shared_data" );
 *       ...
 *       if( Mutex_lock( &data_mutex, 1000 ) )
 *       {
 *         // Uses shared data.
 *         ...
 *         Mutex_unlock( &data_mutex );
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while trying to access." );
 *       }
 *       ...
 *       Mutex_dtor( &data_mutex, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
void Mutex_dtor(struct Mutex* ptr_mutex, int memory_flag);
#endif

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=      object Flagex.      =
//=  Synchronization object. =
//=                          =
//============================

/** @brief synchronization object that combines FLAG and MutEX (FLAGEX).
 *  @ingroup syncronization_group
 *  A Flag with Exclusion. Flagex is a son of Mutex that contains a Flag.
 *  Hence the name, a combination of FLAG and mutEX. It is a sort of synchronization
 *  object you can use when your object needs both to implement a wait-for-notification
 *  function as well as to lock and unlock like Mutex objects can. It works on the same
 *  principle as Java's "monitor lock." The main difference is that Flagex cannot notify
 *  a single thread; Flagex_notify is the equivalent of Java's Object.notifyAll().
 *  <br>
 *  <br>
 *  Flagex offers three main functions: 
 *  <ul>
 *  <li>Locks and unlocks objects with Flagex_lock and Flagex_unlock, the same way you lock and unlock Mutex objects.</li>
 *  <li>Waits for notification with Flagex_wait.</li>
 *  <li>Notifies waiting threads (waking them) with Flagex_notify.</li>
 *  </ul>
 *  The key is that Flagex_wait and Flagex_notify only function when the object is Flagex_locked.
 *  When your object wakes after Flagex_wait, the system regains the lock on the Flagex for your thread.
 *  After you call Flagex_wait, the thread is released so another thread can Flagex_lock
 *  it and then Flagex_notify it. 
 *  <br>
 *  <br>
 *  Use this method when you need to synchronize exclusive access to a specific object until another
 *  process concludes on the same object. This allows the accessing thread both to wait and to synchronize
 *  its action with the other thread.
 *  @see syncronization_group
 */
struct Flagex : public Mutex
{
#ifndef _SKIP_DOC
  char Flagex_Members_STUB[4];
};
#endif

/** @fn struct Flagex* Flagex_ctor(struct Flagex* ptr_flagex, char* name)
 *  Constructs a named Flagex object.
 *  @param ptr_flagex A pointer to the Flagex structure
 *  @param name The name of the Flagex object
 *  @return A pointer to the initialized Flagex object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flagex wait_for_data;
 *       ...
 *       Flagex_ctor( &wait_for_data, "data_ready" );
 *       ...
 *       if( Flagex_wait( &wait_for_data, 1000 ) )
 *       {
 *         // Uses shared data.
 *         ...
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while waiting for data." );
 *       }
 *       ...
 *       Flagex_dtor( &wait_for_data, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 223
#endif
struct Flagex* Flagex_ctor(struct Flagex* ptr_flagex, char* name);

/** @fn bool Flagex_wait(struct Flagex* ptr_flagex, clock_t timeout)
 *  Waits for notification.
 *  Puts the calling thread to sleep until some other thread calls
 *  Flagex_notify(). You must have a lock on this object to call it.
 *  @param ptr_flagex A pointer to the initialized Flagex object
 *  @param timeout Determines how long to wait, in milliseconds ( 0 means forever)
 *  @return TRUE on notification and FALSE on timeout
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flagex wait_for_data;
 *       ...
 *       Flagex_ctor( &wait_for_data, "data_ready" );
 *       ...
 *       if( Flagex_wait( &wait_for_data, 1000 ) )
 *       {
 *         // Uses shared data.
 *         ...
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while waiting for data." );
 *       }
 *       ...
 *       Flagex_dtor( &wait_for_data, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1274
#endif
bool Flagex_wait(struct Flagex* ptr_flagex, clock_t timeout);

/** @fn void Flagex_notify(struct Flagex* ptr_flagex)
 *  Notifies all waiting threads.
 *  All threads that called Flagex_wait() for this object will be
 *  waked up. Note that they won't start until you will unlock the
 *  object, and that most priority one (and, though not guaranteed,
 *  first called wait() from top  priority threads) will get the lock
 *  and actually start.
 *  You must have a lock on the object to call this method.
 *  @param ptr_flagex A pointer to the initialized Flagex object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flagex wait_for_data;
 *       ...
 *       Flagex_ctor( &wait_for_data, "data_ready" );
 *       ...
 *       // Prepares data.
 *       ...
 *       Flagex_notify( &notify ;
 *       ...
 *       Flagex_dtor( &wait_for_data, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 903
#endif
void Flagex_notify(struct Flagex* ptr_flagex);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Flagex_is_locked      Mutex_is_locked 
#else
/** @fn bool Flagex_is_locked (struct Flagex* ptr_flagex)
 *  Checks whether Mutex is locked.
 *  @param ptr_flagex A pointer to the initialized Flagex object
 *  @return TRUE if Mutex is locked
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flagex wait_for_data;
 *       ...
 *       Flagex_ctor( &wait_for_data, "data_ready" );
 *       while( Flagex_is_locked( &wait_for_data ) )
 *       {
 *          cWinApp( main_module.m_process, 250 );
 *       }
 *       Flagex_lock( &wait_for_data, 0 );
 *       ...
 *       // Uses shared data.
 *       ...
 *       Flagex_unlock( &wait_for_data); 
 *       ...
 *       Flagex_dtor( &wait_for_data, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
bool Flagex_is_locked (struct Flagex* ptr_flagex);
#endif
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Flagex_lock           Mutex_lock 
#else
/** @fn bool Flagex_lock (struct Flagex* ptr_flagex, clock_t timeout)
 *  Returns a lock. It attempts to get the lock for specified time. 
 *          Note that if you already locked it, another Flagex_lock() 
 *          operation has no additive effect. The first Flagex_unlock() 
 *          call unlocks it immediately no matter how many times you've 
 *          called the Flagex_lock() method. Be careful when 
 *           designing your Flagexes logic! 
 *  @param ptr_flagex A pointer to the initialized Flagex object
 *  @param timeout Determines how long to wait if a lock can not be acquired;
 *         (0 means forever)
 *  @return TRUE if a lock was acquired, FALSE if timeout expired
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flagex wait_for_data;
 *       ...
 *       Flagex_ctor( &wait_for_data, "data_ready" );
 *       while( Flagex_is_locked( &wait_for_data ) )
 *       {
 *          cWinApp( main_module.m_process, 250 );
 *       }
 *       Flagex_lock( &wait_for_data, 0 );
 *       ...
 *       // Uses shared data.
 *       ...
 *       Flagex_unlock( &wait_for_data); 
 *       ...
 *       Flagex_dtor( &wait_for_data, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Flagex_unlock.
 */
bool Flagex_lock (struct Flagex* ptr_flagex, clock_t timeout);
#endif
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Flagex_unlock         Mutex_unlock ///< 
#else
/** @fn void Flagex_unlock (struct Flagex* ptr_flagex) 
 *  Unlocks Flagex. Note that if you did not lock it before, 
 *         the system ignores the unlock command and prints warning on the console. 
 *         Also be warned: Flagex_unlock() releases the lock immediately, 
 *         no matter how many times Flagex_lock() was called. 
 *         Only the first Flagex_unlock() works; any repeared calls will be ignored.
 *  @param ptr_flagex A pointer to the initialized Flagex object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flagex wait_for_data;
 *       ...
 *       Flagex_ctor( &wait_for_data, "data_ready" );
 *       while( Flagex_is_locked( &wait_for_data ) )
 *       {
 *          cWinApp( main_module.m_process, 250 );
 *       }
 *       Flagex_lock( &wait_for_data, 0 );
 *       ...
 *       // Uses shared data.
 *       ...
 *       Flagex_unlock( &wait_for_data); 
 *       ...
 *       Flagex_dtor( &wait_for_data, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Flagex_lock.
 */
void Flagex_unlock (struct Flagex* ptr_flagex);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Flagex_dtor(ptr_flagex, memory_flag) DLink_dtor(ptr_flagex, memory_flag)
#else
/** @fn void Flagex_dtor(struct Flagex* ptr_flagex, int memory_flag)
 *  Destructor.
 *  @param ptr_flagex A pointer to the initialized Flagex object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in a stack
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct Flagex wait_for_data;
 *       ...
 *       Flagex_ctor( &wait_for_data, "data_ready" );
 *       ...
 *       if( Flagex_wait( &wait_for_data, 1000 ) )
 *       {
 *         // Uses shared data.
 *         ...
 *       }
 *       else
 *       {
 *         TRACE( "Timeout while waiting for data." );
 *       }
 *       ...
 *       Flagex_dtor( &wait_for_data, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
void Flagex_dtor(struct Flagex* ptr_flagex, int memory_flag);
#endif

#ifdef _SKIP_DOC
};
#endif

/** @fn void set_hourglass(bool enable)
 *  @ingroup syncronization_group
 *  When you start this function, the system starts a special Thread to show the picture,
 *  which can eat up to 10-11% of processor time; after you stop it,
 *  don't forget to remove it from the screen by repainting it.
 *  @param enable TRUE to start the show, FALSE to stop it
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       long obtain_new_value();
 *       ...
 *       struct module_t main_module;
 *       struct Output* ptr_output;
 *       long current_value;
 *       long size;
 *       ...
 *       ptr_output = Archive_open_write_Ex ( main_module.m_process->module->archive,
 *                                           "data.txt" );
 *       ...
 *       size = Output_get_size( ptr_output )/sizeof(long);
 *       set_hourglass( TRUE );
 *       for( index = 0; index < size; index++ )
 *       {
 *         current_value = obtain_new_value();
 *         Output_write_dword( ptr_output, current_value );
 *       }
 *       set_hourglass( FALSE );
 *       ...
 *       Output_dtor( ptr_output, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1128
#endif
void set_hourglass(bool enable);

/** @fn cyid_t get_own_id(void)
 *  @ingroup messaging_group
 *  @brief Returns device's CyID
 *  @return Device's CyID
 *  @code
 *       #include <cywin.h>
 *       ...
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *       ptr_message->msgid = MSG_TEST;
 *       // Sends the message to the 'data_receiver' process on the same device.
 *       Message_post( ptr_message, "data_receiver", get_own_id() );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 701
#endif
cyid_t get_own_id(void);

#endif  //  #ifndef __CYBER_H_INCLUDE__



