/** 
 *  @file cybiko.h
 *  Run-Time Library API plus hooks to CyOS API.
 *
 *  This should be included by any vcc1-compiled Cybiko application.
 */
#ifndef __CYBIKO_H_INCLUDED__
#define __CYBIKO_H_INCLUDED__

/**
 *  Frees the memory block allocated for the object(used in destructor)
 * @ingroup memory_overview
 */
#define  FREE_MEMORY     3 

/**
 *  Doesn't free the memory block allocated for the object (used in destructor)
 * @ingroup memory_overview
 */
#define  LEAVE_MEMORY    2

#include "cyber-types.h"
#include "time.h"
#include "cyber.h"
#include "cyber-graph.h" 
#include "cyber-message.h"
#include "archive.h"
#include "module.h"
#include "process.h"
#include "cyber-files.h"
#include "cyber-rf.h"
#include "cyber-lib.h"
#include "keybd.h"
#include "mplayer.h"
#include "tone_gen.h"
#include "vibrator.h"
#include "os_filename.h"
#include "form.h"
#include "varargs.h"
#include "around.h"
#include "finder.h"
#include "score.h"
#include "comport.h"
#include "cy3d.h"

/**
 *  @brief the application module structure.
 *  @ingroup multithreads_overview
 *  Structure of the application module. Contains pointers to the
 *  graphics device context and to the current cywin application.
 *  Use this structure with the init_module() function.  
 *  @see init_module.
 *  @see multithreads_overview
 */
struct module_t // 32 bytes shared with *any* C++ runtime
{
/**
 *  Graphics device context.
*/
  struct DisplayGraphics* m_gfx;         
/**
 *  Pointer to the current cywin application.
*/
  struct cWinApp*         m_process;     
#ifndef _SKIP_DOC
  long                    m_tick;        // current game tick
  short                   m_tick_period; // 100 (def) stands for 0.1 second
  short                   m_key;         // last key pressed
  short                   m_key_mask;    // status bits for that keypress
  char                    m_visible;     // 1 if application has focus
  char                    m_quit;        // 1 if application should terminate
#endif
};

#ifndef DISABLE_STANDARD_RTL

// for the "outer world", most of the following functions look just like
// those of CyOS, although they're actually implemented by the rtl; note
// that there may exist custom rtls with different import sets;
// application using such rtl must #define DISABLE_STANDARD_RTL right
// before #include <cybiko.h>

/**
 *  @ingroup multithreads_overview
 *  Module initializer. You must call this function in each application 
 *  to obtain a module.
 *  @param module The pointer to a module_t object
 *  @return A pointer to the application's initialized module
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       //  Application's main module
 *       struct module_t main_module;
 *       ...
 *       long main( int argc, char* argv[], bool start )
 *       {
 *         //  Initializes the application.
 *         init_module( &main_module );
 *         ...
 *      }
 *  @endcode
 */
#ifndef _SKIP_DOC
import 0
#endif
long init_module(struct module_t* ptr_module);

/**
 *  @ingroup math_group
 *  Returns a pseudo-random value from 0 to max - 1.
 *  @param max the value what determines the range of return values.
 *  This range is [0, max-1].
 *  @return A pseudo-random value from 0 to max-1. 
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int nRandVal;
 *       ...
 *       //  Returns pseudo-random value from 0 to 99.
 *       nRandVal = random( 100 );
 *  @endcode
 */
#ifndef _SKIP_DOC
import 17
#endif
long random(int max);

// extension libraries support (not for explicit use)
#ifndef _SKIP_DOC
import 19
void xlib_load( void );

import 20

void xlib_free( void );
#endif  // #ifndef _SKIP_DOC

#endif  // DISABLE_STANDARD_RTL

/**
 *  @ingroup io_overview
 *  Traces to the system console. Specifies a variable number of arguments
 *  that are used in exactly the same way that a variable number of
 *  arguments are used in the run-time function cprintf.
 *  Use format and variable arguments to format your message and make it 
 *  readable and informative. Please keep in mind that it adds a Thread/timers 
 *  prefix to the line, as well as necessary line ends.
 *  @param sz_format format control(see sprintf() for the full 
 *  format description).
 *  @param argument optional arguments.
 *  @return none.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "Hello, World!" );
 *       ...
 *  @endcode
 *  @see cprintf, sprintf.
 */
#ifndef _SKIP_DOC
__cyos 75
void TRACE(char* sz_format,...);
#else
void TRACE(char* sz_format [, argument]... );
#endif

/**
 *  @ingroup io_overview
 *  Formats and prints the string to the Cybiko console.
 *  @param sz_format format control(see sprintf() for the full 
 *  format description).
 *  @param argument optional arguments.
 *  @return the number of characters printed.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int nVal;
 *       nVal = 54;
 *       ...
 *       cprintf( "nVal = %d", nVal );
 *       ...
 *  @endcode
 *  @see TRACE, sprintf.
 */
#ifndef _SKIP_DOC
__cyos 478
int cprintf(char* sz_format,...);
#else
int cprintf(char* sz_format [, argument]...);
#endif  //  #ifndef _SKIP_DOC

#ifndef _SKIP_DOC
__cyos 407
int cgetch(clock_t timeout);

__cyos 408
int cgets(char* buff,size_t max_size,clock_t timeout);

__cyos 479
void cputch(char chr,clock_t timeout);

__cyos 480
void cputs(char* str,clock_t timeout);
#endif

/**
 *  @ingroup memory_overview
 *  Allocates memory blocks.
 *  Afterward you must free this memory block by calling free().
 *  @param size number of bytes to allocate
 *  @return a void pointer to the allocated space, 
 *  or NULL if there is insufficient memory available
 *  @code
 *       #include <cybiko.h>
 *       #define  MAX_STR_LEN     256
 *       ...
 *       char* sz_string;
 *       ...
 *       sz_string = (char*)malloc( MAX_STR_LEN );
 *       ...
 *       free(sz_string);
 *       ...
 *  @endcode
 *  @see free.
 */
#ifndef _SKIP_DOC
__cyos 876
#endif
char* malloc(size_t size);

/**
 *  @ingroup memory_overview
 *  Allocates an array in memory, with elements initialized to 0.
 *  You can only free this memory block - by calling free().
 *  @param number the number of elements to allocate
 *  @param size the length in bytes of each element.
 *  @return a void pointer to the allocated space, 
 *  or NULL if there is insufficient memory available
 *  @code
 *       #include <cybiko.h>
 *       #define  MAX_OBJECTS     100
 *       ...
 *       struct cObject* ptr_object_array;
 *       ...
 *       ptr_object_array = (struct cObject*)calloc( MAX_OBJECTS, 
 *                                                   sizeof(struct cObject) );
 *       ...
 *       free(ptr_object_array);
 *       ...
 *  @endcode
 *  @see free.
 */
#ifndef _SKIP_DOC
__cyos 399
#endif
void* calloc(size_t number, size_t size);

/**
 *  @ingroup memory_overview
 *  Reallocates memory blocks.
 *  Afterward you must free this memory block by calling free().
 *  @param ptr_memory_block pointer to a previously allocated memory block
 *  @param new_size new size in bytes
 *  @return a pointer to the reallocated memory block
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char* sz_string;
 *       char* sz_source_string;
 *       ...
 *       sz_string = (char*)malloc( 10 );
 *       ...
 *       if( strlen( sz_source_string ) <= 10)
 *       {
 *         sz_string = (char*)realloc( sz_string, 
 *                                     strlen( sz_source_string ) + 1 );
 *       }
 *       ...
 *       free(sz_string);
 *       ...
 *  @endcode
 *  @see free.
 */
#ifndef _SKIP_DOC
__cyos 1008
#endif
void* realloc(void* ptr_memory_block, size_t new_size);

/**
 *  @ingroup memory_overview
 *  Frees a memory block.
 *  @param ptr_memory_block a pointer to previously allocated memory block
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define  MAX_STR_LEN     256
 *       ...
 *       char* sz_string;
 *       ...
 *       sz_string = (char*)malloc( MAX_STR_LEN );
 *       ...
 *       free(sz_string);
 *       ...
 *  @endcode
 *  @see malloc, calloc, realloc.
 */
#ifndef _SKIP_DOC
__cyos 610
#endif
void free(void* ptr_memory_block);

/**
 *  @ingroup memory_overview
 *  Returns low memory alarm. The system sets it automatically when the dynamic memory pool drops 
 *  dangerously low. All applications should check dynamic memory. When TRUE is returned, the application 
 *  should free up all possible dynamically allocated memory.
 *  When it is set to TRUE.
 *  @return TRUE if memory is low
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if( is_low_memory() )
 *       {  
 *         TRACE( "Warning! Low memory." );
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 816
#endif
bool is_low_memory(void);

/**
 *  @ingroup memory_overview
 *  Returns safety pool size. 
 *  @return maximum size of the memory pool that can be allocated.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if ( get_safety_pool_size() < 1024 )
 *       {
 *         TRACE( "Error! Not enough memory." );
 *         return -1;
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 717
#endif
size_t get_safety_pool_size(void);

/**
 *  @ingroup memory_overview
 *  Sets buffers to a specified character.
 *  @param ptr_buffer a pointer to destination.
 *  @param value a character to set.
 *  @param size number of characters.
 *  @return a pointer to destination.
 *  @code
 *       #include <cybiko.h>
 *       #define MAX_PLAYER      10
 *       ...
 *       struct Point
 *       {
 *         int x;
 *         int y;
 *       } players_coordinates[MAX_PLAYER];
 *       ...
 *       //  Initializes players' coordinates.
 *       memset( players_coordinates, 0, MAX_PLAYER*sizeof(struct Point) );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 884
#endif
void* memset(void* ptr_buffer, int value, size_t size);

/**
 *  @ingroup memory_overview
 *  Copies characters between buffers.
 *  If the source and destination overlap, this function does not ensure 
 *  that the original source bytes in the overlapping region are copied 
 *  before being overwritten. Use memmove() to handle overlapping regions.
 *  @param ptr_buffer a pointer to destination.
 *  @param ptr_source a pointer to source.
 *  @param size number of characters to copy.
 *  @return a pointer to destination.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* sz_base_string = "Hello,";
 *       const char* sz_name = "word";  
 *       char* sz_string[20]; 
 *       ... 
 *       memcpy( sz_string, 
 *               sz_base_string, 
 *               strlen( sz_base_string ) );
 *       memcpy( sz_string + strlen( sz_string ), 
 *               sz_name, 
 *               strlen( sz_name ) );
 *       memcpy( sz_string + strlen( sz_string ),
 *               "!",
 *               2 );
 *       //  Traces Hello, world!
 *       TRACE( "Result: %s", sz_string );
 *       ...
 *  @endcode
 *  @see memmove.
 */
#ifndef _SKIP_DOC
__cyos 882
#endif
void* memcpy(void* ptr_buffer, void* ptr_source, size_t size);

/**
 *  @ingroup memory_overview
 *  Moves one buffer to another.
 *  If some regions of the source area and the destination overlap, memmove()
 *  ensures that the original source bytes in the overlapping region are 
 *  copied before being overwritten.
 *  @param ptr_buffer a pointer to destination.
 *  @param ptr_source a pointer to source.
 *  @param size number of characters to copy.
 *  @return a pointer to destination.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* sz_string = "The funny black cat.";
 *       ...
 *       memmove( sz_string + 4, sz_string + 10, 6 );
 *       //  Traces "The black black cat."
 *       TRACE( "Result: %s", string );
 *  @endcode
 *  @see memcpy.
 */
#ifndef _SKIP_DOC
__cyos 883
#endif
void* memmove(void* dst,void* src,size_t size);

/**
 *  @ingroup memory_overview
 *  Compares characters in two buffers.
 *  @param ptr_buff_1 a pointer to first buffer.
 *  @param ptr_buff_2 a pointer to second buffer.
 *  @param size number of characters to compare.
 *  @return 0 if ptr_buff_1 identical to ptr_buff_2. Positive value if
 *  ptr_buff_1 greater(in lexical meaning) than ptr_buff_2. 
 *  Negative value if ptr_buff_1 greater(in lexical meaning) than ptr_buff_2.
 *  @code
 *       char* string_1 = "AAAAAAAAAB";
 *       char* string_2 = "AAAAAAAAAA";
 *       ...
 *       if( !memcmp( string_1, string_2, 10 ) )
 *       {
 *         //  Buffers are not identical, because 
 *         //  string_1 is greater then string_2 in lexical meaning.
 *         ...
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 881
#endif
int memcmp(void* ptr_buff_1,void* ptr_buff_2,size_t size);

/**
 *  @ingroup string_overview
 *  Finds a substring.
 *  @param sz_str_1 A null-terminated string to search
 *  @param sz_str_2 A null-terminated string to search for
 *  @return A pointer to the first occurrence of sz_str_2 in sz_str_1
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char* sz_first_occurrence; 
 *       char* sz_string = "The old string !";
 *       ...                               
 *       if( ( sz_first_occurrence = strstr( string, "old" ) ) )
 *       {
 *         memcpy( sz_first_occurrence, "new", 3 );
 *       }
 *       //  Traces "The new string !".
 *       TRACE( "Result: %s", sz_string );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1218
#endif
char* strstr(char* sz_str_1,char* sz_str_2);

/**
 *  @ingroup string_overview
 *  Copies a string, including the terminating null character to the 
 *  destination buffer. No overflow checking is performed when strings are 
 *  copied or appended. The behavior of strcpy() is undefined if the source 
 *  and destination strings overlap.
 *  @param sz_destination A destination buffer
 *  @param sz_source A null-terminated source string
 *  @return The destination string
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* sz_name = "world";
 *       char sz_string[20];
 *       ...
 *       strcpy( sz_string, "Hello," );
 *       strcat( sz_string, sz_name );
 *       strcat( sz_string, "!" );
 *       ...
 *       //  Traces "Hello, world!". 
 *       TRACE( "Result: %s", sz_string);
 *       ...
 *  @endcode
 *  @see strncpy.
 */
#ifndef _SKIP_DOC
__cyos 1209
#endif
char* strcpy(char* sz_destination, char* sz_source);

/**
 *  @ingroup string_overview
 *  Copies the initial count characters of the source string to the destination 
 *  buffer. If the count is less than or equal to the length of the source string, 
 *  only count - 1 characters are copied from the source string, and a null 
 *  character is appended automatically to the copied string. If the count is 
 *  greater than the length of the source string, the destination string is padded 
 *  with null characters up to the length count. The behavior of strncpy is 
 *  undefined if the source and destination strings overlap.
 *  @param sz_destination A destination buffer
 *  @param sz_source A null-terminated source string
 *  @param count Number of characters to be copied
 *  @return The destination string
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* sz_string = "My nickname is CyCat. I love Cybiko!";
 *       char sz_nicname[NICKNAMESIZE + 1];
 *       ...
 *       strncpy( sz_string + 15, sz_nicname, NICKNAMESIZE + 1);
 *       //  Traces "His nickname is CyCat". 
 *       TRACE("His nickname is %s", sz_nicname);
 *       ...
 *  @endcode
 *  @see strcpy.
 */
#ifndef _SKIP_DOC
__cyos 1215
#endif
int strncpy( char* sz_destination, char* sz_source, size_t count );

/**
 *  @ingroup string_overview
 *  Compares characters from two strings.
 *  @param sz_string_1 First null-terminated string to compare
 *  @param sz_string_2 Second null-terminated string to compare
 *  @return 0 if sz_string_1 is identical to sz_string_2. Positive value if
 *  sz_string_1 is greater (in lexical meaning) than sz_string_2. 
 *  Negative value if sz_string_1 is greater (in lexical meaning) than sz_string_2.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char* string_1 = "AAAAAAAAAB";
 *       char* string_2 = "AAAAAAAAAA";
 *       ...
 *       if( !strcmp( string_1, string_2 ) )
 *       {
 *         //  Strings are not identical, because 
 *         //  string_1 is greater then string_2 in lexical meaning.
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see strncmp.
 */
#ifndef _SKIP_DOC
__cyos 1208
#endif
int strcmp(char* sz_string_1, char* sz_string_2);

/**
 *  @ingroup string_overview
 *  Compares characters from two strings.
 *  @param sz_string_1 First null-terminated string to compare
 *  @param sz_string_2 Second null-terminated string to compare
 *  @param count Number of characters to compare
 *  @return 0 if sz_string_1 is identical to sz_string_2. Positive value if
 *  sz_string_1 is greater (in lexical meaning) than sz_string_2. 
 *  Negative value if sz_string_1 is greater (in lexical meaning) than sz_string_2.
 *  @code
 *       char* string_1 = "AAAAAAAAAB";
 *       char* string_2 = "AAAAAAAAAA";
 *       ...
 *       if( strncmp( string_1, string_2, 6 ) )
 *       {
 *         //  Strings are identical, because we compare
 *         //  only the first 6 characters.
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see strcmp.
 */
#ifndef _SKIP_DOC
__cyos 1214
#endif
int strncmp(char* string_1, char* string_2, size_t count);

/**
 *  @ingroup string_overview
 *  Finds a character in a string.
 *  @param sz_source A null-terminated source string
 *  @param character A character to be located
 *  @return A pointer to the first occurrence of a character in a string the
 *  source string, or NULL if a character is not found
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* sz_string = "I have 2 Cybiko computers!";
 *       char* ptr_amount;
 *       ...
 *       if( ( ptr_amount = strchr( sz_string, '2' ) ) )
 *       {
 *         *ptr_amount = '6';
 *       }
 *       //  Traces "I have 6 Cybiko computers!".
 *       TRACE( sz_string );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1207
#endif
char* strchr(char* sz_source, char character);

/**
 *  @ingroup string_overview
 *  Scans a string for the last occurrence of a character.
 *  @param sz_source A pointer to a string.
 *  @param character A character to be located.
 *  @return A pointer to the last occurrence of character in string the
 *  source string, or NULL if character is not found.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* sz_string = "I have 2 Cybikos! My friend Peter has 1 Cybiko.";
 *       char* ptr_amount;
 *       ...
 *       if( ( ptr_amount = strchr( sz_string, '1' ) ) )
 *       {
 *         *ptr_amount = '2';
 *       }
 *       //  Traces "I have 2 Cybikos! My friend Peter has 1 Cybiko.".
 *       TRACE( sz_string );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1216
#endif
char* strrchr(char* sz_source, int character);

/**
 *  @ingroup string_overview
 *  Duplicate strings.
 *  It calls malloc() to allocate storage space for a copy of the 
 *  source string and then copies it to the allocated space.
 *  Don't forget to free allocated memory afterward.
 *  @param sz_string a pointer to the null-terminated source string.
 *  @return a pointer to the copy of the source string.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* sz_string = "Simple string";
 *       char* sz_copy;
 *       ... 
 *       sz_copy = strdup(sz_string);
 *       ...
 *       //  Traces "Original: Simple string".
 *       TRACE("Original: %s", sz_string);
 *       //  Traces "Copy: Simple string".
 *       TRACE("Copy: %s", sz_copy);
 *       ...
 *       free(sz_copy);
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1210
char* strdup(char* sz_string);
#endif

/**
 *  @ingroup string_overview
 *  Checks whether a string starts with a specified substring.
 *  @param sz_source A pointer to the null-terminated source string
 *  @param sz_start_string A pointer to the null-terminated string to 
 *  be located at the beginning of the source string
 *  @return TRUE if the source string starts with the specified string
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* name_list[6] = {
 *                                    "Dave",
 *                                    "Steve",
 *                                    "Jane",
 *                                    "Bill",
 *                                    "Robin",
 *                                    "John"
 *                                  };
 *       int index;
 *       ...
 *       //  Traces:         
 *       //          "Names beginning with J:"
 *       //          "Jane"
 *       //          "John"  
 *       TRACE("Names beginning with J:");
 *       for( index = 0; index < 6 ; index ++)
 *       {
 *         if( strstarts( name_list[index], "J" ) )
 *         {
 *           TRACE("%s", name_list[index]);   
 *         }
 *       }
 *       ...
 *  @endcode
 *  @see strends.
 */
#ifndef _SKIP_DOC
__cyos 1217
#endif
bool strstarts(char* sz_source, char* sz_start_string);

/**
 *  @ingroup string_overview
 *  Checks whether the string ends with a specified substring.
 *  @param sz_source A pointer to the null-terminated source string.
 *  @param sz_tail_string A pointer to the null-terminated string to 
 *  be located at the end of the source string.
 *  @return TRUE if the source string ends with the specified string.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* name_list[6] = {
 *                                    "Dave",
 *                                    "Steve",
 *                                    "Jane",
 *                                    "Bill",
 *                                    "Robin",
 *                                    "John"
 *                                  };
 *       int index;
 *       ...
 *       //  Traces:         
 *       //          "Names ending with e:"
 *       //          "Dave"
 *       //          "Steve"  
 *       //          "Jane"
 *       TRACE("Names ending with e:");
 *       for( index = 0; index < 6 ; index ++)
 *       {
 *         if( strends( name_list[index], "e" ) )
 *         {
 *           TRACE("%s", name_list[index]);   
 *         }
 *       }
 *       ...
 *  @endcode
 *  @see strstarts.
 */
#ifndef _SKIP_DOC
__cyos 1211
#endif
bool strends(char* sz_source, char* sz_tail_string);

/**
 *  @ingroup string_overview
 *  Checks wildcard match of the string to the pattern.
 *  This function calculates a fast recursive pattern match using character
 *  '*' to match any characters with values of zero or more, and '?' to match exactly
 *  any single character. All other characters require the exact same ones as the
 *  testing string. Since this function is recursive, it does stack checking to 
 *  prevent uncaught stack overflow. Checks wildcard match of a string 
 *  to a pattern.
 *  @param sz_pattern A pointer to the pattern to check match
 *  @param sz_source A pointer to the source string
 *  @return TRUE if the pattern matches the source string, otherwise FALSE.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char* name_list[6] = {
 *                                    "Dave",
 *                                    "Steve",
 *                                    "Jane",
 *                                    "Bill",
 *                                    "Robin",
 *                                    "John"
 *                                  };
 *       int index;
 *       ...
 *       //  Traces:         
 *       //          "Names with the "?a*" pattern:"
 *       //          "Dave"
 *       //          "Jane"
 *       TRACE("Names with the \"?a*\" pattern:");
 *       for( index = 0; index < 6 ; index ++)
 *       {
 *         if( strmatch( "?a*", name_list[index] ) )
 *         {
 *           TRACE("%s", name_list[index]);   
 *         }
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1213
#endif
bool strmatch(char* sz_pattern, char* sz_source);

/**
 *  @ingroup io_overview
 *  Writes formatted data to a string.
 *  This function works almost exactly like standard C one, but it does not
 *  support all format tags and specifiers. If you've never heard about
 *  printf routines, here's the general idea: you specify a format line with
 *  some text that may contain format tags. For each format tag found, sprintf()
 *  takes a parameter for the stack. This method does not allow a compile time check,
 *  but it does give you enormous flexibility and effectiveness.
 *  So, you code it this way:
 *  @code
 *  sprintf( buffer_ptr, "My number is %-10d",123);
 *  @endcode
 *  Here '%-10d' is a format tag that says, "there is a decimal value to be
 *  printed with left justification in the 10 characters field."
 *
 *
 *  The following format specifiers are allowed:
 *
 *  <b>%[-][0][<size>][{l|L}]{i|d|x|X|s|c|%}</b>
 * 
 *  Their meaning is standard:
 * 
 *  @arg leading '-' sets left justification (the default is right)
 * 
 *  @arg if leading zero is specified, leading zeroes will be printed for
 *  numbers.
 *
 *  @arg optional <size> (decimal number) sets the size of the output field
 *
 *  @arg  optional 'l' or 'L' prefix means that number is \a long
 * 
 *  @arg 'i' and 'd' causes a decimal integer to be printed
 *
 *  @arg 'x' and 'X' causes a hexadecimal number (with small or capital letters,
 *  respectively) to be printed
 *
 *  @arg 's' refers to a character string (might be UNICODE or bytes)
 *
 *  @arg 'c' means single character
 *
 *  @arg '%' means "just put out a percent character without taking arguments."
 *
 *  All other characters in the format line will be just printed out as is.
 *  Remember, it is up to you to handle the arguments list and the format line in the
 *  correct correspondence.
 *  @param ptr_buffer a storage location for output.
 *  @param sz_format format-control string.
 *  @argument optional arguments.
 *  @return the number of characters written, not including the 
 *  terminating null character, or a negative value if an output error occurs.
 *  @code
 *       #include <cybiko.h>
 *       #define FIRST_PLAYER_WINS            0
 *       #define SECOND_PLAYER_WINS           1
 *       ...
 *       const char* sz_format = "%s player WINS!";
 *       const char* sz_player_name[2] = {
 *                                         "First",
 *                                         "Second" 
 *                                       };
 *       int game_result;
 *       char sz_result_string[24];
 *       ...
 *       sprintf( sz_result_string, sz_format, sz_player_name[ game_result ] );        
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1170
int sprintf(char* ptr_buffer, char* sz_format,...);
#else
int sprintf(char* ptr_buffer, char* sz_format [, argument]... );
#endif

/**
 *  @ingroup io_overview
 *  Writes formatted output using a pointer to a list of arguments.
 *  @param ptr_buffer a storage location for output.
 *  @param sz_format format-control string(see sprintf() for the full 
 *  format description).
 *  @param argv_list a pointer to a list of arguments.
 *  @return the number of characters written, not including the 
 *  terminating null character, or a negative value if an output error occurs.
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      //  Logs errors to the file.
 *      void trace2logfile(char* sz_format, ...)
 *      {
 *        va_list parameters;
 *        char sz_buffer[64];
 *        struct FileOutput* ptr_log_file; 
 *
 *        //  Formats string.
 *        va_start( parameters, sz_format );
 *        vsprintf( sz_buffer, sz_format, parameters );  
 *        va_end( parameters );
 *
 *        //  Writes it to file "error.log".
 *        ptr_log_file = (struct FileOutput*)malloc( sizeof( struct FileOutput ) );
 *        FileOutput_ctor_Ex( ptr_log_file, "error.log", TRUE );
 *        FileOutput_seekp( ptr_log_file, 0, SEEK_END );
 *        FileOutput_write( ptr_log_file, sz_buffer, strlen(sz_buffer) );
 *        FileOutput_dtor( ptr_log_file, FREE_MEMORY );
 *      }
 *      ...
 *  @endcode
 *  @see va_start, va_end, va_arg.
 */
#ifndef _SKIP_DOC
__cyos 1272
#endif
int vsprintf(char* ptr_buffer, char* sz_format, va_list argv_list); 

/**
 *  @ingroup string_overview
 *  Checks whether string contains '*' or '?' characters.
 *  @param sz_source A source string
 *  @return TRUE if source string contains '*' or '?' characters,
 *  otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       bool open_file( char* sz_name )
 *       {
 *         if( is_pattern( sz_name ) ) 
 *         {
 *           //  Searches file using pattern. 
 *           ...
 *         }
 *         //  Opens file.
 *         ... 
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 817
#endif
bool is_pattern(char* sz_source);

/**
 *  @ingroup string_overview
 *  Skips white spaces.
 *  @param sz_source A source string
 *  @return The source string without white spaces.
 *  @code
 *        #include <cywin.h>
 *        ...
 *        char test_string[ 20 ];
 *        ...
 *        strcpy( test_string, "Any string" );
 *        //  Outputs: Any string
 *        TRACE( "%s", test_string );
 *        //  Outputs: Anystring
 *        TRACE( "%s", skipws( test_string ) );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1166
#endif
char* skipws( char* sz_source );

/**
 *  @ingroup convertion_overview
 *  Converts ASCII decimal into long.
 *  @param sz_source a string to be converted.
 *  @return long value produced by interpreting the input characters 
 *  as a number.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       long amount;
 *       struct cDialog dialog;
 *       struct module_t main_module;
 *       char sz_amount[11];
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cDialog_ctor( &dialog, 
 *                     "Apples", 
 *                     "Enter amount:", 
 *                     mbOk | mbCancel | mbEdit, 
 *                     10, 
 *                     main_module.m_process );
 *       if( cDialog_ShowModal( &dialog ) == mrOk )
 *       {
 *         cDialog_GetEditText( &dialog, sz_amount );
 *       }
 *       cDialog_dtor( &dialog, LEAVE_MEMORY );
 *       amount = atoul( sz_amount );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 372
#endif
long atoul(char* sz_source);

/**
 *  @ingroup convertion_overview
 *  Converts some ASCII number into long.
 *  First, this function skips white spaces (including the spaces at the ends of lines), 
 *  then tries to detect the base by checking the beginning of the number. 
 *  It understands these hexadecimal prefixes:
 *  'x<hex-val> 0x<hex-val> H'<hex-val> h'<hex-val> -<dec-val> <dec-val>'
 *  where hex-val is some set of hexadecimal digits (0123456789ABCDEFabcdef), 
 *  val - decimal value.
 *  @param sz_source a string to be converted.
 *  @return long value produced by interpreting the input characters 
 *  as a number.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       long device_id;
 *       struct cDialog dialog;
 *       struct module_t main_module;
 *       char sz_device_id[11];
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cDialog_ctor( &dialog, 
 *                     "Connection property", 
 *                     "Enter device id:", 
 *                     mbOk | mbCancel | mbEdit, 
 *                     10, 
 *                     main_module.m_process );
 *       if( cDialog_ShowModal( &dialog ) == mrOk )
 *       {
 *         cDialog_GetEditText( &dialog, sz_device_id );
 *       }
 *       cDialog_dtor( &dialog, LEAVE_MEMORY );
 *       device_id = xtoi( sz_device_id );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1299
#endif
long xtoi(char* sz_source);

/**
 *  @ingroup convertion_overview
 *  Converts hexadecimal ASCII into long.
 *  @param sz_source a string to be converted.
 *  @return long value produced by interpreting the input characters as a 
 *  hexadecimal value.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       long device_id;
 *       struct cDialog dialog;
 *       struct module_t main_module;
 *       char sz_device_id[11];
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cDialog_ctor( &dialog, 
 *                     "Connection property", 
 *                     "Enter device id (in hexadecimal form):", 
 *                     mbOk | mbCancel | mbEdit, 
 *                     10, 
 *                     main_module.m_process );
 *       if( cDialog_ShowModal( &dialog ) == mrOk )
 *       {
 *         cDialog_GetEditText( &dialog, sz_device_id );
 *       }
 *       cDialog_dtor( &dialog, LEAVE_MEMORY );
 *       device_id = htoul( sz_device_id );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 765
#endif
long htoul(char* sz_source);

#ifndef _SKIP_DOC
__cyos 938
int parse_int(char** ptr);

__cyos 356
int _rand(void);

__cyos 982
int rand(void);
#endif  //  #ifndef _SKIP_DOC

/**
 *  @ingroup math_group
 *  Initializes a pseudo-random number generator.
 *  @param seed Seed for pseudo-random number generation
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int nValue;
 *       ...
 *       srand( clock() );
 *       ...
 *       nValue = random( 100 );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1171
#endif
void srand(int seed);

/**
 *  @ingroup multithreads_overview
 *  Exits current the thread with some code.
 *  @param exit_code Exit code.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       else
 *       {
 *         exit( 1 );
 *       }
 *       ...
 *       MSequence_dtor(  &intro_music );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 562
#endif
void exit(int exit_code);

/**
 *  @ingroup misc_overview
 *  Returns the version of CYBIKO hardware.
 *  @return version of CYBIKO hardware.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE("Your hardware version is %ld", get_hardware_version() );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1350
#endif
long get_hardware_version(void);

/**
 *  @ingroup string_overview
 *  Returns the length of a string, not including the terminating 
 *  null character. 
 *  @param sz_source A null-terminated source string
 *  @return The number of characters in the string
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char* sz_string;
 *       char* sz_source_string;
 *       ...
 *       sz_string = (char*)malloc( 10 );
 *       ...
 *       if( strlen( sz_source_string ) <= 10)
 *       {
 *         sz_string = (char*)realloc( sz_string, 
 *                                     strlen( sz_source_string ) + 1 );
 *       }
 *       ...
 *       free(sz_string);
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1212
#endif
size_t strlen(char* sz_source);

/**
 *  @ingroup string_overview
 *  Appends one string to another and terminates the resulting string with 
 *  a null character. The initial character of the source string overwrites the 
 *  terminating null character of the destination string. No overflow checking is 
 *  performed when strings are copied or appended. The behavior of strcat is 
 *  undefined if the source and destination strings overlap.
 *  @param sz_destination A null-terminated destination string.
 *  @param sz_source A null-terminated source string.
 *  @return The destination string
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       const char sz_name = "world";
 *       char sz_string[20];
 *       ...
 *       strcpy( sz_string, "Hello," );
 *       strcat( sz_string, sz_name );
 *       strcat( sz_string, "!" );
 *       ...
 *       //  Traces "Hello, world!". 
 *       TRACE( "Result: %s", sz_string);
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1206
#endif
char* strcat(char* sz_destination, char* sz_source);

#endif  //  #ifndef __CYBIKO_H_INCLUDED__
