/** 
 *  @file finder.h 
 *  Hook to finder object.
 *
 *  Note1: name stored in finder.cf[some].cf_fk.f_nick is a real user nickname.
 *  If some visible Cybiko has the same nickname (for instance, "Cy"),
 *  to differ them, you will see one of them in People, Chat
 *  in corrupted form, as "Cy1".
 *  You can get this name form using function
 *  @code
 *       Finder_create_name( struct Finder* ptr_finder,
 *                           char* sz_visible_name,
 *                           struct cyfolk_t* person_info )
 *  @endcode
 *
 *  Note2: You MUST use finder_mutex for access division to Finder fields.
 *
 *  Example show how you can get user name on CyId:
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char* test_get_name( char* sz_nickname, cyid_t id )
 *       {
 *         char index;
 *         Mutex_lock( &finder.finder_mutex, 0 );
 *
 *         index = Finder_find_active( &finder, id );
 *         if( index == NOT_FOUND )
 *         {
 *           strcpy ( sz_nickname, "unknown" );
 *         }
 *         else
 *         {
 *            Finder_create_name (&finder, sz_nickname, &finder.cf[index]);
 *         }
 *
 *         Mutex_unlock(&finder.finder_mutex);
 *         return name;
 *      }
 *  @endcode
 */

#ifndef __FINDER_H_INCLUDED__
#define __FINDER_H_INCLUDED__

/**
 *  Maximum length of the nickname.
 * @ingroup finder_related_group
 */
#define NICKNAMESIZE            8    

/**
 *  Maximum length of the name. 
 * @ingroup finder_related_group
 */
#define NAMESIZE                15   

/**
 *  Maximum length of the Cypet name. 
 * @ingroup finder_related_group
 */
#define CYPETNAMESIZE           7    

/**
 *  Person with the specified parameter not found.
 * @ingroup finder_related_group
 */
#define NOT_FOUND               0xFF 

/**
 *  Maximum length of the ICQ number.
 * @ingroup finder_related_group
 */
#define STR_ICQ_MAX_LENGTH      10

/**
 *  ID of the message sent to the application when any device
 *  appears or disappears from the environment
 * @ingroup messaging_group
 */
#define MSG_USER_FOLK           0x1001

/**
 *  @brief Describes the " You&me/Business Card " information.
 *  @ingroup finder_related_group
 *  Describes the "You&me/Business Card" information.
 *  @see finder_related_group
 */
struct info_t
{
  /**
   *  Real first name.  
   */
  char i_first_name[NAMESIZE + 1];  

  /**
   *  Real last name. 
   */
  char i_last_name[NAMESIZE + 1];   

  /**
   *  Phone number.
   */
  char i_phone[21];                 

  /**
   *  E-mail address.
   */
  char i_mail[31];                  

  /**
   *  Address.
   */
  char i_addr[31];                  

  /**
   *  Other contacts.
   */
  char i_other[41];                 

  /**
   *  Birthday, day of the month. 
   */
  char i_bday;                      

  /**
   *  Birthday, month. 
   */
  char i_bmonth;                  

  /**
   *  Birthday, year.  
   */
  char i_byear;                   

  /**
   *  First name visibility flag.
   */
  char i_first_name_visible;       

  /**
   *  Last name visibility flag.
   */
  char i_last_name_visible;        

  /**
   *  Phone number visibility flag.
   */
  char i_phone_visible;             

  /**
   *  E-mail address visibility flag.
   */
  char i_mail_visible;              

  /**
   *  Address visibility flag.
   */
  char i_addr_visible;             

  /**
   *  Other contacts visibility flag. 
   */
  char i_other_visible;            
};

/**
 *  @brief Describes the " You&me/About Me " and " You&me/About You " information.
 *  @ingroup finder_related_group
 *  Describes the " You&me/About Me " " You&me/About You " information.
 *  @see finder_related_group
 */
struct folk_t 
{
#ifndef _SKIP_DOC
  int f_cypet_gameday;                    
  char f_cypet_name[CYPETNAMESIZE];       
  char f_cypet_info;                      
#endif
  /**
   *  Nickname.
   */
  char f_nick[NICKNAMESIZE + 1];  

  /**
   *  Gender. 0:male, 1:female.
   */
  char f_gender;                  

  /**
   *  Age visibility flag.
   */
  char f_age_visible;             

  /**
   *  Age.
   */
  char f_age;                

  /**
   *  Height.
   */
  char f_height;                  

  /**
   *  Weight.
   */
  char f_weight;                 

  /**
   *  Purpose to find friend.   
   *  0:Business, 1:Romance, 2:Sports, 3:Chat, 4:Other
   */
  char f_purpose;                

  /**
   *  Purpose visibility flag.
   */
  char f_purpose_visible;         

  /**
   *  Height visibility flag.
   */
  char f_height_visible;          

  /**
   *  Weight visibility flag.
   */
  char f_weight_visible;          

  /**
   *  Hobby 1 visibility flag.
   */
  char f_hobby1_visible;          

  /**
   *  Hobby 2 visibility flag.
   */
  char f_hobby2_visible;          

  /**
   *  Hobby 3 visibility flag.
   */
  char f_hobby3_visible;          

  /**
   *  Hobby 1.
   *  1:no, 2:Cybiko, 3:Soccer, 4:Chess, 5:Rollerblading, 6:Skateboarding,
   *  7:Skydiving, 8:BMX, 9:Music, 10:Gaming, 11:Dance, 12:Internet,
   *  13:Computer, 14:Sports, 15:Arts, 16:Science/Tech, 17:Travel,18:Mountain Biking,  
   *  19:Books, 20:Movies, 21:Skiing, 22:Girls, 23:Boys, 24:does not matter 
   */
  char f_hobby1;                 

  /**
   *  Hobby 2.
   *  1:no, 2:Cybiko, 3:Soccer, 4:Chess, 5:Rollerblading, 6:Skateboarding,
   *  7:Skydiving, 8:BMX, 9:Music, 10:Gaming, 11:Dance, 12:Internet,
   *  13:Computer, 14:Sports, 15:Arts, 16:Science/Tech, 17:Travel,18:Mountain Biking,  
   *  19:Books, 20:Movies, 21:Skiing, 22:Girls, 23:Boys, 24:does not matter 
   */
 char f_hobby2;

  /**
   *  Hobby 3.
   *  1:no, 2:Cybiko, 3:Soccer, 4:Chess, 5:Rollerblading, 6:Skateboarding,
   *  7:Skydiving, 8:BMX, 9:Music, 10:Gaming, 11:Dance, 12:Internet,
   *  13:Computer, 14:Sports, 15:Arts, 16:Science/Tech, 17:Travel,18:Mountain Biking,  
   *  19:Books, 20:Movies, 21:Skiing, 22:Girls, 23:Boys, 24:does not matter 
   */
  char f_hobby3;

  /**
   *  secret field 1.
   */
  char f_secret1[NICKNAMESIZE + 1];      

  /**
   *  secret field 2.
   */
  char f_secret2[NICKNAMESIZE + 1];       

  /**
   *  secret field 3.
   */
  char f_secret3[NICKNAMESIZE + 1];      

  /**
   *  Gender of friend to seek. 0:male, 1:female, 2:does not matter.
   */
  char f_seek_gender;                    

  /**
   *  Importance of gender of friend to seek(0 - 5).
   */
  char f_seek_gender_degree;              

  /**
   *  Minimal age of friend to seek.
   */
  char f_min_age;                         

  /**
   *  Maximal age of friend to seek.
   */
  char f_max_age;                         

  /**
   *  Importance of age of friend to seek(0 - 5).
   */
  char f_seek_age_degree;                

  /**
   *  Minimal height of friend to seek.
   */
  char f_min_height;                      

  /**
   *  Maximal height of friend to seek.
   */
  char f_max_height;                      

  /**
   *  Importance of height of friend to seek(0 - 5).
   */
  char f_seek_height_degree;            

  /**
   *  Minimal weight of friend to seek. 
   */
  char f_min_weight;                      

  /**
   *  Maximal weight of friend to seek.
   */
  char f_max_weight;                      

  /**
   *  Importance of weight of friend to seek(0 - 5).
   */
  char f_seek_weight_degree;              

  /**
   *  Purpose to find friend.   
   *  0:Business, 1:Romance, 2:Sports, 3:Chat, 4:Other
   */
  char f_seek_purpose;                    

  /**
   *  Importance of purpose of friend to seek(0 - 5).
   */
  char f_seek_purpose_degree;             

  /**
   *  Hobby 1 of friend to seek.
   *  1:no, 2:Cybiko, 3:Soccer, 4:Chess, 5:Rollerblading, 6:Skateboarding,
   *  7:Skydiving, 8:BMX, 9:Music, 10:Gaming, 11:Dance, 12:Internet,
   *  13:Computer, 14:Sports, 15:Arts, 16:Science/Tech, 17:Travel,18:Mountain Biking,  
   *  19:Books, 20:Movies, 21:Skiing, 22:Girls, 23:Boys, 24:does not matter 
   */
  char f_seek_hobby1;                     

  /**
   *  Hobby 2 of friend to seek.
   *  1:no, 2:Cybiko, 3:Soccer, 4:Chess, 5:Rollerblading, 6:Skateboarding,
   *  7:Skydiving, 8:BMX, 9:Music, 10:Gaming, 11:Dance, 12:Internet,
   *  13:Computer, 14:Sports, 15:Arts, 16:Science/Tech, 17:Travel,18:Mountain Biking,  
   *  19:Books, 20:Movies, 21:Skiing, 22:Girls, 23:Boys, 24:does not matter 
   */
  char f_seek_hobby2;                     

  /**
   *  Hobby 3 of friend to seek.
   *  1:no, 2:Cybiko, 3:Soccer, 4:Chess, 5:Rollerblading, 6:Skateboarding,
   *  7:Skydiving, 8:BMX, 9:Music, 10:Gaming, 11:Dance, 12:Internet,
   *  13:Computer, 14:Sports, 15:Arts, 16:Science/Tech, 17:Travel,18:Mountain Biking,  
   *  19:Books, 20:Movies, 21:Skiing, 22:Girls, 23:Boys, 24:does not matter 
   */
  char f_seek_hobby3;                     

  /**
   *  Importance of hobby 1 of friend to seek(0 - 5).
   */
  char f_seek_hobby1_degree;              

  /**
   *  Importance of hobby 2 of friend to seek(0 - 5).
   */
  char f_seek_hobby2_degree;              

  /**
   *  Importance of hobby 3 of friend to seek(0 - 5).
   */
  char f_seek_hobby3_degree;              

  /**
   *  Importance of secret field 1 of friend to seek(0 - 5).
   */
  char f_seek_secret1_degree;             

  /**
   *  Importance of secret field 2 of friend to seek(0 - 5).
   */
  char f_seek_secret2_degree;             

  /**
   *  Importance of secret field 3 of friend to seek(0 - 5).
   */
  char f_seek_secret3_degree;             

  /**
   *  Secret field 1 of friend to seek.
   */
  char f_seek_secret1[NICKNAMESIZE + 1];

  /**
   *  Secret field 2 of friend to seek.  
   */
  char f_seek_secret2[NICKNAMESIZE + 1];

  /**
   *  Secret field 3 of friend to seek.  
   */
  char f_seek_secret3[NICKNAMESIZE + 1];  
};

/**
 *  @brief Additional information about a person in Finder's list.
 *  @ingroup finder_related_group
 *  Additional information about a person in Finder's list.
 *  @see finder_related_group
 */
struct nfolk_t 
{
#ifndef _SKIP_DOC
  int f_cypet_gameday;                    
  char f_cypet_name[CYPETNAMESIZE];       
  char f_cypet_info;                      
#endif
  /**
   *  Nickname.
   */
  char f_nick[NICKNAMESIZE + 1];          

  /**
   *  Gender.
   */
  char f_gender;                          

  /**
   *  Age visibility flag.
   */
  char f_age_visible;                     

  /**
   *  Age.
   */
  char f_age;                             
};

/**
 *  @brief describes a person on the Finder's list.
 *  @ingroup finder_related_group
 *  Describes a person in the Finder's list.
 *  @see finder_related_group
 */
struct cyfolk_t 
{
  /**
   *  Cy ID of the user's Cybiko device.
   */
  long cf_id;                             

  /**
   *  Additional information about the person listed.
   */
  struct nfolk_t cf_fk;                   

  /**
   *  The mean match coefficient of the listed person - and the device's owner.
   */
  char cf_my_match;                       

  /**
   *  Mean match coefficient of person and device owner.
   */
  char cf_match;                          
#ifndef _SKIP_DOC
  char cf_selected;                       
  char cf_status;                         
#endif
   /**
    *  Number of people with the same nickname in the Finder's list.
    */
   char cf_name_ind;                      
#ifndef _SKIP_DOC
  char cyfolk_PADDING;
#endif
};

#ifndef _SKIP_DOC
struct save_data_t
{
  struct info_t save_mi;                  
  struct folk_t save_mf;                  
};
#endif

/**
 *  @brief Finder is a resident program; that is, it is always in memory. It's main purpose is to gather information about the Cy-environment.
 *  @ingroup finder_related_group
 *  Finder is a resident program, that is, it is always in memory.
 *  It's main purpose is to gather information about the Cy-environment.
 *  If another Cybiko appears within the "field of vision"
 *  (if we get any message from it), the Finder begins the acquaintance procedure.
 *  It sends the  identification information "You&Me/About me" and "You&Me/About you",
 *  and requests the same information from the unknown Cybiko.
 *  Then the Finder calculates the match coefficients (hearts).
 *  If our common coefficient is equal to 3 hearts,
 *  the Finder creates a dialog, informing the user of our Cy-friend's appearance.
 *  The Finder also grants information about other users in the vicinity to other programs, including
 *  People and Chat.
 *
 *  The finder can also tell you other information.
 *  The total number of active people is stored in the field finder.howmany_around.
 *  The total number of unknown users is stored in howmany_unknown. 
 *  The "cf" field holds an array of records, each describing some aspects of other Cybikos:
 *  Cyid, nickname, gender, age etc.
 *  @see finder_related_group
 */
struct Finder: public Process
{
#ifndef _SKIP_DOC
  char sort_method;                       
  bool i_am_active;                       
  bool my_folk_on_flash;                  
#endif
  /**
   *  Mutually exclusive access synchronization object. 
   */
  struct Mutex finder_mutex;              
#ifndef _SKIP_DOC
  struct save_data_t save_data;           
  char marks[5];                          
#endif
  /**
   *  Number of people around.
   */
  char howmany_around;                    

  /**
   *  Number of boys around.
   */
  char howboys_around;                    

  /**
   *  Number of people around whose nicknames are "unknown".
   */
  char howmany_unknown;                   

  /**
   *  Number of people around who temporarily disappeared 
   *  from the finder's environment.
   */
  char howmany_disap;                     
#ifndef _SKIP_DOC
  bool start_chat_from_people;                         
  char start_chat_from_people_name[NICKNAMESIZE+1];    
  long start_chat_from_people_id;                             
#endif
  /**
   *  The user's own profile.
   */
  struct folk_t mf;        

  /** 
   *  The user's own business card.
   */
  struct info_t mi;        

  /**
   *  Array of records describing the people in the vicinity.
   */
  struct cyfolk_t* cf;     

#ifndef _SKIP_DOC
  struct Message* message;
};
#endif

/**
 *  Finds the active person's position in finder.cf array.
 *  @param ptr_finder The Finder object. MUST BE &finder!
 *  @param cyber_id Cy ID of the other person's device
 *  @return Position in the finder.cf, or NOT_FOUND, if the other user is not active
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int finder_list_index;
 *       int friend_candidates_number;
 *       cyid_t friend_candidates[10];
 *       char sz_friend_candidate_name[16];
 *       ...
 *       friend_candidates_number = Finder_get_best_ids(&finder,
 *                                                      friend_candidates,
 *                                                      10);
 *       TRACE("Friend candidates:");
 *       for(index = 0; index < friend_candidates_number; index ++)
 *       {
 *         finder_list_index = Finder_find_active(&finder,
 *                                                friend_candidates[index]);
 *         if(finder_list_index != NOT_FOUND)
 *         {
 *           Finder_create_name(&finder,
 *                              sz_friend_candidate_name,
 *                              &finder.cf[finder_list_index]);
 *           TRACE("%s", sz_friend_candidate_name);
 *         }
 *       }
 *       ...
 *  @endcode
 *  @see NOT_FOUND.
 */
#ifndef _SKIP_DOC
__cyos "finder" 7
#endif
char Finder_find_active(struct Finder* ptr_finder, cyid_t cyber_id);

/**
 *  Finds the position of the unknown person in the finder.cf array.
 *  @param ptr_finder The Finder object. MUST BE &finder!
 *  @param cyber_id The Cy ID of the other person's device
 *  @return Position in finder.cf or NOT_FOUND, if the other user is not found
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char* get_nickname_by_id( char* sz_nickname, cyid_t id )
 *       {
 *         char index;
 *         Mutex_lock( &finder.finder_mutex, 0 );
 *         if( ( index = Finder_find_active( &finder, id ) ) != NOT_FOUND )
 *         {
 *           //  Person is in the list of active people.
 *           Finder_create_name( &finder, sz_nickname, &finder.cf[index] );
 *         }
 *         else if( ( index = Finder_find_unknown( &finder, id ) ) != NOT_FOUND )
 *         {
 *           //  Person is in the list of unknown people.
 *           strcpy( sz_nickname, "unknown" );
 *         }
 *         else if( ( index = Finder_find_disap( &finder, id ) ) != NOT_FOUND)
 *         {
 *           //  Finds people in the list of people who temporarily
 *           //  disappeared from the finder's environment.
 *           Finder_create_name( &finder, sz_nickname, &finder.cf[index] );
 *         }
 *         else
 *         {
 *           //  We have never met the device with the specified Cy ID.
 *           strcpy( sz_nickname, "no info" );
 *         }
 *         Mutex_unlock( &finder.finder_mutex );
 *         return sz_nickname;
 *       }
 *       ...
 *  @endcode
 *  @see NOT_FOUND.
 */
#ifndef _SKIP_DOC
__cyos "finder" 8
#endif
char Finder_find_unknown(struct Finder* ptr_finder, cyid_t cyber_id);

/**
 *  Finds the position in the finder.cf array of the person who temporarily
 *  disappeared from the finder's environment.
 *  @param ptr_finder The Finder object. MUST BE &finder!
 *  @param cyber_id The Cy ID of the other person's device
 *  @return Position in finder.cf or NOT_FOUND, if the other user is not founded
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char* get_nickname_by_id( char* sz_nickname, cyid_t id )
 *       {
 *         char index;
 *         Mutex_lock( &finder.finder_mutex, 0 );
 *         if( ( index = Finder_find_active( &finder, id ) ) != NOT_FOUND )
 *         {
 *           //  Person is in the list of active people.
 *           Finder_create_name( &finder, sz_nickname, &finder.cf[index] );
 *         }
 *         else if( ( index = Finder_find_unknown( &finder, id ) ) != NOT_FOUND )
 *         {
 *           //  Person is in the list of unknown people.
 *           strcpy( sz_nickname, "unknown" );
 *         }
 *         else if( ( index = Finder_find_disap( &finder, id ) ) != NOT_FOUND)
 *         {
 *           //  Finds people in the list of people who temporarily
 *           //  disappeared from the finder's environment.
 *           Finder_create_name( &finder, sz_nickname, &finder.cf[index] );
 *         }
 *         else
 *         {
 *           //  We have never met the device with the specified Cy ID.
 *           strcpy( sz_nickname, "no info" );
 *         }
 *         Mutex_unlock( &finder.finder_mutex );
 *         return sz_nickname;
 *       }
 *       ...
 *  @endcode
 *  @see NOT_FOUND.
 */
#ifndef _SKIP_DOC
__cyos "finder" 9
#endif
char Finder_find_disap(struct Finder* ptr_finder, cyid_t cyber_id);

/**
 *  Calculates the user's age from his/her last birthday.
 *  @param ptr_finder The Finder object. MUST BE &finder!
 *  @param day Day of birth
 *  @param month Month of birth
 *  @param year Year of birth
 *  @return Current age
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE("Your age is: %d", Finder_age(&finder,
 *                                           finder.mi.i_bday,
 *                                           finder.mi.i_bmonth,
 *                                           finder.mi.i_byear));
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "finder" 12
#endif
char Finder_age(struct Finder* ptr_finder, char day, char month, char year);

/**
 *  Alters a real name to create a visible name form. For example, if two "visible" Cybikos
 *  have the same nickname ("Cy", for instance), the Finder can alter one of their names so the user can tell them apart
 *  (for example, showing one as "Cy" and the other as "Cy1").
 *  @param ptr_finder The Finder object. MUST BE &finder!
 *  @param sz_visible_name The buffer where the visible (altered) name will be placed
 *  @param person_info Person information about the other user whose Cybiko name is being altered
 *  @return The visible name.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int finder_list_index;
 *       int friend_candidates_number;
 *       cyid_t friend_candidates[10];
 *       char sz_friend_candidate_name[16];
 *       ...
 *       friend_candidates_number = Finder_get_best_ids(&finder,
 *                                                      friend_candidates,
 *                                                      10);
 *       TRACE("Friend candidates:");
 *       for(index = 0; index < friend_candidates_number; index ++)
 *       {
 *         finder_list_index = Finder_find_active(&finder,
 *                                                friend_candidates[index]);
 *         if(finder_list_index != NOT_FOUND)
 *         {
 *           Finder_create_name(&finder,
 *                              sz_friend_candidate_name,
 *                              &finder.cf[finder_list_index]);
 *           TRACE("%s", sz_friend_candidate_name);
 *         }
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "finder" 14
#endif
char* Finder_create_name(struct Finder* ptr_finder, char* sz_visible_name, struct cyfolk_t* person_info);

/**
 *  Gets the Cy IDs of the devices with the best match coefficient.
 *  @param ptr_finde The Finder object. MUST BE &finder!
 *  @param ptr_cyid_buff The buffer for Cy IDs
 *  @param number The desired number of Cy IDs.
 *  @return Number of Cy IDs stored in the buffer.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int finder_list_index;
 *       int friend_candidates_number;
 *       cyid_t friend_candidates[10];
 *       char sz_friend_candidate_name[16];
 *       ...
 *       friend_candidates_number = Finder_get_best_ids(&finder,
 *                                                      friend_candidates,
 *                                                      10);
 *       TRACE("Friend candidates:");
 *       for(index = 0; index < friend_candidates_number; index ++)
 *       {
 *         finder_list_index = Finder_find_active(&finder,
 *                                                friend_candidates[index]);
 *         if(finder_list_index != NOT_FOUND)
 *         {
 *           Finder_create_name(&finder,
 *                              sz_friend_candidate_name,
 *                              &finder.cf[finder_list_index]);
 *           TRACE("%s", sz_friend_candidate_name);
 *         }
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "finder" 30
#endif
int Finder_get_best_ids(struct Finder* ptr_finder, cyid_t* ptr_cyid_buff, int number);

/**
 *  Global Finder object.
 *  Must be used as the first parameter of all Finder_ functions.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Time current_time;
 *       ...
 *       Time_get_RTC(&current_time);
 *       if((current_time.month == finder.mi.i_bmonth)
 *          &&(current_time.day == finder.mi.i_bday))
 *       {
 *         TRACE("Happy birthday!");
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
#define ptr_Finder    (&finder)     

__cyos "finder" 31
#endif
struct Finder finder;

#ifdef _SKIP_DOC
};
#endif


/**
 *  @ingroup law_level_group
 *  Flag of notification applications about changes in the environment.
 *  If send_user_folk is FALSE MSG_USER_FOLK message will not notify 
 *  applications when some device (dis)appeared (from)in the environment.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       send_user_folk = TRUE;
 *       ...
 *       ptr_message = cWinApp_get_message(main_module.m_process,
 *                                         0,
 *                                         1,
 *                                         MSG_USER);
 *       if(ptr_message->msgid == MSG_USER_FOLK)
 *       {
 *         ...
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "finder" 10
#endif
bool send_user_folk;

/**
 *  @ingroup finder_related_group
 *  ICQ number of the device owner.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE("Your ICQ number is %s", own_icq);
 *       if(own_icq_visible)
 *       {
 *         TRACE("Your ICQ number is visible to people around.");
 *       }
 *       else
 *       {
 *         TRACE("Your ICQ number is not visible to people around.");
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "finder" 39
#endif
char own_icq[];

/**
 *  @ingroup finder_related_group
 *  ICQ number visibility flag.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE("Your ICQ number is %s", own_icq);
 *       if(own_icq_visible)
 *       {
 *         TRACE("Your ICQ number is visible to people around.");
 *       }
 *       else
 *       {
 *         TRACE("Your ICQ number is not visible to people around.");
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "finder" 40
#endif
char own_icq_visible;

/**
 *  @ingroup messaging_group
 *  Returns the number of other users in the vicinity, with maximum activity
 *  @return Number of Cybiko devices the finder detects
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE("Active people: %d", finder.howmany_around);
 *       TRACE("Around desktop: %d", howmany_around_desktop());
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "finder" 38
#endif
char howmany_around_desktop(void);

#endif // #ifndef __FINDER_H_INCLUDED__
         