/** @file form.h
 *  Hooks to cywin cText, cButton, cWinApp, cCustomForm, cDialog and cFrameForm.
 */

#ifndef __FORM_H_INCLUDE__
#define __FORM_H_INCLUDE__

#include "object.h"
#include "appgen.h"

/**
 *   @ingroup application_group
 *   @brief The cWinApp structure is your application object
 *
 *   The cWinApp structure is your application object. It implements cywin's functions, to work with
 *   user's objects such as: cBitmap, cEdit, cList, cItem, etc.
 *   <br>
 *   <br>
 *   You must make a pointer to it using the init_module function.
 *   @see application_group
 */
struct cWinApp : public AppGeneric
{
#ifndef _SKIP_DOC
  char cWinApp_Members_STUB[52];
};
#endif


//============================
//=                          =
//=     object cWinApp.      =
//=  Application template.   =
//=                          =
//============================

/**
 *  The Message-processing function.
 *  @param ptr_win_app A pointer to the cWinApp object
 *  @param ptr_message A pointer to the processed message
 *  @return TRUE if the message was processed
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if ( ptr_message->msgid == MSG_KEYDOWN )
 *     {
 *       //  If user presses the '?' button, help will be shown.
 *       cWinApp_defproc( main_module.m_process, ptr_message);
 *     }
 *     ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 88
#endif
bool cWinApp_defproc(struct cWinApp* ptr_win_app, struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_pause               SystemThread_pause
#else
/**
 *  Pauses execution of the thread.
 *  @param ptr_win_app A pointer to the cWinApp initialized structure
 *  @param timeout The time span to sleep (in milliseconds)
 *  @return FALSE if the timeout has passed
 *  @code
 *   #include <cywin.h>
 *     ...
 *     play_tone(30);
 *     cWinApp_pause(main_module.m_process, 200);
 *     play_tone(-1);
 *     ...
 *  @endcode
 */
bool cWinApp_pause (struct cWinApp* ptr_win_app, clock_t timeout);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_init_display      AppGeneric_init_display
#else
/**
 *  Obtains the graphics context.
 *  @return A pointer to the DisplayGraphics object
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct DisplayGraphics* ptr_gfx;
 *     ...
 *     // Clears the screen.
 *     cWinApp_clear_screen();
 *     ...
 *     // Draws a filled rectangle.
 *     ptr_gfx = cWinApp_init_display();
 *     DisplayGraphics_set_color( ptr_gfx , CLR_BLACK );
 *     DisplayGraphics_fill_rect( ptr_gfx , 5, 5, 30, 30 );
 *     DisplayGraphics_show( ptr_gfx );
 *     ...
 *  @endcode
 */
struct DisplayGraphics* cWinApp_init_display(void);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_clear_screen        AppGeneric_clear_screen
#else
/**
 *  Clears the screen with the color white.
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct DisplayGraphics* ptr_gfx;
 *     ...
 *     // Clears the screen.
 *     cWinApp_clear_screen();
 *     ...
 *     // Draws the filled rectangle.
 *     ptr_gfx = cWinApp_init_display();
 *     DisplayGraphics_set_color( ptr_gfx , CLR_BLACK );
 *     DisplayGraphics_fill_rect( ptr_gfx , 5, 5, 30, 30 );
 *     DisplayGraphics_show( ptr_gfx );
 *     ...
 *  @endcode
 */
void cWinApp_clear_screen (void);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_cancel_shutup       AppGeneric_cancel_shutup
#else
/**
 *  Cancels the MSG_SHUTUP message.
 *  Used to cancel a task switch operation previously launched by
 *  another application in the task manager. Deletes all
 *  MSG_SHUTUP messages from the message queue.
 *  Requires a cWinApp object pointer.
 *  @param ptr_win_app A pointer to the cWinApp initialized structure
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if( ptr_message->msgid == MSG_SHUTUP)
 *     {
 *       // We cancel shutup and delete all MSG_SHUTUP messages.
 *       cWinApp_cancel_shutup( main_module.m_process );
 *       ...
 *     }
 *     ...
 *  @endcode
 *  @see MSG_SHUTUP, cWinApp_ext_cancel_shutup.
 */
void cWinApp_cancel_shutup (struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_ext_cancel_shutup   AppGeneric_ext_cancel_shutup
#else
/**
 *  Cancels the MGS_SHUTUP message statically.
 *  Used to cancel a task switch operation previously launched
 *  by another application in the task manager.
 *  Doesn't require a cWinApp object pointer,
 *  but doesn't delete any other MSG_SHUTUP messages.
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if( ptr_message->msgid == MSG_SHUTUP)
 *     {
 *       // We cancel shutup.
 *       cWinApp_ext_cancel_shutup();
 *       ...
 *     }
 *     ...
 *  @endcode
 *  @see MSG_SHUTUP, cWinApp_cancel_shutup.
 */
void cWinApp_ext_cancel_shutup (void);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_request_focus  Process_request_focus
#else
/**
 *  Requests focus for the application
 *  @param ptr_win_app A pointer to the cWinApp initialized structure
 *  @return A pointer to a Process object
 *  @code
 *   #include <cywin.h>
 *     ...
 *     cWinApp_request_focus( main_module.m_process );
 *     ...
 *     if( cWinApp_has_focus( main_module.m_process ) )
 *     {
 *       // Draws something.
 *       ...
 *     }
 *     ...
 *  @endcode
 */
struct Process* cWinApp_request_focus (struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_has_focus      Process_has_focus
#else
/**
 *  Checks whether the application has the focus
 *  @param ptr_win_app A pointer to the cWinApp initialized structure
 *  @return TRUE if the application has the focus
 *  @code
 *   #include <cywin.h>
 *   ...
 *   cWinApp_request_focus( main_module.m_process );
 *   ...
 *   if( cWinApp_has_focus( main_module.m_process ) )
 *   {
 *     // Draws something.
 *     ...
 *   }
 *   ...
 *  @endcode
 */
bool cWinApp_has_focus (struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_get_display    Process_get_display
#else
/**
 *  Returns the graphics context.
 *  @return The graphics context
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct Bitmap bmp;
 *     ...
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     ...
 *     // Draws bitmap.
 *     Graphics_draw_bitmap( cWinApp_get_display(), &bmp, 30, 40, BM_NORMAL );
 *     ...
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     ...
 *  @endcode
 */
struct DisplayGraphics* cWinApp_get_display (void);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_peek_message   Process_peek_message
#else
/**
 *  Peeks or gets a message from queue.
 *  If 'remove' is TRUE, the message will be removed.
 *  This function does not wait for a message to be available,
 *  it returns 0 immediately it there are no messages
 *  in the specified range.
 *  Don't use it unless you really need it.
 *  @param ptr_win_app A pointer to the initialized cWinApp object
 *  @param remove If TRUE, the message will be removed from the message queue
 *  @param min Retrieves the minimum ID of the message in the queue.
 *  @param max Retrieves the maximum ID of the message in the queue
 *  @return A pointer to the Message if there is one in the process' queue; otherwise 0
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // Removes all keyboard messages from the queue.
 *     while(ptr_message = cWinApp_peek_message( main_module.m_process, 0, MSG_KEYDOWN, MSG_KEYDOWN ))
 *     {
 *       Message_delete( ptr_message );
 *     }
 *     ...
 *  @endcode
 */
struct Message* cWinApp_peek_message (struct cWinApp* ptr_win_app, bool remove,int min, int max);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_get_message    Process_get_message
#else
/**
 *  The calling thread waits until the first message
 *  that fits the specified range appears in the queue.
 *  Be careful not to overload the queue.
 *  Call Message_delete() when you finish with the message.
 *  @param ptr_win_app A pointer to the initialized cWinApp object
 *  @param timeout Timeout value (in milliseconds)
 *  @param min Retrieves the minimum ID of the message in the queue
 *  @param max Retrieves the maximum ID of the message in the queue
 *  @return Pointer to the Message if there is one in the process' queue
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if( ptr_message->msgid == MSG_KEYDOWN )
 *     {
 *       if( Message_get_key_param( ptr_message )->scancode == MSG_ESC )
 *       {
 *         // Processes key 'Esc'.
 *       }
 *     }
 *   ...
 *  @endcode
 *  @see Message_delete
 */
struct Message* cWinApp_get_message (struct cWinApp* ptr_win_app, long timeout, int min, int max);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_put_message    Process_put_message
#else
/**
 *  Puts a Message in the Queue.
 *  @param ptr_win_app A pointer to the initialized cWinApp object
 *  @param ptr_message A pointer to the message to send
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   #define MSG_MAKE_TASK MSG_USER + 1
 *     ...
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_MAKE_TASK );
 *     if( ptr_message->msgid == MSG_MAKE_TASK )
 *     {
 *       cWinApp_pause( main_module.m_process, 250 );
 *       cWinApp_put_message( main_module.m_process, ptr_message );
 *     }
 *     else
 *     {
 *       Message_delete( ptr_message );
 *     }
 *     ...
 *  @endcode
 */
void cWinApp_put_message (struct cWinApp* ptr_win_app, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_get_name       Process_get_name
#else
/**
 *  Returns the message queue's name.
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return The message queue's name
 */
char* cWinApp_get_name (struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_Disconnect(ptr_win_app)  \
        cObject_Disconnect((char*)(ptr_win_app) + 172)
#else
/**
 *  Disconnects a cWinApp from its parent object.
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return None
 */
void cWinApp_Disconnect(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_Select(ptr_win_app)  \
        cObject_Select((char*)(ptr_win_app) + 172)
#else
/**
 *  Selects a cWinApp object.
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return TRUE if the object was selected
 */
bool cWinApp_Select(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_update(ptr_win_app)  \
        cObject_update((char*)(ptr_win_app) + 172)
#else
/**
 *  Updates a cWinApp object.
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return None
 */
void cWinApp_update(struct cWinApp* ptr_win_app);
#endif


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_GetParent(ptr_win_app)  \
        cObject_GetParent((char*)(ptr_win_app) + 172)
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return A pointer to the parent object
 */
struct cClip* cWinApp_GetParent(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_Hide(ptr_win_app)  \
        cObject_Hide((char*)(ptr_win_app) + 172)
#else
/**
 *  Hides a cWinApp object.
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return None
 *  @see cWinApp_Show.
 */
void cWinApp_Hide(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_Show(ptr_win_app)  \
        cObject_Show((char*)(ptr_win_app)  + 172)
#else
/**
 *  Shows a cWinApp object.
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return None
 *  @see cWinApp_Hide.
 */
void cWinApp_Show(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_Disable(ptr_win_app)  \
        cObject_Disable((char*)(ptr_win_app) + 172)
#else
/**
 *  Disables a cWinApp object so that it cannot be selected
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return None
 */
void cWinApp_Disable(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_Enable(ptr_win_app)  \
        cObject_Enable((char*)(ptr_win_app) + 172)
#else
/**
 *  Enables a cWinApp object so that it may be selected.
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @return None
 */
void cWinApp_Enable(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_AddObj(ptr_win_app, ptr_object, x, y)  \
        cClip_AddObj((char*)(ptr_win_app) + 172, (ptr_object), (x), (y))
#else
/**
 *  Adds an object to the cWinApp in position (x, y).
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @param x x-coordinate of the new object
 *  @param y y-coordinate of the new object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     // It's the same as using the cClip_AddObj( &button ) function.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
void cWinApp_AddObj(struct cWinApp* ptr_win_app, struct cObject* ptr_object, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_InsObj(ptr_win_app, ptr_object, x, y)  \
        cClip_InsObj((char*)(ptr_win_app) + 172, (ptr_object), (x), (y))
#else
/**
 *  Adds an object to the cWinApp in position (x, y)
 *  with a z-position index (maximal index is on the top).
 *  @param ptr_win_app A pointer to the initialized cWinApp structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @param x x-coordinate of the new object
 *  @param y y-coordinate of the new object
 *  @param index The index of the new object's z-position
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button1;
 *     struct cButton button2;
 *     struct cButton button3;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button1, "Button text 1", mrOk );
 *     cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *     ...
 *     cButton_ctor( &button2, "Button text 2", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *     ...
 *     // It's the same as using the cClip_InsObj( &button3 ) function.
 *     cButton_ctor( &button3, "Button text 3", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
void cWinApp_InsObj(struct cWinApp* ptr_win_app, struct cObject* ptr_object, int x, int y, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_RemObj(ptr_win_app, ptr_object)   \
        cClip_RemObj((char*)(ptr_win_app) + 172, (ptr_object))
#else
/**
 *  Removes an object from cWinApp.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     // It's the same as using the cClip_AddObj( &button ) function.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     // It's the same as using the cClip_RemObj( &button ) function.
 *     cClip_RemObj( main_module.m_process, &button );
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
void cWinApp_RemObj(struct cClip* ptr_win_app, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_SelectFirst(ptr_win_app)  \
        cClip_SelectFirst((char*)(ptr_win_app) + 172)
#else
/**
 *  Selects the first object in the cWinApp.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @return FALSE if the function failed.
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button1;
 *     struct cButton button2;
 *     struct cButton button3;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button1, "Button text 1", mrOk );
 *     cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *     ...
 *     cButton_ctor( &button2, "Button text 2", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *     ...
 *     // It's the same as using the cClip_InsObj( &button3 ) function.
 *     cButton_ctor( &button3, "Button text 3", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *     ...
 *     // Selects the first button.
 *     cWinApp_SelectFirst( main_module.m_process );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @see cWinApp_SelectPrev, cWinApp_SelectNext.
 */
bool cWinApp_SelectFirst(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_SelectPrev(ptr_win_app, round) \
        cClip_SelectPrev((char*)(ptr_win_app) + 172, (round))
#else
/**
 *  Selects the previous object in cWinApp.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the function failed
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button1;
 *     struct cButton button2;
 *     struct cButton button3;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button1, "Button text 1", mrOk );
 *     cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *     ...
 *     cButton_ctor( &button2, "Button text 2", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *     ...
 *     // It's the same as using the cClip_InsObj( &button3 ) function.
 *     cButton_ctor( &button3, "Button text 3", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *     ...
 *     // Selects the first button.
 *     cWinApp_SelectFirst( main_module.m_process );
 *     ...
 *     // Selects the previous button.
 *     cWinApp_SelectPrev( main_module.m_process , TRUE);
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @see cWinApp_SelectFirst, cWinApp_SelectNext.
 */
bool cWinApp_SelectPrev(struct cWinApp* ptr_win_app, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_SelectNext(ptr_win_app, round) \
        cClip_SelectNext((char*)(ptr_win_app) + 172, (round))
#else
/**
 *  Selects the next object in the cWinApp
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the function failed
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button1;
 *     struct cButton button2;
 *     struct cButton button3;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button1, "Button text 1", mrOk );
 *     cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *     ...
 *     cButton_ctor( &button2, "Button text 2", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *     ...
 *     // It's the same as using the cClip_InsObj( &button3 ) function.
 *     cButton_ctor( &button3, "Button text 3", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *     ...
 *     // Selects the first button.
 *     cWinApp_SelectFirst( main_module.m_process );
 *     ...
 *     // Selects the next button.
 *     cWinApp_SelectNext( main_module.m_process , TRUE);
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @see cWinApp_SelectFirst, cWinApp_SelectPrev.
 */
bool cWinApp_SelectNext(struct cWinApp* ptr_win_app, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_Scroll(ptr_win_app, rectangle)  \
        cClip_Scroll((char*)(ptr_win_app) + 172, (rectangle))
#else
/**
 *  Scrolls a cWinApp to make a rectangle visible. Provides minimal scrolling.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @param rectangle A pointer to the initialized rect_t structure
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cBitmap cbmp;
 *     struct Bitmap bmp;
 *     struct module_t main_module;
 *     struct rect_t rect;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     cBitmap_ctor( &cbmp, &bmp );
 *     // Puts the cBitmap object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *     ...
 *     rect_set( &rect, 160, 240, bmp.h, bmp.w );
 *     // Scrolls the bitmap to the visible area.
 *     cWinApp_Scroll( main_module.m_process, &rect );
 *     ...
 *     cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     ...
 *  @endcode
 *  @see cWinApp_Scroll_Ex.
 */
void cWinApp_Scroll(struct cWinApp* ptr_win_app, struct rect_t* rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_Scroll_Ex(ptr_win_app, x, y)  \
        cClip_Scroll_Ex((char*)(ptr_win_app) + 172, (x), (y))
#else
/**
 *  Scrolls a cWinApp by (x, y), but cannot exceed the child's boundaries.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @param x An "x" coordinate shift
 *  @param y An "y" coordinate shift
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cBitmap cbmp;
 *     struct Bitmap bmp;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     cBitmap_ctor( &cbmp, &bmp );
 *     // Puts the cBitmap object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *     ...
 *     rect_set( &rect, 160, 240, bmp.h, bmp.w );
 *     // Scrolls the bitmap to the visible area.
 *     cWinApp_Scroll_Ex( main_module.m_process, -160, -240);
 *     ...
 *     cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     ...
 *  @endcode
 *  @see cWinApp_Scroll.
 */
void cWinApp_Scroll_Ex(struct cWinApp* ptr_win_app, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_SendScroll(ptr_win_app)  \
        cClip_SendScroll((char*)(ptr_win_app) + 172)
#else
/**
 *  Forces this object to redraw its scrolling arrows.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cBitmap cbmp;
 *     struct Bitmap bmp;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     cBitmap_ctor( &cbmp, &bmp );
 *     // Puts the cBitmap object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *     ...
 *     // Scrolls the bitmap to the visible area.
 *     cWinApp_Scroll_Ex( main_module.m_process, -160, -240 );
 *     cWinApp_SendScroll( main_module.m_process );
 *     ...
 *     cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     ...
 *  @endcode
 */
void cWinApp_SendScroll(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_GetShifty(ptr_win_app)  \
        cClip_GetShifty((char*)(ptr_win_app) + 172)
#else
/**
 *  Returns the 'y' coordinate shift (as a result of scrolling).
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @return 'y' coordinate shift (as a result of scrolling)
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cBitmap cbmp;
 *     struct Bitmap bmp;
 *     struct module_t main_module;
 *     int nShift_Y;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     cBitmap_ctor( &cbmp, &bmp );
 *     // Puts the cBitmap object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *     ...
 *     // Scrolls the bitmap to the visible area.
 *     cWinApp_Scroll_Ex( main_module.m_process, -160, -240 );
 *     cWinApp_SendScroll( main_module.m_process );
 *     // nShift_Y will be equal to -240.
 *     nShift_Y = cWinApp_GetShifty( main_module.m_process );
 *     ...
 *     cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     ...
 *  @endcode
 *  @see cWinApp_GetShiftx.
 */
int cWinApp_GetShifty(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_GetShiftx(ptr_win_app)  \
        cClip_GetShiftx((char*)(ptr_win_app) + 172)
#else
/**
 *  Returns the 'x' coordinate shift (as a result of scrolling).
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @return 'x' coordinate shift (as a result of scrolling).
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cBitmap cbmp;
 *     struct Bitmap bmp;
 *     struct module_t main_module;
 *     int nShift_X;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     cBitmap_ctor( &cbmp, &bmp );
 *     // Puts the cBitmap object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *     ...
 *     // Scrolls the bitmap to the visible area.
 *     cWinApp_Scroll_Ex( main_module.m_process, -160, -240 );
 *     cWinApp_SendScroll( main_module.m_process );
 *     // nShift_X will be equal to -160.
 *     nShift_X = cWinApp_GetShiftx( main_module.m_process );
 *     ...
 *     cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see cWinApp_GetShifty.
 */
int cWinApp_GetShiftx(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_GetCount(ptr_win_app)  \
        cClip_GetCount((char*)(ptr_win_app) + 172)
#else
/**
 *  Returns a count of child objects.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @return Child objects count
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button1;
 *     struct cButton button2;
 *     struct cButton button3;
 *     struct module_t main_module;
 *     int nButtonNumber;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button1, "Button text 1", mrOk );
 *     cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *     ...
 *     cButton_ctor( &button2, "Button text 2", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *     ...
 *     // It's the same as using the cClip_InsObj( &button3 ) function.
 *     cButton_ctor( &button3, "Button text 3", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *     ...
 *     // nButtonNumber will be equal to 3.
 *     nButtonNumber = cWinApp_GetCount( main_module.m_process );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
int cWinApp_GetCount(struct cWinApp* ptr_win_app);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_get_by_index(ptr_win_app, index)    \
        cClip_get_by_index((char*)(ptr_win_app) + 172, (index))
#else
/**
 *  Returns an object with its index.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @param index Index of the object in cWinApp
 *  @return The object with its index
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button1;
 *     struct cButton button2;
 *     struct cButton button3;
 *     struct module_t main_module;
 *     struct cObject * currentObj;
 *     int nButtonNumber;
 *     int i;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button1, "Button text 1", mrOk );
 *     cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *     ...
 *     cButton_ctor( &button2, "Button text 2", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *     ...
 *     // It's the same as using the cClip_InsObj( &button3 ) function.
 *     cButton_ctor( &button3, "Button text 3", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *     ...
 *     // Redraws all objects.
 *     for ( i=0; i<cWinApp_GetCount( main_module.m_process ); i++)
 *     {
 *       cButton_update( cWinApp_get_by_index( main_module.m_process, i) );
 *     }
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
struct cObject* cWinApp_get_by_index(struct cWinApp* ptr_win_app, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_FindObj(ptr_win_app, ptr_object)  \
        cClip_FindObj((char*)(ptr_win_app) + 172, (ptr_object))
#else
/**
 *  Returns a child object's index in cWinApp (or -1).
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @param ptr_object A pointer to the needed cObject
 *  @return The child object's index in cWinApp (or -1).
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button1;
 *     struct cButton button2;
 *     struct cButton button3;
 *     struct module_t main_module;
 *     struct cObject * currentObj;
 *     int nButtonNumber;
 *     int i;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button1, "Button text 1", mrOk );
 *     cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *     ...
 *     cButton_ctor( &button2, "Button text 2", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *     ...
 *     // It's the same as using the cClip_InsObj( &button3 ) function.
 *     cButton_ctor( &button3, "Button text 3", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *     ...
 *      // Finds the object.
 *     if ( cWinApp_FindObj( main_module.m_process, &button2 ) != -1 )
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
int cWinApp_FindObj(struct cWinApp* ptr_win_app, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cWinApp_GetSelectedObject(ptr_win_app)   \
        cClip_GetSelectedObject((char*)(ptr_win_app) + 172)
#else
/**
 *  Returns the currently selected object.
 *  @param ptr_win_app A pointer to the initialized cClip structure
 *  @return The currently selected object
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button1;
 *     struct cButton button2;
 *     struct cButton button3;
 *     struct module_t main_module;
 *     struct cObject * selectedObj;
 *     int nButtonNumber;
 *     int i;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button1, "Button text 1", mrOk );
 *     cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *     ...
 *     cButton_ctor( &button2, "Button text 2", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *     ...
 *     // It's the same as using the cClip_InsObj( &button3 ) function.
 *     cButton_ctor( &button3, "Button text 3", mrOk );
 *     cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *     ...
 *     cObject_Select( &button2 );
 *     ...
 *     // Redraws selected objects.
 *     cButton_update( cWinApp_GetSelectedObject( main_module.m_process );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
struct cObject* cWinApp_GetSelectedObject(struct cWinApp* ptr_win_app);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *   @ingroup controls_group
 *   @brief Static, transparent text line
 *
 *   A static, transparent text line. Use this structure to place text on the Cybiko computer's screen.
 *   <br>
 *   <br>
 *   To place a cText object created by the cText_ctor function onto the Cybiko screen, use the cWinApp_AddObj function.
 *   If you want to replace the onscreen text, you should first clear it with the cWinApp_clear_screen
 *   function. Then use the cText_SetText function to place new text.
 *   <br>
 *   <br>
 *   You must call the cText_ctor function before use and the cText_dtor function after use.
 *   @see controls_group , cWinApp_AddObj , cWinApp_clear_screen
 */
struct cText : public cObject
{
#ifndef _SKIP_DOC
  char cText_Members_STUB[12];
};
#endif

//============================
//=                          =
//=       object cText.      =
//=     Simple text line.    =
//=                          =
//============================

/**
 *  Initializes a cText object.
 *  @param ptr_text A pointer to a cText object
 *  @param text A text string
 *  @param ptr_font A pointer to the structure of the font to use
 *  @param color The text's color
 *  @return A pointer to the initialized cText object
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 108
#endif
struct cText* cText_ctor(struct cText* ptr_text, char* text, struct Font* ptr_font, color_t color);

/**
 *  Deletes a cText object.
 *  @param ptr_text is a pointer to a cText object.
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was
 *  allocated for the object by malloc(), use FREE_MEMORY to free it. Use
 *  LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 209
#endif
void cText_dtor(struct cText* ptr_text, int memory_flag);

/**
 *  Sets a new text line.
 *  @param ptr_text A pointer to the initialized cText object
 *  @param text A new text string
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     ...
 *     cWinApp_clear_screen();
 *     cText_SetText( &text, "Another text" );
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 87
#endif
void cText_SetText(struct cText* ptr_text, char* text);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_proc         cObject_proc
#else
/**
 *  The message-processing function.
 *  @param ptr_text A pointer to the initialized cText object
 *  @param ptr_message A pointer to the processed message
 *  @return TRUE if the message was processed.
 */
bool cText_proc(struct cText* ptr_text, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_Disconnect   cObject_Disconnect
#else
/**
 *  Disconnects a cText object from its parent object.
 *  @param ptr_text A pointer to the initialized cText object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     // Removes the object from the Cybiko screen.
 *     cText_Disconnect( &text);
 *     ...
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     // Afterwards this function disconnects the cText object automatically
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 */
void cText_Disconnect (struct cText* ptr_text);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_Select       cObject_Select
#else
/**
 *  Selects a cText object.
 *  @param ptr_text A pointer to the initialized cText object
 *  @return TRUE if the object was selected
 *  @code
 *   #include <cywin.h>
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     cText_Select(&text)
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 */
bool cText_Select (struct cText* ptr_text);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_update       cObject_update
#else
/**
 *  Updates a cText object.
 *  @param ptr_text A pointer to the initialized cText object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     cWinApp_clear_screen();
 *     // Redraws the text
 *     cText_update(&text);
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 */
void cText_update (struct cText* ptr_text);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_GetParent    cObject_GetParent
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_text A pointer to the initialized cText object
 *  @return A pointer to the parent object
 *  @code
 *   #include <cywin.h>
 *     struct cText text;
 *     struct module_t main_module;
 *     struct cClip *parent;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     parent = cText_GetParent( &text );
 *     if ( parent == main_module.m_process )
 *     {
 *     }
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 */
struct cClip* cText_GetParent (struct cText* ptr_text);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_Hide         cObject_Hide
#else
/**
 *  Hides a cText object.
 *  @param ptr_text A pointer to the initialized cText object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     cText_Hide (&text);
 *      ...
 *     cText_Show (&text);
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 *  @see cText_Show.
 */
void cText_Hide (struct cText* ptr_text);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_Show         cObject_Show
#else
/**
 *  Shows a cText object.
 *  @param ptr_text A pointer to the initialized cText object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     cText_Hide (&text);
 *     ...
 *     cText_Show (&text);
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 *  @see cText_Hide.
 */
void cText_Show (struct cText* ptr_text);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_Disable      cObject_Disable
#else
/**
 *  Disables a cText object so that it cannot be selected.
 *  @param ptr_text A pointer to the initialized cText object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     // Now the user can't use this object
 *     cText_Disable (&text);
 *     ...
 *     // Now the user can use this object again
 *     cText_Enable (&text);
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 *  @see cText_Enable.
 */
void cText_Disable (struct cText* ptr_text);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cText_Enable       cObject_Enable
#else
/**
 *  Enables a cText object so that it may be selected.
 *  @param ptr_text A pointer to the initialized cText object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     struct cText text;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cText_ctor( &text, "Any string", cool_normal_font , CLR_DKGRAY );
 *     cWinApp_clear_screen();
 *     // Puts the 'text' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &text, 10, 40 );
 *     // Now the user can't use this object
 *     cText_Disable (&text);
 *     ...
 *     // Now the user can use this object again
 *     cText_Enable (&text);
 *     ...
 *     cText_dtor( &text, LEAVE_MEMORY );
 *  @endcode
 *  @see cText_Disable.
 */
void cText_Enable (struct cText* ptr_text);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *   @ingroup controls_group
 *   @brief A basic button.
 *
 *   A basic button. You can add a button to any user form, as well as to the Cybiko screen.
 *   <br>
 *   <br>
 *   Every button must have its own modal result, which can be defined in the third parameter of the
 *   cButton_ctor function. Variants of modal results are contained in the tmResults enumeration.
 *   Every button can have a message-processing function.
 *   <br>
 *   <br>
 *   You must call the cButton_ctor function before use and the cButton_dtor function after use.
 *   @see controls_group
 */
struct cButton : public cObject
{
#ifndef _SKIP_DOC
  char cButon_Members_STUB[12];
};
#endif

//============================
//=                          =
//=      object cButton.     =
//=          Button.         =
//=                          =
//============================

/**
 *  Initializes a cButton object.
 *  @param ptr_cbutton A pointer to a cButton object
 *  @param text The button's text
 *  @param modal_result Will be set in cForm when the button is pressed
 *  @return A pointer to the initialized cButton object
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 116
#endif
struct cButton* cButton_ctor(struct cButton* ptr_cbutton, char* text, int modal_result);

/**
 *  Deletes a cButton object.
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated
 *  for the object by malloc(), use FREE_MEMORY to free it.
 *  Use LEAVE_MEMORY if the object was static or allocated in stack
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 111
#endif
void cButton_dtor(struct cButton* ptr_cbutton, int memory_flag);

/**
 *  Returns the button modal result, which was set in the cButton constructor.
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @return The button modal result
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     if ( cButton_GetModalResult( &button ) == mrOk )
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 115
#endif
int cButton_GetModalResult(struct cButton* ptr_cbutton);

/**
 *  The Message-processing function.
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @param ptr_message A pointer to the processed message
 *  @return TRUE if the message was processed
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if ( ptr_message->msgid == MSG_KEYDOWN )
 *     {
 *       //  If user presses 'Enter' button will be pushed.
 *       cButton_proc( &button, ptr_message);
 *     }
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 99
#endif
bool cButton_proc(struct cButton* ptr_cbutton, struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cButton_Enable       cObject_Enable
#else
/**
 *  Enables the cButton object so that it may be selected.
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     struct cClip* parent;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     // It's the same as using the cButton_Disable( &button ) function.
 *     // Now the user can't use this object.
 *     cButton_Disable( &button )
 *     ...
 *     // It's the same as using the cButton_Enable( &button ) function.
 *     // Now the user can use this object again.
 *     cButton_Enable( &button )
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @see cButton_Disable.
 */
void cButton_Enable (struct cButton* ptr_cbutton);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cButton_Disable      cObject_Disable
#else
/**
 *  Disables the cButton object so that it cannot be selected..
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     struct cClip* parent;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     // It's the same as using the cButton_Disable( &button ) function.
 *     // Now the user can't use this object.
 *     cButton_Disable( &button )
 *     ...
 *     // It's the same as using the cButton_Enable( &button ) function.
 *     // Now the user can use this object again.
 *     cButton_Enable( &button )
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @see cButton_Enable.
 */
void cButton_Disable (struct cButton* ptr_cbutton);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cButton_Show         cObject_Show
#else
/**
 *  Shows cButton object.
 *  @param ptr_cbutton A pointer to the initialized cButton object.
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     // It's the same as using the cButton_Hide( &button ) function.
 *     cObject_Hide( &button );
 *     ...
 *     // It's the same as using the cButton_Show( &button ) function.
 *     cObject_Show( &button );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @see cButton_Hide.
 */
void cButton_Show (struct cButton* ptr_cbutton);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cButton_Hide         cObject_Hide
#else
/**
 *  Hides a cButton object.
 *  @param ptr_cbutton A pointer to the initialized cButton object.
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     // It's the same as using the cButton_Hide( &button ) function.
 *     cObject_Hide( &button );
 *     ...
 *     // It's the same as using the cButton_Show( &button ) function.
 *     cObject_Show( &button );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @see cButton_Show.
 */
void cButton_Hide (struct cButton* ptr_cbutton);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cButton_Disconnect   cObject_Disconnect
#else
/**
 *  Disconnects the cButton object from its parent object.
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     // Removes the object from the Cybiko screen.
 *     cButton_Disconnect( &button);
 *     ...
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     // Afterwards this function disconnects the cText object automatically
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
void cButton_Disconnect (struct cButton* ptr_cbutton);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cButton_Select       cObject_Select
#else
/**
 *  Selects a cButton object.
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @return TRUE if the object was selected
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     // It's the same as using the cButton_Select( &button ) function.
 *     // Selects the button.
 *     cButton_Select( &button );
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
bool cButton_Select (struct cButton* ptr_cbutton);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cButton_update       cObject_update
#else
/**
 *  Updates a cButton object.
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     cWinApp_clear_screen();
 *     // It's the same as using the cButton_update( &button ) function.
 *     // Redraws the button.
 *     cButton_update( &button )
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
void cButton_update (struct cButton* ptr_cbutton);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cButton_GetParent    cObject_GetParent
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_cbutton A pointer to the initialized cButton object
 *  @return A pointer to the parent object
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cButton button;
 *     struct module_t main_module;
 *     struct cClip* parent;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cButton_ctor( &button, "Button text", mrOk );
 *     // Puts the 'button' object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *     ...
 *     // It's the same as using the cButton_GetParent( &button ) function.
 *     // Checks for the right parent relation.
 *     parent = cObject_GetParent( &button );
 *     if ( parent == main_module.m_process )
 *     {
 *     }
 *     ...
 *     cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
struct cClip* cButton_GetParent (struct cButton* ptr_cbutton);
#endif

#ifdef _SKIP_DOC
};
#endif

/**  Predefined results that may be returned from modal forms. You can use your own results too.
 *   @ingroup forms_group
 */
enum tMResults
{
  /**
   *  no result available
   */
  mrNone,

  /**
   *  "OK" button pressed
   */
  mrOk,

  /**
   *  "YES" button pressed
   */
  mrYes,

  /**
   *  "NO" button pressed
   */
  mrNo,

  /**
   *  "DRAW" button pressed
   */
  mrDraw,

  /**
   *  "CHAT" button pressed
   */
  mrChat,

  /**
   *  "QUIT" button pressed
   */
  mrQuit,

  /**
   *  "NEW" button pressed
   */
  mrNew        = 6,

  /**
   *  "UPLOAD" button pressed
   */
  mrUpload,

  /**
   *  "RENAME" button pressed
   */
  mrRename,

  /**
   *  "DELETE" button pressed
   */
  mrDelete,

  /**
   *  "VIEW" button pressed
   */
  mrView,

  /**
   *  "PLAY" button pressed
   */
  mrPlay,

  /**
   *  "RESTART" button pressed
   */
  mrRestart,

  /**
   *  "SET" button pressed
   */
  mrSet,

  /**
   *  "LATER" button pressed
   */
  mrLater,

  /**
   *  "CANCEL" button pressed
   */
  mrCancel,

  /**
   *  Message received
   */
  mrMessage,

  /**
   *  beginning of the user's definitions
   */
  mrUser       = 32
};

/**
 *  @ingroup forms_group
 *  @brief This is a structure of the main form.
 *
 *  This is the main form structure, used to produce full-screen rectangle and rounded-rectangle windows (forms).
 *  Forms are useful for creating non-game applications. These applications usually consist of one or more
 *  hierarchically organized screens or windows, and the user moves between them in accordance with its hierarchy.
 *   <br>
 *   <br>
 *  To achieve this using forms, you should simply derive one cCustomForm structure for every window you need. Then in your application, create your main window's object and call its cCustomForm_ShowModal method.
 *  If you need to show another window somewhere, create the object of the form structure you need and fill the form fields. Then call its ShowModal method, read the save form data, and delete the form.
 *   <br>
 *   <br>
 *  You can also create non-modal windows, if you need to. After creating a form, you can view it (see cCustomForm_Show).
 *  When a message comes in, call the form's cCustomForm_Proc function to process it. The key is one form per screen or window.
 *   <br>
 *   <br>
 *  You must call the cCustomForm_ctor function before use and the cCustomForm_dtor function after use.
 *  @see forms_group
 */
struct cCustomForm : public cClip
{
   /**
    *  Form caption
    */
   char FormNameString[30];
   /**
    *  Form type (rect or rounded box)
    */
   char Type;

   #ifndef _SKIP_DOC
   char Type_PADDING;
   #endif

   /**
    *  Parent application.
    */
   struct cWinApp* CurrApplication;
   /**
    *  The client area of the form. All "normal" objects are added to the form's client area,
    *  including all free form space, and excluding the header and border.
    */
   struct cClip* ClientArea;

   /** Result of form or dialog. Modal forms run while the ModalResult is zero. When you set
    *  it to non-zero value, the modal cycle is canceled and ShowModal() returns ModalResult value.
    *  There are some predefined values (see tMResults). ModalResult can be set automatically, when
    *  you press <Esc> in the form, or when you press the in-form cButton.
    */
   int ModalResult;

   /** Window help context. You can set it to a non-zero value in the 0..999 range, and, when the '<?>' key
    *  is pressed, the help file named '<your value>.help' will be showed. The default value is zero.
    */
   int HelpContext;
#ifndef _SKIP_DOC
};
#endif

//============================
//=                          =
//=    object cCustomForm.   =
//=  Common window template. =
//=                          =
//============================

/**
 *  Constructor. Creates a cCustomForm object.
 *  @param ptr_custom_form A pointer to a cCustomForm object
 *  @param rect A window rectangle. If you want to create a full-screen form, you must set 'rect' to {-2,-2,164,104}
 *  @param name The form header caption. If NULL, the form has no header
 *  @param round If TRUE, the form shows a rectangle window with rounded corners; otherwise the form shows a plain rectangular window
 *  @param ptr_win_app A pointer to a cWinApp object. If NULL, the form will not be added to any application, so you can't use it in modal mode. Don't call
 *  cCustomForm_ShowModal if you set ptr_win_app to NULL.
 *  The forms are hidden, when created. To show them, use the cCustomForm_Show() or cCustomForm_ShowModal() functions.
 *  @return A pointer to the initialized cCustomForm object
 *  @code
 *    #include <cywin.h>
 *      ...
 *      struct cCustomForm custom_form;
 *      struct rect_t rect;
 *      struct module_t main_module;
 *      ...
 *      init_module( &main_module );
 *      ...
 *      cCustomForm_ctor( &custom_form, &rect, "Caption", TRUE, main_module.m_process );
 *      cCustomForm_ShowModal( &custom_form );
 *      ...
 *      cCustomForm_dtor( &custom_form, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 89
#endif
struct cCustomForm* cCustomForm_ctor(struct cCustomForm* ptr_custom_form, struct rect_t* rect,
                                     char* name, bool round, struct cWinApp* ptr_win_app);

/**
 *  Deletes a cCustomForm object.
 *  @param ptr_custom_form A pointer to a cCustomForm object.
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY if the object was static or allocated in stack.
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cCustomForm custom_form;
 *     struct rect_t rect;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cCustomForm_ctor( &custom_form, &rect, "Caption", TRUE, main_module.m_process );
 *     cCustomForm_ShowModal( &custom_form );
 *     ...
 *     cCustomForm_dtor( &custom_form, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 112
#endif
void cCustomForm_dtor(struct cCustomForm* ptr_custom_form, int memory_flag);

/**
 *  Shows a form in modal mode.
 *  This function shows the form and enters it to the message loop. The loop continues
 *  until anyone ModalResult variable is set to a non-zero value, or until a MSG_QUIT or MSG_SHUTUP messages is received.
 *  If these messages are received, the function reposts them to the application and leaves the message loop. This is how all forms will close when the applications ends.
 *  @param ptr_custom_form  A pointer to a cCustomForm object
 *  @return tMResults value. mrNone(Zero) if MSG_QUIT or MSG_SHUTUP messages initiates leaving the loop
 *          See tMResults enum for predefined results
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cCustomForm custom_form;
 *     struct rect_t rect;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cCustomForm_ctor( &custom_form, &rect, "Caption", TRUE, main_module.m_process );
 *     cCustomForm_ShowModal( &custom_form );
 *     ...
 *     cCustomForm_dtor( &custom_form, LEAVE_MEMORY );
 *  @endcode
 *  @see cCustomForm_ctor , MSG_QUIT , MSG_SHUTUP
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 94
#endif
int cCustomForm_ShowModal(struct cCustomForm* ptr_custom_form);

/**
 *  The message-processing function.
 *  @param ptr_custom_form A pointer to a cCustomForm object
 *  @param ptr_message A pointer to the processed message
 *  @return TRUE if the message was processed
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cCustomForm custom_form;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cCustomForm_ctor( &custom_form, &rect, "Caption", TRUE, main_module.m_process );
 *     ...
 *     // Adding some objects to the custom form.
 *     ...
 *     cCustomForm_Show( &custom_form );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if ( ptr_message->msgid == MSG_KEYDOWN )
 *     {
 *       //  If 'Tab' was pressed, the next form's object will receive the focus.
 *       cCustomForm_proc( &custom_form, ptr_message );
 *     }
 *     ...
 *     cCustomForm_dtor( &custom_form, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 92
#endif
bool cCustomForm_proc(struct cCustomForm* ptr_custom_form , struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_Disconnect          cObject_Disconnect
#else
/**
 *  Disconnects a cCustomForm from its parent object.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return None
 */
void cCustomForm_Disconnect (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_Select              cObject_Select
#else
/**
 *  Selects a cCustomForm object.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return TRUE if the object was selected.
 */
bool cCustomForm_Select (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_update              cObject_update
#else
/**
 *  Updates a cCustomForm object
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   int cMainMenu_ShowModal( struct cCustomForm* ptr_form )
 *   {
 *    cCustomForm_Show( ptr_form );
 *    ...
 *    ptr_form->ModalResult = mrNone;
 *    ...
 *    while( ptr_form->ModalResult == mrNone )
 *    {
 *      struct Message* ptr_message =
 *        cWinApp_get_message( ptr_form->CurrApplication, 0, 1, MSG_USER );
 *      ...
 *      cMainMenu_proc( ptr_form, ptr_message );
 *      ...
 *      cCustomForm_update( ptr_form );
 *      ...
 *      Message_delete( ptr_message );
 *     }
 *     return ptr_form->ModalResult;
 *   }
 *   ...
 *  @endcode
 */
void cCustomForm_update (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_GetParent           cObject_GetParent
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return A pointer to the parent object
 */
struct cClip* cCustomForm_GetParent (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_Hide                cObject_Hide
#else
/**
 *  Hides a cCustomForm object.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   ptr_menu_form = (struct cCustomForm*) malloc( sizeof *ptr_menu_form );
 *   ...
 *   form_rect.x = 40;
 *   form_rect.y = 20;
 *   form_rect.w = 86;
 *   form_rect.h = 60;
 *   ...
 *   cCustomForm_ctor( ptr_menu_form, &form_rect, "Main Menu",
 *     TRUE, ptr_win_app );
 *   ...
 *   ptr_menu_form->HelpContext = 0;
 *   ptr_menu_form->ModalResult = mrNone;
 *   ...
 *   // Adds the menu to the menu form
 *   cCustomForm_AddObj( ptr_menu_form, ptr_menu, 0, 0 );
 *   ...
 *   cCustomForm_Hide( ptr_menu_form );
 *   ...
 *   // Creates text
 *   ptr_text = (struct cText*) malloc( sizeof *ptr_text );
 *   ...
 *   cText_ctor( ptr_text, "Press <Enter>", cool_normal_font, CLR_BLACK );
 *   ...
 *   // Adds the text to the main form
 *   cFrameForm_AddObj( ptr_form, ptr_text, 42, 35 );
 *   ...
 *  @endcode
 *  @see cCustomForm_Show.
 */
void cCustomForm_Hide (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_Show                cObject_Show
#else
/**
 *  Shows a cCustomForm object.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   int cMainMenu_ShowModal( struct cCustomForm* ptr_form )
 *   {
 *    cCustomForm_Show( ptr_form );
 *    ...
 *    ptr_form->ModalResult = mrNone;
 *    ...
 *    while( ptr_form->ModalResult == mrNone )
 *    {
 *      struct Message* ptr_message =
 *        cWinApp_get_message( ptr_form->CurrApplication, 0, 1, MSG_USER );
 *      ...
 *      cMainMenu_proc( ptr_form, ptr_message );
 *      ...
 *      cCustomForm_update( ptr_form );
 *      ...
 *      Message_delete( ptr_message );
 *    }
 *    return ptr_form->ModalResult;
 *   }
 *   ...
 *  @endcode
 *  @see cCustomForm_Hide.
 */
void cCustomForm_Show (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_Disable             cObject_Disable
#else
/**
 *  Disables the cCustomForm object so that it cannot be selected.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return None
 *  @see cCustomForm_Enable.
 */
void cCustomForm_Disable (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_Enable              cObject_Enable
#else
/**
 *  Enables a cCustomForm object so that it may be selected.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return None
 *  @see cCustomForm_Disable.
 */
void cCustomForm_Enable (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_AddObj(ptr_custom_form, ptr_object, x, y)   \
        if((ptr_custom_form)->ClientArea)           \
        {  \
          cClip_AddObj((ptr_custom_form)->ClientArea, (ptr_object), (x), (y)); \
        }
#else
/**
 *  Adds an object to the cCustomForm in position ( x, y ).
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @param x The x-coordinate of the new object
 *  @param y The y-coordinate of the new object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   ptr_menu_form = (struct cCustomForm*) malloc( sizeof *ptr_menu_form );
 *   ...
 *   form_rect.x = 40;
 *   form_rect.y = 20;
 *   form_rect.w = 86;
 *   form_rect.h = 60;
 *   ...
 *   cCustomForm_ctor( ptr_menu_form, &form_rect, "Main Menu",
 *     TRUE, ptr_win_app );
 *   ...
 *   ptr_menu_form->HelpContext = 0;
 *   ptr_menu_form->ModalResult = mrNone;
 *   ...
 *   // Adds the menu to the menu form
 *   cCustomForm_AddObj( ptr_menu_form, ptr_menu, 0, 0 );
 *   ...
 *   cCustomForm_Hide( ptr_menu_form );
 *   ...
 *   // Creates text
 *   ptr_text = (struct cText*) malloc( sizeof *ptr_text );
 *   ...
 *   cText_ctor( ptr_text, "Press <Enter>", cool_normal_font, CLR_BLACK );
 *   ...
 *   // Adds the text to the main form
 *   cFrameForm_AddObj( ptr_form, ptr_text, 42, 35 );
 *   ...
 *  @endcode
 */
void cCustomForm_AddObj (struct cCustomForm* ptr_custom_form, struct cObject* ptr_object, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_InsObj(ptr_custom_form, ptr_object, x, y, index)   \
        if((ptr_custom_form)->ClientArea)           \
        {  \
          cClip_InsObj((ptr_custom_form)->ClientArea, (ptr_object), (x), (y), (index)); \
        }
#else
/**
 *  Adds an object to the cCustomForm in position ( x, y ) with a z-position 'index' (maximal index is on the top)
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @param x The x-coordinate of the new object
 *  @param y The y-coordinate of the new object
 *  @param index The z-position of the new object
 *  @return None
 */
void cCustomForm_InsObj (struct cCustomForm* ptr_custom_form, struct cObject* ptr_object, int x, int y, int index);
#endif


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_RemObj(ptr_custom_form, ptr_object)   \
        if((ptr_custom_form)->ClientArea)           \
        {  \
          cClip_RemObj((ptr_custom_form)->ClientArea, (ptr_object)); \
        }
#else
/**
 *  Removes an object from a cCustomForm.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @return None
 */
void cCustomForm_RemObj (struct cCustomForm* ptr_custom_form, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_SelectFirst         cClip_SelectFirst
#else
/**
 *  Selects the first object in the cCustomForm.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return FALSE if the function failed
 */
bool cCustomForm_SelectFirst (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_SelectPrev          cClip_SelectPrev
#else
/**
 *  Selects the previous object in the cCustomForm.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the function failed
 */
bool cCustomForm_SelectPrev (struct cCustomForm* ptr_custom_form, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_SelectNext          cClip_SelectNext
#else
/**
 *  Selects the next object in the cCustomForm.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the function failed
 */
bool cCustomForm_SelectNext (struct cCustomForm* ptr_custom_form, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_Scroll              cClip_Scroll
#else
/**
 *  Scrolls a cCustomForm to make a rectangle visible. Provides minimal scrolling.
 *  @param ptr_custom_form A pointer to initialized cCustomForm structure
 *  @param rectangle A pointer to the initialized rect_t structure
 *  @return None
 *  @see cCustomForm_Scroll_Ex.
 */
void cCustomForm_Scroll (struct cCustomForm* ptr_custom_form, struct rect_t* rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_Scroll_Ex           cClip_Scroll_Ex
#else
/**
 *  Scrolls a cCustomForm by ( x, y ), but will not exceed the child's boundaries.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @param x "x" coordinate shift
 *  @param y "y" coordinate shift
 *  @return None
 *  @see cCustomForm_Scroll.
 */
void cCustomForm_Scroll_Ex (struct cCustomForm* ptr_custom_form, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_SendScroll          cClip_SendScroll
#else
/**
 *  Forces this object to redraw its scrolling arrows.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return None
 */
void cCustomForm_SendScroll (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_GetShifty           cClip_GetShifty
#else
/**
 *  Forces this object to redraw its scrolling arrows.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return 'y' coordinate shift (at result of scrolling)
 */
int cCustomForm_GetShifty (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_GetShiftx           cClip_GetShiftx
#else
/**
 *  Returns the 'x' coordinate shift (as a result of scrolling).
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return 'x' coordinate shift (as a result of scrolling)
 */
int cCustomForm_GetShiftx (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_GetCount            cClip_GetCount
#else
/**
 *  Returns the count of child objects.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return Child objects count
 */
int cCustomForm_GetCount (struct cCustomForm* ptr_custom_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_get_by_index        cClip_get_by_index
#else
/**
 *  Returns the object index.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @param Index of object
 *  @return A pointer to the object
 */
struct cObject* cCustomForm_get_by_index (struct cCustomForm* ptr_custom_form, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_FindObj             cClip_FindObj
#else
/**
 *  Returns the child object index in a cCustomForm (or -1 if the object was not found).
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @return The child object index in cCustomForm (or -1 if the object was not found)
 */
int cCustomForm_FindObj (struct cCustomForm* ptr_custom_form, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cCustomForm_GetSelectedObject   cClip_GetSelectedObject
#else
/**
 *  Returns the currently selected object.
 *  @param ptr_custom_form A pointer to the initialized cCustomForm structure
 *  @return The currently selected object
 */
struct cObject* cCustomForm_GetSelectedObject (struct cCustomForm* ptr_custom_form);
#endif

#ifdef _SKIP_DOC
};
#endif

/**  Dialog styles. Constants with value 1..0x8000 are buttons. Other constants are special style constants.
  *  @ingroup forms_group
*/
enum tDialogStyle
{
  ///  no buttons.
  mbNone     = 0x0,
  ///  "OK" button.
  mbOk       = 0x1,
  ///  "YES" button.
  mbYes      = 0x2,
  ///  "NO" button.
  mbNo       = 0x4,
  ///  "DRAW" button.
  mbDraw     = 0x8,
  ///  "CHAT" button.
  mbChat     = 0x10,
  ///  "QUIT" button.
  mbQuit     = 0x20,

  ///  "NEW" button.
  mbNew      = 0x20,
  ///  "UPLOAD" button.
  mbUpload   = 0x40,
  ///  "RENAME" button.
  mbRename   = 0x80,
  ///  "DELETE" button.
  mbDelete   = 0x100,
  ///  "VIEW" button.
  mbView     = 0x200,
  ///  "PLAY" button.
  mbPlay     = 0x400,
  ///  "RESTART" button.
  mbRestart  = 0x800,

  ///  "SET" button.
  mbSet      = 0x1000,

  ///  "LATER" button.
  mbLater    = 0x2000,
  ///  "CANCEL" button.
  mbCancel   = 0x4000,

  ///  edit string attached to dialog.
  mbEdit     = 0x10000,
  ///  dialog's edit is in password mode (all symbols are viewed as '*')
  mbPasswd   = 0x20000,
  ///  do not close dialog when <Esc> button is pressed. (normally, dialog will close with mrCancel ModalResult).
  mbNoEsc    = 0x40000,
  ///  set dialog text flush left flush left (normally, dialog text is centered).
  mbLeft     = 0x80000,
  ///  dialog should ignore "msg_shutup" message. Used in "program close" dialogs.
  mbNoShutup = 0x100000,
  ///  reverse the order of the buttons in the dialog (for example, <Cancel> moves to the left of <OK>).
  mbReverseOrder = 0x200000,
  ///  play default music when the dialog is shown.
  mbsDefault = 0x0,

  ///  don't play any music is playing when the dialog is shows
  mbsNone    = 0x0,

  ///  play Music 1 when the dialog is shown.
  mbs0       = 0x0,
  ///  play Music 2 when the dialog is shown.
  mbs1       = 0x400000,
  ///  play Music 3 when the dialog is shown.
  mbs2       = 0x800000,
  ///  play Music 4 when the dialog is shown.
  mbs3       = 0xC00000,
  ///  play Music 5 when the dialog is shown.
  mbs4       = 0x1000000,
  ///  play Music 6 when the dialog is shown.
  mbs5       = 0x1400000,
  ///  play Music 7 when the dialog is shown.
  mbs6       = 0x1800000,
  ///  play Music 8 when the dialog is shown.
  mbs7       = 0x1C00000,
  ///  shift the music fields.
  mbsMask    = 0x1C00000,
  ///  The same as mbs6
  mbsAbout   = mbs6,
  ///  The same as mbs4
  mbsDraw    = mbs4,
  ///  The same as mbs5
  mbsEsc     = mbs5,
  ///  The same as mbs1
  mbsLinkOff = mbs1,
  ///  The same as mbs7
  mbsNoDraw  = mbs7,
  ///  The same as mbs2
  mbsWelcome = mbs2
};

/**
 *   @ingroup forms_group
 *   @brief The structure for common dialogs.
 *
 *   This is a common dialog structure. You can create a various types of dialog windows
 *   using the third parameter of the cDialog_ctor function.
 *   <br>
 *   <br>
 *   Dialog styles are defined by combinations
 *   of constants of the "tDialogStyle" enumeration (see the "Enumerations" page). For example,
 *   you can create a combination like this: "mbOk | mbCancel | mbEdit". This dialog box
 *   contains "Ok" and "Cancel" buttons, plus editing field that has been defined by the "mbEdit" parameter.
 *   In this case, both the cButton_GetEditText and cButton_SetEditText functions are enabled.
 *   <br>
 *   <br>
 *   After creating a dialog box, you can show it (see cDialog_ShowModal),
 *   and then when a message comes in, you can process it by calling the cDialog_proc function.
 *   <br>
 *   <br>
 *   You must call the cDialog_ctor function before use and the cDialog_dtor function after use.
 *   @see forms_group , cButton_GetEditText , cButton_SetEditText
 */
struct cDialog : public cCustomForm
{
#ifndef _SKIP_DOC
  char cDialog_Members_STUB[16];
};
#endif

//============================
//=                          =
//=      object cDialog.     =
//=      Dialog window.      =
//=                          =
//============================

/**
 *  Initializes a cDialog object.
 *  @param ptr_dialog A pointer to a cDialog object
 *  @param caption Dialog caption
 *  @param text Dialog text
 *  @param style Dialog style. The style is a combination of "tDialogStyle" constants
 *  @param edit_size Edit string size (in symbols). If mbEdit not defined in the style, this parameter is not used
 *  @param ptr_win_app A pointer to the structure of the current cywin application
 *  @return A pointer to the initialized cDialog object
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, "Caption", "Dialog Text",
                     mbYes | mbNo, 0, main_module.m_process );
 *     ...
 *     cDialog_ShowModal( &dialog );
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 101
#endif
struct cDialog* cDialog_ctor(struct cDialog* ptr_dialog, char* caption, char* text,
                             long style, int edit_size, struct cWinApp* ptr_win_app);

/**
 *  Destructor. Deletes the cDialog object.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was
 *  allocated for the object by malloc(), use FREE_MEMORY.
 *  Use LEAVE_MEMORY if object was static or allocated in stack use LEAVE_MEMORY.
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, "Caption", "Dialog Text",
                     mbYes | mbNo, 0, main_module.m_process );
 *     ...
 *     cDialog_ShowModal( &dialog );
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 109
#endif
void cDialog_dtor(struct cDialog* ptr_dialog, int memory_flag);

/**
 *  Copies an edit string from the internal buffer into "ptr_buffer".
 *  You must provide enough space for the buffer.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param ptr_buffer A pointer to the allocated buffer
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     char sz_name[11];
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, "Name", "Enter your name:",
                     mbOk | mbCancel | mbEdit, 10, main_module.m_process );
 *     ...
 *     if ( cDialog_ShowModal( &dialog ) == mrOk )
 *     {
 *       cDialog_GetEditText( &dialog, sz_name );
 *     }
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 *  @see cDialog_SetEditText.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 113
#endif
void cDialog_GetEditText(struct cDialog* ptr_dialog, char* ptr_buffer);

/**
 *  Sets an initial string in an edit line.
 *  Does not work if the mbEdit style is not defined.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param text The text to show
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     char sz_name[11];
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, "Name", "Enter your name:",
                     mbOk | mbCancel | mbEdit, 10, main_module.m_process );
 *     ...
 *     cDialog_SetEditText( &dialog, "Bill Gates" );
 *     if ( cDialog_ShowModal( &dialog ) == mrOk )
 *     {
 *       cDialog_GetEditText( &dialog, sz_name );
 *     }
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 *  @see cDialog_GetEditText.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 114
#endif
void cDialog_SetEditText(struct cDialog* ptr_dialog, char* text);

/**
 *  Plays the standard dialog sound.
 *  @param number The sound's number (from mbs0 to mbs6)
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   {
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     char sz_name[11];
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, "Name", "Enter your name:",
                     mbOk | mbCancel | mbEdit | mbs1, 10, main_module.m_process );
 *     ...
 *     cDialog_SetEditText( &dialog, "Bill Gates" );
 *     if ( cDialog_ShowModal( &dialog ) == mrOk )
 *     {
 *       cDialog_GetEditText( &dialog, sz_name );
 *       if( !strlen( sz_name ) )
 *       {
 *         cDialog_PlaySound( mbs2 );
 *       }
 *     }
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 311
#endif
void cDialog_PlaySound(long number);

/**
 *  Message-processing function.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param ptr_message A pointer to the processed message
 *  @return TRUE if the message was processed
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, NULL, str_Really_exit,
                     mbQuit | mbCancel | mbs1, 0, main_module.m_process );
 *     ...
 *     cDialog_Show( &dialog );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if( cDialog_proc( &dialog, ptr_message ) && dialog.ModalResult != mrNone )
 *     {
 *       if( dialog.ModalResult == mrQuit )
 *       {
 *         ...
 *       }
 *     }
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 104
#endif
bool cDialog_proc(struct cDialog* ptr_dialog, struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_ShowModal           cCustomForm_ShowModal
#else
/**
 *  Shows a cDialog in modal mode. This function shows the cDialog and
 *  enters it into the message loop. The loop continues until any one
 *  ModalResult variable is set a to non-zero value, or until a MSG_QUIT
 *  or MSG_SHUTUP message is received. If these messages are received,
 *  the function reposts them to the application and leaves the message loop.
 *  This is how all forms will close when the application ends.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return tMResults value.
           mrNone(Zero) if MSG_QUIT or MSG_SHUTUP messages initiates
           leaving the loop. There are some predefined results
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     char sz_name[11];
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, "Name", "Enter your name:",
                     mbOk | mbCancel | mbEdit, 10, main_module.m_process );
 *     ...
 *     cDialog_SetEditText( &dialog, "Bill Gates" );
 *     if ( cDialog_ShowModal( &dialog ) == mrOk )
 *     {
 *       cDialog_GetEditText( &dialog, sz_name );
 *     }
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 *  @see tMResults , MSG_QUIT , MSG_SHUTUP.
 */
int cDialog_ShowModal (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_Disconnect          cObject_Disconnect
#else
/**
 *  Disconnects the cDialog from its parent object.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return None
 */
void cDialog_Disconnect (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_Select              cObject_Select
#else
/**
 *  Selects a cDialog object.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return TRUE if object was selected
 */
bool cDialog_Select (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_update              cObject_update
#else
/**
 *  Updates a cDialog object.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return None
 */
void cDialog_update (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_GetParent           cObject_GetParent
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return A pointer to the parent object
 */
struct cClip* cDialog_GetParent (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_Hide  cObject_Hide
#else
/**
 *  Hides the cDialog object.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *    ...
 *    struct cDialog dialog;
 *    struct module_t main_module;
 *    struct Message* ptr_message;
 *    ...
 *    init_module( &main_module );
 *    ...
 *    cDialog_ctor( &dialog, NULL, str_Really_exit,
                    mbQuit | mbCancel | mbs1, 0, main_module.m_process );
 *    ...
 *    cDialog_Show( &dialog );
 *    ...
 *    cDialog_Hide( &dialog );
 *    ...
 *    cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 *  @see cDialog_Show.
 */
void cDialog_Hide (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_Show                cObject_Show
#else
/**
 *  Shows a cDialog object.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, NULL, str_Really_exit,
                     mbQuit | mbCancel | mbs1, 0, main_module.m_process );
 *     ...
 *     cDialog_Show( &dialog );
 *     ...
 *     cDialog_Hide( &dialog );
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 *  @see cDialog_Hide.
 */
void cDialog_Show (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_Disable             cObject_Disable
#else
/**
 *  Disables cDialog object so that it cannot be selected
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *       ...
 *       struct cDialog dialog;
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cDialog_ctor( &dialog, NULL, str_Really_exit,
 *                     mbQuit | mbCancel | mbs1, 0, main_module.m_process );
 *       ...
 *       cDialog_Show( &dialog );
 *       ...
 *       cDialog_Disable( &dialog );
 *       ...
 *       cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 *  @see cDialog_Enable.
 */
void cDialog_Disable (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_Enable              cObject_Enable
#else
/**
 *  Enables the cDialog object so that it may be selected.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cDialog dialog;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cDialog_ctor( &dialog, NULL, str_Really_exit,
 *                   mbQuit | mbCancel | mbs1, 0, main_module.m_process );
 *     ...
 *     cDialog_Show( &dialog );
 *     ...
 *     cDialog_Disable( &dialog );
 *     cDialog_Enable ( &dialog );
 *     ...
 *     cDialog_dtor( &dialog, LEAVE_MEMORY );
 *  @endcode
 *  @see cDialog_Disable.
 */
void cDialog_Enable (struct cDialog* ptr_dialog);
#endif


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_AddObj(ptr_dialog, ptr_object, x, y)   \
        if((ptr_dialog)->ClientArea)           \
        {  \
          cClip_AddObj((ptr_dialog)->ClientArea, (ptr_object), (x), (y)); \
        }
#else
/**
 *  Adds an object to the cDialog in position (x, y).
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @param x The x-coordinate of the new object
 *  @param y The y-coordinate of the new object
 *  @return None
 */
void cDialog_AddObj (struct cDialog* ptr_dialog, struct cObject* ptr_object, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_InsObj(ptr_dialog, ptr_object, x, y, index)   \
        if((ptr_dialog)->ClientArea)           \
        {  \
          cClip_InsObj((ptr_dialog)->ClientArea, (ptr_object), (x), (y), (index)); \
        }
#else
/**
 *  Adds an object to the cDialog in position (x, y) with a z-position index (maximal index is on the top).
 *  @param ptr_dialog A pointer to the initialized cDialog object.
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @param x The x-coordinate of the new object
 *  @param y The y-coordinate of the new object
 *  @param index The z-position of the new object
 *  @return None
 */
void cDialog_InsObj (struct cDialog* ptr_dialog, struct cObject* ptr_object, int x, int y, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_RemObj(ptr_dialog, ptr_object)   \
        if((ptr_dialog)->ClientArea)           \
        {  \
          cClip_RemObj((ptr_dialog)->ClientArea, (ptr_object)); \
        }
#else
/**
 *  Removes an object from the cDialog
 *  @param ptr_dialog A pointer to the initialized cDialog object.
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @return None
 */
void cDialog_RemObj (struct cClip* ptr_dialog, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_SelectFirst         cClip_SelectFirst
#else
/**
 *  Selects the first object in the cDialog.
 *  @param ptr_dialog A pointer to the initialized cDialog object.
 *  @return FALSE if the function failed.
 */
bool cDialog_SelectFirst (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_SelectPrev cClip_SelectPrev
#else
/**
 *  Selects the previous object in the cDialog.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param round TRUE if you need to select the last object after the first
 *  @return FALSE if the function failed
 */
bool cDialog_SelectPrev (struct cDialog* ptr_dialog, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_SelectNext          cClip_SelectNext
#else
/**
 *  Selects the next object in the cDialog.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param round TRUE if you need to select the last object after the first
 *  @return FALSE if the function failed.
 */
bool cDialog_SelectNext (struct cDialog* ptr_dialog, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_Scroll              cClip_Scroll
#else
/**
 *  Scrolls cDialog to make rectangle visible. Provides minimal scrolling.
 *  @param ptr_dialog A pointer to the initialized cDialog object.
 *  @param rectangle A pointer to the initialized rect_t object.
 *  @return None
 *  @see cDialog_Scroll_Ex.
 */
void cDialog_Scroll (struct cDialog* ptr_dialog, struct rect_t* rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_Scroll_Ex           cClip_Scroll_Ex
#else
/**
 *  Scrolls a cDialog by ( x, y ), but cannot exceed the child's boundaries.
 *  @param ptr_dialog A pointer to the initialized cDialog object.
 *  @param x "x" coordinate shift
 *  @param y "y" coordinate shift
 *  @return None
 *  @see cDialog_Scroll.
 */
void cDialog_Scroll_Ex (struct cDialog* ptr_dialog, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_SendScroll          cClip_SendScroll
#else
/**
 *  Forces this object to redraw its scrolling arrows.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return None
 */
void cDialog_SendScroll (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_GetShifty           cClip_GetShifty
#else
/**
 *  Returns the y coordinate shift (as a result of scrolling)
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return y coordinate shift (as a result of scrolling)
 */
int cDialog_GetShifty (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_GetShiftx           cClip_GetShiftx
#else
/**
 *  Returns the x coordinate shift (as a result of scrolling).
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return x coordinate shift (as a result of scrolling)
 */
int cDialog_GetShiftx (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_GetCount            cClip_GetCount
#else
/**
 *  Returns the count of child objects.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return Child objects count
 */
int cDialog_GetCount (struct cDialog* ptr_dialog);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_get_by_index        cClip_get_by_index
#else
/**
 *  Returns the object's index (or NULL).
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param index The object's index
 *  @return Object's index (or NULL)
 */
struct cObject* cDialog_get_by_index (struct cDialog* ptr_dialog, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_FindObj             cClip_FindObj
#else
/**
 *  Returns the child object index in the cDialog (or -1).
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @param ptr_object A pointer to the initialized cObject object
 *  @return Child object index in cDialog (or -1)
 */
int cDialog_FindObj (struct cDialog* ptr_dialog, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cDialog_GetSelectedObject   cClip_GetSelectedObject
#else
/**
 *  Returns the currently selected object.
 *  @param ptr_dialog A pointer to the initialized cDialog object
 *  @return Currently selected object
 */
struct cObject* cDialog_GetSelectedObject (struct cDialog* ptr_dialog);
#endif

#ifdef _SKIP_DOC
};
#endif


/**
 *  @ingroup forms_group
 *  @brief This structure creates a full-screen, framed window
 *
 *  This is the structure for creating a full-screen window with a frame.
 *   <br>
 *   <br>
 *  After creating the form, you can show it (see cFrameForm_ShowModal)
 *  and then when a message comes in, you can process it by calling the cDialog_proc function.
 *   <br>
 *   <br>
 *  You must call the cFrameForm_ctor function before use and cFrameForm_dtor function after use.
 *  @see forms_group , cDialog_proc
 */
struct cFrameForm : public cCustomForm
{
#ifndef _SKIP_DOC
  char cFrameForm_Members_STUB[4];
};
#endif

//============================
//=                          =
//=     object cFrameForm.   =
//=    Full-screen window    =
//=       with frame.        =
//=                          =
//============================

/**
 *  Constructor. Creates the full-screen window form.
 *  @param ptr_frame_form A pointer to a cFrameForm object
 *  @param text The form's caption
 *  @param ptr_win_app A pointer to the structure of the current cywin application
 *  @return A pointer to the initialized cFrameForm object
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cFrameForm frame_form;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cFrameForm_ctor( &frame_form, "Caption", main_module.m_process );
 *     ...
 *     // Adding some objects to the custom form.
 *     ...
 *     cFrameForm_ShowModal( &frame_form );
 *     ...
 *     cFrameForm_dtor( &frame_form, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 96
#endif
struct cFrameForm* cFrameForm_ctor(struct cFrameForm* ptr_frame_form, char* text, struct cWinApp* ptr_win_app);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_dtor                cCustomForm_dtor
#else
/**
 *  Deletes the cFrameForm object
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cFrameForm frame_form;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cFrameForm_ctor( &frame_form, "Caption", main_module.m_process );
 *     ...
 *     // Adding some objects to the custom form.
 *     ...
 *     cFrameForm_ShowModal( &frame_form );
 *     ...
 *     cFrameForm_dtor( &frame_form, LEAVE_MEMORY );
 *  @endcode
 */
void cFrameForm_dtor (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_ShowModal           cCustomForm_ShowModal
#else
/**
 *  Shows a cFrameForm in modal mode.
 *  This function shows the cFrameForm and enters it into the message loop.
 *  The loop continues until any one ModalResult variable is set to
 *  a non-zero value, or until a MSG_QUIT or MSG_SHUTUP messages is received.
 *  If these messages are received, the function reposts them to the application
 *  and leaves the message loop. This is how all forms will close when the application ends.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return tMResults value.
 *          mrNone(Zero) if MSG_QUIT or MSG_SHUTUP messages initiates
 *          leaving the loop. There are some predefined results
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct cFrameForm frame_form;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cFrameForm_ctor( &frame_form, "Caption", main_module.m_process );
 *     ...
 *     // Adding some objects to the custom form.
 *     ...
 *     cFrameForm_ShowModal( &frame_form );
 *     ...
 *     cFrameForm_dtor( &frame_form, LEAVE_MEMORY );
 *  @endcode
 *  @see tMResults , MSG_QUIT , MSG_SHUTUP
 */
int cFrameForm_ShowModal (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_proc                cCustomForm_proc
#else
/**
 *  The Message-processing function.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @param ptr_message A pointer to the processed message
 *  @return TRUE if the message was processed
 *  @code
 *   #include <cywin.h>
 *    ...
 *    bool cMainForm_proc( struct cFrameForm* ptr_form, struct Message* ptr_message )
 *    {
 *      switch( ptr_message->msgid )
 *      {
 *        case MSG_SHUTUP: // Processes the system exit signal
 *        case MSG_QUIT:
 *        ...
 *        ptr_form->ModalResult = mrQuit;
 *        ...
 *        break;
 *        ...
 *        case MSG_KEYDOWN: // Processes the keyboard
 *        ...
 *        switch( Message_get_key_param( ptr_message )->scancode )
 *        {
 *          case KEY_ESC:
 *          ...
 *          ptr_form->ModalResult = mrQuit;
 *          ...
 *          return TRUE;
 *          ...
 *          case KEY_ENTER:
 *          ...
 *          // ModalResult containing the selected menu item
 *          ptr_form->ModalResult = mrOk;
 *          ...
 *          return TRUE;
 *        }
 *      }
 *      return cFrameForm_proc( ptr_form, ptr_message );
 *    }
 *    ...
 *  @endcode
 */
bool cFrameForm_proc (struct cFrameForm* ptr_frame_form, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_Disconnect          cObject_Disconnect
#else
/**
 *  Disconnects a cFrameForm from its parent object.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return None
 */
void cFrameForm_Disconnect (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_Select              cObject_Select
#else
/**
 *  Selects a cFrameForm object.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return TRUE if the object was selected.
 */
bool cFrameForm_Select (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_update              cObject_update
#else
/**
 *  Updates a cFrameForm object.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *    ...
 *    int cMainForm_ShowModal( struct cFrameForm* ptr_form )
 *    {
 *      ptr_form->HelpContext = 0;
 *      ...
 *      cFrameForm_Show( ptr_form );
 *      ...
 *      ptr_form->ModalResult = mrNone;
 *      ...
 *      while( ptr_form->ModalResult == mrNone )
 *      {
 *        struct Message* ptr_message =
 *          cWinApp_get_message( ptr_form->CurrApplication, 0, 1, MSG_USER );
 *        ...
 *        cMainForm_proc( ptr_form, ptr_message );
 *        ...
 *        cFrameForm_update( ptr_form );
 *        ...
 *        Message_delete( ptr_message );
 *      }
 *      return ptr_form->ModalResult;
 *    }
 *    ...
 *  @endcode
 */
void cFrameForm_update (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_GetParent           cObject_GetParent
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return A pointer to the parent object
 */
struct cClip* cFrameForm_GetParent (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_Hide                cObject_Hide
#else
/**
 *  Hides a cFrameForm object.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *    ...
 *    // Application's main form
 *    struct cFrameForm* ptr_main_form;
 *    ...
 *    // Selects the effect
 *    cMainForm_ShowModal( ptr_main_form );
 *    ...
 *    cFrameForm_Hide( ptr_main_form );
 *    ...
 *  @endcode
 *  @see cFrameForm_Show.
 */
void cFrameForm_Hide (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_Show                cObject_Show
#else
/**
 *  Shows a cFrameForm object.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   int cMainForm_ShowModal( struct cFrameForm* ptr_form )
 *   {
 *     cFrameForm_Show( ptr_form );
 *     ...
 *     ptr_form->ModalResult = mrNone;
 *     ...
 *     while( ptr_form->ModalResult == mrNone  &&
 *       ptr_menu_form->ModalResult != mrQuit )
 *     {
 *      struct Message *ptr_message =
 *        cWinApp_get_message( ptr_form->CurrApplication,
 *          0, 1, MSG_USER );
 *      ...
 *      cMainForm_proc( ptr_form, ptr_message );
 *      ...
 *      Message_delete( ptr_message );
 *     }
 *     return ptr_form->ModalResult;
 *   }
 *   ...
 *  @endcode
 *  @see cFrameForm_Hide.
 */
void cFrameForm_Show (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_Disable             cObject_Disable
#else
/**
 *  Disables a cFrameForm object so that it cannot be selected.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return None
 *  @see cFrameForm_Enable.
 */
void cFrameForm_Disable (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_Enable              cObject_Enable
#else
/**
 *  Enables a cFrameForm object so that it may be selected
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return None
 *  @see cFrameForm_Disable.
 */
void cFrameForm_Enable (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_AddObj(ptr_frame_form, ptr_object, x, y)   \
        if((ptr_frame_form)->ClientArea)           \
        {  \
          cClip_AddObj((ptr_frame_form)->ClientArea, (ptr_object), (x), (y)); \
        }
#else
/**
 *  Adds an object to the cFrameForm in position (x, y).
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object.
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @param x x-coordinate of the new object
 *  @param y y-coordinate of the new object
 *  @return None
 */
void cFrameForm_AddObj (struct cFrameForm* ptr_frame_form, struct cObject* ptr_object, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_InsObj(ptr_frame_form, ptr_object, x, y, index)   \
        if((ptr_frame_form)->ClientArea)           \
        {  \
          cClip_InsObj((ptr_frame_form)->ClientArea, (ptr_object), (x), (y), (index)); \
        }
#else
/**
 *  Adds an object to the cFrameForm in a position ( x, y )
    with a z-position index (maximal index is on the top).
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @param ptr_object A pointer to initialized cObject structure
 *  @param x The x-coordinate of the new object
 *  @param y The y-coordinate of the new object
 *  @param index z-position of the new object
 *  @return None
 */
void cFrameForm_InsObj (struct cFrameForm* ptr_frame_form, struct cObject* ptr_object, int x, int y, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_RemObj(ptr_frame_form, ptr_object)   \
        if((ptr_frame_form)->ClientArea)           \
        {  \
          cClip_RemObj((ptr_frame_form)->ClientArea, (ptr_object)); \
        }
#else
/**
 *  Removes an object from a cFrameForm.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object.
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @return None
 */
void cFrameForm_RemObj (struct cClip* ptr_frame_form, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_SelectFirst         cClip_SelectFirst
#else
/**
 *  Selects the first object in the cFrameForm.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return FALSE if the function failed
 */
bool cFrameForm_SelectFirst (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_SelectPrev          cClip_SelectPrev
#else
/**
 *  Selects the previous object in the cFrameForm.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the function failed
 */
bool cFrameForm_SelectPrev (struct cFrameForm* ptr_frame_form, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_SelectNext          cClip_SelectNext
#else
/**
 *  Selects the next object in the cFrameForm.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @param round is TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the function failed
 */
bool cFrameForm_SelectNext (struct cFrameForm* ptr_frame_form, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_Scroll              cClip_Scroll
#else
/**
 *  Scrolls a cFrameForm to make a rectangle visible. Provides minimal scrolling.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @param rectangle A pointer to the initialized rect_t structure
 *  @return None
 *  @see cFrameForm_Scroll_Ex.
 */
void cFrameForm_Scroll (struct cFrameForm* ptr_frame_form, struct rect_t* rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_Scroll_Ex           cClip_Scroll_Ex
#else
/**
 *  Scrolls a cFrameForm by (x, y), but cannot exceed the child's boundaries.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @param x "x" coordinate shift
 *  @param y "y" coordinate shift
 *  @return None
 *  @see cFrameForm_Scroll.
 */
void cFrameForm_Scroll_Ex (struct cFrameForm* ptr_frame_form, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_SendScroll          cClip_SendScroll
#else
/**
 *  Forces this object to redraw its scrolling arrows.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return None
 */
void cFrameForm_SendScroll (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_GetShifty           cClip_GetShifty
#else
/**
 *  Returns the 'y' coordinate shift (as a result of scrolling).
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return 'y' coordinate shift (as a result of scrolling)
 */
int cFrameForm_GetShifty (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_GetShiftx           cClip_GetShiftx
#else
/**
 *  Returns the 'x' coordinate shift (as a result of scrolling).
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return 'x' coordinate shift (as a result of scrolling)
 */
int cFrameForm_GetShiftx (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_GetCount            cClip_GetCount
#else
/**
 *  Returns the count of child objects.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return Child objects count
 */
int cFrameForm_GetCount (struct cFrameForm* ptr_frame_form);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_get_by_index        cClip_get_by_index cClip_get_by_index
#else
/**
 *  Returns an object's  index (or NULL).
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @param index Object's index
 *  @return Object's index (or NULL)
 */
struct cObject* cFrameForm_get_by_index (struct cFrameForm* ptr_frame_form, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_FindObj             cClip_FindObj
#else
/**
 *  Returns the child object's index in the cFrameForm (or -1 ).
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @param ptr_object A pointer to the initialized cObject object
 *  @return The child object's index in the cFrameForm (or -1 )
 */
int cFrameForm_FindObj (struct cFrameForm* ptr_frame_form, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cFrameForm_GetSelectedObject   cClip_GetSelectedObject
#else
/**
 *  Returns the currently selected object.
 *  @param ptr_frame_form A pointer to the initialized cFrameForm object
 *  @return The currently selected object
 */
struct cObject* cFrameForm_GetSelectedObject (struct cFrameForm* ptr_frame_form);
#endif

#ifdef _SKIP_DOC
};
#endif

#endif  //  #ifndef __FORM_H_INCLUDE__
