/** @file highscore.h
 * Functions for working with the HighScore table.
 * For more details see cy_spider example.
 */

#ifndef __HighScore_H__
#define __HighScore_H__

#include <cywin.h>

/**
 * Result of call function highscore_enter
 */
typedef enum
{
   HSR_SHOW    = 0, ///< Success, show HighScore table
   HSR_RESTART = 1, ///< Restart key pressed
   HSR_QUIT    = 2, ///< Quit key pressed
   HSR_NONE    = 3, ///< Esc key pressed
   HSR_INVITE  = 4  ///< Functional key pressed 
}
highscores_result_t;

/**
 * Type of parameter by calling up the function highscore_enter
 */
typedef enum
{
   HSM_CHECKSHOWALL      = 0, ///< Check and show HighScore 
                              ///< and (if "no enter" to HS table) 
                              ///< dialog "U no enter..."
   HSM_CHECKSHOWTABLE    = 1, ///< Check and show the HS table only
   HSM_CHECK             = 2, ///< Check and (if "enter" to HS table)
                              ///< then save in table as "unknown"
   HSM_SHOW              = 3  ///< only show without checking
}
highscores_mode_t;
            

/**
 * Calls up constructors for graphics and music.Initializes HighScore 
 * struct from file score.inf.
 * @param ptr_main_module pointer to the application's
 *  main module   
 * @return TRUE if file score.inf is valid
 *  @code
 *      #include <cybiko.h>
 *      #include <highscore.h>
 *      ...
 *      struct module_t main_module;
 *      long current_score;
 *      int exit_app = FALSE;
 *      ...
 *      init_module( &main_module );
 *      highscore_init( &main_module );
 *      ...
 *      while( !exit_app )  // The application's main loop
 *      {
 *        pMessage = cWinApp_get_message( MainModule.m_process, 1, 1, MSG_USER );
 *        if ( pMessage )
 *        {
 *          switch( pMessage->msgid )
 *          {
 *            ...
 *            case MSG_SHUTUP:   
 *            case MSG_QUIT:
 *              exit_app = TRUE;
 *              highscore_enter( &main_module, current_score, HSM_CHECK );
 *              break;
 *            case MSG_KEYDOWN:
 *              pKeyParam = Message_get_key_param( pMessage );
 *              switch( pKeyParam->scancode )
 *              {
 *                 case KEY_ESC:
 *                   if( highscore_enter( &main_module, current_score, 
 *                                        HSM_CHECKSHOWTABLE ) == HSR_INVITE )
 *                   {
 *                     exit_app = TRUE;
 *                   }
 *                   else
 *                   {
 *                      // do something...
 *                   }
 *                   break;
 *                 case KEY_TAB
 *                   if( highscore_enter( &main_module, current_score, 
 *                                        HSM_SHOW ) == HSR_INVITE )
 *                   {
 *                     exit_app = TRUE;
 *                   }
 *                   break;
 *                 ...
 *              } 
 *              break;
 *            ...
 *          } 
 *          Message_delete( pMessage );
 *        }
 *      } //while( !exit_app )
 *      highscore_cleanup( &main_module );
 *      ...
 *  @endcode
 */
__bytecode bool highscore_init( struct module_t * ptr_main_module );

/**
 * Calls up desstructors for graphics and music. Stores HighScores struct
 * in file score.inf.
 * @param ptr_main_module pointer to the application's
 *  main module   
 * @return TRUE if file score.inf is valid
 *  @code
 *      #include <cybiko.h>
 *      #include <highscore.h>
 *      ...
 *      struct module_t main_module;
 *      long current_score;
 *      int exit_app = FALSE;
 *      ...
 *      init_module( &main_module );
 *      highscore_init( &main_module );
 *      ...
 *      while( !exit_app )  // The application's main loop
 *      {
 *        pMessage = cWinApp_get_message( MainModule.m_process, 1, 1, MSG_USER );
 *        if ( pMessage )
 *        {
 *          switch( pMessage->msgid )
 *          {
 *            ...
 *            case MSG_SHUTUP:   
 *            case MSG_QUIT:
 *              exit_app = TRUE;
 *              highscore_enter( &main_module, current_score, HSM_CHECK );
 *              break;
 *            case MSG_KEYDOWN:
 *              pKeyParam = Message_get_key_param( pMessage );
 *              switch( pKeyParam->scancode )
 *              {
 *                 case KEY_ESC:
 *                   if( highscore_enter( &main_module, current_score, 
 *                                        HSM_CHECKSHOWTABLE ) == HSR_INVITE )
 *                   {
 *                     exit_app = TRUE;
 *                   }
 *                   else
 *                   {
 *                      // do something...
 *                   }
 *                   break;
 *                 case KEY_TAB
 *                   if( highscore_enter( &main_module, current_score, 
 *                                        HSM_SHOW ) == HSR_INVITE )
 *                   {
 *                     exit_app = TRUE;
 *                   }
 *                   break;
 *                 ...
 *              } 
 *              break;
 *            ...
 *          } 
 *          Message_delete( pMessage );
 *        }
 *      } //while( !exit_app )
 *      highscore_cleanup( &main_module );
 *      ...
 *  @endcode
 */
__bytecode void highscore_cleanup( struct module_t * ptr_main_module );

/**
 * This function shows the HighScore table on the 
 * Cybiko computer screen, updates HighScore table, launches events.
 * @param ptr_main_module pointer to the application's
 *  main module
 * @param current_score user's current score
 * @param mode parameter dispatches this function
 * @return Status of exit.
 *  @code
 *      #include <cybiko.h>
 *      #include <highscore.h>
 *      ...
 *      struct module_t main_module;
 *      long current_score;
 *      int exit_app = FALSE;
 *      ...
 *      init_module( &main_module );
 *      highscore_init( &main_module );
 *      ...
 *      while( !exit_app )  // The application's main loop
 *      {
 *        pMessage = cWinApp_get_message( MainModule.m_process, 1, 1, MSG_USER );
 *        if ( pMessage )
 *        {
 *          switch( pMessage->msgid )
 *          {
 *            ...
 *            case MSG_SHUTUP:   
 *            case MSG_QUIT:
 *              exit_app = TRUE;
 *              highscore_enter( &main_module, current_score, HSM_CHECK );
 *              break;
 *            case MSG_KEYDOWN:
 *              pKeyParam = Message_get_key_param( pMessage );
 *              switch( pKeyParam->scancode )
 *              {
 *                 case KEY_ESC:
 *                   if( highscore_enter( &main_module, current_score, 
 *                                        HSM_CHECKSHOWTABLE ) == HSR_INVITE )
 *                   {
 *                     exit_app = TRUE;
 *                   }
 *                   else
 *                   {
 *                      // do something...
 *                   }
 *                   break;
 *                 case KEY_TAB
 *                   if( highscore_enter( &main_module, current_score, 
 *                                        HSM_SHOW ) == HSR_INVITE )
 *                   {
 *                     exit_app = TRUE;
 *                   }
 *                   break;
 *                 ...
 *              } 
 *              break;
 *            ...
 *          } 
 *          Message_delete( pMessage );
 *        }
 *      } //while( !exit_app )
 *      highscore_cleanup( &main_module );
 *      ...
 *  @endcode
 * @see highscores_mode_t, highscores_result_t
 */
__bytecode int  highscore_enter(struct module_t * ptr_main_module,
                     long current_score , int mode );
#endif