/** 
 *  @file keybd.h
 *  A simple way to check if a key is pressed.
 */

#ifndef __KEYBD_H_INCLUDED__
#define __KEYBD_H_INCLUDED__

/**
 *  No click sound.
 * @ingroup sound_overview
 */
#define CK_NO          0 

/**
 *  Normal click sound.
 * @ingroup sound_overview
 */
#define CK_BEEP        1

/**
 *  Short click sound.
 * @ingroup sound_overview
 */
#define CK_BEEP_SHORT  2

/**
 *  @ingroup keyboard_overview
 *  @brief a DirectKeyboard object scans the CYBIKO keyboard to determine what keys the user has pressed.
 *
 *  Use a DirectKeyboard object to scan the CYBIKO keyboard and determine 
 *  what keys are being pressed. You must call the DirectKeyboard_get_instance()
 *  function before use and the DirectKeyboard_dtor() function after use.
 *  @see keyboard_overview
 */
struct DirectKeyboard
{
#ifndef _SKIP_DOC
  char DirectKeyboard_Members_STUB[4];
};
#endif 

/**
 *  Deletes a DirectKeyboard object.
 *  @param ptr_direct_keyboard A pointer to the initialized DirectKeyboard object
 *  @param memory_flag THE MEMORY FLAG MUST BE FREE_MEMORY. 
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct DirectKeyboard* ptr_direct_keyboard;
 *       ptr_direct_keyboard = DirectKeyboard_get_instance();
 *       ...
 *       DirectKeyboard_scan( ptr_direct_keyboard );
 *       if ( DirectKeyboard_is_key_pressed( ptr_direct_keyboard, KEY_ESC ) )
 *       ...
 *       DirectKeyboard_dtor( ptr_direct_keyboard, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "keybd" 0
#endif
void DirectKeyboard_dtor(struct DirectKeyboard* ptr_direct_keyboard, int memory_flag);

/**
 *  Initializes a DirectKeyboard object.
 *  @return A pointer to the initialized DirectKeyboard object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct DirectKeyboard* ptr_direct_keyboard;
 *       ptr_direct_keyboard = DirectKeyboard_get_instance();
 *       ...
 *       DirectKeyboard_scan( ptr_direct_keyboard );
 *       if ( DirectKeyboard_is_key_pressed( ptr_direct_keyboard, KEY_ESC ) )
 *       ...
 *       DirectKeyboard_dtor( ptr_direct_keyboard, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "keybd" 2
#endif
struct DirectKeyboard* DirectKeyboard_get_instance(void);

/**
 *  Scans the CYBIKO keyboard to determine what keys are being pressed.
 *  @param ptr_direct_keyboard A pointer to the initialized DirectKeyboard object
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct DirectKeyboard* ptr_direct_keyboard;
 *       ptr_direct_keyboard = DirectKeyboard_get_instance();
 *       ...
 *       DirectKeyboard_scan( ptr_direct_keyboard );
 *       if ( DirectKeyboard_is_key_pressed( ptr_direct_keyboard, KEY_ESC ) )
 *       ...
 *       DirectKeyboard_dtor( ptr_direct_keyboard, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see DirectKeyboard_is_key_pressed
 */
#ifndef _SKIP_DOC
__cyos "keybd" 4
#endif
void DirectKeyboard_scan(struct DirectKeyboard* ptr_direct_keyboard);

/**
 *  Checks whether a specified key is being pressed.
 *  <b>The DirectKeyboard_scan function must be called before using this function!</b>
 *  @param ptr_direct_keyboard A pointer to the initialized DirectKeyboard object
 *  @param scancode The scancode of the tested key
 *  @return TRUE if the specified key is pressed
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct DirectKeyboard* ptr_direct_keyboard;
 *       ptr_direct_keyboard = DirectKeyboard_get_instance();
 *       ...
 *       DirectKeyboard_scan( ptr_direct_keyboard );
 *       if( DirectKeyboard_is_key_pressed( ptr_direct_keyboard, KEY_ESC ) )
 *       ...
 *       DirectKeyboard_dtor( ptr_direct_keyboard, FREE_MEMORY );
 *  @endcode
 *  @see DirectKeyboard_scan
 */
#ifndef _SKIP_DOC
__cyos "keybd" 5
#endif
bool DirectKeyboard_is_key_pressed(struct DirectKeyboard* ptr_direct_keyboard, short scancode);

#ifdef _SKIP_DOC
};
#endif 

/**
 *  @ingroup keyboard_overview
 *  Returns state of a key clicks prohibition.
 *  @return TRUE if the key click is not prohibited.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       bool clicks_enabled;
 *       ...
 *       clicks_enabled = get_clicks_enabled();
 *       // Turns off the keys clicks.
 *       set_clicks_enabled( FALSE );
 *       ...
 *       // Restors the keys clicks.
 *       set_clicks_enabled( clicks_enabled );
 *  @endcode
 *  @see set_clicks_enabled.
 */
#ifndef _SKIP_DOC
__cyos 654
#endif
bool get_clicks_enabled(void);

/**
 *  @ingroup keyboard_overview
 *  Disables or enables keyboard clicks.
 *  @param enable if TRUE enables all keyboard clicks, else disables
 *  keyboard clicks in accordance to procedure disable_key_click().
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       bool clicks_enabled;
 *       ...
 *       clicks_enabled = get_clicks_enabled();
 *       // Turns off the keys clicks.
 *       set_clicks_enabled( FALSE );
 *       ...
 *       // Restors the keys clicks.
 *       set_clicks_enabled( clicks_enabled );
 *  @endcode
 *  @see get_clicks_enabled.
 */
#ifndef _SKIP_DOC
__cyos 1102
#endif
void set_clicks_enabled(bool enable);

/**
 *  @ingroup keyboard_overview
 *  Disables the click sound for the keys specified, enables the click sound for the rest keys.
 *  To enable the click sound for all the keys, call disable_key_click() with key_list and key_count set to 0.
 *  @param ptr_key_list an array of keys virtual codes.
 *  @param key_count the number of elements in this array.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       static const int arrows_keys[4] = { 
 *                                           KEY_UP, 
 *                                           KEY_DOWN, 
 *                                           KEY_LEFT, 
 *                                           KEY_RIGHT
 *                                         };
 *       ...
 *       // Disables the click sound for the arrows keys.
 *       disable_key_click( arrows_keys, 4 );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 520
#endif
void disable_key_click(int* ptr_key_list, int key_count);

/**
 *  @ingroup keyboard_overview
 *  Returns the time (in the time_t format) of the last keystroke.
 *  @return The last keystroke time.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Time time;
 *       ...
 *       Time_decode( &time, last_key_time() );
 *       TRACE("Last key was pressed at hour: %d, minutes: %d, seconds: %d",
 *              time.hour, time.minute, time.second );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 831
#endif
time_t last_key_time(void);

/**
 *  @ingroup keyboard_overview
 *  Returns current kind of click sound.
 *  @return The current kind of click sound.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       click_kind_t click_type;
 *       ...
 *       click_type = get_click_kind();
 *       // 'click_type' might have parameters as follows: CK_BEEP_SHORT, CK_BEEP, CK_NO.
 *       if ( click_type != CK_BEEP_SHORT )
 *            set_click_kind( CK_BEEP_SHORT );
 *       ...
 *  @endcode
 *  @see set_click_kind, CK_BEEP_SHORT, CK_BEEP, CK_NO.
 */
#ifndef _SKIP_DOC
__cyos 653
#endif
click_kind_t get_click_kind(void);

/**
 *  @ingroup keyboard_overview
 *  Sets current kind of click sound.
 *  @param kind must be one of the following: CK_NO, CK_BEEP, or CK_BEEP_SHORT.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       click_kind_t click_type;
 *       ...
 *       click_type = get_click_kind();
 *       // 'click_type' might have parameters as follows: CK_BEEP_SHORT, CK_BEEP, CK_NO.
 *       if ( click_type != CK_BEEP_SHORT )
 *            set_click_kind( CK_BEEP_SHORT );
 *       ...
 *  @endcode
 *  @see get_click_kind, CK_BEEP_SHORT, CK_BEEP, CK_NO.
 */
#ifndef _SKIP_DOC
__cyos 1101
#endif
void set_click_kind(click_kind_t kind);

/**
 *  @ingroup keyboard_overview
 *  Returns TRUE if "sticking keys" option is enabled.
 *  @return TRUE if "sticking keys" option is enabled.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if ( !is_sticking_keys_enabled() )
 *          enable_sticking_keys( TRUE );
 *  @endcode
 *  @see enable_sticking_keys.
 */
#ifndef _SKIP_DOC
__cyos 822
#endif
bool is_sticking_keys_enabled(void);

/**
 *  @ingroup keyboard_overview
 *  Enables or disables "sticking keys" option.
 *  @param enable if TRUE, "sticking keys" option is enabled, else 
 *  "sticking keys" option is disabled.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if ( !is_sticking_keys_enabled() )
 *          enable_sticking_keys( TRUE );
 *  @endcode
 *  @see is_sticking_keys_enabled.
 */
#ifndef _SKIP_DOC
__cyos 540
#endif
void enable_sticking_keys(bool enable);

/**
 *  @ingroup keyboard_overview
 *  Enables or disables function key processing.
 *  @param enable if TRUE enables function key processing, else disables ones.
 *  @return Previous state.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       bool function_keys;
 *       ...
 *       //  Disables key function processing.  
 *       function_keys = enable_function_keys(FALSE);
 *       ...
 *       //  Restores key function processing state. 
 *       enable_function_keys(function_keys);
 *       ...  
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "servers" 11
#endif
bool enable_function_keys(bool enable);

#endif  //  #ifndef __KEYBD_H_INCLUDED__

