/** @file lists.h
 *  Hooks to cywin cList , cItem, cXItem, cXStr, cSItem and cXByte classes.
 */
#ifndef __LISTS_H_INCLUDED__
#define __LISTS_H_INCLUDED__

/**
 *   @ingroup controls_group
 *   @brief This structure implements methods for working with a list and a menu.
 *
 *   This structure implements methods for working with a list and a menu. A cList
 *   object is a container for cItem, cXItem and cSItem objects.
 *   <br>
 *   <br>
 *   To add these objects to a list, use the cList_AddItem function. To remove objects from
 *   a list or a menu use the cList_RemItem function.
 *   <br>
 *   <br>
 *   When a message comes in, process it using the cList_proc function.
 *   <br>
 *   <br>
 *   You must call the cList_ctor function before use and the cList_dtor function after use.
 *   @see controls_group
 */
struct cList : public cClip
{
#ifndef _SKIP_DOC
};
#endif

//============================
//=                          =
//=      object cList.       =
//=      Various menus.      =
//=                          =
//============================
/**
 *  Constructor. Creates an empty cList object.
 *  @param ptr_clist A pointer to the cList structure
 *  @param width The width of a cList object.
 *  @return A pointer to the initialized cList object
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     int index;
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[3] = {"Start",
 *                              "High Score",
 *                              "Exit"
 *                              };
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ));
 *       cItem_ctor( ptr_menu_item, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item);
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 123
#endif
struct cList* cList_ctor(struct cList* ptr_clist, int width);

/**
 *  Destructor. Deletes the cList object and all child objects (if any).
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory
 *  was allocated for the object by malloc(), use FREE_MEMORY to free it.
 *  Use LEAVE_MEMORY if the object was static or allocated in a stack
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     int index;
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[3] = {
 *                                           "Start",
 *                                           "High Score",
 *                                           "Exit"
 *                                          };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 ) ;
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ));
 *       cItem_ctor( ptr_menu_item, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item);
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 125
#endif
void cList_dtor(struct cList* ptr_clist, int memory_flag);

/**
 *  Adds an object (a cItem object for instance) to a cList object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param ptr_cobject A pointer to the initialized cObject
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     int index;
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[3] = {
 *                                           "Start",
 *                                           "High Score",
 *                                           "Exit"
 *                                          };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *       cItem_ctor( ptr_menu_item, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item );
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 *  @see cList_AddItem_Ex, cList_RemItem.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 177
#endif
void cList_AddItem(struct cList* ptr_clist, struct cObject* ptr_cobject);

/**
 *  The extended version of the cList_AddItem function. Adds an object to a cList object in a specified position.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param ptr_cobject A pointer to the initialized cObject
 *  @param number_items The zero based position in the cList object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     int index;
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[3] = {
 *                                           "Start",
 *                                           "High Score",
 *                                           "Exit"
 *                                          };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ));
 *       cItem_ctor( ptr_menu_item, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item, index );
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 *  @see cList_AddItem, cList_RemItem.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 126
#endif
void cList_AddItem_Ex(struct cList* ptr_clist, struct cObject* ptr_cobject, int number_items);

/**
 *  Removes an object from a cList object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param ptr_cobject A pointer to the initialized cObject
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     int index;
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct cItem* ptr_menu_item[3];
 *     static const char* sz_menu_text[3] = {
 *                                           "Start",
 *                                           "High Score",
 *                                           "Exit"
 *                                          };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item[index] = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *       cItem_ctor( ptr_menu_item + index, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item + index );
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     // "High Score"  will be removed from the list.
 *     cList_RemItem( &menu, ptr_menu_item + 1 );
 *     cItem_dtor( ptr_menu_item + 1 );
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 *  @see cList_AddItem, cList_AddItem_Ex.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 127
#endif
void cList_RemItem(struct cList* ptr_clist, struct cObject* ptr_cobject);

/**
 *  Message-processing function.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param ptr_message A pointer to the Message structure
 *  @return TRUE if the message was processed
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[3] = {
 *                                           "Start",
 *                                           "High Score",
 *                                           "Exit"
 *                                          };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *       cItem_ctor( ptr_menu_item, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item );
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if ( ptr_message->msgid == MSG_KEYDOWN )
 *     {
 *       //  Now we can select menu items using up and down arrows.
 *       cList_proc( &menu, ptr_message );
 *     }
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 128
#endif
bool cList_proc(struct cList* ptr_clist, struct Message* ptr_message);

/**
 *  Returns the index of the currently selected object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return The index of the currently selected object
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[3] = {
 *                                           "Start",
 *                                           "High Score",
 *                                           "Exit"
 *                                          };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *       cItem_ctor( ptr_menu_item, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item );
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     if ( ptr_message->msgid == MSG_KEYDOWN )
 *     {
 *       if( Message_get_key_param( ptr_message )->scancode == KEY_ENTER )
 *       {
 *         switch ( cList_Sel( &menu ) )
 *         {
 *           case 0:
 *           ...
 *         }
 *       }
 *       //  Now we can select menu items using up and down arrows.
 *       cList_proc( &menu, ptr_message );
 *     }
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 176
#endif
int cList_Sel(struct cList* ptr_clist);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_Disconnect            cObject_Disconnect
#else
/**
 *  Disconnects a cList from its parent object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return None
 */
void cList_Disconnect (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_Select                cObject_Select
#else
/**
 *  Selects a cList object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return TRUE if the cList object was selected
 */
bool cList_Select (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_update                cObject_update
#else
/**
 *  Updates a cList object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *   ...
 *   struct cList main_menu;
 *   ...
 *   // Redraws the menu object and the screen
 *   cList_update( &main_menu );
 *   ...
 *  @endcode
 */
void cList_update (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_GetParent             cObject_GetParent
#else
/**
 *  Returns a pointer to the cClip of the parent object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return A pointer to the cClip of the parent object
 */
struct cClip* cList_GetParent (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_Hide                  cObject_Hide
#else
/**
 *  Hides a cList object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *   ...
 *   struct cList main_menu;
 *   ...
 *   case MSG_LOSTFOCUS: // Application has lost focus
 *        ...
 *        cList_Hide( &main_menu );
 *        break;
 *   ...
 *   case MSG_GOTFOCUS: // Application has focus
 *        ...
 *        cList_Show( &main_menu );
 *        break;
 *   ...
 *  @endcode
 *  @see cList_Show.
 */
void cList_Hide (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_Show                  cObject_Show
#else
/**
 *  Shows a cList object.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *   ...
 *   struct cList main_menu;
 *   ...
 *   case MSG_LOSTFOCUS: // Application has lost focus
 *        ...
 *        cList_Hide( &main_menu );
 *        break;
 *   ...
 *   case MSG_GOTFOCUS: // Application has focus
 *        ...
 *        cList_Show( &main_menu );
 *        break;
 *   ...
 *  @endcode
 *  @see cList_Hide.
 */
void cList_Show (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_Disable               cObject_Disable
#else
/**
 *  Disables a cList object so that it cannot be selected.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return None
 *  @see cList_Enable.
 */
void cList_Disable (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_Enable                cObject_Enable
#else
/**
 *  Enables a cList object so that it may be selected.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return None
 *  @see cList_Disable.
 */
void cList_Enable (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_AddObj                cClip_AddObj
#else
/**
 *  Adds an object to the cList in position (x, y).
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param ptr_object A pointer to the initialized cObject object
 *  @param x x-coordinate of the new object
 *  @param y y-coordinate of the new object
 *  @return None
 */
void cList_AddObj (struct cList* ptr_clist, struct cObject* ptr_object, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_InsObj                cClip_InsObj
#else
/**
 *  Adds an object to the cList in position ( x, y ) with a z-position index (maximal index is on the top).
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param ptr_object A pointer to the initialized cObject object
 *  @param x x-coordinate of the new object
 *  @param y y-coordinate of the new object
 *  @param index z-position of the new object
 *  @return None
 */
void cList_InsObj (struct cList* ptr_clist, struct cObject* ptr_object, int x, int y, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_RemObj                cClip_RemObj
#else
/**
 *  Removes an object from a cList.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param ptr_object A pointer to the initialized cObject object
 *  @return None
 */
void cList_RemObj (struct cClip* ptr_clist, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_SelectFirst           cClip_SelectFirst
#else
/**
 *  Selects the first object in the cList.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return FALSE if the function failed
 */
bool cList_SelectFirst (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_SelectPrev            cClip_SelectPrev
#else
/**
 *  Selects the previous object in the cList.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if function failed.
 */
bool cList_SelectPrev (struct cList* ptr_clist, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_SelectNext            cClip_SelectNext
#else
/**
 *  Selects the next object in the cList.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the function failed
 */
bool cList_SelectNext (struct cList* ptr_clist, bool round);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_Scroll                cClip_Scroll
#else
/**
 *  Scrolls a cList to make a rectangle visible. Provides minimal scrolling.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param rectangle A pointer to the initialized rect_t structure
 *  @return None
 *  @see cList_Scroll_Ex.
 */
void cList_Scroll (struct cList* ptr_clist, struct rect_t* rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_Scroll_Ex             cClip_Scroll_Ex
#else
/**
 *  Scrolls a cList by (x, y), but cannot exceed the child's boundaries.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param x "x" coordinate shift
 *  @param y "y" coordinate shift
 *  @return None
 *  @see cList_Scroll.
 */
void cList_Scroll_Ex (struct cList* ptr_clist, int x, int y);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_SendScroll            cClip_SendScroll
#else
/**
 *  Forces this object to redraw its scrolling arrows.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return None
 */
void cList_SendScroll (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_GetShifty             cClip_GetShifty
#else
/**
 *  Returns the 'y' coordinate shift (as a  result of scrolling).
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return 'y' coordinate shift (as a result of scrolling).
 */
int cList_GetShifty (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_GetShiftx             cClip_GetShiftx
#else
/**
 *  Returns the 'x' coordinate shift (as a result of scrolling).
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return 'x' coordinate shift (as a result of scrolling)
 */
int cList_GetShiftx (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_GetCount              cClip_GetCount
#else
/**
 *  Returns a count of child objects.
 *  @param ptr_clist A pointer to the initialized cList object
 *  @return The child objects count
 */
int cList_GetCount (struct cList* ptr_clist);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_get_by_index          cClip_get_by_index
#else
/**
 *  Returns object index (or NULL).
 *  @param ptr_clist A pointer to the initialized cList object
 *  @param index The index of the object in cList
 *  @return The object's index (or NULL)
 */
struct cObject* cList_get_by_index (struct cList* ptr_clist, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_FindObj               cClip_FindObj
#else
/**
 *   Returns a child object's index in the cList (or -1).
 *   @param ptr_clist A pointer to the initialized cList object
 *   @param ptr_object A pointer to the needed cObject
 *   @return The child's object's index in the cList (or -1)
 */
int cList_FindObj (struct cList* ptr_clist, struct cObject* ptr_object);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cList_GetSelectedObject     cClip_GetSelectedObject
#else
/**
 *   Returns the currently selected object.
 *   @param ptr_clist A pointer to the initialized cList object
 *   @return A pointer to the currently selected object
 *   @code
 *    #include <cywin.h>
 *     ...
 *     struct cItem* ptr_selected_item;
 *     ...
 *     // Retrieves the pointer to the list item currently selected
 *     ptr_selected_item = cList_GetSelectedObject( ptr_list_object );
 *     ...
 *   @endcode
 */
struct cObject* cList_GetSelectedObject (struct cList* ptr_clist);
#endif

#ifdef _SKIP_DOC
};
#endif


/**
 *   @ingroup controls_group
 *   @brief This structure implements methods for working with simple units of the list, such as a text string.
 *
 *   This structure implements methods for working with simple units of the list, such as a text string.
 *   It might have an icon on the left and/or an additional text on the right.
 *   <br>
 *   <br>
 *   You must call the cItem_ctor function before use and the cItem_dtor function after use.
 *   @see controls_group
 */
struct cItem : public cObject
{
#ifndef _SKIP_DOC
  char cItem_Members_STUB[20];
};
#endif


//============================
//=                          =
//=       object cItem.      =
//=       Abstract item.     =
//=                          =
//============================
/**
 *  Constructor. Initialize the cItem object.
 *  @param ptr_citem A pointer to the cItem struct
 *  @param width Item's width (in pixels)
 *  @param title_name Item's title
 *  @param submenu Determines whether there is a submenu
 *  @param right_text Creates an additional text area on the right
 *  @param ptr_bitmap Creates  pointer to the icon bitmap
 *  @return A pointer to the initialized cItem object
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct cList menu;
 *     struct BitmapSequence menu_icons;
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[3] = {
 *                                           "CYBIKO",
 *                                           "Microsoft",
 *                                           "Ford"
 *                                          };
 *     static const char* sz_right_text[3] = {
 *                                            "Yang",
 *                                            "Gates",
 *                                            "Ford"
 *                                           };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     BitmapSequence_ctor_Ex( &menu_icons , "logos.pic");
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *       cItem_ctor( ptr_menu_item,
 *                   100,
 *                   sz_menu_text[index],
 *                   FALSE,
 *                   sz_right_text[index],
 *                   BitmapSeqence_get_bitmap( &menu_icons, index ) );
 *       cList_AddItem( &menu, ptr_menu_item );
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *     BitmapSequence_dtor( &menu_icons, LEAVE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 129
#endif
struct cItem* cItem_ctor(struct cItem* ptr_citem, int width, char* title_name,
                         bool submenu, char* right_text,
                         struct Bitmap* ptr_bitmap);

/**
 *  Destructor. Deletes a cItem object.
 *  @param ptr_citem A pointer to the initialized cItem object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was
 *  allocated for the object by malloc(), use FREE_MEMORY to free it. Use
 *  LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     int index;
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct cItem* ptr_menu_item[3];
 *     static const char* sz_menu_text[3] = {
 *                                           "Start",
 *                                           "High Score",
 *                                           "Exit"
 *                                          };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item[index] = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *       cItem_ctor( ptr_menu_item + index, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item + index );
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     // "High Score"  will be removed from the list.
 *     cList_RemItem( &menu, ptr_menu_item + 1 );
 *     cItem_dtor( ptr_menu_item + 1 );
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 172
#endif
void cItem_dtor(struct cItem* ptr_citem, int memory_flag);

/**
 *  Returns the item's title.
 *  @param ptr_citem A pointer to the initialized cItem object
 *  @return The item's title
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     int index;
 *     struct cList menu;
 *     struct module_t main_module;
 *     struct cItem* ptr_menu_item[3];
 *     static const char* sz_menu_text[3] = {
 *                                           "Start",
 *                                           "High Score",
 *                                           "Exit"
 *                                          };
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cList_ctor( &menu, 100 );
 *     for( index = 0; index < 3; index++ )
 *     {
 *       ptr_menu_item[index] = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *       cItem_ctor( ptr_menu_item + index, 100, sz_menu_text[index], FALSE, NULL, NULL );
 *       cList_AddItem( &menu, ptr_menu_item + index );
 *     }
 *     // Puts the cList object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu, 10, 40 );
 *     ...
 *     for( index = 0; index < 3; index++ )
 *     {
 *        TRACE( "%s - item #%d ", cItem_GetTitle( ptr_menu_item + index), index );
 *     }
 *     ...
 *     // Menu items will be destroyed automatically.
 *     cList_dtor( &menu, LEAVE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 175
#endif
char* cItem_GetTitle(struct cItem* ptr_citem);

/**
 *  Message-processing function.
 *  @param ptr_citem A pointer to a cList object
 *  @param ptr_message A pointer to a Message.
 *  @return TRUE if the message was processed.
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     struct Message* ptr_message;
 *     struct cItem menu_item;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     cItem_ctor( &menu_item, 100, "Item", FALSE, NULL, NULL );
 *     // Puts the cItem object on the Cybiko screen.
 *     cWinApp_AddObj( main_module.m_process, &menu_item, 10, 40 );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     // cItem will be redrawn automatically now.
 *     cItem_proc( &menu_item, ptr_message );
 *     ...
 *     cItem_dtor( &menu_item, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 131
#endif
bool cItem_proc(struct cItem* ptr_citem, struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cItem_Disconnect   cObject_Disconnect
#else
/**
 *  Disconnects a cItem from its parent object.
 *  @param ptr_citem A pointer to a cItem object
 *  @return None
 */
void cItem_Disconnect (struct cItem* ptr_citem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cItem_Select       cObject_Select
#else
/**
 *  Selects a cItem object.
 *  @param ptr_citem A pointer to a cItem object
 *  @return TRUE if the object was selected
 */
bool cItem_Select (struct cItem* ptr_citem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cItem_update       cObject_update
#else
/**
 *  Updates a cItem object.
 *  @param ptr_citem A pointer to a cItem object
 *  @return None
 */
void cItem_update (struct cItem* ptr_citem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cItem_GetParent    cObject_GetParent
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_citem A pointer to a cItem object
 *  @return A pointer to the parent object
 */
struct cClip* cItem_GetParent (struct cItem* ptr_citem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cItem_Hide         cObject_Hide
#else
/**
 *  Hides a cItem object.
 *  @param ptr_citem A pointer to a cItem object
 *  @return None
 *  @see cItem_Show.
 */
void cItem_Hide (struct cItem* ptr_citem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cItem_Show         cObject_Show
#else
/**
 *  Shows a cItem object.
 *  @param ptr_citem A pointer to a cItem object.
 *  @return None
 *  @see cItem_Hide.
 */
void cItem_Show (struct cItem* ptr_citem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cItem_Disable      cObject_Disable
#else
/**
 *  Disables a cItem object so that it cannot be selected.
 *  @param ptr_citem A pointer to a cItem object
 *  @return None
 *  @see cItem_Enable.
 */
void cItem_Disable (struct cItem* ptr_citem);
#endif


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cItem_Enable       cObject_Enable
#else
/**
 *  Enables a cItem object so that it may be selected.
 *  @param ptr_citem A pointer to a cItem object
 *  @return None
 *  @see cItem_Disable.
 */
void cItem_Enable (struct cItem* ptr_citem);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *   @ingroup controls_group
 *   @brief The same type of string as in the cItem structure.
 *
 *   The same type of string as in the cItem structure.
 *   However, it is possible to insert text on the right or an icon on the left,
 *   and to select one of the values assigned previously.
 *   <br>
 *   <br>
 *   You must call the cXItem_ctor function before use and the cXItem_dtor function after use.
 *   @see controls_group
 */
struct cXItem : public cItem
{
#ifndef _SKIP_DOC
  char cXItem_Members_STUB[32];
};
#endif

//============================
//=                          =
//=      object cXItem.      =
//=     Three-part item.     =
//=                          =
//============================
/**
 *  Constructor. Creates a cXItem object.
 *  @param ptr_cxitem A pointer to cXItem structure
 *  @param width Item's width in pixels
 *  @param title Item's title.
 *  @param rstr The string in the right-side section - an array of pointers to strings, with the last pointer equaling 0
 *  @param rref A pointer to a variable that can be used to access the current right-side index
 *  @param submenu Determines if there is a submenu
 *  @param readonly Forbids entering and changing the section on the item's right side.
 *  @param lbmp The bitmap list for the left-side section - an array of pointers to bitmaps, with last pointer equaling 0
 *  @param lref A reference to a variable that can be used to access the current left-side index. Initial value is an index of an initially displayed bitmap
 *  @return A pointer to the initialized cXItem object
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char   left_index = 1;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 133
#endif
struct cXItem* cXItem_ctor(struct cXItem* ptr_cxitem, int width, char* title, char** rstr, char* rref, bool submenu, bool readonly, struct Bitmap** lbmp, char* lref);

/**
 *  Destructor. It deletes the cXItem object.
 *  @param ptr_cxitem A pointer to the initialized cXItem object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the
 *  memory was allocated for the object by malloc(), use FREE_MEMORY
 *  to free it. Use LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char   left_index = 1;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 135
#endif
void cXItem_dtor(struct cXItem* ptr_cxitem, int memory_flag);

/**
 *  Selects the left-side section of the item (only if the item has focus).
 *  @param ptr_cxitem A pointer to the initialized cXItem object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char   left_index = 1;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *     // If you on the right item part now you can select the left item part. Note, the focus would be on the left part.
 *     cXItem_SelLeft( ptr_menu_item );
 *     ...
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXItem_SelRight.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 136
#endif
void cXItem_SelLeft(struct cXItem* ptr_cxitem);

/**
 *  Selects the right-side section of the item (only if the item has focus).
 *  @param ptr_cxitem A pointer to the initialized cXItem object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char   left_index = 1;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *     // If you on the left item part now you can select the right item part. Note, the focus would be on the right part.
 *     cXItem_SelRight( ptr_menu_item );
 *     ...
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXItem_SelLeft.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 137
#endif
void cXItem_SelRight(struct cXItem* ptr_cxitem);

/**
 *  Changes the left section's value by 'index'.
 *  @param ptr_cxitem A pointer to the initialized cXItem object
 *  @param index A new left section's value
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char   left_index = 1;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *     // For example, "Digit 1" value is selected now, you can bypass "Digit 2" value and set the "Digit 3" value.
 *     cXItem_SetRight( ptr_menu_item, 2 );
 *     ...
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXItem_SetRight.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 138
#endif
void cXItem_SetLeft(struct cXItem* ptr_cxitem, int index);

/**
 *  Changes the right section's value by 'index'.
 *  @param ptr_cxitem A pointer to the initialized cXItem object
 *  @param index A new right section's value
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char   left_index = 1;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *     // For example, icon "1" is selected now, you can bypass the icon "2" value and set the icon "3" value.
 *     cXItem_SetLeft( ptr_menu_item, 2 );
 *     ...
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXItem_SetLeft.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 139
#endif
void cXItem_SetRight(struct cXItem* ptr_cxitem, int index);

/**
 *  Deselects the left or right section of the item (if either was selected), and saves L/Rsel to L/Rprev.
 *  @param ptr_cxitem A pointer to the initialized cXItem object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char   left_index = 1;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *    // For example, icon "1" is selected now, you can bypass the icon "2" value and set the icon "3" value.
 *     cXItem_SetLeft( ptr_menu_item, 2 );
 *     ...
 *     // Save the current state.
 *     cXItem_OK( ptr_menu_item, FREE_MEMORY );
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXItem_Esc.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 140
#endif
void cXItem_OK(struct cXItem* ptr_cxitem);

/**
 *  Deselects the left or right section of the item (if either was selected), and restores L/Rsel from L/Rprev.
 *  @param ptr_cxitem A pointer to the initialized cXItem object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char  left_index = 1;
 *       struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *     // For example, icon "1" is selected now, you can bypass the icon "2" value and set the icon "3" value.
 *     cXItem_SetLeft( ptr_menu_item, 2 );
 *     ...
 *     // Restores the previous state.
 *     cXItem_Esc( ptr_menu_item, FREE_MEMORY );
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXItem_OK.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 141
#endif
void cXItem_Esc(struct cXItem* ptr_cxitem);

/**
 *  The Message-processing function.
 *  @param ptr_cxitem A pointer to the initialized cXItem object
 *  @param ptr_message A pointer to a Message
 *  @return TRUE if the message was processed
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   right_index = 1;
 *     char   left_index = 1;
 *     struct cItem* ptr_menu_item;
 *     static const char* sz_menu_text[4] = {
 *                                             "Digit 1",
 *                                             "Digit 2",
 *                                             "Digit 3",
 *                                             '\0'
 *                                          };
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     // Item will be shown as follows: '2' | "Now digit is" | "Digit 2"
 *     cXItem_ctor( ptr_menu_item,
 *                  160,
 *                  "Now digit is",
 *                  sz_menu_text,
 *                  &right_index,
 *                  FALSE,
 *                  FALSE,
 *                  bmps,
 *                  &left_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     // cXItem will be redraw automatically now.
 *     cXItem_proc( ptr_menu_item, ptr_message );
 *     ...
 *     // Now you can use the <Select> button to change values on the left part of the item: "Digit 1"->"Digit 2"-> etc.
 *     // To move to the right part use the arrows buttons.
 *     // Now you can use the <Select> button to change values on the right part of the item: icon '1'->icon '2'-> etc.
 *     ...
 *     // For example, icon "1" is selected now, you can bypass the icon "2" value and set the icon "3" value.
 *     cXItem_SetLeft( ptr_menu_item, 2 );
 *     ...
 *     // Restore the previous state.
 *     cXItem_Esc( ptr_menu_item, FREE_MEMORY );
 *     cXItem_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 142
#endif
bool cXItem_proc(struct cXItem* ptr_cxitem, struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_Disconnect   cObject_Disconnect
#else
/**
 *  Disconnects a cXItem from its parent object.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return None
 */
void cXItem_Disconnect (struct cXItem* ptr_cxitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_Select       cObject_Select
#else
/**
 *  Selects cXItem object.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return TRUE if the object was selected
 */
bool cXItem_Select (struct cXItem* ptr_cxitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_update       cObject_update
#else
/**
 *  Updates a cXItem object.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return None
 */
void cXItem_update (struct cXItem* ptr_cxitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_GetParent    cObject_GetParent
#else
/**
 *  Returns a pointer to the cClip of the parent object.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return A pointer to the cClip of the parent object
 */
struct cClip* cXItem_GetParent (struct cXItem* ptr_cxitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_Hide         cObject_Hide
#else
/**
 *  Hides a cXItem object.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return None
 *  @see cXItem_Show.
 */
void cXItem_Hide (struct cXItem* ptr_cxitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_Show         cObject_Show
#else
/**
 *  Shows a cXItem object.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return None
 *  @see cXItem_Hide.
 */
void cXItem_Show (struct cXItem* ptr_cxitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_Disable      cObject_Disable
#else
/**
 *  Disables a cXItem object so that it cannot be selected.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return None
 *  @see cXItem_Enable.
 */
void cXItem_Disable (struct cXItem* ptr_cxitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_Enable       cObject_Enable
#else
/**
 *  Enables a cXItem object so that it may be selected.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return None
 *  @see cXItem_Disable.
 */
void cXItem_Enable (struct cXItem* ptr_cxitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXItem_GetTitle     cItem_GetTitle
#else
/**
 *  Returns the item's title.
 *  @param ptr_cxitem A pointer to a cXItem object
 *  @return The item's title
 */
char* cXItem_GetTitle (struct cXItem* ptr_cxitem);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *  @ingroup controls_group
 *  @brief A cXStr object might have an icon on the left and/or additional text on the right.
 *
 *  A cXStr object might have an icon on the left and/or additional text on the right.
 *  It is possible to input text into the right field. You must call the cXStr_ctor function
 *  before use and the cXStr_dtor function after use.
 *
 *  @see controls_group
 */
struct cXStr : public cXItem
{
#ifndef _SKIP_DOC
  char cXStr_Members_STUB[48];
};
#endif

//============================
//=                          =
//=       object cXStr.      =
//=    Editable left part.   =
//=                          =
//============================
/**
 *  Constructor. Creates a cXStr object.
 *  @param ptr_cxstr A pointer to the cXStr structure
 *  @param w An item's width in pixels (including left and right sections)
 *  @param title A title string
 *  @param st The initial value for the string on the item's right side
 *  @param sln Maximum character length of the string on the item's right side (not including '\0')
 *  @param swd Maximum width, in pixels, of the viewable portion of the string on the item's right side (not including 2 pixels for surrounding the frame)
 *  @param lbmp The bitmap lost for the section on the item's left side - an array of pointers to bitmaps, with last pointer equaling 0
 *  @param lsel A reference to a variable that can be used to access the current left-side index
 *  @param readonly Forbids entering and changing the section on the item's right side
 *  @return A pointer to the initialized cXStr object
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   icon_index = 2;
 *     struct cItem* ptr_menu_item;
 *     char   buffer[100];
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     strcpy( buffer, "Initial value" );
 *     // Item will be shown as follows: '3' | "Any item" | "Initial value"
 *     cXStr_ctor( ptr_menu_item,
 *                  160,
 *                  "Any item",
 *                  buffer,
 *                  30,
 *                  80,
 *                  bmps,
 *                  &icon_index,
 *                  FALSE );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     cXStr_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 145
#endif
struct cXStr* cXStr_ctor(struct cXStr* ptr_cxstr, int w, char* title, char* st, int sln, int swd, struct Bitmap ** lbmp, char* lsel, bool readonly);

/**
 *  Destructor. Deletes the cXStr object.
 *  @param ptr_cxstr A pointer to the initialized cXStr object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   icon_index = 2;
 *     struct cItem* ptr_menu_item;
 *     char   buffer[100];
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     // Item will be shown as follows: '3' | "Any item" | "Initial value"
 *     cXStr_ctor( ptr_menu_item,
 *                  160,
 *                  "Any item",
 *                  buffer,
 *                  30,
 *                  80,
 *                  bmps,
 *                  &icon_index,
 *                  FALSE );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     cXStr_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 148
#endif
void cXStr_dtor(struct cXStr* ptr_cxstr, int memory_flag);

//fn void cXStr_ScrollLR(struct cXStr* ptr_cxstr)
//brief Recalculates StrBox. Should be called if you have made changes to the right string or to the cursor position while it is selected.
//param ptr_cxstr is a pointer to initialized cXStr object.
#ifndef _SKIP_DOC
__cyos "CyWin" 149
void cXStr_ScrollLR(struct cXStr*);
#endif

/**
 *  Highlightes the entire whole right-side string, and positions the cursor at its end (only when the section on the item's right side is selected).
 *  @param ptr_cxstr A pointer to the initialized cXStr object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   icon_index = 2;
 *     struct cItem* ptr_menu_item;
 *     char   buffer[100];
 *     ...
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     // Item will be shown as follows: '3' | "Any item" | "Initial value"
 *     cXStr_ctor( ptr_menu_item,
 *                  160,
 *                  "Any item",
 *                  buffer,
 *                  30,
 *                  80,
 *                  bmps,
 *                  &icon_index,
 *                  FALSE );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     // Highlights the right part.
 *     cXStr_SetStart(  ptr_menu_item  );
 *     ...
 *     cXStr_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 150
#endif
void cXStr_SetStart(struct cXStr* ptr_cxstr);

/**
 *  Selects the section on the item's right side (only if the item has the focus).
 *  @param ptr_cxstr A pointer to the initialized cXStr object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   icon_index = 2;
 *     struct cItem* ptr_menu_item;
 *     char   buffer[100];
 *     ...
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     strcpy( buffer, "Initial value" );
 *     // Item will be shown as follows: '3' | "Any item" | "Initial value"
 *     cXStr_ctor( ptr_menu_item,
 *                  160,
 *                  "Any item",
 *                  buffer,
 *                  30,
 *                  80,
 *                  bmps,
 *                  &icon_index,
 *                  FALSE );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     // Selects the right part.
 *     cXStr_SelRight(  ptr_menu_item  );
 *     ...
 *     cXStr_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXStr_SelLeft.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 153
#endif
void cXStr_SelRight(struct cXStr* ptr_cxstr);

/**
 *  Saves an edited string and deselects the section on the item's right side (if it was selected). Behavior for the left side is not changed.
 *  @param ptr_cxstr A pointer to the initialized cXStr object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   icon_index = 2;
 *     struct cItem* ptr_menu_item;
 *     char   buffer[100];
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     strcpy( buffer, "Initial value" );
 *     // Item will be shown as follows: '3' | "Any item" | "Initial value"
 *     cXStr_ctor( ptr_menu_item,
 *                  160,
 *                  "Any item",
 *                  buffer,
 *                  30,
 *                  80,
 *                  bmps,
 *                  &icon_index,
 *                  FALSE );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     // Saves the current state
 *     cXStr_OK(  ptr_menu_item  );
 *     ...
 *     cXStr_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXStr_Esc.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 155
#endif
void cXStr_OK(struct cXStr* ptr_cxstr);

/**
 *  Clears an edited string and deselects the section on the item's right side(if it was selected). Behavior for the left side is not changed.
 *  @param ptr_cxstr A pointer to the initialized cXStr object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   icon_index = 2;
 *     struct cItem* ptr_menu_item;
 *     char   buffer[100];
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     // Item will be shown as follows: '3' | "Any item" | "Initial value"
 *     cXStr_ctor( ptr_menu_item,
 *                  160,
 *                  "Any item",
 *                  buffer,
 *                  30,
 *                  80,
 *                  bmps,
 *                  &icon_index,
 *                  FALSE );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     // We have made changes then restore the previous state.
 *     cXStr_Esc(  ptr_menu_item  );
 *     ...
 *     cXStr_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see cXStr_OK.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 156
#endif
void cXStr_Esc(struct cXStr* ptr_cxstr);

//fn bool cXStr_Test(struct cXStr* ptr_cxstr)
//brief This function is called when an attempt to save changes to right part has been made (usually when it becomes unselected).
//param ptr_cxstr is a pointer to initialized cXStr object.
//return TRUE if a changes to right part has been made succefully.
#ifndef _SKIP_DOC
__cyos "CyWin" 171
bool cXStr_Test(struct cXStr*);
#endif

/**
 *  Saves a Str to an external string.
 *  @param ptr_cxstr A pointer to the initialized cXStr object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   icon_index = 2;
 *     struct cItem* ptr_menu_item;
 *     char   buffer[100];
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     // Item will be shown as follows: '3' | "Any item" | "Initial value"
 *     cXStr_ctor( ptr_menu_item,
 *                  160,
 *                  "Any item",
 *                  buffer,
 *                  30,
 *                  80,
 *                  bmps,
 *                  &icon_index,
 *                  FALSE );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     // If we made a change in the right section of the item, this function saves it in the 'buffer'.
 *     cXStr_SaveStr(  ptr_menu_item  );
 *     ...
 *     cXStr_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 174
#endif
void cXStr_SaveStr(struct cXStr* ptr_cxstr);

/**
 *  The Message-processing function.
 *  @param ptr_cxstr A pointer to the initialized cXStr object
 *  @param ptr_message A pointer to a Message
 *  @return TRUE if the message was processed
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   icon_index = 2;
 *     struct cItem* ptr_menu_item;
 *     char   buffer[100];
 *     struct Message* ptr_message;
 *     ...
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: digit '1', digit '2', digit '3'.
 *     BitmapSequence_ctor_Ex(&sq, "digits.pic");
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cItem* )malloc ( sizeof ( struct cItem ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     // Item will be shown as follows: '3' | "Any item" | "Initial value"
 *     cXStr_ctor( ptr_menu_item,
 *                  160,
 *                  "Any item",
 *                  buffer,
 *                  30,
 *                  80,
 *                  bmps,
 *                  &icon_index,
 *                  FALSE );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *     // Processes the messages that the cXStr control manages.
 *     cXStr_proc( ptr_menu_item, ptr_message );
 *     ...
 *     // If we made change in the right item part this function saves it in the 'buffer'.
 *     cXStr_SaveStr(  ptr_menu_item  );
 *     ...
 *     cXStr_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 158
#endif
bool cXStr_proc(struct cXStr* ptr_cxstr, struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_Disconnect   cObject_Disconnect
#else
/**
 *  Disconnects a cXStr from its parent object.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @return None
 */
void cXStr_Disconnect (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_Select       cObject_Select
#else
/**
 *  Selects a cXStr object.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @return TRUE if the object was selected
 */
bool cXStr_Select (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_update       cObject_update
#else
/**
 *  Updates a cXStr object.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @return None
 */
void cXStr_update (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_GetParent    cObject_GetParent
#else
/**
 *  Returns a pointer to the cClip of the parent object.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @return A pointer to the cClip of the parent object
 */
struct cClip* cXStr_GetParent (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_Hide         cObject_Hide
#else
/**
 *  Hides a cXStr object.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @return None
 *  @see cXStr_Show.
 */
void cXStr_Hide (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_Show         cObject_Show
#else
/**
 *  Shows a cXStr object.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @return None
 *  @see cXStr_Hide.
 */
void cXStr_Show (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_Disable      cObject_Disable
#else
/**
 *  Disables a cXStr object so that it cannot be selected.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @return None
 *  @see cXStr_Enable.
 */
void cXStr_Disable (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_Enable       cObject_Enable
#else
/**
 *  Enables a cXStr object so that it may be selected.
 *  @param ptr_cxstr A pointer to a cXStr object.
 *  @return None
 *  @see cXStr_Disable.
 */
void cXStr_Enable (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_GetTitle     cItem_GetTitle
#else
/**
 *  Returns the item's title.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @return The item's title
 */
char* cXStr_GetTitle (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_SelLeft      cXItem_SelLeft
#else
/**
 *  Selects the section on the item's left side (only if the item has the focus).
 *  @param ptr_cxstr A pointer to the initialized cXStr object
 *  @return None
 *  @see cXStr_SelRight.
 */
void cXStr_SelLeft (struct cXStr* ptr_cxstr);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_SetLeft      cXItem_SetLeft
#else
/**
 *  Changes the left section's value by 'index'.
 *  @param ptr_cxstr A pointer to a cXStr object
 *  @param index A new left section's value
 *  @return None
 *  @see cXStr_SetRight.
 */
void cXStr_SetLeft (struct cXStr* ptr_cxstr, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXStr_SetRight     cXItem_SetRight
#else
/**
 *  Changes right section's value by 'index'.
 *  @param ptr_cxstr A pointer to a cXStr object.
 *  @param index A new right section's value
 *  @return None
 *  @see cXStr_SetLeft.
 */
void cXStr_SetRight (struct cXStr* ptr_cxstr, int index);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *  @ingroup controls_group
 *  @brief A cSItem object might have an icon on the left and/or additional text on the right
 *
 *  A cSItem object might have an icon on the left and/or additional text on the right
 *  It is possible to input text into the field on the right, but only numerical data. For more
 *  information see the cItem structure definition.
 *   <br>
 *   <br>
 *  You must call the cSItem_ctor function before use and the cSItem_dtor function after use.
 *  @see controls_group
 */
struct cSItem : public cItem
{
#ifndef _SKIP_DOC
  char cSItem_Members_STUB[4];
};
#endif

//============================
//=                          =
//=      object cSItem.      =
//=   Numeric item object.   =
//=                          =
//============================

/**
 *  Constructor. Creates an empty cSItem object.
 *  @param ptr_csitem A pointer to a cSItem structure
 *  @param w Item's width in pixels
 *  @param title Title string
 *  @param ptr_bitmap Creates a pointer to the icon bitmap
 *  @return A pointer to the initialized cSItem object
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     char   buf[15];
 *     struct BitmapSequence * ptr_bmps = (BitmapSequence* ) malloc( sizeof( struct BitmapSequence ) );
 *     struct cList * ptr_clist = (cList* )malloc( sizeof( cList ) );
 *     struct cSItem * ptr_csitem = (cSItem* )malloc( sizeof( cSItem ) );
 *     ...
 *     BitmapSequence_ctor_Ex( ptr_bmps,"gfx.pic" );
 *     cList_ctor( ptr_clist, 80 );
 *     ...
 *     for ( int i=1; i<=5; i++ )
 *     {
 *       sprintf( buf, "menu item %d", i );
 *       cSItem_ctor( ptr_csitem, LCD_WIDTH, buf, BitmapSequence_get_bitmap( ptr_bmps, i-1 ) );
 *       cList_AddItem( ptr_clist, ptr_csitem );
 *     }
 *     ...
 *     cSItem_dtor( ptr_csitem, FREE_MEMORY );
 *     cList_dtor( ptr_clist, FREE_MEMORY );
 *     BitmapSequence_dtor( ptr_bmps, FREE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 173
#endif
struct cSItem* cSItem_ctor(struct cSItem* ptr_csitem, int w, char* title, struct Bitmap* ptr_bitmap);

/**
 *  Destructor. Deletes a cSItem object.
 *  @param ptr_csitem A pointer to the initialized cSItem object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the
 *  memory was allocated for the object by malloc(), use FREE_MEMORY
 *  to free it. Use LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     char   buf[15];
 *     struct BitmapSequence * ptr_bmps = (BitmapSequence* ) malloc( sizeof( struct BitmapSequence ) );
 *     struct cList * ptr_clist = (cList* )malloc( sizeof( cList ) );
 *     struct cSItem * ptr_csitem = (cSItem* )malloc( sizeof( cSItem ) );
 *     ...
 *     BitmapSequence_ctor_Ex( ptr_bmps,"gfx.pic" );
 *     cList_ctor( ptr_clist, 80 );
 *     ...
 *     for ( int i=1; i<=5; i++ )
 *     {
 *       sprintf( buf, "menu item %d", i );
 *       cSItem_ctor( ptr_csitem, LCD_WIDTH, buf, BitmapSequence_get_bitmap( ptr_bmps, i-1 ) );
 *       cList_AddItem( ptr_clist, ptr_csitem );
 *     }
 *     ...
 *     cSItem_dtor( ptr_csitem, FREE_MEMORY );
 *     cList_dtor( ptr_clist, FREE_MEMORY );
 *     BitmapSequence_dtor( ptr_bmps, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 159
#endif
void cSItem_dtor(struct cSItem* ptr_csitem, int memory_flag);


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_Disconnect   cObject_Disconnect
#else
/**
 *  Disconnects a cSItem from its parent object.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return None
 */
void cSItem_Disconnect (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_Select       cObject_Select
#else
/**
 *  Selects a cSItem object.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return TRUE if the object was selected
 */
bool cSItem_Select (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_update       cObject_update
#else
/**
 *  Updates a cSItem object.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return None
 */
void cSItem_update (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_GetParent    cObject_GetParent
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return A pointer to the parent object
 */
struct cClip* cSItem_GetParent (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_Hide         cObject_Hide
#else
/**
 *  Hides a cSItem object.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return None
 *  @see cSItem_Show.
 */
void cSItem_Hide (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_Show         cObject_Show
#else
/**
 *  Shows a cSItem object.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return None
 *  @see cSItem_Hide.
 */
void cSItem_Show (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_Disable      cObject_Disable
#else
/**
 *  Disables a cSItem object so that it cannot be selected.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return None
 *  @see cSItem_Enable.
 */
void cSItem_Disable (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_Enable       cObject_Enable
#else
/**
 *  Enables a cSItem object so that it may be selected.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return None
 *  @see cSItem_Disable.
 */
void cSItem_Enable (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_GetTitle     cItem_GetTitle
#else
/**
 *  Returns the item's title.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @return The item's title
 */
char* cSItem_GetTitle (struct cSItem* ptr_csitem);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cSItem_proc         cItem_proc
#else
/**
 *  The Message-processing function.
 *  @param ptr_csitem A pointer to a cSItem object
 *  @param ptr_message A pointer to a Message
 *  @return TRUE if the message was processed
 */
bool cSItem_proc (struct cSItem* ptr_csitem, struct Message* ptr_message);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *  @ingroup controls_group
 *  @brief a framed string with an icon on the left.
 *
 *  A cXByte object is a a framed string with an icon on the left.
 *  It is used in main application screens. For more information, see the cXStr structure definition.
 *  You must call the cXByte_ctor function before use and the cXByte_dtor function after use.
 *  @see controls_group
 */
struct cXByte : public cXStr
{
#ifndef _SKIP_DOC
  char cXByte_Members_STUB[8];
};
#endif

//============================
//=                          =
//=      object cXByte.      =
//=Item with right text part.=
//=                          =
//============================

/**
 *  Constructor. Creates an empty cXByte object.
 *  @param ptr_cxbyte A pointer to a cSItem struct
 *  @param w Item width in pixels (including left and right sections)
 *  @param title Title string
 *  @param num An integer value that will be converted into a string for the initial string in the item's right section
 *  @param sln Maximum character length of the right string on the item's right side (not including finishing '\0' )
 *  @param swd Maximum width, in pixels, of the visible part of the string on the item's right side (not including 2 pixels for the surrounding frame)
 *  @param min_val The lowest acceptable integer
 *  @param max_val The highest acceptable integer
 *  @param ll The bitmap list for the section on the item's left side - an array of pointers to bitmaps, with last pointer equaling 0
 *  @param lsel A reference to a variable that can be used to access the current left-side index
 *  @return A pointer to the initialized cXByte object
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   picture_index = 2;
 *     struct cXByte* ptr_menu_item;
 *     char   value;
 *     ...
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: picture of '<apples>', picture of 'potatoes', picture of '<oranges>', etc.
 *     BitmapSequence_ctor_Ex( &sq, "food_pictures.pic" );
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = BitmapSequence_get_bitmap(&sq, 3);
 *     bmps[4] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cXByte* )malloc ( sizeof ( struct cXByte ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     value = 20;
 *     // Item will be shown as follows: '<apples>' | "has discounted (in percent)" | ""
 *     cXByte_ctor( ptr_menu_item,
 *                  160,
 *                  "Percent is",
 *                  &value,
 *                  3,
 *                  80,
 *                  0,
 *                  100,
 *                  bmps,
 *                  &picture_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     cXByte_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 162
#endif
struct cXByte* cXByte_ctor(struct cXByte* ptr_cxbyte, int w, char* title, char* num, int sln, int swd, char min_val, char max_val, struct Bitmap ** ll, char* lsel);

/**
 *  Destructor. Deletes the cXByte object.
 *  @param ptr_cxbyte A pointer to the initialized cXByte object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was
 *  allocated for the object by malloc(), use FREE_MEMORY to free it. Use
 *  LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *   {
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   picture_index = 2;
 *     struct cXByte* ptr_menu_item;
 *     char   value;
 *     ...
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: picture of '<apples>', picture of 'potatoes', picture of '<oranges>', etc.
 *     BitmapSequence_ctor_Ex( &sq, "food_pictures.pic" );
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = BitmapSequence_get_bitmap(&sq, 3);
 *     bmps[4] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cXByte* )malloc ( sizeof ( struct cXByte ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     value = 20;
 *     // Item will be shown as follows: '<apples>' | "has discounted (in percent)" | ""
 *     cXByte_ctor( ptr_menu_item,
 *                  160,
 *                  "Percent is",
 *                  &value,
 *                  3,
 *                  80,
 *                  0,
 *                  100,
 *                  bmps,
 *                  &picture_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     cXByte_dtor( ptr_menu_item, FREE_MEMORY );
 *   }
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 169
#endif
void cXByte_dtor(struct cXByte* ptr_cxbyte, int memory_flag);

/**
 *  Saves input information. Also converts the string on the right side to an integer, if it has been changed.
 *  @param ptr_cxbyte A pointer to the initialized cXByte object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   picture_index = 2;
 *     struct cXByte* ptr_menu_item;
 *     char   value;
 *     ...
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: picture of '<apples>', picture of 'potatoes', picture of '<oranges>', etc.
 *     BitmapSequence_ctor_Ex( &sq, "food_pictures.pic" );
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = BitmapSequence_get_bitmap(&sq, 3);
 *     bmps[4] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cXByte* )malloc ( sizeof ( struct cXByte ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     value = 20;
 *     // Item will be shown as follows: '<apples>' | "has discounted (in percent)" | ""
 *     cXByte_ctor( ptr_menu_item,
 *                  160,
 *                  "Percent is",
 *                  &value,
 *                  3,
 *                  80,
 *                  0,
 *                  100,
 *                  bmps,
 *                  &picture_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     // Saves the current state
 *     cXByte_OK(  ptr_menu_item  );
 *     ...
 *     cXByte_dtor( ptr_menu_item, FREE_MEMORY );
 *  @endcode
 *  @see cXByte_Esc.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 164
#endif
void cXByte_OK(struct cXByte* ptr_cxbyte);

/**
 *  Restores information.
 *  @param ptr_cxbyte A pointer to the initialized cXByte object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct BitmapSequence sq;
 *     struct Bitmap* bmps[4];
 *     char   picture_index = 2;
 *     struct cXByte* ptr_menu_item;
 *     char   value;
 *     ...
 *     init_module(&main_module);
 *     // Bitmap sequence contains three icons: picture of '<apples>', picture of 'potatoes', picture of '<oranges>', etc.
 *     BitmapSequence_ctor_Ex( &sq, "food_pictures.pic" );
 *     bmps[0] = BitmapSequence_get_bitmap(&sq, 0);
 *     bmps[1] = BitmapSequence_get_bitmap(&sq, 1);
 *     bmps[2] = BitmapSequence_get_bitmap(&sq, 2);
 *     bmps[3] = BitmapSequence_get_bitmap(&sq, 3);
 *     bmps[4] = NULL;
 *     cWinApp_clear_screen();
 *     ptr_menu_item = (struct cXByte* )malloc ( sizeof ( struct cXByte ) );
 *     strcpy( buffer, "Initial value" );
 *     ...
 *     value = 20;
 *     // Item will be shown as follows: '<apples>' | "has discounted (in percent)" | ""
 *     cXByte_ctor( ptr_menu_item,
 *                  160,
 *                  "Percent is",
 *                  &value,
 *                  3,
 *                  80,
 *                  0,
 *                  100,
 *                  bmps,
 *                  &picture_index );
 *     cWinApp_AddObj( main_module.m_process, ptr_menu_item, 0, 40 );
 *     ...
 *     // We have made changes then restored the previous state.
 *     cXByte_Esc(  ptr_menu_item  );
 *     ...
 *     cXByte_dtor( ptr_menu_item, FREE_MEMORY );
 *  @endcode
 *  @see cXByte_OK.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 165
#endif
void cXByte_Esc(struct cXByte* ptr_cxbyte);

//fn bool cXByte_Test(struct cXByte* ptr_cxbyte)
//brief Checks whether input string is an integer within the given bounds.
//param ptr_cxbyte is a pointer to initialized cXByte object.
//return TRUE if value belong to cXByte range.
#ifndef _SKIP_DOC
__cyos "CyWin" 166
bool cXByte_Test(struct cXByte*);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_Disconnect   cObject_Disconnect
#else
/**
 *  Disconnects a cXByte from its parent object.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 */
void cXByte_Disconnect (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_Select       cObject_Select
#else
/**
 *  Selects a cXByte object.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return TRUE if the object was selected
 */
bool cXByte_Select (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_update       cObject_update
#else
/**
 *  Updates a cXByte object
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 */
void cXByte_update (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_GetParent    cObject_GetParent
#else
/**
 *  Returns a pointer to the parent object.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return A pointer to the parent object
 */
struct cClip* cXByte_GetParent (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_Hide         cObject_Hide
#else
/**
 *  Hides a cXByte object.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 *  @see cXByte_Show.
 */
void cXByte_Hide (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_Show         cObject_Show
#else
/**
 *  Shows a cXByte object.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 *  @see cXByte_Hide.
 */
void cXByte_Show (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_Disable      cObject_Disable
#else
/**
 *  Disables the cXByte object so that it cannot be selected.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 *  @see cXByte_Enable.
 */
void cXByte_Disable (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_Enable       cObject_Enable
#else
/**
 *  Enables a cXByte object so that it may be selected.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 *  @see cXByte_Disable.
 */
void cXByte_Enable (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_GetTitle     cItem_GetTitle
#else
/**
 *  Returns the item's title.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return The item's title
 */
char* cXByte_GetTitle (struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_proc         cXStr_proc
#else
/**
 *  The Message-processing function.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @param ptr_message A pointer to the Message
 *  @return TRUE if the message was processed
 */
bool cXByte_proc(struct cXByte* ptr_cxbyte, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_SelLeft      cXItem_SelLeft
#else
/**
 *  Selects the section on the item's left side (only if the item has the focus).
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 *  @see cXByte_SelRight.
 */
void cXByte_SelLeft(struct cXByte* ptr_cxbyte);
#endif


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_SelRight      cXItem_SelRight
#else
/**
 *  Selects the section on the item's right side (only if the item has the focus).
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 *  @see cXByte_SelLeft.
 */
void cXByte_SelRight(struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_SetLeft      cXItem_SetLeft
#else
/**
 *  Changes the left side's value by 'index'.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @param index A new left side's value
 *  @return None
 *  @see cXByte_SetRight.
 */
void cXByte_SetLeft(struct cXByte* ptr_cxbyte, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_SetRight      cXItem_SetRight
#else
/**
 *  Changes the right side's value by 'index'.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @param index A new right side's value
 *  @return None
 *  @see cXByte_SetLeft.
 */
void cXByte_SetRight(struct cXByte* ptr_cxbyte, int index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_ScrollLR     cXStr_ScrollL
#else
/**
 *  Recalculates a StrBox.
 *  Should be called if you moved the cursor or made changes to the string on the right side while it was selected.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 */
void cXByte_ScrollLR(struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_SetStart     cXStr_SetStart
#else
/**
 *  Highlightes the entire string on the item's right side and positions the cursor at its end
 *  (only when the section on the item's right side is selected).
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 */
void cXByte_SetStart(struct cXByte* ptr_cxbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cXByte_SaveStr      cXStr_SaveStr
#else
/**
 *  Saves a Str to an external string.
 *  @param ptr_cxbyte A pointer to a cXByte object
 *  @return None
 */
void cXByte_SaveStr(struct cXByte* ptr_cxbyte);
#endif

#ifdef _SKIP_DOC
};
#endif

#endif  // #ifndef __LISTS_H_INCLUDED__
