/** 
 *  @file module.h
 *  Interfaces to CyOS executable modules.
 *
 */

#ifndef __MODULE_H_INCLUDED__
#define __MODULE_H_INCLUDED__

/** 
 *  @brief the CyOS Module.
 *  @ingroup multithreads_overview
 *  Modules are compound execution units. Thay consist of executable, relocatable
 *  sections that can be started in separate threads to create more threads and other CyOS objects.
 *  Modules know the archive files used to load them, and provide easy one-call
 *  access to the other resources stored inside them.
 *   @see multithreads_overview
 */
struct Module
{
#ifndef _SKIP_DOC
  char Module_Members_STUB_1[84];
#endif

  /**
   *  A pointer to the module's Archive object.
   */
  struct Archive* archive;

#ifndef _SKIP_DOC
  char Module_Members_STUB_2[32];
#endif
};

/**
 *  @ingroup multithreads_overview
 *  The result of the module's execution
 *  @see execute_module.
 */
enum modulestate_t
{
  /**
   *  Module was successfully initialized.
   */
  MODULE_OK = 0,

  /**
   *  Module was bad.
   */
  MODULE_BAD,

  /** 
   *  Intialization timeout.
   */
  MODULE_TIMEOUT
};

/**
 *  @ingroup multithreads_overview
 *  Loads and executes the module, then waits for it's module_main to exits.
 *  It blocks a timeout while module_main finishes, 
 *  then stores it's returned value in the retcode parameter. Use it
 *  when you need to get returned code, and don't mind waiting till
 *  the module_main exits.
 *  @param sz_command_line A command string to start the module
 *  @param ptr_retcode The storage site for the module's returned code
 *  @param timeout Determines how long to wait for module_main to finish
 *  @return The result of the execution (one of the modulestate_t members)
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       long retcode;
 *       ...
 *       // Executes an application.
 *       execute_module( "billiard.app", &retcode, 1000*60*10 );
 *       if( retcode ==  MODULE_TIMEOUT)
 *       {
 *         // The game time exceeded 10 minutes.  
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see start_module, modulestate_t
 */
#ifndef _SKIP_DOC
__cyos 555
#endif
int execute_module(char* sz_command_line, long* ptr_retcode, clock_t timeout);

/**
 *  @ingroup multithreads_overview
 *  Starts execution of the designated module (do not wait until it initializes!).
 *  @return TRUE if the module was launched, otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       long retcode;
 *       ...
 *       // Executes an application.
 *       if( !start_module( "billiard.app" ) )
 *       {
 *         // Unable to launch the 'Billiard' application.
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see execute_module.
 */
#ifndef _SKIP_DOC
__cyos 1177
#endif
bool start_module(char* command_line);

/**
 *  @ingroup filesystem_overview
 *  Returns the resource from the current archive, using resource index.
 *  @return Pointer to the initialized Input object.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Input* ptr_input;
 *       struct Bitmap bmp;
 *       ...
 *       Bitmap_ctor(&bmp);
 *       ...
 *       //  Your make file must have the archive resource "bmp.pic" linked 
 *       //  by the filer. "bmp.pic" must be the first in the resource 
 *       //  list (index = 0). For example, see the Cybiko Compiler's 
 *       //  Introduction Section
 *       ptr_input = open_resource( 0 );
 *       if( !Bitmap_load( &bmp, ptr_input ) )
 *           return FALSE;
 *       else
 *           ...
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see open_resource_Ex.
 */
#ifndef _SKIP_DOC
__cyos 920
#endif
struct Input* open_resource(int index);

/**
 *  @ingroup filesystem_overview
 *  Returns the resource from the current archive,  using the resource name.
 *  @return Pointer to the initialized Input object.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Input* ptr_input;
 *       struct Bitmap bmp;
 *       ...
 *       Bitmap_ctor(&bmp);
 *       ...
 *       //  Your make file must have the archive resource "bmp.pic" linked 
 *       //  by filer. For example, see the Cybiko Compiler's 
 *       //  Introduction Section
 *       ptr_input = open_resource_Ex( "bmp.pic" );
 *       if( !Bitmap_load( &bmp, ptr_input ) )
 *           return FALSE;
 *       else
 *           ...
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 *  @see open_resource.
 */
#ifndef _SKIP_DOC
__cyos 919
#endif
struct Input* open_resource_Ex(char* sz_file_name);

/**
 *  @ingroup multithreads_overview
 *  Increments the reference count for the current module
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       increment_usage();
 *       ...
 *       //  During this time, the module cannot be unloaded.
 *       ...
 *       decrement_usage();
 *       ...
 *  @endcode
 *  @see decrement_usage.
 */
#ifndef _SKIP_DOC
__cyos 770
#endif
void increment_usage(void);

/**
 *  @ingroup multithreads_overview
 *  Decrements the reference count for the current module.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       increment_usage();
 *       ...
 *       //  During this time, the module cannot be unloaded.
 *       ...
 *       decrement_usage();
 *       ...
 *  @endcode
 *  @see increment_usage.
 */
#ifndef _SKIP_DOC
__cyos 495
#endif
void decrement_usage(void);

#endif  //  #ifndef __MODULE_H_INCLUDED__

