/** 
 *  @file mplayer.h
 *  Consists the MSequence structure that allows you play of music sequences
 *  Make sure, you have mplayer.app.
 */

#ifndef __MPLAYER_H_INCLUDED__
#define __MPLAYER_H_INCLUDED__

/**
 *  @brief The music sequence object.
 *  @ingroup sound_overview
 *  The music sequence object. Allows the end user to manipulate the
 *  music-playing on Cybiko computers. You must call MSequence_ctor 
 *  before use and MSequence_dtor after use.
 *  @see MSequence_ctor, MSequence_dtor, sound_overview
 */
struct MSequence
{
#ifndef _SKIP_DOC
  char MSequence_Members_STUB[12];
};
#endif

/**
 *  Creates a music sequence from a specified resource (file) name.
 *  @param ptr_msequence A pointer to a MSequence structure.
 *  @param sz_file_name The music seqence's resource file name (*.mus)
 *  @return A pointer to the initialized music sequence
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       // Loops background music.
 *       if( !MSequence_is_playing ( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 5
#endif
struct MSequence* MSequence_ctor(struct MSequence* ptr_msequence, char* sz_file_name);

/**
 *  Checks whether the current content appears to be valid.
 *  @param ptr_msequence A pointer to the initialized music sequence
 *  @return TRUE if the content is valid
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       // Loops background music.
 *       if( !MSequence_is_playing ( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 0
#endif
bool MSequence_is_sane(struct MSequence* ptr_msequence);

/**
 *  Plays a music sequence in the foreground.
 *  @param ptr_msequence A pointer to the initialized music sequence.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if ( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 1
#endif
void MSequence_play(struct MSequence* ptr_msequence);

/**
 *  Stops playing a music sequence.
 *  @param ptr_msequence A pointer to the initialized music sequence.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if ( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       // Stops background music.
 *       if( MSequence_is_playing ( &intro_music ) )
 *       {
 *         MSequence_stop( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 2
#endif
void MSequence_stop(struct MSequence* ptr_msequence);

/**
 *  Checks whether a music sequence is playing.
 *  @param ptr_msequence A pointer to the initialized music sequence
 *  @return TRUE if music is playing
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       // Loops background music.
 *       if( !MSequence_is_playing ( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 3
#endif
bool MSequence_is_playing(struct MSequence* ptr_msequence);

/**
 *  Returns the current position (0xFFFF if it's about to start playing the first event)
 *  @param ptr_msequence A pointer to the initialized music sequence
 *  @return The current position
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       // Loops background music in the first of 20 music events.
 *       if( MSequence_get_curr_pos ( &intro_music ) > 20 )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 4
#endif
int MSequence_get_curr_pos(struct MSequence* ptr_msequence);

/**
 *  Loads a sequence from a specified Input stream.
 *  @param ptr_msequence A pointer to the initialized music sequence
 *  @param ptr_input A pointer to the Input stream structure of an opened 
 *  music resource
 *  @return TRUE if initialization succeeded
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       struct Input* ptr_input;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       ptr_input = open_recourse_Ex( "new_intro.mus" );
 *       MSequence_load( &intro_music, ptr_input );
 *       if( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       Input_dtor( ptr_input, FREE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 6
#endif
bool MSequence_load(struct MSequence* ptr_msequence, struct Input* ptr_input);

/**
 *  Destructor. Performs cleanup.
 *  @param ptr_msequence A pointer to the initialized music sequence.
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in a stack
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       // Loops background music.
 *       if( !MSequence_is_playing ( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 7
#endif
void MSequence_dtor(struct MSequence* ptr_msequence, int memory_flag);

/**
 *  Plays the a music sequence in the background.
 *  @param ptr_msequence A pointer to the initialized music sequence.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct MSequence intro_music;
 *       ...
 *       MSequence_ctor( &intro_music, "intro.mus" );
 *       ...
 *       if ( MSequence_is_scene( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       // Loops background music.
 *       if( !MSequence_is_playing ( &intro_music ) )
 *       {
 *         MSequence_play_background( &intro_music );
 *       }
 *       ...
 *       MSequence_dtor( &intro_music, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 9
#endif
void MSequence_play_background(struct MSequence* ptr_msequence);

/**
 *  Sets or clears the global mute flag.
 *  @param mute_flag If TRUE all sounds will be disabled; otherwise enabled
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       // Turns off music at all.
 *       MSequence_global_mute( TRUE );
 *       ...
 *       // Turns on music again.
 *       MSequence_global_mute( FALSE );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 10
#endif
void MSequence_global_mute(bool mute_flag);

/**
 *  Sets or clears the foreground music(effects) mute flag.
 *  @param mute_flag If TRUE, foreground sounds will be disabled; 
 *  otherwise enabled
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       //  Turns off sound effects.
 *       MSequence_mute_foreground( TRUE );
 *       ...
 *       //  Turns on sound effects again.
 *       MSequence_mute_foreground( FALSE );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 11
#endif
void MSequence_mute_foreground(bool mute_flag);

/**
 *  Sets or clears the background music(effects) mute flag.
 *  @param mute_flag If TRUE background sounds will be disabled; 
 *  otherwise enabled
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       // Turns off background sound.
 *       MSequence_mute_background( TRUE );
 *       ...
 *       // Turns on background sound again.
 *       MSequence_mute_background( FALSE );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "mplayer" 12
#endif
void MSequence_mute_background(bool mute_flag);

#ifdef _SKIP_DOC
};
#endif

#endif  //  #ifndef __MPLAYER_H_INCLUDED__

