/**
 *  @file object.h
 *  Hooks to CyWin cObject and cClip classes.
 */

#ifndef __OBJECT_H_INCLUDE__
#define __OBJECT_H_INCLUDE__

//============================
//=                          =
//=      object cObject.     =
//=     Base CyWin object.   =
//=                          =
//============================

/**
 *  @ingroup base_group
 *  @brief represents a simple graphic object, and implements base message-delivery and command-processing functions.
 *
 *  The CyOS base object.
 *  Represents a simple graphic object and implements base message-delivery
 *  and command-processing functions.
 *   @see base_group
 */
struct cObject
{
#ifndef _SKIP_DOC
  char cObject_Members_STUB[20];
};

/**
 *  @brief A base window structure - a clipped box, which may contains other objects.
 *  @ingroup base_group
 *  The CyOS base object.
 *  This is a base window structure - a clipped box, which may contains other objects. cClip implements base message-delivery and command-processing functions. It supports scrolling, painting child objects, and so on. You must initialize the structure before use and release object resources after use.
 *  @see base_group
 */
struct cClip : public cObject
{
  char cClip_Members_STUB[28];
};

/**
 *  Constructor.
 *  @param ptr_object a pointer to the cObject structure.
 *  @return a pointer to initialized the cObject structure.
 */
__cyos "CyWin" 227
struct cObject* cObject_ctor(struct cObject* ptr_object);

/**
 *  Destructor. Automaticaly disconnects an object from it parent.
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @param memory_flag can be FREE_MEMORY or LEAVE_MEMORY. If memory_flag is
 *  FREE_MEMORY when destructor frees memory , allocated for the object.
 *  FREE_MEMORY used if memory was allocated by malloc().
 *  If object was static or allocated in stack use LEAVE_MEMORY.
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
__cyos "CyWin" 182
void cObject_dtor(struct cObject*, int);

/**
 *  Disconnects an object form it parent.
 *  You must call this function in destructors of your object-derived
 *  structures, if destructor blocks object painting.
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @return none.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       cBitmap_ctor( &cbmp, &bmp );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
 *       ...
 *       // Removes the object from the Cybiko screen. It's the same as use the cObject_Disconnect( &cbmp ) fnuction.
 *       cBitmap_Disconnect( &cbmp );
 *       ...
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
 *       ...
 *       // Afterwards this function disconnects the cBitmap object automatically.
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 */
__cyos "CyWin" 184
void cObject_Disconnect(struct cObject* ptr_object);

/** @fn bool cObject_Select(struct cObject* ptr_object)
 *   Selects an object.
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       // It's the same as use the cButton_Select( &button ) function.
 *       // Selects the button.
 *       cObject_Select( &button );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @return TRUE if object was selected.
 */
__cyos "CyWin" 186
bool cObject_Select(struct cObject*);

/** @fn void cObject_update(struct cObject* ptr_object)
 *   Sends signal to the system, that object must be redrawed.
 *  After you change state of an object you must call this function to redraw your object.
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       cWinApp_clear_screen();
 *       // It's the same as use the cButton_update( &button ) function.
 *       // Redraws the button.
 *       cObject_update( &button )
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
__cyos "CyWin" 188
void cObject_update(struct cObject*);

/** @fn struct cClip* cObject_GetParent(struct cObject* ptr_object)
 *   Returns the parent of cClip structure.
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       struct cClip* parent;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       // It's the same as use the cButton_GetParent( &button ) function.
 *       // Checks for right parent relation.
 *       parent = cObject_GetParent( &button );
 *       if ( parent == main_module.m_process )
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 *  @return A pointer to the parent object.
 */
__cyos "CyWin" 221
struct cClip* cObject_GetParent(struct cObject*);

/** @fn void cObject_Hide(struct cObject* ptr_object)
 *    Hides an object.
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       struct cClip* parent;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       // It's the same as use the cButton_Hide( &button ) function.
 *       cObject_Hide( &button );
 *       ...
 *       // It's the same as use the cButton_Show( &button ) function.
 *       cObject_Show( &button );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
__cyos "CyWin" 223
void cObject_Hide(struct cObject*);

/** @fn void cObject_Show(struct cObject* ptr_object)
 *   Shows an object.
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       struct cClip* parent;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       // It's the same as use the cButton_Hide( &button ) function.
 *       cObject_Hide( &button );
 *       ...
 *       // It's the same as use the cButton_Show( &button ) function.
 *       cObject_Show( &button );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 224
#endif
void cObject_Show(struct cObject*);

/** @fn void cObject_Disable(struct cObject* ptr_object)
 *   Disables an object (disabled object can not be selected).
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       struct cClip* parent;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       // It's the same as use the cButton_Disable( &button ) function.
 *       // Now the user can't use this object.
 *       cObject_Disable( &button )
 *       ...
 *       // It's the same as use the cButton_Enable( &button ) function.
 *       // Now the user can use this object again.
 *       cObject_Enable( &button )
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
__cyos "CyWin" 225
void cObject_Disable(struct cObject*);

/** @fn void cObject_Enable(struct cObject* ptr_object)
 *   Enables an object (enabled object can be selected).
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       struct cClip* parent;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       // It's the same as use the cButton_Disable( &button ) function.
 *       // Now the user can't use this object.
 *       cObject_Disable( &button )
 *       ...
 *       // It's the same as use the cButton_Enable( &button ) function.
 *       // Now the user can use this object again.
 *       cObject_Enable( &button )
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
__cyos "CyWin" 226
void cObject_Enable(struct cObject*);

/** @fn bool cObject_proc(struct cObject* ptr_object, struct Message* ptr_message)
 *   Message-processing function.
 *  @param ptr_object is a pointer to initialized cObject structure.
 *  @param ptr_message is a pointer to processed message.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       cBitmap_ctor( &cbmp, &bmp );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
 *       ...
 *       ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *       // It's the same as use the cObject_proc( &cbmp, ptr_message ) function.
 *       // Processes mesages what manages cBitmap control.
 *       cBitmap_proc( &cbmp, ptr_message );
 *       ...
 *       // Afterwards this function disconnects the cBitmap object automatically.
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @return TRUE if message was processed.
 */
__cyos "CyWin" 185
bool cObject_proc(struct cObject*, struct Message*);

#else
};
#endif

//============================
//=                          =
//=       object cClip       =
//=    Clipped box object.   =
//=                          =
//============================

#ifdef _SKIP_DOC
/**
 *   @brief This is a base window structure - a clipped box, which may contain other objects.
 *   @ingroup base_group
 *   This is a base window structure - a clipped box, which may contain other objects.
 *   cClip implements base message-delivery and command-processing functions.
 *   It supports scrolling, painting child objects, and so on.
 *   You must the initialize structure before use and release object resources after use.
 *   @see base_group
 */
struct cClip : public cObject
{
#endif

/**  Scroll mode.
*/
enum tScrolls
{
  ///  no scrolling available.
  sNone     = 0,
  ///  enable scrolling up.
  sUp       = 1,
  ///  enable scrolling down.
  sDown     = 2,
  ///  enable scrolling both up and down.
  sUpDown   = 3
};

/**
 *  Constructor.
 *  @param ptr_clip is a pointer to cClip structure.
 *  @param width is the width of the clipped box.
 *  @param height is the height of the clipped box.
 *  @return a pointer to initialized cClip structure.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 204
struct cClip* cClip_ctor(struct cClip* ptr_clip, int width, int height);
#endif

//fn void cClip_dtor(struct cClip* ptr_clip, int memory_flag)
//brief Destructor.
//It destroys also all child objects.
//param ptr_clip is a pointer to initialized cClip structure.
//param memory_flag can be FREE_MEMORY or LEAVE_MEMORY. If memory_flag is
//FREE_MEMORY when destructor frees memory , allocated for the object.
//FREE_MEMORY used if memory was allocated by malloc().
//If object was static or allocated in stack use LEAVE_MEMORY.
//@see FREE_MEMORY, LEAVE_MEMORY.
#ifndef _SKIP_DOC
__cyos "CyWin" 206
void cClip_dtor(struct cClip*, int);
#endif

/** @fn void cClip_AddObj(struct cClip* ptr_clip, struct cObject* ptr_object, int x, int y)
 *  Adds the object 'obj' to the cClip in position (x,y).
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @param x The x-coordinate of the new object.
 *  @param y The y-coordinate of the new object.
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       // It's the same as using the cClip_AddObj( &button ) function.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 220
#endif
void cClip_AddObj(struct cClip*, struct cObject*, int, int);

/** @fn void cClip_InsObj(struct cClip* ptr_clip, struct cObject* ptr_object, int x, int y, int index)
 *   Adds the object 'obj' to the cClip in position (x,y), with a z-position index (maximal index is on the top).
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param ptr_object A pointer to initialized cObject structure
 *  @param x The x-coordinate of the new object
 *  @param y The y-coordinate of the new object
 *  @param index The z-position of the new object
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button1;
 *       struct cButton button2;
 *       struct cButton button3;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button1, "Button text 1", mrOk );
 *       cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *       ...
 *       cButton_ctor( &button2, "Button text 2", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *       ...
 *       // It's the same as using the cClip_InsObj( &button3 ) function.
 *       cButton_ctor( &button3, "Button text 3", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 207
#endif
void cClip_InsObj(struct cClip*, struct cObject*, int, int, int);

/** @fn void cClip_RemObj(struct cClip* ptr_clip, struct cObject* ptr_object)
 *   Removes the object from cClip.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param ptr_object A pointer to the initialized cObject structure
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button, "Button text", mrOk );
 *       // Puts the 'button' object on the Cybiko screen.
 *       // It's the same as using the cClip_AddObj( &button ) function.
 *       cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
 *       ...
 *       // It's the same as using the cClip_RemObj( &button ) function.
 *       cClip_RemObj( main_module.m_process, &button );
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 189
#endif
void cClip_RemObj(struct cClip*, struct cObject*);

/** @fn bool cClip_SelectFirst(struct cClip* ptr_clip)
 *   Selects the first object in the cClip.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return FALSE if the operation failed
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button1;
 *       struct cButton button2;
 *       struct cButton button3;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button1, "Button text 1", mrOk );
 *       cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *       ...
 *       cButton_ctor( &button2, "Button text 2", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *       ...
 *       // It's the same as using the cClip_InsObj( &button3 ) function.
 *       cButton_ctor( &button3, "Button text 3", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *       ...
 *       // Selects the first button.
 *       cClip_SelectFirst( main_module.m_process );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 197
#endif
bool cClip_SelectFirst(struct cClip*);

/** @fn bool cClip_SelectPrev(struct cClip* ptr_clip, bool round)
 *   Selects the  previous object in the cClip.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the operation failed
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button1;
 *       struct cButton button2;
 *       struct cButton button3;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button1, "Button text 1", mrOk );
 *       cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *       ...
 *       cButton_ctor( &button2, "Button text 2", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *       ...
 *       // It's the same as using the cClip_InsObj( &button3 ) function.
 *       cButton_ctor( &button3, "Button text 3", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *       ...
 *       // Selects the first button.
 *       cClip_SelectFirst( main_module.m_process );
 *       ...
 *       // Selects the previous button.
 *       cClip_SelectPrev( main_module.m_process, TRUE );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 198
#endif
bool cClip_SelectPrev(struct cClip*, bool);

/** @fn bool cClip_SelectNext(struct cClip* ptr_clip, bool round)
 *   Selects the next object in the cClip.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param round TRUE if you need to select the last object after selecting the first
 *  @return FALSE if the operation failed
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button1;
 *       struct cButton button2;
 *       struct cButton button3;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button1, "Button text 1", mrOk );
 *       cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *       ...
 *       cButton_ctor( &button2, "Button text 2", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *       ...
 *       // It's the same as using the cClip_InsObj( &button3 ) function.
 *       cButton_ctor( &button3, "Button text 3", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *       ...
 *       // Selects the first button.
 *       cClip_SelectFirst( main_module.m_process );
 *       ...
 *       // Selects the next button.
 *       cClip_SelectNext( main_module.m_process, TRUE );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 199
#endif
bool cClip_SelectNext(struct cClip*, bool);

/** @fn void cClip_Scroll(struct cClip* ptr_clip, struct rect_t* rectangle)
 *   Scrolls the cClip to make a specified rectangle visible. Provides minimal scrolling.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param rectangle A pointer to the initialized rect_t structure
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       struct rect_t   rect;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       cBitmap_ctor( &cbmp, &bmp );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *       ...
 *       rect_set( &rect, 160, 240, bmp.h, bmp.w );
 *       // Scrolls the bitmap to the visible area.
 *       cClip_Scroll( main_module.m_process, &rect );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 201
#endif
void cClip_Scroll(struct cClip* ptr_clip, struct rect_t* rectangle);

/** @fn void cClip_Scroll_Ex(struct cClip* ptr_clip, int x, int y)
 *  Scrolls the cClip by ( x, y ), but does not exceed the child's boundaries.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param x "x" coordinate shift
 *  @param y "y" coordinate shift
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       cBitmap_ctor( &cbmp, &bmp );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *       ...
 *       // Scrolls the bitmap to the visible area.
 *       cClip_Scroll_Ex( main_module.m_process, -160, -240 );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 203
#endif
void cClip_Scroll_Ex(struct cClip* ptr_clip, int x, int y);

/** @fn void cClip_SendScroll(struct cClip* ptr_clip)
 *   Forces this object to redraw its scrolling arrows.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       cBitmap_ctor( &cbmp, &bmp );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *       ...
 *       // Scrolls the bitmap to the visible area.
 *       cClip_Scroll_Ex( main_module.m_process, -160, -240 );
 *       cClip_SendScroll( main_module.m_process );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 211
#endif
void cClip_SendScroll(struct cClip* ptr_clip);

/** @fn int cClip_GetShifty(struct cClip* ptr_clip)
 *   Returns the 'y' coordinate shift (as a result of scrolling).
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return 'y' coordinate shift
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       int    nShift_Y;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       cBitmap_ctor( &cbmp, &bmp );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *       ...
 *       // Scrolls the bitmap to the visible area.
 *       cClip_Scroll_Ex( main_module.m_process, -160, -240 );
 *       cClip_SendScroll( main_module.m_process );
 *       // nShift_Y will be equal to -240.
 *       nShift_Y = cClip_GetShifty( main_module.m_process );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see cClip_GetShiftx.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 212
#endif
int cClip_GetShifty(struct cClip* ptr_clip);

/** @fn int cClip_GetShiftx(struct cClip* ptr_clip)
 *   Returns the 'x' coordinate shift (as a result of scrolling).
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return 'x' coordinate shift
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       int    nShift_X;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       cBitmap_ctor( &cbmp, &bmp );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 160, 240 );
 *       ...
 *       // Scrolls the bitmap to the visible area.
 *       cClip_Scroll_Ex( main_module.m_process, -160, -240 );
 *       cClip_SendScroll( main_module.m_process );
 *       // nShift_X will be equal to -160.
 *       nShift_X = cClip_GetShiftx( main_module.m_process );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see cClip_GetShifty.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 213
#endif
int cClip_GetShiftx(struct cClip* ptr_clip);

/** @fn int cClip_GetCount(struct cClip* ptr_clip)
 *   Returns the count of child objects.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return Child objects count
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button1;
 *       struct cButton button2;
 *       struct cButton button3;
 *       struct module_t main_module;
 *       int    nButtonNumber;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button1, "Button text 1", mrOk );
 *       cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *       ...
 *       cButton_ctor( &button2, "Button text 2", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *       ...
 *       // It's the same as using the cClip_InsObj( &button3 ) function.
 *       cButton_ctor( &button3, "Button text 3", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *       ...
 *       // nButtonNumber will be equal to 3.
 *       nButtonNumber = cClip_GetCount( main_module.m_process );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 216
#endif
int cClip_GetCount(struct cClip* ptr_clip);

/** @fn struct cObject* cClip_get_by_index(struct cClip* ptr_clip, int index)
 *   Returns an object with its index.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param index The index of the object in cClip
 *  @return If the index is valid, a pointer to the cObject under the specified index; else NULL.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button1;
 *       struct cButton button2;
 *       struct cButton button3;
 *       struct module_t main_module;
 *       struct cObject * currentObj;
 *       int    nButtonNumber;
 *       int    i;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button1, "Button text 1", mrOk );
 *       cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *       ...
 *       cButton_ctor( &button2, "Button text 2", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *       ...
 *       // It's the same as using the cClip_InsObj( &button3 ) function.
 *       cButton_ctor( &button3, "Button text 3", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *       ...
 *       // Redraws all objects.
 *       for ( i=0; i<cClip_GetCount( main_module.m_process ); i++)
 *       {
 *         cButton_update( cClip_get_by_index( main_module.m_process, i) );
 *       }
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 217
#endif
struct cObject* cClip_get_by_index(struct cClip*, int);

/** @fn int cClip_FindObj(struct cClip* ptr_clip, struct cObject* ptr_object)
 *   Returns the child object's index in the cClip.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param ptr_object A pointer to the needed cObject
 *  @return Object index; -1 if the index cannot be found
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button1;
 *       struct cButton button2;
 *       struct cButton button3;
 *       struct module_t main_module;
 *       struct cObject * currentObj;
 *       int    nButtonNumber;
 *       int    i;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button1, "Button text 1", mrOk );
 *       cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *       ...
 *       cButton_ctor( &button2, "Button text 2", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *       ...
 *       // It's the same as using the cClip_InsObj( &button3 ) function.
 *       cButton_ctor( &button3, "Button text 3", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *       ...
 *       // Finds the object.
 *       if ( cClip_FindObj( main_module.m_process, &button2 ) != -1 )
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 218
#endif
int cClip_FindObj(struct cClip*, struct cObject*);

/** @fn struct cObject* cClip_GetSelectedObject(struct cClip* ptr_clip)
 *   Returns the currently selected object in the cClip.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return The currently selected object in the cClip
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cButton button1;
 *       struct cButton button2;
 *       struct cButton button3;
 *       struct module_t main_module;
 *       struct cObject * selectedObj;
 *       int    nButtonNumber;
 *       int    i;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cButton_ctor( &button1, "Button text 1", mrOk );
 *       cWinApp_AddObj( main_module.m_process, &button1, 20, 50 );
 *       ...
 *       cButton_ctor( &button2, "Button text 2", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button2, 20, 50, 1 );
 *       ...
 *       // It's the same as using the cClip_InsObj( &button3 ) function.
 *       cButton_ctor( &button3, "Button text 3", mrOk );
 *       cWinApp_InsObj( main_module.m_process, &button3, 20, 50, 2 );
 *       ...
 *       cObject_Select( &button2 );
 *       ...
 *       // Redraws selected objects.
 *       cButton_update( cClip_GetSelectedObject( main_module.m_process );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 219
#endif
struct cObject* cClip_GetSelectedObject(struct cClip* ptr_clip);

/**
 *  Message-processing function.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @param ptr_message A pointer to the processed message
 *  @return TRUE if the message was processed
 *  @code
 *  #include <cywin.h>
 *  ...
 *  struct cClip* ptr_messages_clip;
 *  struct Message* ptr_message;
 *  ...
 *  case KEY_DOWN:
 *    cClip_proc( ptr_chat_form->ptr_messages_clip, ptr_message );
 *    return TRUE;
 *  ...
 *  @endcode
*/
#ifndef _SKIP_DOC
__cyos "CyWin" 202
#endif
bool cClip_proc(struct cClip* ptr_clip, struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cClip_Disconnect   cObject_Disconnect
#else
/** @fn void cClip_Disconnect (struct cClip* ptr_clip)
 *  Disconnects the cClip from its parent object.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return None
 *  @code
 *  @endcode
 */
void cClip_Disconnect (struct cClip* ptr_clip);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cClip_Select       cObject_Select
#else
/** @fn bool cClip_Select (struct cClip* ptr_clip)
 *  Selects the cClip object.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return TRUE if the object was selected
 *  @code
 *  @endcode
 */
bool cClip_Select (struct cClip* ptr_clip);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cClip_update       cObject_update
#else
/** @fn void cClip_update (struct cClip* ptr_clip)
 *  Updates a cClip object.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return None
 *  @code
 *  @endcode
 */
void cClip_update (struct cClip* ptr_clip);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cClip_GetParent    cObject_GetParent
#else
/** @fn struct cClip* cClip_GetParent (struct cClip* ptr_clip)
 *  Returns a pointer to the parent object's cClip.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return A pointer to the parent object's cClip
 *  @code
 *  @endcode
 */
struct cClip* cClip_GetParent (struct cClip* ptr_clip);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cClip_Hide         cObject_Hide
#else
/** @fn void cClip_Hide (struct cClip* ptr_clip)
 *  Hides the cClip object.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return None
 *  @code
 *  @endcode
 */
void cClip_Hide (struct cClip* ptr_clip);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cClip_Show         cObject_Show
#else
/** @fn void cClip_Show (struct cObject* ptr_clip)
 *  Shows the cClip object.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return None
 *  @code
 *  @endcode
 */
void cClip_Show (struct cObject* ptr_clip);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cClip_Disable      cObject_Disable
#else
/** @fn void cClip_Disable (struct Clip* ptr_clip)
 *  Disables the cClip object so that it cannot be selected.
 *  @param ptr_clip A pointer to the initialized cClip structure
 *  @return None
 *  @code
 *  @endcode
 */
void cClip_Disable (struct Clip* ptr_clip);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cClip_Enable       cObject_Enable
#else
/** @fn void cClip_Enable (struct cObject* ptr_clip)
 *  Enables the cClip object - so that it may be selected.
 *  @param ptr_clip is A pointer to the initialized cClip structure
 *  @return None
 *  @code
 *  @endcode
 */
void cClip_Enable (struct cObject* ptr_clip);
#endif

#ifdef _SKIP_DOC
};
#endif

#endif  //  #ifndef __OBJECT_H_INCLUDE__
