/**
 *  @file os_filename.h
 *  CyOS processing of the messages.
 *
 */
#ifndef __OS_FILENAME_H_INCLUDED__
#define __OS_FILENAME_H_INCLUDED__

/**
 *  @ingroup filesystem_overview
 *  Name of MP3 player device.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "Names of the devices:" );
 *       TRACE( "\tMp3 player: %s", MP3_DRIVE_NAME );
 *       TRACE( "\tDrive A: %s", DRIVE_A_NAME );
 *       TRACE( "\tDrive B: %s", DRIVE_B_NAME );
 *       TRACE( "\tDefault Drive: %s", DEFAULT_DRIVE_NAME );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1412 
#endif
char MP3_DRIVE_NAME[];

/**
 *  @ingroup filesystem_overview
 *  Name of the default flash device.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "Names of the devices:" );
 *       TRACE( "\tMp3 player: %s", MP3_DRIVE_NAME );
 *       TRACE( "\tDrive A: %s", DRIVE_A_NAME );
 *       TRACE( "\tDrive B: %s", DRIVE_B_NAME );
 *       TRACE( "\tDefault Drive: %s", DEFAULT_DRIVE_NAME );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1409
#endif 
char DEFAULT_DRIVE_NAME[];

/**
 *  @ingroup filesystem_overview
 *  Name of flash device A.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "Names of the devices:" );
 *       TRACE( "\tMp3 player: %s", MP3_DRIVE_NAME );
 *       TRACE( "\tDrive A: %s", DRIVE_A_NAME );
 *       TRACE( "\tDrive B: %s", DRIVE_B_NAME );
 *       TRACE( "\tDefault Drive: %s", DEFAULT_DRIVE_NAME );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1410
#endif 
char DRIVE_A_NAME[];

/**
 *  @ingroup filesystem_overview
 *  Name of flash device B.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "Names of the devices:" );
 *       TRACE( "\tMp3 player: %s", MP3_DRIVE_NAME );
 *       TRACE( "\tDrive A: %s", DRIVE_A_NAME );
 *       TRACE( "\tDrive B: %s", DRIVE_B_NAME );
 *       TRACE( "\tDefault Drive: %s", DEFAULT_DRIVE_NAME );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1411
#endif 
char DRIVE_B_NAME[];

/**
 *  @ingroup filesystem_overview
 *  Retrieves the maximum file name length.
 *  @return Maximum length of the file name.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       char* sz_tmp_full_name;
 *       char sz_tmp_name[16]; 
 *       int max_file_name = mFileName_max_file_name_length();
 *       ...
 *       //  Generates name for the temporary file;
 *       sz_tmp_full_name = (char*) malloc( max_file_name );
 *       do
 *       {
 *         sprintf( sz_tmp_name, "%ld.tmp", time() );  
 *         mFileName_make_path( sz_tmp_full_name, 
 *                              mFileName_current_device_name(),
 *                              sz_tmp_name );
 *       }while( ! mFileName_is_valid_file_name( sz_tmp_full_name ) );
 *       ...
 *       free( sz_tmp_full_name );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1448 
#endif
int mFileName_max_file_name_length(void);

/** 
 *  @ingroup filesystem_overview
 *  Verifies that the file name is valid.
 *  @param sz_full_name absolute name of the file.
 *  @return TRUE if name of the file is valid.  
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       char* sz_tmp_full_name;
 *       char sz_tmp_name[16]; 
 *       int max_file_name = mFileName_max_file_name_length();
 *       ...
 *       //  Generates name for the temporary file;
 *       sz_tmp_full_name = (char*) malloc( max_file_name );
 *       do
 *       {
 *         sprintf( sz_tmp_name, "%ld.tmp", time() );  
 *         mFileName_make_path( sz_tmp_full_name, 
 *                              mFileName_current_device_name(),
 *                              sz_tmp_name );
 *       }while( ! mFileName_is_valid_file_name( sz_tmp_full_name ) );
 *       ...
 *       free( sz_tmp_full_name );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 825
#endif 
bool mFileName_is_valid_file_name(char* sz_full_name); 

/**
 *  @ingroup filesystem_overview
 *  Splits the absolute file name into device name and file name.
 *  @param sz_full_name absolute file name.
 *  @param sz_device_name buffer in which the device name will be stored.
 *  @param sz_file_name buffer in which the file name will be stored.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       char* sz_application_name;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       //  Retrieves application name.
 *       sz_application_name = (char*) malloc( mFileName_max_file_name_length() );
 *       mFileName_split_path( Archive_archive_name( main_module.m_process->module->archive ), 
 *                             NULL,
 *                             sz_application_name );
 *       ...
 *       TRACE("Name of the application is %s", sz_application_name );
 *       ...
 *       free(sz_application_name);
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1168 
#endif
void mFileName_split_path(char* sz_full_name, char* sz_device_name, char* sz_file_name);

/**
 *  @ingroup filesystem_overview
 *  Creates an absolute file name by combining the file name with the device name.
 *  @param sz_full_name buffer in which the absolute file name will be stored.
 *  @param sz_device_name device name.
 *  @param sz_file_name file name.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       char* sz_tmp_full_name;
 *       char sz_tmp_name[16]; 
 *       int max_file_name = mFileName_max_file_name_length();
 *       ...
 *       //  Generates name for the temporary file;
 *       sz_tmp_full_name = (char*) malloc( max_file_name );
 *       do
 *       {
 *         sprintf( sz_tmp_name, "%ld.tmp", time() );  
 *         mFileName_make_path( sz_tmp_full_name, 
 *                              mFileName_current_device_name(),
 *                              sz_tmp_name );
 *       }while( ! mFileName_is_valid_file_name( sz_tmp_full_name ) );
 *       ...
 *       free( sz_tmp_full_name );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 875 
#endif
void mFileName_make_path(char* sz_full_name, char* sz_device_name, char* sz_file_name);

/**
 *  @ingroup filesystem_overview
 *  Retrieves the path to the file from the absolute file name.
 *  @param sz_full_name absolute name of the file. 
 *  @param sz_path buffer in which the file path will be stored.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       char* sz_application_path;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       //  Retrieves application name.
 *       sz_application_path = (char*) malloc( mFileName_max_file_name_length() );
 *       mFileName_get_path( Archive_archive_name( main_module.m_process->module->archive ), 
 *                           sz_application_path );
 *       ...
 *       TRACE("Path to the application is %s", sz_application_path );
 *       ...
 *       free(sz_application_path);
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1446
#endif 
void mFileName_get_path(char* sz_full_name, char* sz_path);

/**
 *  @ingroup filesystem_overview
 *  Retrieves the name of the current flash device.
 *  @return Name of the current flash device.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       char* sz_tmp_full_name;
 *       char sz_tmp_name[16]; 
 *       int max_file_name = mFileName_max_file_name_length();
 *       ...
 *       //  Generates name for the temporary file;
 *       sz_tmp_full_name = (char*) malloc( max_file_name );
 *       do
 *       {
 *         sprintf( sz_tmp_name, "%ld.tmp", time() );  
 *         mFileName_make_path( sz_tmp_full_name, 
 *                              mFileName_current_device_name(),
 *                              sz_tmp_name );
 *       }while( ! mFileName_is_valid_file_name( sz_tmp_full_name ) );
 *       ...
 *       free( sz_tmp_full_name );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1451 
#endif
char* mFileName_current_device_name(void);

#endif  //  #ifndef __OS_FILENAME_H_INCLUDED__

