/** 
 *  @file process.h
 *  CyOS processing of the messages.
 */
#ifndef __PROCESS_H_INCLUDED__
#define __PROCESS_H_INCLUDED__

/**
 *  Very low priority (for background tasks).
 * @ingroup multithreads_overview
 */
#define PR_IDLE    5

/**
 *  Normal (medium) priority.
 * @ingroup multithreads_overview
 */
#define PR_NORMAL  50

/**
 *  Very high priority (for runtime threads).
 * @ingroup multithreads_overview
 */
#define PR_RUNTIME 100

//============================
//=                          =
//=   object SystemThread.   =
//=   Basic execution unit.  =
//=                          =
//============================

/** 
 *  @brief Base object for all execution units (threads).
 *  @ingroup multithreads_overview
 *  Base object for all execution units (threads).
 *  This object defines the basic object that makes an execution thread.
 *   @see multithreads_overview
 */
struct SystemThread
{
#ifndef _SKIP_DOC
  char SystemThread_Members_STUB[40];
  bool wake_state;
  char wake_state_PADDING[3];
  void* controller;
  clock_t total_time;
  clock_t last_time;
  clock_t start_time;
  clock_t end_time;
  bool resume_status;
  char resume_status_PADDING[3];
  char* stack_top;
  char* stack_bottom;
  long flags;
#endif

  /**
   *  Pointer to the module to which the thread belongs.
   */
  struct Module* module;  

#ifndef _SKIP_DOC
  char priority;
  char saved_priority;
  char priority_PADDING[2];
  struct SystemThread* next;
  void* sp;
  void* sp_peak;
  void* queue;
};
#endif

/**
 *  Pauses execution of the thread.
 *  @param ptr_thread A pointer to the SystemThread initialized object
 *  @param timeout The time span to sleep, in milliseconds
 *  @return FALSE if the timeout has passed
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       play_tone( 30 );
 *       SystemThread_pause( main_module.m_process, 200 );
 *       play_tone( -1 );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 939
#endif
bool SystemThread_pause(struct SystemThread* ptr_thread, clock_t timeout);

/**
 *  Returns a thread's priority.
 *  @param ptr_system_thread A pointer to the SystemThread initialized object
 *  @return Priority of the thread
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       if( SystemThread_get_priority( main_module.m_process ) <  PR_NORMAL )
 *           SystemThread_set_priority( main_module.m_process, PR_NORMAL );
 *       ...
 *  @endcode
 *  @see SystemThread_set_priority, PR_IDLE, PR_NORMAL, PR_RUNTIME.
 */
#ifndef _SKIP_DOC
__cyos 711
#endif
priority_t SystemThread_get_priority(struct SystemThread* ptr_system_thread);

/**
 *  Changes the thread's priority.
 *  @param ptr_system_thread A pointer to the SystemThread initialized structure
 *  @param priority The thread priority, on a scale from PR_IDLE to PR_RUNTIME
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       if( SystemThread_get_priority( main_module.m_process ) <  PR_NORMAL )
 *           SystemThread_set_priority( main_module.m_process, PR_NORMAL );
 *       ...
 *  @endcode
 *  @see SystemThread_get_priority, PR_IDLE, PR_NORMAL, PR_RUNTIME.
 */
#ifndef _SKIP_DOC
__cyos 1147
#endif
void SystemThread_set_priority(struct SystemThread* ptr_system_thread, priority_t priority);

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=     object Process.      =
//=  Thread & message queue. =
//=                          =
//============================

/** @brief the main thread object for the CyOS.
 *  @ingroup multithreads_overview
 *  The CyOS threading model is based on the Threads' status.
 *  Threads are executed concurrently by priority: the higher the
 *  priority (bigger numerical value), the more time the Thread can have.
 *  No low-priority thread will be executed until there is at least one other
 *  thread with a higher priority; Threads of the same priority share more or
 *  less the same time slices for the CPU.
 *  A Thread's natural state of being is to wait. Threads spend most of the
 *  time waiting, for a timeout, a message, a file operation, a Flag or Mutex
 *  state, or many other reasons. If the OS dispatcher detects a Thread with a
 *  higher priority than PR_IDLE that does not spend much time waiting, the OS
 *  decides that it is a zombie thread (a dead thread that just eats system
 *  resources) and kills it.
 *   @see multithreads_overview
 */
struct Thread : public SystemThread
{
#ifndef _SKIP_DOC
  int stack_size;
  int stack_size_PADDING;
#endif
};

/**
 *  @brief a kind of thread that can process messages.
 *  @ingroup multithreads_overview
 *  A thread enabled to process messages.
 *  The System has the Thread function to do tasks, and the Process
 *  function to process messages such as input events from the keyboard,
 *  system messages, and radio-sent communication messages.
 *  The Process function can have input focus; you can and ask for it by
 *  calling Process_request_focus(). When the Process has the focus,
 *  it gets all input events from the keyboard.
 *  @see multithreads_overview
 */
struct Process : public Thread
{
#ifndef _SKIP_DOC
  char Process_Members_STUB[56];
};
#endif

/**
 *  Pauses execution of the Process.
 *  @param ptr_process A pointer to the Process initialized object
 *  @param timeout The time span to sleep, in milliseconds
 *  @return FALSE if the timeout is passed
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       play_tone(30);
 *       Process_pause(main_module.m_process, 200);
 *       play_tone(-1);
 *       ...
 *  @endcode
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Process_pause   SystemThread_pause
#else
bool Process_pause(struct Process* ptr_process, clock_t timeout);
#endif

/**
 *  Asks the system to activate the Process (pass the focus).
 *  Once it has the focus, the Process has actual access to the screen and receives the keyboard 
 *  and pointing device events. When the system gets this call, it passes the focus 
 *  to the Process that asked for it by sending it the message MSG_GOTFOCUS.
 *  Then the system sends the message MSG_LOSTFOCUS to the Process that had
 *  the focus previously.
 *  @param ptr_process A pointer to the initialized Process object
 *  @return Process that had the focus before this call
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       Process_request_focus( main_module.m_process );
 *       ...
 *       if( Process_has_focus( main_module.m_process ) )
 *       {
 *         //  Draws something.
 *         ...
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1034
#endif
struct Process* Process_request_focus(struct Process* ptr_process);

/**
 *  Tests whether the Process has the focus
 *  @param ptr_process A pointer to the initialized Process object
 *  @return TRUE if the Process has the focus, otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       Process_request_focus( main_module.m_process );
 *       ...
 *       if( Process_has_focus( main_module.m_process ) )
 *       {
 *         //  Draws something.
 *         ...
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 758
#endif
bool Process_has_focus(struct Process* ptr_process);

/**
 *  Retrieves a DisplayGraphics object to the access display.
 *  @return A pointer to the DisplayGraphics object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Bitmap bmp;
 *       ...
 *       //  Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       ...
 *       //  Draws bitmap.
 *       Graphics_draw_bitmap( Process_get_display(), &bmp, 30, 40, BM_NORMAL );
 *       ...
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 666
#endif
struct DisplayGraphics* Process_get_display(void);

/**
 *  Retrieves the Process' name.
 *  @param ptr_process A pointer to the initialized Process object
 *  @return The Process' name
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       TRACE("Hello from the %s process", 
 *             Process_get_name( main_module.m_process ) );
 *       ...
 *  @endcode
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Process_get_name(ptr_process)  Queue_get_name((char*)(ptr_process) + 108) ///< char* Process_get_name (struct Process* ptr_process) Returns the name of a process.
#else
char* Process_get_name(struct Process* ptr_process);
#endif

/**
 *  Puts a Message in the Process' Queue.
 *  @param ptr_process A pointer to the initialized Process object
 *  @param ptr_message A pointer to the message to send
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       #define MSG_MAKE_TASK  MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       ptr_message = Process_get_message( main_module.m_process, 0, 1, MSG_MAKE_TASK );
 *       if( ptr_message->msgid == MSG_MAKE_TASK )
 *       {
 *         ...
 *         //  Performs some periodical calculation.
 *         ...
 *         Process_pause( main_module.m_process, 250 );
 *         Process_put_message( main_module.m_process, ptr_message );
 *       }
 *       else
 *       {
 *         Message_delete( ptr_message );
 *       }
 *       ...
 *  @endcode
 *  @see Process_get_message , Process_peek_message.
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Process_put_message(ptr_process, ptr_message) Queue_put_message((char*)(ptr_process) + 108, (ptr_message)) ///< void Process_put_message (struct Process* ptr_process, struct Message* ptr_message) Puts a Message to the queue. It calls Queue_put_message.
#else
void Process_put_message(struct Process* ptr_process, struct Message* ptr_message);
#endif

/**
 *  Returns the message after the specified interval. The calling thread waits 
 *  until the first message appears in the queue. Be careful not to overload the queue
 *  overflow. Delete message when you finished with it.
 *  @param ptr_process A pointer to the initialized Process object.
 *  @param timeout Timeout value(in milliseconds)
 *  @param min deprecated
 *  @param max deprecated
 *  @return Pointer to the Message if one is in the Process' queue, 
 *  or 0 if a timeout occurs
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       ptr_message = Process_get_message( main_module.m_process, 0, 1, MSG_USER );
 *       if( ptr_message->msgid == MSG_KEYDOWN )
 *       {
 *         if( Message_get_key_param( ptr_message )->scancode == MSG_ESC )
 *         {
 *           //  Processes key 'Esc'.
 *         }
 *       }
 *       ...
 *       Message_delete( ptr_message );
 *       ...
 *  @endcode
 *  @see Process_peek_message.
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Process_get_message(ptr_process, timeout, min, max) \
        Queue_get_message((char*)(ptr_process) + 108, (timeout), (min), (max)) ///< struct Message* Process_get_message (struct Process* ptr_process, long timeout, int min, int max) Returns the message from the specified interval, the calling thread waits until first message that fits the specified range will appear in the queue. Be careful with it, the queue may overflow. Delete message when you finished with it. It calls Queue_get_message.
#else
struct Message* Process_get_message(struct Process* ptr_process, long timeout, int min, int max);
#endif

/**
 *  Peeks or gets message from the queue. 
 *  If remove is TRUE, the message will be removed. This function does not wait for a message 
 *  to be available; it returns 0 immediately if there are no messages in 
 *  the specified range. Don't use this function unless you really need to.
 *  @param ptr_process A pointer to the initialized Process object
 *  @param remove If TRUE the message will be removed from the message queue
 *  @param min Specifies the lowest message value to be retrieved, as an integer value
 *  @param max Specifies the highest message value to be retrieved, as an integer value 
 *  @return A pointer to the Message if one is in the Process' queue, 
 *  otherwise 0 
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       //  Removes all keyboard messages from the queue.
 *       while(ptr_message = Process_peek_message( main_module.m_process, TRUE, MSG_KEYDOWN, MSG_KEYDOWN ))
 *       {
 *         Message_delete( ptr_message );
 *       }
 *       ...
 *  @endcode
 *  @see Process_get_message.
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Process_peek_message(ptr_process, remove, min, max) \
        Queue_peek_message((char*)(ptr_process) + 108, (remove), (min), (max))
#else
struct Message* Process_peek_message(struct Process* ptr_process, bool remove, int min, int max);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *  Pointer to the current thread.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE("Hello from the %s process!", 
 *             Process_get_name( current_thread ) );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 124 
#endif
struct Thread* current_thread;

/**
 *  @ingroup multithreads_overview
 *  Tests whether the current process has the focus
 *  @return TRUE if the current process has the focus, otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       Process_request_focus( main_module.m_process );
 *       ...
 *       if( has_focus() )
 *       {
 *         //  Draws something.
 *         ...
 *       }
 *       ...
 *  @endcode
 */
#ifdef _REPLACE_DEFINE_BY_FUNCTION
bool has_focus(void);
#else
#define has_focus()  Process_has_focus( current_thread ) 
#endif

/**
 *  @ingroup multithreads_overview
 *  Pauses execution of the current thread.
 *  @param The time span to sleep, measured in milliseconds
 *  @return FALSE if timeout has passed.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       play_tone( 30 );
 *       sleep( 200 );
 *       play_tone( -1 );
 *       ...
 *  @endcode
 */
#ifdef _REPLACE_DEFINE_BY_FUNCTION
bool sleep(clock_t timeout);
#else
#define sleep(timeout)  SystemThread_pause( ( current_thread ), ( timeout ) )
#endif

#endif  //  #ifndef __PROCESS_H_INCLUDED__
