/** 
 *  @file score.h
 *  Hook to the high score archive.
 *
 *  Before use of this object not compressed file "score.inf" must be added 
 *  to the application archive. It must have size (number of records)*20 byte.
 *  To disable file compressing use "-" before score.inf in the filer file list.
 */
#ifndef __SCORE_H_INCLUDED__
#define __SCORE_H_INCLUDED__

/**
 *  @brief The game's high score structure.
 *  @ingroup misc_overview
 *  Structure of the high score of the game.
 *  @see Score, misc_overview
 */
struct score_t
{
  /**
   *  High score.
   */
  long score;
  /**
   *  Cy ID of the device on which the score was reached.
   */
  cyid_t cyid;
  /**
   *  Time the high score was reached.
   */
  time_t time;
  /**
   *  Cy name of the player, who reach the high score.
   */
  char nickname[8];
};

/**
 *  @brief The game's high score archive.
 *  @ingroup misc_overview
 *  The game's high score archive.
 *  @see score_t, misc_overview
 */
struct Score
{
#ifndef _SKIP_DOC
  struct Archive* ptr_archive;
  void* reserved;
  int count;
  int count_PADDING;
};
#endif

/**
 *  High score archive constructor. Before using this object,
 *  you must add an uncompressed "score.inf" file to the application archive.
 *  Specify its size as (number of records)*20 bytes.
 *  To disable file compression, use "-" before "score.inf" in the filer's file list.
 *  @param ptr_score A pointer to the high score archive structure
 *  @param ptr_module A pointer to the application's thread module
 *  @return A pointer to the initialized high score archive object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int record_number;
 *       struct module_t main_module;
 *       struct Score score_archive;
 *       struct score_t* high_scores;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       Score_ctor( &score_archive, main_module.m_process->module );
 *       if( Score_is_valid ( &score_archive ) )
 *       {
 *         record_number = Score_get_record_count( &score_archive );
 *         high_scores = (struct score_t*)malloc( sizeof(struct score_t)*record_number );
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_read( &score_archive, index, high_scores + index );
 *         }
 *         //  Changes high_scores.
 *         ...
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_write( &score_archive, index, high_scores + index );
 *         }
 *         ...
 *         free(high_scores);
 *       }
 *       ...
 *       Score_dtor( &score_archive, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "gamelib.dl" 3
#endif
struct Score* Score_ctor(struct Score* ptr_score, struct Module* ptr_module);

/**
 *  High score archive destructor.
 *  @param ptr_score A pointer to the initialized high score archive object.
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY.
 *  If the memory was allocated for the object by malloc(),
 *  use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or  
 *  allocated in a stack
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int record_number;
 *       struct module_t main_module;
 *       struct Score score_archive;
 *       struct score_t* high_scores;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       Score_ctor( &score_archive, main_module.m_process->module );
 *       if( Score_is_valid ( &score_archive ) )
 *       {
 *         record_number = Score_get_record_count( &score_archive );
 *         high_scores = (struct score_t*)malloc( sizeof(struct score_t)*record_number );
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_read( &score_archive, index, high_scores + index );
 *         }
 *         //  Changes high_scores.
 *         ...
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_write( &score_archive, index, high_scores + index );
 *         }
 *         ...
 *         free(high_scores);
 *       }
 *       ...
 *       Score_dtor( &score_archive, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "gamelib.dl" 4
#endif
void Score_dtor(struct Score* ptr_score, int memory_flag);

/**
 *  Checks the validity of the high score archive.
 *  @param ptr_score A pointer to the initialized high score archive object
 *  @return TRUE if the high score archive is valid, otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int record_number;
 *       struct module_t main_module;
 *       struct Score score_archive;
 *       struct score_t* high_scores;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       Score_ctor( &score_archive, main_module.m_process->module );
 *       if( Score_is_valid ( &score_archive ) )
 *       {
 *         record_number = Score_get_record_count( &score_archive );
 *         high_scores = (struct score_t*)malloc( sizeof(struct score_t)*record_number );
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_read( &score_archive, index, high_scores + index );
 *         }
 *         //  Changes high_scores.
 *         ...
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_write( &score_archive, index, high_scores + index );
 *         }
 *         ...
 *         free(high_scores);
 *       }
 *       ...
 *       Score_dtor( &score_archive, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Score_is_valid(ptr_score)          ((ptr_score)->count) 
#else
bool Score_is_valid( struct Score* ptr_score );
#endif

/** 
 *  Returns the number of records in the high score archive. 
 *  @param ptr_score A pointer to the initialized high score archive object
 *  @return Number of records in the high score archive
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int record_number;
 *       struct module_t main_module;
 *       struct Score score_archive;
 *       struct score_t* high_scores;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       Score_ctor( &score_archive, main_module.m_process->module );
 *       if( Score_is_valid ( &score_archive ) )
 *       {
 *         record_number = Score_get_record_count( &score_archive );
 *         high_scores = (struct score_t*)malloc( sizeof(struct score_t)*record_number );
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_read( &score_archive, index, high_scores + index );
 *         }
 *         //  Changes high_scores.
 *         ...
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_write( &score_archive, index, high_scores + index );
 *         }
 *         ...
 *         free(high_scores);
 *       }
 *       ...
 *       Score_dtor( &score_archive, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Score_get_record_count(ptr_score)  ((ptr_score)->count) 
#else
int Score_get_record_count(struct Score* ptr_score);
#endif

/**
 *  Reads the record from the high score archive.
 *  @param ptr_score A pointer to the initialized high score archive object
 *  @param index The index of the record in the high score archive
 *  @param ptr_result A pointer to the buffer that receives the score
 *  @return TRUE if the record was read successfully
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int record_number;
 *       struct module_t main_module;
 *       struct Score score_archive; 
 *       struct score_t* high_scores; 
 *       ...
 *       init_module( &main_module );
 *       ...
 *       Score_ctor( &score_archive, main_module.m_process->module );
 *       if( Score_is_valid ( &score_archive ) )
 *       {
 *         record_number = Score_get_record_count( &score_archive );
 *         high_scores = (struct score_t*)malloc( sizeof(struct score_t)*record_number );
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_read( &score_archive, index, high_scores + index );
 *         }
 *         //  Changes high_scores. 
 *         ...
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_write( &score_archive, index, high_scores + index );
 *         }
 *         ...
 *         free(high_scores);
 *       }
 *       ...
 *       Score_dtor( &score_archive, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see score_t.
 */
#ifndef _SKIP_DOC
__cyos "gamelib.dl" 0
#endif
bool Score_read(struct Score* ptr_score, int index, struct score_t* ptr_result);

/**
 *  Writes the record to the high score archive.
 *  @param ptr_score Pointer to the initialized high score archive object
 *  @param index Index of the record in the high score archive
 *  @param ptr_result A pointer to the game's high score structure
 *  @return TRUE if the record was stored successfully
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int record_number;
 *       struct module_t main_module;
 *       struct Score score_archive;
 *       struct score_t* high_scores;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       Score_ctor( &score_archive, main_module.m_process->module );
 *       if( Score_is_valid ( &score_archive ) )
 *       {
 *         record_number = Score_get_record_count( &score_archive );
 *         high_scores = (struct score_t*)malloc( sizeof(struct score_t)*record_number );
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_read( &score_archive, index, high_scores + index );
 *         }
 *         //  Changes high_scores.
 *         ...
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_write( &score_archive, index, high_scores + index );
 *         }
 *         ...
 *         free(high_scores);
 *       }
 *       ...
 *       Score_dtor( &score_archive, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Score_write_Ex, score_t.
 */
#ifndef _SKIP_DOC
__cyos "gamelib.dl" 1
#endif
bool Score_write(struct Score* ptr_score,  int index, struct score_t* ptr_result);

/**
 *  Writes the record to the high score archive.
 *  @param ptr_score A pointer to the initialized high score archive object
 *  @param index The index of the record in the high score archive
 *  @param score The high score value
 *  @param sz_nickname The Cy name of the player, who has reached the high score
 *  @param cyid The Cy ID of the device on which the score was reached
 *  @param time Time the high score was reached
 *  @return TRUE if the record was stored successfully.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int record_number;
 *       struct module_t main_module;
 *       struct Score score_archive;
 *       struct score_t* high_scores;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       Score_ctor( &score_archive, main_module.m_process->module );
 *       if( Score_is_valid ( &score_archive ) )
 *       {
 *         record_number = Score_get_record_count( &score_archive );
 *         high_scores = (struct score_t*)malloc( sizeof(struct score_t)*record_number );
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_read( &score_archive, index, high_scores + index );
 *         }
 *         //  Changes high_scores.
 *         ...
 *         for( index = 0; index < record_number; index ++ )
 *         {
 *           Score_write_Ex( &score_archive,
 *                           index,
 *                           high_scores[index].score,
 *                           high_scores[index].nickname,
 *                           high_scores[index].cyid,
 *                           high_scores[index].time );
 *         }
 *         ...
 *         free(high_scores);
 *       }
 *       ...
 *       Score_dtor( &score_archive, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Score_write, score_t.
 */
#ifndef _SKIP_DOC
__cyos "gamelib.dl" 2
#endif
bool Score_write_Ex(struct Score* ptr_score, int index, long score, char* sz_nickname, cyid_t cyid, time_t time);

#ifdef _SKIP_DOC
};
#endif

#endif //  #ifndef __SCORE_H_INCLUDED__


