/**
 *  @file time.h
 *  Time manipulations, Real Time Clock driver.
 */

#ifndef __TIME_H_INCLUDED__
#define __TIME_H_INCLUDED__

/**
 *  @brief A time manipulation structure.
 *  @ingroup time_overview
 *  A time manipulation structure.
 *  @see time_overview
 */
struct Time
{
  /**
   *  Year - 1900, 99 means 1999.
   */
  char year;

  /**
   *  Month, 1..12.
   */
  char month;

  /**
   *  Day of month, 1..31.
   */
  char day;

  /**
   *  Hour, 0..23.
   */
  char hour;

  /**
   *  Minute, 0..59.
   */
  char minute;

  /**
   *  Second, 0..59.
   */
  char second;

  /**
   *  Hundredths of a second, 0..99.
   */
  char hundreds;

#ifndef _SKIP_DOC
};
#endif

/**
 *  Decodes 'time' and fill fields.
 *  @param ptr_time A pointer to a Time object
 *  @param time The time_t format.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Time time;
 *       ...
 *       Time_decode( &time, last_key_time() );
 *       TRACE("Last key was pressed at hour: %d, minutes: %d, seconds: %d",
 *             time.hour, 
 *             time.minute, 
 *             time.second );
 *       ...
 *  @endcode
 *  @see Time_encode.
 */
#ifndef _SKIP_DOC
__cyos 489
#endif
void Time_decode(struct Time* ptr_time, time_t time);

/**
 *  Packs fields into the time_t type.
 *  @param ptr_time A pointer to a Time object
 *  @return Packed field in the time_t type
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       time_t packed_time;
 *       struct Time time;
 *       ...
 *       // 99 is 1999, 100 is 2000 and etc.
 *       time.year  = 100;
 *       time.month = 02;
 *       time.day = 07;
 *       time.hour = 22;
 *       time.minutes = 15;
 *       time.second = 15;
 *       time.hundreds = 10;
 *       ...
 *       packed_time = Time_encode( &time );
 *       ...
 *  @endcode
 *  @see Time_decode , time_t.
 */
#ifndef _SKIP_DOC
__cyos 543
#endif
time_t Time_encode(struct Time* ptr_time);

/**
 *  Sets the Real Time Clock (RTC) according to its fields.
 *  @param ptr_time A pointer to a Time object
 *  @return TRUE if the time was set successfully
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Time time;
 *       Time_get_RTC( &time );
 *       time.hour++;
 *       time.hour %= 24;
 *       //  We have changed the clock one hour ahead. 
 *       Time_set_RTC( &time );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1082
#endif
bool Time_set_RTC(struct Time* ptr_time);

/**
 *  Sets the fields according to the values read from the Real Time Clock (RTC).
 *  @param ptr_time A pointer to a Time object
 *  @return TRUE if the field was set successfully
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Time time;
 *       Time_get_RTC( &time );
 *       time.hour++;
 *       time.hour %= 24;
 *       // We have changed the clock one hour ahead.
 *       Time_set_RTC( &time );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 621
#endif
bool Time_get_RTC(struct Time* ptr_time);

/**
 *  Calculates the weekday for the current date (Sunday is 0, Saturday is 6).
 *  @param ptr_time A pointer to a Time object
 *  @return The weekday number
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int week_day;
 *       struct Time time;
 *       ...
 *       Time_get_RTC( &time );
 *       ...
 *       week_day = Time_get_weekday( &time );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 749
#endif
int Time_get_weekday(struct Time* ptr_time);

/**
 *  Resets the alarm flag.
 *  @return 1 if alarm flag was set, 0 if not set, -1 if the flag could not be read
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       // It's the same as using the Time_enable_alarm_int( FALSE ) function.
 *       // Turns off the alarm.
 *       Time_clear_alarm_flag();
 *       ...
 *       // Turns on the alarm.
 *       Time_enable_alarm_int( TRUE );
 *  @endcode
 *  @see Time_enable_alarm_int.
 */
#ifndef _SKIP_DOC
__cyos 1316
#endif
int Time_clear_alarm_flag(void);

/**
 *  Sets or resets the alarm flag.
 *  @param enable If TRUE, the alarm flag will be set, else will be reset
 *  @return The previous state of the alarm flag
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       // It's the same as using the Time_enable_alarm_int( FALSE ) function.
 *       // Turns off the alarm.
 *       Time_clear_alarm_flag();
 *       ...
 *       // Turns on the alarm.
 *       Time_enable_alarm_int( TRUE );
 *  @endcode
 *  @see Time_clear_alarm_flag.
 */
#ifndef _SKIP_DOC
__cyos 1317
#endif
bool Time_enable_alarm_int(bool enable);

/**
 *  @ingroup time_overview
 *  Returns the time from the system startup in milliseconds; updated every 10 mS.
 *  @return a clock_t value.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int value;
 *       ...
 *       // Initializes a pseudorandom sequence.
 *       srand( clock() );
 *       ...
 *       value = random( 10 );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 451
#endif
clock_t clock(void);

/**
 *  @ingroup time_overview
 *  Returns the current time in the system format (seconds since some date).
 *  @return Time in the time_t format (seconds since some date).
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int week_day;
 *       struct Time time;
 *       ...
 *       Time_decode( &time, time() );
 *       ...
 *       week_day = Time_get_weekday( &time );
 *  @endcode
 *  @see Time_encode.
 */
#ifndef _SKIP_DOC
__cyos 1249
#endif
time_t time(void);

/**
 *  @ingroup time_overview
 *  Returns trusted time.
 *  This time is synchronized with the Cybiko web server. 
 *  The user cannot change the trusted time. 
 *  @return The trusted time.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Time time;
 *       ...
 *       Time_decode( &time, get_trusted_time() );
 *       //  Synchronizes real time clock with the trusted time.
 *       Time_set_RTC( &time );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1368
#endif
time_t get_trusted_time(void);

#ifdef _SKIP_DOC
};
#endif


#endif  //  #ifndef __TIME_H_INCLUDED__
