/** 
 *  @file tone_gen.h
 *  Tones generation.
 */

#ifndef __TONE_GEN_H_INCLUDED__
#define __TONE_GEN_H_INCLUDED__

/**
 *  The beep function argument for the OK dialog.
 * @ingroup sound_overview
 */
#define BEEP_OK        0       

/**
 *  The beep function argument for the INFO dialog.
 * @ingroup sound_overview
 */
#define BEEP_INFO      1       

/**
 *  The beep function argument for the QUESTION dialog.
 * @ingroup sound_overview
 */
#define BEEP_QUESTION  2 

/**
 *  The beep function argument for the ERROR dialog.
 * @ingroup sound_overview
 */
#define BEEP_ERROR     3   

/**
 *  @ingroup sound_overview
 *  Returns a sounds-enabled flag state.
 *  @return Sounds-enabled flag state.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if ( !get_sounds_enabled() )
 *             enable_sounds( TRUE );
 *       ...
 *  @endcode
 *  @see enable_sounds.
 */
#ifndef _SKIP_DOC
__cyos 729
#endif
bool get_sounds_enabled(void);

/**
 *  @ingroup sound_overview
 *  Enables or disables tone generation.
 *  When tone generation is disabled, play_tone() has no effect (the device's 
 *  speaker keeps silent). Tone generation is enabled on startup.
 *  @param enable - If TRUE, tone generation is 
 *  enabled, otherwise it is disabled.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if ( !get_sounds_enabled() )
 *             enable_sounds( TRUE );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 539
#endif
void enable_sounds(bool enable);

/**
 *  @ingroup sound_overview
 *  Produces a beep tone.
 *  @param beep_type Is one of the following: BEEP_OK, BEEP_INFO, 
 *  BEEP_QUESTION or BEEP_ERROR.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       beep( BEEP_QUESTION );
 *       ...
 *  @endcode
 *  @see BEEP_OK, BEEP_INFO, BEEP_QUESTION, BEEP_ERROR.
 */
#ifndef _SKIP_DOC
__cyos 380
#endif
void beep(int beep_type);

/**
 *  @ingroup sound_overview
 *  Allows the device to determine whether or not a tone is being played.
 *  @return TRUE if a tone is being played, FALSE if no tone is being played.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if ( !is_tone_playing() )
 *             beep( BEEP_ERROR );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 823
#endif
bool is_tone_playing(void);

/**
 *  @ingroup sound_overview
 *  Starts tone generation. 
 *  The range of valid values is from 0 (85Hz) to 67 (about 4075Hz)
 *  inclusive. If the index is negative, calling play_tone() is the same as 
 *  calling stop_tone().
 *  @param index Specifies the frequency of the generated sound.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       play_tone(30);
 *       cWinApp_pause(main_module.m_process, 200);
 *       play_tone(-1);
 *       ...
 *  @endcode
 *  @see play_raw.
 */
#ifndef _SKIP_DOC
__cyos 942
#endif
void play_tone(int index);

/**
 *  @ingroup sound_overview
 *  Starts playing a tone. This function works the same way as the play_tone function.
 *  The difference is that, instead of the musical tone index, you provide the divisor 
 *  for the timer so you can get any kind of sound this hardware can generate.
 *  @param divisor Value from the range [6505 - 1356] (or 85Hz - 4075Hz )
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int count = 0;
 *       ...
 *       // Here is a simple example that screams like a crazy cat:
 *       for( index = 0; index < 0xFFF; index++ )
 *       {
 *         count = ( count + 1 ) % 64;
 *         if( count < 32 )
 *             play_raw( count * 10 + 1000 );
 *         else
 *             play_raw( ( 64 - count ) * 10 + 1000 );
 *       }
 *       stop_tone();
 *  @endcode
 *  @see play_tone.
 */
#ifndef _SKIP_DOC
__cyos 941
#endif
void play_raw(int divisor);

/**
 *  @ingroup sound_overview
 *  Stops tone playing (turns the sound off). 
 *  It's safe to call it even if no tone is being played.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int index;
 *       int count = 0;
 *       ...
 *       // Here is a simple example that screams like a crazy cat:
 *       for( index = 0; index < 0xFFF; index++ )
 *       {
 *         count = ( count + 1 ) % 64;
 *         if( count < 32 )
 *             play_raw( count * 10 + 1000 );
 *         else
 *             play_raw( ( 64 - count ) * 10 + 1000 );
 *       }
 *       stop_tone();
 *  @endcode
 *  @see play_tone.
 */
#ifndef _SKIP_DOC
__cyos 1190
#endif
void stop_tone(void);

#endif  //  #ifndef __TONE_GEN_H_INCLUDED__


