/** 
 *  @file utilfunc.h
 *  Auxilary functions of CyWin.
 */

#ifndef __UTILFUNC_H_INCLUDED__
#define __UTILFUNC_H_INCLUDED__

/**
 *  @ingroup drawing_primitives_group
 *   Fills a rectangle with a specified color, to form a color bar.
 *   @param x the x-coordinate of the upper-left corner
 *   @param y the y-coordinate of the upper-left corner
 *   @param width the bar's width
 *   @param height the bar's height
 *   @param color the bar's color
 *   @param xor if TRUE xor operation will be performed
 *   @return None.
 *   @code
 *        #include <cywin.h>
 *        ...
 *        draw_bar( 10, 10, 100, 100, CLR_DKGRAY, FALSE );
 *        ... 
 *   @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 290
#endif
void draw_bar(int x, int y, int width, int height, color_t color, bool xor);

/**
 *  @ingroup drawing_primitives_group
 *   Draws a rectangle with the specified color.
 *   @param x the x-coordinate of the upper-left corner.
 *   @param y the y-coordinate of the upper-left corner.
 *   @param width the rectangle's width.
 *   @param height the rectangle's  height.
 *   @param color the rectangle's color.
 *   @return None.
 *   @code
 *        #include <cywin.h>
 *        ...
 *        rect( 10, 10, 100, 100, CLR_DKGRAY );
 *        ...
 *   @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 291
#endif
void rect(int x, int y, int width, int height, color_t color);

/**
 *  @ingroup fonts_and_text_group
 *   Draws text in the specified color.
 *   @param x the x-coordinate of the upper-left corner.
 *   @param y the y-coordinate of the upper-left corner.
 *   @param ptr_font the pointer to the initialized font structure.
 *   @param sz_text a pointer to the string of a text to be drawn.
 *   @param width the width of a text block (in symbols).
 *   @param color the text color.
 *   @return None.
 *   @code
 *        #include <cywin.h>
 *        ...
 *        text( 10, 10, cool_normal_font, "Any text", 6, CLR_DKGRAY );
 *        ...
 *   @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 292
#endif
void text(int x, int y, struct Font* ptr_font, char* sz_text, int, color_t color);

/**
 *   @ingroup fonts_and_text_group
 *   Draws text with specified colors for the text and the background.
 *   @param x the x-coordinate of the upper-left corner.
 *   @param y the y-coordinate of the upper-left corner.
 *   @param ptr_font a pointer to the initialized font structure.
 *   @param sz_text a pointer to the string of a text to be drawn.
 *   @param width the width of a text block (in symbols).
 *   @param color the text color.
 *   @param bk_color the background color.
 *   @return None.
 *   @code
 *        #include <cywin.h>
 *        ...
 *        textC( 10, 10, cool_normal_font, "Any text", 3, CLR_WHITE, CLR_BLACK );
 *        ...
 *   @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 293
#endif
void textC(int x, int y, struct Font* ptr_font, char* sz_text, int width, color_t color, color_t bk_color);

/**
 *  @ingroup math_group
 *  Returns the minimum of the two values.
 *  @param first_value first value to be compared.  
 *  @param second_value second value to be compared.
 *  @return The minimum of the two values.
 *  @code
 *        #include <cywin.h>
 *        ...
 *        struct rect_t object_1;
 *        struct rect_t object_2;
 *        struct rect_t rect_and;
 *        ...
 *        //  Obtains intersection of two rectangles.
 *        rect_and.x = imax( object_1.x, object_2.x );
 *        rect_and.y = imax( object_1.y, object_2.y );
 *        rect_and.w = imin( object_1.x + object_1.w, 
 *                           object_2.x + object_2.w ) - rect_and.x;
 *        rect_and.h = imin( object_1.y + object_1.h, 
 *                           object_2.y + object_2.h ) - rect_and.y;
 *        ...
 *  @endcode
 *  @see imax.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 294
#endif
int imin(int first_value, int second_value);

/**
 *  @ingroup math_group
 *  Returns the maximum of the two values.
 *  @param first_value first value to be compared.  
 *  @param second_value second value to be compared.
 *  @return The maximum of the two values
 *  @code
 *        #include <cywin.h>
 *        ...
 *        struct rect_t object_1;
 *        struct rect_t object_2;
 *        struct rect_t rect_and;
 *        ...
 *        //  Obtains intersection of two rectangles.
 *        rect_and.x = imax( object_1.x, object_2.x );
 *        rect_and.y = imax( object_1.y, object_2.y );
 *        rect_and.w = imin( object_1.x + object_1.w, 
 *                           object_2.x + object_2.w) - rect_and.x;
 *        rect_and.h = imin( object_1.y + object_1.h, 
 *                           object_2.y + object_2.h) - rect_and.y;
 *        ...
 *  @endcode
 *  @see imin.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 295
#endif
int imax(int first_value ,int second_value);

/**
 *  @ingroup drawing_primitives_group
 *  Returns TRUE if the point is in the rectangle area.
 *  @param x the point's x-coordinate.
 *  @param y the point's y-coordinate.
 *  @param ptr_rectangle the pointer to the rectangle.
 *  @return TRUE if the point is in the rectangle. 
 *  @code
 *        #include <cywin.h>
 *        ...
 *        int x;
 *        int y;
 *        struct rect_t rect_screen;
 *        ...
 *        rect_set( &rect_screen, 0, 0, 160, 100 );
 *        ...
 *        if ( !in_rect( x, y, &rect_screen ) )
 *        {
 *          TRACE( "Warning: Object is out of screen.");
 *        }
 *        ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 296
#endif
bool in_rect(int x, int y, struct rect_t* ptr_rectangle);

/**
 *  @ingroup fonts_and_text_group
 *  Draws bounded text with trailing dots if text is cut.
 *  @param ptr_gfx a pointer to the graphics context.
 *  @param x the x-coordinate of a text block.
 *  @param y the y-coordinate of a text block.
 *  @param width the width of a text block (in symbols).
 *  @param sz_text a pointer to the text to be drawn.
 *  @return The number of characters actually drawn, not including dots.
 *  @code
 *        #include <cywin.h>
 *        ...
 *        struct module_t  main_module;
 *        ...
 *        init_module( &main_module );
 *        ...
 *        draw_bounded_text( main_module.m_gfx, 10, 10, 4, "Any text" );
 *        ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 302
#endif
int draw_bounded_text(struct Graphics* ptr_gfx, int x, int y, int width, char* sz_text);

/**
 *  @ingroup string_overview
 *  Removes spaces from the string.
 *  @param sz_text String to be modified
 *  @return A pointer to the string without spaces
 *  @code
 *        #include <cywin.h>
 *        ...
 *        char test_string[ 20 ];
 *        ...
 *        strcpy( test_string, "Any string" );
 *        //  Outputs: Any string
 *        TRACE( "%s", test_string );
 *        //  Outputs: Anystring
 *        TRACE( "%s", trunc_spaces( test_string ) );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 304
#endif
char* trunc_spaces(char* sz_text);

/**
 *  @ingroup memory_overview
 *  Trace memory dump. 
 *  @param ptr_memory pointer to the memory dump
 *  @param size the size of the memory dump
 *  @param string_length the maximum length of a string on the console
 *  @return None
 *  @code
 *        #include <cywin.h>
 *        ...
 *        char raw_data[ 256 ];
 *        ...
 *        //  Fills raw_data array.
 *        ...
 *        memory_dump( raw_data, 256, 16 );
 *        ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 303
#endif
void memory_dump(void* ptr_memory, int size, int string_length);

#endif  //  #ifndef __UTILFUNC_H_INCLUDED__

