/** 
 *  @file varargs.h
 *  Macros for variable argument functions.
 */

#ifndef __VARARGS_H_INCLUDED__
#define __VARARGS_H_INCLUDED__

/**
 *  @typedef typedef char *va_list;
 *  @ingroup vararg_overview
 *  Storage to hold information needed by the va_start(), va_arg() and va_end() 
 *  macros. The called function declares a variable of type va_list that can be 
 *  passed as an argument to another function. 
 *  @see va_start, va_end, va_arg.
 */
typedef char *va_list;

#ifndef _SKIP_DOC
#define _LONGSIZEOF(n)  ( (sizeof(n) + sizeof(long) - 1) & ( (sizeof(long) - 1)^(-1) ) )
#endif

/**
 *  @ingroup vararg_overview
 *  Initializes a list of optional arguments for variable argument function.
 *  @param ptr_arg A pointer to list of arguments
 *  @param prev_param The parameter that precedes the first optional argument
 *  @return None
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      //  Logs errors to the file.
 *      void trace2logfile(char* sz_format, ...)
 *      {
 *        va_list parameters;
 *        char sz_buffer[64];
 *        struct FileOutput* ptr_log_file; 
 *        //  Formats string.
 *        va_start( parameters, sz_format );
 *        vsprintf( sz_buffer, sz_format, parameters );  
 *        va_end( parameters );
 *        //  Writes it to file "error.log".
 *        ptr_log_file = (struct FileOutput*)malloc( sizeof( struct FileOutput ) );
 *        FileOutput_ctor_Ex( ptr_log_file, "error.log", TRUE );
 *        FileOutput_seekp( ptr_log_file, 0, SEEK_END );
 *        FileOutput_write( ptr_log_file, sz_buffer, strlen(sz_buffer) );
 *        FileOutput_dtor( ptr_log_file, FREE_MEMORY );
 *      }
 *      ...
 *  @endcode
 *  @see va_end, va_arg.
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define va_start(list, prev_param) list = ( (va_list)&prev_param + sizeof(prev_param) )
#else
void va_start(va_list ptr_arg, prev_param);
#endif

/**
 *  @ingroup vararg_overview
 *  Retrieves the current argument from the list of optional arguments 
 *  for variable argument function.
 *  @param ptr_arg A pointer to the list of arguments
 *  @param type Type of the argument to retrieve
 *  @return Current argument from the list of optional arguments 
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      //  Traces strings.
 *      void trace_multistring(int string_number, ...)
 *      {
 *        int index;
 *        va_list parameters;
 *        va_start( parameters, string_number );
 *        for( index = 0; index < string_number; index ++ )
 *        {
 *          TRACE( va_arg( parameters, char* ) );
 *        }
 *        va_end( parameters );
 *      }
 *      ...
 *      trace_multistring( 4, "One", "Two", "Three", "Four");
 *      ...
 *  @endcode
 *  @see va_start, va_end.
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define va_arg(list,type) (*(type*)( ( list += _LONGSIZEOF(type) ) - sizeof(type) ) )
#else
type va_arg(va_list ptr_arg, type);
#endif

/**
 *  @ingroup vararg_overview
 *  Releases a list of optional arguments for variable argument function
 *  @param ptr_arg A pointer to a list of arguments
 *  @return None
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      //  Logs errors to the file.
 *      void trace2logfile(char* sz_format, ...)
 *      {
 *        va_list parameters;
 *        char sz_buffer[64];
 *        struct FileOutput* ptr_log_file; 
 *        //  Formats string.
 *        va_start( parameters, sz_format );
 *        vsprintf( sz_buffer, sz_format, parameters );  
 *        va_end( parameters );
 *        //  Writes it to file "error.log".
 *        ptr_log_file = (struct FileOutput*)malloc( sizeof( struct FileOutput ) );
 *        FileOutput_ctor_Ex( ptr_log_file, "error.log", TRUE );
 *        FileOutput_seekp( ptr_log_file, 0, SEEK_END );
 *        FileOutput_write( ptr_log_file, sz_buffer, strlen(sz_buffer) );
 *        FileOutput_dtor( ptr_log_file, FREE_MEMORY );
 *      }
 *      ...
 *  @endcode
 *  @see va_start, va_arg.
 */
#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define va_end(list) ( list = (va_list)0 )
#else
void va_end(va_list ptr_arg);
#endif

#endif  //  #ifndef __VARARGS_H_INCLUDED__
