/*
================================ COPYRIGHT ================================
The contents of this file are subject to the VTech Informations Ltd. License
of VT-OS Ver. 1.1 operating system (the "License"); you may not use this 
file except in compliance with the License.  

Software distributed under the License is distributed on an "AS IS" basis, 
WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the License 
for the specific language governing rights and limitations under the License.
  
The Original Code is VT-OS Ver. 1.1 Operating System, released 
on October 1st, 1999
	
The Initial Developer of the Original Code is VTech Informations Ltd.  All 
codes are Copyright (C) VTech Informations Ltd. 1999.  All Rights Reserved.
===========================================================================
*/

/*
===========================================================================
File        :   uartmgr.c
Author(s)   :   Kenng Ng
Company     :   VTech Informations Ltd.
Project     :   Helio 
Date:	    :   October 1st, 1999
Purpose:	:   Uart Manager
Revision    :   1.1
Note        :   None
===========================================================================
*/              

#include "datatype.h"
#include "uartmgr.h"
#include "errcode.h"
#include "dm_os.h"
#include "qtype.h"
#include "msg.h"
#include "intc.h"
#include <stddef.h>
#include "print.h"
#include "eventmgr.h"
#include "LcdApi.h"
#include "lcddebug.h"
#include "system.h"

#define GetHandlePort(handle) 	(WORD) (0)
#define GetHandleApp(handle)  	(WORD) (1)

UBYTE		UartMgrBuffer[3072];
int 		uartmgr_mode;
UBYTE 		need_hb;


void		UartRxInit();
BOOLEAN		UartGetBytes(UBYTE	*RxPtr, UWORD Length);
UWORD		(*UartMgrChkUartANumBytes)();
void 		UartAOn();
void 		UartAOff();

extern char 				*ovrbuf;
extern int 					ovrcount;
extern volatile UBYTE		UartNumBytes;


char *strcpy(char *dest, const char *src);
void UartPrints(BYTE *c)
{
	while (*c)
	{
		UartPutc(UART_A,*c);
		c++;
    }
}
/*void UartPrints(char *buf) { UartPrintf(UART_A, buf); }*/
BYTE UartGetc(WORD port)
{   char data;
while (UartRxData(port,  &data)!=UART_SUCCESS);
return (char) data;
}
void UartPutc(WORD port, BYTE data)
{   while (UartTxData(port, data)!=UART_SUCCESS);
}
void UartPrintf(WORD port, BYTE *buffer)
{   WORD i=0;
while (buffer[i])
{
	UartPutc (port,  (BYTE) buffer[i] ) ;
	i++;
}
}


/* send buffer */
struct com_lnk_struct
{
	UWORD size;			/* number of valid data in the buffer */
	UWORD byte_send;			/* no. of byte sent out */
	BYTE **data; 			/* pointer to the buffer */
	struct com_lnk_struct *next;	/* next buffer node */
};
typedef struct com_lnk_struct ComSendLnk;

typedef struct
{
	WORD    handle;         /* the handle ID */
	ComMode mode;           /* mode, buffer or char */
	BYTE *rec_buffer;
	UWORD rec_buffer_size;
	UWORD   rec_buffer_count;   /* number of byte recevied in buffer */
	BYTE status;
	struct com_lnk_struct *send_buffer;        /* if buffer mode, the buffer pointer */
} UartPortCtrl;

UartPortCtrl uart_port_ctrl[UART_PORT_MAX];




WORD  UartRxData(WORD port, BYTE *data)
{
    if (port==UART_A)
    {
        if (~CPU ->REG_UARTA_CTRL_1 & RXHOLDFULL)   /* receive buffer empty */
            return (UART_BUF_EMPTY);
        *data = (char)(CPU ->REG_UARTA_HLDG );
    }
    if (port==UART_B)
    {
        if (~CPU ->REG_UARTB_CTRL_1 & RXHOLDFULL)   /* receive buffer empty */
            return (UART_BUF_EMPTY);
        *data = (char)(CPU ->REG_UARTB_HLDG );
    }
    return (UART_SUCCESS);
}

/*---------------------------------------------------------------------------
Prototype :   WORD UartTxData(WORD port, BYTE data)
Purpose	:	Write data to UART port
Scope 	:	ALL
input     :   WORD  port     UART_A: port A,   UART_B: port B
return	:	 UART_SUCCESS
UART_BUF_FULL
---------------------------------------------------------------------------*/
WORD  UartTxData(WORD port, BYTE data)
{
	if (port==UART_A)
	{
		if (~CPU ->REG_UARTA_CTRL_1 & EMPTY)		/* transmit buffer full */
			return (UART_BUF_FULL);
		
		CPU ->REG_UARTA_HLDG = data;
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
	}
	else /* port == UART_B */
	{
		if (~CPU ->REG_UARTB_CTRL_1 & EMPTY)		/* transmit buffer full */
			return (UART_BUF_FULL);
		CPU ->REG_UARTB_HLDG = data;
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
        __asm(" nop ");
		
	}
	return (UART_SUCCESS);
}


/*********************************************************************
Purpose:  Get the lower 2-byte of a 4-byte data
Scope:  All
Input parameters:  word - the 4-byte data
Output parameters:  None
Return:  The lower 2-byte                                       
Comment:
************************************************************************/
USHORT LParam(UWORD word)
{
	return (USHORT) (word & 0x0000ffff);
}

/*********************************************************************
Purpose: Get the upper 2-byte of a 4-byte data 
Scope: All
Input parameters: word - the 4-byte data                     
Output parameters: None
Return: The upper 2-byte                                        
Comment:
************************************************************************/
USHORT HParam(UWORD word)
{
	return ((USHORT) (word >> 16));
}

/*********************************************************************
Purpose: Set the value of lower 2-byte of a 4-byte data 
Scope: All
Input parameters: word - the 4-byte data
ldata - the 2-byte data
Output parameters: word - the new data
Return: None
Comment:
************************************************************************/
void SetLParam(UWORD *word, USHORT ldata)
{
	(USHORT) word[0] = ldata;
}

/*********************************************************************
Purpose: Set the value of upper 2-byte of a 4-byte data
Scope: All
Input parameters: word - the 4-byte data
hdata - the 2-byte data
Output parameters: word - the new data
Return: None                                        
Comment:
************************************************************************/
void SetHParam(UWORD *word, USHORT hdata)
{
	*word &= 0x0000ffff;
	*word |= (((UWORD) hdata) << 16);
}


void ComSetDataMode(int mode)
{
    uartmgr_mode = mode;
}

int ComGetDataMode()
{
	return uartmgr_mode;
}


UBYTE DecodeData(UBYTE in, UBYTE *out)
{
    if(uartmgr_mode == COM_DATA_BIN)
    {
        *out = in;
        return 0;
    }
	
    if(need_hb == 2)
    {
		if( in == 0)
		{  /* rec: 0x7f -> 0x7f -> 0  => 0x7f*/
			*out = 0x7f;
		}
		else
		{ /* rec: 0x7f -> 0x7f -> 1  => 0xff*/
			*out = 0xff;
		}
		need_hb = 0;
	}
	else
	{
		if( in == (UBYTE)0x7f)
		{
			need_hb++;
		}
		else
		{
			if(need_hb==1)
			{
				*out = in | (UBYTE)0x80;
				need_hb = 0;
			}
			else
			{
				*out = in;
			}
		}
	}
	return need_hb;
}

/*********************************************************************
Purpose: UWORD	UartMgrChkBufBytes()
Scope: EventMgr
Input parameters: None
Output parameters: None
Return: Number of bytes in buffer
Comment: None
************************************************************************/
UWORD	UartMgrChkBufBytes()
{
	return UartNumBytes;	
}

/*********************************************************************
Purpose: UWORD	UartMgrChkBufBytesDummy()
Scope: EventMgr
Input parameters: None
Output parameters: None
Return: 0
Comment: None
************************************************************************/
UWORD	UartMgrChkBufBytesDummy()
{
	return 0;	
}


/*********************************************************************
Purpose:  Open an uart port
Scope:  Application
Input parameters:  port  - port to open, UART_A or UART_B
Output parameters:  port_hdle - the handle to access this connection
Return:  TRUE - Success
ERR_COM_PORT_BUSY - opened by other app
ERR_COM_INV_PORT - invalid port number
Comment:
************************************************************************/
Err ComOpenPort(WORD port, WORD *port_hdle)
{
    AppID app;
    
    if (port > UART_PORT_MAX)
		return ERR_COM_INV_PARAM;  /* no such port */
	if (HParam(uart_port_ctrl[port].handle) != COM_FREE)  /* port busy */
        return ERR_COM_PORT_BUSY;
    app = SysGetActiveAppID();
	
	/* build the handle			    */
	/* format of a handle :
	16          8             8
	|  app id  |  port num  | random number |
	
	  if app id = COM_FREE (0x0000), the port is considered to be free
	*/
	
    uartmgr_mode = COM_DATA_BIN;
	SetHParam(&uart_port_ctrl[port].handle, (USHORT) app);
	SetLParam(&uart_port_ctrl[port].handle, (uart_port_ctrl[port].handle & 0x0000ffff) +1);
	uart_port_ctrl[port].handle &= 0xffff00ff;
	uart_port_ctrl[port].handle |=  ( (port & 0x000000ff) << 8 );
	*port_hdle = uart_port_ctrl[port].handle;
	uart_port_ctrl[port].mode = COM_MODE_CHAR;	/* default char. mode */
	uart_port_ctrl[port].rec_buffer_size = 0;
	uart_port_ctrl[port].rec_buffer = NULL;	        /* set buffer to NULL */
    uart_port_ctrl[port].status = COM_RX_DISABLE;
    UartRxInit();
    UartMgrChkUartANumBytes = UartMgrChkBufBytes;
	return TRUE;
}

/*********************************************************************
Purpose: Check if the handle is valid or not
Scope: Internal
Input parameters: port_hdle - the handle to check
Output parameters: None
Return: TRUE - the handle is valid
False - the handle is invalid
Comment:
************************************************************************/
BOOLEAN ComIsHandleValid(WORD port_hdle)
{
	/* check port range */
	if (GetHandlePort(port_hdle) > UART_PORT_MAX)
		return FALSE;
	
	/* check handle signature */
	if ((uart_port_ctrl[GetHandlePort(port_hdle)].handle != port_hdle))
		return FALSE;
	
	return TRUE;
}

/*********************************************************************
Purpose: Close an uart port
Scope: Application
Input parameters: port_hdle - handle of the port
Output parameters: None
Return: TRUE - Success
ERR_COM_INV_HANDLE - invalid handle
Comment:
************************************************************************/
Err ComClosePort(WORD port_hdle)
{
	if (!ComIsHandleValid(port_hdle))
		return ERR_COM_INV_HANDLE;
    
    UartAOff();
	/* mark the port free */
	SetHParam(&uart_port_ctrl[GetHandlePort(port_hdle)].handle, COM_FREE);
    uart_port_ctrl[GetHandlePort(port_hdle)].status = COM_FREE;
    UartDisable(GetHandlePort(port_hdle));  /* disable the port */
    UartMgrChkUartANumBytes = UartMgrChkBufBytesDummy;
	return TRUE;
}

/*********************************************************************
Purpose: Reset uart ports and initilize Uart Manager
Scope: OS
Input parameters: None                     
Output parameters: None 
Return: None                                        
Comment: All buffer & internal variables will reset
************************************************************************/
void ComReset()
{
	USHORT i;
	ComSendLnk *free_buf;
	ComSendLnk *current_buf;
	
	for(i=0;i<=UART_PORT_MAX;i++)	/* mark each port free */
    {
		uart_port_ctrl[i].status = COM_FREE;
		SetHParam(&uart_port_ctrl[i].handle, COM_FREE);
    }
}

/*********************************************************************
Purpose: Start to capture the uart port data
Scope: Application
Input parameters: port_hdle - handle of the port
mode - port mode, COM_MODE_CHAR or COM_MODE_BUFFER
time_out - time out if in buffer mode
buffer_size - buffer size, if in buffer mode
buffer - the buffer pointer, if in buffer mode
Output parameters: None
Return: TRUE - Success
ERR_COM_INV_HANDLE - invalid handle
ERR_COM_INV_PARAM  - mode is invalid
ERR_COM_INV_BUFFER - buffer = NULL or buffer_size = 0, if in buffer mode
Comment: uart data will not capture with ComOpenPort(), call
ComCapture to start capture uart data
************************************************************************/
Err ComCapture(WORD port_hdle, ComMode mode, UWORD time_out, UWORD buffer_size, BYTE *buffer)
{
    if (!ComIsHandleValid(port_hdle))
		return ERR_COM_INV_HANDLE;
    
	if (uart_port_ctrl[0].status == COM_RX_ENABLE)          
		return FALSE;
    
    UartAOn();
	
    need_hb = 0;
	
	switch(mode)
	{
	case COM_MODE_BUFFER:
		if ((buffer == NULL) || (buffer_size == 0))
			return ERR_COM_INV_BUFFER;			            	            	
		uart_port_ctrl[GetHandlePort(port_hdle)].rec_buffer 		= buffer;
		uart_port_ctrl[GetHandlePort(port_hdle)].rec_buffer_size 	= buffer_size;
		uart_port_ctrl[GetHandlePort(port_hdle)].rec_buffer_count 	= 0;
		uart_port_ctrl[GetHandlePort(port_hdle)].mode 				= COM_MODE_BUFFER;
		break;
	case COM_MODE_CHAR:
		uart_port_ctrl[GetHandlePort(port_hdle)].rec_buffer 		= NULL;
		uart_port_ctrl[GetHandlePort(port_hdle)].rec_buffer_size 	= 1;
		uart_port_ctrl[GetHandlePort(port_hdle)].rec_buffer_count 	= 0;
		uart_port_ctrl[GetHandlePort(port_hdle)].mode 				= COM_MODE_CHAR;
		break;
	default:
		return ERR_COM_INV_PARAM;
	}
    uart_port_ctrl[GetHandlePort(port_hdle)].status = COM_RX_ENABLE;
	UartEnable(GetHandlePort(port_hdle));  
	return TRUE;	
}

BOOLEAN		UartMgrCaptureData()
{
	UBYTE		RxByte;	
	EvtType		event;
	BOOLEAN		RetVal = FALSE;
	
    if(uart_port_ctrl[0].status != COM_RX_ENABLE)      
		return FALSE;
	
    if (uart_port_ctrl[0].mode == COM_MODE_CHAR)
	{	
		if (UartGetBytes(&RxByte, uart_port_ctrl[0].rec_buffer_size))
		{
			event.eventType = EVT_UART_GET_CHAR;
			event.eventID = 0;
			event.para1 = (WORD) RxByte;
			event.para2 = 0;
			event.evtPBP = NULL;
			EvtAppend(event, &evtQueue);
			RetVal = TRUE;
           	uart_port_ctrl[GetHandlePort(port_hdle)].status = COM_RX_DISABLE;
		}				
	}
	else
	{				
		if (UartGetBytes((UBYTE*)(uart_port_ctrl[0].rec_buffer), uart_port_ctrl[0].rec_buffer_size))
		{
			uart_port_ctrl[0].rec_buffer_size = 0;
			uart_port_ctrl[0].rec_buffer_count = 0;			
			event.eventType = EVT_UART_GET_BUFFER;
			event.eventID = 0;
			event.para1 = 0;
			event.para2 = 0;
			event.evtPBP = uart_port_ctrl[0].rec_buffer;
			EvtAppend(event, &evtQueue);
			RetVal = TRUE;					
			uart_port_ctrl[GetHandlePort(port_hdle)].status = COM_RX_DISABLE;						
		}
	}
	return RetVal;
}		

/*********************************************************************
Purpose: Get the port mode
Scope: Application
Input parameters: port_hdle - handle of the port
Output parameters: mode - port mode
Return: TRUE - Success
ERR_COM_INV_HANDLE - invalid handle
Comment:
************************************************************************/
Err ComGetMode(WORD port_hdle, ComMode *mode)
{
	if(!ComIsHandleValid(port_hdle))	/* invalid handle */
		return ERR_COM_INV_HANDLE;
	
	*mode = uart_port_ctrl[GetHandlePort(port_hdle)].mode;
	return TRUE;
}

/*********************************************************************
Purpose: set the baud, parity, stop bit of the port
Scope: Application
Input parameters: port_hdle - handle of the port
baud - the baud rate to set
e.g. UART_BAUD115200
config - Parity | Databit | Stop bit
e.g. UART_PARITY_NONE | UART_DATABIT8 | UART_STOPBIT1
Output parameters: None 
Return: TRUE - Success
ERR_COM_INV_HANDLE - invalid handle
ERR_COM_INV_BUAD_RATE - invalid baud rate
Comment: call this after ComOpenPort()
************************************************************************/
Err ComConfigPort(WORD port_hdle, WORD baud, WORD config)
{
	if(!ComIsHandleValid(port_hdle))
		return ERR_COM_INV_HANDLE;
	
	/* chk baud  */
	switch (baud)
	{
	case UART_BAUD115200:
	case UART_BAUD57600 :
	case UART_BAUD38400 :
	case UART_BAUD19200 :
	case UART_BAUD9600  :
	case UART_BAUD4800  :
	case UART_BAUD2400  :
	case UART_BAUD1200  :
	case UART_BAUD600   :
	case UART_BAUD300   :
		break;
	default:
		return ERR_COM_INV_PARAM;
	}
	
	UartInit(GetHandlePort(port_hdle), baud, config);
	
	return TRUE;
}

/*********************************************************************
Purpose: get uart port configuration 
Scope: Application 
Input parameters: port_hdle - handle of the port
Output parameters: baud - baud rate, e.g. UART_BAUD38400
config - Parity | Databit | Stop bit
Return: TRUE - Success
ERR_COM_INV_HANDLE - invalid handle
Comment:
************************************************************************/
Err ComGetConfig(WORD port_hdle, WORD *baud, WORD *config)
{
	if(!ComIsHandleValid(port_hdle))
		return ERR_COM_INV_HANDLE;
	
	*config = UartChkCfg(GetHandlePort(port_hdle));
	return TRUE;
}

/*********************************************************************
Purpose: Send data
Scope: Application 
Input parameters: port_hdle - handle of the port
buffer - data to send
buffer_size - size of buffer
Output parameters: None 
Return: TRUE - Success
ERR_COM_INV_HANDLE - invalid handle
ERR_MEM_NO_MEM   - not enough memory
Comment:
************************************************************************/
Err ComSendData(WORD port_hdle, BYTE *buffer, UWORD buffer_size)
{
	UWORD	Count;  
	
	while (UartChkTxDmaBusy(UART_A) || !UartAChkEmpty());
	
    for (Count = 0; Count < buffer_size; Count++)  
		UartMgrBuffer[Count] = buffer[Count];
	
    UartADisableDmaInt();
    UartADmaAddr((WORD)(UartMgrBuffer) & 0x1FFFFFFF);   /*must be physical address*/
    UartASetDmaLen(buffer_size - 1);
    UartAEnableDmaInt();
    UartAEnableTxDma();
	return TRUE;
}

/*********************************************************************
Purpose: Send a 8-bit data
Scope: Application 
Input parameters: port_hdle - handle of the port
data - data to send
Output parameters: None 
Return: TRUE - Success
ERR_COM_INV_HANDLE - invalid handle
ERR_MEM_NO_MEM   - not enough memory
Comment:
************************************************************************/
Err ComSendChar(WORD port_hdle, BYTE data)
{
	BYTE 	*buf;
	Err 	result;
	UBYTE	Buffer[100];
	
	/* port_hdle is checked in ComSendData, so, no need to chk here */
	if(!ComIsHandleValid(port_hdle))
		return ERR_COM_INV_HANDLE;
	
    while (UartChkTxDmaBusy(UART_A) || !UartAChkEmpty());
    UartWriteBuf(GetHandlePort(port), (BYTE*)&data, 1);
	
	return result;
}



/*********************************************************************
Purpose: Init. UartMgr
Scope: OS
Input parameters: None
Output parameters: None
Return: None
Comment: Call when system reboot
************************************************************************/
void UartMgrInit()
{
    BYTE *x;
    
	uart_port_ctrl[0].send_buffer = NULL;
	uart_port_ctrl[1].send_buffer = NULL;
    uart_port_ctrl[0].rec_buffer = NULL;
    uart_port_ctrl[1].rec_buffer = NULL;
    ComReset();
    
    UartMgrChkUartANumBytes = UartMgrChkBufBytesDummy;
    EvtRemoveEventHandler(UartMgrCaptureData);
    EvtAddEventHandler(UartMgrCaptureData);    
    
    UartRxInit();
    
    x=pmalloc(10*4092);
    pfree(x);
}

/*********************************************************************
Purpose: Get uart data from DD 
Scope: Obsoleted
Input parameters: Obsoleted      
Output parameters: Obsoleted
Return: Obsoleted
Comment: Obsoleted
************************************************************************/
void ComGetUartBuffer(WORD port)
{
	return;
}


/*********************************************************************
Purpose: Send data through DD
Scope: Obsoleted
Input parameters: Obsoleted
Output parameters: Obsoleted
Return: Obsoleted
Comment: Obsoleted
************************************************************************/
void ComWriteOutData(WORD port)
{
	return;
}

/*********************************************************************
Purpose: DD interface
Scope: Obsoleted
Input parameters: Obsoleted
Output parameters: Obsoleted
Return: Obsoleted
Comment: Obsoleted
************************************************************************/
void UartMgr(MsgType *msg)
{
	return;
}

