/*
 * @(#)device.c
 *
 * Copyright 1999-2001, Aaron Ardiri (mailto:aaron@ardiri.com)
 * All rights reserved.
 *
 * The  source code outlines a number of basic Helio Computing Programming
 * principles and you  should be able to take the core structure and write 
 * a large complex program. It is distributed WITHOUT ANY WARRANTY; use it
 * "AS IS" and at your own risk.
 *
 * The code presented is Copyright 2000 by Aaron Ardiri. It should be used
 * for educational purposes only.  You  shall not modify the Cube3D source 
 * code in any way and re-distribute it as your own, however you  are free 
 * to  use the  code as a  guide  for  developing  programs on  the  Helio 
 * Computing Platform.
 */

#include "helio.h"

// global variable structure
typedef struct
{
  BYTE colorMode;
} DeviceGlobals;

// globals reference
static DeviceGlobals *globals;

// interface
static void DeviceGetEventInt();

/**
 * Initialize the device.
 */
void 
DeviceInitialize()
{
  // create a globals object
  globals = (DeviceGlobals *)pmalloc(sizeof(DeviceGlobals));

  // ok, if running 1.1.08 or better.. go to grayscale mode
  {
    BYTE osVersion[16];

    DeviceGetOSVersionNumber(osVersion);
    if (strncmp(osVersion, "1.1.08", 6) >= 0) {
      LcdGetColorMode(globals->colorMode);
      LcdSetColorMode(GREYSCALE_MODE);
    }
  }
}

/**
 * Get an event from the event queue.
 * 
 * @param event the event generated.
 * @param timeout the timeout period (milliseconds), timeout = NULL_EVENT.
 */
void
DeviceGetEvent(EvtType *event, SHORT timeout)
{
  // normal event handling (for helio - wait forever, no timeout)
  if (timeout == evtWaitForever)  
    EvtGetEvent(event);

  // ok, timeout time :)) lets work baby
  else {

    UWORD tmrTimeout = NULL;

    // trigger a timer interrupt after "timeout" millisecs
    tmrTimeout = TmrIntEnable(timeout, DeviceGetEventInt);

    // get the event
    EvtGetEvent(event);

    // clean up timer code
    if (tmrTimeout) {
      TmrIntDisable(tmrTimeout);
      tmrTimeout = NULL;
    }
  }
}

/**
 * Get the OS version number for the device.
 *
 * @param osVersion a string to store the version string into.
 */
void
DeviceGetOSVersionNumber(BYTE *osVersion)
{
  BYTE *tmpString;

  // allocate memory for the version
  tmpString = (BYTE *)pmalloc(16 * sizeof(BYTE));

  // 1.1.08 and greater call
  SysGetOSVersionNo(tmpString);

  // if we dont get a nice response, lets try the 1.1.03 way :)
  if ((tmpString[0] < '0') || (tmpString[0] > '9') || (tmpString[1] != '.'))
    SysGetOSVersionNo(&tmpString);

  // transfer the "version" string
  strcpy(osVersion, tmpString);

  // clean up
  pfree(tmpString);
}

/**
 * Reset the device to its original state.
 */
void 
DeviceTerminate()
{
  // ok, if running 1.1.08 or better.. restore the gfx mode
  {
    BYTE osVersion[16];

    DeviceGetOSVersionNumber(osVersion);
    if (strncmp(osVersion, "1.1.08", 6) >= 0) {
      LcdSetColorMode(globals->colorMode);
    }
  }

  // clean up memory
  pfree(globals);
}

/**
 * Callback Interrupt for DeviceGetEvent() - to add NULL_EVENT to the queue.
 */
static void
DeviceGetEventInt()
{
  EvtAppendEvt(NULL_EVENT,0,0,0,NULL);
}
