/*
*	SmallBASIC RTL - FILESYSTEM, FILE and DEVICE I/O
*
*	2000-05-27, Nicholas Christopoulos
*
*	This program is distributed under the terms of the GPL v2.0 or later
*	Download the GNU Public License (GPL) from www.gnu.org
*/

#include "sys.h"
#include "kw.h"
#include "var.h"
#include "pproc.h"
#include "device.h"
#include "blib.h"

struct file_encoded_var	{
	byte	sign;		// always '$'
	byte	version;	//
	byte	type;
	dword	size;
	};

/*
*	OPEN "file" [FOR {INPUT|OUTPUT|APPEND}] AS #fileN
*/
void	cmd_fopen()
{
	var_t	file_name;
	int		flags = 0;
	int		handle;

	// filename
	par_getstr(&file_name);		if ( prog_error )	return;

	// mode
	if	( code_peek() == kwFORSEP )	{
		code_skipnext();
		switch ( code_peek() )	{
		case	kwINPUTSEP:
			flags = DEV_FILE_INPUT;
			break;
		case	kwOUTPUTSEP:
			flags = DEV_FILE_OUTPUT;
			break;
		case	kwAPPENDSEP:
			flags = DEV_FILE_APPEND;
			break;
		default:
			err_syntax();
			v_free(&file_name);
			return;
			}

		code_skipnext();
		}
	else
		flags = 0;	// ????

	// file handle
	if	( code_peek() == kwAS )	{
		code_skipnext();

		par_getsharp();
		if	( !prog_error )	{
			handle = par_getint();
			if	( !prog_error )	{
				if	( dev_fstatus(handle) == 0 )	
					dev_fopen(handle, file_name.v.p.ptr, flags);
				else
					rt_raise("OPEN: FILE IS ALREADY OPENED");
				}
			}
		}
	else
		err_syntax();

	v_free(&file_name);
}

/*
*	CLOSE #fileN
*/
void	cmd_fclose()
{
	int		handle;

	// file handle
	par_getsharp();
	if	( !prog_error )	{
		handle = par_getint();
		if	( !prog_error )	{
			if	( dev_fstatus(handle) )
				dev_fclose(handle);
			else
				rt_raise("CLOSE: FILE IS NOT OPENED");
			}
		}
}

/*
*	SEEK #fileN, pos
*/
void	cmd_fseek()
{
	int		handle;
	dword	pos;

	// file handle
	par_getsharp();
	if	( !prog_error )	{
		handle = par_getint();
		if	( !prog_error )	{
			if	( dev_fstatus(handle) )	{
				par_getsep();
				if	( !prog_error )	{
					pos = par_getint();
					if	( !prog_error )	{
						dev_fseek(handle, pos);
						}
					}
				}
			else
				rt_raise("SEEK: FILE IS NOT OPENED");
			}
		}
}

/*
*	PRINT #fileN; var1 [, varN]
*/
void	cmd_fprint()
{
	cmd_print(PV_FILE);
}

/*
*/
void	write_encoded_var(int handle, var_t *var) SEC(BIO);
void	write_encoded_var(int handle, var_t *var)
{
	struct file_encoded_var	fv;
	var_t	*elem;
	int		i;

	fv.sign = '$';
	fv.version = 1;
	fv.type = var->type;
	switch ( var->type )	{
	case	V_INT:
	   	fv.size = 4;
		dev_fwrite(handle, (byte *) &fv, sizeof(struct file_encoded_var));
		dev_fwrite(handle, (byte *) &var->v.i, fv.size);
		break;	
	case	V_NUM:
	   	fv.size = 8;
		dev_fwrite(handle, (byte *) &fv, sizeof(struct file_encoded_var));
		dev_fwrite(handle, (byte *) &var->v.n, fv.size);
		break;	
	case	V_STR:
	   	fv.size = strlen(var->v.p.ptr);
		dev_fwrite(handle, (byte *) &fv, sizeof(struct file_encoded_var));
		dev_fwrite(handle, (byte *) var->v.p.ptr, fv.size);
		break;	
	case	V_ARRAY:
	   	fv.size = var->v.a.size;
		dev_fwrite(handle, (byte *) &fv, sizeof(struct file_encoded_var));

		// write additional data about array
		dev_fwrite(handle, &var->v.a.maxdim, 1);
		for ( i = 0; i < var->v.a.maxdim; i ++ )	{
			dev_fwrite(handle, (byte *) &var->v.a.lbound[i], sizeof(int));
			dev_fwrite(handle, (byte *) &var->v.a.ubound[i], sizeof(int));
			}

		// write elements
		for ( i = 0; i < var->v.a.size; i ++ )	{
			elem = (var_t *) (var->v.a.ptr + sizeof(var_t) * i);
			write_encoded_var(handle, elem);
			}
		break;
		};
}

/*
*/
int		read_encoded_var(int handle, var_t *var) SEC(BIO);
int		read_encoded_var(int handle, var_t *var)
{
	struct file_encoded_var	fv;
	var_t	*elem;
	int		i;
	
	dev_fread(handle, (byte *) &fv, sizeof(struct file_encoded_var));
	if	( fv.sign != '$' )	{
		rt_raise("READ: BAD SIGNATURE");
		return -1;			// bad signature
		}

	v_free(var);
	switch ( fv.type )	{
	case	V_INT:
		var->type = V_INT;
		dev_fread(handle, (byte *) &var->v.i, fv.size);
		break;	
	case	V_NUM:
		var->type = V_NUM;
		dev_fread(handle, (byte *) &var->v.n, fv.size);
		break;	
	case	V_STR:
		var->type = V_STR;
		var->v.p.ptr = tmp_alloc(fv.size+1);
		dev_fread(handle, (byte *) var->v.p.ptr, fv.size);
		var->v.p.ptr[fv.size] = '\0';
		break;	
	case	V_ARRAY:
		var->type = V_ARRAY;
		var->v.a.ptr = tmp_alloc(fv.size * sizeof(var_t));
	   	var->v.a.size = fv.size;

		// read additional data about array
		dev_fread(handle, (byte *) &var->v.a.maxdim, 1);
		for ( i = 0; i < var->v.a.maxdim; i ++ )	{
			dev_fread(handle, (byte *) &var->v.a.lbound[i], sizeof(int));
			dev_fread(handle, (byte *) &var->v.a.ubound[i], sizeof(int));
			}

		// write elements
		for ( i = 0; i < var->v.a.size; i ++ )	{
			elem = (var_t *) (var->v.a.ptr + sizeof(var_t) * i);
			v_init(elem);
			read_encoded_var(handle, elem);
			}
		break;
	default:
		return -2;	// unknown data-type
		};

	return 0;
}

/*
*	WRITE #fileN; var1 [, varN]
*/
void	cmd_fwrite()
{
	int		handle;

	// file handle
	par_getsharp();
	if	( !prog_error )	{
		handle = par_getint();
		if	( !prog_error )	{

			if	( code_peek() == kwTYPE_EOC || code_peek() == kwTYPE_LINE )	{	// There are no parameters
				if	( !dev_fstatus(handle) )	
//					dev_fwrite(handle, "\n", 1);
					;
				else
					rt_raise("FIO: FILE IS NOT OPENED");
				return;
				}

			par_getsep();			// allow commas

			if	( !prog_error )	{

				if	( dev_fstatus(handle) )	{

					byte	code, exitf = 0;
					var_t	*var_p;

					do	{
						code = code_peek();
						switch ( code )	{
						case	kwTYPE_LINE:
						case	kwTYPE_EOC:
							exitf = 1;
							break;
						case	kwTYPE_SEP:
							code_skipsep();
							break;
						case	kwTYPE_VAR:
							var_p = par_getvar_ptr();
							if	( !prog_error )	
								write_encoded_var(handle, var_p);
							break;
						default:
							rt_raise("WRITE: ONLY VARIABLES ARE ALLOWED");
							};

						if	( prog_error )
							return;
						} while ( !exitf );
					}
				else
					rt_raise("FIO: FILE IS NOT OPENED");
				}
			}
		}
}

/*
*	READ #fileN; var1 [, var2 [, ...]]
*/
void	cmd_fread()
{
	int		handle;
	var_t	*var_p;
	byte	code;

	// file handle
	par_getsharp();
	if	( !prog_error )	{
		handle = par_getint();
		if	( prog_error )	return;

		par_getsep();			// allow commas
		if	( prog_error )	return;

		if	( dev_fstatus(handle) )	{

			// get the variables
			do	{
				if	( prog_error )	return;

				// get variable's ptr
				var_p = par_getvar_ptr();
				if	( prog_error )	return;

				read_encoded_var(handle, var_p);
				if	( prog_error )	return;

				// next
				code = code_peek();
				if	( code == kwTYPE_SEP )	
					par_getsep();			// allow commas
				else
					break;
				} while ( 1 );
			}
		else
			rt_raise("FIO: FILE IS NOT OPENED");
		}
}

/*
*	LINEINPUT [#fileN;] var$
*/
void	cmd_flineinput()
{
	int		handle, size, index;
	var_t	*var_p;
	byte	code, ch;

	if	( code_peek() == kwTYPE_SEP )	{
		//
		//	FILE OR DEVICE
		//

		// file handle
		par_getsharp();
		if	( !prog_error )	{

			handle = par_getint();
			if	( !prog_error )	{
				// par_getsemicolon();
				par_getsep();			// allow commas

				if	( !prog_error )	{

					if	( dev_fstatus(handle) )	{

						// get the variable
						code = code_peek();
						if	( code != kwTYPE_VAR )	{
							err_syntax();
							return;
							}
						var_p = code_getvarptr();
		
						if	( !prog_error )	{
							v_free(var_p);
							size = 256;
							index = 0;
							var_p->type = V_STR;
							var_p->v.p.ptr = tmp_alloc(size);

							// READ IT
							while ( !dev_feof(handle) )	{
								dev_fread(handle, &ch, 1);
								if	( prog_error )	{
									v_free(var_p);
									var_p->type = V_INT;
									var_p->v.i = -1;
									return;
									}
								else if	( ch == '\n' )
									break;
								else if	( ch != '\r' )	{
									// store char
									if	( index == (size-1) )	{
										size += 256;
										var_p->v.p.ptr = tmp_realloc(var_p->v.p.ptr, size);
										}
									var_p->v.p.ptr[index] = ch;
									index ++;
									}
								}
					
							//
							var_p->v.p.ptr[index] = '\0';
							var_p->v.p.size = index+1;

							} // read
						else
							rt_raise("FIO: FILE IS NOT OPENED");
						}
					}
				}
			}
		}
	else	{
		//
		//	CONSOLE
		//
		var_p = par_getvar_ptr();
		if	( !prog_error )	{
			v_free(var_p);
			var_p->type = V_STR;
			var_p->v.p.ptr  = tmp_alloc(SB_TEXTLINE_SIZE+1);
			var_p->v.p.size = SB_TEXTLINE_SIZE+1;
			dev_gets(var_p->v.p.ptr, SB_TEXTLINE_SIZE);
			dev_print("\n");
			}
		}
}

/*
*	INPUT #fileN; var$ [, var2 [, ...]]
*/
void	cmd_finput()
{
	cmd_input(PV_FILE);
}

/*
*	KILL filename
*/
void	cmd_fkill()
{
	var_t	file_name;

	// filename
	v_init(&file_name);
	par_getstr(&file_name);		if ( prog_error )	return;
	if	( dev_fexists(file_name.v.p.ptr) )	
		dev_fremove(file_name.v.p.ptr);
//	else
//		rt_raise("KILL: FILE DOES NOT EXIST");
	v_free(&file_name);
}

/*
*	COPY/RENAME filem newfile
*/
void	cmd_filecp(int mv)
{
	var_t	src, dst;

	// filename
	v_init(&src);
	v_init(&dst);
	par_getstr(&src);		if ( prog_error )	return;
	par_getcomma();			if ( prog_error )	{ v_free(&src); return; }
	par_getstr(&dst);		if ( prog_error )	{ v_free(&src); return; }

	if	( dev_fexists(src.v.p.ptr) )	{
		if	( !mv )
			dev_fcopy(src.v.p.ptr, dst.v.p.ptr);
		else
			dev_frename(src.v.p.ptr, dst.v.p.ptr);
		}
	else
		rt_raise("COPY/RENAME: FILE DOES NOT EXIST");

	v_free(&src);
	v_free(&dst);
}

/*
*/
void	cmd_chdir()
{
	var_t	dir;

	// filename
	v_init(&dir);
	par_getstr(&dir);
	if ( prog_error )	return;
	dev_chdir(dir.v.p.ptr);
	v_free(&dir);
}

/*
*/
void	cmd_rmdir()
{
	var_t	dir;

	// filename
	v_init(&dir);
	par_getstr(&dir);
	if ( prog_error )	return;
	dev_rmdir(dir.v.p.ptr);
	v_free(&dir);
}

/*
*/
void	cmd_mkdir()
{
	var_t	dir;

	// filename
	v_init(&dir);
	par_getstr(&dir);
	if ( prog_error )	return;
	dev_mkdir(dir.v.p.ptr);
	v_free(&dir);
}

/*
*	LOADLN filename, array
*/
#if defined(OS_LIMITED)
#define	LDLN_INC	16
#else
#define	LDLN_INC	256
#endif

void	cmd_floadln()
{
	var_t	file_name, *array_p = NULL, *var_p = NULL;
	int		flags = DEV_FILE_INPUT;
	int		handle, index, bcount, size, array_size;
	byte	ch, type = 0;

	// filename
	par_getstr(&file_name);		if ( prog_error )	return;
	par_getcomma();				if ( prog_error )	return;
	array_p = var_p = code_getvarptr();	if ( prog_error )	return;
	if	( code_peek() == kwTYPE_SEP )	{
		par_getcomma();				if ( prog_error )	return;
		type = par_getint();
		}

	handle = dev_freefilehandle();
	if	( !prog_error )	{
		if	( dev_fstatus(handle) == 0 )	{
			dev_fopen(handle, file_name.v.p.ptr, flags);
			if	( !prog_error )	{

				if	( type == 0 )	{
					//
					//	build array
					//
					array_size = LDLN_INC;
					v_toarray1(array_p, array_size);	// v_free() is here
					index = 0;

					while ( !dev_feof(handle) )	{
						// build var for line
						var_p = (var_t *) (array_p->v.a.ptr + (sizeof(var_t) * index));
						#if defined(OS_LIMITED)
						size = 128;
						#else
						size = 1024;
						#endif
						var_p->type = V_STR;
						var_p->v.p.ptr  = tmp_alloc(size);
						index ++;

						// READ LINE
						bcount = 0;
						while ( !dev_feof(handle) )	{
							dev_fread(handle, &ch, 1);
							if	( prog_error )	
								break;
							else if	( ch == '\n' )
								break;
							else if	( ch != '\r' )	{
								// store char
								if	( bcount >= (size-1) )	{
									#if defined(OS_LIMITED)
									size += 128;
									#else
									size += 1024;
									#endif
									var_p->v.p.ptr = tmp_realloc(var_p->v.p.ptr, size);
									}
								var_p->v.p.ptr[bcount] = ch;
								bcount ++;
								}
							} // read line
					
						if	( prog_error )	{
							// clear & exit
							v_free(array_p);
							v_init(array_p);
							break;
							}

						// store text-line
						var_p->v.p.ptr[bcount] = '\0';
						var_p->v.p.size = bcount+1;
						var_p->v.p.ptr  = tmp_realloc(var_p->v.p.ptr, var_p->v.p.size);

						// resize array
						if	( index >= (array_size-1) )	{
							array_size += LDLN_INC;
							v_resize_array(array_p, array_size);
							}
						} // read file


					if	( index )	
						v_resize_array(array_p, index);
					else
						v_resize_array(array_p, 0);	// v_free() is here
					}
				else	{	// if type=1
					//
					//	build string
					//
					v_free(var_p);
					var_p->type = V_STR;
					var_p->v.p.size = dev_flength(handle)+1;
					var_p->v.p.ptr  = tmp_alloc(var_p->v.p.size);
					if	( var_p->v.p.size > 1 )	
						dev_fread(handle, var_p->v.p.ptr, var_p->v.p.size-1);
					var_p->v.p.ptr[var_p->v.p.size-1] = '\0';
					}

				//
				dev_fclose(handle);
				}
			}
		else
			rt_raise("LOADLN: I/O ERROR");
		}

	v_free(&file_name);
}

/*
*	SAVELN filename, array
*/
void	cmd_fsaveln()
{
	var_t	file_name, *array_p = NULL, *var_p = NULL;
	int		flags = DEV_FILE_OUTPUT;
	int		handle, i;

	// filename
	par_getstr(&file_name);		if ( prog_error )	return;
	par_getcomma();				if ( prog_error )	return;
	var_p = code_getvarptr();	if ( prog_error )	return;

	if	( var_p->type == V_ARRAY )
		array_p = var_p;

	handle = dev_freefilehandle();
	if	( !prog_error )	{
		if	( dev_fstatus(handle) == 0 )	{
			dev_fopen(handle, file_name.v.p.ptr, flags);

			if	( !prog_error )	{
				// write
				if	( var_p->type == V_ARRAY )	{
					// parameter is an array
					for ( i = 0; i < array_p->v.a.size; i ++ )	{
						var_p = (var_t *) (array_p->v.a.ptr + (sizeof(var_t) * i));
						fprint_var(handle, var_p);
						dev_fwrite(handle, "\n", 1);
						}
					}
				else	{
					// parameter is an string
					fprint_var(handle, var_p);
//					dev_fwrite(handle, "\n", 1);
					}

				dev_fclose(handle);
				}
			}
		else
			rt_raise("SAVE: I/O ERROR");
		}

	v_free(&file_name);
}

//
//	LOCK #1, [record]|[start TO end]
//
void	cmd_flock()
{
	var_t	str;

	par_getstr(&str);
	if	( prog_error )	return;
	v_free(&str);
}

//
//	CHMOD file, mode
//
void	cmd_chmod()
{
	var_t	str;
	dword	mode;

	par_getstr(&str);	if	( prog_error )	return;
	par_getcomma();		if	( prog_error )	{ v_free(&str); return; }
	mode=par_getint();	if	( prog_error )	{ v_free(&str); return; }

	#if !defined(_PalmOS) && !defined(_VTOS)
	chmod(str.v.p.ptr, mode);
	#endif
	v_free(&str);
}
