/**
*	SmallBasic byte-code executor
*
*	Nicholas Christopoulos
*
*	This program is distributed under the terms of the GPL v2.0 or later
*	Download the GNU Public License (GPL) from www.gnu.org
*/

#include "sys.h"
#include "panic.h"
#include "blib.h"
#include "str.h"
#include "fmt.h"
#include "extlib.h"

#define BRUN_MODULE

#include "kw.h"
#include "var.h"
#include "scan.h"
#include "smbas.h"

#include "device.h"
#include "pproc.h"

#if defined(_WinBCB)
#include <dir.h>
#endif

// command-line (compiler/executor) options
byte	opt_graphics = 1;		// command-line option: start in graphics mode
byte	opt_quite = 0;			// command-line option: quite
int		opt_retval = 0;			// return-value 
byte	opt_decomp = 0; 		// decompile
byte	opt_safedraw = 0;		// use safe routines for drawing
byte	opt_loadmod = 0;		// load all modules
#if defined(OS_LIMITED)
char	opt_command[128];		// command-line parameters
char	opt_modlist[128];
#else
char	opt_command[1024];
char	opt_modlist[1024];
#endif

//#define _CHECK_STACK

int		prog_line;				// The current source line
int		prog_error;				// The last RTL error code (its work as flag)
extern int scan_error;			// The last compiler error code

addr_t	prog_length;			// The byte-code length (program length in bytes)
addr_t	prog_ip;				// The instruction pointer
byte	*prog_source;			// The byte-code
addr_t	data_ip;				// READ/DATA position
addr_t	brun_first_data_ip;			// READ/DATA position

var_t	**tvar;					// The table of variables
lab_t	*tlab;					// The table of labels

stknode_t	*prog_stack;		// The program stack 
dword	prog_stack_alloc;		// The stack size
dword	prog_stack_count;		// The stack pointer
dword	prog_varcount;
dword	prog_labcount;
char	*prog_file;
static dword evt_check_every;

static char	fileName[OS_FILENAME_SIZE+1];
mem_t	bytecode_h;
#if defined(_PalmOS)
DmOpenRef	bytecode_fp;
word		bytecode_recidx;
#endif

/*
*	returns the next 32bit and moves the instruction pointer to the next instruction
*/
dword	code_getnext32()
{
	dword	v;

	memcpy(&v, prog_source+prog_ip, 4);
	prog_ip += 4;
	return v;
}

#if defined(OS_PREC64)
/*
*	returns the next 64bit and moves the instruction pointer to the next instruction
*/
long long	code_getnext64i()
{
	long long	v;

	memcpy(&v, prog_source+prog_ip, 8);
	prog_ip += 8;
	return v;
}
#endif

/*
*	returns the next 64bit and moves the instruction pointer to the next instruction
*/
double	code_getnext64f()
{
	double	v;

	memcpy(&v, prog_source+prog_ip, 8);
	prog_ip += 8;
	return v;
}

#if defined(OS_PREC64)
/*
*	returns the next 128bit and moves the instruction pointer to the next instruction
*/
long double	code_getnext128f()
{
	long double	v;

	memcpy(&v, prog_source+prog_ip, 16);
	prog_ip += 16;
	return v;
}
#endif

/*
*	jump to label
*/
void	code_jump_label(word label_id)
{
	prog_ip = tlab[label_id].ip;
}

/*
*	Put the node 'node' in stack (PUSH)
*/
void	code_push(stknode_t *node)
{
#if defined(_UnixOS) && defined(_CHECK_STACK)
	int		i;
#endif
	if	( prog_stack_count + 1 >= prog_stack_alloc )	{
		err_stackoverflow();
		return;
		}

	prog_stack[prog_stack_count] = *node;
#if defined(_UnixOS) && defined(_CHECK_STACK)
	for ( i = 0; keyword_table[i].name[0] != '\0'; i ++ )	{
		if	( node->type == keyword_table[i].code )	{
			printf("%3d: PUSH %s (%d)\n", prog_stack_count, keyword_table[i].name, prog_line);
			break;
			}
		}
#endif
	prog_stack_count ++;
}

/*
*	Returns and deletes the topmost node from stack (POP)
*/
void	code_pop(stknode_t *node)
{
#if defined(_UnixOS) && defined(_CHECK_STACK)
	int		i;
#endif
	if	( prog_stack_count )	{
		prog_stack_count --;
		if	( node )
			*node = prog_stack[prog_stack_count];
#if defined(_UnixOS) && defined(_CHECK_STACK)
		for ( i = 0; keyword_table[i].name[0] != '\0'; i ++ )	{
			if	( prog_stack[prog_stack_count].type == keyword_table[i].code )	{
				printf("%3d: POP %s (%d)\n", prog_stack_count, keyword_table[i].name, prog_line);
				break;
				}
			}
#endif
		}
	else	{
		if	( node )	{
			err_stackunderflow();
			node->type = 0xFF;
			}
		}
}

/*
*	Returns and deletes the topmost node from stack (POP)
*/
void	code_pop_and_free(stknode_t *node)
{
#if defined(_UnixOS) && defined(_CHECK_STACK)
	int		i;
#endif

	if	( prog_stack_count )	{
		stknode_t *cur_node;

		prog_stack_count --;
		if	( node )
			*node = prog_stack[prog_stack_count];

#if defined(_UnixOS) && defined(_CHECK_STACK)
		for ( i = 0; keyword_table[i].name[0] != '\0'; i ++ )	{
			if	( prog_stack[prog_stack_count].type == keyword_table[i].code )	{
				printf("%3d: POP %s (%d)\n", prog_stack_count, keyword_table[i].name, prog_line);
				break;
				}
			}
#endif
		
		/*
		*	free node's data
		*/
		cur_node = &prog_stack[prog_stack_count];
		switch ( cur_node->type )	{
		case	kwTYPE_CRVAR:
			v_free(tvar[cur_node->x.vdvar.vid]);					// free local variable data
			tmp_free(tvar[cur_node->x.vdvar.vid]);
			tvar[cur_node->x.vdvar.vid] = cur_node->x.vdvar.vptr;		// restore ptr
			break;
		case	kwBYREF:
			tvar[cur_node->x.vdvar.vid] = cur_node->x.vdvar.vptr;
			break;
		case	kwTYPE_VAR:
			if	( (cur_node->x.param.vcheck == 1) || (cur_node->x.param.vcheck == 0x81) ) {
				v_free(cur_node->x.param.res);
				tmp_free(cur_node->x.param.res);
				}
			break;
		case	kwTYPE_RET:
			v_free(cur_node->x.vdvar.vptr);				// free ret-var
			tmp_free(cur_node->x.vdvar.vptr);
			break;
		case	kwFUNC:
			tvar[cur_node->x.vcall.rvid] = cur_node->x.vcall.retvar;	// restore ptr
			break;
		case	kwFOR:
			if	( cur_node->x.vfor.subtype == kwIN )	{
				if	( cur_node->x.vfor.flags & 1 )	{	// allocated in for
					v_free(cur_node->x.vfor.arr_ptr);
					tmp_free(cur_node->x.vfor.arr_ptr);
					}
				}
			}
		}
	else	{
		if	( node )	{
			err_stackunderflow();
			node->type = 0xFF;
			}
		}
}

/*
*/
void	code_pop_until(int type)
{
	stknode_t	node;

	code_pop(&node);
	// 'GOTO' SHIT
	while ( node.type != type )	{
		code_pop(&node); 
		if ( prog_error ) return; 
		}
}

/*
*	Peek the topmost node of stack
*/
stknode_t*	code_stackpeek()
{
	if	( prog_stack_count )
		return &prog_stack[prog_stack_count-1];
	
	return NULL;
}

/*
*	Convertion multi-dim index to one-dim index
*/
addr_t	getarrayidx(var_t *array)
{
	byte	code;
	var_t	var;
	addr_t	idx = 0, lev = 0, m = 0;
	addr_t	idim, i;

	do	{
		v_init(&var);
		eval(&var);
		if	( prog_error )	return 0;
		idim = v_getint(&var);
		v_free(&var);

		if	( prog_error )	
			return 0;

		idim = idim - array->v.a.lbound[lev];

		m = idim;
		for ( i = lev+1; i < array->v.a.maxdim; i ++ )	
			m = m * (ABS(array->v.a.ubound[i] - array->v.a.lbound[i]) + 1);
		idx += m;

		// skip separator
		code = code_peek();
		if	( code == kwTYPE_SEP )	{
			code_skipnext();
			if	( code_getnext() != ',' )
				err_syntax_error();
			}

		// next
		lev ++;
		} while ( code_peek() != kwTYPE_LEVEL_END );

	if	( !prog_error )		{
		if	( array->v.a.maxdim != lev )	
			err_missing_sep();
		}
	return idx;
}

/*
*	Used by code_getvarptr() to retrieve an element ptr of an array
*/
var_t	*code_getvarptr_arridx(var_t *basevar_p)
{
	addr_t	array_index;
	var_t	*var_p = NULL;

	if	( code_peek() != kwTYPE_LEVEL_BEGIN )
		err_arrmis_lp();
	else	{
		code_skipnext();	// '('
		array_index = getarrayidx(basevar_p);
		if	( !prog_error )	{
			if	( array_index < basevar_p->v.a.size )	{
				var_p = (var_t*) (basevar_p->v.a.ptr + (array_index * sizeof(var_t)));

				if	( code_peek() == kwTYPE_LEVEL_END )		{
					code_skipnext();		// ')', ')' level
					if	( code_peek() == kwTYPE_LEVEL_BEGIN )	{	// there is a second array inside
						if	( var_p->type != V_ARRAY )
							err_varisnotarray();
						else
							return code_getvarptr_arridx(var_p);
						}
					}
				else
					err_arrmis_rp();

				}
			else	
				err_arridx();
			}
		}

	return var_p;
}

/*
*	Returns the varptr of the next variable
*	if the variable is an array returns the element ptr
*/
var_t	*code_getvarptr()
{
	var_t	*basevar_p, *var_p = NULL;

	if	( code_peek() == kwTYPE_VAR )	{
		code_skipnext();
		var_p = basevar_p = tvar[code_getaddr()];
		if	( basevar_p->type == V_ARRAY )	{		/* variable is an array	*/
			if	( code_peek() == kwTYPE_LEVEL_BEGIN )
				var_p = code_getvarptr_arridx(basevar_p);
			}
		else	{
			if	( code_peek() == kwTYPE_LEVEL_BEGIN )
				err_varisnotarray();
			}
		}
	
	if	( var_p == NULL && !prog_error  )	{
		rt_raise("NOT A VARIABLE");
		#if defined(_UnixOS)
//		memcpy(NULL, var_p, 1024);	// crash
		#else
		return tvar[0];
		#endif
		}

	return var_p;
}

/*
*	Used by code_isvar() to retrieve an element ptr of an array
*/
var_t	*code_isvar_arridx(var_t *basevar_p)
{
	addr_t	array_index;
	var_t	*var_p = NULL;

	if	( code_peek() != kwTYPE_LEVEL_BEGIN )
		return NULL;
	else	{
		code_skipnext();	// '('
		array_index = getarrayidx(basevar_p);
		if	( !prog_error )	{
			if	( array_index < basevar_p->v.a.size )	{
				var_p = (var_t*) (basevar_p->v.a.ptr + (array_index * sizeof(var_t)));

				if	( code_peek() == kwTYPE_LEVEL_END )		{
					code_skipnext();		// ')', ')' level
					if	( code_peek() == kwTYPE_LEVEL_BEGIN )	{	// there is a second array inside
						if	( var_p->type != V_ARRAY )
							return NULL;
						else
							return code_isvar_arridx(var_p);
						}
					}
				else
					return NULL;

				}
			else	
				return NULL;
			}
		}

	return var_p;
}

/*
*	returns true if the next code is a variable (on expressions returns false)
*/
int		code_isvar()	
{
	var_t	*basevar_p, *var_p = NULL;
	addr_t	cur_ip;

	cur_ip = prog_ip;	// store IP

	if	( code_peek() == kwTYPE_VAR )	{
		code_skipnext();
		var_p = basevar_p = tvar[code_getaddr()];
		if	( basevar_p->type == V_ARRAY )	{	/* variable is an array	*/
			if	( code_peek() == kwTYPE_LEVEL_BEGIN )	
				var_p = code_isvar_arridx(basevar_p);
			}
		else	{
			if	( code_peek() == kwTYPE_LEVEL_BEGIN )
				var_p = NULL;
			}
		}

	if	( var_p )	{
		if	( kw_check_evexit(code_peek()) || code_peek() == kwTYPE_LEVEL_END )	{
			prog_ip = cur_ip;	// restore IP
			return 1;
			}
		}

	prog_ip = cur_ip;	// restore IP
	return 0;
}

/*
*	sets the value of an integer system-variable
*/
void	setsysvar_int(int index, long value)
{
	var_t	*var_p = tvar[index];
	
	var_p->type = V_INT;
	var_p->const_flag = 1;
	var_p->v.i = value;
}

/*
*	sets the value of a real system-variable
*/
void	setsysvar_num(int index, double value)
{
	var_t	*var_p = tvar[index];
	
	var_p->type = V_NUM;
	var_p->const_flag = 1;
	var_p->v.n = value;
}

/*
*	sets the value of an string system-variable
*/
void	setsysvar_str(int index, const char *value)
{
	var_t	*var_p = tvar[index];
	int		l = strlen(value)+1;
	
	if	( var_p->type == V_STR )
		tmp_free(var_p->v.p.ptr);

	var_p->type = V_STR;
	var_p->const_flag = 1;
	var_p->v.p.ptr = tmp_alloc(l);
	strcpy(var_p->v.p.ptr, value);
	var_p->v.p.size = l;
}

/*
*	RUN byte-code
*	
*	source structure:
*		[bc_head_t]
*		for each label:
* 			[word]	... label-ip
*		[the bytecode itself]
*
*	brun_init(source)
*	brun()
*	brun_close()
*/
void	brun_init(byte *source)
{
	#if defined(OS_ADDR16)
	word	i;
	#else
	dword	i;
	#endif
	#if defined(_UnixOS)
	char	homedir[OS_PATHNAME_SIZE+1];
	int		l;
	#endif
	bc_head_t	hdr;
	byte		*cp;

	memcpy(&hdr, source, sizeof(bc_head_t));
	cp = source + sizeof(bc_head_t);

	prog_varcount = hdr.var_count;
	prog_labcount = hdr.lab_count;

	/*
	*	create variable-table
	*/
	if	( prog_varcount == 0 )	prog_varcount ++;
	tvar = tmp_alloc(sizeof(var_t *) * prog_varcount);
	for ( i = 0; i < prog_varcount; i ++ )	
		tvar[i] = v_new();

	/*
	*	create label-table
	*/
	if	( prog_labcount == 0 )	
		tlab = tmp_alloc(sizeof(lab_t));
	else	{
		tlab = tmp_alloc(sizeof(lab_t) * prog_labcount);
		for ( i = 0; i < prog_labcount; i ++ )		{
			memcpy(&tlab[i].ip, cp, ADDRSZ);
			cp += ADDRSZ;
			}
		}

	/*
	*	create system stack
	*/
	prog_stack_alloc = SB_EXEC_STACK_SIZE;
	prog_stack = tmp_alloc(sizeof(stknode_t) * prog_stack_alloc);
	prog_stack_count = 0;

	/*
	*	initialize the rest globals
	*/
	prog_error  = 0;
	prog_line   = 0;
	
	data_ip = brun_first_data_ip = hdr.data_ip;
	prog_length = hdr.bc_count;
	prog_source = cp;
	prog_ip     = 0;

	/*
	*	predefined variables
	*/
	setsysvar_int(SYSVAR_OSVER, 	os_ver);
	setsysvar_str(SYSVAR_OSNAME, 	OS_NAME);
	setsysvar_int(SYSVAR_SBVER, 	SB_DWORD_VER);
	setsysvar_num(SYSVAR_PI,    	3.14159265358979323846);
	setsysvar_int(SYSVAR_XMAX,		159);
	setsysvar_int(SYSVAR_YMAX,		159);
	setsysvar_int(SYSVAR_BPP,		1);
	setsysvar_int(SYSVAR_TRUE,  	1);
	setsysvar_int(SYSVAR_FALSE, 	0);
	setsysvar_int(SYSVAR_LINECHART, 1);
	setsysvar_int(SYSVAR_BARCHART,  2);
	setsysvar_str(SYSVAR_PWD,   	dev_getcwd());
	setsysvar_str(SYSVAR_COMMAND, 	opt_command);

	#if defined(_UnixOS)
	if	( dev_getenv("HOME") )
		strcpy(homedir, dev_getenv("HOME"));
	else
		strcpy(homedir, "/tmp/");
	
	l = strlen(homedir);
	if	( homedir[l-1] != OS_DIRSEP )	{
		homedir[l] = OS_DIRSEP;
		homedir[l+1] = '\0';
		}
	setsysvar_str(SYSVAR_HOME, homedir);
	#elif defined(_DOS)
	setsysvar_str(SYSVAR_HOME, "c:/");	// djgpp uses /
	#elif defined(_Win32)
    #if defined(_WinBCB)
	setsysvar_str(SYSVAR_HOME, "c:\\");
    #else
	if	( OS_DIRSEP == '/' )
		setsysvar_str(SYSVAR_HOME, "c:/");	// mingw32
	else
		setsysvar_str(SYSVAR_HOME, "c:\\");
    #endif
	#else
	setsysvar_str(SYSVAR_HOME, "");
	#endif

	/*
	*	ready
	*/
	eval_alloc_stack();
	cmd_play_reset();
}

/*
*	load & compile the source file 'fileName'
*/
void	brun_load(char *fileName, int nocomp)
{
	char		*ptr;
#if defined(_PalmOS)
	DmOpenRef	fp;
	LocalID		lid;
	word		recIndex;
#endif

	if	( opt_loadmod )
		sblmgr_init(1, opt_modlist);
	else
		sblmgr_init(0, NULL);

	if	( !nocomp )	{
		bc_init();
		prog_file = fileName;

		/*
		*	compile - pass1
		*/
		bc_load(fileName);

		/*
		*	compile - pass2 & binary
		*/
		bytecode_h = 0;
		if	( !bc_get_error() )	{
			bc_pass2();
			if	( !bc_get_error() )	{
				bc_check_labels();
				if	( !bc_get_error() )	
					bytecode_h = bc_createbin();
				}
			}
		bc_close();

		/*
		*	run
		*/
		if	( bytecode_h )	{
			#if defined(_PalmOS)
			/*
			*	PalmOS - Memory optimization
			*
			*	We'll free 'bytecode' memory by using a database record handle
			*	Because 'bytecode' it is working as read-only we have no problems
			*	using record's memory handle instead of a dynamic-RAM handle
			*/
			mem_t		rec_h;
			char		*prevrec, *newrec;

			lid = DmFindDatabase(0, (char *) "SBVM.BIN");
			if	( lid )	
				DmDeleteDatabase(0, lid);
			DmCreateDatabase(0, "SBVM.BIN", ID_SmBa, ID_DATA, 0);
			lid = DmFindDatabase(0, (char *) "SBVM.BIN");
			fp = DmOpenDatabase(0, lid, dmModeReadWrite);

			// save bytecode_h
			recIndex = dmMaxRecordIndex;
			rec_h = DmNewRecord(fp, &recIndex, MemHandleSize(bytecode_h));

			newrec = mem_lock(bytecode_h);
			prevrec = mem_lock(rec_h);
			DmWrite(prevrec, 0, newrec, MemHandleSize(bytecode_h));
			mem_unlock(rec_h);
			mem_unlock(bytecode_h);
			DmReleaseRecord(fp, recIndex, 1);

			// free bytecode_h
			mem_free(bytecode_h);
			bytecode_h = DmGetRecord(fp, recIndex);

			bytecode_fp = fp;
			bytecode_recidx = recIndex;

			/*
			*	Try to defrag dynamic memory
			*/
			MemHeapCompact(0);
			#endif

			ptr = mem_lock(bytecode_h);
			brun_init(ptr);
			#if defined(_PalmOS)
			EvtWakeup();		// !@$!@#$@#!$#@#$$!
			#endif
			}
		else
			prog_error = -32;
		}
	else	{
		/*
		*	Restart
		*/
		#if defined(_PalmOS)
		lid = DmFindDatabase(0, (char *) "SBVM.BIN");
		if	( !lid )	
			panic("SBVM.BIN not found");
		fp = DmOpenDatabase(0, lid, dmModeReadWrite);
		recIndex = 0;
		bytecode_h = DmGetRecord(fp, recIndex);

		bytecode_fp = fp;
		bytecode_recidx = recIndex;

		/*
		*	Try to defrag dynamic memory
		*/
		MemHeapCompact(0);

		ptr = mem_lock(bytecode_h);
		brun_init(ptr);
		EvtWakeup();		// !@$!@#$@#!$#@#$$!
		#endif
		}
}

/*
*	close the executor
*/
int		brun_close(void)
{
	word		i;
	stknode_t	node;

	if	( bytecode_h )	{
		// clean up - format list
		free_format();

		// clean up - eval stack
		eval_free_stack();

		// clean up - prog stack
		while ( prog_stack_count > 0 )	
			code_pop_and_free(&node);
		tmp_free(prog_stack);

		// clean up - variables
		for ( i = 0; i < prog_varcount; i ++ )	{
			v_free(tvar[i]);
			tmp_free(tvar[i]);
			}
		tmp_free(tvar);

		// clean up - the rest
		tmp_free(tlab);
		mem_unlock(bytecode_h);

		#if defined(_PalmOS)
		DmReleaseRecord(bytecode_fp, bytecode_recidx, 1);
// restart
//		DmRemoveRecord(bytecode_fp, bytecode_recidx);
		DmCloseDatabase(bytecode_fp);
		#else
		mem_free(bytecode_h);
		#endif

		bytecode_h = 0;
		}

	//
	sblmgr_close();
	if	( prog_error != -1 && prog_error != 0 )
		return 1;
	return 0;
}

/*
*	BREAK
*/
void	brun_stop()
{
	prog_error = -3;
}

/*
*	returns the status of executor (runing or stopped)
*/										
int		brun_status()
{
	if	( prog_error )
		return BRUN_STOPPED;
	return BRUN_RUNNING;
}

/*
*	BREAK - display message, too
*/
void	brun_break()
{
	if	( brun_status() == BRUN_RUNNING )	{
		dev_settextcolor(15, 0);
		#if defined(_PalmOS)
		dev_print("\a\n");
		dev_setxy(0,os_graf_my-9);
		dev_printf("\033[91m\033[0m\033[7m * BREAK AT %d * - Press '.' to return\4 \033[80m\033[0m", prog_line);
		#else
		dev_printf("\n\033[0m\033[80m\a\033[7m * BREAK AT LINE %d * \033[0m\n", prog_line);
		#endif
		}
	brun_stop();
}

/*
*/
void	cmd_chain(void)
{
	var_t	var;

	v_init(&var);
	eval(&var);
	if	( prog_error )	
		return;
	else	{
		if	( var.type != V_STR )	{
			err_typemismatch();
			return;
			}
		else	{
			strcpy(fileName, var.v.p.ptr);
			if	( !basFileExist(fileName) )	{
				rt_raise("CHAIN: FILE '%s' DOES NOT EXIST", fileName);
				v_free(&var);
				return;
				}
			v_free(&var);
			}
		}

	brun_close();
	#if defined(_PalmOS)
	dev_cls();
	#endif
	brun_load(fileName, 0);
}

/*
*	RUN "program"
*
*	For PalmOS use RUN CreatorID (i.e. run "calc")
*/
void	cmd_run(void)
{
	var_t	var;

	v_init(&var);
	eval(&var);
	if	( prog_error )	
		return;
	else	{
		if	( var.type != V_STR )	{
			err_typemismatch();
			return;
			}
		else	{
			strcpy(fileName, var.v.p.ptr);
			v_free(&var);
			}
		}

//	brun_close();
	if	( !dev_run(fileName) )
		rt_raise("RUN \"%s\" FAILED", fileName);
}

/*
*/
void	bc_loop(int isf)
{
	register dword now;
	static dword next_check;
	stknode_t	node;
	byte	code, pops;
	byte	trace_flag = 0;
	addr_t	next_ip;
	int		udf_level = 0;
	pcode_t	pcode;
	int		i;


	/**
	*	DO NOT CREATE FUNCTION-PTR-TABLE
	*	function tables does not works with multiseg
	*
	*	For commands witch changes the IP use
	*
	*   case mycommand:
	*		command();
	*		if	( prog_error )	break;
	*		continue;
	*/
	while ( prog_ip < prog_length )	{

//		run_events();

		#if defined(_PalmOS)
		now = TimGetTicks();
		#elif defined(_FRANKLIN_EBM)
		now = time_get_onOS();
		#else
		now = clock();
		#endif

		// events
		if	( now >= next_check )	{
			next_check = now + evt_check_every;

			switch ( dev_events(0) )		{
			case	-1:
				break;				// break event
			case	-2:
				prog_error=-2;
				#if defined(_PalmOS)
				dev_settextcolor(15, 0);
				dev_print("\a\n");
				dev_setxy(0,os_graf_my-9);
				dev_printf("\033[91m\033[0m\033[7m * BREAK AT %d * - Press '.' to return\4 \033[80m\033[0m", prog_line);
				#else
				dev_printf("\n\033[0m\033[80m\a\033[7m * BREAK AT LINE %d * \033[0m\n", prog_line);
				#endif
				break;
				};
			}

		//
		if	( !prog_error )	{
			// next command
			code = prog_source[prog_ip];
			prog_ip ++;

			switch ( code )	{
			case	kwLABEL:
			case	kwREM:
			case	kwTYPE_EOC:
				continue;
			case	kwTYPE_LINE:
				prog_line = code_getaddr();
				if	( trace_flag )
					dev_printf("<%d>", prog_line);
				continue;
			case 	kwLET:
				cmd_let(0);
				break;
			case 	kwCONST:
				cmd_let(1);
				break;
			case	kwGOTO:
				next_ip = code_getaddr();

				// clear the stack (whatever you can)
				pops = code_getnext();
				while ( pops > 0 )	{
					code_pop_and_free(NULL);
					pops --;
					}

				// jump
				prog_ip = next_ip;
				continue;
			case	kwGOSUB:
				cmd_gosub();
				if	( prog_error )	break;
				continue;
			case	kwRETURN:
				cmd_return();
				if	( prog_error )	break;
				continue;
			case	kwONJMP:
				cmd_on_go();
				if	( prog_error )	break;
				continue;
			case	kwPRINT:
				cmd_print(PV_CONSOLE);
				break;
			case	kwINPUT:
				cmd_input(PV_CONSOLE);
				break;
			case	kwIF:
				cmd_if();
				if	( prog_error )	break;
				continue;
			case	kwELIF:
				cmd_elif();
				if	( prog_error )	break;
				continue;
			case	kwELSE:
				cmd_else();
				if	( prog_error )	break;
				continue;
			case	kwENDIF:
				code_pop(&node);
				// 'GOTO' SHIT
				while ( node.type != kwIF )	{
					code_pop(&node); 
					if ( prog_error ) break; 
					}

				if	( !prog_error )	{
					prog_ip += (ADDRSZ+ADDRSZ);
					continue;
					}
				break;
			case	kwFOR:
				cmd_for();
				if	( prog_error )	break;
				continue;
			case	kwNEXT:
				cmd_next();
				if	( prog_error )	break;
				continue;
			case	kwWHILE:
				cmd_while();
				if	( prog_error )	break;
				continue;
			case	kwWEND:
				cmd_wend();
				if	( prog_error )	break;
				continue;
			case	kwREPEAT:
				node.type = kwREPEAT;
				code_skipaddr();
				next_ip = (code_getaddr()) + 1;
				node.exit_ip = code_peekaddr(next_ip);
				code_push(&node);
				continue;
			case	kwUNTIL:
				cmd_until();
				if	( prog_error )	break;
				continue;
			case 	kwDIM:
				cmd_dim(0);
				break;
			case 	kwREDIM:
				cmd_redim();
				break;
			case	kwAPPEND:
				cmd_ladd();
				break;
			case	kwINSERT:
				cmd_lins();
				break;
			case	kwDELETE:
				cmd_ldel();
				break;
			case 	kwERASE:
				cmd_erase();
				break;
			case	kwREAD:
				cmd_read();
				break;
			case	kwDATA:
				cmd_data();
				break;
			case	kwRESTORE:
				cmd_restore();
				break;

			/* -----------------------------------------
			*	external procedures
			*/
			case	kwTYPE_CALLEXTP:		// [lib][index]
				{
					addr_t	lib, idx;

					lib = code_getaddr();
					idx = code_getaddr();
					sblmgr_procexec(lib, idx);
				}
				break;
			/* -----------------------------------------
			*	buildin procedures
			*/
			case	kwTYPE_CALLP:
				pcode = code_getaddr();
				switch ( pcode )	{
				case kwCLS:
					dev_cls();
					graph_reset();
					break;
				case kwRTE:
					cmd_RTE();
					break;
//				case kwSHELL:
//					break;
				case kwENVIRON:
					cmd_environ();
					break;
				case kwLOCATE:
					cmd_locate();
					break;
				case kwAT:
					cmd_at();
					break;
				case kwPEN:
					cmd_pen();
					break;
				case kwDATEDMY:
					cmd_datedmy();
					break;
				case kwBEEP:
					cmd_beep();
					break;
				case kwSOUND:
					cmd_sound();
					break;
				case kwPSET:
					cmd_pset();
					break;
				case kwRECT:
					cmd_rect();
					break;
				case kwCIRCLE:
					cmd_circle();
					break;
				case kwRANDOMIZE:
					cmd_randomize();
					break;
				case kwWSPLIT:
					cmd_wsplit();
					break;
				case kwSPLIT:
					cmd_split();
					break;
				case kwWJOIN:
					cmd_wjoin();
					break;
				case kwPAUSE:
					cmd_pause();
					break;
				case kwDELAY:
					cmd_delay();
					break;
				case kwARC:
					cmd_arc();
					break;
				case kwDRAW:
					cmd_draw();
					break;
				case kwPAINT:
					cmd_paint();
					break;
				case kwPLAY:
					cmd_play();
					break;
				case kwSORT:
					cmd_sort();
					break;
				case kwSEARCH:
					cmd_search();
					break;
				case kwROOT:
					cmd_root();
					break;
				case kwDIFFEQ:
					cmd_diffeq();
					break;
				case kwCHART:
					cmd_chart();
					break;
				case kwWINDOW:
					cmd_window();
					break;
				case kwVIEW:
					cmd_view();
					break;
				case kwDRAWPOLY:
					cmd_drawpoly();
					break;
				case kwM3IDENT:
					cmd_m3ident();
					break;
				case kwM3ROTATE:
					cmd_m3rotate();
					break;
				case kwM3SCALE:
					cmd_m3scale();
					break;
				case kwM3TRANSLATE:
					cmd_m3translate();
					break;
				case kwM3APPLY:
					cmd_m3apply();
					break;
				case kwSEGINTERSECT:
					cmd_intersect();
					break;
				case kwPOLYEXT:
					cmd_polyext();
					break;
				case kwDERIV:
					cmd_deriv();
					break;
				case kwLOADLN:
					cmd_floadln();
					break;
				case kwSAVELN:
					cmd_fsaveln();
					break;
				case kwKILL:
					cmd_fkill();
					break;
				case kwRENAME:
					cmd_filecp(1);
					break;
				case kwCOPY:
					cmd_filecp(0);
					break;
				case kwCHDIR:
					cmd_chdir();
					break;
				case kwMKDIR:
					cmd_mkdir();
					break;
				case kwRMDIR:
					cmd_rmdir();
					break;
				case kwFLOCK:
					cmd_flock();
					break;
				case kwCHMOD:
					cmd_chmod();
					break;
				case kwPLOT:
					cmd_plot();
					break;
				case kwSWAP:
					cmd_swap();
					break;
				case kwSTKDUMP:
					dev_print("\nSTKDUMP:\n");
					dump_stack();
					prog_error = -1;
					break;
				default:
					rt_raise("Unsupported buildin procedure call %ld, please report this bug", pcode);
					}
				break;

			case	kwTYPE_CALL_UDP:		// user defined procedure
				cmd_udp(kwPROC);
				if	( isf )	
					udf_level ++;
				if	( prog_error )	break;
				continue;
			case	kwTYPE_CALL_UDF:		// user defined function
				if	( isf )	{
					cmd_udp(kwFUNC);
					udf_level ++;
					}
				else
					err_syntax();
				if	( prog_error )	break;
				continue;
			case	kwTYPE_RET:
				cmd_udpret();
				if	( isf )	{
					udf_level --;
					if	( udf_level == 0 )
						return;
					}
				if	( prog_error )	break;
				continue;
			case	kwTYPE_CRVAR:
				cmd_crvar();
				break;
			case	kwTYPE_PARAM:
				cmd_param();
				break;
			case	kwEXIT:
				pops = cmd_exit();
				if	( isf && pops )	{
					udf_level --;
					if	( udf_level == 0 )
						return;
					}
				if	( prog_error )	break;
				continue;

			////////////////
			case	kwLINE:
				cmd_line();
				break;

			// third class
			case	kwCOLOR:
				cmd_color();
				break;
//			case	kwINTEGRAL:
//				cmd_integral();
//				break;

			// --- at end ---
			case	kwOPEN:
				cmd_fopen();
				break;
			case	kwCLOSE:
				cmd_fclose();
				break;
			case	kwFILEWRITE:
				cmd_fwrite();
				break;
			case	kwFILEREAD:
				cmd_fread();
				break;
			case kwLOGPRINT:
				cmd_print(PV_LOG);
				break;
			case	kwFILEPRINT:
				cmd_print(PV_FILE);
				break;
			case	kwSPRINT:
				cmd_print(PV_STRING);
				break;
			case	kwLINEINPUT:
				cmd_flineinput();
				break;
			case	kwSINPUT:
				cmd_input(PV_STRING);
				break;
			case	kwFILEINPUT:
				cmd_input(PV_FILE);
				break;
			case	kwSEEK:
				cmd_fseek();
				break;
			case	kwTRON:
				trace_flag = 1;
				continue;
			case	kwTROFF:
				trace_flag = 0;
				continue;

			case	kwSTOP:
			case	kwEND:
				if	( (prog_length-1) > prog_ip )	{
					if	( code_peek() != kwTYPE_EOC && code_peek() != kwTYPE_LINE )	{
						var_t	ec;

						v_init(&ec);
						eval(&ec);
						opt_retval = v_igetval(&ec);
						v_free(&ec);
						}
					}
				prog_error = -1;
				break;
			case	kwCHAIN:
				cmd_chain();
				break;

			case	kwRUN:
				cmd_run();
				break;

			default:
				rt_raise("SEG:CODE[%d]=%02X", prog_ip, prog_source[prog_ip]);
				//////////
				dev_printf("OUT OF ADDRESS SPACE\n");
				for ( i = 0; keyword_table[i].name[0] != '\0'; i ++)	{
					if ( prog_source[prog_ip] == keyword_table[i].code )	{
						dev_printf("OR ILLEGAL CALL TO '%s'", keyword_table[i].name);
						break;
						}
					}
				//////////
				}
			}

		// too many parameters
		code = prog_source[prog_ip];
		if	( code != kwTYPE_EOC && code != kwTYPE_LINE && !prog_error )	{
			if	( code == kwTYPE_SEP )	
				rt_raise("COMMAND SEPARATOR '%c' FOUND!", prog_source[prog_ip+1]);
			else
				rt_raise("PARAM COUNT ERROR @%d=%X", prog_ip, prog_source[prog_ip]);
			}
		else	{
			prog_ip ++;
			if ( code == kwTYPE_LINE )	{
				prog_line = code_getaddr();
				if	( trace_flag )
					dev_printf("<%d>", prog_line);
				}
			}

		// quit on error
		if	( prog_error )
			break;
		}
}

/*
*	executor
*
*	file   = the source file
*	nocomp = no-compilation (use the previous bc-file)
*/
#if defined(SONY_CLIE)
extern int font_h;
extern int maxline;
#endif
void	brun(char *file, int nocomp)
{
	dword	tps;

	#if defined(SONY_CLIE)
	if	( use_sony_clie )	{
		HRFntSetFont(sony_refHR, hrStdFont);
		font_h = FntCharHeight();
		maxline = 320/font_h;
		os_graf_mx = 320;
		os_graf_my = 320;
		}
	#endif

 	#if defined(_VTOS)
	// Speed up initial parsing by 2x under VTOS
 	extern void CpuChangeSpeed(int);
 	CpuChangeSpeed(0<<10);
 	#endif

	brun_load(file, nocomp);
	if	( prog_error )
		return;

 	#if defined(_VTOS)
	// Run at normal CPU speed
 	CpuChangeSpeed(1<<10);
 	#endif

	dev_init(opt_graphics,0);

	#if defined(_PalmOS)
	tps = SysTicksPerSecond();
	#else
	tps = CLOCKS_PER_SEC;
	#endif
	evt_check_every = (50 * tps) / 1000;	// 50ms

	// randomize ticks
	#if defined(_PalmOS)
	SysRandom(TimGetTicks());
	#else
	srand(clock());
	#endif

	// begin
	bc_loop(0);

	if	( prog_error == -1 || prog_error == 0 )	{	// normal exit
		if	( os_graphics )	
			dev_settextcolor(0, 15);

		if	( !opt_quite )	{
			#if defined(_PalmOS)
			dev_print("\a\n");
			dev_settextcolor(15,0);
			dev_setxy(0,os_graf_my-9);
			dev_print("\033[91m\033[0m\033[7m * DONE * - Press '.' to return\4 \033[80m\033[0m");
			#else
			dev_print("\n\033[0m\033[80m\a\033[7m * DONE * \033[0m\n");
			#endif
			}
		prog_error = -1;
		}

	brun_close();

	dev_restore();

	#if defined(_PalmOS)
	{
		EventType	e;

		e.eType = menuEvent;
		e.data.menu.itemID = 1601;	// SBFinishNotify;
		EvtAddEventToQueue(&e);
	}
	#endif
}

/*
*	debug info
*	stack dump
*/
void	dump_stack()
{
	stknode_t	node;
	int			i;

	do	{
		code_pop(&node);
		if	( node.type != 0xFF )	{
			for ( i = 0; keyword_table[i].name[0] != '\0'; i ++ )	{
				if	( node.type == keyword_table[i].code )	{
					dev_printf("\t%s", keyword_table[i].name);
					switch ( node.type )	{
					case kwGOSUB:
						dev_printf(" RIP: %d", node.x.vgosub.ret_ip);
						if	( prog_source[node.x.vgosub.ret_ip] == kwTYPE_LINE )	{
							dev_printf(" = LI %d", (*((word *) (prog_source+node.x.vgosub.ret_ip+1))) - 1);
							}
						break;
						}
					dev_print("\n");
					break;
					}
				}
			}
		else
			break;
		} while ( 1 );
}

#if !defined(_PalmOS) && !defined(_VTOS)
//////////////////////////////////////////////////////////////////////////////////////
//////// Command-line interface (Unix)

/*
*/
void	prcmd(code_t code)
{
	char	cmd[16];						

	if	( kw_getcmdname(code, cmd) )
		fprintf(stderr, "{%s}", cmd);
	else
		fprintf(stderr, "{N/A:%d}", code);
}

/* ---------------------------------------------------
*
*	decompiler
*
*/
void	dump_bytecode(const char *srcfile, FILE *stream)
{
	int		i, c, b, d, h, l, j;
	long	lng;
	int		len, new_ip;
	char	cmd[33];
	FILE	*fp;
	char_p_t *source = NULL;
	int		lcount = 0, lsize = 0;

	/////////////////
	// read source
	fp = fopen(srcfile, "rt");
	if	( fp )	{
		char	buf[512];

		lsize = 1024;
		source = tmp_alloc(lsize * sizeof(char_p_t));
		lcount = 0;
		while ( fgets(buf, 511, fp) )	{
			source[lcount] = tmp_alloc(strlen(buf)+1);
			strcpy(source[lcount], buf);
			lcount ++;
			if	( lcount >= lsize )	{
				lsize += 1024;
				source = tmp_realloc(source, lsize * sizeof(char_p_t));
				}
			}
		fclose(fp);
		}

	/////////////////
	fprintf(stream, "Program size   = %d\n", prog_length);
    fprintf(stream, "Variable-count = %d\n", prog_varcount);
    fprintf(stream, "Label-count    = %d\n", prog_labcount);

    fprintf(stream, "\n=== Label-table ===\n");
    for ( i = 0; i < prog_labcount; i ++ )	
	    fprintf(stream, "%d = %d\n", i, tlab[i].ip);

    fprintf(stream, "\n=== Code ===\n");
	do	{
		b = code_getnext();
		h = b >> 4;
		l = b & 0xF;

		fprintf(stream, "%08d: %c%c %c ",
			(int) prog_ip-1,
			"0123456789ABCDEF"[h],
			"0123456789ABCDEF"[l],
			(b>=32) ? b : 42 );

		switch ( b )	{
		case	kwTYPE_LINE:
			c = code_getaddr();
			if	( source && lcount >= c )		{
				fprintf(stream, "--LINE: %d\n\n%d: ", c, c);
				fprintf(stream, "%s", source[c-1]);
				}
			else
				fprintf(stream, "--LINE: %d", c);
			break;
		case	kwTYPE_EVPUSH:
			fprintf(stream, "PUSH RESULT");
			break;
		case	kwTYPE_EVPOP:
			fprintf(stream, "POP LEFT");
			break;
		case	kwTYPE_EOC:
			fprintf(stream, "--EOC");
			break;
		case	kwTYPE_CALLF:
			lng = code_getaddr();
			kw_getfuncname(lng, cmd);
			fprintf(stream, "$bfn %ld: %s ", lng, cmd);
			break;
		case	kwTYPE_CALLP:
			lng = code_getaddr();
			kw_getprocname(lng, cmd);
			fprintf(stream, "$bpc %ld: %s ", lng, cmd);
			break;
		case	kwTYPE_CALLEXTF:
		case	kwTYPE_CALLEXTP:		// [lib][index]
			c = code_getaddr();
			d = code_getaddr();
			if	( b == kwTYPE_CALLEXTF )	{
				sblmgr_getfuncname(c, d, cmd);
				fprintf(stream, "$extfn %d:%d (%s)", c, d, cmd);
				}
			else	{
				sblmgr_getprocname(c, d, cmd);
				fprintf(stream, "$extpc %d:%d (%s)", c, d, cmd);
				}
			break;
		case	kwPROC:
			fprintf(stream, "=== PROCEDURE ===");
			break;
		case	kwFUNC:
			fprintf(stream, "=== FUNCTION ===");
			break;
		case	kwFILEPRINT:
			fprintf(stream, "FILE PRINT");
			break;
		case	kwFILEINPUT:
			fprintf(stream, "FILE INPUT");
			break;
		case	kwLINEINPUT:
			fprintf(stream, "LINE INPUT");
			break;
		case	kwFORSEP:
			fprintf(stream, "FOR-SEP");
			break;
		case	kwLOOPSEP:
			fprintf(stream, "LOOP-SEP");
			break;
		case	kwINPUTSEP:
			fprintf(stream, "INPUT-SEP");
			break;
		case	kwAPPEND:
			fprintf(stream, "APPEND");
			break;
		case	kwCODEARRAY:
			fprintf(stream, "ARRAY");
			break;
		case	kwTYPE_PARAM:
		case	kwTYPE_CRVAR:
			c = code_getnext();
			if	( b == kwTYPE_PARAM )
				fprintf(stream, "PARAM = %d: ", c);
			else
				fprintf(stream, "CRVAR = %d: ", c);
				
			d = 0;
			for ( j = 0; j < c; j ++ )	{
				if	( b == kwTYPE_PARAM )	{
					d = code_getnext();
					if	( d & 0x80 )
						fprintf(stream, "&");
					}
				fprintf(stream, "%d ", code_getaddr());
				}
			break;
		case	kwUSE:
			fprintf(stream, "USE %d ", code_getaddr());
			fprintf(stream, "EXIT %d", code_getaddr());
			break;
		case	kwTYPE_RET:
			fprintf(stream, "=== UDP/F RET ===");
			break;
		case	kwTYPE_CALL_UDP:
			fprintf(stream, "CALL UDP %d", code_getaddr());
			code_skipaddr();
			break;
		case	kwTYPE_CALL_UDF:
			fprintf(stream, "CALL UDF %d", code_getaddr());
			fprintf(stream, ", RVID: %d", code_getaddr());
			break;
		case	kwEXIT:
			fprintf(stream, "EXIT ");
			c = code_getnext();
			switch ( c )	{
			case kwFORSEP:
				fprintf(stream, "FOR-SEP");
				break;
			case kwLOOPSEP:
				fprintf(stream, "LOOP-SEP");
				break;
			case kwPROCSEP:
				fprintf(stream, "PROC-SEP");
				break;
			case kwFUNCSEP:
				fprintf(stream, "FUNC-SEP");
				break;
			default:
				fprintf(stream, "AUTO");
				};

			break;

		default:
			for ( c = 0; keyword_table[c].name[0] != '\0'; c ++)	{
				if ( b == keyword_table[c].code )	{
					fprintf(stream, "%s ", keyword_table[c].name);
					break;
					}
				}

			switch ( b )	{
			case	kwTYPE_SEP:
				c = code_getnext();
				fprintf(stream, "\'%c\' ", (c>=32) ? c : '?');
				break;
			case	kwTYPE_LOGOPR:
			case	kwTYPE_CMPOPR:
			case	kwTYPE_ADDOPR:
			case	kwTYPE_MULOPR:
			case	kwTYPE_POWOPR:
			case	kwTYPE_UNROPR:
				c = code_getnext();
				fprintf(stream, "DATA %d \'%c\' ", c, (c>=32) ? c : '?');
				break;
			case	kwTYPE_LEVEL_BEGIN:
				fprintf(stream, "(");
				break;
			case	kwTYPE_LEVEL_END:
				fprintf(stream, ")");
				break;
			case	kwONJMP:
				new_ip = code_getaddr();
				code_skipaddr();
				b = code_getnext();
				c = code_getnext();
				fprintf(stream, "ON %s CNT %d\n", ((b == kwGOTO)?"GOTO":"GOSUB"), c);
				fprintf(stream, "      NEXT IP %d\n", new_ip);
				for ( j = 0; j < c; j ++ )	
					fprintf(stream, "       JMP IP %d\n", code_getaddr());
				fprintf(stream, "ON %s EXPR:", ((b == kwGOTO)?"GOTO":"GOSUB"));
				break;
			case	kwGOTO:
				new_ip = code_getaddr();
				fprintf(stream, "= JMP %d ", new_ip);
				c = code_getnext();
				fprintf(stream, "POPs %d", c);
				break;
			case	kwGOSUB:
				c = code_getaddr();
				fprintf(stream, "ID %d ", c);
				new_ip = tlab[c].ip;
				fprintf(stream, "= JMP %d ", new_ip);
				break;
			case	kwLET:
				break;
			case	kwRESTORE:
			case	kwTYPE_LINE:
			case	kwLABEL:
			case	kwTYPE_VAR:
				fprintf(stream, "ID %d ", code_getaddr());
				break;
			case	kwTYPE_INT:
				fprintf(stream, "VAL %d ", code_getint());
				break;
			case	kwTYPE_NUM:
				fprintf(stream, "VAL %f ", code_getreal());
				break;
			case	kwTYPE_STR:
				len = code_getstrlen();
				fprintf(stream, "\"");
				for ( j = 0; j < len; j ++ )
					fprintf(stream, "%c", prog_source[prog_ip+j]);
				fprintf(stream, "\"");
				prog_ip += len;
				break;
			case	kwIF:		case	kwFOR: 		case	kwWHILE:	case	kwREPEAT:
			case	kwELSE:		case	kwELIF:
			case	kwENDIF:	case	kwNEXT:		case	kwWEND:		case	kwUNTIL:
				fprintf(stream, "IP1 %d IP2 %d ", code_getaddr(), code_getaddr());
				break;
				}
			}

		fprintf(stream, "\n");
		} while	( prog_ip < prog_length );

	/////////////////
	// delete source
	for ( i = 0; i < lcount; i ++ )	
		tmp_free(source[i]);
	tmp_free(source);
}

/* ---------------------------------------------------------
*
*		main
*
*/

// global the filename (its needed for CTRL+C signal - delete temporary)
char	g_file[1025];

// global command-line args
char	**g_argv;
int		g_argc;

/*
*	remove temporary files
*
*	its called by atexit() only if the
*	source file it had been created in /tmp
*/
void	sbunx_remove_temp(void)
{
	unlink(g_file);
}

#if !defined(_FRANKLIN_EBM)

#if !defined(_Win32) && !defined(_WinBCB)
int	main(int argc, char *argv[])
#else
int	realmain(int argc, char *argv[])
#endif
{
	char	prev_cwd[1025];
	char	cwd[1025], *slash;
	int		i, j, use_stdin = 0, c;
	int		opt_ihavename = 0, opt_nomore = 0;
	FILE	*fp;

	strcpy(g_file, "");
	opt_graphics = 0;
	opt_quite = 0;
	opt_command[0] = '\0';

	/*
	*	command-line parameters
	*/
	for ( i = 1; i < argc; i ++ )	{
		if	( argv[i][0] == '-' )	{
			if	( opt_nomore )	{
				if	( strlen(opt_command) )
					strcat(opt_command, " ");
				strcat(opt_command, argv[i]);
				}
			else	{
				switch ( argv[i][1] )	{
				case '-':
					// the following parameters are going to script (COMMAND$)
					opt_nomore = 1;
					break;
				case 's':
					// decompile
					opt_decomp ++;
					break;
				case 'g':
					// run in graphics mode
					opt_graphics = 2;
					break;
				case 'p':
					if	( strcmp(argv[i]+1, "pkw") == 0 )	{
						printf("--- Remarks:\n");
						printf("'\n");
						printf("REM\n");
						// operators
						printf("--- Operators:\n");
						printf("() \"\"\n");
						printf("%s\n", "+ - * / \\ % ^");
						printf("%s\n", "= <= =< >= => <> != !");
						printf("%s\n", "&& & || | ~");
						for ( j = 0; opr_table[j].name[0] != '\0'; j ++ )	
							printf("%s\n", opr_table[j].name);

						// print keywords
						printf("--- Generic Keywords:\n");
						for ( j = 0; keyword_table[j].name[0] != '\0'; j ++ )	{
							if	( keyword_table[j].name[0] != '$' )
								printf("%s\n", keyword_table[j].name);
							}

						// special separators
						for ( j = 0; spopr_table[j].name[0] != '\0'; j ++ )	
							printf("%s\n", spopr_table[j].name);

						// functions
						printf("--- Functions:\n");
						for ( j = 0; func_table[j].name[0] != '\0'; j ++ )	
							printf("%s\n", func_table[j].name);

						// procedures
						printf("--- Procedures:\n");
						for ( j = 0; proc_table[j].name[0] != '\0'; j ++ )	
							printf("%s\n", proc_table[j].name);

						// external procedures
						// ....
						exit(1);
						}
					break;
				case 'm':
					// load run-time modules (linux only, shared libraries)
					#if defined(__linux__)
					opt_loadmod = 1;
					strcpy(opt_modlist, argv[i]+2);
					#else
					printf("\n\a* Modules are supported only on Linux platform\n\n");
					#endif
					break;
				case 'q':
					// shutup
					opt_quite = 1;
					break;
				case 'x':
					// setup graphics mode
					slash = &argv[i][2];
					sprintf(cwd, "SBGRAF=%s", slash);
					dev_putenv(cwd);
					break;
				case 'h':
					// print command-line parameters
					printf("SmallBASIC version %s - kw:%d, pc:%d, fc:%d, ae:%d\n", SB_STR_VER, kwNULL, (kwNULLPROC-kwCLS)+1, (kwNULLFUNC-kwASC)+1, 65536 / sizeof(var_t));
					printf("Written by Nicholas Christopoulos\n");
					printf("http://smallbasic.sourceforge.net\n\n");

					fprintf(stderr, "usage: sbasic [options] source [--] [program parameters]\n");
					fprintf(stderr, "-s		decompiler\n");
					fprintf(stderr, "-g		enable graphics\n");
					fprintf(stderr, "-m[mod1,mod2,...] load all or the specified modules\n");
					fprintf(stderr, "-q		quite\n");
					fprintf(stderr, "-pkw	prints all keywords (useful to create color-syntax macros for editors)\n");
					fprintf(stderr, "-x<width>x<height>[x<bpp>] graphics mode (only for DOSGRX, XF and Win32 driver)\n");
/*
					fprintf(stderr, "\ncharset (default: utf8)\n");
					fprintf(stderr, "-j		enable sjis\n");
					fprintf(stderr, "-b		enable big5\n");
					fprintf(stderr, "-m		enable generic multibyte\n");
					fprintf(stderr, "-u		enable unicode!\n");
*/
					return 255;
				default:
					fprintf(stderr, "unknown option: %s\n", argv[i]);
					return 255;
					};
				}
			}
		else	{
			// no - switch
			// this is the filename or script-parameters
			if	( opt_ihavename == 0 )	{
				strcpy(g_file, argv[i]);
				if	( access(g_file, F_OK) )	{
					strcat(g_file, ".bas");
					if	( access(g_file, F_OK) )		{
						perror("sbasic");
						return 255;
						}
					}
				if	( access(g_file, R_OK) )	{
					perror("sbasic");
					return 255;
					}
				opt_ihavename = 1;
				}
			else	{
				if	( strlen(opt_command) )
					strcat(opt_command, " ");
				strcat(opt_command, argv[i]);
				}
			}
		}

	/*
	*	initialization
	*/
	if	( !opt_quite )	{
		printf("SmallBASIC version %s, by Nicholas Christopoulos\n", SB_STR_VER);
		if	( strlen(g_file) )
			printf("http://smallbasic.sourceforge.net\n\n");
		else
			printf("\nPress CTRL+D to run.\n\n* READY *\n\n");
		}

	getcwd(prev_cwd, 1024);
	strcpy(cwd, prev_cwd);

	if	( strlen(g_file) == 0 )	{
		/*
		*	stdin
		*/
		use_stdin ++;
		#if defined(_Win32) || defined(_DOS)
		slash = strchr(argv[0], OS_DIRSEP);
		if	( slash )	{
			strcpy(g_file, argv[0]);
			slash = strrchr(g_file, OS_DIRSEP);
			*slash = OS_DIRSEP;
			*(slash+1) = '\0';
			strcat(g_file, "sbasic.tmp");
			}
		else
			sprintf(g_file, "sbasic.tmp");
		#else
		sprintf(g_file, "%ctmp%csb%d.bas", OS_DIRSEP, OS_DIRSEP, getpid());
		#endif

		// its a temporary and it must be deleted
		atexit(sbunx_remove_temp);

		// get it from stdin
		fp = fopen(g_file, "wb");
		if	( fp )	{
			while ( (c = fgetc(stdin)) != -1 )	
				fputc(c, fp);
			fclose(fp);
			}
		else
			perror("sbasic");
		}
	else	{
		/*
		*	file
		*/
		if	( g_file[0] == OS_DIRSEP )
			cwd[0] = '\0';
		#if defined(_Win32) || defined(_DOS)
		if	( strlen(g_file) > 2 )	{
			if ( g_file[1] == ':' )	
				cwd[0] = '\0';
			}
		#endif
		slash = strrchr(g_file, OS_DIRSEP);
		if	( slash )	{
			char	tmp[1024];
			char	sep[2];
			*slash = '\0';
			
			sep[0] = OS_DIRSEP;
			sep[1] = '\0';
			strcat(cwd, sep);
			strcat(cwd, g_file);
			strcpy(tmp, slash+1);
			strcpy(g_file, tmp);

//			printf("Current directory changed to [%s]\n", cwd);
//			printf("Source file changed to: [%s]\n", file);
			chdir(cwd);
			}
		}

	/*
	*	run it
	*/
	memmgr_init();
	if	( opt_decomp )	{
		// decompile
		brun_load(g_file, 0);
		sblmgr_close();
		if	( bc_get_error() )
			return 1;
		memmgr_setabort(1);
		dump_bytecode(g_file, stdout);
		}
	else	{
		// run
		brun(g_file, 0);
		}

	/*
	*	cleanup
	*/
	if	( use_stdin )
		remove(g_file);
	chdir(prev_cwd);

	if	( prog_error || scan_error )
		return 1;
	return opt_retval;
}
#endif

#endif	// EBM

