/*
*	Generic Platform Driver
*
*	2001-02-14: ndc, created
*/

#if !defined(_device_h)
#define _device_h

#if defined(__cplusplus)
extern "C" {
#endif

#include "sys.h"
#include "keys.h"		// Keyboard codes
#if !defined(SCAN_MODULE)
#include "var.h"
#endif

#if defined(_UnixOS)
#include <termios.h>
#endif

typedef struct {
	double	x, y;
	} pt_t;

typedef struct {
	int32	x, y;
	} ipt_t;

/* for low-level graphics drivers, like term and SDL */
void	g_line(int x1,int y1,int x2, int y2, void (*dotproc)(int,int));

/* ===========================================================================================================================
*
*	colors - VGA16 compatible
*
*	There are 16 colors. The second set of 8 colors its the high-light of the first set.
*	Example:
*		dark gray its CLR_BLACK+8
*		light green its CLR_GREEN+8
*		yellow its CLR_BROWN+8
*/
#define	CLR_BLACK	0
#define	CLR_BLUE	1
#define	CLR_GREEN	2
#define	CLR_CYAN	3
#define	CLR_RED		4
#define	CLR_MAGENTA	5
#define	CLR_BROWN	6
#define	CLR_GRAY	7

#define	CLR_WHITE	15

/*
*	OS Global Information Variables
*
*	Globals are needed for speed (Unix is fast, but PalmOS is not)
*/
enum os_charset_codes { 
	enc_utf8,					// 8bit - All European languages - default
	enc_sjis,					// Japanese characters support
	enc_big5,					// Chinese characters support
	enc_gmb,					// Generic multibyte
	enc_unicode					// Unicode
	};
#if !defined(DEVICE_MODULE)
extern dword	os_ver;									// OS version (ex: 0x330 for PalmOS 3.3)
extern byte		os_charset;

#if defined(SONY_CLIE)
extern byte     use_sony_clie;                          // true if the program runs under a Sony CLIE
extern UInt16	sony_refHR;                             // Sony's HiRes lib ref
#endif

extern byte		os_color;								// true if the output has real colors (256+ colors)
extern dword	os_color_depth;							// the number of bits of the supported colors
														// (ex: 8 for 256 colors, 15 or 16 for 64K, 24 or 32 for 1.6M)
extern byte		os_graphics;							// non-zero if the driver supports graphics
extern int		os_graf_mx;								// graphic mode: maximum x 
extern int		os_graf_my;								// graphic mode: maximum y

// graphics - viewport
extern int		dev_Vx1;
extern int		dev_Vy1;
extern int		dev_Vx2;
extern int		dev_Vy2;

extern int		dev_Vdx;
extern int		dev_Vdy;

// graphics - window world coordinates
extern int		dev_Wx1;
extern int		dev_Wy1;
extern int		dev_Wx2;
extern int		dev_Wy2;

extern int		dev_Wdx;
extern int		dev_Wdy;

// graphics - current colors
extern long		dev_fgcolor;
extern long		dev_bgcolor;

// cache
extern word	os_cclabs1;
extern word	os_ccpass2;
#endif

/* ===========================================================================================================================
*
*	Driver basics
*
*/

int		dev_init(int mode, int flags)	SEC(BIO);		// driver initialization
int		dev_restore(void) 				SEC(BIO);		// driver (close the driver & restore computer mode)

/* ===========================================================================================================================
*
*	System basics
*
*/
void	dev_pushkey(word ch)			SEC(BIO);		// Internal keyboard buffer: store a key into the buffer
int		dev_kbhit(void)					SEC(BIO);		// Internal keyboard buffer: returns true if there is a key
void	dev_clrkb(void)					SEC(BIO);		// Internal keyboard buffer: clear

int		dev_run(const char *prog) 		SEC(BIO);		// system()

////////////////////
//	Check's the system events queue
//
//	a) stores keys into the keyboard buffer
//	b) store pen/mouse events
//	c) any other supported device
//
//	Returns  0 if there is not new events in the queue
//	Returns >0 the number of the new events
//	Returns -1 that means BREAK (brun_break() it was called)
//	Returns -2 that means BREAK (executor displays "BREAK" message)
//
//	if dead_loop is zero, the dev_events() will checks the events and then will return immediatly.
//	if dead_loop is true, the dev_events() will wait for a new event.
//
int		dev_events(int dead_loop);
void	dev_delay(dword ms);

////////////////////
// Mouse & lightpen!
// Since 1988 the mouse was an new device for PC's, there is no mouse support on QB.
// We shall use the PEN(x) to support the mouse, and we must maintain the Palm compatibility.
//
// PalmOS PEN, Lightpen & Mouse API
// ================================
// PEN(0) -> true (non zero) if there is a new pen or mouse event
// PEN(1) -> PEN: last pen-down x; MOUSE: last mouse button down x 
// PEN(2) -> PEN: last pen-down y; MOUSE: last mouse button down y
// PEN(3) -> QB compatiblity, don't use it
// PEN(4) -> PEN: last/current x, MOUSE: the current x position only if the left mouse button is pressed (like PEN is down)
// PEN(5) -> PEN: last/current y, MOUSE: the current y position only if the left mouse button is pressed (like PEN is down)
// PEN(6) -> QB compatiblity, don't use it
// PEN(7) -> QB compatiblity, don't use it
// PEN(8) -> QB compatiblity, don't use it
// PEN(9) -> QB compatiblity, don't use it
// Mouse buttons:
// PEN(10) -> current mouse x position
// PEN(11) -> current mouse y position
// PEN(12) -> true if the left mouse button is pressed
// PEN(13) -> true if the right mouse button is pressed
// PEN(14) -> true if the middle mouse button is pressed
//

void	dev_setpenmode(int enable)			SEC(BIO);	// enables or disables pen/mouse driver (PEN ON|OFF)
int		dev_getpen(int code)				SEC(BIO);	// returns the pen/mouse status (PEN(code))

/* ===========================================================================================================================
*
*	terminal input/output
*
*/
void	dev_print(const char *str)			SEC(BIO); 	// write a string to the console
void	dev_printf(const char *fmt, ...);				// write a formated string to the console

void	dev_cls(void)						SEC(BIO);	// clear the screen
void	dev_clreol(void)					SEC(BIO);	// clear the console line from the current pos. until the end of the line
void	dev_setxy(int x, int y)				SEC(BIO);	// set cursor position. parameters in pixels.
int		dev_getx(void)						SEC(BIO);	// returns the current cursor x position (in pixels)
int		dev_gety(void)						SEC(BIO);	// returns the current cursor y position (in pixels)
void	dev_settextcolor(long fg, long bg)	SEC(BIO); 	// set the text colors. (fg = foreground, bg = background)

int		dev_getch(void)						SEC(BIO);	// waits until a key is pressed and returns its code
char	*dev_gets(char *buf, int size)		SEC(BIO);	// the real INPUT command. Gets a string from the console

/* ===========================================================================================================================
*
*	Graphics
*
*/
void	dev_viewport(int x1, int y1, int x2, int y2)	SEC(BIO);
void	dev_window(int x1, int y1, int x2, int y2)	SEC(BIO);

int		dev_textwidth(const char *str)		SEC(BIO);
int		dev_textheight(const char *str)		SEC(BIO);

void	dev_setcolor(long color)			SEC(BIO);
void	dev_setpixel(int x, int y)			SEC(BIO);
long	dev_getpixel(int x, int y)			SEC(BIO);
void	dev_clipline(int *x1, int *y1, int *x2, int *y2, int *visible) SEC(BIO);
void	dev_line(int x1, int y1, int x2, int y2)	SEC(BIO);
void	dev_rect(int x1, int y1, int x2, int y2, int fill)	SEC(BIO);
void	dev_ellipse(int xc, int yc, int xr, int yr, double aspect, int fill)	SEC(BIO);
void	dev_arc(int xc, int yc, double r, double as, double ae, double aspect)	SEC(BIO);

//// floodfill
void	dev_ffill(word, word, long, long)				SEC(BIO);
//// polyfill
void	dev_pfill(ipt_t *pts, int ptNum)                  SEC(BIO);

/* ===========================================================================================================================
*
*	Sound
*
*/

void	dev_beep(void)						SEC(BIO); 	// just a BEEP! :)
void	dev_sound(int freq, int dur_ms, int vol_prc, int bgplay) SEC(BIO);	// note: duration in ms, volume in percent, bgplay = play on background

/* ===========================================================================================================================
*
*	FILE SYSTEM BASICS
*
*/

#define	VFS_ATTR_FILE		1
#define	VFS_ATTR_DIR		2
#define	VFS_ATTR_LINK		4

int		wc_match(const char *mask, char *name)				SEC(BIO);

int		dev_initfs(void)									SEC(BIO);
void	dev_closefs(void)									SEC(BIO);

int		dev_fexists(const char *file)						SEC(BIO);
int		dev_fcopy(const char *file, const char *newfile)	SEC(BIO);
int		dev_frename(const char *file, const char *newname)	SEC(BIO);
int		dev_fremove(const char *file)						SEC(BIO);

int		dev_faccess(const char *file)						SEC(BIO);
int		dev_fattr(const char *file)							SEC(BIO);

/* ===========================================================================================================================
*
*	FILE I/O
*
*/

// file types
typedef enum { 
    ft_stream, 
    ft_random, 

    ft_serial_port,   	// COMx:
    ft_printer_port,  	// LPTx:
    ft_irda_port,		// IRDx:

    ft_socket_client,           // SCLT:address:port
    ft_socket_server,           // SSVR:address:port
    ft_ftp_client,              // FTP is a good example also
    ft_mail_client,             // MAIL (at least SMTP)
    ft_memo, 
    ft_pdoc, 
    ft_ztxt 
    } dev_ftype_t;

// file_t structure
typedef struct {
	char	name[OS_PATHNAME_SIZE+1];
	dev_ftype_t	type;
	int		port;
	long	devspeed;
    byte    *drv_data;      // driver data (don't forget to set it to null on close())
    dword   drv_dw[4];      // driver data

#if defined(_UnixOS)
	struct termios oldtio,newtio;
#endif

#if defined(_PalmOS)
	UInt16	libHandle;
//	DmOpenRef	ref;
	FileHand	handle;
	Err			last_error;
#elif defined(_VTOS)
	FILE		*handle;
	int			last_error;
#else
	int			handle;
	int			last_error;
#endif
	int			open_flags;
	} dev_file_t;

// flags for dev_fopen()
#define	DEV_FILE_INPUT		1
#define DEV_FILE_OUTPUT		2	// create
#define DEV_FILE_APPEND		4	// (if its not exists then create) & open for append
#define	DEV_FILE_EXCL		8	// open exclusive 

int		dev_freefilehandle(void)								SEC(BIO);
dev_file_t*	dev_getfileptr(int handle)						SEC(BIO);
int		dev_fstatus(int handle)								SEC(BIO);

int			dev_fopen(int SBHandle, const char *name, int flags)	SEC(BIO);
dword		dev_flength(int SBHandle)								SEC(BIO);
int			dev_fclose(int SBHandle)								SEC(BIO);
dword		dev_fseek(int SBHandle, dword offset)					SEC(BIO);
dword		dev_ftell(int SBHandle)									SEC(BIO);
int			dev_feof(int SBHandle)									SEC(BIO);
int			dev_fwrite(int SBHandle, byte *buff, dword size)		SEC(BIO);
int			dev_fread(int SBHandle, byte *buff, dword size)			SEC(BIO);
void        dev_mkdir(const char *dir)                              SEC(BIO);
void        dev_rmdir(const char *dir)                              SEC(BIO);
void        dev_chdir(const char *dir)                              SEC(BIO);
char        *dev_getcwd(void)                                        SEC(BIO);

char		*dev_getenv(const char *)								SEC(BIO);
int			dev_putenv(const char *)								SEC(BIO);

char_p_t    *dev_create_file_list(const char *wc, int *count)       SEC(BIO);
void    	dev_destroy_file_list(char_p_t *list, int count)          SEC(BIO);

#if defined(__cplusplus)
}
#endif

#endif
