/*
*	SmallBASIC module manager
*
*	2001/12/07, Nicholas Christopoulos
*
*	This program is distributed under the terms of the GPL v2.0 or later
*	Download the GNU Public License (GPL) from www.gnu.org
*/

#include "sys.h"
#include "kw.h"
#include "extlib.h"
#include "pproc.h"

#if defined(__linux__)
#define	LNX_EXTLIB
#endif

#if defined(LNX_EXTLIB)
#include <dlfcn.h>
#include <dirent.h>
#endif

int		bc_add_extproc(const char *proc_name, int lib_id)	SEC(BCSCAN);
int		bc_add_extfunc(const char *func_name, int lib_id)	SEC(BCSCAN);
void	bc_reset_extlist(void)		SEC(BCSCAN);

extern int		prog_error;			// The last RTL error code (its work as flag)
extern addr_t	prog_length;		// The byte-code length (program length in bytes)
extern addr_t	prog_ip;			// The instruction pointer
extern byte		*prog_source;		// The byte-code
extern byte 	opt_quite;			// CLI option

#if defined(OS_LIMITED)
#define	MAX_SLIB_N	16
#else
#define	MAX_SLIB_N	256
#endif

#if defined(LNX_EXTLIB)
static slib_t	slib_table[MAX_SLIB_N];
static int		slib_count;
#endif

/*
*	retrieve the function pointer
*/
void    *slib_getoptptr(slib_t *lib, const char *name)
{
#if defined(LNX_EXTLIB)
	return	dlsym(lib->handle, name);
#else
	return NULL;
#endif
}

/*
*	retrieve the function pointer; error if its not exists
*/
void    *slib_getptr(slib_t *lib, const char *name)
{
#if defined(LNX_EXTLIB)
	void    *ptr;

	ptr = slib_getoptptr(lib->handle, name);
	if   ( !ptr )
		panic("SB-LibMgr: %s, missing function %s\n", lib->name, name);
	return ptr;
#else
	return NULL;
#endif
}

/*
*	open library (basic open)
*/
int		slib_llopen(slib_t *lib)
{
#if defined(LNX_EXTLIB)
	lib->handle = dlopen(lib->fullname, RTLD_NOW);
	if	( lib->handle == NULL )
		panic("SB-LibMgr: error on loading %s\n%s", lib->name, dlerror());
	return (lib->handle != NULL);
#else
	return 0;
#endif
}

/*
*	close library (basic close)
*/
int		slib_llclose(slib_t *lib)
{
#if defined(LNX_EXTLIB)
	if	( !lib->handle )
		return 0;

	dlclose(lib->handle);
	lib->handle = NULL;
	return 1;
#else
	return 1;
#endif
}

/*
*/
void	slib_import_routines(slib_t *lib)
{
#if defined(LNX_EXTLIB)
	int		i, count;
	char	buf[33];
	int		(*fgetname)(int,char*);
	int		(*fcount)(void);

	fcount   = slib_getoptptr(lib, "sblib_proc_count");
	fgetname = slib_getoptptr(lib, "sblib_proc_getname");
	if	( fcount )	{
		count = fcount();
		for ( i = 0; i < count; i ++ )	{ 
			if	( fgetname(i, buf) )	
				bc_add_extproc(buf, lib->id);
			}
		}

	fcount   = slib_getoptptr(lib, "sblib_func_count");
	fgetname = slib_getoptptr(lib, "sblib_func_getname");
	if	( fcount )	{
		count = fcount();
		for ( i = 0; i < count; i ++ )	{ 
			if	( fgetname(i, buf) )
				bc_add_extfunc(buf, lib->id);
			}
		}
#endif
}

/*
*	load a lib
*/
void	slib_import(const char *name, const char *fullname)
{
#if defined(LNX_EXTLIB)
	slib_t	*lib;
	int		(*minit)(void);
	int		(*mtype)(void);
	void	(*mdrvname)(char*);
	int		mok = 0;
	
	lib = &slib_table[slib_count];
	memset(lib, 0, sizeof(slib_t));
	strncpy(lib->name, name, 255);
	strncpy(lib->fullname, fullname, 1023);
	lib->id = slib_count;

	if	( !opt_quite )
		printf("SB-LibMgr: importing %s", fullname);

	if	( slib_llopen(lib) )	{
		mok = 1;

		// init
		minit = slib_getoptptr(lib, "sblib_init");
		if	( minit )	{
			if	( !minit() )	{
				mok = 0;
				panic("SB-LibMgr: %s->sblib_init(), failed", lib->name);
				}
			}
		
		// get type
		mtype = slib_getoptptr(lib, "sblib_type");
		if	( mtype )
			lib->type = mtype();
		else
			lib->type = lib_lang_ext;	// default type

		// get info
		switch ( lib->type )	{
		case	lib_lang_ext:
			slib_import_routines(lib);
			mok = 1;
			break;
		case	lib_vfs_driver:
			mdrvname = slib_getptr(lib, "sblib_vfsname");
			memset(lib->vfs_drvname, 0, 5);
			mdrvname(lib->vfs_drvname);
			mok = 1;
			break;
		default:
			panic("SB-LibMgr: %s->sbmod_type(), type %d is not supported", lib->type);
			mok = 0;
			};
		}

	if	( mok )	{
		slib_count ++;
		if	( !opt_quite )
			printf("... done\n");
		}
	else	{
		if	( !opt_quite )
			printf("... error\n");
		}
#endif
}

/*
*	scan libraries
*/
void	sblmgr_scanlibs(const char *path)
{
#if defined(LNX_EXTLIB)
	DIR				*dp;
	struct dirent	*e;
	char			*name, *p;
	char			full[1024], libname[256];

	if	( (dp = opendir(path)) == NULL )
		return;

	while ( (e = readdir(dp)) != NULL )	{
		name = e->d_name;
		if	( (strcmp(name, ".") == 0) || (strcmp(name, "..") == 0) )
			continue;
		if	( (p = strstr(name, ".so")) != NULL )	{
			if	( strcmp(p, ".so") == 0 )	{
				// store it
				
				strcpy(libname, name);
				p = strchr(libname, '.');
				*p = '\0';

				strcpy(full, path);
				strcat(full, name);

				slib_import(libname, full);
				}
			}
		}

	closedir(dp);
#endif
}

/*
*	slib-manager: initialize manager
*/
void	sblmgr_init(int mcount, const char *mlist)
{
#if defined(LNX_EXTLIB)
	int		all=0;

	slib_count = 0;
	bc_reset_extlist();

	if	( !opt_quite && mcount )
		printf("SB-LibMgr: scanning for modules...\n");
	if	( mcount )	{
		if	( mlist )	{
			if	( strlen(mlist) == 0 )
				all = 1;
			else
				all = 1;
			// TODO: else load the specified modules
			}
		else
			all = 1;

		if	( all )	{
			sblmgr_scanlibs("/usr/lib/sbasic/modules/");
			sblmgr_scanlibs("/usr/local/lib/sbasic/modules/");
			}
		}
	if	( !opt_quite )
		printf("\n");
#endif
}

/*
*	slib-manager: close everything
*/
void	sblmgr_close()
{
#if defined(LNX_EXTLIB)
	int		i;
	slib_t	*lib;
	void	(*mclose)(void);

	for ( i = 0; i < slib_count; i ++ )	{
		lib = &slib_table[i];
		if	( lib->handle )	{
			mclose = slib_getoptptr(lib, "sblib_init");
			if	( mclose )	
				mclose();
			slib_llclose(lib);
			}
		}

	bc_reset_extlist();
#endif
}

/*
*	returns the 'index' function-name of the 'lib'
*/
int		sblmgr_getfuncname(int lib_id, int index, char *buf)
{
#if defined(LNX_EXTLIB)
	slib_t	*lib;
	int		(*mgf)(int,char*);

	buf[0] = '\0';
	if	( lib_id < 0 || lib_id >= slib_count )	
		return 0;	// error

	lib = &slib_table[lib_id];
	mgf = slib_getoptptr(lib, "sblib_func_getname");
	if	( mgf == NULL )
		return 0;	// error

	return mgf(index, buf);
#else
	return 0;
#endif
}

/*
*	returns the 'index' procedure-name of the 'lib'
*/
int		sblmgr_getprocname(int lib_id, int index, char *buf)
{
#if defined(LNX_EXTLIB)
	slib_t	*lib;
	int		(*mgp)(int,char*);

	buf[0] = '\0';
	if	( lib_id < 0 || lib_id >= slib_count )	
		return 0;	// error

	lib = &slib_table[lib_id];
	mgp = slib_getoptptr(lib, "sblib_proc_getname");
	if	( mgp == NULL )
		return 0;	// error

	return mgp(index, buf);
#else
	return 0;
#endif
}

/*
*	build parameter table
*/
int		slib_build_ptable(slib_par_t *ptable)
{
#if defined(LNX_EXTLIB)
	int			pcount = 0;
	var_t		*arg = NULL;
	byte		ready, code;
	addr_t		ofs;

	if	( code_peek() == kwTYPE_LEVEL_BEGIN )	{
		code_skipnext();		// kwTYPE_LEVEL_BEGIN

		ready = 0;
		do	{
			code = code_peek();
			switch ( code )	{
			case	kwTYPE_EOC:
				code_skipnext();
				break;
			case	kwTYPE_SEP:			// separator 
				code_skipsep();
				break;
			case	kwTYPE_LEVEL_END:	// ) -- end of parameters
				ready = 1;
				break;
			case	kwTYPE_VAR:		// variable
				ofs = prog_ip;		// store IP

				if	( code_isvar() )	{
					// push parameter
					ptable[pcount].var_p = code_getvarptr();
					ptable[pcount].byref = 1;
					pcount ++;
					break;
					}

				prog_ip = ofs;	// restore IP
				// no 'break' here
			default:
				// default --- expression (BYVAL ONLY)
				arg = v_new();
				eval(arg);
				if	( !prog_error )	{
					// push parameter
					ptable[pcount].var_p = arg;
					ptable[pcount].byref = 0;
					pcount ++;
					}
				else	{
					v_free(arg);
					tmp_free(arg);
					return pcount;
					}

				}

			} while ( !ready );

		code_skipnext();		// kwTYPE_LEVEL_END
		}

	return pcount;
#else
	return 0;
#endif
}

/*
*	free parameter table
*/
void	slib_free_ptable(slib_par_t *ptable, int pcount)
{
#if defined(LNX_EXTLIB)
	int	i;

	for ( i = 0; i < pcount; i ++ )	{
		if	( ptable[i].byref == 0 )	{
			v_free(ptable[i].var_p);
			tmp_free(ptable[i].var_p);
			}
		}
#endif
}

/*
*	execute a procedure
*/
int		sblmgr_procexec(int lib_id, int index)
{
#if defined(LNX_EXTLIB)
	slib_t	*lib;
	var_t	ret;
	slib_par_t *ptable = NULL;
	int		(*pexec)(int, int, slib_par_t *, var_t *);
	int		pcount = 0;
	int		success = 0;

	if	( lib_id < 0 || lib_id >= slib_count )	{
		// rt_raise(...)
		return 0;	// error
		}

	lib = &slib_table[lib_id];
	pexec = slib_getoptptr(lib, "sblib_proc_exec");
	if	( pexec == NULL )	{
		// rt_raise(...)
		return 0;	// error
		}

	//	build parameter table
	ptable = tmp_alloc(sizeof(slib_par_t) * 64);	// 64 = maximum parameter
	pcount = slib_build_ptable(ptable);
	if	( prog_error )	{
		slib_free_ptable(ptable, pcount);
		return 0;
		}

	// exec
	v_init(&ret);
	success = pexec(index, pcount, ptable, &ret);
	
	// error
	if	( !success )	{
		if	( ret.type == V_STR )
			rt_raise("SB-LibMgr:\n%s: %s\n", lib->name, ret.v.p.ptr);
		else
			rt_raise("SB-LibMgr:\n%s: not specified error\n", lib->name);
		}

	//	clean-up
	if	( ptable )	{
		slib_free_ptable(ptable, pcount);
		tmp_free(ptable);
		}

	v_free(&ret);

	return success;
#else
	return 0;
#endif
}

/*
*	execute a function
*/
int		sblmgr_funcexec(int lib_id, int index, var_t *ret)
{
#if defined(LNX_EXTLIB)
	slib_t	*lib;
	slib_par_t *ptable = NULL;
	int		(*pexec)(int, int, slib_par_t *, var_t *);
	int		pcount = 0;
	int		success = 0;

	if	( lib_id < 0 || lib_id >= slib_count )	{
		// rt_raise(...)
		return 0;	// error
		}

	lib = &slib_table[lib_id];
	pexec = slib_getoptptr(lib, "sblib_func_exec");
	if	( pexec == NULL )	{
		// rt_raise(...)
		return 0;	// error
		}

	//	build parameter table
	ptable = tmp_alloc(sizeof(slib_par_t) * 64);	// 64 = maximum parameter
	pcount = slib_build_ptable(ptable);
	if	( prog_error )	{
		slib_free_ptable(ptable, pcount);
		return 0;
		}

	// exec
	v_init(ret);
	success = pexec(index, pcount, ptable, ret);
	
	// error
	if	( !success )	{
		if	( ret->type == V_STR )
			rt_raise("SB-LibMgr:\n%s: %s\n", lib->name, ret->v.p.ptr);
		else
			rt_raise("SB-LibMgr:\n%s: (error not specified)\n", lib->name);
		}

	//	clean-up
	if	( ptable )	{
		slib_free_ptable(ptable, pcount);
		tmp_free(ptable);
		}

	return success;
#else
	return 0;
#endif
}

