/*
*	streams (normal files), driver
*
*	Nicholas Christopoulos
*
*	This program is distributed under the terms of the GPL v2.0 or later
*	Download the GNU Public License (GPL) from www.gnu.org
*/

#include "sys.h"
#include "device.h"
#include "pproc.h"
#include "match.h"
#if defined(_PalmOS)
	#include <FileStream.h>
#else
	#include <errno.h>

	#if defined(_UnixOS)
	#include <sys/time.h>
	#include <unistd.h>
	#endif
        #if defined(_VTOS)
        typedef FILE *FileHand;
        #else
	#include <dirent.h>		// POSIX standard (Note: Borland C++ compiler supports it; VC no)

	typedef int FileHand;
        #endif
#endif
#include "fs_stream.h"

/*
*	open a file
*/
int		stream_open(dev_file_t *f)
{
	#if defined(_PalmOS)
	///////////////////////////////////////////////////////////////////////////////////////////
	//	PalmOS
	int		flags;
	dword	osflags;

	flags = f->open_flags;
	osflags = fileModeAnyTypeCreator;
	
	if	( flags == DEV_FILE_OUTPUT )	{
		if	( DmFindDatabase(0, (char*) f->name) != 0 )
			FileDelete(0, (char*) f->name);
		}

	if	( flags & DEV_FILE_APPEND )	
		osflags |= fileModeAppend;
	else if	( flags & DEV_FILE_OUTPUT )	
	 	osflags |= fileModeReadWrite;
	else if	( flags & DEV_FILE_INPUT )	
	 	osflags |= fileModeReadOnly;
	else
	 	osflags |= (fileModeReadWrite | fileModeDontOverwrite);

	if	( flags & DEV_FILE_EXCL )
		osflags |= fileModeExclusive;

	f->handle = FileOpen(0, (char *) f->name, ID_UFST, ID_SmBa, osflags, &f->last_error);
	if	( f->last_error )
		err_file(f->last_error);

	return (f->last_error == 0);
	#elif defined(_VTOS)
	///////////////////////////////////////////////////////////////////////////////////////////
	//	VTOS
	char *mode;
	if (f->open_flags&DEV_FILE_OUTPUT)
	  unlink(f->name);
	if (f->open_flags&DEV_FILE_APPEND)
	  mode = "w+t";
	else if (f->open_flags & DEV_FILE_OUTPUT)
	  mode = "wt";
	else
	  mode = "rt";
	f->handle = my_fopen(f->name, mode);
	if      ( f->handle==NULL ) {
	  f->last_error = -1;
	  f->handle = (FILE*)-1;
	  err_file(f->last_error);
	}
	return (f->last_error == 0);
	#else
	///////////////////////////////////////////////////////////////////////////////////////////
	//	C
	int		osflags, osshare, flags = f->open_flags;

	#if defined(_Win32)
	osflags = O_BINARY;
	#else
	osflags = 0;
	#endif

	if	( flags == DEV_FILE_OUTPUT )	
		remove(f->name);

	if	( flags & DEV_FILE_EXCL )
		osshare = 0;
	else
		osshare = S_IREAD|S_IWRITE;	// default is 'shared'

	osflags |= O_RDWR;				// there is no reason to use O_RDONLY or O_WRONLY
									// (warning: bcc changes the file attributes on O_RDONLY)
	if	( flags & DEV_FILE_OUTPUT )
		osflags |= O_CREAT;
	if	( flags & DEV_FILE_APPEND )	
		osflags |= O_APPEND;

	f->handle = open(f->name, osflags, osshare);
	if	( f->handle < 0 )	
		err_file((f->last_error = errno));
	return (f->handle >= 0);
	#endif
}

/*
*	close the stream
*/
int		stream_close(dev_file_t *f)
{
	#if defined(_PalmOS)
	f->last_error = FileClose(f->handle);
	f->handle = (FileHand) -1;
	if	( f->last_error )
		err_file(f->last_error);
	return (f->last_error == 0);
	#elif defined(_VTOS)
	fclose(f->handle);
	f->handle = (FILE*)-1;
	return 1;
	#else
	int r;

	r = close(f->handle);
	f->handle = -1;
	if	( r )
		err_file((f->last_error = errno));
	return (r == 0);
	#endif
}

/*
*/
int		stream_write(dev_file_t *f, byte *data, dword size)
{
	#if defined(_PalmOS)
	FileWrite(f->handle, data, size, 1, &f->last_error);
	if	( f->last_error )
		err_file(f->last_error);
	return (f->last_error == 0);
	#elif defined(_VTOS)
	int		r;

	r = fwrite(data, 1, size, f->handle);
	if ( r != (int) size )
		err_file((f->last_error = errno));
	return (r == (int) size);
	#else
	int		r;
	
	r = write(f->handle, data, size);
	if	( r != (int) size )
		err_file((f->last_error = errno));
	return (r == (int) size);
	#endif
}

/*
*/
int		stream_read(dev_file_t *f, byte *data, dword size)
{
	#if defined(_PalmOS)
	FileRead(f->handle, data, size, 1, &f->last_error);
	if	( f->last_error )	{
		if	( f->last_error == fileErrEOF )
			f->last_error = 0;
		else
			err_file(f->last_error);
		}
	return (f->last_error == 0);
	#elif defined(_VTOS)
	int r;
	r=fread(data, 1, size, f->handle);
	if      ( r != (int) size )
	  err_file((f->last_error = -1));
	return (r == (int) size);
	#else
	int		r;

	r = read(f->handle, data, size);
	if	( r != (int) size )
		err_file((f->last_error = errno));
	return (r == (int) size);
	#endif
}

/*
*	returns the current position
*/
dword	stream_tell(dev_file_t *f)
{
	#if defined(_PalmOS)
	return FileTell(f->handle, NULL, &f->last_error);
	#elif defined(_VTOS)
	return ftell(f->handle);
	#else
	return lseek(f->handle, 0, SEEK_CUR);
	#endif
}

/*
*	returns the file-length
*/
dword	stream_length(dev_file_t *f)
{
	#if defined(_PalmOS)
	dword	fsize;

	FileTell(f->handle, &fsize, &f->last_error);
	if	( f->last_error )	{
		fsize = (dword) -1;
		err_file(f->last_error);
		}
	return fsize;
	#elif defined (_VTOS)
	return fsize(f->handle);
	#else
	long	pos, endpos;

	pos = lseek(f->handle, 0, SEEK_CUR);
	if	( pos != -1 )	{
		endpos = lseek(f->handle, 0, SEEK_END);
		lseek(f->handle, pos, SEEK_SET);
		return endpos;
		}
	else
		err_file((f->last_error = errno));
	return 0;
	#endif
}

/*
*/
dword	stream_seek(dev_file_t *f, dword offset)
{
	#if defined(_PalmOS)
	dword	pos;

	f->last_error = FileSeek(f->handle, offset, fileOriginBeginning);
	pos = FileTell(f->handle, NULL, &f->last_error);
	if	( f->last_error )	{
		err_file(f->last_error);
		pos = (dword) -1;
		}
	return pos;
	#elif defined(_VTOS)
	return fseek(f->handle, offset, SEEK_SET);
	#else 
	return lseek(f->handle, offset, SEEK_SET);
	#endif
}

/*
*/
int		stream_eof(dev_file_t *f)
{
	#if defined(_PalmOS)
	dword	fsize, curpos;

	curpos = FileTell(f->handle, &fsize, &f->last_error);
	if	( f->last_error )	{
		err_file(f->last_error);
		return 1;
		}
	return (curpos == fsize);
	#elif defined(_VTOS)
    return feof(f->handle);
	#else
	long	pos, endpos;

	pos = lseek(f->handle, 0, SEEK_CUR);
	if	( pos != -1 )	{
		endpos = lseek(f->handle, 0, SEEK_END);
		lseek(f->handle, pos, SEEK_SET);
		return (pos == endpos);
		}
	else	
		err_file((f->last_error = errno));
	return 1;
	#endif
}


