#include <system.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "../stdlib/my_stdio.h"
#include "glk.h"
#include "cheapglk.h"
#include "getfile.h"
#include "heliomain.h"
#include "glkiface.h"

/* This file implements filerefs as they work in a stdio system: a
    fileref contains a pathname, a text/binary flag, and a file
    type.
*/

/* Linked list of all filerefs */
static fileref_t *gli_filereflist = NULL; 

static char workingdir[256] = ".";

fileref_t *gli_new_fileref(char *filename, glui32 usage, glui32 rock, char *type)
{
    fileref_t *fref = (fileref_t *)qmalloc(sizeof(fileref_t));
    if (!fref)
        return NULL;
    
    fref->magicnum = MAGIC_FILEREF_NUM;
    fref->rock = rock;
    
    fref->filename = qmalloc(1 + strlen(filename));
    strcpy(fref->filename, filename);
    
    fref->textmode = ((usage & fileusage_TextMode) != 0);
    fref->filetype = (usage & fileusage_TypeMask);
    strcpy(fref->type, type);

    fref->prev = NULL;
    fref->next = gli_filereflist;
    gli_filereflist = fref;
    if (fref->next) {
        fref->next->prev = fref;
    }
    
    if (gli_register_obj)
        fref->disprock = (*gli_register_obj)(fref, gidisp_Class_Fileref);
    else
        fref->disprock.ptr = NULL;

    return fref;
}

void gli_delete_fileref(fileref_t *fref)
{
    fileref_t *prev, *next;
    
    if (gli_unregister_obj) {
        (*gli_unregister_obj)(fref, gidisp_Class_Fileref, fref->disprock);
        fref->disprock.ptr = NULL;
    }
        
    fref->magicnum = 0;
    
    if (fref->filename) {
        qfree(fref->filename);
        fref->filename = NULL;
    }
    
    prev = fref->prev;
    next = fref->next;
    fref->prev = NULL;
    fref->next = NULL;

    if (prev)
        prev->next = next;
    else
        gli_filereflist = next;
    if (next)
        next->prev = prev;
    
    qfree(fref);
}

void glk_fileref_destroy(fileref_t *fref)
{
    if (!fref) {
        gli_strict_warning("fileref_destroy: invalid ref");
        return;
    }
    gli_delete_fileref(fref);
}

frefid_t glk_fileref_create_temp(glui32 usage, glui32 rock)
{
    char *filename;
    fileref_t *fref;
    static int tmpcnt = 1000;

    /* This is a pretty good way to do this on Unix systems. On Macs,
        it's pretty bad, but this library won't be used much on the Mac 
        -- I hope. I have no idea about the DOS/Windows world. */
    filename = qmalloc(32);
    sprintf(filename,"glktemp%d",tmpcnt++);
    //    filename = tmpnam(NULL);
    
    fref = gli_new_fileref(filename, usage, rock, "GLK Temporary");
    if (!fref) {
        gli_strict_warning("fileref_create_temp: unable to create fileref.");
        return NULL;
    }
    
    return fref;
}

frefid_t glk_fileref_create_from_fileref(glui32 usage, frefid_t oldfref,
    glui32 rock)
{
    fileref_t *fref; 

    if (!oldfref) {
        gli_strict_warning("fileref_create_from_fileref: invalid ref");
        return NULL;
    }

    fref = gli_new_fileref(oldfref->filename, usage, rock, oldfref->type);
    if (!fref) {
        gli_strict_warning("fileref_create_from_fileref: unable to create fileref.");
        return NULL;
    }
    
    return fref;
}

frefid_t glk_fileref_create_by_name(glui32 usage, char *name,
    glui32 rock)
{
    fileref_t *fref;
    char buf[256];
    char buf2[256];
    int len;
    char *cx, *type;

    len = strlen(name);
    if (len > 255)
        len = 255;
    
    /* Take out all '/' characters, and make sure the length is greater 
        than zero. Again, this is the right behavior in Unix. 
        DOS/Windows might want to take out '\' instead, unless the
        stdio library converts slashes for you. They'd also want to trim 
        to 8 characters. Remember, the overall goal is to make a legal 
        platform-native filename, without any extra directory 
        components.
       Suffixes are another sore point. Really, the game program 
        shouldn't have a suffix on the name passed to this function. So
        in DOS/Windows, this function should chop off dot-and-suffix,
        if there is one, and then add a dot and a three-letter suffix
        appropriate to the file type (as gleaned from the usage 
        argument.)
    */
    
    memcpy(buf, name, len);
    if (len == 0) {
        buf[0] = 'X';
        len++;
    }
    buf[len] = '\0';
    
    for (cx=buf; *cx; cx++) {
        if (*cx == '/')
            *cx = '-';
    }
    
    //    sprintf(buf2, "%s/%s", workingdir, buf);
    strcpy(buf2, buf);
    switch (usage & fileusage_TypeMask) {
        case fileusage_SavedGame:
	  type = GetSaveGameType();
            break;
        case fileusage_Transcript:
	    type = "Text";
            break;
        case fileusage_InputRecord:
	    type = "Text";
            break;
        case fileusage_Data:
        default:
	    type = GetSaveGameType();
            break;
    }
    

    fref = gli_new_fileref(buf2, usage, rock, type);
    if (!fref) {
        gli_strict_warning("fileref_create_by_name: unable to create fileref.");
        return NULL;
    }
    
    return fref;
}

frefid_t glk_fileref_create_by_prompt(glui32 usage, glui32 fmode,
    glui32 rock)
{
    fileref_t *fref;
    char buf[256], newbuf[256];
    char *cx;
    int val;
    char *prompt, *prompt2, *type;
    int checkover;

    switch (usage & fileusage_TypeMask) {
        case fileusage_SavedGame:
            prompt = "Select Saved Game ";
	    type = GetSaveGameType();
            break;
        case fileusage_Transcript:
            prompt = "Select Transcript ";
	    type = "Text";
            break;
        case fileusage_InputRecord:
            prompt = "Select Command Record ";
	    type = "Text";
            break;
        case fileusage_Data:
        default:
            prompt = "Select Data File ";
	    type = GetSaveGameType();
            break;
    }
    
    checkover = 0;
    if (fmode == filemode_Read)
        prompt2 = "To Load";
    else {
      checkover = 1;
      prompt2 = "To Store";
    }
    sprintf(newbuf, "%s%s", prompt, prompt2);
    if (!GetFileName("", type, newbuf, buf, checkover))
      return NULL;
    val = strlen(buf);
    
    while (val 
        && (buf[val-1] == '\n' 
            || buf[val-1] == '\r' 
            || buf[val-1] == ' '))
        val--;
    buf[val] = '\0';
    
    for (cx = buf; *cx == ' '; cx++) { }
    
    val = strlen(cx);
    if (!val) {
        /* The player just hit return. It would be nice to provide a
            default value, but this implementation is too cheap. */
        return NULL;
    }
    
    //    if (cx[0] == '/')
        strcpy(newbuf, cx);
	//    else
	//        sprintf(newbuf, "%s/%s", workingdir, cx);

    fref = gli_new_fileref(newbuf, usage, rock, type);
    if (!fref) {
        gli_strict_warning("fileref_create_by_prompt: unable to create fileref.");
        return NULL;
    }
    
    return fref;
}

frefid_t glk_fileref_iterate(fileref_t *fref, glui32 *rock)
{
    if (!fref) {
        fref = gli_filereflist;
    }
    else {
        fref = fref->next;
    }
    
    if (fref) {
        if (rock)
            *rock = fref->rock;
        return fref;
    }
    
    if (rock)
        *rock = 0;
    return NULL;
}

glui32 glk_fileref_get_rock(fileref_t *fref)
{
    if (!fref) {
        gli_strict_warning("fileref_get_rock: invalid ref.");
        return 0;
    }
    
    return fref->rock;
}

glui32 glk_fileref_does_file_exist(fileref_t *fref)
{
    if (!fref) {
        gli_strict_warning("fileref_does_file_exist: invalid ref");
        return FALSE;
    }
    
    /* This is sort of Unix-specific, but probably any stdio library
        will implement at least this much of stat(). */
    
    if (fexists(fref->filename))
        return 1;
    else
        return 0;
}

void glk_fileref_delete_file(fileref_t *fref)
{
    if (!fref) {
        gli_strict_warning("fileref_delete_file: invalid ref");
        return;
    }
    
    /* If you don't have the unlink() function, obviously, change it
        to whatever file-deletion function you do have. */
        
    unlink(fref->filename);
}

/* This should only be called from startup code. */
void glkunix_set_base_file(char *filename)
{
    int ix;
  
    for (ix=strlen(filename)-1; ix >= 0; ix--) 
        if (filename[ix] == '/')
            break;

    if (ix >= 0) {
        /* There is a slash. */
        strncpy(workingdir, filename, ix);
        workingdir[ix] = '\0';
        ix++;
    }
    else {
        /* No slash, just a filename. */
        ix = 0;
    }
}

