#include <system.h>
#include <stdio.h>
#include <stdlib.h>
#include "glk.h"
#include "cheapglk.h"
#include "../stdlib/fnt.h"
#include "screenmap.h"
#include "heliomain.h"

/* Since we're not using any kind of cursor movement or terminal 
    emulation, we're dreadfully limited in what kind of windows we
    support. In fact, we can only support one window at a time,
    and that must be a wintype_TextBuffer. Printing to this window
    simply means printing to stdout, and reading from it means
    reading from stdin. (The input code is in glk_select(), and
    the output is in glk_put_char() etc.) */

static window_t *mainwin = NULL;
static window_t *statuswin = NULL;
static window_t *auxwin = NULL;

int textFont = smallFont;
int fixedFont = smallFixedFont;

window_t *gli_new_window(glui32 rock)
{
    window_t *win = (window_t *)qmalloc(sizeof(window_t));
    if (!win)
        return NULL;
    
    win->magicnum = MAGIC_WINDOW_NUM;
    win->rock = rock;
    
    win->str = gli_new_stream(strtype_Window, FALSE, TRUE, 0);
    win->str->win = win;
    win->echostr = NULL;
    
    win->line_request = FALSE;
    win->char_request = FALSE;
    win->linebuf = NULL;
    win->linebuflen = 0;
    win->tempptr = 0;
    if (gli_register_obj)
        win->disprock = (*gli_register_obj)(win, gidisp_Class_Window);
    else
        win->disprock.ptr = NULL;
    
    win->x = win->y = win->w = win->h = 0;
    win->style = style_Normal;
    win->curline = 0;
    
    win->isstatus = win->ismain = win->isaux = 0;

    return win;
}

void gli_delete_window(window_t *win)
{
    if (win->linebuf) {
        if (gli_unregister_arr) {
            (*gli_unregister_arr)(win->linebuf, win->linebuflen, "&+#!Cn", 
                win->inarrayrock);
        }
        win->linebuf = NULL;
    }

    if (gli_unregister_obj) {
        (*gli_unregister_obj)(win, gidisp_Class_Window, win->disprock);
        win->disprock.ptr = NULL;
    }
        
    win->magicnum = 0;
    
    /* Close window's stream. */
    gli_delete_stream(win->str);
    win->str = NULL;

    /* The window doesn't own its echostr; closing the window doesn't close
        the echostr. */
    win->echostr = NULL;
    
    qfree(win);
}

winid_t glk_window_open(winid_t split, glui32 method, glui32 size, 
    glui32 wintype, glui32 rock)
{
    window_t *win;
    int oldy, delta;

    if (mainwin || split) {
      if (split==mainwin) {
	if ((statuswin==NULL) && (wintype == wintype_TextGrid) && (method==(winmethod_Above|winmethod_Fixed))) {
	  statuswin = gli_new_window(rock);
	  statuswin->font = fixedFont;
	  statuswin->x = split->x;
	  statuswin->y = 0; //split->y;
	  statuswin->w = split->w;
	  statuswin->type = wintype;
	  FntSetFont(statuswin->font);
	  statuswin->h = FntLineHeight()*size;
	  while (statuswin->h > 148) statuswin->h -= FntLineHeight();
	  oldy = split->y;
	  split->y = statuswin->h+1;
	  split->h = 149-split->y;
	  statuswin->isstatus = 1;
	  
	  // Clear out remains of last line
	  if (oldy!=split->y) {
	    FntSetFont(split->font);
	    delta = statuswin->h % FntLineHeight();
	    os_scroll_area(split->y, split->x, 149, split->x+split->w, FntLineHeight()-delta);
	    split->curr_y = split->y;
	    while (split->curr_y<148) split->curr_y += FntLineHeight();
	    split->curr_y -= FntLineHeight();
#if 0
	    split->curr_y -= FntLineHeight()-delta;
	    if (split->curr_y < split->y) split->curr_y = split->y;
	    if (split->curr_y < 0) split->curr_y = 0;
#endif
	  }
	  return statuswin;
	} else if ((auxwin==NULL) && (wintype == wintype_TextGrid) && (method==(winmethod_Below|winmethod_Proportional))) {
	  auxwin = gli_new_window(rock);
	  auxwin->font = fixedFont;
	  auxwin->x = split->x;
	  auxwin->y = split->y;
	  auxwin->w = split->w;
	  auxwin->h = split->h;
	  auxwin->type = wintype;
	  auxwin->isaux = 1;
	  return auxwin;
	} else
	  return NULL;
      } 
    }
    
    if (wintype != wintype_TextBuffer) {
      /* This cheap library only allows you to open text buffer windows. 
	 Again, don't print a warning. */
      return NULL;
    }

      win = gli_new_window(rock);
      if (!win) {
        gli_strict_warning("window_open: unable to create window.");
        return NULL;
      }
      
      win->font = textFont;
      win->type = wintype;
      win->x = win->y = 0;
      win->w = 160;
      win->h = 149;
      win->ismain = 1;
      mainwin = win;
      return mainwin;
}

static window_t *gli_current_win = NULL;

void glk_window_close(window_t *win, stream_result_t *result)
{
    if (!win) {
        gli_strict_warning("window_close: invalid id.");
        return;
    }
    
    gli_stream_fill_result(win->str, result);
    
    gli_delete_window(win);

    if (win->ismain) {
      if (gli_current_win==mainwin) gli_current_win=NULL;
      mainwin = NULL;
    } else if (win->isstatus) {
      if (gli_current_win==statuswin) gli_current_win=mainwin;
      statuswin = NULL;
    } else if (win->isaux) {
      if (gli_current_win==auxwin) gli_current_win=mainwin;
      auxwin = NULL;
    }
}


window_t *gli_window_get()
{
  return gli_current_win;
}
window_t *gli_window_get_main()
{
  return mainwin;
}
window_t *gli_window_get_aux()
{
  return auxwin;
}
window_t *gli_window_get_status()
{
  return statuswin;
}

winid_t glk_window_get_root()
{
    /* If there's a window, it's the root window. */
    if (mainwin)
        return mainwin;
    else
        return NULL;
}

winid_t glk_window_iterate(window_t *win, glui32 *rockptr)
{
    /* Iteration is really simple when there can only be one window. */
    
    if (!win) {
        /* They're asking for the first window. Return the main window 
            if it exists, or 0 if there is none. */
        if (!mainwin) {
            if (rockptr)
                *rockptr = 0;
            return NULL;
        }
        
        if (rockptr)
            *rockptr = mainwin->rock;
        return mainwin;
    } else if (win == mainwin) {
        /* They're asking for the next window. There is none. */
        if (rockptr)
            *rockptr = 0;
        return NULL;
    }
    else {
        gli_strict_warning("window_iterate: invalid id.");
        return NULL;
    }
}

glui32 glk_window_get_rock(window_t *win)
{
    if (!win) {
        gli_strict_warning("window_get_rock: invalid id.");
        return 0;
    }
    
    return win->rock;
}

glui32 glk_window_get_type(window_t *win)
{
    if (!win) {
        gli_strict_warning("window_get_type: invalid id.");
        return 0;
    }
    
    return win->type;
}

winid_t glk_window_get_parent(window_t *win)
{
  if (!win) {
    gli_strict_warning("window_get_parent: invalid id.");
    return NULL;
  }
  if(win!=mainwin) return mainwin;
  return NULL;
}

winid_t glk_window_get_sibling(window_t *win)
{
    if (!win) {
        gli_strict_warning("window_get_sibling: invalid id.");
        return NULL;
    }
    
    return NULL;
}

strid_t glk_window_get_stream(window_t *win)
{
    stream_t *str;
    
    if (!win) {
        gli_strict_warning("window_get_stream: invalid id.");
        return NULL;
    }
    
    str = win->str;
    
    return str;
}

void glk_window_set_echo_stream(window_t *win, stream_t *str)
{
    if (!win) {
        gli_strict_warning("window_set_echo_stream: invalid window id.");
        return;
    }

    win->echostr = str;
}

strid_t glk_window_get_echo_stream(window_t *win)
{
    stream_t *str;
    
    if (!win) {
        gli_strict_warning("window_get_echo_stream: invalid id.");
        return NULL;
    }
    
    str = win->echostr;
    
    if (str)
        return str;
    else
        return NULL;
}

void glk_set_window(window_t *win)
{
    if (!win) {
        gli_stream_set_current(NULL);
    }
    else {
	gli_current_win = win;
        gli_stream_set_current(win->str);
    }
}

void glk_request_char_event(window_t *win)
{
    if (!win) {
        gli_strict_warning("request_char_event: invalid id");
        return;
    }
    
    if (win->char_request || win->line_request) {
        gli_strict_warning("request_char_event: window already has keyboard request");
        return;
    }
    
    win->char_request = TRUE;
}

void glk_request_line_event(window_t *win, char *buf, glui32 maxlen, 
    glui32 initlen)
{
    if (!win) {
        gli_strict_warning("request_line_event: invalid id");
        return;
    }
    
    if (win->char_request || win->line_request) {
        gli_strict_warning("request_line_event: window already has keyboard request");
        return;
    }
    
    win->line_request = TRUE;
    win->linebuf = buf;
    win->linebuflen = maxlen;

    if (gli_register_arr) {
        win->inarrayrock = (*gli_register_arr)(buf, maxlen, "&+#!Cn");
    }
}

void glk_request_mouse_event(window_t *win)
{
    if (!win) {
        gli_strict_warning("request_mouse_event: invalid id");
        return;
    }
    /* Yeah, right */
    return;
}

void glk_cancel_char_event(window_t *win)
{
    if (!win) {
        gli_strict_warning("cancel_char_event: invalid id");
        return;
    }
    
    win->char_request = FALSE;
}

void glk_cancel_line_event(window_t *win, event_t *ev)
{
    event_t dummyev;
    
    if (!ev) {
        ev = &dummyev;
    }

    gli_event_clearevent(ev);
    
    if (!win) {
        gli_strict_warning("cancel_line_event: invalid id");
        return;
    }
    
    if (win->line_request) {
        if (gli_unregister_arr) {
            (*gli_unregister_arr)(win->linebuf, win->linebuflen, 
                "&+#!Cn", win->inarrayrock);
        }

        win->line_request = FALSE;
        win->linebuf = NULL;
        win->linebuflen = 0;
        
        /* Since there's only one window and no arrangement events,
            once a glk_select() starts, it can only end with actual
            line or character input. But it's possible that the
            program will set a line input request and then immediately
            cancel it. In that case, no input has occurred, so we
            set val1 to zero. */
        
        ev->type = evtype_LineInput;
        ev->val1 = 0;
        ev->val2 = 0;
        ev->win = win;
    }
}

void glk_cancel_mouse_event(window_t *win)
{
    if (!win) {
        gli_strict_warning("cancel_mouse_event: invalid id");
        return;
    }
    /* Yeah, right */
    return;
}

void glk_window_clear(window_t *win)
{
    if (!win) {
        gli_strict_warning("window_clear: invalid id.");
        return;
    }
    
    if (win->line_request) {
        gli_strict_warning("window_clear: window has pending line request");
        return;
    }

    os_erase_area(win->y, win->x, win->y+win->h, win->x+win->w);
    win->tempptr = 0;
    win->curline = 0;
    win->curr_x = win->x;
    win->curr_y = win->y;
}

void glk_window_move_cursor(window_t *win, glui32 xpos, glui32 ypos)
{
    if (!win) {
        gli_strict_warning("window_move_cursor: invalid id.");
        return;
    }
    
    if (win->type==wintype_TextGrid) {
      FntSetFont(win->font);
      win->curr_x = win->x+xpos*FntCharWidth('0');
      win->curr_y = win->y+ypos*FntLineHeight();
    } else
      gli_strict_warning("window_move_cursor: cannot move cursor in a TextBuffer window.");
}

void glk_window_get_size(window_t *win, glui32 *widthptr, 
    glui32 *heightptr)
{
    if (!win) {
        gli_strict_warning("window_get_size: invalid id.");
        return;
    }
    
    FntSetFont(win->font);

    if (widthptr)
        *widthptr = win->w / FntCharWidth('0');
    if (heightptr)
        *heightptr = win->h / FntLineHeight();
}

void glk_window_get_arrangement(window_t *win, glui32 *methodptr,
    glui32 *sizeptr, winid_t *keywinptr)
{
    if (!win) {
        gli_strict_warning("window_get_arrangement: invalid id.");
        return;
    }
    
    gli_strict_warning("window_get_arrangement: not a Pair window.");
}

void glk_window_set_arrangement(window_t *win, glui32 method,
    glui32 size, winid_t keywin)
{
    if (!win) {
        gli_strict_warning("window_set_arrangement: invalid id.");
        return;
    }
    
    gli_strict_warning("window_set_arrangement: not a Pair window.");
}

#ifdef GLK_MODULE_IMAGE

glui32 glk_image_draw(winid_t win, glui32 image, glsi32 val1, glsi32 val2)
{
    gli_strict_warning("image_draw: graphics not supported.");
    return FALSE;
}

glui32 glk_image_draw_scaled(winid_t win, glui32 image, 
    glsi32 val1, glsi32 val2, glui32 width, glui32 height)
{
    gli_strict_warning("image_draw_scaled: graphics not supported.");
    return FALSE;
}

glui32 glk_image_get_info(glui32 image, glui32 *width, glui32 *height)
{
    gli_strict_warning("image_get_info: graphics not supported.");
    return FALSE;
}

void glk_window_flow_break(winid_t win)
{
    gli_strict_warning("window_flow_break: graphics not supported.");
}

void glk_window_erase_rect(winid_t win, 
    glsi32 left, glsi32 top, glui32 width, glui32 height)
{
    gli_strict_warning("window_erase_rect: graphics not supported.");
}

void glk_window_fill_rect(winid_t win, glui32 color, 
    glsi32 left, glsi32 top, glui32 width, glui32 height)
{
    gli_strict_warning("window_fill_rect: graphics not supported.");
}

void glk_window_set_background_color(winid_t win, glui32 color)
{
    gli_strict_warning("window_set_background_color: graphics not supported.");
}

#endif /* GLK_MODULE_IMAGE */

#ifdef GLK_MODULE_HYPERLINKS

void glk_set_hyperlink(glui32 linkval)
{
    gli_strict_warning("set_hyperlink: hyperlinks not supported.");
}

void glk_set_hyperlink_stream(strid_t str, glui32 linkval)
{
    gli_strict_warning("set_hyperlink_stream: hyperlinks not supported.");
}

void glk_request_hyperlink_event(winid_t win)
{
    gli_strict_warning("request_hyperlink_event: hyperlinks not supported.");
}

void glk_cancel_hyperlink_event(winid_t win)
{
    gli_strict_warning("cancel_hyperlink_event: hyperlinks not supported.");
}

#endif /* GLK_MODULE_HYPERLINKS */
