/*
	HEMISC.C

	Miscellaneous functions:

		AP                      PassLocals
		CallRoutine             Peek, PeekWord
		ContextCommand		Poke, PokeWord
		Dict                    PrintHex
		FatalError              Printout
		FileIO                  PromptMore
		Flushpbuffer            RecordCommands
		GetCommand              SaveUndo
		GetString               SetStackFrame
		GetText                 SetupDisplay
		GetWord                 SpecialChar
		InitGame                TrytoOpen
		LoadGame		Undo
		ParseCommandLine	

	for the Hugo Engine

	Copyright (c) 1995-2001 by Kent Tessman
*/


#include "heheader.h"


void Banner(void);                      	/* from he.c */

void hugo_stopmusic(void);
void hugo_stopsample(void);


/* game_version is needed for backward compatibility.  It holds the
   version number of the game*10 + the revision number.  The object
   size of pre-v2.2 games was only 12 bytes.
*/
int game_version;
int object_size = 24;

/* File pointers, etc. */
HUGO_FILE game = NULL;
HUGO_FILE script = NULL;
HUGO_FILE save = NULL;
HUGO_FILE record = NULL;
HUGO_FILE playback = NULL;
HUGO_FILE io = NULL;     char ioblock = 0; char ioerror = 0;
char gamefile[MAXPATH];
char gamepath[MAXPATH];
#if !defined (GLK)
char scriptfile[MAXPATH];
char savefile[MAXPATH];
char recordfile[MAXPATH];
#endif

/* Header information */
char id[3];
char serial[9];
unsigned int codestart;			/* start of executable code	*/
unsigned int objtable;                	/* object table			*/
unsigned int eventtable;              	/* event table			*/
unsigned int proptable;               	/* property table		*/
unsigned int arraytable;              	/* array data table		*/
unsigned int dicttable;               	/* dictionary			*/
unsigned int syntable;                	/* synonyms			*/
unsigned int initaddr;                	/* "init" routine		*/
unsigned int mainaddr;                	/* "main"			*/
unsigned int parseaddr;               	/* "parse"			*/
unsigned int parseerroraddr;          	/* "parseerror"			*/
unsigned int findobjectaddr;          	/* "findobject"			*/
unsigned int endgameaddr;             	/* "endgame"			*/
unsigned int speaktoaddr;             	/* "speakto"			*/
unsigned int performaddr;		/* "perform"			*/

/* Totals */
int objects;
int events;
int dictcount;		/* dictionary entries */
int syncount;		/* synonyms, etc.     */

#if !defined (COMPILE_V25)
char context_command[MAX_CONTEXT_COMMANDS][64];
int context_commands;
#endif

/* Loaded memory image */
unsigned char *mem;			/* the memory buffer            */
int loaded_in_memory = true;		/* i.e., the text bank          */
unsigned int defseg;			/* holds segment indicator      */
unsigned int gameseg;			/* code segment                 */
long codeptr;                           /* code pointer                 */
long codeend;                           /* end of loaded code           */

/* Text output */
char pbuffer[MAXBUFFER*2+1];            /* print buffer for line-wrapping  */
int currentpos = 0;                     /* column position (pixel or char) */
int currentline = 0;                    /* row number (line)               */
int full = 0;                           /* page counter for PromptMore     */
char fcolor = 16,			/* default fore/background colors  */
	bgcolor = 17,			/* (16 = default foreground,	   */
	icolor = 16;			/*  17 = default background)	   */
char default_bgcolor = 17;		/* default for screen background   */
int currentfont = NORMAL_FONT;		/* current font bitmasks           */
char capital = 0;                       /* if next letter is to be capital */
unsigned int textto = 0;		/* for printing to an array        */
int SCREENWIDTH, SCREENHEIGHT;		/* screen dimensions               */
					/*   (in pixels or characters)     */
int physical_windowwidth,		/* "physical_..." measurements	   */
	physical_windowheight,		/*   are in pixels (or characters) */
	physical_windowtop, physical_windowleft,
	physical_windowbottom, physical_windowright;
int inwindow = 0;
int charwidth, lineheight, FIXEDCHARWIDTH, FIXEDLINEHEIGHT;
int current_text_x, current_text_y;

char skipping_more = false;

/* SaveUndo() and Undo() */
int undostack[MAXUNDO][5];		/* for saving undo information     */
int undoptr;                            /* number of operations undoable   */
int undoturn = 0;                       /* number of operations this turn  */
char undoinvalid = 0;                   /* for start of game, and restarts */
char undorecord = 0;                    /* true when recording             */


/* AP

	The all-purpose printing routine that takes care of word-wrapping.
*/

void AP (char *a)
{
	char c, sticky = false, skipspchar = false, startofline = 0;
	int b = 0, i, j, slen;
	int newline = 0;
	int thisline, thisword;           /* widths in pixels or characters */
	int tempfont;

	char q[MAXBUFFER*2+1],            /* current word      */
		r = '\0';                 /* current character */

	static int lastfcolor = -1, lastbgcolor = -1, lastfont = -1;
	static int lastprintedfont = NORMAL_FONT;


	/* Shameless little trick to override control characters in engine-
	   printed text, such as MS-DOS filenames that contain '\'s:
	*/
	if (a[0]==NO_CONTROLCHAR)
	{
		skipspchar = true;
		a = &a[1];
	}

	/* Semi-colon overrides LF */
	if ((strlen(a)>=2) and a[strlen(a)-1]==';' and a[strlen(a)-2]=='\\')
	{
		/* No longer needed--see decrease of slen, below
		a[strlen(a)-2] = '\0'; */
		sticky = true;
	}

	thisline = hugo_textwidth(pbuffer);

	slen = strlen(pbuffer);
	if (slen==0) startofline = true;

	/* Check for color changes */
	if (lastfcolor!=fcolor or lastbgcolor!=bgcolor or startofline)
	{
		pbuffer[slen++] = COLOR_CHANGE;
		pbuffer[slen++] = (char)(fcolor+1);
		pbuffer[slen++] = (char)(bgcolor+1);
		pbuffer[slen] = '\0';
		lastfcolor = fcolor;
		lastbgcolor = bgcolor;
	}

	/* Check for font changes */
	if (lastfont!=currentfont or startofline)
	{
		pbuffer[slen++] = FONT_CHANGE;
		pbuffer[slen++] = (char)(currentfont+1);
		pbuffer[slen] = '\0';
		lastfont = currentfont;
	}


	/* Begin by looping through the entire provided string: */

	slen = (int)strlen(a);
	if (sticky) slen-=2;

	for (i=0; i<slen; i++)
	{
		q[0] = '\0';

		do
		{
			r = a[i];

			/* First check control characters */
			if (r=='\\' and !skipspchar)
			{
				r = a[++i];

				switch (r)
				{
					case 'n':
					{
						if (!textto)
						{
							r = '\0';
							newline = true;
						}
						else
							r = '\n';
						break;
					}
					case 'B':
					{
						currentfont |= BOLD_FONT;
						goto AddFontCode;
					}
					case 'b':
					{
						currentfont &= ~BOLD_FONT;
						goto AddFontCode;
					}
					case 'I':
					{
						currentfont |= ITALIC_FONT;
						goto AddFontCode;
					}
					case 'i':
					{
						currentfont &= ~ITALIC_FONT;
						goto AddFontCode;
					}
					case 'P':
					{
						currentfont |= PROP_FONT;
						goto AddFontCode;
					}
					case 'p':
					{
						currentfont &= ~PROP_FONT;
						goto AddFontCode;
					}
					case 'U':
					{
						currentfont |= UNDERLINE_FONT;
						goto AddFontCode;
					}
					case 'u':
					{
						currentfont &= ~UNDERLINE_FONT;
AddFontCode:
						if (!textto)
						{
							q[strlen(q)+2] = '\0';
							q[strlen(q)+1] = (char)(currentfont+1);
							q[strlen(q)] = FONT_CHANGE;
							hugo_font(currentfont);
						}
						goto GetNextChar;
					}
					case '_':       /* forced space */
					{
						if (textto) r = ' ';
						else r = FORCED_SPACE;
						break;
					}
					default:
						r = SpecialChar(a, &i);
				}
			}
			else if (game_version<=22)
			{
				if (r=='~')
					r = '\"';
				else if (r=='^')
				{
					r = '\0';
					newline++;
				}
			}
			else if (r=='\n')
			{
				r = '\0';
				newline++;
			}

			/* Add the new character */
			q[strlen(q)+1] = '\0';
			q[strlen(q)] = r;
GetNextChar:
			i++;
		}
		while ((r!=' ' or textto) and r!=1 and r!='\0' and i<slen);


		/* Text may be sent to an address in the array table instead
		   of being output to the screen
		*/
		if (textto)
		{
			if (q[0]=='\0') return;

			for (j=0; j<=(int)strlen(q); j++)
			{
				/* space for array length */
				b = (game_version>=23)?2:0;

				if ((unsigned char)q[j] >= ' ')
				{
					mem[arraytable*16L + textto*2 + b] = q[j];
					textto++;
				}
			}
			/* Add a terminating zero in case we don't
			   print any more to the array */
			mem[arraytable*16L + textto*2 + b] = 0;

			if (i>=slen) return;
			continue;       /* back to for (i=0; i<slen; i++) */
		}

		thisword = hugo_textwidth(q);

		i--;


	/* At this point, we know we've either ended the string, hit a space,
	   etc., i.e., something that signals the end of the current word.
	*/
		strcat(pbuffer, q);
		thisline += thisword;

		if (strlen(pbuffer) >= MAXBUFFER*2) FatalError(OVERFLOW_E);


		/* Check if the current screen position plus the width of the
		   to-be-printed text exceeds the width of the screen */

		if (thisline+currentpos > physical_windowwidth and strlen(pbuffer)>2)
		{
			/* If so, skim backward for a place to break the
			   line, i.e., a space or a hyphen */

			thisword = thisline;    /* smaller line length */

			/* A complicated little loop, mainly that way to enable
			   breaks on appropriate punctuation, but also to make
			   sure we don't break in the middle of a '--' dash
			*/

			/* (This loop is also the reason why FORCED_SPACE must
			   be greater than the number of colors/font codes + 1;
			   if not, a color/font change might be confused for
			   FORCED_SPACE.)
			*/
			for (j=strlen(pbuffer)-2;
				((pbuffer[j]!=FORCED_SPACE
				and pbuffer[j]!=' ' and pbuffer[j]!='-'
				and (pbuffer[j]!='/' or pbuffer[j-1]==' '))
				or thisword+currentpos-
					((pbuffer[j]!='-')?hugo_charwidth(pbuffer[j]):0)
					> physical_windowwidth)
				and j>0; j--)
			{
				thisword -= hugo_charwidth(pbuffer[j]);
			}

			strcpy(q, pbuffer);

			if (!j) /* No place to break it--infrequent case */
			{
				/* So start from the beginning and trim off a single
				screen width */
				b=0;
				for (j=0; j<=(int)strlen(pbuffer); j++)
				{
					if ((b+=hugo_charwidth(pbuffer[j]))
						> physical_windowwidth)
					break;
				}
				j--;
			}

			if (q[j]=='-' and q[j+1]=='-') j--;

			/* Print the first part of the line (with the rest
			   stored in pbuffer
			*/
			c = q[j+1];
			q[j+1] = '\0';
			tempfont = currentfont;

                        /* Can't use Rtrim() because a '\t' might be
                           embedded as a font/color code
                        */
                        while (q[strlen(q)-1]==' ') q[strlen(q)-1] = '\0';

			hugo_font(currentfont = lastprintedfont);
			Printout(q);
			lastprintedfont = currentfont;
			q[j+1] = c;

			/* Make sure that the to-be-printed line starts out
			   with the right font (i.e., if a font change was
			   processed by Printout() and is now stored in
			   currentfont)
			*/

                        /* Can't use Ltrim() because a '\t' might be
                           embedded as a font/color code
                        */
                        while (q[j+1]==' ') j++;

                        sprintf(pbuffer, "%c%c%s", FONT_CHANGE, currentfont+1,
				q+j+1);
			currentfont = tempfont;

			thisline = hugo_textwidth(pbuffer);
		}

		if (newline)
		{
			hugo_font(currentfont = lastprintedfont);
			Printout(pbuffer);
			lastprintedfont = currentfont;
			strcpy(pbuffer, "");
			thisline = 0;

			while (--newline) Printout("");
		}
	}

	if (!sticky)
	{
		hugo_font(currentfont = lastprintedfont);
		Printout(pbuffer);
		lastprintedfont = currentfont;
		strcpy(pbuffer, "");
	}
}


/* CALLROUTINE

	Used whenever a routine is called, assumes the routine address
	and begins with the arguments (if any).
*/

int CallRoutine(unsigned int addr)
{
	int arg, i;
	int val;
	int templocals[MAXLOCALS], temppass[MAXLOCALS];
	int temp_stack_depth;
	long tempptr;
#if defined (DEBUGGER)
	int tempdbnest;
#endif

	arg = 0;

	/* Pass local variables to routine, if specified */
	if (mem[codeptr]==OPEN_BRACKET_T)
	{
		codeptr++;
		while (mem[codeptr] != CLOSE_BRACKET_T)
		{
			if (arg)
			{
				for (i=0; i<arg; i++)
					temppass[i] = passlocal[i];
			}

			passlocal[arg++] = GetValue();

			if (arg > 1)
			{
				for (i=0; i<arg-1; i++)
					passlocal[i] = temppass[i];
			}

			if (mem[codeptr]==COMMA_T) codeptr++;
		}
		codeptr++;
	}

	for (i=0; i<MAXLOCALS; i++)
		templocals[i] = var[MAXGLOBALS+i];
	PassLocals(arg);

	temp_stack_depth = stack_depth;

	SetStackFrame(stack_depth, RUNROUTINE_BLOCK, 0, 0);

	tempptr = codeptr;      /* store calling address */
	ret = 0;

#if defined (DEBUGGER)
	tempdbnest = dbnest;
	DebugRunRoutine((long)addr*address_scale);
	dbnest = tempdbnest;
#else
	RunRoutine((long)addr*address_scale);
#endif
	retflag = 0;
	val = ret;
	codeptr = tempptr;

	stack_depth = temp_stack_depth;

	for (i=0; i<MAXLOCALS; i++)
		var[MAXGLOBALS+i] = templocals[i];

	return val;
}


/* CONTEXTCOMMAND

	Adds a command to the context command list.  A zero value
	(i.e., an empty string) resets the list.
*/

void ContextCommand(void)
{
	unsigned int n;

ContextCommandLoop:

	codeptr++;
	
	n = GetValue();
#if !defined (COMPILE_V25)
	if (n==0)
	{
		context_commands = 0;
	}
	else if (context_commands < MAX_CONTEXT_COMMANDS)
	{
		char *cc;

		strncpy(context_command[context_commands], cc = GetWord(n), 64);
		context_command[context_commands][63] = '\0';
		if (strlen(cc)>=64)
			sprintf(context_command[context_commands]+60, "...");
		context_commands++;
	}
#endif
	if (Peek(codeptr)==COMMA_T) goto ContextCommandLoop;
	codeptr++;
}


/* DICT

	Dynamically creates a new dictionary entry.
*/

unsigned int Dict()
{
	int i, len = 256;
	unsigned int arr;
	unsigned int pos = 2, loc;

	codeptr += 2;                           /* "(" */

	if (mem[codeptr]==PARSE_T or mem[codeptr]==WORD_T)
		strcpy(line, GetWord(GetValue()));
	else
	{
		/* Get the array address to read the to-be-
		   created dictionary entry from:
		*/
		arr = GetValue();
		if (game_version>=22)
		{
			/* Convert the address to a word
			   value:
			*/
			arr*=2;

			if (game_version>=23)
				/* space for array length */
				arr+=2;
		}

		defseg = arraytable;
		for (i=0; i<len and PeekWord(arr+i*2)!=0; i++)
			line[i] = (char)PeekWord(arr+i*2);
		defseg = gameseg;
		line[i] = '\0';
	}

	if (Peek(codeptr)==COMMA_T) codeptr++;
	len = GetValue();

	if ((loc = FindWord(line))!=UNKNOWN_WORD) return loc;

	defseg = dicttable;

	for (i=1; i<=dictcount; i++)
		pos += Peek(pos) + 1;

	loc = pos - 2;
	
	if ((long)(pos+strlen(line)) > (long)(codeend-dicttable*16L))
	{
#ifdef DEBUGGER
		sprintf(line, "$MAXDICTEXTEND dictionary space exceeded");
		RuntimeWarning(line);
#endif
		defseg = gameseg;
		return 0;
	}

	Poke(pos++, (unsigned char)strlen(line));
	for (i=0; i<(int)strlen(line) and i<len; i++)
		Poke(pos++, (unsigned char)(line[i]+CHAR_TRANSLATION));
	PokeWord(0, ++dictcount);

	defseg = gameseg;

	SaveUndo(DICT_T, strlen(line), 0, 0, 0);

	return loc;
}


/* FATALERROR */

void FatalError(int n)
{
	char fatalerrorline[64];

#if defined (DEBUGGER)
	hugo_stopmusic();
	hugo_stopsample();

	/* If the Debugger has already issued an error, it will
	   try to recover instead of issuing a stream of
	   identical errors.
	*/
	if (runtime_error) return;
#else
	hugo_cleanup_screen();
#endif

	switch (n)
	{
		case MEMORY_E:
			{sprintf(line, "Out of memory\n");
			break;}

		case OPEN_E:
			{sprintf(line, "Cannot open file\n");
			break;}

		case READ_E:
			{sprintf(line, "Cannot read from file\n");
			break;}

		case WRITE_E:
			{sprintf(line, "Cannot write to save file\n");
			break;}

		case EXPECT_VAL_E:
			{sprintf(line, "Expecting value at $%s\n", PrintHex(codeptr));
			break;}

		case UNKNOWN_OP_E:
			{sprintf(line, "Unknown operation at $%s\n", PrintHex(codeptr));
			break;}

		case ILLEGAL_OP_E:
			{sprintf(line, "Illegal operation at $%s\n", PrintHex(codeptr));
			break;}

		case OVERFLOW_E:
			{sprintf(line, "Overflow at $%s\n", PrintHex(codeptr));
			break;}

		case DIVIDE_E:
			{sprintf(line, "Divide by zero at $%s\n", PrintHex(codeptr));
			break;}
	}

#if defined (DEBUGGER)

	if (routines > 0)
        {
		SwitchtoDebugger();

		if (n==MEMORY_E) DebuggerFatal(D_MEMORY_ERROR);

		RuntimeWarning(line);
		debugger_interrupt = true;
		debugger_skip = true;
		runtime_error = true;

		if (n!=EXPECT_VAL_E)
			RecoverLastGood();

		codeptr = this_codeptr;

		return;
	}

	hugo_cleanup_screen();
#endif

/* crash dump */
/*
if (n==UNKNOWN_OP_E or n==ILLEGAL_OP_E or n==EXPECT_VAL_E or n==OVERFLOW_E)
{
	for (n=-8; n<0; n++)
		printf(" %c%2X", (n==-8)?'$':' ', mem[codeptr+n]);
	printf("\n");
	for (n=-8; n<0; n++)
		printf(" %3d", mem[codeptr+n]);
	printf("\n\n> %2X", mem[codeptr]);
	for (n=1; n<8; n++)
		printf("  %2X", mem[codeptr+n]);
	printf("\n");
	printf(">%3d", mem[codeptr]);
	for (n=1; n<8; n++)
		printf(" %3d", mem[codeptr+n]);
	printf("\n");
}
*/
	sprintf(fatalerrorline, "\nFatal Error:  %s", line);
	PRINTFATALERROR(fatalerrorline);

	hugo_closefiles();
	hugo_blockfree(mem);
	exit(n);
}


/* FILEIO */

void FileIO(void)
{
	char iotype;
	unsigned int fname;
	long skipaddr;
	int temp_stack_depth = stack_depth;
#if defined (DEBUGGER)
	int tempdbnest;
#endif

	iotype = mem[codeptr++];
	skipaddr = (long)PeekWord(codeptr)*address_scale;
	codeptr+=2;
	fname = GetValue();
	if (game_version>=23) codeptr++; /* eol */

	ioerror = 0;

	if (ioblock) goto LeaveFileIO;  /* can't nest file operations */

	if (iotype==WRITEFILE_T)        /* "writefile" */
	{
#if !defined (GLK)
		/* stdio implementation */
		if ((io = HUGO_FOPEN(GetWord(fname), "wb"))==NULL) goto LeaveFileIO;
#else
		/* Glk implementation */
		frefid_t fref = NULL;

		fref = glk_fileref_create_by_name(fileusage_Data | fileusage_BinaryMode,
			GetWord(fname), 0);
		io = glk_stream_open_file(fref, filemode_Write, 0);
		glk_fileref_destroy(fref);
		if (io==NULL) goto LeaveFileIO;
#endif
		ioblock = 1;
	}
	else                            /* "readfile"  */
	{
#if !defined (GLK)
		/* stdio implementation */
		if ((io = HUGO_FOPEN(GetWord(fname), "rb"))==NULL) goto LeaveFileIO;
#else
		/* Glk implementation */
		frefid_t fref = NULL;

		fref = glk_fileref_create_by_name(fileusage_Data | fileusage_BinaryMode,
			GetWord(fname), 0);
		io = glk_stream_open_file(fref, filemode_Read, 0);
		glk_fileref_destroy(fref);
		if (io==NULL) goto LeaveFileIO;
#endif
		ioblock = 2;
	}

#if defined (DEBUGGER)
	tempdbnest = dbnest++;
#endif

	SetStackFrame(stack_depth, RUNROUTINE_BLOCK, 0, 0);

	RunRoutine(codeptr);

#if defined (DEBUGGER)
	dbnest = tempdbnest;
#endif

	stack_depth = temp_stack_depth;

	if (ioerror) retflag = 0;

	fclose(io);
	ioblock = 0;

LeaveFileIO:
	ioerror = 0;
	codeptr = skipaddr;
}


/* FLUSHPBUFFER */

void Flushpbuffer()
{
	if (pbuffer[0]=='\0') return;

	pbuffer[strlen(pbuffer)+1] = '\0';
	pbuffer[strlen(pbuffer)] = (char)NO_NEWLINE;
	Printout(Ltrim(pbuffer));
	currentpos = hugo_textwidth(pbuffer)-charwidth;
	strcpy(pbuffer, "");
}


/* GETCOMMAND */

void GetCommand(void)
{
	char a[256];

	AP("");

	/* This should actually be physical_windowheight/lineheight--I can't
	   figure out why the extra +1 is need (only with variable
	   lineheights)
	*/
	hugo_settextpos(1, physical_windowheight/lineheight+1);
	hugo_settextcolor(fcolor);
	hugo_setbackcolor(bgcolor);

	strncpy(a, GetWord(var[prompt]), 255);
	during_player_input = true;
	full = 0;
	hugo_getline(a);
	during_player_input = false;
	strcpy(buffer, Rtrim(buffer));

	strcpy(parseerr, "");

	full = 1;
	remaining = 0;
}


/* GETSTRING

	From any address <addr>; the segment must be defined prior to
	calling the function.
*/

char *GetString(long addr)
{
	static char a[256];
	int i, length;

	length = Peek(addr);

	for (i=1; i<=length; i++)
		a[i-1] = (char)(Peek(addr + i) - CHAR_TRANSLATION);
	a[i-1] = '\0';

	return a;
}


/* GETTEXT

	Get text block from position <textaddr> in the text bank.  If
	the game was not fully loaded in memory, i.e., if loaded_in_memory
	is not true, the block is read from disk.
*/

char *GetText(long textaddr)
{
	static char g[1025];
	int i, a;
	int tdatal, tdatah, tlen;       /* low byte, high byte, length */


	/* Read the string from memory... */
	if (loaded_in_memory)
	{
		tlen = mem[codeend+textaddr] + mem[codeend+textaddr+1]*256;
		for (i=1; i<=tlen; i++)
		{
			g[i-1] = (char)(mem[codeend+textaddr+1+i] - CHAR_TRANSLATION);
		}
		g[i-1] = '\0';

		return g;
	}

	/* ...Or load the string from disk */
	if (fseek(game, codeend+textaddr, SEEK_SET)) FatalError(READ_E);

	tdatal = fgetc(game);
	tdatah = fgetc(game);
	if (tdatal==EOF or tdatah==EOF or ferror(game)) FatalError(READ_E);

	tlen = tdatal + tdatah * 256;

	for (i=1; i<=tlen; i++)
	{
		if ((a = fgetc(game))==EOF) FatalError(READ_E);
		g[i-1] = (char)(a - CHAR_TRANSLATION);
	}
	g[i-1] = '\0';

	return g;
}


/* GETWORD

	From the dictionary table.
*/

char *GetWord(unsigned int w)
{
	static char *b;
	unsigned short a;

	a = w;

	if (a==0) return "";

	if (a==PARSE_STRING_VAL) return parseerr;
	if (a==SERIAL_STRING_VAL) return serial;

	/* bounds-checking to avoid some sort of memory arena error */
	if ((long)(a+dicttable*16L) > codeend)
	{
		b = "";
		return b;
	}

	defseg = dicttable;
	b = GetString((long)a + 2);
	defseg = gameseg;

	return b;
}


/* INITGAME */

void InitGame(void)
{
#if defined (DEBUGGER)
	int i;
#endif
	/* Stop any audio if this is a restart */
	hugo_stopsample();
	hugo_stopmusic();

#if !defined (COMPILE_V25)
	context_commands = 0;
#endif

	SetStackFrame(stack_depth, RUNROUTINE_BLOCK, 0, 0);

#if defined (DEBUGGER)
	for (i=0; i<MAXLOCALS; i++) strcpy(localname[i], "");
	window[VIEW_LOCALS].count = current_locals = 0;

	PassLocals(0);
	DebugRunRoutine((long)initaddr*address_scale);
#else
	PassLocals(0);
	RunRoutine((long)initaddr*address_scale);
#endif

	ret = 0;
	retflag = 0;
	var[actor] = var[player];
}


/* LOADGAME */

void LoadGame(void)
{
/* unsigned char *cbuf; */
	size_t ccount;
	int i, data;
	int check_version;
	unsigned int synptr;
	long c, textbank, filelength;

#if defined (DEBUGGER)
	if (!strcmp(gamefile, ""))
	{
		game = NULL;
		strcpy(gamefile, "(no file)");
		return;
	}
#endif

#if !defined (GLK) /* since in Glk the game stream is always open */
	if ((game = TrytoOpen(gamefile, "rb", "games"))==NULL)
	{
		if ((game = TrytoOpen(gamefile, "rb", "object"))==NULL)
			FatalError(OPEN_E);
	}
#endif

	fseek(game, 0, SEEK_END);
	filelength = ftell(game);
	fseek(game, 0, SEEK_SET);

	if (ferror(game)) FatalError(READ_E);
	if ((game_version = fgetc(game))==EOF) FatalError(OPEN_E);

	/* Earlier versions of the compiler wrote the version code as
	   1 or 2 instead of 10 or 20.
	*/
	if (game_version==1 or game_version==2)
		game_version*=10;

	if (game_version < 21) object_size = 12;

	check_version = HEVERSION*10 + HEREVISION;
#if defined (COMPILE_V25)
	if (check_version==25 and game_version==30)
		check_version = 30;
#endif

	defseg = gameseg;

	if (game_version < HEVERSION)
	{
#if defined (DEBUGGER)
		hugo_cleanup_screen();
		hugo_clearfullscreen();
#endif
		sprintf(line, "Hugo Compiler v%d.%d or later required.\n", HEVERSION, HEREVISION);
		if (game_version>0)
			sprintf(line+strlen(line), "File \"%s\" is v%d.%d.\n", gamefile, game_version/10, game_version%10);

#if defined (DEBUGGER_PRINTFATALERROR)
		DEBUGGER_PRINTFATALERROR(line);
#else
		printf(line);
#endif
		hugo_closefiles();
		hugo_blockfree(mem);

		exit(OPEN_E);
	}
	else if (game_version > check_version)
	{
#if defined (DEBUGGER)
		hugo_cleanup_screen();
		hugo_clearfullscreen();
#endif
		sprintf(line, "File \"%s\" is incorrect or unknown version.\n", gamefile);

#if defined (DEBUGGER_PRINTFATALERROR)
		DEBUGGER_PRINTFATALERROR(line);
#else
		printf(line);
#endif
		hugo_closefiles();
		hugo_blockfree(mem);
		exit(OPEN_E);           /* ditto */
	}

	hugo_settextpos(1, physical_windowheight/lineheight);

	if (game_version>=25)
		fseek(game, H_TEXTBANK, SEEK_SET);
	else
		/* Because pre-v2.5 didn't have performaddr in the header */
		fseek(game, H_TEXTBANK-2L, SEEK_SET);

	data = fgetc(game);
	textbank = fgetc(game);
	if (data==EOF or textbank==EOF or ferror(game)) FatalError(READ_E);
	textbank = (textbank*256L + (long)data) * 16L;

	/* Use a 1024-byte read block */
	ccount = 1024;
/*
	if ((cbuf = malloc(ccount*sizeof(unsigned char)))==NULL)
		FatalError(MEMORY_E);
*/

	/* Allocate as much memory as is required */
	codeend = textbank;
	if ((!loaded_in_memory) or (mem = (unsigned char *)hugo_blockalloc(filelength))==NULL)
	{
		loaded_in_memory = 0;
		if ((mem = (unsigned char *)hugo_blockalloc(codeend))==NULL)
			FatalError(MEMORY_E);
	}

	c = 0;

	if (fseek(game, 0, SEEK_SET)) FatalError(READ_E);

	/* Load either the entire file or just up to the start of the
	   text bank
	*/
	while (c < (loaded_in_memory ? filelength:codeend))
	{
		/* Complicated, but basically just makes sure that
		   the last read (whether loaded_in_memory or not)
		   doesn't override the end of the file.  Shouldn't
		   normally be a problem for fread(), but it caused
		   a crash under MSVC++.
		*/
		i = fread((unsigned char *)&mem[c], sizeof(unsigned char),
			(loaded_in_memory)?
				((filelength-c>(long)ccount)?ccount:(size_t)(filelength-c)):
				((codeend-c>(long)ccount)?ccount:(size_t)(codeend-c)),
			game);

		if (!i) break;
		c += i;
	}

/* The old way:
		i = fread(cbuf, sizeof(unsigned char), ccount, game);
		if (!i) break;
		for (data=0; data<i and c<((loaded_in_memory)?filelength:codeend); data++)
			mem[c++] = (unsigned char)cbuf[data];
	}
	free(cbuf);
*/
	if (ferror(game)) FatalError(READ_E);

	defseg = gameseg;

	/* Decode header: */

	id[0] = Peek(H_ID);
	id[1] = Peek(H_ID+1);
	id[2] = '\0';

	for (i=0; i<8; i++)
		serial[i] = Peek(H_SERIAL+i);
	serial[8] = '\0';

	codestart  = PeekWord(H_CODESTART);
	objtable   = PeekWord(H_OBJTABLE) + gameseg;
	proptable  = PeekWord(H_PROPTABLE) + gameseg;
	eventtable = PeekWord(H_EVENTTABLE) + gameseg;
	arraytable = PeekWord(H_ARRAYTABLE) + gameseg;
	dicttable  = PeekWord(H_DICTTABLE) + gameseg;
	syntable   = PeekWord(H_SYNTABLE) + gameseg;

	initaddr       = PeekWord(H_INIT);
	mainaddr       = PeekWord(H_MAIN);
	parseaddr      = PeekWord(H_PARSE);
	parseerroraddr = PeekWord(H_PARSEERROR);
	findobjectaddr = PeekWord(H_FINDOBJECT);
	endgameaddr    = PeekWord(H_ENDGAME);
	speaktoaddr    = PeekWord(H_SPEAKTO);
	performaddr    = PeekWord(H_PERFORM);


	/* Read totals: */

	defseg = objtable;
	objects = PeekWord(0);

	defseg = eventtable;
	events = PeekWord(0);

	defseg = dicttable;
	dictcount = PeekWord(0);

	defseg = syntable;
	syncount = PeekWord(0);


	/* Additional information to be found: */

	/* display object */
	if (game_version >= 24)
	{
		data = FindWord("(display)");

		for (i=0; i<objects; i++)
		{
			if (GetProp(i, 0, 1, true)==data)
			{
				display_object = i;
				break;
			}
		}
	}
	
	/* build punctuation string (additional user-specified punctuation) */
	synptr = 2;
	strcpy(punc_string, "");
	for (i=1; i<=syncount; i++)
	{
		defseg = syntable;
		if (Peek(synptr)==3)	/* 3 = punctuation */
		{
			strcpy(line, GetWord(PeekWord(synptr+1)));
			if (strlen(line) + strlen(punc_string) > 63) break;
			strcat(punc_string, line);
		}
		synptr+=5;
	}
}


/* PARSECOMMANDLINE */

#if !defined (GLK)	/* ParseCommandLine() is omitted for Glk */

void ParseCommandLine(int argc, char *argv[])
{
	char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILENAME], ext[MAXEXT];

/*
#if defined (DEBUGGER)
	if (argc==1)
	{
		strcpy(gamefile, "");
		return;
	}
#endif
*/
	if (argc==1)
	{
		Banner();
		hugo_blockfree(mem);
		exit(0);
	}

	hugo_splitpath(argv[1], drive, dir, fname, ext);

	if (strcmp(ext, ""))
		strcpy(gamefile, argv[1]);
	else
		hugo_makepath(gamefile, drive, dir, fname,
#if defined (DEBUGGER)
			"hdx");
#else
			"hex");
#endif

	if (getenv("HUGO_SAVE"))
		hugo_makepath(savefile, "", getenv("HUGO_SAVE"), fname, "sav");
	else
		hugo_makepath(savefile, drive, dir, fname, "sav");

	if (getenv("HUGO_RECORD"))
		hugo_makepath(recordfile, "", getenv("HUGO_RECORD"), fname, "rec");
	else
		hugo_makepath(recordfile, drive, dir, fname, "rec");

	strcpy(scriptfile, DEF_PRN);

	hugo_makepath(gamepath, drive, dir, "", "");
}

#endif	/* GLK */


/* PASSLOCALS

	Must be called before running every new routine, i.e. before
	calling RunRoutine().  Unfortunately, the current locals must
	be saved in a temp array prior to calling.  The argument n
	gives the number of arguments passed.
*/

void PassLocals(int n)
{
	int i;

	for (i=0; i<MAXLOCALS; i++)
		{var[MAXGLOBALS+i] = passlocal[i];
		passlocal[i] = 0;}
	arguments_passed = n;
}


/* PEEK */

unsigned char Peek(long a)
{
	return mem[defseg * 16L + a];
}


/* PEEKWORD */

unsigned int PeekWord(long a)
{
	return (unsigned char)mem[defseg*16L+a] + (unsigned char)mem[defseg*16L+a+1]*256;
}


/* POKE */

void Poke(unsigned int a, unsigned char v)
{
	mem[defseg * 16L + a] = v;
}


/* POKEWORD */

void PokeWord(unsigned int a, unsigned int v)
{
	char low, high;

	high = (char)(v / 256);
	low = (char)(v - high * 256);

	mem[defseg * 16L + a] = low;
	mem[defseg * 16L + a + 1] = high;
}


/* PRINTHEX

	Returns <a> as a hex-number string in XXXXXX format.
*/

char *PrintHex(long a)
{
	static char hex[7];
	int h = 0;

	strcpy(hex, "");

	if (a<0L) a = 0;

	hex[h++] = '0';
	if (a<65536L) hex[h++] = '0';
	else
	{
		sprintf(hex+h, "%1X", (unsigned int)(a/65536L));
		a = a%65536L;
		h++;
	}

	if (a < 4096L) hex[h++] = '0';
	if (a < 256L) hex[h++] = '0';
	if (a < 16L) hex[h++] = '0';

	hex[h] = '\0';
	sprintf(hex+h, "%X", (unsigned int)a);

	return hex;
}


/* PRINTOUT

	Since printf() does not take into account cursor-relocation, 
	color-setting and window-scrolling.
*/

void Printout(char *a)
{
	char b[2], sticky = 0, trimmed = 0;
	char tempfcolor;
	int i, l;
	int last_printed_font = currentfont;


	tempfcolor = fcolor;

	if ((a[0]!='\0') and a[strlen(a)-1]==(char)NO_NEWLINE)
	{
		a[strlen(a)-1] = '\0';
		sticky = true;
	}

	b[0] = b[1] = '\0';


	/* The easy part is just skimming <a> and processing each code
	   or printed character, as the case may be:
	*/
	
	l = 0;	/* physical length of string */

	for (i=0; i<(int)strlen(a); i++)
	{
		if ((a[i]==' ') and !trimmed and currentpos==0)
		{
			continue;
		}

		if ((unsigned char)a[i] > ' ' or a[i]==FORCED_SPACE)
		{
			trimmed = true;
			last_printed_font = currentfont;
		}

		switch (b[0] = a[i])
		{
			case FORCED_SPACE:
				hugo_print(" ");
				l += hugo_charwidth(b[0] = ' ');
				break;

			case FONT_CHANGE:
				hugo_font(currentfont = (int)a[++i]-1);
				break;

			case COLOR_CHANGE:
				fcolor = (char)(a[++i]-1);
				hugo_settextcolor((int)(fcolor));
				hugo_setbackcolor((int)a[++i]-1);
				hugo_font(currentfont);
				break;

			default:
				l += hugo_charwidth(b[0]);
				hugo_print(b);
		}

		if (script and (unsigned char)b[0]>=' ')
			/* fprintf() this way for Glk */
			if (fprintf(script, "%s", b) < 0) FatalError(WRITE_E);

#if defined (SCROLLBACK_DEFINED)
		if (!inwindow) hugo_sendtoscrollback(b);
#endif
	}

	/* If we've got a linefeed and didn't hit the right edge of the
	   window
	*/
	if (!sticky and currentpos+l < physical_windowwidth)
	{
		/* The background color may have to be temporarily set if we're
		   not in a window--the reason is that full lines of the
		   current background color might be printed by the OS-specific
		   scrolling function.  (This behavior is overridden by the
		   Hugo engine for in-window printing, which always adds new
		   lines in the current background color when scrolling.)
		*/
		hugo_print("\r");
		hugo_setbackcolor((inwindow)?bgcolor:default_bgcolor);

		i = currentfont;
		hugo_font(currentfont = last_printed_font);

		if (currentline > physical_windowheight/lineheight)
		{
			hugo_scrollwindowup();
		}
		else
		{
			hugo_print("\n");
		}

		hugo_font(currentfont = i);
		hugo_setbackcolor(bgcolor);
	}

#if defined (AMIGA)
	else
	{
		if (currentpos + l >= physical_windowwidth)
			AmigaForceFlush();
	}
#endif

	/* If no newline is to be printed after the current line: */
	if (sticky)
	{
		currentpos += l;
	}

	/* Otherwise, take care of all the line-feeding, line-counting,
	   etc.
	*/
	else
	{
		currentpos = 0;
		if (currentline++ > physical_windowheight/lineheight)
			currentline = physical_windowheight/lineheight;

		if (!playback) skipping_more = false;

		if (++full >= physical_windowheight/lineheight-1)
			PromptMore();

		if (script)
		{
			/* fprintf() this way for Glk */
			if (fprintf(script, "%s", "\n")<0)
				FatalError(WRITE_E);
		}

#if defined (SCROLLBACK_DEFINED)
		if (!inwindow) hugo_sendtoscrollback("\n");
#endif
	}

	fcolor = tempfcolor;
}


/* PROMPTMORE */

#ifndef PROMPTMORE_REPLACED

void PromptMore(void)
{
	char temp_during_player_input;
	int k, tempcurrentfont;
	int temp_current_text_y = current_text_y;
	
	if (playback and skipping_more)
	{
		full = 0;
		return;
	}
	skipping_more = false;
	
	/* Clear the key buffer */
	while (hugo_iskeywaiting()) hugo_getkey();

	temp_during_player_input = during_player_input;
	during_player_input = false;

	tempcurrentfont = currentfont;
	hugo_font(currentfont = NORMAL_FONT);

	/* +1 to force positioning to the very, very bottom of the window */
	hugo_settextpos(1, physical_windowheight/lineheight+1);

	#if (defined (BEOS) && !defined (GCC_BEOS)) || defined (WIN32) || defined (WXWINDOWS)
		/* For ports where it's possible, do a better "MORE..." prompt
		   without a flashing caret */
/*		hugo_font(currentfont |= BOLD_FONT); */
		if (bgcolor!=DEF_SLBGCOLOR)
		{
			/* system statusline colors */
			hugo_settextcolor(DEF_SLFCOLOR);
			hugo_setbackcolor(DEF_SLBGCOLOR);
		}
		else
		{
			/* system colors */
			if (DEF_FCOLOR < 8)
				hugo_settextcolor(DEF_FCOLOR+8); /* bright */
			else
				hugo_settextcolor(DEF_FCOLOR);
			hugo_setbackcolor(DEF_BGCOLOR);
		}
		hugo_print("     [MORE...]     ");
		/* Note that hugo_iskeywaiting() must flush the printing buffer,
		   if one is being employed */
		while (!hugo_iskeywaiting())
		{
			hugo_timewait(10);
		}
	#else
		/* The normal, with-caret way of doing it */
		/* system colors */
		hugo_settextcolor(16); /* DEF_FCOLOR);  */
		hugo_setbackcolor(17); /* DEF_BGCOLOR); */
		/* hugo_font(BOLD_FONT); */
		hugo_print("[MORE...]");
		/* hugo_font(NORMAL_FONT); */
	#endif

	k = hugo_waitforkey();

	hugo_settextcolor(fcolor);      /* program colors */
	hugo_setbackcolor(bgcolor);

	if (playback and k==27)         /* if ESC is pressed during playback */
	{
		if (fclose(playback))
			FatalError(READ_E);
		playback = 0;
	}
	else if (playback and k=='+')
		skipping_more = true;

	hugo_settextpos(1, physical_windowheight/lineheight+1);
#if defined (BEOS) || defined (WIN32) || defined (WXWINDOWS)
	hugo_print("                    ");
#else
	hugo_print("         ");
#endif
	hugo_font(currentfont = tempcurrentfont);
	hugo_settextpos(1, physical_windowheight/lineheight);
	current_text_y = temp_current_text_y;
	full = 0;

	during_player_input = temp_during_player_input;
}

#endif	// ifndef PROMPTMORE_REPLACED


/* RECORDCOMMANDS */

int RecordCommands(void)
{
	remaining = 0;
	skipping_more = false;

	switch (Peek(codeptr))
	{
		case RECORDON_T:
		{
			if (!record and !playback)
			{
#if !defined (GLK)
				/* stdio implementation */
				hugo_getfilename("for command recording", recordfile);
				if (!strcmp(line, ""))
					return 0;
				if (!hugo_overwrite(line))
					return 0;
				if (!(record = HUGO_FOPEN(line, "wt")))
					return 0;
				strcpy(recordfile, line);
#else
				/* Glk implementation */
				frefid_t fref = NULL;

				fref = glk_fileref_create_by_prompt(fileusage_Transcript | fileusage_TextMode,
					filemode_Write, 0);
				record = glk_stream_open_file(fref, filemode_Write, 0);
				glk_fileref_destroy(fref);
				if (!record)
					return 0;
#endif
				return 1;
			}
			break;
		}

		case RECORDOFF_T:
		{
			if (playback) return 1;

			if (record)
			{
				if (fclose(record)) return (0);

				record = 0;
				return 1;
			}
			break;
		}

		case PLAYBACK_T:
		{
			if (!playback)
			{
#if !defined (GLK)
				/* stdio implementation */
				hugo_getfilename("for command playback", recordfile);
				if (!strcmp(line, ""))
					return 0;
				if (!(playback = HUGO_FOPEN(line, "rt")))
					return 0;
				strcpy(recordfile, line);
#else
				/* Glk implementation */
				frefid_t fref = NULL;

				fref = glk_fileref_create_by_prompt(fileusage_InputRecord | fileusage_TextMode,
					filemode_Read, 0);
				playback = glk_stream_open_file(fref, filemode_Read, 0);
				glk_fileref_destroy(fref);
				if (!playback)
					return 0;
#endif
				return 1;
			}
			break;
		}
	}
	return 0;
}


/* SAVEUNDO

	Formats:

	end of turn:    (0, undoturn, 0, 0, 0)
	move obj.:      (MOVE_T, obj., parent, 0, 0)
	property:       (PROP_T, obj., prop., # or PROP_ROUTINE, val.)
	attribute:      (ATTR_T, obj., attr., 0 or 1, 0)
	variable:       (VAR_T, var., value, 0, 0)
	array:          (ARRAYDATA_T, array addr., element, val., 0)
	dict:           (DICT_T, entry length, 0, 0, 0)
	word setting:   (WORD_T, word number, new word, 0, 0)
*/

void SaveUndo(int a, int b, int c, int d, int e)
{
	int tempptr;

	if (undorecord)
	{
		undostack[undoptr][0] = a;      /* save the operation */
		undostack[undoptr][1] = b;
		undostack[undoptr][2] = c;
		undostack[undoptr][3] = d;
		undostack[undoptr][4] = e;

		/* Put zeroes at end of this operation in case
		   the stack wraps around */
		tempptr = undoptr;
		if (++undoptr==MAXUNDO) undoptr = 0;
		undostack[undoptr][0] = 0;
		undostack[undoptr][1] = 0;
		undoptr = tempptr;

		if (++undoturn==MAXUNDO)        /* turn too complex */
			{undoptr = 0;
			undoturn = MAXUNDO;
			undoinvalid = 1;}

		if (++undoptr==MAXUNDO) undoptr = 0;
	}
}


/* SETSTACKFRAME

        Properly sets up the code_block structure for the current stack
        depth depending on if this is a called block (RUNROUTINE_BLOCK)
        or otherwise.
*/

void SetStackFrame(int depth, int type, long brk, long returnaddr)
{
	if (depth==RESET_STACK_DEPTH) stack_depth = 0;
	else if (++stack_depth>=MAXSTACKDEPTH) FatalError(MEMORY_E);

	code_block[stack_depth].type = type;
	code_block[stack_depth].brk = brk;
	code_block[stack_depth].returnaddr = returnaddr;

#if defined (debugger)
	code_block[stack_depth].dbnest = dbnest;
#endif
}


/* SETUPDISPLAY */

void SetupDisplay(void)
{
	hugo_settextmode();

	hugo_settextwindow(1, 1,
		SCREENWIDTH/FIXEDCHARWIDTH, SCREENHEIGHT/FIXEDLINEHEIGHT);

	last_window_left = 1;
	last_window_top = 1;
	last_window_right = SCREENWIDTH/FIXEDCHARWIDTH;
	last_window_bottom = SCREENHEIGHT/FIXEDLINEHEIGHT;

	hugo_clearfullscreen();
}


/* SPECIALCHAR

	SpecialChar() is passed <a> as the string and <*i> as the
	position in the string.  The character(s) at a[*i], a[*(i+1)],
	etc. are converted into a single Latin-1 (i.e., greater than
	127) character value.

	Assume that the AP() has already encountered a control 
	character ('\'), and that a[*i]... is one of:

		`a	accent grave on following character (e.g., 'a')
		'a	accent acute on following character (e.g., 'a')
		~n	tilde on following (e.g., 'n' or 'N')
		:a	umlaut on following (e.g., 'a')
		^a	circumflex on following (e.g., 'a')
		,c	cedilla on following (e.g., 'c' or 'C')
		<	Spanish left quotation marks
		>	Spanish right quotation marks
		!	upside-down exclamation mark
		?	upside-down question mark
		ae	ae ligature
		AE	AE ligature
		c	cents symbol
		L	British pound
		Y	Japanese Yen
		-	em (long) dash
		#nnn	character value given by nnn

	Note that the return value is a single character--which will
	be either unchanged or a Latin-1 character value.
*/

char SpecialChar(char *a, int *i)
{
	char r, s, skipbracket = 0;

	r = a[*i];
	s = r;

	if (r=='\"') return r;

	/* For a couple of versions, Hugo allowed Inform-style
	   punctuation control characters; I don't remember
	   exactly why.
	*/
	if (game_version <= 22)
		if (r=='~' or r=='^') return r;

	if (r=='(')
		{r = a[++*i];
		skipbracket = true;}

	switch (r)
	{
		case '`':               /* accent grave */
		{
			/* Note that the "s = '...'" characters are
			   Latin-1 and may not display properly under,
			   e.g., DOS */

			s = a[++*i];
#ifndef NO_LATIN1_CHARSET
			switch (s)
			{
				case 'a':  s = (char)0xe0; break; /*  */
				case 'e':  s = (char)0xe8; break; /*  */
				case 'i':  s = (char)0xec; break; /*  */
				case 'o':  s = (char)0xf2; break; /*  */
				case 'u':  s = (char)0xf9; break; /*  */
				case 'A':  s = (char)0xc0; break; /*  */
				case 'E':  s = (char)0xc8; break; /*  */
				case 'I':  s = (char)0xcc; break; /*  */
				case 'O':  s = (char)0xd2; break; /*  */
				case 'U':  s = (char)0xd9; break; /*  */
			}
#endif
			break;
		}
		case '\'':              /* accent acute */
		{
			s = a[++*i];
#ifndef NO_LATIN1_CHARSET
			switch (s)
			{
				case 'a':  s = (char)0xe1; break; /*  */
				case 'e':  s = (char)0xe9; break; /*  */
				case 'i':  s = (char)0xed; break; /*  */
				case 'o':  s = (char)0xf3; break; /*  */
				case 'u':  s = (char)0xfa; break; /*  */
				case 'y':  s = (char)0xfd; break;
				case 'A':  s = (char)0xc1; break; /*  */
				case 'E':  s = (char)0xc9; break; /*  */
				case 'I':  s = (char)0xcd; break; /*  */
				case 'O':  s = (char)0xd3; break; /*  */
				case 'U':  s = (char)0xda; break; /*  */
				case 'Y':  s = (char)0xdd; break; /*  */
			}
#endif
			break;
		}
		case '~':               /* tilde */
		{
			s = a[++*i];
#ifndef NO_LATIN1_CHARSET
			switch (s)
			{
				case 'a':  s = (char)0xe3; break; /*  */
				case 'n':  s = (char)0xf1; break; /*  */
				case 'o':  s = (char)0xf5; break; /*  */
				case 'A':  s = (char)0xc3; break; /*  */
				case 'N':  s = (char)0xd1; break; /*  */
				case 'O':  s = (char)0xd5; break; /*  */
			}
#endif
			break;
		}
		case '^':               /* circumflex */
		{
			s = a[++*i];
#ifndef NO_LATIN1_CHARSET
			switch (s)
			{
				case 'a':  s = (char)0xe2; break; /*  */
				case 'e':  s = (char)0xea; break; /*  */
				case 'i':  s = (char)0xee; break; /*  */
				case 'o':  s = (char)0xf4; break; /*  */
				case 'u':  s = (char)0xfb; break; /*  */
				case 'A':  s = (char)0xc2; break; /*  */
				case 'E':  s = (char)0xca; break; /*  */
				case 'I':  s = (char)0xce; break; /*  */
				case 'O':  s = (char)0xd4; break; /*  */
				case 'U':  s = (char)0xdb; break; /*  */
			}
#endif
			break;
		}
		case ':':               /* umlaut */
		{
			s = a[++*i];
#ifndef NO_LATIN1_CHARSET
			switch (s)
			{
				case 'a':  s = (char)0xe4; break; /*  */
				case 'e':  s = (char)0xeb; break; /*  */
				case 'i':  s = (char)0xef; break; /*  */
				case 'o':  s = (char)0xf6; break; /*  */
				case 'u':  s = (char)0xfc; break; /*  */
				/* case 'y':  s = (char)0xff; break; */ /*  */
				case 'A':  s = (char)0xc4; break; /*  */
				case 'E':  s = (char)0xcb; break; /*  */
				case 'I':  s = (char)0xcf; break; /*  */
				case 'O':  s = (char)0xd6; break; /*  */
				case 'U':  s = (char)0xdc; break; /*  */
			}
#endif
			break;
		}
		case ',':               /* cedilla */
		{
			s = a[++*i];
#ifndef NO_LATIN1_CHARSET
			switch (s)
			{
				case 'C':  s = (char)0xc7; break; /*  */
				case 'c':  s = (char)0xe7; break; /*  */
			}
#endif
			break;
		}
		case '<':               /* Spanish left quotation marks */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xab; /*  */
#endif
			break;
		case '>':               /* Spanish right quotation marks */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xbb; /*  */
			break;
#endif
		case '!':               /* upside-down exclamation mark */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xa1; /*  */
#endif
			break;
		case '?':               /* upside-down question mark */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xbf; /*  */
#endif
			break;
		case 'a':               /* ae ligature */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xe6; ++*i; /*  */
#else
			s = 'e'; ++*i;
#endif
			break;
		case 'A':               /* AE ligature */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xc6; ++*i; /*  */
#else
			s = 'E'; ++*i;
#endif
			break;
		case 'c':               /* cents symbol */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xa2; /*  */
#endif
			break;
		case 'L':               /* British pound */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xa3; /*  */
#endif
			break;
		case 'Y':               /* Japanese Yen */
#ifndef NO_LATIN1_CHARSET
			s = (char)0xa5; /*  */
#endif
			break;
		case '-':               /* em dash */
#ifndef NO_LATIN1_CHARSET
			/* s = (char)0x97; */ /*  */
#endif
			break;
		case '#':               /* 3-digit decimal ASCII code */
		{
			s = (char)((a[++*i]-'0')*100);
			s += (a[++*i]-'0')*10;
			s += (a[++*i]-'0');
#ifdef NO_LATIN1_CHARSET
			if ((unsigned)s>127) s = '?';
#endif
		}
	}

	if (skipbracket)
	{
		++*i;
		if (a[*i+1]==')') ++*i;
		if (s==')') s = r;
	}

	return s;
}


/* TRYTOOPEN

	Tries to open a particular filename (based on a given environment
	variable), trying the current directory first.
*/

#if !defined (GLK)	/* not used for Glk */

FILE *TrytoOpen(char *f, char *p, char *d)
{
	char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILENAME], ext[MAXEXT];
	char envvar[32];
	FILE *tempfile; char temppath[MAXPATH];

	/* Try to open the given, vanilla filename */
	if ((tempfile = HUGO_FOPEN(f, p)))
		return tempfile;

	hugo_splitpath(f, drive, dir, fname, ext);      /* file to open */

	/* If the given filename doesn't already specify where to find it */
	if (!strcmp(drive, "") and !strcmp(dir, ""))
	{
		/* Check gamefile directory */
		hugo_makepath(temppath, "", gamepath, fname, ext);
		if ((tempfile = HUGO_FOPEN(temppath, p)))
		{
			strcpy(f, temppath);    /* the new pathname */
			return tempfile;
		}

		/* Check environment variables */
		strcpy(envvar, "hugo_");        /* the actual var. name */
		strcat(envvar, d);

		if (getenv(strupr(envvar)))
		{
			hugo_makepath(temppath, "", getenv(strupr(envvar)), fname, ext);

			if ((tempfile = HUGO_FOPEN(temppath, p)))
			{
				strcpy(f, temppath);  /* the new pathname */
				return tempfile;
			}
		}
	}

	return NULL;            /* return NULL if not openable */
}

#endif	/* GLK */


/* UNDO */

int Undo()
{
	int count = 0, n;
	int turns, turncount, tempptr;
	int obj, prop, attr, v;
	unsigned int addr;

	if (--undoptr < 0) undoptr = MAXUNDO-1;

	if (undostack[undoptr][1]!=0)
	{
		/* Get the number of operations to be undone for
		   the last turn.
		*/
		if ((turns = undostack[undoptr][1]) >= MAXUNDO)
			goto CheckUndoFailed;

		turns--;

		turncount = 0;
		tempptr = undoptr;

		/* Count the number of operations available to see if there
		   are enough to undo the last turn (as per the number
		   required in <turns>.
		*/
		do
		{
			if (--undoptr < 0) undoptr = MAXUNDO-1;
			turncount++;

			/* if end of turn */
			if (undostack[undoptr][0]==0)
				break;
		}
		while (true);

		if (turncount<turns) goto CheckUndoFailed;

		undoptr = tempptr;

		if (--undoptr < 0) undoptr = MAXUNDO-1;

		while (undostack[undoptr][0] != 0)
		{
			switch (undostack[undoptr][0])
			{
				case MOVE_T:
				{
					MoveObj(undostack[undoptr][1], undostack[undoptr][2]);
					count++;
					break;
				}

				case PROP_T:
				{
					obj = undostack[undoptr][1];
					prop = undostack[undoptr][2];
					n = undostack[undoptr][3];
					v = undostack[undoptr][4];

					if ((addr = PropAddr(obj, prop, 0))!=0)
					{
						defseg = proptable;

						if (n==PROP_ROUTINE)
						{
							Poke(addr+1, PROP_ROUTINE);
							n = 1;
						}

						/* Use this new prop count number if the
						   existing one is too low or a prop routine
						*/
						else if (Peek(addr+1)==PROP_ROUTINE or Peek(addr+1)<(unsigned char)n)
							Poke(addr+1, (unsigned char)n);

						/* property length */
						if (n<=(int)Peek(addr+1))
							PokeWord(addr+2+(n-1)*2, v);
					}
					count++;
					break;
				}

				case ATTR_T:
				{
					obj = undostack[undoptr][1];
					attr = undostack[undoptr][2];
					n = undostack[undoptr][3];
					SetAttribute(obj, attr, n);
					count++;
					break;
				}

				case VAR_T:
				{
					n = undostack[undoptr][1];
					v = undostack[undoptr][2];
					var[n] = v;
					count++;
					break;
				}

				case ARRAYDATA_T:
				{
					defseg = arraytable;
					addr = undostack[undoptr][1];
					n = undostack[undoptr][2];
					v = undostack[undoptr][3];

				/* The array length was already accounted for before calling
				   SaveUndo(), so there is no adjustment of
				   +2 here.
				*/
					PokeWord(addr+n*2, v);
					count++;
					break;
				}

				case DICT_T:
				{
					defseg = dicttable;
					PokeWord(0, --dictcount);
					count++;
					break;
				}
				case WORD_T:
				{
					n = undostack[undoptr][1];
					v = undostack[undoptr][2];
					wd[n] = v;
					word[n] = GetWord(wd[n]);
					count++;
				}
			}
			defseg = gameseg;

			if (--undoptr < 0) undoptr = MAXUNDO-1;
		}
	}

CheckUndoFailed:
	if (!count)
	{
		undoinvalid = 1;
		return 0;
	}

	undoptr++;
	return 1;
}


/*
 * Random-number generator replacement by Andrew Plotkin:
 *
 */

#if defined (BUILD_RANDOM)

static unsigned int rand_table[55];	/* state for the RNG */
static int rand_index1, rand_index2;

int random()
{
    rand_index1 = (rand_index1 + 1) % 55;
    rand_index2 = (rand_index2 + 1) % 55;
    rand_table[rand_index1] = rand_table[rand_index1] - rand_table[rand_index2];
    return rand_table[rand_index1];
}

void srandom(int seed)
{
	int k = 1;
	int i, loop;

	rand_table[54] = seed;
	rand_index1 = 0;
	rand_index2 = 31;
	
	for (i = 0; i < 55; i++)
	{
		int ii = (21 * i) % 55;

		rand_table[ii] = k;
		k = seed - k;
		seed = rand_table[ii];
	}
	
	for (loop = 0; loop < 4; loop++)
	{
		for (i = 0; i < 55; i++)
			rand_table[i] = rand_table[i] - rand_table[ (1 + i + 30) % 55];
	}
}

#endif /* BUILD_RANDOM */
