/*
	HEPARSE.C

	Input parsing functions:

		FindWord
		KillWord
		Match routines
			AddObj
			AddPObject
			AdvanceGrammar
			Available
			DomainObj
			InList
			ObjWord
			SubtractObj
			SubtractPObject
			TryObj
			ValidObj
		Parse
		ParseError
		RemoveWord
		SeparateWords

	for the Hugo Engine

	Copyright (c) 1995-2001 by Kent Tessman
*/


#include "heheader.h"


/* Function prototypes: */
void AddObj(int obj);
void AddPObject(int obj, char type, unsigned int w);
void AdvanceGrammar(void);
int DomainObj(int obj);
int InList(int obj);
int MatchObject(int *a);
int MatchWord(int *a);
void SubtractObj(int obj);
void SubtractPObject(int obj);
void TryObj(int obj);

int GetVal(void);			/* from heexpr.c */


char buffer[MAXBUFFER+MAXWORDS*2];      /* input buffer                    */
char errbuf[MAXBUFFER+1];               /* last invalid input              */
char line[1025];                        /* line buffer                     */

int words = 0;                          /* parsed word count               */
char *word[MAXWORDS+1];                 /* breakdown into words            */
unsigned int wd[MAXWORDS+1];		/*     "      "   dict. entries    */
unsigned int parsed_number;		/* needed for numbers in input	   */

signed char remaining = 0;              /* multiple commands in input      */
char parseerr[MAXBUFFER+1];             /* for passing to RunPrint         */
char parsestr[MAXBUFFER+1];             /* for passing quoted string       */
char xverb;                             /* flag; 0 = regular verb          */
unsigned int grammaraddr;             	/* address in grammar              */
int domain, odomain;                  	/* of object(s)                    */
int objlist[32];                      	/* for objects of verb             */
char objcount;                          /* of objlist                      */
char parse_allflag = false;		/* for "all" in MatchObject()      */
struct pobject_structure
	pobjlist[MAXPOBJECTS];          /* for possible objects            */
int pobjcount;                          /* of pobjlist                     */
int pobj;				/* last remaining suspect          */
int obj_match_state;			/* see MatchCommand() for details  */
char object_is_number;			/* number used in player command   */
unsigned int objgrammar;                /* for 2nd pass                    */
char objstart;                          /*  "   "   "                      */
char objfinish;                         /*  "   "   "                      */
char addflag;                           /* true if adding to objlist[]     */
int speaking;				/* if command is addressed to obj. */

char oops[MAXBUFFER+1];                 /* illegal word                    */
int oopscount = 0;                      /* # of corrections in a row       */

char reparse_everything;
char punc_string[64];			/* punctuation string */

char full_buffer = false;
static char recursive_call = false;	/* to MatchObject() */


/* ADDOBJ

	Adds the object <obj> to objlist[], making all related adjustments.
*/

void AddObj(int obj)
{
	int i;

	for (i=0; i<objcount; i++)
		if (objlist[i]==obj) return;

	objlist[(int)objcount] = obj;
	if (++objcount> 32) objcount = 32;
}


/* ADDPOBJECT

	Adds <obj> as a contender to the possible object list, noting that
	it was referred to as either a noun or an adjective.
*/

void AddPObject(int obj, char type, unsigned int w)
{
	int i;
	int j, num;
	unsigned int pa;

	if (pobjcount==MAXPOBJECTS) return;

	for (i=0; i<pobjcount; i++)
	{
		/* If it is already in the list */
		if (pobjlist[i].obj==obj)
		{
			/* Being referred to with a noun outweighs being
			   referred to previously with an adjective
			*/
			if (type==(char)noun) pobjlist[i].type = (char)noun;

			return;
		}
	}

	/* Getting this to point is presuming that we're adding an object
	   referred to with an adjective, but check just to be sure it isn't
	   also a noun for that same object
	*/

	pa = PropAddr(obj, noun, 0);          /* ...or a noun? */
	if (pa)
	{
		defseg = proptable;
		num = Peek(pa + 1);
		for (j=1; j<=num; j++)
		{
			if (PeekWord(pa + j * 2)==w)
				type = (char)noun;
		}
		defseg = gameseg;
	}

	pobjlist[pobjcount].obj = obj;
	pobjlist[pobjcount].type = type;

	pobjcount++;

}


/* ADVANCEGRAMMAR

	Move the address in the grammar table past the current token.
*/

void AdvanceGrammar(void)
{
	int a;

	defseg = gameseg;

	switch (a = Peek(grammaraddr))
	{
		case FORWARD_SLASH_T:
		case HELD_T:
		case MULTI_T:
		case MULTIHELD_T:
		case ANYTHING_T:
		case NUMBER_T:
		case PARENT_T:
		case NOTHELD_T:
		case MULTINOTHELD_T:
		case WORD_T:
		case OBJECT_T:
		case XOBJECT_T:
		case STRING_T:
			{grammaraddr++;
			break;}

		case ASTERISK_T:
		case ATTR_T:
			{grammaraddr += 2;
			break;}

		case DICTENTRY_T:
		case ROUTINE_T:
		case OBJECTNUM_T:
			{grammaraddr += 3;
			break;}

		case OPEN_BRACKET_T:
			{grammaraddr +=5;
			break;}
	}
}


/* AVAILABLE

	The non_grammar argument is true when called from a non-
	grammar function such as RunEvents().
*/

int Available(int obj, char non_grammar)
{
	int temp_stack_depth;

	if (findobjectaddr)
	{
		passlocal[0] = obj;

		/* if anything or (Routine) */
		if ((Peek(grammaraddr)==ANYTHING_T
			or (Peek(grammaraddr)==OPEN_BRACKET_T and Peek(grammaraddr+1)==ROUTINE_T))
			and non_grammar==0)
		{
			passlocal[1] = 0;
		}
		else
		{
			/* domain of -1 is an explicit 'parent' */
			if (domain==-1)
				/* passlocal[1] = 0; */
				passlocal[1] = var[location];
			else if (domain > 0)
				passlocal[1] = domain;
			else if (speaking) passlocal[1] = GrandParent(speaking);
			else passlocal[1] = var[location];
		}

		ret = 0;

		PassLocals(2);
		temp_stack_depth = stack_depth;

		SetStackFrame(stack_depth, RUNROUTINE_BLOCK, 0, 0);

#if defined (DEBUGGER)
		DebugRunRoutine((long)findobjectaddr*address_scale);
#else
		RunRoutine((long)findobjectaddr*address_scale);
#endif
		retflag = 0;
		stack_depth = temp_stack_depth;
		return ret;
	}
	else
		return 1;
}


/* DOMAINOBJ

	Takes into account the preset domain for checking an object's
	presence; <domain> is 0, -1, or an object number..
*/

int DomainObj(int obj)
{
	int yes = false;

	if (obj != var[player])
	{
		switch (domain)
		{
			case 0:
			case -1:
			{
				if (Parent(obj)==var[location])
					yes = true;
				else
				{
					if (Parent(obj)==var[location] and !InList(Parent(obj)))
						yes = true;
				}

				if (Peek(grammaraddr)==MULTINOTHELD_T)
				{
					if (Parent(obj)==var[player])
						yes = false;
				}
				break;
			}

			default:
			{
				if (Parent(obj)==domain)
					yes = true;
			}
		}
	}

	return yes;
}


/* FINDWORD

	Returns the dictionary address of <a>.
*/

unsigned int FindWord(char *a)
{
	unsigned int ptr = 0;
	int i;

	if (a[0]=='\0') return 0;

	defseg = dicttable;

	for (i=1; i<=dictcount; i++)
	{
		if ((unsigned char)(mem[dicttable*16L+ptr+3]-CHAR_TRANSLATION)==(unsigned char)a[0])
		{
			if (!strcmp(GetString(ptr + 2), a))
			{
				defseg = gameseg;
				return ptr;
			}
		}

		ptr += Peek(ptr + 2) + 1;
	}
	
	/* As a last resort, see if the first 6 characters of the word (if it
	   has at least six characters) match a dictionary word:
	*/
	if (strlen(a) >= 6)
	{
		ptr = 0;
	
		for (i=1; i<=dictcount; i++)
		{
			if ((unsigned char)mem[dicttable*16L+ptr+3]-CHAR_TRANSLATION==a[0])
			{
				if (!strncmp(GetString(ptr + 2), a, strlen(a)))
				{
					/* As long as the dictionary word 
					   doesn't contain a space */
					if (strrchr(GetString(ptr+2), ' '))
						goto NextPossibleWord;
						
					defseg = gameseg;
					return ptr;
				}
			}
	
NextPossibleWord:
			ptr += Peek(ptr + 2) + 1;
		}
	}


	defseg = gameseg;

	return UNKNOWN_WORD;                    /* not found */
}


/* INLIST

	Checks to see if <obj> is in objlist[].
*/

int InList(int obj)
{
	int i;

	for (i=0; i<objcount; i++)
		if (objlist[i]==obj) return true;
	return false;
}


/* KILLWORD

	Deletes word[a].
*/

void KillWord(int a)
{
	int i;

	if (a>words) return;

	for (i=a; i<words; i++)
		word[i] = word[i+1];
	word[words] = "";

	RemoveWord(a);
	words--;
/*      var[wordcount] = words;*/
}


/* MATCHCOMMAND

	Here, briefly, is how MatchCommand() works:

		1.  Match the verb.

		2.  If no match, check to see if the line begins with an
		    object (character) and try to match it.

		3.  If found, try to match a syntax for that verb, including
		    objects, dictionary words, numbers, attributes, and
		    routines.  If any objects are specified, skip over them
		    for now, marking the start and finish.  This is done
		    mostly in MatchWord().

		4.  Match the xobject, if there is one--via MatchObject().

		5.  If all is well, return to match the objects that were
		    previously skipped over, loading them into objlist[].
		    Once again, this is done by MatchObject().

	(The reason the objects are initially skipped is because it may
	be necessary to know where to look for them--this may require
	knowing what the xobject is, if the syntax is something like:

		"get" <object> "from" <xobject>)

	The variable <obj_match_state> is the indicator of what stage
	object-matching is at:

		obj_match_state = 0  - haven't matched anything yet

		obj_match_state = 1  - xobject has been matched

		obj_match_state = 2  - matching object(s), loading
				       objlist[]

		obj_match_state = 5  - matching first word/name, i.e.,
				       "Bob, <do something>"
*/

int MatchCommand()
{
	int i, j, flag, a, mw = 0, gotspeaker = 0;
	int numverbs = 0, nextverb = 0;
	unsigned int ptr, verbptr, nextgrammar;
	unsigned int obj, propaddr;

	odomain = 0;

	if (!strcmp(word[1], "~oops"))
	{
		strcpy(parseerr, "");

		/* "oops" on its own */
		if (words==1 or !strcmp(oops, ""))
		{
			ParseError(16, 0);       /* "You'll have to make a mistake..." */
			return 0;
		}

		/* trying to correct more than one word */
		if (words > 2 or oopscount)
		{
			ParseError(17, 0);       /* "...one word at a time..." */
			return 0;
		}

		/* trying to correct a correction */
		if (!strcmp(Left(errbuf, 5), "~oops"))
		{
			ParseError(13, 0);
			return 0;
		}

		/* Rebuild the corrected buffer */
		oopscount = 1;
		strcpy(line, word[2]);
		for (i=1; i<=(int)strlen(errbuf); i++)
		{
			if (!strcmp(Mid(errbuf, i, strlen(oops)), oops))
				break;
		}

		strcpy(buffer, errbuf);
		buffer[i-1] = '\0';
		strcat(buffer, line);

		strcat(buffer, Right(errbuf, strlen(errbuf) - i - strlen(oops) + 1));

		SeparateWords();
		if (!Parse()) return 0;
	}

	if (word[1][0]=='.') KillWord(1);


	/*
	 * STEP 1:  Match verb
	 *
	 */

	ptr = 64;

MatchVerb:
	if (words==0) return 0;

	defseg = gameseg;

	grammaraddr = 0;
	domain = 0;
	obj_match_state = 0;
	xverb = 0;
	objcount = 0;
	parse_allflag = false;
	objstart = 0;
	object_is_number = false;

	var[object] = 0;
	var[xobject] = 0;
	var[self] = 0;
	var[verbroutine] = 0;

	while (Peek(ptr) != 255)
	{
		defseg = gameseg;

		/* verb or xverb header */
		if (Peek(ptr)==VERB_T or Peek(ptr)==XVERB_T)
		{
			/* Skim through 1 or more verb words */
			numverbs = Peek(ptr + 1);
			verbptr = ptr + 2;
			for (i=1; i<=numverbs; i++)
			{
				/* 0xffff signals something other than a 
				   dictionary word--see BuildVerb() in hcbuild.c.
				   This will be the case, like, 1% of the time.
				*/

				/* If it is a dictionary word... */
				if (PeekWord(verbptr)!=0xffff)
				{
					/* If one of the verb words matches the first
					   word in the input line
					*/
					if (wd[1]==PeekWord(verbptr))
					{
						grammaraddr = ptr;
						goto GotVerb;
					}

					verbptr += 2;
				}

				/* ...otherwise assume it's an object (value) */
				else
				{
					codeptr = verbptr + 1; /* skip 0xffff */

					/* GetVal(), not GetValue(), since it's
					   always a simple value
					*/
					obj = GetVal();

					/* codeptr can't >65535 on a 16-bit
					   compiler
					*/
					verbptr = (unsigned int)codeptr;
					propaddr = PropAddr(obj, noun, 0);
					if (propaddr)
					{
						defseg = proptable;
						a = Peek(propaddr+1);	/* obj.#prop */
						defseg = gameseg;

						for (j=1; j<=a; j++)
						{
							if (wd[1]==(unsigned)GetProp(obj, noun, j, 0))
							{
								grammaraddr = ptr;
								goto GotVerb;
							}
						}
					}
				}
			}

			/* Otherwise skip over this verb header */
			ptr += 2 + numverbs * 2;
		}

		/* anything else */
		else
			ptr += Peek(ptr + 1) + 1;
	}


	/*
	 * STEP 2:  Match object/character (if no verb match)
	 *
	 */

	/* If we hit the end of the grammar without finding a verb match: */
	if (Peek(ptr)==255)
	{
		/* See if the command begins with an object
		   (character) name:
		*/
		flag = 0;
		for (i=0; i<objects; i++)
		{
			if (ObjWord(i, wd[1]))
			{
				flag = 1;
				break;
			}
		}

		/* No match, ergo an invalid command */
		if (flag==0 and nextverb==true)
		{
			strcpy(parseerr, "");
			ParseError(6, 0);        /* "...doesn't make any sense..." */
			return 0;
		}

		/* No provision made for addressing objects (characters) */
		if (flag==0 or speaktoaddr==0)
		{
			strcpy(parseerr, "");
			ParseError(2, 0);        /* "Better start with a verb..." */
			return 0;
		}

		/* Count how many object words there are */
		for (i=2; i<=words; i++)
		{
			flag = 0;
			for (j=0; j<objects; j++)
			{
				if (ObjWord(j, wd[i]))
				{
					flag = 1;
					break;
				}
			}
			if (flag==0) break;
		}

		/* Try to match the first word to a valid object */
		objfinish = (char)(i - 1);
		obj_match_state = 5;
		i = 1;
		recursive_call = 0;
		
		if (MatchObject(&i) != true) return 0;  /* unsuccessful */
		
		speaking = pobj;                        /* successful */
		gotspeaker = true;

		/* So erase the object name from the start of the line */
		for (i=1; i<=objfinish; i++)
			KillWord(1);
		if (word[1][0]=='~') KillWord(1);
		
		/* If it's a name and that's all...*/
		if (words==0) return true;

		/* ...or else proceed as usual */
		ptr = 64;
		goto MatchVerb;
	}
	else
	{
		if (!gotspeaker)
			speaking = 0;
	}

GotVerb:

	obj_match_state = 0;
	strcpy(parseerr, word[1]);

	if (Peek(grammaraddr)==XVERB_T) xverb = true;
	grammaraddr += 2 + numverbs * 2;

	/*
	 * STEP 3:  Match proper grammar structure (syntax)
	 *
	 */

	/*
	 * (STEP 4:  Match xobject, if any)
	 *
	 */

	/* Loop until end of grammar table, or next verb:
	 */
	while (Peek(grammaraddr)!=255 and
		Peek(grammaraddr)!=VERB_T and Peek(grammaraddr)!=XVERB_T)
	{
		nextgrammar = grammaraddr + Peek(grammaraddr + 1) + 1;
		a = 1;

		/* Loop until end of table or next verb: */
		while (Peek(grammaraddr) != 255 and Peek(grammaraddr) != VERB_T and Peek(grammaraddr) != XVERB_T)
		{
			mw = MatchWord(&a);

			if (mw==true)
			{
				/* end of both input and grammar */
				if (wd[a]==0 and Peek(grammaraddr)==ROUTINE_T)
					{full_buffer = (char)a;
					break;}

				/* end of grammar, not input */
				if (Peek(grammaraddr)==ROUTINE_T)
					{mw = false;
					goto NextStructure;}
			}
			else
			{
				/* If error already signalled */
				if (mw==2) return 0;

				/* No match, so try next structure */
				else
				{
NextStructure:
					grammaraddr = nextgrammar;
					var[object] = 0;
					var[xobject] = 0;
					var[verbroutine] = 0;
					domain = 0;
					obj_match_state = 0;
					xverb = 0;
					objcount = 0;
					objstart = 0;
					break;
				}
			}
		}

		/* Matched the complete syntax of a verb */
		if (mw==true)
		{
			var[verbroutine] = PeekWord(grammaraddr + 1);
			break;
		}
	}

	if (mw != true)
	{
		if (mw==false)	/* mw = 2 if error already printed */
		{
			/* If there's more grammar to check... */
			if (Peek(grammaraddr) != 255)
			{
				ptr = grammaraddr;
				nextverb = true;
				goto MatchVerb;
			}

			/* ...or if we reached the end without a sensible
			   syntax matched:
			*/
			strcpy(parseerr, "");

			/* "...doesn't make any sense..." */
			ParseError(6, 0);
		}
		return 0;
	}


	/*
	 * STEP 5:  Match remaining object(s), if any
	 *
	 */

	/* If there are objects waiting to be loaded into objlist[] */
	if (objstart)
	{
		/* Call FindObject(0, 0) to reset library's disambiguation
		   mechanism
		*/
		SetStackFrame(RESET_STACK_DEPTH, RUNROUTINE_BLOCK, 0, 0);
		PassLocals(0);
#if defined (DEBUGGER)
		DebugRunRoutine((long)findobjectaddr*address_scale);
#else
		RunRoutine((long)findobjectaddr*address_scale);
#endif
		retflag = 0;

	
		obj_match_state = 2;
		recursive_call = false;
		if (odomain) domain = odomain;

		grammaraddr = objgrammar;
		i = objstart;

		/* If there was a problem matching them */
		if (MatchObject(&i)==false) return 0;
	}

	/* Got a successfully matched verb with all the requisite
	   parameters (if any).
	*/
	if (parsestr[0]=='\"')
	{
		strcpy(parsestr, Right(parsestr, strlen(parsestr)-1));
		if (parsestr[strlen(parsestr)-1]=='\"')
			parsestr[strlen(parsestr)-1] = '\0';
	}

	if (words<a) remaining = 0;
	else
		remaining = (char)(words - a);

	return true;
}


/* MATCHOBJECT

	A dastardly function that cost weeks of sleep to design, code,
	and tweak into workability.  Handle with care--it's a house of
	cards.

	NOTE:  recusive_call is set to 0 if this is the first call.
	MatchObject() sets it to 1 when calling itself when asking
	for clarification as to which object is meant.

	Return -1 on a recursive call to allow parsing to continue.
*/

#define MAX_MOBJ 16     /* maximum number of matchable object words */

int MatchObject(int *a)
{
	int i, j, k, flag;
	int mobjs; unsigned int mobj[MAX_MOBJ];
	int allmatch, roomloc;
	int nextobj = 0, bestobj;
	int m;
	int wtemp;
	unsigned int wdtemp[MAXWORDS+1];
	int tempobjlist[MAXWORDS+1], tempobjcount;
	struct pobject_structure temppobjlist[MAXPOBJECTS];
	int temppobjcount;
	int tempaddflag;

	/* If this is a recursive call, we're not adding new objects */
	if (!recursive_call) addflag = true;

	stack_depth = 0;

	pobj = 0;                          /* possible object */
	objcount = 0;                      /* of objlist[] */
	pobjcount = 0;                     /* of pobjlist[] */
	mobjs = 0;                         /* # of previous words in phrase */

	strcpy(parseerr, "");

	do                                 /* starting at word #a */
	{
		/* Check first to make sure it's not a housekeeping word
		   such as "~and" or "~all".
		*/
		if (word[*a][0]!='~' and word[*a][0]!='\0')
		{
			if (parseerr[0]!='\0') strcat(parseerr, " ");
			strcat(parseerr, word[*a]);

			flag = 0;
			for (i=0; i<objects; i++)
			{
				/* Might be this object if wd[*a] is an
				   adjective or noun of object i
				*/
				m = ObjWord(i, wd[*a]);

				if (m)
				{
					flag = true;
					allmatch = true;

					/* check previously matched words */
					for (j=1; j<=mobjs; j++)
					{
						if (!ObjWord(i, mobj[j])) /* or wd[*a]==mobj[j]) */
							allmatch = false;
					}

					/* matches all previous words */
					if (allmatch==true)
					{
						AddPObject(i, (char)m, wd[*a]);
						pobj = i;
					}

					/* doesn't match previous words */
					else
						SubtractPObject(i);
				}

				/* definitely not this object */
				else
					SubtractPObject(i);
			}


			/* If checking the start of an input line, i.e. for
			   a command addressed to an object (character):
			*/
			if (obj_match_state==5 and !flag) goto Clarify;
		}

		else if (!strcmp(word[*a], "~any"))
			goto NextLoop;

		/* "~and", "~all",... */
		else
			goto Clarify;

		/* Didn't get any suspects */
		if (pobjcount==0)
		{
			/* If "~and", "~all",... */
			if (word[*a][0]=='~')
			{
				/* If checking the xobject */
				if (obj_match_state==1)
					{strcpy(parseerr, word[1]);
					/* "...can't use multiple objects..."
					   (as indirect objects) */
					ParseError(7, 0);
					return 0;}
				goto Clarify;
			}

			/* Got an unmatchable sequence of words */
			else
			{
				/* "(no such thing)..." */
				ParseError(5, 0);
				return 0;
			}
		}

		if (word[*a][0]!='~')
		{
			/* Go back for next word in this object phrase */

			mobjs++;
			if (mobjs==MAX_MOBJ)
			{
				/* "(no such thing)..." */
				ParseError(5, 0);
				return 0;
			}
			mobj[mobjs] = wd[*a];
		}
		else
		{
			/* Since hitting "~and" or "~all", we've obviously
			   finished an object phrase
			*/
			(*a)++;
			goto Clarify;
		}

NextLoop:
		(*a)++;                         /* next word */
		if ((*a > words or word[*a][0]=='\0')
			or (obj_match_state != 1 and *a > objfinish))
			goto Clarify;
	}
	while (true);                           /* endless loop */


Clarify:;

	if (!strcmp(word[*a], "~all"))          /* if "~all" is specified */
	{
		parse_allflag = true;
		
		/* If one or more words were already matched, however... */

		if (mobjs > 0)
		{
			ParseError(6, 0);       /* "...doesn't make any sense..." */
			return 0;
		}

		if (!domain)    /* no particular domain object specified */
			roomloc = var[location];
		else
			roomloc = domain;

		/* Try to add everything in the specified domain
		   to objlist[]
		*/
		for (i=1; i<=Children(roomloc); i++)
		{
			if (i==1)
				nextobj = Child(roomloc);
			else
				nextobj = Sibling(nextobj);

			for (j=0; j<=Children(nextobj); j++)
			{
				if (j==0)
					TryObj(nextobj);
				else
				{
					if (j==1)
						TryObj(Child(nextobj));
					else
						TryObj(Sibling(nextobj));
				}
			}
		}
		(*a)++;


		/* Done processing the object phrase yet? */

		if ((*a > words or word[*a][0]=='\0')
			or (obj_match_state != 1 and *a >= objfinish))
		{
			if (!objcount and !speaking)
			{
				strcpy(parseerr, word[1]);
				ParseError(9, 0);   /* "Nothing to (verb)..." */
				return 0;
			}
			return true;
		}

		/* Go back for the next piece of the phrase */
		pobjcount = 0;
		(*a)--;
		goto NextLoop;
	}


	/* If this is a clarification-required call to MatchObject(),
	   return -1 if more than one possible object is found.
	*/
	if (pobjcount > 1 and recursive_call) return -1;


	/* If we have a possible object or set of objects, go through the
	   disqualification process, either to sort out any confusion, or
	   even if there's only one possible object, to make sure it's
	   available
	*/
	if (pobjcount >= 1)
	{
		bestobj = 0;

		for (j=1; j<=2; j++)		/* twice for ordering issues */
		for (k=0; k<pobjcount; k++)	/* disqualify if unavailable */
		{
			i = pobjlist[k].obj;

			/* held or multiheld */
			if ((domain) and domain==var[player])
			{
				if (Parent(i) != var[player])
				{
					SubtractPObject(i);
					k--;
				}
				else
					pobj = i;
			}

			else if ((Peek(grammaraddr)==NOTHELD_T or Peek(grammaraddr)==MULTINOTHELD_T) 
				and Parent(i)==var[player])
			{
				SubtractPObject(i);
				k--;

				/* if this was the last suspect */
				if (pobjcount<=1)  /* i.e., 0 */
				{
					ParseError(11, i);  /* "You don't see that..." */
					return 0;
				}
			}

			/* otherwise */
			else if (Available(i, 0)==false) /* and obj_match_state!=5) */
			{
				SubtractPObject(i);
				k--;

				m = domain;
				/* Temporary parent domain */
				domain = -1;
				if (Available(i, 0)) bestobj = i;
				domain = m;
			}
			else
				pobj = i;

			if (!bestobj) bestobj = i;
		}


		/* Disqualify if an object is less exact--i.e., if the
		   word is a noun for one object but only an adjective for
		   another, disqualify the one for which it is an
		   adjective.
		*/
		if (pobjcount > 1)
		{
			for (i=0; i<pobjcount; i++)
			{
				if (pobjlist[i].type==(char)noun)
				{
					for (j=0; j<pobjcount; j++)
					{
						if (pobjlist[j].type==(char)adjective)
						{
							bestobj = pobjlist[i].obj;
							SubtractPObject(pobjlist[j].obj);
							j--;
							if (pobjcount<=1) break;
							if (pobj==pobjlist[j].obj) pobj = bestobj;
						}
					}
				}
				if (pobjcount==1) break;
			}
		}


		/* And finally, use whatever disambiguation the FindObject
		   routine might provide, if necessary.
		*/
		if (pobjcount > 1 and findobjectaddr)
		{
			for (k=0; k<pobjcount; k++)
			{
				i = pobjlist[k].obj;
				ret = 0;
				passlocal[0] = i;
				PassLocals(2);

				SetStackFrame(RESET_STACK_DEPTH, RUNROUTINE_BLOCK, 0, 0);
#if defined (DEBUGGER)
				DebugRunRoutine((long)findobjectaddr*address_scale);
#else
				RunRoutine((long)findobjectaddr*address_scale);
#endif
				retflag = 0;

				if (ret==0)     /* returned false */
				{
					SubtractPObject(i);
					k--;	/* so we don't skip one */
					if (pobjcount<=1) break;
				}
				else
				{
					bestobj = i;
					pobj = i;
				}

				if (pobjcount==1) break;
			}
		}


		/* On the other hand, if we've managed to disqualify
		   everything:
		*/
		if (pobjcount==0 and !speaking)
		{
			if ((!domain) or domain != var[player])
			{
				if (Peek(grammaraddr)==ANYTHING_T)
					/* "...haven't seen..." */
					ParseError(10, bestobj);
				else
					/* "...don't see any..." */
					ParseError(11, bestobj);
			}
			else
				ParseError(15, bestobj);  /* "...don't have any..." */
			return 0;
		}


		/* If, after all this, there still exists some confusion
		   about what object is meant:
		*/
		if (pobjcount > 1)
		{
			char tempxverb = xverb;
			ParseError (8, 0);      /* "Which...do you mean..." */
			xverb = tempxverb;

			for (i=1; i<=words; i++)      /* save word arrays */
				wdtemp[i] = wd[i];
			wtemp = words;

			for (i=0; i<=objcount; i++)   /* save object arrays */
				tempobjlist[i] = objlist[i];
			for (i=0; i<=pobjcount; i++)
				temppobjlist[i] = pobjlist[i];

			tempobjcount = objcount;
			temppobjcount = pobjcount;
			tempaddflag = addflag;


			/* Get a new input and properly parse it; after
			   all, it may be returned to MatchCommand() as
			   a potentially valid command.
			*/
			GetCommand();
			SeparateWords();
			if (words==0)
				{ParseError(0, 1);
				return 0;}
			if (!Parse()) return 0;

			/* Do we not care?  i.e. is "any", "either", etc.
			   given? */

			if (!strcmp(word[1], "~any") and words==1)
			{
				for (k=0; k<pobjcount; k++)
				{
					i = pobjlist[k].obj;

					if (strcmp(Name(i), ""))
					{
						pobj = i;
						strcpy(line, "(");
						/*
						if (GetProp(i, article, 1, 0))
							sprintf(line+strlen(line), "the ");
						*/
						sprintf(line+strlen(line), "%s)", Name(i));
						AP(line);
						goto RestoreTempArrays;
					}
				}
			}

			/* Check first to see if the first word is a noun
			   or adjective for any of the possible suspect
			   objects.
			*/
			flag = 0;
			for (i=0; i<pobjcount; i++)
			{
				if (ObjWord(pobjlist[i].obj, wd[1])) flag = 1;
			}

			/* If not, tell MatchCommand() that there's a 
			   new command coming down the hopper.
			*/
			if (flag==0)
				{full_buffer = true;
				return 0;}

			/* Here, tell MatchObject()--this function--
			   that this isn't a virgin call.
			*/
			recursive_call = true;
			addflag = true;
			i = 1;
			j = MatchObject(&i);
			if (j==false)	/* parsing error */
			{
				full_buffer = false;
				return 0;
			}
			else if (j==-1)	/* multiple matches found */
			{
				/* Now, even if we weren't able to find a
				   match, check to see if the last word
				   belongs to only one of the possible
				   suspects, especially if it's a noun.
				*/
				flag = 0;
				bestobj = 0;
				i = words;
				while (wd[i]==0) i--;
				for (j=0; j<temppobjcount; j++)
				{
					if ((m = ObjWord(temppobjlist[j].obj, wd[i])) != 0)
					{
						if (!bestobj or m<=bestobj)
							{flag++;
							bestobj = m;
							pobj = temppobjlist[j].obj;}
					}
				}
				if (flag != 1)
				{
					/* "You'll have to be more specific...*/
					ParseError(13, 0);
					full_buffer = false;
					return 0;
				}
				full_buffer = false;
			}

RestoreTempArrays:

		/* Restore temporary arrays */

			for (i=1; i<=wtemp; i++)
				wd[i] = wdtemp[i];
			words = wtemp;

		/* Rebuild <buffer> and word[] array */

			strcpy(buffer, "");
			for (i=1; i<=wtemp; i++)
				{strcat(buffer, GetWord(wd[i]));
				strcat(buffer, " ");}
			SeparateWords();

		/* Restore object lists */

			nextobj = objlist[0];
			for (i=0; i<temppobjcount; i++)
				pobjlist[i] = temppobjlist[i];
			pobjcount = temppobjcount;
			for (i=0; i<=tempobjcount; i++)
				objlist[i] = tempobjlist[i];
			objcount = (char)tempobjcount;
			addflag = (char)tempaddflag;

			if (word[*a][0]=='~') (*a)--;
		}
	}

	i = 0;                        /* count nouns */
	k = 10;                       /* best (i.e., lowest) match */
	for (j=1; j<=mobjs; j++)
	{
		if ((m = ObjWord(pobj, mobj[j]))==noun)
		{
			k = noun;
			i++;
		}
		else if (k==noun) i = 2;
	}

	/* Can't use more than one noun for a given object */
	if (i > 1)
	{
		/* "...no such thing" */
		ParseError(5, 0);
		return 0;
	}

	/* Check finally to make sure it's valid */
	if (!ValidObj(pobj)) return false;

	/* Finally--now add it or subtract it from the list, as
	   appropriate:
	*/
	if (addflag==true)
	{
		if (pobj != 0) AddObj(pobj);
	}
	else
		SubtractObj(pobj);

	/* If the object wasn't where it was specifically claimed to be,
	   applies to second pass through object phrase(s) after xobject
	   is found (since <obj_match_state> is 2):
	*/
	if (obj_match_state==2 and domain != 0 and
		Parent(pobj) != domain and pobj != 0 and !speaking)
	{
		if (domain==var[player])
			ParseError(15, pobj);    /* "You don't have any..." */
		else
			ParseError(14, pobj);    /* "You don't see any...there..." */
		return 0;
	}

	if (!strcmp(word[*a], "~except"))
		addflag = false;

	if (strcmp(word[*a], "~and") and strcmp(word[*a], "~except"))
	{
		if (++(*a) > words + 1)
			*a = words + 1;

		/* Check for problems with, for example,
		   "Put dart on table.  Look on table.", where
		   a "You haven't seen any 'on'..." will be
		   generated when the check for an xobject runs
		   past the first "table".  (Trust me.)
		*/
		else if ((obj_match_state==1) and *a < words)
			(*a)--;

		/* At the end yet? */
		if ((*a > words or word[*a][0]=='\0')
			or (obj_match_state != 1 and *a > objfinish))
		{
			if ((objcount > 0 and pobj != 0) or recursive_call or speaking)
				return true;
			else
			{
				/* No objects found */
				strcpy(parseerr, word[1]);
				ParseError(9, 0);   /* "Nothing to (verb)..." */
				return 0;
			}
		}
	}

	/* Go back for the next object phrase */
	pobjcount = 0;
	mobjs = 0;
	strcpy(parseerr, "");

	goto NextLoop;
}


/* MATCHWORD */

int MatchWord(int *a)
{
	char num[18];
	int i, t, flag, finish;
	unsigned int thissyntax, nextsyntax, thisword;

	if (wd[*a]==0) return 0;

	t = Peek(grammaraddr);

	/* the verb ("*") */
	if (t==ASTERISK_T)
	{
		(*a)++;
		AdvanceGrammar();
		return true;
	}

	/* a non-specific dictionary word */
	else if (t==WORD_T)
	{
		if (obj_match_state==1)
			var[xobject] = wd[*a];
		else
			{var[object] = wd[*a];
			obj_match_state = 1;}
		(*a)++;
		AdvanceGrammar();
		return true;
	}

	/* a specific dictionary entry */
	else if (t==DICTENTRY_T)
	{
CheckWord:
		/* matches word */
		if (wd[*a]==PeekWord(grammaraddr + 1))
		{
			(*a)++;
			AdvanceGrammar();
			while (Peek(grammaraddr)==9)
				grammaraddr += 4;
			return true;
		}
		else
		{
			/* if next word is a "/" */
			if (Peek(grammaraddr + 3)==9)
			{
				AdvanceGrammar();       /* this word */
				AdvanceGrammar();       /* "/" */
				goto CheckWord;
			}

			return 0;
		}
	}

	/* alternative dictionary words */
	else if (t==FORWARD_SLASH_T)
		{grammaraddr++;
		return true;}

	/* a number */
	else if (t==NUMBER_T)
	{
		if (!strcmp(itoa(atoi(word[*a]), num, 10), word[*a]))
		{
			if (obj_match_state==1)
				var[xobject] = atoi(word[*a]);
			else
				{var[object] = atoi(word[*a]);
				obj_match_state = 1;}
			object_is_number = true;
			AdvanceGrammar();
			(*a)++;
			return true;
		}
	}

	/* a string enclosed in quotes */
	else if (t==STRING_T)
	{
		if (parsestr[0]=='\"')
		{
			AdvanceGrammar();
			(*a)++;
			return true;
		}
		else
			return 0;
	}

	/* Some manifestation of an object (or objects) before domain is
	   found, since <obj_match_state> is initially set to 0:
	*/
	else if (obj_match_state==0)
	{
		if (Peek(grammaraddr)==HELD_T or Peek(grammaraddr)==MULTIHELD_T)
			odomain = var[player];

		obj_match_state = 1;    /* since next set of object words
					   must be the xobject */
		objstart = (char)*a;
		objgrammar = grammaraddr;

		while (wd[*a] != 0)
		{
			finish = *a;

			/* Check what's coming up in case it's a dictionary
			   word--which would override an object phrase.
			*/
			thissyntax = grammaraddr;
			AdvanceGrammar();
			nextsyntax = grammaraddr;
			grammaraddr = thissyntax;

			/* dictionary word or string */
CheckWordorString:
			if (Peek(nextsyntax)==DICTENTRY_T or Peek(nextsyntax)==STRING_T)
			{
				if ((PeekWord(nextsyntax + 1)==wd[*a]) or (Peek(nextsyntax)==STRING_T and wd[*a]==UNKNOWN_WORD))
				{
					grammaraddr = nextsyntax;
					if (*a != objstart)
						return true;
					else
						return false;
				}
				else if (Peek(nextsyntax + 3)==FORWARD_SLASH_T)
				{
					thissyntax = grammaraddr;
					grammaraddr = nextsyntax + 3;
					AdvanceGrammar();
					nextsyntax = grammaraddr;
					grammaraddr = thissyntax;
					goto CheckWordorString;
				}
			}

			/* or a number */
			if (Peek(nextsyntax)==NUMBER_T and !strcmp(word[*a], itoa(atoi(word[*a]), num, 10)))
			{
				grammaraddr = nextsyntax;
				if (*a != objstart)
					return true;
				else
					return false;
			}

			/* Pass over any object words--they'll be matched
			   specifically later in MatchCommand().
			*/
			flag = 0;
			thisword = FindWord(word[*a]);
			for (i=0; i<objects; i++)
			{
				if (ObjWord(i, thisword))
				{
					flag = 1;
					(*a)++;
					break;
				}

				/* if "~and", "~all",... */
				if (word[*a][0]=='~')
				{
					(*a)++;
					flag = 1;

					/* multi or multi(something) */
					if (Peek(grammaraddr)!=MULTI_T and
						Peek(grammaraddr)!=MULTIHELD_T and
						Peek(grammaraddr) != MULTINOTHELD_T)
					{
						strcpy(parseerr, word[1]);
						/* "You can't...multiple objects." */
						ParseError(3, 0);
						return 2;
					}
				}
			}
			objfinish = (char)finish;

			if (flag==0) return 0;
		}

		AdvanceGrammar();
		return true;
	}

	/* hitting xobject */
	else if (obj_match_state==1)
	{
		/* If the xobject is specifically a parent of the
		   object(s) to be matched later:
		*/
		if (Peek(grammaraddr)==PARENT_T) domain = -1;

		/* Also deal with held xobjects */
		t = domain;
		if (Peek(grammaraddr)==HELD_T or Peek(grammaraddr)==MULTIHELD_T)
			domain = var[player];

		/* Regardless, try to match the xobject */
		recursive_call = false;
		if (MatchObject(&(*a))==0)
		{
			return 2;
		}
		else
		{
			domain = 0;
			if (ValidObj(pobj)==false)
				return 2;
			domain = t;

			var[xobject] = pobj;
			if (domain==-1) domain = pobj; /* parent */
			obj_match_state = 2;

			AdvanceGrammar();

			/* Can't have multiple xobjects */
			if (objcount > 1)
				{ParseError(7, 0);
				return 2;}
			objcount = 0;
			return true;
		}
	}
	return 0;
}


/* OBJWORD

	Returns <adjective> if the word at dictionary address <a> is
	an adjective of <obj>, or <noun> if it is a noun.
*/

int ObjWord(int obj, unsigned int a)
{
	int j, num;
	unsigned int pa;

	pa = PropAddr(obj, adjective, 0);     /* is it an adjective... */
	if (pa)
	{
		defseg = proptable;
		num = Peek(pa + 1);
		for (j=1; j<=num; j++)
		{
			defseg = proptable;
			if (PeekWord(pa + j * 2)==a)
			{
				defseg = gameseg;
				return adjective;
			}
		}
		defseg = gameseg;
	}

	pa = PropAddr(obj, noun, 0);          /* ...or a noun? */
	if (pa)
	{
		defseg = proptable;
		num = Peek(pa + 1);
		for (j=1; j<=num; j++)
		{
			defseg = proptable;
			if (PeekWord(pa + j * 2)==a)
			{
				defseg = gameseg;
				return noun;
			}
		}
	}

	defseg = gameseg;

	return 0;
}


/* PARSE

	Turns word[] into dictionary addresses stored in wd[].  Takes care
	of fingering illegal (unknown) words and doing alterations such
	as compounds, removals, and synonyms.
*/

int Parse()
{
	char foundstring = 0;           /* allow one unknown word/phrase */
	int i, j, k, m;
	char num[33];
	char tempword[81];
	unsigned int period, comma;
	unsigned int synptr;

	period = FindWord(".");
	comma = FindWord(",");

	strcpy(parsestr, "");           /* for storing any unknown string */
	parsed_number = 0;		/*  "     "     "  parsed number  */

	for (i=1; i<=words; i++)        /* find dictionary addresses */
	{
		if (word[i][0]=='\"' and foundstring==0)
			{strcpy(parsestr, word[i]);
			foundstring = 1;
			wd[i] = UNKNOWN_WORD;}
		else
		{
			wd[i] = FindWord(word[i]);

			/* Numbers -32768 to 32767 are valid...*/
			if (!strcmp(word[i], itoa(atoi(word[i]), num, 10)))
			{
#if !defined (MATH_16BIT)
				if (atoi(word[i]) > 32767 or atoi(word[i]) < -32768)
					goto NotinDictionary;
#endif
				parsed_number = atoi(word[i]);
				strcpy(parseerr, word[i]);
			}

			/* Otherwise it must be a dictionary entry */
			else
			{
				/* If it's not in the dictionary */
				if (wd[i]==UNKNOWN_WORD)
				{
NotinDictionary:
					strcpy(parseerr, word[i]);
					strcpy(oops, word[i]);

					/* "...can't use the word..." */
					ParseError(1, 0);
					strcpy(errbuf, "");
					for (i=1; i<=words; i++)
					{
						strcat(errbuf, word[i]);
						if (i != words) strcat(errbuf, " ");
					}
					return 0;
				}
			}
		}
	}
	wd[words+1] = 0;
	oopscount = 0;


	/* Do synonyms, removals, compounds, punctuation */

	for (i=1; i<=words; i++)                /* Look through words... */
	{
		synptr = 2;

		for (j=1; j<=syncount; j++)  /* ...and alterations */
		{
			defseg = syntable;
			if (wd[i]==PeekWord(synptr + 1))
			{
				switch (Peek(synptr))
				{
					case 0:        /* synonym */
					{
						defseg = syntable;
						wd[i] = PeekWord(synptr + 3);
						m = strlen(GetWord(wd[i])) - strlen(word[i]);
						if (m)
						{
							if (m + (int)strlen(buffer) > 81)
								{strcpy(buffer, "");
								words = 0;
								ParseError(0, 0);
								return 0;}

							for (k=words; k>i; k--)
							{
								strcpy(tempword, word[k]);
								word[k] += m;
								strcpy(word[k], tempword);
							}
						}
						strcpy(word[i], GetWord(wd[i]));
						i--;
						break;
					}

					case 1:        /* removal */
					{
						KillWord(i);
						i--;
						break;
					}

					case 2:        /* compound */
					{
						if (wd[i+1]==PeekWord(synptr+3))
						{
							strcat(word[i], word[i+1]);
							wd[i] = FindWord(word[i]);
							KillWord(i+1);
						}
						break;
					}
				}
				goto NextSyn;
			}
NextSyn:
			synptr += 5;
		}

		if (wd[i]==comma)
		{
			if (strcmp(word[i+1], "~and"))
			{
				word[i] = "~and";
				wd[i] = FindWord("~and");
			}
			else
				KillWord(i);
		}

		if (wd[i]==period)
		{
			wd[i] = 0;
			word[i] = "";
		}
	}

	defseg = gameseg;

	if (strcmp(word[1], "~oops")) strcpy(oops, "");
/*      var[words] = w;*/

	if (words==0)
		ParseError(0,0);        /* What? */
		
	return true;
}


/* PARSEERROR */

void ParseError(int e, int a)
{
	int i, k, count;

	remaining = 0;
	xverb = true;

	if (e==5 and !strcmp(parseerr, "")) e = 6;

	if (parseerroraddr)
	{
		ret = 0;
		passlocal[0] = e;
		passlocal[1] = a;
		PassLocals(2);

		SetStackFrame(RESET_STACK_DEPTH, RUNROUTINE_BLOCK, 0, 0);

#if defined (DEBUGGER)
		DebugRunRoutine((long)parseerroraddr*address_scale);
#else
		RunRoutine((long)parseerroraddr*address_scale);
#endif
		stack_depth = 0;
		retflag = 0;
		if (ret)
		{
			if (ret==2) reparse_everything = true;
			return;
		}
	}

	switch (e)
	{
		case 0:
			{AP("What?");
			break;}

		case 1:
			{sprintf(line, "You can't use the word \"%s\".", parseerr);
			AP(line);
			break;}

		case 2:
			{AP("Better start with a verb.");
			break;}

		case 3:
			{sprintf(line, "You can't %s multiple objects.", parseerr);
			AP(line);
			break;}

		case 4:
			{AP("Can't do that.");
			break;}

		case 5:
			{sprintf(line, "You haven't seen any \"%s\", nor are you likely to in the near future even if such a thing exists.", parseerr);
			AP(line);
			break;}

		case 6:
			{AP("That doesn't make any sense.");
			break;}

		case 7:
			{AP("You can't use multiple objects like that.");
			break;}

		case 8:
		{
			sprintf(line, "Which %s do you mean, ", parseerr);
			count = 1;
			for (k=0; k<pobjcount; k++)
			{
				i = pobjlist[k].obj;

				if (strcmp(Name(i), ""))
				{
					if (count==pobjcount)
					{
						if (count > 2) strcat(line, ",");
						strcat(line, " or ");
					}
					else
					{
						if (count != 1)
							strcat(line, ", ");
					}
					if (GetProp(i, article, 1, 0))
						strcat(line, "the ");
					strcat(line, Name(i));
					count++;
				}
			}
			strcat(line, "?");
			AP(line);
			break;
		}

		case 9:
			{sprintf(line, "Nothing to %s.", parseerr);
			AP(line);
			break;}

		case 10:
			{AP("You haven't seen anything like that.");
			break;}

		case 11:
			{AP("You don't see that.");
			break;}

		case 12:
			{sprintf(line, "You can't do that with the %s.", Name(a));
			AP(line);
			break;}

		case 13:
			{AP("You'll have to be a little more specific.");
			break;}

		case 14:
			{AP("You don't see that there.");
			break;}

		case 15:
			{AP("You don't have that.");
			break;}

		case 16:
			{AP("You'll have to make a mistake first.");
			break;}

		case 17:
			{AP("You can only correct one word at a time.");
			break;}
	}
}


/* REMOVEWORD

	Deletes wd[a].
*/

void RemoveWord(int a)
{
	if (a > words) return;

	for (; a<words; a++)
		wd[a] = wd[a + 1];
	wd[words] = 0;
}


/* SEPARATEWORDS

	Splits <buffer> into the word[] array.  Also does nifty things
	such as turning time values such as hh:mm into a single number
	(representing minutes from midnight).
*/

void SeparateWords(void)
{
	char inquote = 0;
	char a[1025];
	char b[2];
	char w1[17], w2[17];            /* for time conversions */
	char temp[17];
	short n1, n2;                   /* must be 16 bits */
	int bloc = 0;                   /* buffer location */
	int i;


	/* First filter the line of any user-specified punctuation */
	do
	{
		i = strcspn(buffer, punc_string);
		if (buffer[i]) buffer[i] = ' ';
	} while (buffer[i]);


	/* Begin the word-splitting proper: */
	
	words = 1;                      /* Setup a blank string */

	for (i=0; i<MAXWORDS+1; i++)
	{
		word[i] = "";
		wd[i] = 0;
	}
	word[1] = buffer;

	strcpy(a, buffer);
	strcpy(buffer, "");
	
	for (i=1; i<=(int)strlen(a); i++)
	{
		if (inquote!=1 and isascii(a[i-1]))
			b[0] = (char)tolower(a[i-1]);
		else b[0] = a[i-1];
		b[1] = '\0';

		if (b[0]=='\"' and inquote==1)
			{strcpy(buffer+bloc, b);
			bloc++;
			inquote++;}

		if (b[0]=='\"' or ((b[0]==' ' or b[0]=='!' or b[0]=='?') and inquote!=1))
		{
			if (word[words][0]!='\0')
			{
				bloc++;
				if (++words > MAXWORDS) words = MAXWORDS;
				word[words] = buffer + bloc;
				strcpy(word[words], "");
			}

			if (b[0]=='\"' and inquote==0)
				{strcpy(buffer+bloc, b);
				bloc++;
				inquote = 1;}
		}
		else
		{
			if ((b[0]=='.' or b[0]==',') and inquote!=1)
			{
				if (word[words][0]!='\0')
				{
					bloc++;
					if (++words > MAXWORDS) words = MAXWORDS;
				}
				word[words] = buffer + bloc;
				strcpy(word[words], b);
				bloc += strlen(b) + 1;
				if (++words > MAXWORDS) words = MAXWORDS;
				word[words] = buffer + bloc;
				strcpy(word[words], "");
			}
			else
				{strcpy(buffer+bloc, b);
				bloc++;}
		}
	}

	if (!strcmp(word[words], "")) words--;

	for (i=1; i<=words; i++)
	{
		/* Convert hours:minutes time to minutes only */
		if (strcspn(word[i], ":")!=strlen(word[i]) and strlen(word[i])<=5)
		{
			strcpy(w1, Left(word[i], strcspn(word[i], ":")));
			strcpy(w2, Right(word[i], strlen(word[i]) - strcspn(word[i], ":") - 1));
			n1 = (short)atoi(w1);
			n2 = (short)atoi(w2);

			if (!strcmp(Left(w2, 1), "0"))
				strcpy(w2, Right(w2, strlen(w2) - 1));

			/* If this is indeed a hh:mm time, write it back
			   as the modified word, storing the original hh:mm
			   in parse$:
			*/
			if (!strcmp(w1, itoa((int)n1, temp, 10)) and !strcmp(w2, itoa((int)n2, temp, 10)) and (n1 > 0 and n1 < 25) and (n2 >= 0 and n2 < 60))
			{
				strcpy(parseerr, word[i]);
				itoa(n1 * 60 + n2, word[i], 10);
			}
		}
	}
}


/* SUBTRACTOBJ

	Removes object <obj> from objlist[], making all related adjustments.
*/

void SubtractObj(int obj)
{
	int i, j;

	for (i=0; i<objcount; i++)
	{
		if (objlist[i]==obj)
		{
			for (j=i; j<objcount; j++)
				objlist[j] = objlist[j+1];
			objcount--;
			return;
		}
	}
}


/* SUBTRACTPOBJECT

	Removes <obj> as a possible contender for object disambiguation.
*/

void SubtractPObject(int obj)
{
	int i, j, last = 0;

	for (i=0; i<pobjcount; i++)
	{
		if (pobjlist[i].obj==obj)
		{
			if (pobjlist[i].obj==pobj and last!=0) pobj = last;

			for (j=i; j+1<pobjcount; j++)
			{
				pobjlist[j] = pobjlist[j+1];
			}
			pobjcount--;

			return;
		}
		else last = pobjlist[i].obj;
	}
}


/* TRYOBJ

	Called by MatchObject() to see if <obj> is available, and add it to
	or subtract it from objlist[] accordingly.
*/

void TryObj(int obj)
{
	unsigned int tempdomain;

	if (DomainObj(obj))
	{
		tempdomain = domain;
		domain = 0;

		if (Available(obj, 0) and !InList(Parent(obj)))
			AddObj(obj);
		else
			SubtractObj(obj);

		domain = tempdomain;
	}
}


/* VALIDOBJ

	Checks first of all to see if an object is available, then checks
	if it meets all the qualifications demanded by the grammar syntax.
*/

int ValidObj(int obj)
{
	int attr, nattr = 0;
	unsigned int addr;

	defseg = gameseg;

	if (!Available(obj, 0) and !speaking and
		(Peek(grammaraddr)!=OPEN_BRACKET_T or
		Peek(grammaraddr+1)!=ROUTINE_T))
	{
		if (Peek(grammaraddr)==ANYTHING_T)
			ParseError(10, obj);    /* "...haven't seen..." */
		else
			ParseError(11, obj);    /* "...don't see any..." */
		return 0;
	}

	switch (Peek(grammaraddr))
	{
		case OPEN_BRACKET_T:
		{
			if (Peek(grammaraddr+1)==ROUTINE_T)
			{
				addr = PeekWord(grammaraddr+2);
				ret = 0;
				passlocal[0] = obj;
				PassLocals(1);

				SetStackFrame(RESET_STACK_DEPTH, RUNROUTINE_BLOCK, 0, 0);

#if defined (DEBUGGER)
				DebugRunRoutine((long)addr*address_scale);
#else
				RunRoutine((long)addr*address_scale);
#endif
				retflag = 0;

				/* If the routine doesn't return true,
				   the object doesn't qualify.
				*/
				if (!ret) return (0);
			}
			else if (Peek(grammaraddr+1)==OBJECTNUM_T)
			{
				if (obj != (int)PeekWord(grammaraddr+2))
				{
					strcpy(parseerr, "");
					if (GetProp(obj, article, 1, 0))
						strcpy(parseerr, "the ");
					strcat(parseerr, Name(obj));

					/* "...can't do that with..." */
					ParseError(12, obj);
					return 0;
				}
			}
			break;
		}

		case ATTR_T:
		case NOT_T:
		{
			if (Peek(grammaraddr)==NOT_T) nattr = 1;
			attr = Peek(grammaraddr + 1 + nattr);

			/* If the attribute match is not made,
			   the object doesn't qualify.
			*/
			if (!TestAttribute(obj, attr, nattr))
			{
				strcpy(parseerr, "");
				if (GetProp(obj, article, 1, 0))
					strcpy(parseerr, "the ");
				strcat(parseerr, Name(obj));

				/* "...can't do that with..." */
				ParseError(12, obj);
				return 0;
			}
			break;
		}
	}
	return true;
}

