/*
	STRINGFN.C

	String manipulation functions

	Copyright (c) 1995-2001 by Kent Tessman
*/


#include <stdio.h>
#include <string.h>
#include <ctype.h>

char *Left(char *a, int l);
char *Ltrim(char *a);
char *Mid(char *a, int pos, int n);
char *Right(char *a, int l);
char *Rtrim(char *a);

static char *GetTempString(void);

#if !defined (QUICKC)
char *itoa(int a, char *buf, int base);
char *strlwr(char *s);
char *strnset(char *s, int c, size_t l);
char *strupr(char *s);
#endif


/* GETTEMPSTRING

	Prevents having to make sure each string-returning function has its
	own static char array to copy into.  In other words, for five
	functions, we don't need 5 * 1K each of static char array.

	NOTE:  Assumes that no more than NUM_TEMPSTRINGS nested
	string-manipulations will be done at once.
*/

#define NUM_TEMPSTRINGS 2
static char tempstring[NUM_TEMPSTRINGS][1025];
static char tempstring_count = 0;

static char *GetTempString(void)
{
	static char *r;

	r = &tempstring[(int)tempstring_count][0];
	if (++tempstring_count >= NUM_TEMPSTRINGS) tempstring_count = 0;

	return r;
}


/* The following string-manipulation functions closely mimic BASIC-language
   string functionality.   They do not alter the provided string; instead,
   they return a pointer to a static (modified) copy.
*/


/* LEFT */

char *Left(char a[], int l)
{
	static char *temp;
	int i;

	temp = GetTempString();

	if (l > (int)strlen(a))
		l = strlen(a);
	for (i = 0; i<l; i++)
		temp[i] = a[i];
	temp[i] = '\0';
	return temp;
}


/* LTRIM */

char *Ltrim(char a[])
{
	static char *temp;

	temp = GetTempString();

	strcpy(temp, a);
	while (temp[0]==' ' || temp[0]=='\t')
		strcpy(temp, Right(temp, strlen(temp)-1));
	return temp;
}


/* MID */

char *Mid(char a[], int pos, int n)
{
	static char *temp;
	int i;

	temp = GetTempString();

	pos--;
	if (pos+n > (int)strlen(a))
		n = strlen(a)-pos;
	for (i = 0; i<n; i++)
		temp[i] = a[pos+i];
	temp[i] = '\0';
	return temp;
}


/* RIGHT */

char *Right(char a[], int l)
{
	static char *temp;
	int i;

	temp = GetTempString();

	if (l > (int)strlen(a))
		l = strlen(a);
	for (i = 0; i<l; i++)
		temp[i] = a[strlen(a)-l+i];
	temp[i] = '\0';
	return temp;
}


/* RTRIM */

char *Rtrim(char a[])
{
	static char *temp;
	int len;

	temp = GetTempString();

	strcpy(temp, a);
	while (((len = strlen(temp))) && (temp[len-1]==' ' || temp[len-1]=='\t'))
		strcpy(temp, Left(temp, len-1));
	return temp;
}


#if !defined (QUICKC)

/* Because the following functions are part of the standard QuickC
   library, but they don't seem too common elsewhere.  Thanks to
   David Kinder for these handy replacements.
*/

char *itoa(int a, char *buf, int base)
{
	/* This only works if base is 10 (which it will be) */

	sprintf(buf, "%d" ,a);
	return buf;
}

char *strlwr(char *s)
{
	int i;

	i = 0;
	while (*(s+i)!='\0')
	{
		*(s+i) = tolower(*(s+i));
		i++;
	}
	return s;
}

char *strnset(char *s, int c, size_t l)
{
	int i;

	for (i=0; i<(int)l; i++) *(s+i) = (char)c;
	return s;
}

char *strupr(char *s)
{
	int i;

	i = 0;
	while (*(s+i)!='\0')
	{
		*(s+i) = toupper(*(s+i));
		i++;
	}
	return s;
}

#endif
