/* Makegray V1.4 */
/* Earle F. Philhower, III */
/* earle@ziplabel.com */
/* http://www.ziplabel.com/helio */
/* To be used w/grayscale.app */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <windows.h>

void MakeOneZHDB(char *recordName, char *owner, int size, unsigned char *data, char *outFile);
void LoadBMP2Gray( char *filename, int *xsize, int *ysize, unsigned char **data);
void PackGray2LCD( int x, int y, unsigned char *data, unsigned char **pack, unsigned int *size );
void MakeOneHDB(char *recordName, char *owner, int size, unsigned char *data, char *outFile);
void ScaleImage(unsigned char *in, unsigned char **out, int xin, int yin, int *xout, int *yout);
void LoadJPG2Gray( char *filename, int *xsize, int *ysize, unsigned char **data);
unsigned char *Normalize(int size, unsigned char *source);
unsigned char *Dither(int xsize, int ysize, unsigned char *source);

void err()
{
	printf("MAKEGRAY v1.5:                    Earle F. Philhower, III - earle@ziplabel.com\n");
	printf("Converts JPEG or BMP images to GrayscaleFile2 ZHDBs for use with GRAYSCAL.APP.\n\n");
	printf("usage: makegray [-nobig] image.[bmp/jpg] imagename image.hdb\n");
	printf("   or: makegray [-nobig] image.[bmp/jpg]\n");
	exit(-1);
}

int main(int argc, char **argv)
{
	int x, y, xscale, yscale;
	unsigned char *data, *scale, *pack, *packscale;
	unsigned int size, sizescale;
	char *infile, *desc, *outfile;
	char locdesc[64], locout[MAX_PATH+1];
	int i, j;
	int nobig;

	// See if -nobig specified
	nobig = 0;
	if (argc>1) {
		if (stricmp("-nobig", argv[1])==0) {
			nobig = 1;
			for (i=0; i<argc-1; i++)
				argv[i] = argv[i+1];
			argc--;
		}
	}

	if (argc==2) {
		infile = argv[1];
		strcpy(locout, infile);
		for (i=strlen(locout)-1; i>0; i--) if (locout[i]=='.') break;
		if (i==0) err();
		locout[i+1] = 0;
		strcat(locout, "hdb");
		for (j=i; i>0; i--)  if (locout[i]=='\\') break;
		if (locout[i]=='\\') i++;
		memset(locdesc, 0, 64);
		memcpy(locdesc, locout+i, j-i);
		outfile = locout;
		desc = locdesc;
	} else if (argc==4) {
		infile = argv[1];
		desc = argv[2];
		outfile = argv[3];
	} else
		err();

	if (nobig) printf("Will not generate zoomed image\n");
	printf("Input BMP: %s\nDescription: %s\nOutput HDB: %s\n", infile, desc, outfile);
	if (stricmp(".jpg", infile+strlen(infile)-4)==0)
		LoadJPG2Gray( infile, &x, &y, &data );
	else
		LoadBMP2Gray( infile, &x, &y, &data );
	printf("Input BMP Size: %d x %d\n", x, y);

	if (nobig) {
		ScaleImage( data, &scale, x, y, &xscale, &yscale );
		x = xscale;
		y = yscale;
		data = scale;
	}
	PackGray2LCD( x, y, data, &pack, &size );
	
	ScaleImage( data, &scale, x, y, &xscale, &yscale );
	printf("Scaled Size: %d x %d\n", xscale, yscale);
	PackGray2LCD( xscale, yscale, scale, &packscale, &sizescale );
	
	data = malloc(size+sizescale);
	memcpy(data, pack, size);
	memcpy(data+size,packscale, sizescale);

	MakeOneZHDB(desc, "GrayscaleFile2", size+sizescale, data, outfile);

}			

void LoadBMP2Gray( char *filename, int *xsize, int *ysize, unsigned char **data)
{
	FILE *in;
	BITMAPFILEHEADER bmfh;
	BITMAPINFOHEADER bmih;
	unsigned char palette[256][4];
	char *rgb, *ptr;
	int x, y, c;
	int off;

	in = fopen(filename, "rb");
	if (in==NULL) {printf("*** ERROR *** Unable to open %s\n", filename); exit(-1); }
	fread(&bmfh, sizeof(BITMAPFILEHEADER), 1, in);
	fread(&bmih, sizeof(BITMAPINFOHEADER), 1, in);
	switch (bmih.biBitCount) {
		case 1: printf("Unable to read BnW images\n"); exit(-1);
		case 4: printf("Unable to read 16c images\n"); exit(-1);
		case 8: fread(palette, 256, 4, in); break;
	}
	ptr = rgb = malloc( bmih.biWidth*bmih.biHeight );
	fseek(in, bmfh.bfOffBits, SEEK_SET);
	off = 0;

	ptr = rgb + bmih.biWidth*(bmih.biHeight-1);
	for (y=0; y<bmih.biHeight; y++) {
		for (x=0; x<bmih.biWidth; x++) {
			switch (bmih.biBitCount) {
			case 8:
				c=fgetc(in); off++;
				*(ptr++) = (palette[c][0]+palette[c][1]+palette[c][2])/3;
				break;
			case 24:
				*(ptr++) = (fgetc(in)+fgetc(in)+fgetc(in))/3;
				off += 3;
				break;
			}
		}
		ptr -= 2*bmih.biWidth;
		while (off%sizeof(DWORD)) {
			fgetc(in);
			off++;
		}
	}

	*xsize = bmih.biWidth;
	*ysize = bmih.biHeight;
	*data = rgb;
}


unsigned char *Normalize(int size, unsigned char *source)
{
	int i;
	unsigned char *data;
	unsigned char min, max, elem;
	double scale;

	data = malloc(size);
	// First find normalization range
	min = 255; max = 0;
	for (i=0; i<size; i++) {
		if (source[i]>max) max = source[i];
		if (source[i]<min) min = source[i];
	}
	scale = 255.0/((double)(max-min));

	for (i=0; i<size; i++) {
		elem = source[i];
		elem = (unsigned char)floor((((double)elem)-min)*scale);
		data[i] = elem;
	}

	return data;
}




/* Floyd/Steinberg error diffusion dithering algorithm in color.  The array
** line[][] contains the RGB values for the current line being processed;
** line[0][x] = red, line[1][x] = green, line[2][x] = blue.  It uses the
** external functions getline() and putdot(), whose purpose should be easy
** to see from the code.
*/

unsigned char gray[16] = {0xf0,0xe0,0xd0,0xc0,0xb0,0xa0,0x90,0x80,0x70,0x60,0x50,0x40,0x30,0x20,0x10,0x00};


unsigned char *Dither(int xsize, int ysize, unsigned char *source)
{
    static int *ed; //[WIDTH] = {0};      /* Errors distributed down, i.e., */
	/* to the next line.              */
    int x, y, h, c, nc, v,              /* Working variables              */
        e[4],                           /* Error parts (7/8,1/8,5/8,3/8). */
        ef;                          /* Error distributed forward.     */
    long dist, sdist;                   /* Used for least-squares match.  */
	unsigned char *line;
	unsigned char *dst;
	
	dst = malloc(xsize*ysize);
	line = malloc(xsize);
	ed = malloc(xsize*sizeof(int));
    for (x=0; x<xsize; ++x) {
        ed[x] =0;
    }
    y = 0;                              /* Get one line at a time from    */
    while (y<ysize) {          /* input image.                   */
		memcpy(line, source+y*xsize, xsize);
		
        ef = 0;      /* No forward error for first dot */
		
        for (x=0; x<xsize; ++x) {
			v = line[x] + ef + ed[x];  /* Add errors from    */
			if (v < 0) v = 0;                   /* previous pixels    */
			if (v > 255) v = 255;               /* and clip.          */
			line[x] = v;
			
			sdist = 255L * 255L * 255L + 1L;        /* Compute the color  */
			for (c=0; c<15; ++c) {              /* in colormap[] that */
				/* is nearest to the  */
				/* corrected color.   */
				
				dist = (line[x]-gray[c]);
				dist = dist*dist;
				if (dist < sdist) {
					nc = c;
					sdist = dist;
				}
			}
			dst[y*xsize+x] = nc;
			//			putdot(x, y, nc);           /* Nearest color found; plot it.  */
			
			v = line[x] - gray[nc];   /* V = new error; h = */
			h = v >> 1;                         /* half of v, e[1..4] */
			e[1] = (7 * h) >> 3;                /* will be filled     */
			e[2] = h - e[1];                    /* with the Floyd and */
			h = v - h;                          /* Steinberg weights. */
			e[3] = (5 * h) >> 3;
			e[4] = h = e[3];
			
			ef = e[1];                       /* Distribute errors. */
			if (x < xsize-1) ed[x+1] = e[2];
			if (x == 0) ed[x] = e[3]; else ed[x] += e[3];
			if (x > 0) ed[x-1] += e[4];
			
		} /* next x */
		
		++y;
	} /* next y */
	
	return dst;
}





void PackGray2LCD( int x, int y, unsigned char *data, unsigned char **pack, unsigned int *size )
{
	int i,j;
	int linesize;
	unsigned char *ptr;
	unsigned char elem, tdat;


	data = Normalize( x*y, data );
	data = Dither(x, y, data);

	// How many words per line
	linesize = (int)(ceil(x/8.0));
	*size=y*linesize*4+8;
	*pack = ptr = malloc(*size);

	*(ptr++) = (linesize>>24)&255;
	*(ptr++) = (linesize>>16)&255;
	*(ptr++) = (linesize>>8)&255;
	*(ptr++) = (linesize>>0)&255;
	*(ptr++) = (y>>24)&255;
	*(ptr++) = (y>>16)&255;
	*(ptr++) = (y>>8)&255;
	*(ptr++) = (y>>0)&255;

	for (i=0; i<y; i++) {
		for (j=0; j<x; j++) {
			elem = data[i*x + j];
			elem &= 0x0f;
			if ((j&1)==0)
				tdat = elem;
			else {
				tdat = (tdat<<4) | (elem<<0);
				*(ptr++) = tdat;
			}
		}
		// Clean out remaining
		if (j%1) { *(ptr++) = elem; j++; }
		for (; j<(linesize*8); j+=2) *(ptr++) = 0;
	}
}


void ScaleImage(unsigned char *in, unsigned char **out, int xin, int yin, int *xout, int *yout)
{
	int vx,vy;
	unsigned char vcol;
	double rx,ry;
	double width_scale, height_scale;
	double gray;
	int   i,j;
	int half_square_width, half_square_height;
	double round_width, round_height;
	unsigned char *buff;

	if (xin<=160 && yin<=160) {
		*out = in;
		*xout = xin;
		*yout = yin;
		return;
	} else if (xin>yin) {
		double scale;
		scale = 160.0/((double)xin);
		*xout = (int)(xin*scale);
		*yout = (int)(yin*scale);
	} else {
		double scale;
		scale = 160.0/((double)yin);
		*xout = (int)(xin*scale);
		*yout = (int)(yin*scale);
	}
	*out = buff = malloc((*xout) * (*yout));
	width_scale  = (double)xin  / (double)*xout;
	height_scale = (double)yin / (double)*yout;

	half_square_width  = (int)(width_scale  / 2.0);
	half_square_height = (int)(height_scale / 2.0);
	round_width  = (width_scale  / 2.0) - (double)half_square_width;
	round_height = (height_scale / 2.0) - (double)half_square_height;
	if(round_width  > 0.0)
		half_square_width++;
	else
	    round_width = 1.0;
	if(round_height > 0.0)
		half_square_height++;
	else
		round_height = 1.0;

	for(vy = 0;vy < *yout; vy++)  
	{
		for(vx = 0;vx < *xout; vx++)
		{
			rx = vx * width_scale;
			ry = vy * height_scale;
			vcol = in[(int)rx +(int)ry*xin];
		
			gray = 0.0;

			for(j=0;j<half_square_height*2;j++)
			{
				for(i=0;i<half_square_width*2;i++)
				{
					int idx, idy;
					idx = ((int)rx)-half_square_width+i;
					idy = ((int)ry)-half_square_height+j;
					if (idx<0) idx=0; if (idx>xin-1) idx=xin-1;
					if (idy<0) idy=0; if (idy>yin-1) idy=yin-1;
					vcol = in[idx+xin*idy];
					if(((j == 0) || (j == (half_square_height*2)-1)) && 
						((i == 0) || (i == (half_square_width*2)-1))) {
						gray += round_width*round_height*(double)(vcol);
					} else if((j == 0) || (j == (half_square_height*2)-1)) {
						gray += round_height*(double)(vcol);
					} else if((i == 0) || (i == (half_square_width*2)-1)) {
						gray += round_width*(double)(vcol);
					} else {
						gray += (double)(vcol);
					}
				}
			}
      
			gray /= width_scale*height_scale;
	      
			vcol = (unsigned char)((gray > 255.0)? 255.0 : ((gray< 0.0)? 0.0:gray));
	    
			buff[vx+vy*(*xout)] = vcol;
	    }
	}
}
