#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mount.h>

FILE *console;

int mount_root(const char *source, const char *fstype) {
    char cmd[256];

    snprintf(cmd, sizeof(cmd), "mount -n -t %s %s /mnt", fstype, source);
    if (system(cmd))
	return -1;

    if (chdir("/mnt") < 0) {
	perror("chdir");
	umount("/mnt");
	return -1;
    }

    snprintf(cmd, sizeof(cmd), "mount -n -t %s -o move . /", fstype);
    if (system(cmd)) {
	chdir("/");
	umount("/mnt");
	return -1;
    }

    if (chroot(".") < 0) {
	perror("chroot");
	exit(1);
    }

    return 0;
}

void boot(void) {
    execl("/sbin/init", "init", NULL);
    fprintf(console, "Failed to start /bin/init.  Do Panic.");
    perror("/sbin/init");

    exit(1);
}

struct choice {
    const char *text;
    int keycode;
    const char *device;
    const char *fstype;
} choices [] = {
    { "1. Ramdisk", 0x00ac, NULL, NULL },
    { "2. Internal Flash", 0x018d, "/dev/mtdblock3", "jffs2" },
    { "3. MMC partition 2", 0x0098, "/dev/mmcblock0p2", "ext3" },
    { "4. MMC partition 3", 0x009b, "/dev/mmcblock0p3", "ext3" },
    { 0 },
};

int get_button_device() {
    int i;
    int fd;
    int n;
    char fn[256];
    char buf[256];
    char *p;
    int major, minor;

    for (i = 0; ; i++) {
	snprintf(fn, sizeof(fn), "/sys/class/input/input%d/name", i);
	if ((fd = open(fn, O_RDONLY)) == -1)
	    return -1;

	if ((n = read(fd, buf, sizeof(buf)-1)) < 0) {
	    perror(fn);
	    close(fd);
	    continue;
	} 

	close(fd);

	while (n > 1 && isspace(buf[n-1]))
	    n--;
	buf[n] = '\0';
	printf("%s\n", buf);
	if (strcmp(buf, "s3c2410-buttons"))
	    continue;

	printf("event device %s\n", fn);

	snprintf(fn, sizeof(fn), "/sys/class/input/input%d/event%d/dev", i, i);
	if ((fd = open(fn, O_RDONLY)) == -1) {
	    perror(fn);
	    continue;
	}

	if ((n = read(fd, buf, sizeof(buf)-1)) < 0) {
	    perror(fn);
	    close(fd);
	    continue;
	} 

	close(fd);

	buf[n] = '\0';

	if ((p = strchr(buf, ':')) == NULL) {
	    fprintf(stderr, "bad device specification in %s: %s\n", fn, buf);
	    continue;
	}

	major = atoi(buf);
	minor = atoi(p + 1);

	sprintf(buf, "mknod /tmp/buttons c %d %d\n", major, minor);
	if (system(buf))
	    continue;

	if ((fd = open("/tmp/buttons", O_RDONLY)) < 0) {
	    perror("event");
	    continue;
	}

	unlink("/tmp/buttons");

	return fd;
    }

    return -1;
}

int main()
{
    int event_fd;
    char event_buf[16];
    int n;
    int i;

    if ((console = fopen("/dev/tty1", "w")) < 0) {
	perror("tty");
	exit(1);
    }
    fcntl(fileno(console), F_SETFD, FD_CLOEXEC);

    system("stty -F /dev/tty1 cooked -echo");

    system("mount -n -t sysfs none /sys");

    if ((event_fd = get_button_device()) < 0) {
	fprintf(stderr, "failed to open button device, using ramdisk\n");
	boot();
    }
    fcntl(event_fd, F_SETFD, FD_CLOEXEC);

    umount("/sys");


retry:
    fprintf(console, "\n");
    fprintf(console, "Choose a root file system\n");
    fprintf(console, "\n");
    for (i = 0; choices[i].text; i++) {
	fprintf(console, "%s", choices[i].text);
	if (choices[i].fstype)
	    fprintf(console, " (%s)", choices[i].fstype);
	fprintf(console, "\n");
    }
    fprintf(console, "\n");

    while (1) {
	n = read(event_fd, event_buf, sizeof(event_buf));

	if (((short *)event_buf)[6]) {
	    int keycode = ((short *)event_buf)[5];
	    printf("keypress 0x%04x\n", keycode);

	    for (i = 0; choices[i].text; i++) {
		if (choices[i].keycode == keycode) {
		    if (choices[i].device) {
			fprintf(console, "mount -t %s %s\n", choices[i].fstype, choices[i].device);
			if (mount_root(choices[i].device, choices[i].fstype) < 0) {
			    fprintf(console, "Failed to mount root device.\n");
			    fprintf(console, "Try again.\n");
			    goto retry;
			}
		    }
		    
		    fprintf(console, "Starting init...\n");
		    
		    boot();

		    exit(1);
		}
	    }
	}
    }
}
 
/*
  Local variables:
  compile-command: "make -k "
  End:
*/

