/*
 * Copyright (c) 2005 Christer Weinigel <christer@weinigel.se>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This is a Linux Boot Loader for Windows CE running on a
 * Samsung s3c2410 processor.
 *
 * This file must be compiled with -O2 or it will fail in mysterious
 * ways.  It seems that the wince gcc is buggy as hell.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include "build.h"

#include "winhack.h"

typedef unsigned long u32;
typedef unsigned short u16;
typedef unsigned char u8;

#include "setup.h"

unsigned long ram_addr = 0x30000000;
unsigned long ram_size = 64 * 1024 * 1024;
unsigned long tag_addr = ~0;
unsigned long initrd_addr = ~0;
unsigned long kernel_addr = ~0;

void *allocate_phys(unsigned long size, unsigned long *pphys) {
    unsigned long phys;
    LPVOID virt;

    /* Try to allocate pages until we have an address above the low
     * limit.  The reason for doing this is that the kernel expects to
     * reside at the beginning of RAM and we have to copy it there
     * after we have killed off WinCE.  It's much easier to do the
     * copy if we are sure that the source and destination wont't
     * overlap, so be a bit lazy here.  We could make this a bit
     * smarter though, if we check the size of the initrd+kernel
     * before starting the copy, we could change the below into
     * ram_addr + 0x8000 + kernel_size + initrd_size */
    unsigned low_limit = ram_addr + ram_size/2;

    do {
	virt = AllocPhysMem(size, PAGE_EXECUTE_READWRITE, 0, 0, &phys);
	if (0)
	    printf("virt: %p phys: 0x%08lx  size: 0x%lx\n", virt, phys, size);

	if (!virt) {
	    fprintf(stderr, "Failed to AllocPhysMem: 0x%x\n", (unsigned)GetLastError());
	    exit(1);
	}
    } while (phys < low_limit);

    if (pphys)
	*pphys = phys;

    return virt;
}

/*
 * dest -- physical destination address
 * src -- physical source address
 * count -- number of bytes to copy
 */
void add_dir_entry(unsigned long dest, unsigned long src, unsigned long count,
		   unsigned long **pdir)
{
    unsigned long *dir = *pdir;

    if (0)
	printf("add_dir_entry(0x%08lx, 0x%08lx, 0x%08lx, %p, %p)\n", dest, src, count, pdir, *pdir);

    *dir++ = (count + 3) >> 2;
    *dir++ = src;
    *dir++ = dest;

    *pdir = dir;
}

/**
 * load a file into physical memory and fill in the directory
 */
unsigned load_file(const char *fn, unsigned long dest_phys,
		   unsigned long **pdir)
{
    FILE *fp;
    unsigned n;
    void *src;
    unsigned long src_phys;
    unsigned long size;

    /* no use in having anything less than 64k here, we'll just waste
     * virtual address space if we do */
    unsigned long block_size = 64 * 1024;

    fp = fopen(fn, "r");
    if (!fp) {
	perror(fn);
	exit(1);
    }

    size = 0;

    while (1) {
	src = allocate_phys(block_size, &src_phys);

	n = fread(src, 1, block_size, fp);
	if (ferror(fp)) {
	    perror(fn);
	    exit(1);
	}
	if (n == 0)
	    break;

	add_dir_entry(dest_phys, src_phys, n, pdir);
	size += n;
	dest_phys += n;
    }

    fclose(fp);

    return size;
}

void *create_tags(void *ptr, const char *cmdline,
		 unsigned long initrd_addr, unsigned initrd_size)
{
    struct tag *tag;

    tag = (struct tag *)ptr;

    /* start tags */
    tag->hdr.tag = ATAG_CORE;
    tag->hdr.size = tag_size(tag_core);
    tag->u.core.flags = 0;
    tag->u.core.pagesize = 0x00001000;
    tag->u.core.rootdev = 0x00ff;
    tag = tag_next(tag);

    /* command line */
    tag->hdr.tag = ATAG_CMDLINE;
    /* Add 1 for the \0 and 3 more to round up the the shift */
    tag->hdr.size = (strlen(cmdline) + 1 + sizeof(struct tag_header) + 3) >> 2;
    strcpy(tag->u.cmdline.cmdline, cmdline);
    tag = tag_next(tag);

    /* RAM */
    tag->hdr.tag = ATAG_MEM;
    tag->hdr.size = tag_size(tag_mem32);
    tag->u.mem.size = ram_size;
    tag->u.mem.start = ram_addr;
    tag = tag_next(tag);

    /* initrd */
    if (initrd_addr && initrd_size) {
	tag->hdr.tag = ATAG_INITRD2;
	tag->hdr.size = tag_size(tag_initrd);
	tag->u.initrd.start = initrd_addr;
	tag->u.initrd.size = initrd_size;
	tag = tag_next(tag);
    }

    tag->hdr.tag = ATAG_VIDEOTEXT;
    tag->hdr.size = tag_size(tag_videotext);
    tag->u.videotext.video_lines = 40;
    tag->u.videotext.video_cols = 48;
    tag = tag_next(tag);

    /* no more tags */
    tag->hdr.tag = ATAG_NONE;
    tag->hdr.size = 0;
    tag = tag_next(tag);

    return tag;
}

void dump_dir(void *dir)
{
    unsigned long *p = dir;

    while (*p) {
	unsigned long count = *p++;
	unsigned long src = *p++;
	unsigned long dest = *p++;

	printf("entry(0x%08lx, 0x%08lx, 0x%08lx)\n", dest, src, count);
    }
}

#define PAGE_SIZE (64 * 1024)

unsigned page_round_up(unsigned u) {
    return (u + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1);
}

void disable_interrupts();

/**
 * Disable the MMU and jump to phys_addr.
 */
void enter_physical(unsigned long phys_addr);

static const char *search_paths[] = {
    "/My Documents",
    "/SD-MMC Card",
    "/Storage Card",
    NULL
};

static const char *find_file(const char *fn, const char **pdir)
{
    int i;
    char *path;

    for (i = 0; search_paths[i]; i++) {
	asprintf(&path, "%s/%s", search_paths[i], fn);
	printf("looking for: %s\n", path);
	if (!access(path, R_OK)) {
	    if (pdir)
		*pdir = search_paths[i];
	    return path;
	}
	free(path);
    }

    return NULL;
}

#define DEFAULT_FN "default.txt"
#define KERNEL_FN "zImage"

static const char *basedir = NULL;
static int default_line;
static const char *default_path = NULL;

static int mtype = 0;
static const char *kernel_path = NULL;
static const char *initrd_path = NULL;
static const char *cmdline = "root=/dev/ram rw keepinitrd";
static int dry_run = 1;

/**
 * Get the next token from a line.  A token is either a white space
 * delimited word or a quoted string on the form "foo bar".
 */
char *next_token(char **ptr) {
    char *p = *ptr;
    char *start = NULL;

    while (*p && isspace(*p))
	p++;

    if (*p) {
	if (*p == '"') {
	    p++;
	    start = p;
	    while (*p && *p != '"')
		p++;
	} else {
	    start = p;
	    while (*p && !isspace(*p))
		p++;
	}

	if (*p)
	    *p++ = '\0';
    }

    *ptr = p;
    return start;
}

const char *fix_path(const char *path) {
    char *p;

    /* don't modify an absolute path */
    if (path[0] != '/' && path[0] != '\\') {
	if (asprintf(&p, "%s/%s", basedir, path) == -1) {
	    perror("asprintf failed");
	    exit(1);
	}
	path = p;
    } else {
	path = strdup(path);
    }

    return path;
}

unsigned long parse_unsigned(const char *s)
{
    char *p;
    unsigned long value;

    value = strtoul(s, &p, 0);
    if (*p) {
	fprintf(stderr, "%s:%d: invalid integer value: \"%s\"\n", default_path, default_line, s);
	exit(1);
    }

    return value;
}

void on_set(const char *var, const char *value)
{
    if (!strcasecmp(var, "MTYPE")) {
	mtype = parse_unsigned(value);
    } else if (!strcasecmp(var, "KERNEL")) {
	kernel_path = fix_path(value);
    } else if (!strcasecmp(var, "INITRD")) {
	initrd_path = fix_path(value);
    } else if (!strcasecmp(var, "RAMADDR")) {
	ram_addr = parse_unsigned(value);
    } else if (!strcasecmp(var, "RAMSIZE")) {
	ram_size = parse_unsigned(value);
    } else if (!strcasecmp(var, "TAG_ADDR")) {
	tag_addr = parse_unsigned(value);
    } else if (!strcasecmp(var, "KERNEL_ADDR")) {
	kernel_addr = parse_unsigned(value);
    } else if (!strcasecmp(var, "INITRD_ADDR")) {
	initrd_addr = parse_unsigned(value);
    } else if (!strcasecmp(var, "CMDLINE")) {
	cmdline = strdup(value);
    } else {
	fprintf(stderr, "%s:%d: ignoring set of unknown variable: %s \"%s\"\n", default_path, default_line, var, value);
    }
}

void read_default(void)
{
    FILE *fp;
    char s[1024];

    if ((fp = fopen(default_path, "r")) == NULL) {
	perror(default_path);
	exit(1);
    }

    default_line = 0;
    while (fgets(s, sizeof(s), fp)) {
	char *p = s;
	const char *cmd = next_token(&p);
	default_line++;

	if (cmd == NULL || cmd[0] == '#')
	    continue;

	if (!strcasecmp(cmd, "set")) {
	    const char *var = next_token(&p);
	    const char *value = next_token(&p);

	    on_set(var, value);
	} else 	if (!strcasecmp(cmd, "bootlinux")) {
	    dry_run = 0;
	} else {
	    fprintf(stderr, "%s:%d: ignoring unknown command: %s\n", default_path, default_line, cmd);
	}
    }

    fclose(fp);
}

/**
 * Copy everything between boot_linux_start and boot_linux_end into a
 * page.
 *
 * Returns the physical address of boot_linux_start.
 */
unsigned long setup_boot_linux(unsigned dir,
			       unsigned mach, unsigned tags, unsigned kernel)
{
    /* the following are labels from asm.S */
    extern char boot_linux_start;
    extern char boot_linux_end;
    extern char boot_linux_param;

    /* note that this structure must mach the one in asm.S */
    struct {
	unsigned dir;
	unsigned mach;
	unsigned tags;
	unsigned kernel;
	unsigned led_reg;
	unsigned led_val;
    } *param;

    unsigned long size;
    void *virt;
    unsigned long phys;

    size = &boot_linux_end - &boot_linux_start;

    virt = allocate_phys(size, &phys);

    printf("Copying 0x%lx bytes of boot_linux to 0x%08lx (%p)\n",
	   size, phys, virt);
    memcpy(virt, &boot_linux_start, size);

    param = virt + (&boot_linux_param - &boot_linux_start);

    param->dir = dir;
    param->mach = mach;
    param->tags = tags;
    param->kernel = kernel;

    if (mach == 656) {
	/* blink the BLUE led on the n30 */
	param->led_reg = 0x56000064;
	param->led_val = 0x00000040;
    }

    return phys;
}

int main(int argc, char *argv[])
{
    unsigned long dir_phys;
    void *dir_virt;
    unsigned long *dir;
    unsigned long kernel_size;
    unsigned long initrd_size;
    unsigned long boot_linux;
    void *tag;
    unsigned long tag_phys;

    printf("Linux Boot Loader for WinCE, build %s.\n", BUILD);

    if ((default_path = find_file(DEFAULT_FN, &basedir)) == NULL) {
	fprintf(stderr, "no %s found, can't continue\n", DEFAULT_FN);
	exit(1);
    }

    read_default();

    if (mtype == -1) {
	fprintf(stderr, "MTYPE must be specified\n");
	exit(1);
    }

    if (!kernel_path)
	kernel_path = fix_path(KERNEL_FN);

    if (tag_addr == ~0)
	tag_addr = ram_addr + 0x100;

    if (kernel_addr == ~0)
	kernel_addr = ram_addr + 0x8000;

    if (initrd_addr == ~0)
	initrd_addr = ram_addr + 0x800000;

    fprintf(stderr, "MTYPE %u\n", mtype);
    fprintf(stderr, "KERNEL %s\n", kernel_path);
    fprintf(stderr, "INITRD %s\n", initrd_path);
    fprintf(stderr, "RAMADDR 0x%08lx\n", ram_addr);
    fprintf(stderr, "RAMSIZE 0x%08lx\n", ram_size);
    fprintf(stderr, "TAG_ADDR 0x%08lx\n", tag_addr);
    fprintf(stderr, "KERNEL_ADDR 0x%08lx\n", kernel_addr);
    fprintf(stderr, "INITRD_ADDR 0x%08lx\n", initrd_addr);
    fprintf(stderr, "CMDLINE %s\n", cmdline);

    printf("Allocating directory\n");

    /* hopefully 64 kByte is enough for the whole directory */
    dir_virt = allocate_phys(64 * 1024, &dir_phys);
    printf("Directory at 0x%08lx (%p)\n", dir_phys, dir_virt);

    printf("Setting up bootstrap code\n");
    boot_linux = setup_boot_linux(dir_phys, mtype, tag_addr, kernel_addr);

    dir = dir_virt;

    fprintf(stderr, "Reading kernel from \"%s\"\n", kernel_path);
    kernel_size = load_file(kernel_path, kernel_addr, &dir);

    if (initrd_path) {
	fprintf(stderr, "Reading initrd from \"%s\"\n", initrd_path);
	initrd_size = load_file(initrd_path, initrd_addr, &dir);
    } else {
	initrd_addr = 0;
	initrd_size = 0;
    }

    /* hopefully 0x700 is enough space for the tags */
    tag = allocate_phys(0x700, &tag_phys);
    create_tags(tag, cmdline, initrd_addr, initrd_size);
    add_dir_entry(tag_addr, tag_phys, 0x700, &dir);

    /* directory end marker */
    *dir++ = 0;

    if (1)
	dump_dir(dir_virt);

    fprintf(stderr, "Becoming root\n");

    /* now becoming supervisor. */
    CeSetThreadPriority(GetCurrentThread(), 0);
    CeSetThreadQuantum(GetCurrentThread(), 0);
    SetKMode(1);
    SetProcPermissions(0xffffffff);

    fprintf(stderr, "Booting Linux\n");

    /* the point of no return */

    if (!dry_run) {
	disable_interrupts();
	enter_physical(boot_linux);
    }

    exit(0);
}

