/*
 * Copyright (c) 2005 Christer Weinigel <christer@weinigel.se>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This is a program that looks for changes of the specified
 * addresses.  I've used it to figure out which GPIO pin is connected
 * to which button.  It also gives an indication of what pins Windows
 * CE is modifying.
 *
 * \todo The output format could be a lot clearer.
 * \todo So far I can only dump the GPIO registers, this needs fixing.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "winhack.h"

/* Modify the register and filter list to match your hardware. */

#define N35 1

static struct reg {
    const char *name;
    unsigned base;
    unsigned offset;
    unsigned mask;
    
    /* used by the program */
    void *virt;
    unsigned value;
    unsigned count[32];
} regs[] = {
    { "GPADAT", 0x56000000, 0x0004, 0xffffffff },
    { "GPBDAT", 0x56000000, 0x0014, 0xfffffffe },
    { "GPCDAT", 0x56000000, 0x0024, 0xffff07ed },
#ifdef N35
    { "GPDDAT", 0x56000000, 0x0034, 0xffff0601 },
#else
    { "GPDDAT", 0x56000000, 0x0034, 0xffff0701 },
#endif
    { "GPEDAT", 0x56000000, 0x0044, 0xffff3800 },
    { "GPFDAT", 0x56000000, 0x0054, 0xffffffff },
    { "GPGDAT", 0x56000000, 0x0064, 0xffffffff },
    { "GPHDAT", 0x56000000, 0x0074, 0x00000103 },

#if 1
    { "GPACON", 0x56000000, 0x0000, 0xffffffff },
    { "GPBCON", 0x56000000, 0x0010, 0xffffffff },
    { "GPCCON", 0x56000000, 0x0020, 0xffffffff },
    { "GPDCON", 0x56000000, 0x0030, 0xffffffff },
    { "GPECON", 0x56000000, 0x0040, 0xffffffff },
    { "GPFCON", 0x56000000, 0x0050, 0xffffffff },
    { "GPGCON", 0x56000000, 0x0060, 0xffffffff },
    { "GPHCON", 0x56000000, 0x0070, 0xffffffff },

    { "GPBUP", 0x56000000, 0x0018, 0xffffffff },
    { "GPCUP", 0x56000000, 0x0028, 0xffffffff },
    { "GPDUP", 0x56000000, 0x0038, 0xffffffff },
    { "GPEUP", 0x56000000, 0x0048, 0xffffffff },
    { "GPFUP", 0x56000000, 0x0058, 0xffffffff },
    { "GPGUP", 0x56000000, 0x0068, 0xffffffff },
    { "GPHUP", 0x56000000, 0x0078, 0xffffffff },

    { "GSTATUS0", 0x56000000, 0x00AC, 0xffffffff },
    { "GSTATUS1", 0x56000000, 0x00B0, 0xffffffff },
    { "GSTATUS2", 0x56000000, 0x00B4, 0xffffffff },
    { "GSTATUS3", 0x56000000, 0x00B8, 0xffffffff },
    { "GSTATUS4", 0x56000000, 0x00BC, 0xffffffff },

    { "MISCCR", 0x56000000, 0x0080, 0xffffffff },
    { "PWR_REG", 0x52000000, 0x144, 0xffffffff },
    // { "DMASKTRIG0", 0x4b000000, 0x0020, 0xffffffff },
    { "TCFG0", 0x51000000, 0x0000, 0xffffffff },
    { "TCFG0", 0x51000000, 0x0000, 0xffffffff },
    { "TCFG1", 0x51000000, 0x0004, 0xffffffff },
    // { "TCON", 0x51000000, 0x0008, 0xffffffff },
    { "TCNTB0", 0x51000000, 0x000c, 0xffffffff },
    { "TCMPB0", 0x51000000, 0x0010, 0xffffffff },
    // { "TCNTO0", 0x51000000, 0x0014, 0xffffffff },
    { "TCNTB1", 0x51000000, 0x0018, 0xffffffff },
    { "TCMPB1", 0x51000000, 0x001c, 0xffffffff },
    // { "TCNTO1", 0x51000000, 0x0020, 0xffffffff },
    { "TCNTB2", 0x51000000, 0x0024, 0xffffffff },
    { "TCMPB2", 0x51000000, 0x0028, 0xffffffff },
    // { "TCNTO2", 0x51000000, 0x002c, 0xffffffff },
    { "TCNTB3", 0x51000000, 0x0030, 0xffffffff },
    { "TCMPB3", 0x51000000, 0x0034, 0xffffffff },
    // { "TCNTO3", 0x51000000, 0x0038, 0xffffffff },
    { "TCNTB4", 0x51000000, 0x003c, 0xffffffff },
    // { "TCNTO4", 0x51000000, 0x0040, 0xffffffff },

    // { "INTMSK", 0x4a000000, 0x0008, 0xffffffff },

#endif

#if 0
    { "ADCCON", 0x58000000, 0x0000, 0xffffffff },
    { "ADCTSC", 0x58000000, 0x0004, 0xffffffff },
    { "ADCDLY", 0x58000000, 0x0008, 0xffffffff },
    { "ADCDAT0", 0x58000000, 0x000C, 0xfffffc00 },
    { "ADCDAT1", 0x58000000, 0x0010, 0xfffffc00 },
#endif
};
#define NREGS (sizeof(regs) / sizeof(*regs))

/* when QUIT is true the loop finished.  The following will quit if
 * the the 6th bit of the 5th register listed above, i.e. F6, changes.
 * On the n30 that is the calendar button and on the H1940 it is the
 * select button */

#define QUIT() regs[6].count[7]

/* A limit on the number of times a change will show up */
#define MAX_COUNT 100

static void map_regs(void) 
{
    unsigned long base;
    void *virt;
    int i;

    base = ~0;
    virt = NULL;
    for (i = 0; i < NREGS; i++) {
	if (base == regs[i].base) {
	    regs[i].virt = virt;
	    continue;
	}

	base = regs[i].base;

	printf("allocating: 0x%08lx\n", base);
	virt = VirtualAlloc(NULL, 0x10000, MEM_RESERVE, PAGE_NOACCESS);
	if (!virt) {
	    fprintf(stderr, "VirtualAlloc failed\n");
	    exit(1);
	}

	if (!VirtualCopy(virt, (LPVOID)(base >> 8), 0x10000, 
			 PAGE_READWRITE | PAGE_PHYSICAL | PAGE_NOCACHE)) {
	    printf("VirtualCopy of 0x%08lx failed\n", base);
	    exit(1);
	}

	regs[i].virt = virt;

	if (0 && base == 0x58000000) {
	    printf("base=0x58000...\n");
	    writel(10000, virt + 0x08);
	}
    }
}

int main()
{
    unsigned t0;
    unsigned t;
    int r;
    int b;

    map_regs();

    printf("done\n");

    for (r = 0; r < NREGS; r++) {
	regs[r].value = readl(regs[r].virt + regs[r].offset);
	for (b = 0; b < 32; b++)
	    regs[r].count[b] = 0;

	if (1) {
	    printf("%u: reg %s (0x%08x) value 0x%08x\n", 
		   0, regs[r].name, regs[r].base + regs[r].offset, 
		   regs[r].value);
	}
    }

    t0 = GetTickCount();
    printf("start\n");
    while (1) {
	for (r = 0; r < NREGS; r++) {
	    unsigned value;
	    unsigned delta;

	    value = readl(regs[r].virt + regs[r].offset);

	    delta = (value ^ regs[r].value) & regs[r].mask;
	    if (!delta)
		continue;

	    for (b = 0; b < 32; b++) {
		int show = 0;

		if (delta & (1<<b)) {
		    if (regs[r].count[b] < MAX_COUNT) {
			show = 1;
			regs[r].count[b]++;
		    } else 
			delta &= ~(1<<b);
		}

		if (show) {
		    t = GetTickCount() - t0;
		    printf("%u: reg %s (0x%08x) value 0x%08x delta 0x%08x \n", 
			   t, regs[r].name, regs[r].base + regs[r].offset, 
			   value, delta);
		}
	    }

	    regs[r].value = value;
	}

	if (QUIT())
	    break;

	if (1)
	    Sleep(1);
    }	

    exit(0);
}
