/*
 * Decompiled with CFR 0.152.
 */
package de.joergjahnke.gameboy.core;

import de.joergjahnke.common.io.Serializable;
import de.joergjahnke.common.io.SerializationUtils;
import de.joergjahnke.common.util.DefaultObservable;
import de.joergjahnke.common.util.Observer;
import de.joergjahnke.gameboy.core.Gameboy;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.Vector;

public class Cartridge
extends DefaultObservable
implements Serializable,
Observer {
    public static final int ROM_BANK_SIZE = 16384;
    public static final int RAM_BANK_SIZE = 8192;
    private static final int ROM_BANK_AREA = 16384;
    private static final int RAM_BANK_AREA = 40960;
    private static final boolean DEFAULT_RTC_WITH_SYSTEM_CLOCK = true;
    private static final boolean SYNCHRONIZE_RTC_WITH_CPU = false;
    public static final Vector SUPPORTED_EXTENSIONS = new Vector();
    private final Gameboy gameboy;
    private String title;
    private boolean isGBC;
    private int cartridgeType;
    private int romSize;
    private int ramSize;
    private byte[][] romBanks;
    private byte[][] ramBanks;
    private CartridgeImpl cartridgeImpl;

    public Cartridge(Gameboy gameboy) {
        this.gameboy = gameboy;
    }

    public void load(InputStream romStream) throws IOException {
        this.setChanged(true);
        this.notifyObservers(new Integer(0));
        this.ramBanks = null;
        byte[] buffer = new byte[16384];
        romStream.read(buffer);
        StringBuffer titleBuffer = new StringBuffer();
        for (int i = 308; i < 324 && buffer[i] != 0; ++i) {
            titleBuffer.append((char)buffer[i]);
        }
        this.title = titleBuffer.toString();
        this.isGBC = (buffer[323] & 0x80) != 0;
        this.cartridgeType = buffer[327] & 0xFF;
        String cartridgeTypeName = this.getCartridgeTypeName();
        if (cartridgeTypeName.startsWith("MBC1") || cartridgeTypeName.startsWith("ROM")) {
            this.cartridgeImpl = new MBC1CartridgeImpl();
        } else if (cartridgeTypeName.startsWith("MBC2")) {
            this.cartridgeImpl = new MBC2CartridgeImpl();
        } else if (cartridgeTypeName.startsWith("MBC3")) {
            this.cartridgeImpl = new MBC3CartridgeImpl();
        } else if (cartridgeTypeName.startsWith("MBC5")) {
            this.cartridgeImpl = new MBC5CartridgeImpl();
        } else {
            this.cartridgeImpl = new MBC1CartridgeImpl();
            this.gameboy.getLogger().warning("Unsupported cartridge type: " + cartridgeTypeName + "! Trying with MBC1 cartridge handling.");
        }
        switch (buffer[328]) {
            case 82: {
                this.romSize = 0x120000;
                break;
            }
            case 83: {
                this.romSize = 0x140000;
                break;
            }
            case 84: {
                this.romSize = 0x180000;
                break;
            }
            default: {
                this.romSize = 32768 << (buffer[328] & 0xFF);
            }
        }
        switch (buffer[329]) {
            case 1: {
                this.ramSize = 2048;
                break;
            }
            case 2: {
                this.ramSize = 8192;
                break;
            }
            case 3: {
                this.ramSize = 32768;
                break;
            }
            case 4: 
            case 5: 
            case 6: {
                this.ramSize = 131072;
                break;
            }
            default: {
                this.ramSize = 0;
            }
        }
        this.ramBanks = new byte[Math.max(1, this.ramSize / 8192)][8192];
        int numROMBanks = this.romSize / 16384;
        this.romBanks = new byte[this.romSize / 16384][16384];
        System.arraycopy(buffer, 0, this.romBanks[0], 0, buffer.length);
        for (int i = 1; i < this.romBanks.length; ++i) {
            this.setChanged(true);
            this.notifyObservers(new Integer(i * 100 / numROMBanks));
            romStream.read(buffer);
            System.arraycopy(buffer, 0, this.romBanks[i], 0, buffer.length);
        }
        this.setChanged(true);
        this.notifyObservers(new Integer(100));
    }

    public String getTitle() {
        return this.title;
    }

    public boolean isGBC() {
        return this.isGBC;
    }

    public int getCartridgeType() {
        return this.cartridgeType;
    }

    public String getCartridgeTypeName() {
        switch (this.cartridgeType) {
            case 0: {
                return "ROM Only";
            }
            case 1: {
                return "MBC1";
            }
            case 2: {
                return "MBC1+RAM";
            }
            case 3: {
                return "MBC1+RAM+Battery";
            }
            case 5: {
                return "MBC2";
            }
            case 6: {
                return "MBC2+Battery";
            }
            case 8: {
                return "ROM+RAM";
            }
            case 9: {
                return "ROM+RAM+Battery";
            }
            case 11: {
                return "MMM1";
            }
            case 12: {
                return "MMM1+RAM";
            }
            case 13: {
                return "MMM1+RAM+Battery";
            }
            case 15: {
                return "MBC3+Timer+Battery";
            }
            case 16: {
                return "MBC3+Timer+RAM+Battery";
            }
            case 17: {
                return "MBC3";
            }
            case 18: {
                return "MBC3+RAM";
            }
            case 19: {
                return "MBC3+RAM+Battery";
            }
            case 21: {
                return "MBC4";
            }
            case 22: {
                return "MBC4+RAM";
            }
            case 23: {
                return "MBC4+RAM+Battery";
            }
            case 25: {
                return "MBC5";
            }
            case 26: {
                return "MBC5+RAM";
            }
            case 27: {
                return "MBC5+RAM+Battery";
            }
            case 28: {
                return "MBC5+Rumble";
            }
            case 29: {
                return "MBC5+Rumble+RAM";
            }
            case 30: {
                return "MBC5+Rumble+RAM+Battery";
            }
            case 254: {
                return "HuC3";
            }
            case 255: {
                return "HuC1+RAM+Battery";
            }
        }
        return "Unknown (" + this.cartridgeType + ")";
    }

    public boolean hasBatterySupport() {
        return this.getCartridgeTypeName().indexOf("Battery") >= 0;
    }

    public int getROMSize() {
        return this.romSize;
    }

    public int getRAMSize() {
        return this.ramSize;
    }

    public byte[][] getROMBanks() {
        return this.romBanks;
    }

    public byte[][] getRAMBanks() {
        return this.ramBanks;
    }

    public final void writeByte(int adr, byte data) {
        this.cartridgeImpl.writeByte(adr, data);
    }

    public void saveData(OutputStream saveStream) throws IOException {
        DataOutputStream out = new DataOutputStream(saveStream);
        int to = this.getRAMBanks().length;
        for (int i = 0; i < to; ++i) {
            out.write(this.getRAMBanks()[i]);
        }
        this.serialize(out);
        out.flush();
    }

    public void loadData(InputStream loadStream) throws IOException {
        DataInputStream in = new DataInputStream(loadStream);
        int to = this.getRAMBanks().length;
        for (int i = 0; i < to; ++i) {
            in.read(this.getRAMBanks()[i]);
        }
        try {
            this.deserialize(in);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public void serialize(DataOutputStream out) throws IOException {
        this.cartridgeImpl.serialize(out);
        for (int i = 0; i < this.ramBanks.length; ++i) {
            SerializationUtils.serialize(out, this.ramBanks[i]);
        }
    }

    public void deserialize(DataInputStream in) throws IOException {
        this.cartridgeImpl.deserialize(in);
        for (int i = 0; i < this.ramBanks.length; ++i) {
            SerializationUtils.deserialize(in, this.ramBanks[i]);
        }
    }

    public void update(Object observed, Object arg) {
        this.cartridgeImpl.update(observed, arg);
    }

    static {
        SUPPORTED_EXTENSIONS.addElement("gbc");
        SUPPORTED_EXTENSIONS.addElement("cgb");
        SUPPORTED_EXTENSIONS.addElement("gb");
    }

    class MBC5CartridgeImpl
    extends MBC1CartridgeImpl {
        MBC5CartridgeImpl() {
        }

        public void writeByte(int adr, byte data) {
            switch (adr & 0xE000) {
                case 8192: {
                    int romBank = this.currentROMBank;
                    romBank = (adr & 0x1000) != 0 ? romBank & 0xFF | ((data & 1) != 0 ? 256 : 0) : romBank & 0x100 | data & 0xFF;
                    this.setROMBank(romBank);
                    break;
                }
                case 16384: {
                    if (Cartridge.this.ramSize <= 0) break;
                    this.setRAMBank(data & 0xF);
                    break;
                }
                default: {
                    super.writeByte(adr, data);
                }
            }
        }
    }

    class MBC3CartridgeImpl
    extends MBC1CartridgeImpl {
        private static final int SECONDS = 0;
        private static final int MINUTES = 1;
        private static final int HOURS = 2;
        private static final int DAYS_LOW = 3;
        private static final int DAYS_HIGH = 4;
        private final Date clock;
        private long lastRTCUpdate = 0L;
        private int[] rtc = new int[5];
        private int rtcIndex = -1;
        private boolean latchRTC = false;
        private boolean isClockActive = true;
        private long cpuSpeed = 0x400000L;

        protected MBC3CartridgeImpl() {
            this.isROMBankingMode = false;
            this.clock = new Date();
            Calendar calendar = Calendar.getInstance();
            calendar.set(1, 1970);
            this.clock.setTime(calendar.getTime().getTime());
        }

        public void writeByte(int adr, byte data) {
            block0 : switch (adr & 0xE000) {
                case 8192: {
                    this.setROMBank(Math.max(1, data & 0x7F));
                    break;
                }
                case 16384: {
                    if (data >= 8 && data <= 12) {
                        this.rtcIndex = data - 8;
                        byte[] memory_ = ((Cartridge)Cartridge.this).gameboy.getCPU().memory;
                        memory_[40960] = (byte)this.rtc[this.rtcIndex];
                        for (int len = 1; len < 8192; len <<= 1) {
                            System.arraycopy(memory_, 40960, memory_, 40960 + len, len);
                        }
                        this.currentRAMBank = -1;
                        break;
                    }
                    this.rtcIndex = -1;
                    super.writeByte(adr, (byte)(data & 3));
                    break;
                }
                case 24576: {
                    if (this.latchRTC && data == 1) {
                        this.latchClock();
                        this.latchRTC = false;
                        break;
                    }
                    if (data == 0) {
                        this.latchRTC = true;
                        break;
                    }
                    this.latchRTC = false;
                    break;
                }
                case 40960: {
                    switch (this.rtcIndex) {
                        case 0: {
                            this.updateClock();
                            this.clock.setTime(this.clock.getTime() + (long)(((data & 0xFF) - this.getRTCSeconds()) * 1000));
                            break block0;
                        }
                        case 1: {
                            this.updateClock();
                            this.clock.setTime(this.clock.getTime() + (long)(((data & 0xFF) - this.getRTCMinutes()) * 1000 * 60));
                            break block0;
                        }
                        case 2: {
                            this.updateClock();
                            this.clock.setTime(this.clock.getTime() + (long)(((data & 0xFF) - this.getRTCHours()) * 1000 * 60 * 60));
                            break block0;
                        }
                        case 3: {
                            this.updateClock();
                            this.clock.setTime(this.clock.getTime() + (long)((data & 0xFF) - this.getRTCDays() % 256) * 1000L * 60L * 60L * 24L);
                            this.latchClock();
                            break block0;
                        }
                        case 4: {
                            this.updateClock();
                            int days = this.getRTCDays() % 256 + ((data & 1) != 0 ? 256 : 0) + ((data & 0x80) != 0 ? 512 : 0);
                            this.clock.setTime(this.clock.getTime() + (long)(days - this.getRTCDays()) * 1000L * 60L * 60L * 24L);
                            this.isClockActive = (data & 0x40) == 0;
                            break block0;
                        }
                    }
                    super.writeByte(adr, data);
                    break;
                }
                default: {
                    super.writeByte(adr, data);
                }
            }
        }

        private boolean isClockActive() {
            return this.isClockActive;
        }

        private int getRTCSeconds() {
            return (int)(this.clock.getTime() / 1000L % 60L);
        }

        private int getRTCMinutes() {
            return (int)(this.clock.getTime() / 1000L / 60L % 60L);
        }

        private int getRTCHours() {
            return (int)(this.clock.getTime() / 1000L / 60L / 60L % 24L);
        }

        private int getRTCDays() {
            return (int)(this.clock.getTime() / 1000L / 60L / 60L / 24L);
        }

        private void latchClock() {
            this.updateClock();
            this.rtc[0] = this.getRTCSeconds();
            this.rtc[1] = this.getRTCMinutes();
            this.rtc[2] = this.getRTCHours();
            this.rtc[3] = this.getRTCDays() % 256;
            this.rtc[4] = (this.getRTCDays() % 512 >> 8) + (this.isClockActive() ? 0 : 64) + (this.getRTCDays() >= 512 ? 128 : 0);
        }

        private void updateClock() {
            if (this.isClockActive()) {
                long now = new Date().getTime();
                long passedMillis = now - this.lastRTCUpdate;
                this.clock.setTime(this.clock.getTime() + passedMillis);
                this.lastRTCUpdate = now;
            }
        }

        public void serialize(DataOutputStream out) throws IOException {
            this.updateClock();
            super.serialize(out);
            out.writeLong(this.clock.getTime());
            out.writeBoolean(this.latchRTC);
            out.writeBoolean(this.isClockActive);
            SerializationUtils.serialize(out, this.rtc);
            out.writeLong(System.currentTimeMillis());
        }

        public void deserialize(DataInputStream in) throws IOException {
            this.lastRTCUpdate = new Date().getTime();
            super.deserialize(in);
            this.clock.setTime(in.readLong());
            this.latchRTC = in.readBoolean();
            this.isClockActive = in.readBoolean();
            SerializationUtils.deserialize(in, this.rtc);
            long oldTime = in.readLong();
            this.clock.setTime(this.clock.getTime() + System.currentTimeMillis() - oldTime);
        }

        public void update(Object observed, Object arg) {
            if (observed == Cartridge.this.gameboy.getCPU() && arg instanceof Long) {
                this.updateClock();
                this.cpuSpeed = (Long)arg;
            }
        }
    }

    class MBC2CartridgeImpl
    extends MBC1CartridgeImpl {
        MBC2CartridgeImpl() {
        }

        public void writeByte(int adr, byte data) {
            switch (adr & 0xE000) {
                case 0: {
                    if ((adr & 0x100) == 0) {
                        super.writeByte(adr, data);
                    }
                    this.areRAMWritesEnabled = (data & 0xF) == 10;
                    break;
                }
                case 8192: {
                    if ((adr & 0x100) == 0) break;
                    super.writeByte(adr, (byte)(data & 0xF));
                    break;
                }
                default: {
                    super.writeByte(adr, data);
                }
            }
        }
    }

    class MBC1CartridgeImpl
    extends CartridgeImpl {
        MBC1CartridgeImpl() {
        }

        public void writeByte(int adr, byte data) {
            switch (adr & 0xE000) {
                case 0: {
                    this.areRAMWritesEnabled = (data & 0xF) == 10;
                    break;
                }
                case 8192: {
                    this.setROMBank((this.currentROMBank & 0xE0) + Math.max(1, data & 0x1F));
                    break;
                }
                case 16384: {
                    if (this.isROMBankingMode) {
                        this.setROMBank((this.currentROMBank & 0x1F) + ((data & 3) << 5));
                        break;
                    }
                    this.setRAMBank(data & 3);
                    break;
                }
                case 24576: {
                    this.isROMBankingMode = (data & 1) == 0;
                    break;
                }
                case 40960: {
                    if (!this.areRAMWritesEnabled) break;
                    ((Cartridge)Cartridge.this).gameboy.getCPU().memory[adr] = data;
                    ((Cartridge)Cartridge.this).ramBanks[this.currentRAMBank][adr & 0x1FFF] = data;
                }
            }
        }
    }

    abstract class CartridgeImpl
    implements Serializable,
    Observer {
        protected boolean areRAMWritesEnabled = true;
        protected boolean isROMBankingMode = true;
        protected int currentROMBank = 1;
        protected int currentRAMBank = 0;

        CartridgeImpl() {
        }

        public abstract void writeByte(int var1, byte var2);

        protected final void setROMBank(int romBank) {
            if (romBank != this.currentROMBank) {
                if (romBank >= Cartridge.this.getROMBanks().length) {
                    Cartridge.this.gameboy.getLogger().warning("Tried to access ROM bank " + romBank + " of only " + Cartridge.this.getROMBanks().length + " ROM banks!");
                }
                this.currentROMBank = romBank % Cartridge.this.getROMBanks().length;
                System.arraycopy(Cartridge.this.getROMBanks()[this.currentROMBank], 0, ((Cartridge)Cartridge.this).gameboy.getCPU().memory, 16384, 16384);
            }
        }

        protected final void setRAMBank(int ramBank) {
            if (ramBank != this.currentRAMBank) {
                this.currentRAMBank = ramBank;
                System.arraycopy(Cartridge.this.getRAMBanks()[this.currentRAMBank], 0, ((Cartridge)Cartridge.this).gameboy.getCPU().memory, 40960, 8192);
            }
        }

        public void serialize(DataOutputStream out) throws IOException {
            out.writeBoolean(this.areRAMWritesEnabled);
            out.writeBoolean(this.isROMBankingMode);
            out.writeInt(this.currentROMBank);
            out.writeInt(this.currentRAMBank);
        }

        public void deserialize(DataInputStream in) throws IOException {
            this.areRAMWritesEnabled = in.readBoolean();
            this.isROMBankingMode = in.readBoolean();
            this.currentROMBank = in.readInt();
            this.currentRAMBank = in.readInt();
        }

        public void update(Object observed, Object arg) {
        }
    }
}

