/* Arrays.java -- Utility class with methods to operate on arrays
Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007,
Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */
package com.classpath.util;

import java.util.Vector;

/**
 * This class contains various static utility methods performing operations on
 * arrays, and a method to provide a List "view" of an array to facilitate
 * using arrays with Collection-based APIs. All methods throw a
 * {@link NullPointerException} if the parameter array is null.
 * <p>
 *
 * Implementations may use their own algorithms, but must obey the general
 * properties; for example, the sort must be stable and n*log(n) complexity.
 * Sun's implementation of sort, and therefore ours, is a tuned quicksort,
 * adapted from Jon L. Bentley and M. Douglas McIlroy's "Engineering a Sort
 * Function", Software-Practice and Experience, Vol. 23(11) P. 1249-1265
 * (November 1993). This algorithm offers n*log(n) performance on many data
 * sets that cause other quicksorts to degrade to quadratic performance.
 *
 * @author Original author unknown
 * @author Bryce McKinlay
 * @author Eric Blake (ebb9@email.byu.edu)
 * @see Comparable
 * @see Comparator
 * @since 1.2
 * @status updated to 1.4
 */
public class Arrays
{
	/**
	 * This class is non-instantiable.
	 */
	private Arrays()
	{
	}

	/**
	 * Compare two byte arrays for equality.
	 *
	 * @param a1 the first array to compare
	 * @param a2 the second array to compare
	 * @return true if a1 and a2 are both null, or if a2 is of the same length
	 *         as a1, and for each 0 <= i < a1.length, a1[i] == a2[i]
	 */
	public static boolean equals(byte[] a1, byte[] a2)
	{
		// Quick test which saves comparing elements of the same array, and also
		// catches the case that both are null.
		if(a1 == a2)
		{
			return true;
		}

		if(null == a1 || null == a2)
		{
			return false;
		}

		// If they're the same length, test each element
		if(a1.length == a2.length)
		{
			int i = a1.length;
			while(--i >= 0)
			{
				if(a1[i] != a2[i])
				{
					return false;
				}
			}
			return true;
		}
		return false;
	}

	/**
	 * Fill an array with an int value.
	 *
	 * @param a the array to fill
	 * @param val the value to fill it with
	 */
	public static void fill(int[] a, int val)
	{
		fill(a, 0, a.length, val);
	}

	/**
	 * Fill a range of an array with an int value.
	 *
	 * @param a the array to fill
	 * @param fromIndex the index to fill from, inclusive
	 * @param toIndex the index to fill to, exclusive
	 * @param val the value to fill with
	 * @throws IllegalArgumentException if fromIndex &gt; toIndex
	 * @throws ArrayIndexOutOfBoundsException if fromIndex &lt; 0
	 *         || toIndex &gt; a.length
	 */
	public static void fill(int[] a, int fromIndex, int toIndex, int val)
	{
		if(fromIndex > toIndex)
		{
			throw new IllegalArgumentException();
		}
		for(int i = fromIndex; i < toIndex; i++)
		{
			a[i] = val;
		}
	}

	/**
	 * Fill an array with a byte value.
	 *
	 * @param a the array to fill
	 * @param val the value to fill it with
	 */
	public static void fill(byte[] a, byte val)
	{
		fill(a, 0, a.length, val);
	}

	/**
	 * Fill a range of an array with a byte value.
	 *
	 * @param a the array to fill
	 * @param fromIndex the index to fill from, inclusive
	 * @param toIndex the index to fill to, exclusive
	 * @param val the value to fill with
	 * @throws IllegalArgumentException if fromIndex &gt; toIndex
	 * @throws ArrayIndexOutOfBoundsException if fromIndex &lt; 0
	 *         || toIndex &gt; a.length
	 */
	public static void fill(byte[] a, int fromIndex, int toIndex, byte val)
	{
		if(fromIndex > toIndex)
		{
			throw new IllegalArgumentException();
		}
		for(int i = fromIndex; i < toIndex; i++)
		{
			a[i] = val;
		}
	}

	/**
	 * Fill an array with an Object value.
	 *
	 * @param a the array to fill
	 * @param val the value to fill it with
	 * @throws ClassCastException if val is not an instance of the element
	 *         type of a.
	 */
	public static void fill(Object[] a, Object val)
	{
		fill(a, 0, a.length, val);
	}

	/**
	 * Fill a range of an array with an Object value.
	 *
	 * @param a the array to fill
	 * @param fromIndex the index to fill from, inclusive
	 * @param toIndex the index to fill to, exclusive
	 * @param val the value to fill with
	 * @throws ClassCastException if val is not an instance of the element
	 *         type of a.
	 * @throws IllegalArgumentException if fromIndex &gt; toIndex
	 * @throws ArrayIndexOutOfBoundsException if fromIndex &lt; 0
	 *         || toIndex &gt; a.length
	 */
	public static void fill(Object[] a, int fromIndex, int toIndex, Object val)
	{
		if(fromIndex > toIndex)
		{
			throw new IllegalArgumentException();
		}
		for(int i = fromIndex; i < toIndex; i++)
		{
			a[i] = val;
		}
	}

	public static Vector asList(Object[] a)
	{
		Vector res = new Vector(a.length);

		for(int i = 0; i < a.length; i++)
		{
			res.addElement(a[i]);
		}

		return res;
	}
}
