/* java.lang.UnicodeCharacter -- Wrapper class for char, and Unicode subsets
Copyright (C) 1998, 1999, 2001, 2002, 2004, 2005 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */
package com.classpath.util;

import com.vmx.AuxClass;
import java.io.DataInputStream;
import java.io.IOException;

/**
 * Wrapper class for the primitive char data type.  In addition, this class
 * allows one to retrieve property information and perform transformations
 * on the defined characters in the Unicode Standard, Version 4.0.0.
 * java.lang.UnicodeCharacter is designed to be very dynamic, and as such, it
 * retrieves information on the Unicode character set from a separate
 * database, gnu.java.lang.CharData, which can be easily upgraded.
 *
 * <p>For predicates, boundaries are used to describe
 * the set of characters for which the method will return true.
 * This syntax uses fairly normal regular expression notation.
 * See 5.13 of the Unicode Standard, Version 4.0, for the
 * boundary specification.
 *
 * <p>See <a href="http://www.unicode.org">http://www.unicode.org</a>
 * for more information on the Unicode Standard.
 *
 * @author Tom Tromey (tromey@cygnus.com)
 * @author Paul N. Fisher
 * @author Jochen Hoenicke
 * @author Eric Blake (ebb9@email.byu.edu)
 * @author Andrew John Hughes (gnu_andrew@member.fsf.org)
 * @see CharData
 * @since 1.0
 * @status partly updated to 1.5; some things still missing
 */
public final class Characters
{
	/**
	 * A class to encompass all the properties of characters in the
	 * private use blocks in the Unicode standard.  This class extends
	 * UnassignedCharacters because the return type from getType() is
	 * different.
	 * @author Anthony Balkissoon abalkiss at redhat dot com
	 *
	 */
	private static class PrivateUseCharacters extends UnassignedCharacters
	{
		/**
		 * Returns the type of the character cp.
		 */
		static int getType(int cp)
		{
			// The upper 2 code points in any plane are considered unassigned,
			// even in the private-use planes.
			if((cp & 0xffff) >= 0xfffe)
			{
				return UnassignedCharacters.getType(cp);
			}
			return PRIVATE_USE;
		}

		/**
		 * Returns true if the character cp is defined.
		 */
		static boolean isDefined(int cp)
		{
			// The upper 2 code points in any plane are considered unassigned,
			// even in the private-use planes.
			if((cp & 0xffff) >= 0xfffe)
			{
				return UnassignedCharacters.isDefined(cp);
			}
			return true;
		}

		/**
		 * Gets the directionality for the character cp.
		 */
		static byte getDirectionality(int cp)
		{
			if((cp & 0xffff) >= 0xfffe)
			{
				return UnassignedCharacters.getDirectionality(cp);
			}
			return DIRECTIONALITY_LEFT_TO_RIGHT;
		}
	}

	/**
	 * A class to encompass all the properties of code points that are
	 * currently undefined in the Unicode standard.
	 * @author Anthony Balkissoon abalkiss at redhat dot com
	 *
	 */
	private static class UnassignedCharacters
	{
		/**
		 * Returns the numeric value for the unassigned characters.
		 * @param cp the character
		 * @param radix the radix (not used)
		 * @return the numeric value of this character in this radix
		 */
		static int digit(int cp, int radix)
		{
			return -1;
		}

		/**
		 * Returns the Unicode directionality property for unassigned
		 * characters.
		 * @param cp the character
		 * @return DIRECTIONALITY_UNDEFINED
		 */
		static byte getDirectionality(int cp)
		{
			return DIRECTIONALITY_UNDEFINED;
		}

		/**
		 * Returns -1, the numeric value for unassigned Unicode characters.
		 * @param cp the character
		 * @return -1
		 */
		static int getNumericValue(int cp)
		{
			return -1;
		}

		/**
		 * Returns UNASSIGNED, the type of unassigned Unicode characters.
		 * @param cp the character
		 * @return UNASSIGNED
		 */
		static int getType(int cp)
		{
			return UNASSIGNED;
		}

		/**
		 * Returns false to indiciate that the character is not defined in the
		 * Unicode standard.
		 * @param cp the character
		 * @return false
		 */
		static boolean isDefined(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character is not a digit.
		 * @param cp the character
		 * @return false
		 */
		static boolean isDigit(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character cannot be ignored
		 * within an identifier
		 * @param cp the character
		 * @return false
		 */
		static boolean isIdentifierIgnorable(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character cannot be part of a
		 * Java identifier.
		 * @param cp the character
		 * @return false
		 */
		static boolean isJavaIdentifierPart(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character cannot be start a
		 * Java identifier.
		 * @param cp the character
		 * @return false
		 */
		static boolean isJavaIdentiferStart(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character is not a letter.
		 * @param cp the character
		 * @return false
		 */
		static boolean isLetter(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character cannot is neither a letter
		 * nor a digit.
		 * @param cp the character
		 * @return false
		 */
		static boolean isLetterOrDigit(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character is not a lowercase letter.
		 * @param cp the character
		 * @return false
		 */
		static boolean isLowerCase(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character cannot is not mirrored.
		 * @param cp the character
		 * @return false
		 */
		static boolean isMirrored(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character is not a space character.
		 * @param cp the character
		 * @return false
		 */
		static boolean isSpaceChar(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character it not a titlecase letter.
		 * @param cp the character
		 * @return false
		 */
		static boolean isTitleCase(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character cannot be part of a
		 * Unicode identifier.
		 * @param cp the character
		 * @return false
		 */
		static boolean isUnicodeIdentifierPart(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character cannot start a
		 * Unicode identifier.
		 * @param cp the character
		 * @return false
		 */
		static boolean isUnicodeIdentifierStart(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character is not an uppercase letter.
		 * @param cp the character
		 * @return false
		 */
		static boolean isUpperCase(int cp)
		{
			return false;
		}

		/**
		 * Returns false to indicate that the character is not a whitespace
		 * character.
		 * @param cp the character
		 * @return false
		 */
		static boolean isWhiteSpace(int cp)
		{
			return false;
		}

		/**
		 * Returns cp to indicate this character has no lowercase conversion.
		 * @param cp the character
		 * @return cp
		 */
		static int toLowerCase(int cp)
		{
			return cp;
		}

		/**
		 * Returns cp to indicate this character has no titlecase conversion.
		 * @param cp the character
		 * @return cp
		 */
		static int toTitleCase(int cp)
		{
			return cp;
		}

		/**
		 * Returns cp to indicate this character has no uppercase conversion.
		 * @param cp the character
		 * @return cp
		 */
		static int toUpperCase(int cp)
		{
			return cp;
		}
	}
	/**
	 * Smallest value allowed for radix arguments in Java. This value is 2.
	 *
	 * @see #digit(char, int)
	 * @see #forDigit(int, int)
	 * @see Integer#toString(int, int)
	 * @see Integer#valueOf(String)
	 */
	public static final int MIN_RADIX = 2;
	/**
	 * Largest value allowed for radix arguments in Java. This value is 36.
	 *
	 * @see #digit(char, int)
	 * @see #forDigit(int, int)
	 * @see Integer#toString(int, int)
	 * @see Integer#valueOf(String)
	 */
	public static final int MAX_RADIX = 36;
	/**
	 * The minimum value the char data type can hold.
	 * This value is <code>'\\u0000'</code>.
	 */
	public static final char MIN_VALUE = '\u0000';
	/**
	 * The maximum value the char data type can hold.
	 * This value is <code>'\\uFFFF'</code>.
	 */
	public static final char MAX_VALUE = '\uFFFF';
	/**
	 * The minimum Unicode 4.0 code point.  This value is <code>0</code>.
	 * @since 1.5
	 */
	public static final int MIN_CODE_POINT = 0;
	/**
	 * The maximum Unicode 4.0 code point, which is greater than the range
	 * of the char data type.
	 * This value is <code>0x10FFFF</code>.
	 * @since 1.5
	 */
	public static final int MAX_CODE_POINT = 0x10FFFF;
	/**
	 * The minimum Unicode high surrogate code unit, or
	 * <emph>leading-surrogate</emph>, in the UTF-16 character encoding.
	 * This value is <code>'\uD800'</code>.
	 * @since 1.5
	 */
	public static final char MIN_HIGH_SURROGATE = '\uD800';
	/**
	 * The maximum Unicode high surrogate code unit, or
	 * <emph>leading-surrogate</emph>, in the UTF-16 character encoding.
	 * This value is <code>'\uDBFF'</code>.
	 * @since 1.5
	 */
	public static final char MAX_HIGH_SURROGATE = '\uDBFF';
	/**
	 * The minimum Unicode low surrogate code unit, or
	 * <emph>trailing-surrogate</emph>, in the UTF-16 character encoding.
	 * This value is <code>'\uDC00'</code>.
	 * @since 1.5
	 */
	public static final char MIN_LOW_SURROGATE = '\uDC00';
	/**
	 * The maximum Unicode low surrogate code unit, or
	 * <emph>trailing-surrogate</emph>, in the UTF-16 character encoding.
	 * This value is <code>'\uDFFF'</code>.
	 * @since 1.5
	 */
	public static final char MAX_LOW_SURROGATE = '\uDFFF';
	/**
	 * The minimum Unicode surrogate code unit in the UTF-16 character encoding.
	 * This value is <code>'\uD800'</code>.
	 * @since 1.5
	 */
	public static final char MIN_SURROGATE = MIN_HIGH_SURROGATE;
	/**
	 * The maximum Unicode surrogate code unit in the UTF-16 character encoding.
	 * This value is <code>'\uDFFF'</code>.
	 * @since 1.5
	 */
	public static final char MAX_SURROGATE = MAX_LOW_SURROGATE;
	/**
	 * The lowest possible supplementary Unicode code point (the first code
	 * point outside the basic multilingual plane (BMP)).
	 * This value is <code>0x10000</code>.
	 */
	public static final int MIN_SUPPLEMENTARY_CODE_POINT = 0x10000;
	/**
	 * The number of bits needed to represent a <code>char</code>.
	 * @since 1.5
	 */
	public static final int SIZE = 16;
	/**
	 * Lu = Letter, Uppercase (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte UPPERCASE_LETTER = 1;
	/**
	 * Ll = Letter, Lowercase (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte LOWERCASE_LETTER = 2;
	/**
	 * Lt = Letter, Titlecase (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte TITLECASE_LETTER = 3;
	/**
	 * Mn = Mark, Non-Spacing (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte NON_SPACING_MARK = 6;
	/**
	 * Mc = Mark, Spacing Combining (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte COMBINING_SPACING_MARK = 8;
	/**
	 * Me = Mark, Enclosing (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte ENCLOSING_MARK = 7;
	/**
	 * Nd = Number, Decimal Digit (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte DECIMAL_DIGIT_NUMBER = 9;
	/**
	 * Nl = Number, Letter (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte LETTER_NUMBER = 10;
	/**
	 * No = Number, Other (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte OTHER_NUMBER = 11;
	/**
	 * Zs = Separator, Space (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte SPACE_SEPARATOR = 12;
	/**
	 * Zl = Separator, Line (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte LINE_SEPARATOR = 13;
	/**
	 * Zp = Separator, Paragraph (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte PARAGRAPH_SEPARATOR = 14;
	/**
	 * Cc = Other, Control (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte CONTROL = 15;
	/**
	 * Cf = Other, Format (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte FORMAT = 16;
	/**
	 * Cs = Other, Surrogate (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte SURROGATE = 19;
	/**
	 * Co = Other, Private Use (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte PRIVATE_USE = 18;
	/**
	 * Cn = Other, Not Assigned (Normative).
	 *
	 * @since 1.1
	 */
	public static final byte UNASSIGNED = 0;
	/**
	 * Lm = Letter, Modifier (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte MODIFIER_LETTER = 4;
	/**
	 * Lo = Letter, Other (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte OTHER_LETTER = 5;
	/**
	 * Pc = Punctuation, Connector (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte CONNECTOR_PUNCTUATION = 23;
	/**
	 * Pd = Punctuation, Dash (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte DASH_PUNCTUATION = 20;
	/**
	 * Ps = Punctuation, Open (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte START_PUNCTUATION = 21;
	/**
	 * Pe = Punctuation, Close (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte END_PUNCTUATION = 22;
	/**
	 * Pi = Punctuation, Initial Quote (Informative).
	 *
	 * @since 1.4
	 */
	public static final byte INITIAL_QUOTE_PUNCTUATION = 29;
	/**
	 * Pf = Punctuation, Final Quote (Informative).
	 *
	 * @since 1.4
	 */
	public static final byte FINAL_QUOTE_PUNCTUATION = 30;
	/**
	 * Po = Punctuation, Other (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte OTHER_PUNCTUATION = 24;
	/**
	 * Sm = Symbol, Math (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte MATH_SYMBOL = 25;
	/**
	 * Sc = Symbol, Currency (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte CURRENCY_SYMBOL = 26;
	/**
	 * Sk = Symbol, Modifier (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte MODIFIER_SYMBOL = 27;
	/**
	 * So = Symbol, Other (Informative).
	 *
	 * @since 1.1
	 */
	public static final byte OTHER_SYMBOL = 28;
	/**
	 * Undefined bidirectional character type. Undefined char values have
	 * undefined directionality in the Unicode specification.
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_UNDEFINED = -1;
	/**
	 * Strong bidirectional character type "L".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_LEFT_TO_RIGHT = 0;
	/**
	 * Strong bidirectional character type "R".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_RIGHT_TO_LEFT = 1;
	/**
	 * Strong bidirectional character type "AL".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC = 2;
	/**
	 * Weak bidirectional character type "EN".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_EUROPEAN_NUMBER = 3;
	/**
	 * Weak bidirectional character type "ES".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_EUROPEAN_NUMBER_SEPARATOR = 4;
	/**
	 * Weak bidirectional character type "ET".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_EUROPEAN_NUMBER_TERMINATOR = 5;
	/**
	 * Weak bidirectional character type "AN".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_ARABIC_NUMBER = 6;
	/**
	 * Weak bidirectional character type "CS".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_COMMON_NUMBER_SEPARATOR = 7;
	/**
	 * Weak bidirectional character type "NSM".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_NONSPACING_MARK = 8;
	/**
	 * Weak bidirectional character type "BN".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_BOUNDARY_NEUTRAL = 9;
	/**
	 * Neutral bidirectional character type "B".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_PARAGRAPH_SEPARATOR = 10;
	/**
	 * Neutral bidirectional character type "S".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_SEGMENT_SEPARATOR = 11;
	/**
	 * Strong bidirectional character type "WS".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_WHITESPACE = 12;
	/**
	 * Neutral bidirectional character type "ON".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_OTHER_NEUTRALS = 13;
	/**
	 * Strong bidirectional character type "LRE".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_LEFT_TO_RIGHT_EMBEDDING = 14;
	/**
	 * Strong bidirectional character type "LRO".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_LEFT_TO_RIGHT_OVERRIDE = 15;
	/**
	 * Strong bidirectional character type "RLE".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_RIGHT_TO_LEFT_EMBEDDING = 16;
	/**
	 * Strong bidirectional character type "RLO".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_RIGHT_TO_LEFT_OVERRIDE = 17;
	/**
	 * Weak bidirectional character type "PDF".
	 *
	 * @since 1.4
	 */
	public static final byte DIRECTIONALITY_POP_DIRECTIONAL_FORMAT = 18;
	/**
	 * The character shift amount to look up the block offset. In other words,
	 * <code>(char) (BLOCKS.value[ch >> SHIFT[p]] + ch)</code> is the index
	 * where <code>ch</code> is described in <code>DATA</code> if <code>ch</code>
	 * is in Unicode plane <code>p</code>.  Note that <code>p</code> is simply
	 * the integer division of ch and 0x10000.
	 */
	private static int[] SHIFT;
	/**
	 * Stores unicode block offset lookup table. Exploit package visibility of
	 * String.value to avoid copying the array.
	 * @see #readCodePoint(int)
	 * @see CharData#BLOCKS
	 */
	private static char[][] blocks;
	/**
	 * Stores unicode attribute offset lookup table. Exploit package visibility
	 * of String.value to avoid copying the array.
	 * @see CharData#DATA
	 */
	private static char[][] data;
	/**
	 * Stores unicode uppercase attribute table. Exploit package visibility
	 * of String.value to avoid copying the array.
	 * @see CharData#UPPER
	 */
	private static char[][] upper;
	/**
	 * Stores unicode lowercase attribute table. Exploit package visibility
	 * of String.value to avoid copying the array.
	 * @see CharData#LOWER
	 */
	private static char[][] lower;

	static
	{
		try
		{
			DataInputStream dis = new DataInputStream(AuxClass.getResourceAsStream("/chars/chars.dat"));

			SHIFT = new int[dis.readInt()];

			for(int i = 0; i < SHIFT.length; i++)
			{
				SHIFT[i] = dis.readInt();
			}

			blocks = new char[dis.readInt()][];

			for(int i = 0; i < blocks.length; i++)
			{
				blocks[i] = readDeltaCodedString(dis);
			}

			data = new char[dis.readInt()][];

			for(int i = 0; i < data.length; i++)
			{
				data[i] = readDeltaCodedString(dis);
			}

			upper = new char[dis.readInt()][];

			for(int i = 0; i < upper.length; i++)
			{
				upper[i] = readDeltaCodedString(dis);
			}

			lower = new char[dis.readInt()][];

			for(int i = 0; i < lower.length; i++)
			{
				lower[i] = readDeltaCodedString(dis);
			}

			dis.close();
		}
		catch(Exception e)
		{
			e.printStackTrace();
		}
	}

	public static char[] readDeltaCodedString(DataInputStream dis) throws IOException
	{
		char[] cs = new char[dis.readInt()];

		if(cs.length > 0)
		{
			cs[0] = dis.readChar();

			for(int i = 1; i < cs.length; i++)
			{
				cs[i] = (char)(cs[i - 1] + dis.readChar());
			}
		}

		return cs;
	}
//	/**
//	 * Stores unicode numeric value attribute table. Exploit package visibility
//	 * of String.value to avoid copying the array.
//	 * @see CharData#NUM_VALUE
//	 */
//	private static final char[][] numValue =
//		new char[][]
//	{
//		CharData.NUM_VALUE[0].toCharArray(),
//		CharData.NUM_VALUE[1].toCharArray(),
//		CharData.NUM_VALUE[2].toCharArray(),
//		CharData.NUM_VALUE[3].toCharArray(),
//		CharData.NUM_VALUE[4].toCharArray(),
//		CharData.NUM_VALUE[5].toCharArray(),
//		CharData.NUM_VALUE[6].toCharArray(),
//		CharData.NUM_VALUE[7].toCharArray(),
//		CharData.NUM_VALUE[8].toCharArray(),
//		CharData.NUM_VALUE[9].toCharArray(),
//		CharData.NUM_VALUE[10].toCharArray(),
//		CharData.NUM_VALUE[11].toCharArray(),
//		CharData.NUM_VALUE[12].toCharArray(),
//		CharData.NUM_VALUE[13].toCharArray(),
//		CharData.NUM_VALUE[14].toCharArray(),
//		CharData.NUM_VALUE[15].toCharArray(),
//		CharData.NUM_VALUE[16].toCharArray()
//	};
//	/**
//	 * Stores unicode direction attribute table. Exploit package visibility
//	 * of String.value to avoid copying the array.
//	 * @see CharData#DIRECTION
//	 */
//	// Package visible for use by String.
//	static final char[][] direction =
//		new char[][]
//	{
//		CharData.DIRECTION[0].toCharArray(),
//		CharData.DIRECTION[1].toCharArray(),
//		CharData.DIRECTION[2].toCharArray(),
//		CharData.DIRECTION[3].toCharArray(),
//		CharData.DIRECTION[4].toCharArray(),
//		CharData.DIRECTION[5].toCharArray(),
//		CharData.DIRECTION[6].toCharArray(),
//		CharData.DIRECTION[7].toCharArray(),
//		CharData.DIRECTION[8].toCharArray(),
//		CharData.DIRECTION[9].toCharArray(),
//		CharData.DIRECTION[10].toCharArray(),
//		CharData.DIRECTION[11].toCharArray(),
//		CharData.DIRECTION[12].toCharArray(),
//		CharData.DIRECTION[13].toCharArray(),
//		CharData.DIRECTION[14].toCharArray(),
//		CharData.DIRECTION[15].toCharArray(),
//		CharData.DIRECTION[16].toCharArray()
//	};
//	/**
//	 * Stores unicode titlecase table. Exploit package visibility of
//	 * String.value to avoid copying the array.
//	 * @see CharData#TITLE
//	 */
//	private static final char[] title = CharData.TITLE.toCharArray();
	/**
	 * Mask for grabbing the type out of the contents of data.
	 * @see CharData#DATA
	 */
	private static final int TYPE_MASK = 0x1F;
	/**
	 * Mask for grabbing the non-breaking space flag out of the contents of
	 * data.
	 * @see CharData#DATA
	 */
	private static final int NO_BREAK_MASK = 0x20;
	/**
	 * Mask for grabbing the mirrored directionality flag out of the contents
	 * of data.
	 * @see CharData#DATA
	 */
	private static final int MIRROR_MASK = 0x40;

	/**
	 * Grabs an attribute offset from the Unicode attribute database. The lower
	 * 5 bits are the character type, the next 2 bits are flags, and the top
	 * 9 bits are the offset into the attribute tables.
	 *
	 * @param codePoint the character to look up
	 * @return the character's attribute offset and type
	 * @see #TYPE_MASK
	 * @see #NO_BREAK_MASK
	 * @see #MIRROR_MASK
	 * @see CharData#DATA
	 * @see CharData#SHIFT
	 */
	// Package visible for use in String.
	static char readCodePoint(int codePoint)
	{
		int plane = codePoint >>> 16;


		char offset = (char) (codePoint & 0xffff);


		return data[plane][(char) (blocks[plane][offset >> SHIFT[plane]] + offset)];


	}

	/**
	 * Determines if a character is a Unicode lowercase letter. For example,
	 * <code>'a'</code> is lowercase.  Returns true if getType() returns
	 * LOWERCASE_LETTER.
	 * <br>
	 * lowercase = [Ll]
	 *
	 * @param ch character to test
	 * @return true if ch is a Unicode lowercase letter, else false
	 * @see #isUpperCase(char)
	 * @see #isTitleCase(char)
	 * @see #toLowerCase(char)
	 * @see #getType(char)
	 */
	public static boolean isLowerCase(char ch)
	{
		return isLowerCase((int) ch);


	}

	/**
	 * Determines if a character is a Unicode lowercase letter. For example,
	 * <code>'a'</code> is lowercase.  Returns true if getType() returns
	 * LOWERCASE_LETTER.
	 * <br>
	 * lowercase = [Ll]
	 *
	 * @param codePoint character to test
	 * @return true if ch is a Unicode lowercase letter, else false
	 * @see #isUpperCase(char)
	 * @see #isTitleCase(char)
	 * @see #toLowerCase(char)
	 * @see #getType(char)
	 *
	 * @since 1.5
	 */
	public static boolean isLowerCase(int codePoint)
	{
		return getType(codePoint) == LOWERCASE_LETTER;


	}

	/**
	 * Determines if a character is a Unicode uppercase letter. For example,
	 * <code>'A'</code> is uppercase.  Returns true if getType() returns
	 * UPPERCASE_LETTER.
	 * <br>
	 * uppercase = [Lu]
	 *
	 * @param ch character to test
	 * @return true if ch is a Unicode uppercase letter, else false
	 * @see #isLowerCase(char)
	 * @see #isTitleCase(char)
	 * @see #toUpperCase(char)
	 * @see #getType(char)
	 */
	public static boolean isUpperCase(char ch)
	{
		return isUpperCase((int) ch);


	}

	/**
	 * Determines if a character is a Unicode uppercase letter. For example,
	 * <code>'A'</code> is uppercase.  Returns true if getType() returns
	 * UPPERCASE_LETTER.
	 * <br>
	 * uppercase = [Lu]
	 *
	 * @param codePoint character to test
	 * @return true if ch is a Unicode uppercase letter, else false
	 * @see #isLowerCase(char)
	 * @see #isTitleCase(char)
	 * @see #toUpperCase(char)
	 * @see #getType(char)
	 *
	 * @since 1.5
	 */
	public static boolean isUpperCase(int codePoint)
	{
		return getType(codePoint) == UPPERCASE_LETTER;


	}

	/**
	 * Determines if a character is a Unicode titlecase letter. For example,
	 * the character "Lj" (Latin capital L with small letter j) is titlecase.
	 * True if getType() returns TITLECASE_LETTER.
	 * <br>
	 * titlecase = [Lt]
	 *
	 * @param ch character to test
	 * @return true if ch is a Unicode titlecase letter, else false
	 * @see #isLowerCase(char)
	 * @see #isUpperCase(char)
	 * @see #toTitleCase(char)
	 * @see #getType(char)
	 */
	public static boolean isTitleCase(char ch)
	{
		return isTitleCase((int) ch);


	}

	/**
	 * Determines if a character is a Unicode titlecase letter. For example,
	 * the character "Lj" (Latin capital L with small letter j) is titlecase.
	 * True if getType() returns TITLECASE_LETTER.
	 * <br>
	 * titlecase = [Lt]
	 *
	 * @param codePoint character to test
	 * @return true if ch is a Unicode titlecase letter, else false
	 * @see #isLowerCase(char)
	 * @see #isUpperCase(char)
	 * @see #toTitleCase(char)
	 * @see #getType(char)
	 *
	 * @since 1.5
	 */
	public static boolean isTitleCase(int codePoint)
	{
		return getType(codePoint) == TITLECASE_LETTER;


	}

	/**
	 * Determines if a character is a Unicode decimal digit. For example,
	 * <code>'0'</code> is a digit.  A character is a Unicode digit if
	 * getType() returns DECIMAL_DIGIT_NUMBER.
	 * <br>
	 * Unicode decimal digit = [Nd]
	 *
	 * @param ch character to test
	 * @return true if ch is a Unicode decimal digit, else false
	 * @see #digit(char, int)
	 * @see #forDigit(int, int)
	 * @see #getType(char)
	 */
	public static boolean isDigit(char ch)
	{
		return isDigit((int) ch);


	}

	/**
	 * Determines if a character is a Unicode decimal digit. For example,
	 * <code>'0'</code> is a digit. A character is a Unicode digit if
	 * getType() returns DECIMAL_DIGIT_NUMBER.
	 * <br>
	 * Unicode decimal digit = [Nd]
	 *
	 * @param codePoint character to test
	 * @return true if ch is a Unicode decimal digit, else false
	 * @see #digit(char, int)
	 * @see #forDigit(int, int)
	 * @see #getType(char)
	 *
	 * @since 1.5
	 */
	public static boolean isDigit(int codePoint)
	{
		return getType(codePoint) == DECIMAL_DIGIT_NUMBER;


	}

	/**
	 * Determines if a character is part of the Unicode Standard. This is an
	 * evolving standard, but covers every character in the data file.
	 * <br>
	 * defined = not [Cn]
	 *
	 * @param ch character to test
	 * @return true if ch is a Unicode character, else false
	 * @see #isDigit(char)
	 * @see #isLetter(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isLowerCase(char)
	 * @see #isTitleCase(char)
	 * @see #isUpperCase(char)
	 */
	public static boolean isDefined(char ch)
	{
		return isDefined((int) ch);


	}

	/**
	 * Determines if a character is part of the Unicode Standard. This is an
	 * evolving standard, but covers every character in the data file.
	 * <br>
	 * defined = not [Cn]
	 *
	 * @param codePoint character to test
	 * @return true if ch is a Unicode character, else false
	 * @see #isDigit(char)
	 * @see #isLetter(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isLowerCase(char)
	 * @see #isTitleCase(char)
	 * @see #isUpperCase(char)
	 *
	 * @since 1.5
	 */
	public static boolean isDefined(int codePoint)
	{
		return getType(codePoint) != UNASSIGNED;


	}

	/**
	 * Determines if a character is a Unicode letter. Not all letters have case,
	 * so this may return true when isLowerCase and isUpperCase return false.
	 * A character is a Unicode letter if getType() returns one of
	 * UPPERCASE_LETTER, LOWERCASE_LETTER, TITLECASE_LETTER, MODIFIER_LETTER,
	 * or OTHER_LETTER.
	 * <br>
	 * letter = [Lu]|[Ll]|[Lt]|[Lm]|[Lo]
	 *
	 * @param ch character to test
	 * @return true if ch is a Unicode letter, else false
	 * @see #isDigit(char)
	 * @see #isJavaIdentifierStart(char)
	 * @see #isJavaLetter(char)
	 * @see #isJavaLetterOrDigit(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isLowerCase(char)
	 * @see #isTitleCase(char)
	 * @see #isUnicodeIdentifierStart(char)
	 * @see #isUpperCase(char)
	 */
	public static boolean isLetter(char ch)
	{
		return isLetter((int) ch);


	}

	/**
	 * Determines if a character is a Unicode letter. Not all letters have case,
	 * so this may return true when isLowerCase and isUpperCase return false.
	 * A character is a Unicode letter if getType() returns one of
	 * UPPERCASE_LETTER, LOWERCASE_LETTER, TITLECASE_LETTER, MODIFIER_LETTER,
	 * or OTHER_LETTER.
	 * <br>
	 * letter = [Lu]|[Ll]|[Lt]|[Lm]|[Lo]
	 *
	 * @param codePoint character to test
	 * @return true if ch is a Unicode letter, else false
	 * @see #isDigit(char)
	 * @see #isJavaIdentifierStart(char)
	 * @see #isJavaLetter(char)
	 * @see #isJavaLetterOrDigit(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isLowerCase(char)
	 * @see #isTitleCase(char)
	 * @see #isUnicodeIdentifierStart(char)
	 * @see #isUpperCase(char)
	 *
	 * @since 1.5
	 */
	public static boolean isLetter(int codePoint)
	{
		return ((1 << getType(codePoint))
			& ((1 << UPPERCASE_LETTER)
			| (1 << LOWERCASE_LETTER)
			| (1 << TITLECASE_LETTER)
			| (1 << MODIFIER_LETTER)
			| (1 << OTHER_LETTER))) != 0;


	}

	/**
	 * Returns the index into the given char subarray that is offset
	 * <code>codePointOffset</code> code points from <code>index</code>.
	 * @param a the char array
	 * @param start the start index of the subarray
	 * @param count the length of the subarray
	 * @param index the index to be offset
	 * @param codePointOffset the number of code points offset from <code>index
	 * </code>
	 * @return the index into the char array
	 *
	 * @throws NullPointerException if a is null
	 * @throws IndexOutOfBoundsException if start or count is negative or if
	 * start + count is greater than the length of the array
	 * @throws IndexOutOfBoundsException if index is less than start or larger
	 * than start + count
	 * @throws IndexOutOfBoundsException if codePointOffset is positive and the
	 * subarray from index to start + count - 1 has fewer than codePointOffset
	 * code points.
	 * @throws IndexOutOfBoundsException if codePointOffset is negative and the
	 * subarray from start to index - 1 has fewer than (-codePointOffset) code
	 * points
	 *
	 * @since 1.5
	 */
	public static int offsetByCodePoints(char[] a,
		int start,
		int count,
		int index,
		int codePointOffset)
	{
		int len = a.length;


		int end = start + count;


		if(start < 0 || count < 0 || end > len || index < start || index > end)
		{
			throw new IndexOutOfBoundsException();
		}



		int numToGo = codePointOffset;


		int offset = index;


		int adjust = 1;


		if(numToGo >= 0)
		{
			for(; numToGo
				> 0; offset++)
			{
				numToGo--;


				if(isHighSurrogate(a[offset])
					&& (offset + 1) < len
					&& isLowSurrogate(a[offset + 1]))
				{
					offset++;
				}


			}
			return offset;


		}
		else
		{
			numToGo *= -1;


			for(; numToGo
				> 0;)
			{
				numToGo--;
				offset--;

				if(isLowSurrogate(a[offset])
					&& (offset - 1) >= 0
					&& isHighSurrogate(a[offset - 1]))
				{
					offset--;
				}


				if(offset < start)
				{
					throw new IndexOutOfBoundsException();
				}


			}
			return offset;


		}

	}

	/**
	 * Returns the number of Unicode code points in the specified range of the
	 * given char array.  The first char in the range is at position
	 * offset and the length of the range is count.  Paired surrogates
	 * (supplementary characters are represented by a pair of chars -
	 * one from the high surrogates and one from the low surrogates)
	 * count as just one code point.
	 * @param a the char array to inspect
	 * @param offset the beginning of the range
	 * @param count the length of the range
	 * @return the number of Unicode code points in the given range of the
	 * array
	 * @throws NullPointerException if a is null
	 * @throws IndexOutOfBoundsException if offset or count is negative or if
	 * offset + countendIndex is larger than the length of a.
	 * @since 1.5
	 */
	public static int codePointCount(char[] a, int offset,
		int count)
	{
		int len = a.length;


		int end = offset + count;


		if(offset < 0 || count < 0 || end > len)
		{
			throw new IndexOutOfBoundsException();
		}



		int counter = 0;


		for(int i = offset; i
			< end; i++)
		{
			counter++;
			// If there is a pairing, count it only once.


			if(isHighSurrogate(a[i]) && (i + 1) < end
				&& isLowSurrogate(a[i + 1]))
			{
				i++;
			}


		}
		return counter;


	}

	/**
	 * Determines if a character is a Unicode letter or a Unicode digit. This
	 * is the combination of isLetter and isDigit.
	 * <br>
	 * letter or digit = [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nd]
	 *
	 * @param ch character to test
	 * @return true if ch is a Unicode letter or a Unicode digit, else false
	 * @see #isDigit(char)
	 * @see #isJavaIdentifierPart(char)
	 * @see #isJavaLetter(char)
	 * @see #isJavaLetterOrDigit(char)
	 * @see #isLetter(char)
	 * @see #isUnicodeIdentifierPart(char)
	 */
	public static boolean isLetterOrDigit(char ch)
	{
		return isLetterOrDigit((int) ch);


	}

	/**
	 * Determines if a character is a Unicode letter or a Unicode digit. This
	 * is the combination of isLetter and isDigit.
	 * <br>
	 * letter or digit = [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nd]
	 *
	 * @param codePoint character to test
	 * @return true if ch is a Unicode letter or a Unicode digit, else false
	 * @see #isDigit(char)
	 * @see #isJavaIdentifierPart(char)
	 * @see #isJavaLetter(char)
	 * @see #isJavaLetterOrDigit(char)
	 * @see #isLetter(char)
	 * @see #isUnicodeIdentifierPart(char)
	 *
	 * @since 1.5
	 */
	public static boolean isLetterOrDigit(int codePoint)
	{
		return ((1 << getType(codePoint))
			& ((1 << UPPERCASE_LETTER)
			| (1 << LOWERCASE_LETTER)
			| (1 << TITLECASE_LETTER)
			| (1 << MODIFIER_LETTER)
			| (1 << OTHER_LETTER)
			| (1 << DECIMAL_DIGIT_NUMBER))) != 0;


	}

	/**
	 * Determines if a character can start a Java identifier. This is the
	 * combination of isLetter, any character where getType returns
	 * LETTER_NUMBER, currency symbols (like '$'), and connecting punctuation
	 * (like '_').
	 *
	 * @param ch character to test
	 * @return true if ch can start a Java identifier, else false
	 * @deprecated Replaced by {@link #isJavaIdentifierStart(char)}
	 * @see #isJavaLetterOrDigit(char)
	 * @see #isJavaIdentifierStart(char)
	 * @see #isJavaIdentifierPart(char)
	 * @see #isLetter(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isUnicodeIdentifierStart(char)
	 */
	public static boolean isJavaLetter(char ch)
	{
		return isJavaIdentifierStart(ch);


	}

	/**
	 * Determines if a character can follow the first letter in
	 * a Java identifier.  This is the combination of isJavaLetter (isLetter,
	 * type of LETTER_NUMBER, currency, connecting punctuation) and digit,
	 * numeric letter (like Roman numerals), combining marks, non-spacing marks,
	 * or isIdentifierIgnorable.
	 *
	 * @param ch character to test
	 * @return true if ch can follow the first letter in a Java identifier
	 * @deprecated Replaced by {@link #isJavaIdentifierPart(char)}
	 * @see #isJavaLetter(char)
	 * @see #isJavaIdentifierStart(char)
	 * @see #isJavaIdentifierPart(char)
	 * @see #isLetter(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isUnicodeIdentifierPart(char)
	 * @see #isIdentifierIgnorable(char)
	 */
	public static boolean isJavaLetterOrDigit(char ch)
	{
		return isJavaIdentifierPart(ch);


	}

	/**
	 * Determines if a character can start a Java identifier. This is the
	 * combination of isLetter, any character where getType returns
	 * LETTER_NUMBER, currency symbols (like '$'), and connecting punctuation
	 * (like '_').
	 * <br>
	 * Java identifier start = [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nl]|[Sc]|[Pc]
	 *
	 * @param ch character to test
	 * @return true if ch can start a Java identifier, else false
	 * @see #isJavaIdentifierPart(char)
	 * @see #isLetter(char)
	 * @see #isUnicodeIdentifierStart(char)
	 * @since 1.1
	 */
	public static boolean isJavaIdentifierStart(char ch)
	{
		return isJavaIdentifierStart((int) ch);


	}

	/**
	 * Determines if a character can start a Java identifier. This is the
	 * combination of isLetter, any character where getType returns
	 * LETTER_NUMBER, currency symbols (like '$'), and connecting punctuation
	 * (like '_').
	 * <br>
	 * Java identifier start = [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nl]|[Sc]|[Pc]
	 *
	 * @param codePoint character to test
	 * @return true if ch can start a Java identifier, else false
	 * @see #isJavaIdentifierPart(char)
	 * @see #isLetter(char)
	 * @see #isUnicodeIdentifierStart(char)
	 * @since 1.5
	 */
	public static boolean isJavaIdentifierStart(int codePoint)
	{
		return ((1 << getType(codePoint))
			& ((1 << UPPERCASE_LETTER)
			| (1 << LOWERCASE_LETTER)
			| (1 << TITLECASE_LETTER)
			| (1 << MODIFIER_LETTER)
			| (1 << OTHER_LETTER)
			| (1 << LETTER_NUMBER)
			| (1 << CURRENCY_SYMBOL)
			| (1 << CONNECTOR_PUNCTUATION))) != 0;


	}

	/**
	 * Determines if a character can follow the first letter in
	 * a Java identifier.  This is the combination of isJavaLetter (isLetter,
	 * type of LETTER_NUMBER, currency, connecting punctuation) and digit,
	 * numeric letter (like Roman numerals), combining marks, non-spacing marks,
	 * or isIdentifierIgnorable.
	 * <br>
	 * Java identifier extender =
	 *   [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nl]|[Sc]|[Pc]|[Mn]|[Mc]|[Nd]|[Cf]
	 *   |U+0000-U+0008|U+000E-U+001B|U+007F-U+009F
	 *
	 * @param ch character to test
	 * @return true if ch can follow the first letter in a Java identifier
	 * @see #isIdentifierIgnorable(char)
	 * @see #isJavaIdentifierStart(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isUnicodeIdentifierPart(char)
	 * @since 1.1
	 */
	public static boolean isJavaIdentifierPart(char ch)
	{
		return isJavaIdentifierPart((int) ch);


	}

	/**
	 * Determines if a character can follow the first letter in
	 * a Java identifier.  This is the combination of isJavaLetter (isLetter,
	 * type of LETTER_NUMBER, currency, connecting punctuation) and digit,
	 * numeric letter (like Roman numerals), combining marks, non-spacing marks,
	 * or isIdentifierIgnorable.
	 * <br>
	 * Java identifier extender =
	 *   [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nl]|[Sc]|[Pc]|[Mn]|[Mc]|[Nd]|[Cf]
	 *   |U+0000-U+0008|U+000E-U+001B|U+007F-U+009F
	 *
	 * @param codePoint character to test
	 * @return true if ch can follow the first letter in a Java identifier
	 * @see #isIdentifierIgnorable(char)
	 * @see #isJavaIdentifierStart(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isUnicodeIdentifierPart(char)
	 * @since 1.5
	 */
	public static boolean isJavaIdentifierPart(int codePoint)
	{
		int category = getType(codePoint);


		return ((1 << category)
			& ((1 << UPPERCASE_LETTER)
			| (1 << LOWERCASE_LETTER)
			| (1 << TITLECASE_LETTER)
			| (1 << MODIFIER_LETTER)
			| (1 << OTHER_LETTER)
			| (1 << NON_SPACING_MARK)
			| (1 << COMBINING_SPACING_MARK)
			| (1 << DECIMAL_DIGIT_NUMBER)
			| (1 << LETTER_NUMBER)
			| (1 << CURRENCY_SYMBOL)
			| (1 << CONNECTOR_PUNCTUATION)
			| (1 << FORMAT))) != 0
			|| (category == CONTROL && isIdentifierIgnorable(codePoint));


	}

	/**
	 * Determines if a character can start a Unicode identifier.  Only
	 * letters can start a Unicode identifier, but this includes characters
	 * in LETTER_NUMBER.
	 * <br>
	 * Unicode identifier start = [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nl]
	 *
	 * @param ch character to test
	 * @return true if ch can start a Unicode identifier, else false
	 * @see #isJavaIdentifierStart(char)
	 * @see #isLetter(char)
	 * @see #isUnicodeIdentifierPart(char)
	 * @since 1.1
	 */
	public static boolean isUnicodeIdentifierStart(char ch)
	{
		return isUnicodeIdentifierStart((int) ch);


	}

	/**
	 * Determines if a character can start a Unicode identifier.  Only
	 * letters can start a Unicode identifier, but this includes characters
	 * in LETTER_NUMBER.
	 * <br>
	 * Unicode identifier start = [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nl]
	 *
	 * @param codePoint character to test
	 * @return true if ch can start a Unicode identifier, else false
	 * @see #isJavaIdentifierStart(char)
	 * @see #isLetter(char)
	 * @see #isUnicodeIdentifierPart(char)
	 * @since 1.5
	 */
	public static boolean isUnicodeIdentifierStart(int codePoint)
	{
		return ((1 << getType(codePoint))
			& ((1 << UPPERCASE_LETTER)
			| (1 << LOWERCASE_LETTER)
			| (1 << TITLECASE_LETTER)
			| (1 << MODIFIER_LETTER)
			| (1 << OTHER_LETTER)
			| (1 << LETTER_NUMBER))) != 0;


	}

	/**
	 * Determines if a character can follow the first letter in
	 * a Unicode identifier. This includes letters, connecting punctuation,
	 * digits, numeric letters, combining marks, non-spacing marks, and
	 * isIdentifierIgnorable.
	 * <br>
	 * Unicode identifier extender =
	 *   [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nl]|[Mn]|[Mc]|[Nd]|[Pc]|[Cf]|
	 *   |U+0000-U+0008|U+000E-U+001B|U+007F-U+009F
	 *
	 * @param ch character to test
	 * @return true if ch can follow the first letter in a Unicode identifier
	 * @see #isIdentifierIgnorable(char)
	 * @see #isJavaIdentifierPart(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isUnicodeIdentifierStart(char)
	 * @since 1.1
	 */
	public static boolean isUnicodeIdentifierPart(char ch)
	{
		return isUnicodeIdentifierPart((int) ch);


	}

	/**
	 * Determines if a character can follow the first letter in
	 * a Unicode identifier. This includes letters, connecting punctuation,
	 * digits, numeric letters, combining marks, non-spacing marks, and
	 * isIdentifierIgnorable.
	 * <br>
	 * Unicode identifier extender =
	 *   [Lu]|[Ll]|[Lt]|[Lm]|[Lo]|[Nl]|[Mn]|[Mc]|[Nd]|[Pc]|[Cf]|
	 *   |U+0000-U+0008|U+000E-U+001B|U+007F-U+009F
	 *
	 * @param codePoint character to test
	 * @return true if ch can follow the first letter in a Unicode identifier
	 * @see #isIdentifierIgnorable(char)
	 * @see #isJavaIdentifierPart(char)
	 * @see #isLetterOrDigit(char)
	 * @see #isUnicodeIdentifierStart(char)
	 * @since 1.5
	 */
	public static boolean isUnicodeIdentifierPart(int codePoint)
	{
		int category = getType(codePoint);


		return ((1 << category)
			& ((1 << UPPERCASE_LETTER)
			| (1 << LOWERCASE_LETTER)
			| (1 << TITLECASE_LETTER)
			| (1 << MODIFIER_LETTER)
			| (1 << OTHER_LETTER)
			| (1 << NON_SPACING_MARK)
			| (1 << COMBINING_SPACING_MARK)
			| (1 << DECIMAL_DIGIT_NUMBER)
			| (1 << LETTER_NUMBER)
			| (1 << CONNECTOR_PUNCTUATION)
			| (1 << FORMAT))) != 0
			|| (category == CONTROL && isIdentifierIgnorable(codePoint));


	}

	/**
	 * Determines if a character is ignorable in a Unicode identifier. This
	 * includes the non-whitespace ISO control characters (<code>'\u0000'</code>
	 * through <code>'\u0008'</code>, <code>'\u000E'</code> through
	 * <code>'\u001B'</code>, and <code>'\u007F'</code> through
	 * <code>'\u009F'</code>), and FORMAT characters.
	 * <br>
	 * Unicode identifier ignorable = [Cf]|U+0000-U+0008|U+000E-U+001B
	 *    |U+007F-U+009F
	 *
	 * @param ch character to test
	 * @return true if ch is ignorable in a Unicode or Java identifier
	 * @see #isJavaIdentifierPart(char)
	 * @see #isUnicodeIdentifierPart(char)
	 * @since 1.1
	 */
	public static boolean isIdentifierIgnorable(char ch)
	{
		return isIdentifierIgnorable((int) ch);


	}

	/**
	 * Determines if a character is ignorable in a Unicode identifier. This
	 * includes the non-whitespace ISO control characters (<code>'\u0000'</code>
	 * through <code>'\u0008'</code>, <code>'\u000E'</code> through
	 * <code>'\u001B'</code>, and <code>'\u007F'</code> through
	 * <code>'\u009F'</code>), and FORMAT characters.
	 * <br>
	 * Unicode identifier ignorable = [Cf]|U+0000-U+0008|U+000E-U+001B
	 *    |U+007F-U+009F
	 *
	 * @param codePoint character to test
	 * @return true if ch is ignorable in a Unicode or Java identifier
	 * @see #isJavaIdentifierPart(char)
	 * @see #isUnicodeIdentifierPart(char)
	 * @since 1.5
	 */
	public static boolean isIdentifierIgnorable(int codePoint)
	{
		if((codePoint >= 0 && codePoint <= 0x0008)
			|| (codePoint >= 0x000E && codePoint <= 0x001B)
			|| (codePoint >= 0x007F && codePoint <= 0x009F)
			|| getType(codePoint) == FORMAT)
		{
			return true;
		}


		return false;


	}

	/**
	 * Converts a Unicode character into its lowercase equivalent mapping.
	 * If a mapping does not exist, then the character passed is returned.
	 * Note that isLowerCase(toLowerCase(ch)) does not always return true.
	 *
	 * @param ch character to convert to lowercase
	 * @return lowercase mapping of ch, or ch if lowercase mapping does
	 *         not exist
	 * @see #isLowerCase(char)
	 * @see #isUpperCase(char)
	 * @see #toTitleCase(char)
	 * @see #toUpperCase(char)
	 */
	public static char toLowerCase(char ch)
	{
		return (char) (lower[0][readCodePoint((int) ch) >>> 7] + ch);
	}

	/**
	 * Converts a Unicode character into its lowercase equivalent mapping.
	 * If a mapping does not exist, then the character passed is returned.
	 * Note that isLowerCase(toLowerCase(ch)) does not always return true.
	 *
	 * @param codePoint character to convert to lowercase
	 * @return lowercase mapping of ch, or ch if lowercase mapping does
	 *         not exist
	 * @see #isLowerCase(char)
	 * @see #isUpperCase(char)
	 * @see #toTitleCase(char)
	 * @see #toUpperCase(char)
	 *
	 * @since 1.5
	 */
	public static int toLowerCase(int codePoint)
	{
		// If the code point is unassigned or in one of the private use areas
		// then we delegate the call to the appropriate private static inner class.
		int plane = codePoint >>> 16;


		if(plane > 2 && plane < 14)
		{
			return UnassignedCharacters.toLowerCase(codePoint);
		}


		if(plane > 14)
		{
			return PrivateUseCharacters.toLowerCase(codePoint);
		}

		// The short value stored in lower[plane] is the signed difference between
		// codePoint and its lowercase conversion.


		return ((short) lower[plane][readCodePoint(codePoint) >>> 7]) + codePoint;
	}

	/**
	 * Converts a Unicode character into its uppercase equivalent mapping.
	 * If a mapping does not exist, then the character passed is returned.
	 * Note that isUpperCase(toUpperCase(ch)) does not always return true.
	 *
	 * @param ch character to convert to uppercase
	 * @return uppercase mapping of ch, or ch if uppercase mapping does
	 *         not exist
	 * @see #isLowerCase(char)
	 * @see #isUpperCase(char)
	 * @see #toLowerCase(char)
	 * @see #toTitleCase(char)
	 */
	public static char toUpperCase(char ch)
	{
		return (char) (upper[0][readCodePoint((int) ch) >>> 7] + ch);
	}

	/**
	 * Converts a Unicode character into its uppercase equivalent mapping.
	 * If a mapping does not exist, then the character passed is returned.
	 * Note that isUpperCase(toUpperCase(ch)) does not always return true.
	 *
	 * @param codePoint character to convert to uppercase
	 * @return uppercase mapping of ch, or ch if uppercase mapping does
	 *         not exist
	 * @see #isLowerCase(char)
	 * @see #isUpperCase(char)
	 * @see #toLowerCase(char)
	 * @see #toTitleCase(char)
	 *
	 * @since 1.5
	 */
	public static int toUpperCase(int codePoint)
	{
		// If the code point is unassigned or in one of the private use areas
		// then we delegate the call to the appropriate private static inner class.
		int plane = codePoint >>> 16;


		if(plane > 2 && plane < 14)
		{
			return UnassignedCharacters.toUpperCase(codePoint);
		}


		if(plane > 14)
		{
			return PrivateUseCharacters.toUpperCase(codePoint);
		}

		// The short value stored in upper[plane] is the signed difference between
		// codePoint and its uppercase conversion.


		return ((short) upper[plane][readCodePoint(codePoint) >>> 7]) + codePoint;
	}

	public static String toLowerCase(String text)
	{
		char[] cs = text.toCharArray();

		for(int i = 0; i < cs.length; i++)
		{
			cs[i] = toLowerCase(cs[i]);
		}

		return new String(cs);
	}

	public static String toUpperCase(String text)
	{
		char[] cs = text.toCharArray();

		for(int i = 0; i < cs.length; i++)
		{
			cs[i] = toUpperCase(cs[i]);
		}

		return new String(cs);
	}

//	/**
//	 * Converts a Unicode character into its titlecase equivalent mapping.
//	 * If a mapping does not exist, then the character passed is returned.
//	 * Note that isTitleCase(toTitleCase(ch)) does not always return true.
//	 *
//	 * @param ch character to convert to titlecase
//	 * @return titlecase mapping of ch, or ch if titlecase mapping does
//	 *         not exist
//	 * @see #isTitleCase(char)
//	 * @see #toLowerCase(char)
//	 * @see #toUpperCase(char)
//	 */
//	public static char toTitleCase(char ch)
//	{
//		// As title is short, it doesn't hurt to exhaustively iterate over it.
//		for(int i = title.length - 2; i
//			>= 0; i -= 2)
//		{
//			if(title[i] == ch)
//			{
//				return title[i + 1];
//			}
//		}
//
//
//		return toUpperCase(ch);
//
//
//	}
//	/**
//	 * Converts a Unicode character into its titlecase equivalent mapping.
//	 * If a mapping does not exist, then the character passed is returned.
//	 * Note that isTitleCase(toTitleCase(ch)) does not always return true.
//	 *
//	 * @param codePoint character to convert to titlecase
//	 * @return titlecase mapping of ch, or ch if titlecase mapping does
//	 *         not exist
//	 * @see #isTitleCase(char)
//	 * @see #toLowerCase(char)
//	 * @see #toUpperCase(char)
//	 *
//	 * @since 1.5
//	 */
//	public static int toTitleCase(int codePoint)
//	{
//		// As of Unicode 4.0.0 no characters outside of plane 0 have
//		// titlecase mappings that are different from their uppercase
//		// mapping.
//		if(codePoint < 0x10000)
//		{
//			return (int) toTitleCase((char) codePoint);
//		}
//
//
//		return toUpperCase(codePoint);
//
//
//	}
//	/**
//	 * Converts a character into a digit of the specified radix. If the radix
//	 * exceeds MIN_RADIX or MAX_RADIX, or if the result of getNumericValue(ch)
//	 * exceeds the radix, or if ch is not a decimal digit or in the case
//	 * insensitive set of 'a'-'z', the result is -1.
//	 * <br>
//	 * character argument boundary = [Nd]|U+0041-U+005A|U+0061-U+007A
//	 *    |U+FF21-U+FF3A|U+FF41-U+FF5A
//	 *
//	 * @param ch character to convert into a digit
//	 * @param radix radix in which ch is a digit
//	 * @return digit which ch represents in radix, or -1 not a valid digit
//	 * @see #MIN_RADIX
//	 * @see #MAX_RADIX
//	 * @see #forDigit(int, int)
//	 * @see #isDigit(char)
//	 * @see #getNumericValue(char)
//	 */
//	public static int digit(char ch, int radix)
//	{
//		if(radix < MIN_RADIX || radix > MAX_RADIX)
//		{
//			return -1;
//		}
//
//
//		char attr = readCodePoint((int) ch);
//
//
//		if(((1 << (attr & TYPE_MASK))
//			& ((1 << UPPERCASE_LETTER)
//			| (1 << LOWERCASE_LETTER)
//			| (1 << DECIMAL_DIGIT_NUMBER))) != 0)
//		{
//			// Signedness doesn't matter; 0xffff vs. -1 are both rejected.
//			int digit = numValue[0][attr >> 7];
//
//
//			return (digit < radix) ? digit : -1;
//
//
//		}
//		return -1;
//
//
//	}
//	/**
//	 * Converts a character into a digit of the specified radix. If the radix
//	 * exceeds MIN_RADIX or MAX_RADIX, or if the result of getNumericValue(ch)
//	 * exceeds the radix, or if ch is not a decimal digit or in the case
//	 * insensitive set of 'a'-'z', the result is -1.
//	 * <br>
//	 * character argument boundary = [Nd]|U+0041-U+005A|U+0061-U+007A
//	 *    |U+FF21-U+FF3A|U+FF41-U+FF5A
//	 *
//	 * @param codePoint character to convert into a digit
//	 * @param radix radix in which ch is a digit
//	 * @return digit which ch represents in radix, or -1 not a valid digit
//	 * @see #MIN_RADIX
//	 * @see #MAX_RADIX
//	 * @see #forDigit(int, int)
//	 * @see #isDigit(char)
//	 * @see #getNumericValue(char)
//	 */
//	public static int digit(int codePoint, int radix)
//	{
//		if(radix < MIN_RADIX || radix > MAX_RADIX)
//		{
//			return -1;
//		}
//
//		// If the code point is unassigned or in one of the private use areas
//		// then we delegate the call to the appropriate private static inner class.
//
//
//		int plane = codePoint >>> 16;
//
//
//		if(plane > 2 && plane < 14)
//		{
//			return UnassignedCharacters.digit(codePoint, radix);
//		}
//
//
//		if(plane > 14)
//		{
//			return PrivateUseCharacters.digit(codePoint, radix);
//		}
//
//
//		char attr = readCodePoint(codePoint);
//
//
//		if(((1 << (attr & TYPE_MASK))
//			& ((1 << UPPERCASE_LETTER)
//			| (1 << LOWERCASE_LETTER)
//			| (1 << DECIMAL_DIGIT_NUMBER))) != 0)
//		{
//			// Signedness doesn't matter; 0xffff vs. -1 are both rejected.
//			int digit = numValue[plane][attr >> 7];
//
//			// If digit is less than or equal to -3 then the numerical value was
//			// too large to fit into numValue and is stored in CharData.LARGENUMS.
//
//
//			if(digit <= -3)
//			{
//				digit = CharData.LARGENUMS[-digit - 3];
//			}
//
//
//			return (digit < radix) ? digit : -1;
//
//
//		}
//		return -1;
//
//
//	}
//	/**
//	 * Returns the Unicode numeric value property of a character. For example,
//	 * <code>'\\u216C'</code> (the Roman numeral fifty) returns 50.
//	 *
//	 * <p>This method also returns values for the letters A through Z, (not
//	 * specified by Unicode), in these ranges: <code>'\u0041'</code>
//	 * through <code>'\u005A'</code> (uppercase); <code>'\u0061'</code>
//	 * through <code>'\u007A'</code> (lowercase); and <code>'\uFF21'</code>
//	 * through <code>'\uFF3A'</code>, <code>'\uFF41'</code> through
//	 * <code>'\uFF5A'</code> (full width variants).
//	 *
//	 * <p>If the character lacks a numeric value property, -1 is returned.
//	 * If the character has a numeric value property which is not representable
//	 * as a nonnegative integer, such as a fraction, -2 is returned.
//	 *
//	 * character argument boundary = [Nd]|[Nl]|[No]|U+0041-U+005A|U+0061-U+007A
//	 *    |U+FF21-U+FF3A|U+FF41-U+FF5A
//	 *
//	 * @param ch character from which the numeric value property will
//	 *        be retrieved
//	 * @return the numeric value property of ch, or -1 if it does not exist, or
//	 *         -2 if it is not representable as a nonnegative integer
//	 * @see #forDigit(int, int)
//	 * @see #digit(char, int)
//	 * @see #isDigit(char)
//	 * @since 1.1
//	 */
//	public static int getNumericValue(char ch)
//	{
//		// Treat numValue as signed.
//		return (short) numValue[0][readCodePoint((int) ch) >> 7];
//
//
//	}
//	/**
//	 * Returns the Unicode numeric value property of a character. For example,
//	 * <code>'\\u216C'</code> (the Roman numeral fifty) returns 50.
//	 *
//	 * <p>This method also returns values for the letters A through Z, (not
//	 * specified by Unicode), in these ranges: <code>'\u0041'</code>
//	 * through <code>'\u005A'</code> (uppercase); <code>'\u0061'</code>
//	 * through <code>'\u007A'</code> (lowercase); and <code>'\uFF21'</code>
//	 * through <code>'\uFF3A'</code>, <code>'\uFF41'</code> through
//	 * <code>'\uFF5A'</code> (full width variants).
//	 *
//	 * <p>If the character lacks a numeric value property, -1 is returned.
//	 * If the character has a numeric value property which is not representable
//	 * as a nonnegative integer, such as a fraction, -2 is returned.
//	 *
//	 * character argument boundary = [Nd]|[Nl]|[No]|U+0041-U+005A|U+0061-U+007A
//	 *    |U+FF21-U+FF3A|U+FF41-U+FF5A
//	 *
//	 * @param codePoint character from which the numeric value property will
//	 *        be retrieved
//	 * @return the numeric value property of ch, or -1 if it does not exist, or
//	 *         -2 if it is not representable as a nonnegative integer
//	 * @see #forDigit(int, int)
//	 * @see #digit(char, int)
//	 * @see #isDigit(char)
//	 * @since 1.5
//	 */
//	public static int getNumericValue(int codePoint)
//	{
//		// If the code point is unassigned or in one of the private use areas
//		// then we delegate the call to the appropriate private static inner class.
//		int plane = codePoint >>> 16;
//
//
//		if(plane > 2 && plane < 14)
//		{
//			return UnassignedCharacters.getNumericValue(codePoint);
//		}
//
//
//		if(plane > 14)
//		{
//			return PrivateUseCharacters.getNumericValue(codePoint);
//		}
//
//		// If the value N found in numValue[plane] is less than or equal to -3
//		// then the numeric value was too big to fit into 16 bits and is
//		// stored in CharData.LARGENUMS at offset (-N - 3).
//
//
//		short num = (short) numValue[plane][readCodePoint(codePoint) >> 7];
//
//
//		if(num <= -3)
//		{
//			return CharData.LARGENUMS[-num - 3];
//		}
//
//
//		return num;
//
//
//	}
	/**
	 * Determines if a character is a ISO-LATIN-1 space. This is only the five
	 * characters <code>'\t'</code>, <code>'\n'</code>, <code>'\f'</code>,
	 * <code>'\r'</code>, and <code>' '</code>.
	 * <br>
	 * Java space = U+0020|U+0009|U+000A|U+000C|U+000D
	 *
	 * @param ch character to test
	 * @return true if ch is a space, else false
	 * @deprecated Replaced by {@link #isWhitespace(char)}
	 * @see #isSpaceChar(char)
	 * @see #isWhitespace(char)
	 */
	public static boolean isSpace(char ch)
	{
		// Performing the subtraction up front alleviates need to compare longs.
		return ch-- <= ' ' && ((1 << ch)
			& ((1 << (' ' - 1))
			| (1 << ('\t' - 1))
			| (1 << ('\n' - 1))
			| (1 << ('\r' - 1))
			| (1 << ('\f' - 1)))) != 0;


	}

	/**
	 * Determines if a character is a Unicode space character. This includes
	 * SPACE_SEPARATOR, LINE_SEPARATOR, and PARAGRAPH_SEPARATOR.
	 * <br>
	 * Unicode space = [Zs]|[Zp]|[Zl]
	 *
	 * @param ch character to test
	 * @return true if ch is a Unicode space, else false
	 * @see #isWhitespace(char)
	 * @since 1.1
	 */
	public static boolean isSpaceChar(char ch)
	{
		return isSpaceChar((int) ch);


	}

	/**
	 * Determines if a character is a Unicode space character. This includes
	 * SPACE_SEPARATOR, LINE_SEPARATOR, and PARAGRAPH_SEPARATOR.
	 * <br>
	 * Unicode space = [Zs]|[Zp]|[Zl]
	 *
	 * @param codePoint character to test
	 * @return true if ch is a Unicode space, else false
	 * @see #isWhitespace(char)
	 * @since 1.5
	 */
	public static boolean isSpaceChar(int codePoint)
	{
		return ((1 << getType(codePoint))
			& ((1 << SPACE_SEPARATOR)
			| (1 << LINE_SEPARATOR)
			| (1 << PARAGRAPH_SEPARATOR))) != 0;


	}

	/**
	 * Determines if a character is Java whitespace. This includes Unicode
	 * space characters (SPACE_SEPARATOR, LINE_SEPARATOR, and
	 * PARAGRAPH_SEPARATOR) except the non-breaking spaces
	 * (<code>'\u00A0'</code>, <code>'\u2007'</code>, and <code>'\u202F'</code>);
	 * and these characters: <code>'\u0009'</code>, <code>'\u000A'</code>,
	 * <code>'\u000B'</code>, <code>'\u000C'</code>, <code>'\u000D'</code>,
	 * <code>'\u001C'</code>, <code>'\u001D'</code>, <code>'\u001E'</code>,
	 * and <code>'\u001F'</code>.
	 * <br>
	 * Java whitespace = ([Zs] not Nb)|[Zl]|[Zp]|U+0009-U+000D|U+001C-U+001F
	 *
	 * @param ch character to test
	 * @return true if ch is Java whitespace, else false
	 * @see #isSpaceChar(char)
	 * @since 1.1
	 */
	public static boolean isWhitespace(char ch)
	{
		return isWhitespace((int) ch);


	}

	/**
	 * Determines if a character is Java whitespace. This includes Unicode
	 * space characters (SPACE_SEPARATOR, LINE_SEPARATOR, and
	 * PARAGRAPH_SEPARATOR) except the non-breaking spaces
	 * (<code>'\u00A0'</code>, <code>'\u2007'</code>, and <code>'\u202F'</code>);
	 * and these characters: <code>'\u0009'</code>, <code>'\u000A'</code>,
	 * <code>'\u000B'</code>, <code>'\u000C'</code>, <code>'\u000D'</code>,
	 * <code>'\u001C'</code>, <code>'\u001D'</code>, <code>'\u001E'</code>,
	 * and <code>'\u001F'</code>.
	 * <br>
	 * Java whitespace = ([Zs] not Nb)|[Zl]|[Zp]|U+0009-U+000D|U+001C-U+001F
	 *
	 * @param codePoint character to test
	 * @return true if ch is Java whitespace, else false
	 * @see #isSpaceChar(char)
	 * @since 1.5
	 */
	public static boolean isWhitespace(int codePoint)
	{
		int plane = codePoint >>> 16;


		if(plane > 2 && plane < 14)
		{
			return UnassignedCharacters.isWhiteSpace(codePoint);
		}


		if(plane > 14)
		{
			return PrivateUseCharacters.isWhiteSpace(codePoint);
		}



		int attr = readCodePoint(codePoint);


		return ((((1 << (attr & TYPE_MASK))
			& ((1 << SPACE_SEPARATOR)
			| (1 << LINE_SEPARATOR)
			| (1 << PARAGRAPH_SEPARATOR))) != 0)
			&& (attr & NO_BREAK_MASK) == 0)
			|| (codePoint <= '\u001F' && ((1 << codePoint)
			& ((1 << '\t')
			| (1 << '\n')
			| (1 << '\u000B')
			| (1 << '\u000C')
			| (1 << '\r')
			| (1 << '\u001C')
			| (1 << '\u001D')
			| (1 << '\u001E')
			| (1 << '\u001F'))) != 0);


	}

	/**
	 * Determines if a character has the ISO Control property.
	 * <br>
	 * ISO Control = [Cc]
	 *
	 * @param ch character to test
	 * @return true if ch is an ISO Control character, else false
	 * @see #isSpaceChar(char)
	 * @see #isWhitespace(char)
	 * @since 1.1
	 */
	public static boolean isISOControl(char ch)
	{
		return isISOControl((int) ch);


	}

	/**
	 * Determines if the character is an ISO Control character.  This is true
	 * if the code point is in the range [0, 0x001F] or if it is in the range
	 * [0x007F, 0x009F].
	 * @param codePoint the character to check
	 * @return true if the character is in one of the above ranges
	 *
	 * @since 1.5
	 */
	public static boolean isISOControl(int codePoint)
	{
		if((codePoint >= 0 && codePoint <= 0x001F)
			|| (codePoint >= 0x007F && codePoint <= 0x009F))
		{
			return true;
		}


		return false;


	}

	/**
	 * Returns the Unicode general category property of a character.
	 *
	 * @param ch character from which the general category property will
	 *        be retrieved
	 * @return the character category property of ch as an integer
	 * @see #UNASSIGNED
	 * @see #UPPERCASE_LETTER
	 * @see #LOWERCASE_LETTER
	 * @see #TITLECASE_LETTER
	 * @see #MODIFIER_LETTER
	 * @see #OTHER_LETTER
	 * @see #NON_SPACING_MARK
	 * @see #ENCLOSING_MARK
	 * @see #COMBINING_SPACING_MARK
	 * @see #DECIMAL_DIGIT_NUMBER
	 * @see #LETTER_NUMBER
	 * @see #OTHER_NUMBER
	 * @see #SPACE_SEPARATOR
	 * @see #LINE_SEPARATOR
	 * @see #PARAGRAPH_SEPARATOR
	 * @see #CONTROL
	 * @see #FORMAT
	 * @see #PRIVATE_USE
	 * @see #SURROGATE
	 * @see #DASH_PUNCTUATION
	 * @see #START_PUNCTUATION
	 * @see #END_PUNCTUATION
	 * @see #CONNECTOR_PUNCTUATION
	 * @see #OTHER_PUNCTUATION
	 * @see #MATH_SYMBOL
	 * @see #CURRENCY_SYMBOL
	 * @see #MODIFIER_SYMBOL
	 * @see #INITIAL_QUOTE_PUNCTUATION
	 * @see #FINAL_QUOTE_PUNCTUATION
	 * @since 1.1
	 */
	public static int getType(char ch)
	{
		return getType((int) ch);


	}

	/**
	 * Returns the Unicode general category property of a character.
	 *
	 * @param codePoint character from which the general category property will
	 *        be retrieved
	 * @return the character category property of ch as an integer
	 * @see #UNASSIGNED
	 * @see #UPPERCASE_LETTER
	 * @see #LOWERCASE_LETTER
	 * @see #TITLECASE_LETTER
	 * @see #MODIFIER_LETTER
	 * @see #OTHER_LETTER
	 * @see #NON_SPACING_MARK
	 * @see #ENCLOSING_MARK
	 * @see #COMBINING_SPACING_MARK
	 * @see #DECIMAL_DIGIT_NUMBER
	 * @see #LETTER_NUMBER
	 * @see #OTHER_NUMBER
	 * @see #SPACE_SEPARATOR
	 * @see #LINE_SEPARATOR
	 * @see #PARAGRAPH_SEPARATOR
	 * @see #CONTROL
	 * @see #FORMAT
	 * @see #PRIVATE_USE
	 * @see #SURROGATE
	 * @see #DASH_PUNCTUATION
	 * @see #START_PUNCTUATION
	 * @see #END_PUNCTUATION
	 * @see #CONNECTOR_PUNCTUATION
	 * @see #OTHER_PUNCTUATION
	 * @see #MATH_SYMBOL
	 * @see #CURRENCY_SYMBOL
	 * @see #MODIFIER_SYMBOL
	 * @see #INITIAL_QUOTE_PUNCTUATION
	 * @see #FINAL_QUOTE_PUNCTUATION
	 *
	 * @since 1.5
	 */
	public static int getType(int codePoint)
	{
		// If the codePoint is unassigned or in one of the private use areas
		// then we delegate the call to the appropriate private static inner class.
		int plane = codePoint >>> 16;


		if(plane > 2 && plane < 14)
		{
			return UnassignedCharacters.getType(codePoint);
		}


		if(plane > 14)
		{
			return PrivateUseCharacters.getType(codePoint);
		}



		return readCodePoint(codePoint) & TYPE_MASK;


	}

//	/**
//	 * Returns the Unicode directionality property of the character. This
//	 * is used in the visual ordering of text.
//	 *
//	 * @param ch the character to look up
//	 * @return the directionality constant, or DIRECTIONALITY_UNDEFINED
//	 * @see #DIRECTIONALITY_UNDEFINED
//	 * @see #DIRECTIONALITY_LEFT_TO_RIGHT
//	 * @see #DIRECTIONALITY_RIGHT_TO_LEFT
//	 * @see #DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC
//	 * @see #DIRECTIONALITY_EUROPEAN_NUMBER
//	 * @see #DIRECTIONALITY_EUROPEAN_NUMBER_SEPARATOR
//	 * @see #DIRECTIONALITY_EUROPEAN_NUMBER_TERMINATOR
//	 * @see #DIRECTIONALITY_ARABIC_NUMBER
//	 * @see #DIRECTIONALITY_COMMON_NUMBER_SEPARATOR
//	 * @see #DIRECTIONALITY_NONSPACING_MARK
//	 * @see #DIRECTIONALITY_BOUNDARY_NEUTRAL
//	 * @see #DIRECTIONALITY_PARAGRAPH_SEPARATOR
//	 * @see #DIRECTIONALITY_SEGMENT_SEPARATOR
//	 * @see #DIRECTIONALITY_WHITESPACE
//	 * @see #DIRECTIONALITY_OTHER_NEUTRALS
//	 * @see #DIRECTIONALITY_LEFT_TO_RIGHT_EMBEDDING
//	 * @see #DIRECTIONALITY_LEFT_TO_RIGHT_OVERRIDE
//	 * @see #DIRECTIONALITY_RIGHT_TO_LEFT_EMBEDDING
//	 * @see #DIRECTIONALITY_RIGHT_TO_LEFT_OVERRIDE
//	 * @see #DIRECTIONALITY_POP_DIRECTIONAL_FORMAT
//	 * @since 1.4
//	 */
//	public static byte getDirectionality(char ch)
//	{
//		// The result will correctly be signed.
//		return getDirectionality((int) ch);
//
//
//	}
//	/**
//	 * Returns the Unicode directionality property of the character. This
//	 * is used in the visual ordering of text.
//	 *
//	 * @param codePoint the character to look up
//	 * @return the directionality constant, or DIRECTIONALITY_UNDEFINED
//	 * @see #DIRECTIONALITY_UNDEFINED
//	 * @see #DIRECTIONALITY_LEFT_TO_RIGHT
//	 * @see #DIRECTIONALITY_RIGHT_TO_LEFT
//	 * @see #DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC
//	 * @see #DIRECTIONALITY_EUROPEAN_NUMBER
//	 * @see #DIRECTIONALITY_EUROPEAN_NUMBER_SEPARATOR
//	 * @see #DIRECTIONALITY_EUROPEAN_NUMBER_TERMINATOR
//	 * @see #DIRECTIONALITY_ARABIC_NUMBER
//	 * @see #DIRECTIONALITY_COMMON_NUMBER_SEPARATOR
//	 * @see #DIRECTIONALITY_NONSPACING_MARK
//	 * @see #DIRECTIONALITY_BOUNDARY_NEUTRAL
//	 * @see #DIRECTIONALITY_PARAGRAPH_SEPARATOR
//	 * @see #DIRECTIONALITY_SEGMENT_SEPARATOR
//	 * @see #DIRECTIONALITY_WHITESPACE
//	 * @see #DIRECTIONALITY_OTHER_NEUTRALS
//	 * @see #DIRECTIONALITY_LEFT_TO_RIGHT_EMBEDDING
//	 * @see #DIRECTIONALITY_LEFT_TO_RIGHT_OVERRIDE
//	 * @see #DIRECTIONALITY_RIGHT_TO_LEFT_EMBEDDING
//	 * @see #DIRECTIONALITY_RIGHT_TO_LEFT_OVERRIDE
//	 * @see #DIRECTIONALITY_POP_DIRECTIONAL_FORMAT
//	 * @since 1.5
//	 */
//	public static byte getDirectionality(int codePoint)
//	{
//		// If the code point is unassigned or in one of the private use areas
//		// then we delegate the call to the appropriate private static inner class.
//		int plane = codePoint >>> 16;
//
//
//		if(plane > 2 && plane < 14)
//		{
//			return UnassignedCharacters.getDirectionality(codePoint);
//		}
//
//
//		if(plane > 14)
//		{
//			return PrivateUseCharacters.getDirectionality(codePoint);
//		}
//
//		// The result will correctly be signed.
//
//
//		return (byte) (direction[plane][readCodePoint(codePoint) >> 7] >> 2);
//
//
//	}
	/**
	 * Determines whether the character is mirrored according to Unicode. For
	 * example, <code>\u0028</code> (LEFT PARENTHESIS) appears as '(' in
	 * left-to-right text, but ')' in right-to-left text.
	 *
	 * @param ch the character to look up
	 * @return true if the character is mirrored
	 * @since 1.4
	 */
	public static boolean isMirrored(char ch)
	{
		return (readCodePoint((int) ch) & MIRROR_MASK) != 0;


	}

	/**
	 * Determines whether the character is mirrored according to Unicode. For
	 * example, <code>\u0028</code> (LEFT PARENTHESIS) appears as '(' in
	 * left-to-right text, but ')' in right-to-left text.
	 *
	 * @param codePoint the character to look up
	 * @return true if the character is mirrored
	 * @since 1.5
	 */
	public static boolean isMirrored(int codePoint)
	{
		// If the code point is unassigned or part of one of the private use areas
		// then we delegate the call to the appropriate private static inner class.
		int plane = codePoint >>> 16;


		if(plane > 2 && plane < 14)
		{
			return UnassignedCharacters.isMirrored(codePoint);
		}


		if(plane > 14)
		{
			return PrivateUseCharacters.isMirrored(codePoint);
		}



		return (readCodePoint(codePoint) & MIRROR_MASK) != 0;


	}

	/**
	 * Reverse the bytes in val.
	 * @since 1.5
	 */
	public static char reverseBytes(char val)
	{
		return (char) (((val >> 8) & 0xff) | ((val << 8) & 0xff00));


	}

	/**
	 * Converts a unicode code point to a UTF-16 representation of that
	 * code point.
	 *
	 * @param codePoint the unicode code point
	 *
	 * @return the UTF-16 representation of that code point
	 *
	 * @throws IllegalArgumentException if the code point is not a valid
	 *         unicode code point
	 *
	 * @since 1.5
	 */
	public static char[] toChars(int codePoint)
	{
		if(!isValidCodePoint(codePoint))
		{
			throw new IllegalArgumentException("Illegal Unicode code point : "
				+ codePoint);
		}


		char[] result = new char[charCount(codePoint)];


		int ignore = toChars(codePoint, result, 0);


		return result;


	}

	/**
	 * Converts a unicode code point to its UTF-16 representation.
	 *
	 * @param codePoint the unicode code point
	 * @param dst the target char array
	 * @param dstIndex the start index for the target
	 *
	 * @return number of characters written to <code>dst</code>
	 *
	 * @throws IllegalArgumentException if <code>codePoint</code> is not a
	 *         valid unicode code point
	 * @throws NullPointerException if <code>dst</code> is <code>null</code>
	 * @throws IndexOutOfBoundsException if <code>dstIndex</code> is not valid
	 *         in <code>dst</code> or if the UTF-16 representation does not
	 *         fit into <code>dst</code>
	 *
	 * @since 1.5
	 */
	public static int toChars(int codePoint, char[] dst, int dstIndex)
	{
		if(!isValidCodePoint(codePoint))
		{
			throw new IllegalArgumentException("not a valid code point: "
				+ codePoint);


		}

		int result;


		if(isSupplementaryCodePoint(codePoint))
		{
			// Write second char first to cause IndexOutOfBoundsException
			// immediately.
			final int cp2 = codePoint - 0x10000;
			dst[dstIndex + 1] = (char) ((cp2 % 0x400) + (int) MIN_LOW_SURROGATE);
			dst[dstIndex] = (char) ((cp2 / 0x400) + (int) MIN_HIGH_SURROGATE);
			result = 2;


		}
		else
		{
			dst[dstIndex] = (char) codePoint;
			result = 1;


		}
		return result;


	}

	/**
	 * Return number of 16-bit characters required to represent the given
	 * code point.
	 *
	 * @param codePoint a unicode code point
	 *
	 * @return 2 if codePoint >= 0x10000, 1 otherwise.
	 *
	 * @since 1.5
	 */
	public static int charCount(int codePoint)
	{
		return (codePoint >= MIN_SUPPLEMENTARY_CODE_POINT)
			? 2
			: 1;


	}

	/**
	 * Determines whether the specified code point is
	 * in the range 0x10000 .. 0x10FFFF, i.e. the character is within the Unicode
	 * supplementary character range.
	 *
	 * @param codePoint a Unicode code point
	 *
	 * @return <code>true</code> if code point is in supplementary range
	 *
	 * @since 1.5
	 */
	public static boolean isSupplementaryCodePoint(int codePoint)
	{
		return codePoint >= MIN_SUPPLEMENTARY_CODE_POINT
			&& codePoint <= MAX_CODE_POINT;


	}

	/**
	 * Determines whether the specified code point is
	 * in the range 0x0000 .. 0x10FFFF, i.e. it is a valid Unicode code point.
	 *
	 * @param codePoint a Unicode code point
	 *
	 * @return <code>true</code> if code point is valid
	 *
	 * @since 1.5
	 */
	public static boolean isValidCodePoint(int codePoint)
	{
		return codePoint >= MIN_CODE_POINT && codePoint <= MAX_CODE_POINT;


	}

	/**
	 * Return true if the given character is a high surrogate.
	 * @param ch the character
	 * @return true if the character is a high surrogate character
	 *
	 * @since 1.5
	 */
	public static boolean isHighSurrogate(char ch)
	{
		return ch >= MIN_HIGH_SURROGATE && ch <= MAX_HIGH_SURROGATE;


	}

	/**
	 * Return true if the given character is a low surrogate.
	 * @param ch the character
	 * @return true if the character is a low surrogate character
	 *
	 * @since 1.5
	 */
	public static boolean isLowSurrogate(char ch)
	{
		return ch >= MIN_LOW_SURROGATE && ch <= MAX_LOW_SURROGATE;


	}

	/**
	 * Return true if the given characters compose a surrogate pair.
	 * This is true if the first character is a high surrogate and the
	 * second character is a low surrogate.
	 * @param ch1 the first character
	 * @param ch2 the first character
	 * @return true if the characters compose a surrogate pair
	 *
	 * @since 1.5
	 */
	public static boolean isSurrogatePair(char ch1, char ch2)
	{
		return isHighSurrogate(ch1) && isLowSurrogate(ch2);


	}

	/**
	 * Given a valid surrogate pair, this returns the corresponding
	 * code point.
	 * @param high the high character of the pair
	 * @param low the low character of the pair
	 * @return the corresponding code point
	 *
	 * @since 1.5
	 */
	public static int toCodePoint(char high, char low)
	{
		return ((high - MIN_HIGH_SURROGATE) * 0x400)
			+ (low - MIN_LOW_SURROGATE) + 0x10000;


	}

	/**
	 * Get the code point at the specified index in the CharSequence.
	 * If the character is the start of a surrogate pair, and there is a
	 * following character, and this character completes the pair, then
	 * the corresponding supplementary code point is returned.
	 * Otherwise, the character at the index is returned.
	 *
	 * @param chars the character array in which to look
	 * @param index the index of the codepoint to get, starting at 0
	 * @return the codepoint at the specified index
	 * @throws IndexOutOfBoundsException if index is negative or &gt;= length()
	 * @since 1.5
	 */
	public static int codePointAt(char[] chars, int index)
	{
		return codePointAt(chars, index, chars.length);


	}

	/**
	 * Get the code point at the specified index in the CharSequence.
	 * If the character is the start of a surrogate pair, and there is a
	 * following character within the specified range, and this
	 * character completes the pair, then the corresponding
	 * supplementary code point is returned.  Otherwise, the character
	 * at the index is returned.
	 *
	 * @param chars the character array in which to look
	 * @param index the index of the codepoint to get, starting at 0
	 * @param limit the limit past which characters should not be examined
	 * @return the codepoint at the specified index
	 * @throws IndexOutOfBoundsException if index is negative or &gt;=
	 * limit, or if limit is negative or &gt;= the length of the array
	 * @since 1.5
	 */
	public static int codePointAt(char[] chars, int index, int limit)
	{
		if(index < 0 || index >= limit || limit < 0 || limit > chars.length)
		{
			throw new IndexOutOfBoundsException();
		}


		char high = chars[index];


		if(!isHighSurrogate(high) || ++index >= limit)
		{
			return high;
		}


		char low = chars[index];


		if(!isLowSurrogate(low))
		{
			return high;
		}


		return toCodePoint(high, low);


	}

	/**
	 * Get the code point before the specified index.  This is like
	 * #codePointAt(char[], int), but checks the characters at
	 * <code>index-1</code> and <code>index-2</code> to see if they form
	 * a supplementary code point.  If they do not, the character at
	 * <code>index-1</code> is returned.
	 *
	 * @param chars the character array
	 * @param index the index just past the codepoint to get, starting at 0
	 * @return the codepoint at the specified index
	 * @throws IndexOutOfBoundsException if index is negative or &gt;= length()
	 * @since 1.5
	 */
	public static int codePointBefore(char[] chars, int index)
	{
		return codePointBefore(chars, index, 1);


	}

	/**
	 * Get the code point before the specified index.  This is like
	 * #codePointAt(char[], int), but checks the characters at
	 * <code>index-1</code> and <code>index-2</code> to see if they form
	 * a supplementary code point.  If they do not, the character at
	 * <code>index-1</code> is returned.  The start parameter is used to
	 * limit the range of the array which may be examined.
	 *
	 * @param chars the character array
	 * @param index the index just past the codepoint to get, starting at 0
	 * @param start the index before which characters should not be examined
	 * @return the codepoint at the specified index
	 * @throws IndexOutOfBoundsException if index is &gt; start or &gt;
	 * the length of the array, or if limit is negative or &gt;= the
	 * length of the array
	 * @since 1.5
	 */
	public static int codePointBefore(char[] chars, int index, int start)
	{
		if(index < start || index > chars.length
			|| start < 0 || start >= chars.length)
		{
			throw new IndexOutOfBoundsException();
		}


		--index;


		char low = chars[index];


		if(!isLowSurrogate(low) || --index < start)
		{
			return low;
		}


		char high = chars[index];


		if(!isHighSurrogate(high))
		{
			return low;
		}


		return toCodePoint(high, low);


	}
} // class UnicodeCharacter

