/* java.util.zip.ZipEntry
   Copyright (C) 2001, 2002 Free Software Foundation, Inc.
 
This file is part of GNU Classpath.
 
GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.
 
GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.
 
Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.
 
As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

package com.classpath.zip;

import com.one.ArchiveEntry;
import com.vmx.AuxClass;
import com.vmx.StringEncoder;
import java.util.Calendar;
import java.util.Date;

import java.io.*;

/**
 * This class represents a member of a zip archive.  ZipFile and
 * ZipInputStream will give you instances of this class as information
 * about the members in an archive.  On the other hand ZipOutputStream
 * needs an instance of this class to create a new member.
 *
 * @author Jochen Hoenicke
 */
public class ZipEntry extends ArchiveEntry
{
	private static int KNOWN_SIZE = 1;
	private static int KNOWN_CSIZE = 2;
	private static int KNOWN_CRC = 4;
	private static int KNOWN_TIME = 8;

	private static Calendar cal = Calendar.getInstance();

	private int crc;
	private int dostime;
	private short known = 0;
	private short method = -1;
	private byte[] extra = null;
	private String comment = null;
	
	public int flags;              /* used by ZipOutputStream */

	/**
	 * Creates an empty zip entry.
	 */
	public ZipEntry()
	{
	}
	
	/**
	 * Creates a zip entry with the given name.
	 * @param name the name. May include directory components separated by '/'.
	 *
	 * @exception NullPointerException when name is null.
	 * @exception IllegalArgumentException when name is bigger then 65535 chars.
	 */
	public ZipEntry(String name)
	{
		setName(name);
	}

	/**
	 * Creates a copy of the given zip entry.
	 * @param e the entry to copy.
	 */
	public ZipEntry(ZipEntry e)
	{
		name = e.name;
		known = e.known;
		size = e.size;
		compressedSize = e.compressedSize;
		crc = e.crc;
		dostime = e.dostime;
		method = e.method;
		extra = e.extra;
		comment = e.comment;
	}
	
	public int getLocalSize()
	{
		if((known & KNOWN_CSIZE) == 0)
		{
			return -1;
		}
		
		int localSize = ZipConstants.LOCHDR + compressedSize;
		
		if(name != null)
		{
			localSize += StringEncoder.getEncodedLength(name, StringEncoder.ENC_ARCHIVE);
		}
		
		if(extra != null)
		{
			localSize += extra.length;
		}
		
		if((flags & 8) != 0)
		{
			localSize += ZipConstants.EXTHDR;
		}
		
		return localSize;
	}
	
	/**
	 * Read this entry from central file header.
	 * Return true if successful, false if central header not found.
	 */
	public boolean readCentralHeader(InputStream is) throws IOException
	{
		if(AuxClass.readLeInt(is) != ZipConstants.CENSIG)
		{
			return false;
		}
		
		is.skip(4);
		
		flags = AuxClass.readLeShort(is);
		
		setMethod(AuxClass.readLeShort(is));
		setDOSTime(AuxClass.readLeInt(is));
		setCRC(AuxClass.readLeInt(is));
		setCompressedSize(AuxClass.readLeInt(is));
		setSize(AuxClass.readLeInt(is));
		
		int nameLen = AuxClass.readLeShort(is);
		int extraLen = AuxClass.readLeShort(is);
		int commentLen = AuxClass.readLeShort(is);
		
		is.skip(8);
		
		offset = AuxClass.readLeInt(is);
		
		if(nameLen > 0)
		{
			setName(StringEncoder.readString(is, nameLen, StringEncoder.ENC_ARCHIVE));
		}
		
		if(extraLen > 0)
		{
			byte[] b = new byte[extraLen];
			is.read(b, 0, extraLen);
			
			setExtra(b);
		}
		
		if(commentLen > 0)
		{
			setComment(StringEncoder.readString(is, commentLen, StringEncoder.ENC_ARCHIVE));
		}
		
		return true;
	}
	
	/**
	 * Write central file header for this entry,
	 * return number of bytes written.
	 */
	public int writeCentralHeader(OutputStream os) throws IOException
	{
		AuxClass.writeLeInt(os, ZipConstants.CENSIG);
		
		int method = getMethod();
		
		AuxClass.writeLeShort(os, method == ZipFile.STORED ? ZipFile.ZIP_STORED_VERSION : ZipFile.ZIP_DEFLATED_VERSION);
		AuxClass.writeLeShort(os, method == ZipFile.STORED ? ZipFile.ZIP_STORED_VERSION : ZipFile.ZIP_DEFLATED_VERSION);
		AuxClass.writeLeShort(os, flags);
		AuxClass.writeLeShort(os, method);
		AuxClass.writeLeInt(os, getDOSTime());
		AuxClass.writeLeInt(os, getCRC());
		AuxClass.writeLeInt(os, getCompressedSize());
		AuxClass.writeLeInt(os, getSize());

		byte[] name = StringEncoder.encodeString(getName(), StringEncoder.ENC_ARCHIVE);
		
		if(name.length > 0xFFFF)
		{
			throw new ZipException("Name too long.");
		}
		
		byte[] extra = getExtra();
		
		if(extra == null)
		{
			extra = new byte[0];
		}
		
		String strComment = getComment();
		
		byte[] comment = strComment != null ? StringEncoder.encodeString(strComment, StringEncoder.ENC_ARCHIVE) : new byte[0];
		
		if(comment.length > 0xFFFF)
		{
			throw new ZipException("Comment too long.");
		}

		AuxClass.writeLeShort(os, name.length);
		AuxClass.writeLeShort(os, extra.length);
		AuxClass.writeLeShort(os, comment.length);
		AuxClass.writeLeShort(os, 0); /* disk number */
		AuxClass.writeLeShort(os, 0); /* internal file attr */
		AuxClass.writeLeInt(os, 0);   /* external file attr */
		AuxClass.writeLeInt(os, offset);

		if(name.length > 0)
		{
			os.write(name);
		}

		if(extra.length > 0)
		{
			os.write(extra);
		}

		if(comment.length > 0)
		{
			os.write(comment);
		}
		
		return ZipConstants.CENHDR + name.length + extra.length + comment.length;
	}
	
	/**
	 * Read this entry from local file header.
	 * Return true if successful, false if local header not found.
	 */
	public boolean readLocalHeader(InputStream is) throws IOException
	{
		if(AuxClass.readLeInt(is) != ZipConstants.LOCSIG)
		{
			return false;
		}
		
		is.skip(2);
		
		flags = AuxClass.readLeShort(is);
		
		setMethod(AuxClass.readLeShort(is));
		setDOSTime(AuxClass.readLeInt(is));
		setCRC(AuxClass.readLeInt(is));
		setCompressedSize(AuxClass.readLeInt(is));
		setSize(AuxClass.readLeInt(is));
		
		int nameLen = AuxClass.readLeShort(is);
		int extraLen = AuxClass.readLeShort(is);
		
		if(nameLen > 0)
		{
			setName(StringEncoder.readString(is, nameLen, StringEncoder.ENC_ARCHIVE));
		}
		
		if(extraLen > 0)
		{
			byte[] b = new byte[extraLen];
			is.read(b, 0, extraLen);
			
			setExtra(b);
		}
		
		return true;
	}
	
	/**
	 * Write local file header for this entry,
	 * return number of bytes written.
	 */
	public int writeLocalHeader(OutputStream os) throws IOException
	{
		AuxClass.writeLeInt(os, ZipConstants.LOCSIG);
		
		int method = getMethod();
		
		AuxClass.writeLeShort(os, method == ZipFile.STORED ? ZipFile.ZIP_STORED_VERSION : ZipFile.ZIP_DEFLATED_VERSION);
		AuxClass.writeLeShort(os, flags);
		AuxClass.writeLeShort(os, method);
		AuxClass.writeLeInt(os, getDOSTime());
		
		if((flags & 8) == 0)
		{
			AuxClass.writeLeInt(os, getCRC());
			AuxClass.writeLeInt(os, getCompressedSize());
			AuxClass.writeLeInt(os, getSize());
		}
		else
		{
			AuxClass.writeLeInt(os, 0);
			AuxClass.writeLeInt(os, 0);
			AuxClass.writeLeInt(os, 0);
		}
		
		byte[] name = StringEncoder.encodeString(getName(), StringEncoder.ENC_ARCHIVE);
		
		if(name.length > 0xFFFF)
		{
			throw new ZipException("Name too long.");
		}
		
		byte[] extra = getExtra();
		
		if(extra == null)
		{
			extra = new byte[0];
		}
		
		AuxClass.writeLeShort(os, name.length);
		AuxClass.writeLeShort(os, extra.length);

		if(name.length > 0)
		{
			os.write(name);
		}

		if(extra.length > 0)
		{
			os.write(extra);
		}
		
		return ZipConstants.LOCHDR + name.length + extra.length;
	}
	
	public void setDOSTime(int dostime)
	{
		this.dostime = dostime;
		known |= KNOWN_TIME;
	}

	public int getDOSTime()
	{
		if((known & KNOWN_TIME) == 0)
		{
			return 0;
		}
		else
		{
			return dostime;
		}
	}
	
	public void setName(String name)
	{
		if(name.length() > 65535)
		{
			throw new IllegalArgumentException("Name too long");
		}
		
		this.name = name;
	}

	/**
	 * Returns the entry name.  The path components in the entry are
	 * always separated by slashes ('/').
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * Sets the time of last modification of the entry.
	 * @time the time of last modification of the entry.
	 */
	public void setLastModified(long time)
	{
		synchronized(cal)
		{
			cal.setTime(new Date(time * 1000L));
			
			dostime = (cal.get(Calendar.YEAR) - 1980 & 0x7f) << 25
					| (cal.get(Calendar.MONTH) + 1) << 21
					| (cal.get(Calendar.DAY_OF_MONTH)) << 16
					| (cal.get(Calendar.HOUR_OF_DAY)) << 11
					| (cal.get(Calendar.MINUTE)) << 5
					| (cal.get(Calendar.SECOND)) >> 1;
		}
		
		dostime = (int)(dostime / 1000L);
		this.known |= KNOWN_TIME;
	}

	/**
	 * Gets the time of last modification of the entry.
	 * @return the time of last modification of the entry, or -1 if unknown.
	 */
	public long getLastModified()
	{
		if((known & KNOWN_TIME) == 0)
		{
			return -1;
		}

		int sec = 2 * (dostime & 0x1f);
		int min = (dostime >> 5) & 0x3f;
		int hrs = (dostime >> 11) & 0x1f;
		int day = (dostime >> 16) & 0x1f;
		int mon = ((dostime >> 21) & 0xf) - 1;
		int year = ((dostime >> 25) & 0x7f) + 1980; /* since 1900 */

		try
		{
			synchronized(cal)
			{
				cal.set(cal.YEAR, year);
				cal.set(cal.MONTH, mon);
				cal.set(cal.DAY_OF_MONTH, day);
				cal.set(cal.HOUR_OF_DAY, hrs);
				cal.set(cal.MINUTE, min);
				cal.set(cal.SECOND, sec);
				
				return cal.getTime().getTime();
			}
		}
		catch(RuntimeException ex)
		{
			/* Ignore illegal time stamp */
			known &= ~KNOWN_TIME;
			return -1;
		}
	}

	/**
	 * Sets the size of the uncompressed data.
	 * @exception IllegalArgumentException if size is not in 0..0xffffffffL
	 */
	public void setSize(int size)
	{
		this.size = size;
		this.known |= KNOWN_SIZE;
	}

	/**
	 * Gets the size of the uncompressed data.
	 * @return the size or -1 if unknown.
	 */
	public int getSize()
	{
		return (known & KNOWN_SIZE) != 0 ? size : -1;
	}

	/**
	 * Sets the size of the compressed data.
	 * @exception IllegalArgumentException if size is not in 0..0xffffffffL
	 */
	public void setCompressedSize(int csize)
	{
		this.compressedSize = csize;
		this.known |= KNOWN_CSIZE;
	}

	/**
	 * Gets the size of the compressed data.
	 * @return the size or -1 if unknown.
	 */
	public int getCompressedSize()
	{
		return (known & KNOWN_CSIZE) != 0 ? compressedSize : -1;
	}

	/**
	 * Sets the crc of the uncompressed data.
	 * @exception IllegalArgumentException if crc is not in 0..0xffffffffL
	 */
	public void setCRC(int crc)
	{
		this.crc = crc;
		this.known |= KNOWN_CRC;
	}

	/**
	 * Gets the crc of the uncompressed data.
	 * @return the crc or -1 if unknown.
	 */
	public int getCRC()
	{
		return (known & KNOWN_CRC) != 0 ? crc : -1;
	}

	/**
	 * Sets the compression method.  Only DEFLATED and STORED are
	 * supported.
	 * @exception IllegalArgumentException if method is not supported.
	 * @see ZipOutputStream#DEFLATED
	 * @see ZipOutputStream#STORED
	 */
	public void setMethod(int method)
	{
		if(method != ZipFile.STORED && method != ZipFile.DEFLATED)
		{
			throw new IllegalArgumentException("Method not STORED or DEFLATED");
		}
		
		this.method = (short)method;
	}

	/**
	 * Gets the compression method.
	 * @return the compression method or -1 if unknown.
	 */
	public int getMethod()
	{
		return method;
	}

	/**
	 * Sets the extra data.
	 * @exception IllegalArgumentException if extra is longer than 0xffff bytes.
	 */
	public void setExtra(byte[] extra)
	{
		if(extra == null)
		{
			this.extra = null;
			return;
		}

		if(extra.length > 0xffff)
		{
			throw new IllegalArgumentException("Extra too long");
		}
		
		this.extra = extra;
		
		try
		{
			int pos = 0;
			
			while(pos < extra.length)
			{
				int sig = (extra[pos++] & 0xff) | (extra[pos++] & 0xff) << 8;
				int len = (extra[pos++] & 0xff) | (extra[pos++] & 0xff) << 8;
				
				if(sig == 0x5455)
				{
					/* extended time stamp */
					int flags = extra[pos];
					
					if((flags & 1) != 0)
					{
						long time = ((extra[pos + 1] & 0xff) |
						             (extra[pos + 2] & 0xff) << 8 |
						             (extra[pos + 3] & 0xff) << 16 |
						             (extra[pos + 4] & 0xff) << 24);
						
						setLastModified(time);
					}
				}
				
				pos += len;
			}
		}
		catch(ArrayIndexOutOfBoundsException ex)
		{
			/* be lenient */
			return;
		}
	}

	/**
	 * Gets the extra data.
	 * @return the extra data or null if not set.
	 */
	public byte[] getExtra()
	{
		return extra;
	}

	/**
	 * Sets the entry comment.
	 * @exception IllegalArgumentException if comment is longer than 0xffff.
	 */
	public void setComment(String comment)
	{
		if(comment != null && comment.length() > 0xffff)
		{
			throw new IllegalArgumentException("Comment too long");
		}
		
		this.comment = comment;
	}

	/**
	 * Gets the comment.
	 * @return the comment or null if not set.
	 */
	public String getComment()
	{
		return comment;
	}

	/**
	 * Gets true, if the entry is a directory.  This is solely
	 * determined by the name, a trailing slash '/' marks a directory.
	 */
	public boolean isDirectory()
	{
		int nlen = name.length();
		return nlen > 0 && name.charAt(nlen - 1) == '/';
	}

	/**
	 * Gets the string representation of this ZipEntry.  This is just
	 * the name as returned by getName().
	 */
	public String toString()
	{
		return name;
	}

	/**
	 * Gets the hashCode of this ZipEntry.  This is just the hashCode
	 * of the name.  Note that the equals method isn't changed, though.
	 */
	public int hashCode()
	{
		return name.hashCode();
	}
}
