package com.one.vector;

import javax.microedition.lcdui.*;
import javax.microedition.io.*;
import java.io.*;
import java.lang.Math;

public class TexturePolygon extends VectorElement
{
	protected double[][] data;
	protected int[][] sdata;
	protected ArrayData pdata;
	protected int count;
	
	protected double[] bounds = new double[4];
	protected int[] sbounds = new int[4];
	
	protected String filename;
	protected int[] texdata;
	protected int texwidth, texheight;
	
	protected int[] imagedata;
	protected int scanlength;
	
	protected boolean ready = false;
	
	public TexturePolygon()
	{
		pdata = new ArrayData();
		pdata.type = ArrayData.HARMONIC;
		
		count = 3;
	}
	
	public void read(DataInputStream dis) throws IOException
	{
		name = dis.readUTF();
		color = dis.readInt();
		parseFlag(dis.readUnsignedByte());
		
		pdata = new ArrayData(dis);
		count = dis.readUnsignedShort();
		
		System.gc();
		
		try
		{
			if(dis.readBoolean())
			{
				int blen = dis.readInt();
				byte[] b = new byte[blen];
				
				dis.readFully(b);
				Image texture = Image.createImage(b, 0, blen);
				b = null;
				
				texwidth = texture.getWidth();
				texheight = texture.getHeight();
				
				System.gc();
				texdata = new int[texwidth * texheight];
				texture.getRGB(texdata, 0, texwidth, 0, 0, texwidth, texheight);
			}
			else
			{
				texwidth = dis.readInt();
				texheight = dis.readInt();
				
				texdata = new int[texwidth * texheight];
				
				for(int i = 0; i < texdata.length; i++)
				{
					texdata[i] = dis.readInt();
				}
			}
		}
		catch(OutOfMemoryError oome)
		{
			texwidth = Bitmap.DEFTEX_WIDTH;
			texheight = Bitmap.DEFTEX_HEIGHT;
			
			texdata = new int[texwidth * texheight];
			
			for(int i = 0; i < texdata.length; i++)
			{
				texdata[i] = color;
			}
		}
		
		readTransformList(dis);
	}
	
	public void write(DataOutputStream dos) throws IOException
	{
		dos.writeUTF(name);
		dos.writeInt(color);
		dos.writeByte(getFlag());
		
		pdata.write(dos);
		dos.writeShort(count);
		
		try
		{
			InputStream is = Connector.openInputStream("file:///" + filename);
			
			byte[] b = new byte[4096];
			
			dos.writeBoolean(true);
			dos.writeInt(is.available());
			
			while(is.available() > 0)
			{
				dos.write(b, 0, is.read(b));
			}
			
			is.close();
		}
		catch(Exception e)
		{
			dos.writeBoolean(false);
			dos.writeInt(texwidth);
			dos.writeInt(texheight);
			
			for(int i = 0; i < texdata.length; i++)
			{
				dos.writeInt(texdata[i]);
			}
		}
		
		writeTransformList(dos);
	}
	
	public void create()
	{
		data = AuxMath.plotPolygon(pdata.vala, pdata.valb, pdata.diva, pdata.divb, count, pdata.inta, pdata.intb);
		sdata = new int[data.length][2];
		
		double[] r = AuxMath.plotRotRect(pdata.vala, pdata.valb, pdata.diva, pdata.divb, pdata.intb);
		
		bounds[0] = Math.min(r[0], Math.min(r[2], Math.min(r[4], r[6])));
		bounds[1] = Math.min(r[1], Math.min(r[3], Math.min(r[5], r[7])));
		bounds[2] = Math.max(r[0], Math.max(r[2], Math.max(r[4], r[6]))) - bounds[0];
		bounds[3] = Math.max(r[1], Math.max(r[3], Math.max(r[5], r[7]))) - bounds[1];
		
		if(texdata == null)
		{
			try
			{
				InputStream is = Connector.openInputStream("file:///" + filename);
				Image texture = Image.createImage(is);
				is.close();
				
				System.gc();
				
				texwidth = texture.getWidth();
				texheight = texture.getHeight();
				
				texdata = new int[texwidth * texheight];
				texture.getRGB(texdata, 0, texwidth, 0, 0, texwidth, texheight);
			}
			catch(Throwable t)
			{
				texwidth = Bitmap.DEFTEX_WIDTH;
				texheight = Bitmap.DEFTEX_HEIGHT;
				
				texdata = new int[texwidth * texheight];
				
				for(int i = 0; i < texdata.length; i++)
				{
					texdata[i] = color;
				}
			}
		}
	}
	
	public void rescale()
	{
		sbounds[0] = getScaled(bounds[0], 0);
		sbounds[1] = getScaled(bounds[1], 1);
		sbounds[2] = getScaled(bounds[2], 2);
		sbounds[3] = getScaled(bounds[3], 3);
		
		for(int i = 0; i < data.length; i++)
		{
			sdata[i][0] = getScaled(data[i][0] - bounds[0], 0);
			sdata[i][1] = getScaled(data[i][1] - bounds[1], 1);
		}
		
		ready = false;
		
		try
		{
			int repx = (sbounds[2] + texwidth - 1) / texwidth;
			int repy = (sbounds[3] + texheight - 1) / texheight;
			
			scanlength = texwidth * repx;
			
			System.gc();
			imagedata = new int[scanlength * texheight * repy];
			
			int i, j, k;
			
			for(i = 0; i < repx; i++)
			{
				for(j = 0; j < repy; j++)
				{
					for(k = 0; k < texheight; k++)
					{
						System.arraycopy(texdata, texwidth * k, imagedata, texwidth * i + (texheight * j + k) * scanlength, texwidth);
					}
				}
			}
			
			Image temp = Image.createImage(sbounds[2], sbounds[3]);
			Graphics g = temp.getGraphics();
			
			g.setColor(0);
			g.fillRect(0, 0, sbounds[2], sbounds[3]);
			g.setColor(0xFFFFFFFF);
			
			for(i = 2; i < data.length; i++)
			{
				g.fillTriangle(sdata[0][0], sdata[0][1], sdata[i][0], sdata[i][1], sdata[i - 1][0], sdata[i - 1][1]);
			}
			
			System.gc();
			int[] shapedata = new int[imagedata.length];
			temp.getRGB(shapedata, 0, scanlength, 0, 0, sbounds[2], sbounds[3]);
			
			for(i = 0; i < shapedata.length; i++)
			{
				if((shapedata[i] & 0xFFFFFF) == 0)
				{
					imagedata[i] = 0;
				}
			}
			
			ready = true;
		}
		catch(Throwable t)
		{
			for(int i = 0; i < data.length; i++)
			{
				sdata[i][0] = getScaled(data[i][0], 0);
				sdata[i][1] = getScaled(data[i][1], 1);
			}
		}
		
		System.gc();
	}
	
	public void paint(Graphics g)
	{
		if(hidden)
		{
			return;
		}
		
		if(ready && group.image.usecolor)
		{
			g.drawRGB(imagedata, 0, scanlength, sbounds[0], sbounds[1], sbounds[2], sbounds[3], true);
		}
		else
		{
			if(group.image.usecolor)
			{
				g.setColor(color);
			}
			
			for(int i = 2; i < data.length; i++)
			{
				g.fillTriangle(sdata[0][0], sdata[0][1], sdata[i][0], sdata[i][1], sdata[i - 1][0], sdata[i - 1][1]);
			}
		}
	}
	
	public void update(boolean flag, boolean changed)
	{
		Transform transform;
		int param;
		
		for(int i = 0; i < transcount; i++)
		{
			transform = (Transform)transforms.elementAt(i);
			transform.setFrame(group.image.frame);
			
			if(transform.isApplied())
			{
				param = transform.getParam();
				
				switch(param)
				{
					case SHOW:
						setHidden(AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition()));
						break;
					
					case COLOR:
						color = AuxMath.linearFunctionRGB(transform.getData(), transform.getPosition());
						break;
					
					case X:
						pdata.vala = AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition());
						changed = true;
						break;
					
					case Y:
						pdata.valb = AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition());
						changed = true;
						break;
					
					case WIDTH:
						pdata.diva = AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition());
						changed = true;
						break;
					
					case HEIGHT:
						pdata.divb = AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition());
						changed = true;
						break;
					
					case ROTATION:
						pdata.inta = AuxMath.specificFunctionInt(transform.getData(), transform.getPosition());
						changed = true;
						break;
					
					case ANGLE:
						pdata.intb = AuxMath.specificFunctionInt(transform.getData(), transform.getPosition());
						changed = true;
						break;
				}
			}
		}
		
		if(flag && changed)
		{
			create();
			rescale();
		}
	}
	
	public Object getValue(int param, int subparam)
	{
		switch(param)
		{
			case X:
				return new Double(pdata.vala);
			
			case Y:
				return new Double(pdata.valb);
			
			case WIDTH:
				return new Double(pdata.diva);
			
			case HEIGHT:
				return new Double(pdata.divb);
			
			case ROTATION:
				return new Integer(pdata.inta);
			
			case ANGLE:
				return new Integer(pdata.intb);
			
			case POLY_COUNT:
				return new Integer(count);
			
			case FILE:
				return filename;
		}
		
		return null;
	}
	
	public void setValue(Object value, int param, int subparam)
	{
		switch(param)
		{
			case X:
				pdata.vala = ((Double)value).doubleValue();
				break;
			
			case Y:
				pdata.valb = ((Double)value).doubleValue();
				break;
			
			case WIDTH:
				pdata.diva = ((Double)value).doubleValue();
				break;
			
			case HEIGHT:
				pdata.divb = ((Double)value).doubleValue();
				break;
			
			case ROTATION:
				pdata.inta = ((Integer)value).intValue();
				break;
			
			case ANGLE:
				pdata.intb = ((Integer)value).intValue();
				break;
			
			case POLY_COUNT:
				count = ((Integer)value).intValue();
				break;
			
			case FILE:
				filename = (String)value;
				texdata = null;
				break;
		}
	}
}