package com.one.vector;

import javax.microedition.lcdui.Graphics;
import java.io.*;

public class ThickBezierCurve extends VectorElement
{
	protected ThickLine[] lines;
	protected double[][] points;
	protected double thickness;
	protected int count;
	
	public ThickBezierCurve()
	{
		points = new double[3][2];
		points[0][0] = 0;
		points[0][1] = 0;
		points[1][0] = 0;
		points[1][1] = 0;
		points[2][0] = 0;
		points[2][1] = 0;
		
		count = 3;
		
		thickness = 0;
	}
	
	public void read(DataInputStream dis) throws IOException
	{
		name = dis.readUTF();
		color = dis.readInt();
		parseFlag(dis.readUnsignedByte());
		
		count = dis.readUnsignedShort() + 1;
		
		points = new double[dis.readUnsignedShort()][2];
		
		for(int i = 0; i < points.length; i++)
		{
			points[i][0] = dis.readFloat();
			points[i][1] = dis.readFloat();
		}
		
		thickness = dis.readFloat();
		
		readTransformList(dis);
	}
	
	public void write(DataOutputStream dos) throws IOException
	{
		dos.writeUTF(name);
		dos.writeInt(color);
		dos.writeByte(getFlag());
		
		dos.writeShort(count - 1);
		
		dos.writeShort(points.length);
		
		for(int i = 0; i < points.length; i++)
		{
			dos.writeFloat((float)points[i][0]);
			dos.writeFloat((float)points[i][1]);
		}
		
		dos.writeFloat((float)thickness);
		
		writeTransformList(dos);
	}
	
	public void create()
	{
		double[][] data = AuxMath.plotBezierCurve(points, count);
		
		lines = new ThickLine[count - 1];
		
		for(int i = 0; i < lines.length; i++)
		{
			lines[i] = new ThickLine(color, NORMAL, data[i][0], data[i][1], data[i + 1][0], data[i + 1][1], thickness);
			lines[i].setGroup(group);
		}
	}
	
	public void rescale()
	{
		for(int i = 0; i < lines.length; i++)
		{
			lines[i].rescale();
		}
	}
	
	public void setGroup(Group grp)
	{
		super.setGroup(grp);
		
		for(int i = 0; i < lines.length; i++)
		{
			lines[i].setGroup(grp);
		}
	}
	
	public void paint(Graphics g)
	{
		if(hidden)
		{
			return;
		}
		
		for(int i = 0; i < lines.length; i++)
		{
			lines[i].paint(g);
		}
	}
	
	public void update(boolean flag, boolean changed)
	{
		Transform transform;
		int param, subparam;
		
		for(int i = 0; i < transcount; i++)
		{
			transform = (Transform)transforms.elementAt(i);
			transform.setFrame(group.image.frame);
			
			if(transform.isApplied())
			{
				param = transform.getParam();
				
				if(param == SHOW)
				{
					setHidden(AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition()));
				}
				else if(param == COLOR)
				{
					color = AuxMath.linearFunctionRGB(transform.getData(), transform.getPosition());
					changed = true;
				}
				else if(param == THICKNESS)
				{
					thickness = AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition());
					changed = true;
				}
				else
				{
					subparam = transform.getSubParam() - 1;
					
					if(subparam >= 0 && subparam < points.length)
					{
						switch(param)
						{
							case X:
								points[subparam][0] = AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition());
								changed = true;
								break;
							
							case Y:
								points[subparam][1] = AuxMath.specificFunctionFloat(transform.getData(), transform.getPosition());
								changed = true;
								break;
						}
					}
				}
			}
		}
		
		if(flag && changed)
		{
			create();
			rescale();
		}
	}
	
	public Object getValue(int param, int subparam)
	{
		if(param == COUNT)
		{
			return new Integer(points.length);
		}
		else if(param == QUALITY)
		{
			return new Integer(count - 1);
		}
		else if(param == X)
		{
			subparam--;
			
			if(subparam >= 0 && subparam < points.length)
			{
				return new Double(points[subparam][0]);
			}
		}
		else if(param == Y)
		{
			subparam--;
			
			if(subparam >= 0 && subparam < points.length)
			{
				return new Double(points[subparam][1]);
			}
		}
		else if(param == THICKNESS)
		{
			return new Double(thickness);
		}
		
		return null;
	}
	
	public void setValue(Object value, int param, int subparam)
	{
		if(param == COUNT)
		{
			double[][] p = new double[((Integer)value).intValue()][2];
			int minlength = Math.min(points.length, p.length);
			
			for(int i = 0; i < minlength; i++)
			{
				p[i][0] = points[i][0];
				p[i][1] = points[i][1];
			}
			
			points = p;
		}
		else if(param == QUALITY)
		{
			count = ((Integer)value).intValue() + 1;
		}
		else if(param == X)
		{
			subparam--;
			
			if(subparam >= 0 && subparam < points.length)
			{
				points[subparam][0] = ((Double)value).doubleValue();
			}
		}
		else if(param == Y)
		{
			subparam--;
			
			if(subparam >= 0 && subparam < points.length)
			{
				points[subparam][1] = ((Double)value).doubleValue();
			}
		}
		else if(param == THICKNESS)
		{
			thickness = ((Double)value).doubleValue();
		}
	}
	
	public void writeSVG(PrintStream ps)
	{
		double[][] data = AuxMath.plotBezierCurve(points, count);
		
		ps.print("\t\t\t<polyline");
		
		ps.print(" points=\"");
		ps.print(Integer.toString(getScaled(data[0][0], 0)) + " " + Integer.toString(getScaled(data[0][1], 1)));
		
		for(int i = 1; i < data.length; i++)
		{
			ps.print(", " + Integer.toString(getScaled(data[i][0], 0)) + " " + Integer.toString(getScaled(data[i][1], 1)));
		}
		
		ps.print("\"");
		
		ps.print(" stroke-width=\"" + Integer.toString(getScaled(thickness, 4)) + "\"");
		ps.print(" stroke-linecap=\"round\"");
		ps.print(" stroke-linejoin=\"round\"");
		
		writeSVGTail(ps, true, false);
		
		ps.println("/>");
	}
}