package modules.audio;

import com.one.ExtGraphics;
import com.one.SlidingNumber;
import com.one.VolumeUnit;
import com.one.vector.AuxMath;
import filemanager.images;
import java.util.Random;

/**
 * Стрелочный индикатор
 */
class AnalogIndicator
{
	protected VolumeUnit vu;

	protected int width, height;
	protected int orgx, orgy, radius, baseangle;

	public AnalogIndicator(int width, int height, int viswidth, int visheight, int radius)
	{
		this.width = width;
		this.height = height;

		orgx = width / 2;
		orgy = height - visheight + radius;

		this.radius = radius;

		double la = Math.toDegrees(AuxMath.atan2(radius, -viswidth / 2));
		double ra = Math.toDegrees(AuxMath.atan2(radius, viswidth / 2));

		baseangle = (int)la;
		vu = new VolumeUnit(0, baseangle - (int)ra, 8, 35, 2, 3);
	}

	public void setTarget(int value)
	{
		vu.setTarget(value);
	}

    public void setValue(int value)
    {
        vu.setValue(value);
    }

	public void paint(ExtGraphics g, int x, int y, boolean update)
	{
		g.translate(x, y);
		g.setClip(0, 0, width, height);

		double angle = Math.toRadians((double)(baseangle - vu.nextValue(update)));

        double angsin = Math.sin(angle);
        double angcos = Math.cos(angle);

		int x1 = orgx + (int)(radius * 88 / 100 * angcos);
        int y1 = orgy - (int)(radius * 88 / 100 * angsin);

        int x2 = orgx + (int)(radius * angcos);
        int y2 = orgy - (int)(radius * angsin);

        g.setColor(images.cVisArrowLine);
		g.drawLine(orgx, orgy, x1, y1);

		g.setColor(images.cVisArrowCap);

        if(x2 < x1) // стрелка наклонена влево
        {
            g.setClip(x2, y2, x1 - x2 + 1, y1 - y2 + 1);
        }
        else if(x2 > x1) // стрелка наклонена вправо
        {
            g.setClip(x1, y2, x2 - x1 + 1, y1 - y2 + 1);
        }
        else // стрелка стоит вертикально
        {
            g.setClip(x2, y2, 1, y1 - y2 + 1);
        }

		g.drawLine(orgx, orgy, x2, y2);

		g.translate(-x, -y);
	}
}

/**
 * Визуалка aNNiMON'а
 */
class VisualEffect
{
	protected static Random rnd = new Random();

	protected SlidingNumber slidingInnerR, slidingInnerG, slidingInnerB;
	protected SlidingNumber slidingOuterR, slidingOuterG, slidingOuterB;

	protected int colorVariant;
	protected int chains, links;
	protected float[] data;

	protected int innerR, innerG, innerB;
	protected int outerR, outerG, outerB;

	public VisualEffect(int chains, int links)
	{
		this.chains = chains;
		this.links = links;

		data = new float[links];

		colorVariant = 1;

		slidingInnerR = new SlidingNumber(18, 253, 64, 768);
		slidingInnerG = new SlidingNumber(18, 253, 64, 768);
		slidingInnerB = new SlidingNumber(18, 253, 64, 768);

		slidingOuterR = new SlidingNumber(18, 253, 64, 768);
		slidingOuterG = new SlidingNumber(18, 253, 64, 768);
		slidingOuterB = new SlidingNumber(18, 253, 64, 768);

		innerR = slidingInnerR.nextValue();
		innerG = slidingInnerG.nextValue();
		innerB = slidingInnerB.nextValue();

		outerR = slidingOuterR.nextValue();
		outerG = slidingOuterG.nextValue();
		outerB = slidingOuterB.nextValue();
	}

	public void nextColor()
	{
		if(++colorVariant > 14)
		{
			colorVariant = 1;
		}
	}

	public void prevColor()
	{
		if(--colorVariant < 1)
		{
			colorVariant = 14;
		}
	}

	public void randomColor()
	{
		colorVariant = 1 + (rnd.nextInt() & 0x7FFFFFFF) % 14;
	}

	public void drawVisual(ExtGraphics g, int px, int py, int width, int height, boolean update)
	{
		float len, d, k, alpha1, alphaI;
		float x1, y1, x2, y2;

		if(width > height)
		{
			len = height / links;
		}
		else
		{
			len = width / links;
		}

		d = (float)(Math.PI * 2 / chains);

		if(update)
		{
			if(Math.abs(rnd.nextFloat()) * 50 < 5)
			{
				k = (float)(Math.abs(rnd.nextFloat() * 1440) * Math.PI / 180);
			}
			else
			{
				k = (float)(Math.abs(rnd.nextFloat() * 360) * Math.PI / 180);
			}

			data[0] = (float)(data[0] + Math.sin(k) / (float)((Math.abs(rnd.nextInt()) % 9) + 1));

			innerR = slidingInnerR.nextValue();
			innerG = slidingInnerG.nextValue();
			innerB = slidingInnerB.nextValue();

			outerR = slidingOuterR.nextValue();
			outerG = slidingOuterG.nextValue();
			outerB = slidingOuterB.nextValue();
		}

		for(int i = 1; i < links; i++)
		{
			data[i] = (float)(data[i] + (data[i - 1] - data[i]) * 0.1);
		}

		for(int j = 0; j < chains; j++)
		{
			x1 = (float)(0.5 * width);
			y1 = (float)(0.5 * height);

			x2 = x1;
			y2 = y1;

			for(int i = 1; i < links; i++)
			{
				alpha1 = j * d + data[1];
				alphaI = j * d + data[i];

				if(AudioOptions.playerVisualVariant == 1)
				{
					x2 = (float)(x1 + Math.cos(alpha1) * len);
					y2 = (float)(y1 + Math.sin(alphaI) * len);
					x1 = x1 + (float)(Math.cos(alphaI) * len);
				}
				else if(AudioOptions.playerVisualVariant == 2)
				{
					x2 = (float)(x1 + Math.sin(alphaI) * len);
					y2 = (float)(y1 + Math.cos(alphaI) * len);
				}
				else if(AudioOptions.playerVisualVariant == 3)
				{
					x2 = (float)(x1 + Math.tan(alpha1) * len);
					y2 = (float)(y1 + Math.cos(alphaI) * len);
					y1 = y1 - (float)(Math.sin(alphaI) * len);
				}
				else if(AudioOptions.playerVisualVariant == 4)
				{
					x2 = (float)(x1 + Math.tan(alpha1) * len);
					y2 = (float)(y1 + Math.cos(alphaI) * len);
					x1 = y1 - (float)(Math.sin(alphaI) * len);
				}
				else if(AudioOptions.playerVisualVariant == 5)
				{
					x2 = (float)(x1 + Math.tan(alpha1) * len);
					y2 = (float)(y1 + Math.cos(alphaI) * len);
					x1 = (float)(x1 * y2 * d + data[i] - (float)(Math.sin(alphaI) * len));
				}
				else if(AudioOptions.playerVisualVariant == 6)
				{
					x2 = (float)(x1 + Math.cos(alpha1) * len);
					y2 = (float)(y1 + Math.sin(alpha1) * len);
					x1 = x1 + (float)(Math.sin(alphaI) * len);
					y1 = y1 + (float)(Math.cos(alphaI) * len);
				}

				if(colorVariant == 1)
				{
					g.setColor(innerR * (links - i) / links + outerR * i / links, innerG * (links - i) / links + outerG * i / links, innerB * (links - i) / links + outerB * i / links);
				}
				else if(colorVariant == 2)
				{
					g.setColor(255, 255 - 255 * i / links, 0);
				}
				else if(colorVariant == 3)
				{
					g.setColor(innerR, 255 - 255 * i / links, 255);
				}
				else if(colorVariant == 4)
				{
					g.setColor(255 - 255 * i / links, 255, 0);
				}
				else if(colorVariant == 5)
				{
					g.setColor(255 - 255 * i / links, 255 - 255 * i / links, 255);
				}
				else if(colorVariant == 6)
				{
					g.setColor(255 - 255 * i / links, 255, 255);
				}
				else if(colorVariant == 7)
				{
					g.setColor(0, 255, 255 - 255 * i / links);
				}
				else if(colorVariant == 8)
				{
					g.setColor(255, innerG, 255 - 255 * i / links);
				}
				else if(colorVariant == 9)
				{
					g.setColor(255, 255 - 255 * i / links, 255 - 255 * i / links);
				}
				else if(colorVariant == 10)
				{
					g.setColor(innerR, 255 - 255 * i / links, 0);
				}
				else if(colorVariant == 11)
				{
					g.setColor(255, 255 - 255 * i / links, 255);
				}
				else if(colorVariant == 12)
				{
					g.setColor(255 - 255 * i / links, 0, 255);
				}
				else if(colorVariant == 13)
				{
					g.setColor(255 - 255 * i / links, 255 * i / links, 0);
				}
				else if(colorVariant == 14)
				{
					g.setColor(innerB, 255, 255 - 255 * i / links);
				}

				g.drawLine(px + (int)x1, py + (int)y1, px + (int)x2, py + (int)y2);

				x1 = x2;
				y1 = y2;
			}
		}
	}
}