package app;

import java.io.*;
import dma.*;
import time.*;

public final class GBACore extends Thread {

    // OBJECTS
    private DirectMemoryAccess dma;
    private Time time;
    private GBACanvas gui;
    // CONTROL
    public static final int REQUEST_START = 0;
    public static final int REQUEST_RESET = 1;
    public static final int REQUEST_STOP = 2;
    public static final int REQUEST_LOAD = 3;
    public static final int REQUEST_SAVE = 4;
    public static final int REQUEST_SLEEP = 5;
    private int request;
    private int param;
    public int[] pixels;
    private boolean op1;
    private boolean op2;
    private boolean res;
	private boolean draw;
    //public String rom_name;

    public GBACore(GBACanvas gui) {
        dma = new DirectMemoryAccess();
        time = new Time();
        this.gui = gui;
        pixels = new int[240 * 160];
        connectToDMA(dma);
        connectToTime(time);
        dma.connectToMemory(this);
        connectToGBACanvas(gui);
        time.connectToMemory(this);

        registers = new int[37];
        initARM();
    }

    public void reset() {
        cpu_reset();
        Memory_reset();
        dma.reset();
        gfx_reset();
        time.reset();
    }

    public void request(int _request, int _param) {
        request = _request;
        param = _param;
    }

    public final void run() {
        while (true) {
            switch (request) {
                case REQUEST_RESET:
                    reset();
                    break;
                case REQUEST_STOP:
                    request = 0;
                    return;
                /*case REQUEST_LOAD: {
                    SDbinding sd = null;
                    DataInputStream dis = null;
                    try {
                        sd = new SDbinding();
                        dis = sd.openDataInputStream(gui.getSaveFile(param));
                        if (dis != null) {
                            state_load(dis);
                        }
                    } catch (IOException e) {
                    } finally {
                        sd.close();
                        dis = null;
                        sd = null;
                    }
                    break;
                }
                case REQUEST_SAVE: {
                    DataOutputStream dos = null;
                    try {
                        dos = sd.openDataOutputStream(gui.getSaveFile(param));
                        if (dos != null) {
                            state_save(dos);
                            dos.flush();
                        }
                    } catch (IOException e) {
                    } finally {
                        dos = null;
                    }
                    break;
                }*/
                case REQUEST_SLEEP:
                    continue;
                default:
            }
            request = 0;

            draw = gui.startFrame();

			short inp = gui.readJoyPad();
            writeIO16(0x130, (short)(inp & 0x3FF) );
			if( inp<0 ) ioram[ 515 ] |= (byte)( ioram[ 513 ] & 0x10 );
            //System.out.println("gui.readJoyPad() = " + gui.readJoyPad());

            for (int scanline = 0; scanline < 228; scanline++) {
                ioram[0x0006] = (byte) scanline;
                cpu(960);
                enterHBlank();
                cpu(272);
                byte val = ioram[0x0004];
                val &= ~0x02;
                ioram[0x0004] = val;
                time.addTime(1232);
                if (scanline == 159) {
                    dma0.signalVBlank();
                    dma1.signalVBlank();
                    dma2.signalVBlank();
                    dma3.signalVBlank();
                    writeIO16(0x0202, (short) (readIO16(0x0202) | (readIO16(0x0200) & 0x0001)));
                    ioram[0x0004] |= 0x01;
                } else if (scanline == 227) {
                    ioram[0x0004] &= ~0x01;
                }
            }
            if (draw) {
                gui.endFrame();
            }
        }
    }
    private int[] registers;
    private int pipelineStage1, pipelineStage2;
    private static byte SPSR;
    private int mFlag;
    private boolean tFlag,
            fFlag,
            iFlag,
            vFlag,
            cFlag,
            nFlag,
            zFlag;

    // ----- Gestion des drapeaux -----
    private final void setMode(int mode) {
        int oldMode = mFlag;
        int newMode = mode;

        if (newMode != oldMode) {
            // Sauvegarder le contexte des registres de l'ancien mode
            switch (oldMode) {
                case 0x00000011: {
                    int temp = registers[17];
                    registers[17] = registers[8];
                    registers[8] = temp;

                    temp = registers[18];
                    registers[18] = registers[9];
                    registers[9] = temp;

                    temp = registers[19];
                    registers[19] = registers[10];
                    registers[10] = temp;

                    temp = registers[20];
                    registers[20] = registers[11];
                    registers[11] = temp;

                    temp = registers[21];
                    registers[21] = registers[12];
                    registers[12] = temp;

                    temp = registers[22];
                    registers[22] = registers[13];
                    registers[13] = temp;

                    temp = registers[23];
                    registers[23] = registers[14];
                    registers[14] = temp;
                }
                break;
                case 0x00000013: {
                    int temp = registers[25];
                    registers[25] = registers[13];
                    registers[13] = temp;

                    temp = registers[26];
                    registers[26] = registers[14];
                    registers[14] = temp;
                }
                break;
                case 0x00000017: {
                    int temp = registers[28];
                    registers[28] = registers[13];
                    registers[13] = temp;

                    temp = registers[29];
                    registers[29] = registers[14];
                    registers[14] = temp;
                }
                break;
                case 0x00000012: {
                    int temp = registers[31];
                    registers[31] = registers[13];
                    registers[13] = temp;

                    temp = registers[32];
                    registers[32] = registers[14];
                    registers[14] = temp;
                }
                break;
                case 0x0000001B: {
                    int temp = registers[34];
                    registers[34] = registers[13];
                    registers[13] = temp;

                    temp = registers[35];
                    registers[35] = registers[14];
                    registers[14] = temp;
                }
                break;
            }

            // Installer le contexte des registres du nouveau mode
            switch (newMode) {
                case 0x00000010:
                case 0x0000001F:
                    SPSR = -1;
                    break;
                case 0x00000011: {
                    int temp = registers[17];
                    registers[17] = registers[8];
                    registers[8] = temp;

                    temp = registers[18];
                    registers[18] = registers[9];
                    registers[9] = temp;

                    temp = registers[19];
                    registers[19] = registers[10];
                    registers[10] = temp;

                    temp = registers[20];
                    registers[20] = registers[11];
                    registers[11] = temp;

                    temp = registers[21];
                    registers[21] = registers[12];
                    registers[12] = temp;

                    temp = registers[22];
                    registers[22] = registers[13];
                    registers[13] = temp;

                    temp = registers[23];
                    registers[23] = registers[14];
                    registers[14] = temp;

                    SPSR = 24;
                }
                break;
                case 0x00000013: {
                    int temp = registers[25];
                    registers[25] = registers[13];
                    registers[13] = temp;

                    temp = registers[26];
                    registers[26] = registers[14];
                    registers[14] = temp;

                    SPSR = 27;
                }
                break;
                case 0x00000017: {
                    int temp = registers[28];
                    registers[28] = registers[13];
                    registers[13] = temp;

                    temp = registers[29];
                    registers[29] = registers[14];
                    registers[14] = temp;

                    SPSR = 30;
                }
                break;
                case 0x00000012: {
                    int temp = registers[31];
                    registers[31] = registers[13];
                    registers[13] = temp;

                    temp = registers[32];
                    registers[32] = registers[14];
                    registers[14] = temp;

                    SPSR = 33;
                }
                break;
                case 0x0000001B: {
                    int temp = registers[34];
                    registers[34] = registers[13];
                    registers[13] = temp;

                    temp = registers[35];
                    registers[35] = registers[14];
                    registers[14] = temp;

                    SPSR = 36;
                }
                break;
            }
            mFlag = newMode;
        }
    }

    // ----- Gestion des interruptions -----
    private final void generateInterrupt(int newMode, int interruptVector, int pcValue) {
        int cpsr = 0x00000000;
        cpsr |= mFlag;
        cpsr = tFlag ? (cpsr | 0x00000020) : (cpsr & ~0x00000020);
        cpsr = fFlag ? (cpsr | 0x00000040) : (cpsr & ~0x00000040);
        cpsr = iFlag ? (cpsr | 0x00000080) : (cpsr & ~0x00000080);
        cpsr = vFlag ? (cpsr | 0x10000000) : (cpsr & ~0x10000000);
        cpsr = cFlag ? (cpsr | 0x20000000) : (cpsr & ~0x20000000);
        cpsr = zFlag ? (cpsr | 0x40000000) : (cpsr & ~0x40000000);
        cpsr = nFlag ? (cpsr | 0x80000000) : (cpsr & ~0x80000000);
        int oldCPSR = cpsr;
        setMode(newMode);
        tFlag = false;
        iFlag = true;
        registers[SPSR] = oldCPSR;
        registers[14] = pcValue;
        registers[15] = interruptVector;
        registers[15] &= 0xFFFFFFFC;
        pipelineStage1 = getWord(registers[15]);
        registers[15] += 4;
        pipelineStage2 = getWord(registers[15]);
    }

    // ----- Fonctions �l�mentaires du noyau CPU -----
    private final void cpu_reset() {
        // Initialiser les registres
        for (byte i = 0; i < 37; i++) {
            registers[i] = 0;
        }
        // Initialiser les drapeaux
        mFlag = 0;
        tFlag = vFlag = cFlag = zFlag = nFlag = false;
        fFlag = iFlag = true;
        setMode(0x00000013);
        // Initialiser le PC
        registers[15] = 0x00000000;

        // Sauter l'ex�cution du BIOS
        iFlag = false;
        setMode(0x0000001F);
        registers[15] = 0x08000000;
        registers[13] = 0x03007F00;
        registers[25] = 0x03007FE0;
        registers[31] = 0x03007FA0;

        if (tFlag) {
            registers[15] &= 0xFFFFFFFE;
            pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
            registers[15] += 2;
            pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
        } else {
            registers[15] &= 0xFFFFFFFC;
            pipelineStage1 = getWord(registers[15]);
            registers[15] += 4;
            pipelineStage2 = getWord(registers[15]);
        }
    }

    private final void cpu(int cycles) {
        while (cycles > 0) {
            if (!iFlag
                    && (ioram[0x00000208] != 0)
                    && (((short) ((ioram[0x00000200] & 0x00FF) | (ioram[0x00000201] << 8)) & (short) ((ioram[0x00000202] & 0x00FF) | (ioram[0x00000203] << 8))) != 0)) {
                generateInterrupt(0x00000012, 0x00000018, registers[15] + (tFlag ? 2 : 0)); // (getPC() + 2) & 0xFFFFFFFC
            } else {
                int opcode;
                byte instruction;
                if (tFlag) { // THUMB state
                    registers[15] += 2;
                    opcode = pipelineStage1;
                    pipelineStage1 = pipelineStage2;
                    pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                    instruction = thumbInstruction[opcode >>> 8];
                    //executeTHUMB(opcode, instruction);
                    switch (instruction) {
                        case 0x01: {
                            int rdIndex = opcode & 0x0007;
                            int rsIndex = (opcode >>> 3) & 0x0007;
                            int rsValue = registers[rsIndex];
                            int rdValue = 0;
                            int shiftAmount = (opcode >>> 6) & 0x001F;

                            switch (opcode & 0x1800) {
                                case 0x0000: // LSL Rd, Rs, #Offset
                                    if (shiftAmount != 0) {
                                        cFlag = ((rsValue & (1 << (32 - shiftAmount))) != 0);
                                        rdValue = rsValue << shiftAmount;
                                    }
                                    break;

                                case 0x0800: // LSR Rd, Rs, #Offset
                                    if (shiftAmount == 0) {
                                        cFlag = (rsValue < 0);
                                        rdValue = 0;
                                    } else {
                                        cFlag = ((rsValue & (1 << (shiftAmount - 1))) != 0);
                                        rdValue = rsValue >>> shiftAmount;
                                    }
                                    break;

                                case 0x1000: // ASR Rd, Rs, #Offset
                                    if (shiftAmount == 0) {
                                        cFlag = (rsValue < 0);
                                        rdValue = rsValue >> 31;
                                    } else {
                                        cFlag = ((rsValue & (1 << (shiftAmount - 1))) != 0);
                                        rdValue = rsValue >> shiftAmount;
                                    }
                                    break;

                                default: // Unknown
                            }

                            registers[rdIndex] = rdValue;
                            zFlag = (rdValue == 0);
                            nFlag = (rdValue < 0);
                        }
                        break;

                        case 0x02: {
                            int rdIndex = opcode & 0x0007;
                            int rsIndex = (opcode >>> 3) & 0x0007;
                            int rsValue = registers[rsIndex];
                            int rdValue;
                            int operand3 = (opcode >>> 6) & 0x0007;

                            if ((opcode & 0x0400) == 0) { // operand3 is a register
                                operand3 = registers[operand3];
                            }

                            if ((opcode & 0x0200) == 0) { // ADD Rd, Rs, (Rn / #nn)
                                rdValue = rsValue + operand3;
                                op1 = (rsValue < 0);
                                op2 = (operand3 < 0);
                                res = (rdValue < 0);

                                vFlag = (op1 && op2 && !res) || (!op1 && !op2 && res);
                                cFlag = (op1 && op2) || (op1 && !res) || (op2 && !res);
                            } else { // SUB Rd, Rs, (Rn / #nn)
                                rdValue = rsValue - operand3;
                                op1 = (rsValue < 0);
                                op2 = (operand3 < 0);
                                res = (rdValue < 0);

                                vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                                cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                            }

                            registers[rdIndex] = rdValue;
                            zFlag = (rdValue == 0);
                            nFlag = (rdValue < 0);
                        }
                        break;

                        case 0x03: {
                            int rdIndex = (opcode >>> 8) & 0x0007;
                            int rdOldValue = registers[rdIndex];
                            int rdNewValue = 0;
                            int immediate = opcode & 0x00FF;

                            switch (opcode & 0x1800) {
                                case 0x0000: // MOV Rd, #nn
                                    rdNewValue = immediate;
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x0800: // CMP Rd, #nn
                                    rdNewValue = rdOldValue - immediate;
                                    op1 = (rdOldValue < 0);
                                    op2 = (immediate < 0);
                                    res = (rdNewValue < 0);

                                    vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                                    cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                                    break;

                                case 0x1000: // ADD Rd, #nn
                                    rdNewValue = rdOldValue + immediate;
                                    op1 = (rdOldValue < 0);
                                    op2 = (immediate < 0);
                                    res = (rdNewValue < 0);

                                    vFlag = (op1 && op2 && !res) || (!op1 && !op2 && res);
                                    cFlag = (op1 && op2) || (op1 && !res) || (op2 && !res);
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x1800: // SUB Rd, #nn
                                    rdNewValue = rdOldValue - immediate;
                                    op1 = (rdOldValue < 0);
                                    op2 = (immediate < 0);
                                    res = (rdNewValue < 0);

                                    vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                                    cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                default: // Unknown
                            }

                            zFlag = (rdNewValue == 0);
                            nFlag = (rdNewValue < 0);
                        }
                        break;

                        case 0x04: {
                            int rdIndex = opcode & 0x0007;
                            int rsIndex = (opcode >>> 3) & 0x0007;
                            int rsValue = registers[rsIndex];
                            int rdOldValue = registers[rdIndex];
                            int rdNewValue = rdOldValue;

                            switch (opcode & 0x03C0) {
                                case 0x0000: // AND Rd, Rs
                                    rdNewValue &= rsValue;
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x0040: // EOR Rd, Rs
                                    rdNewValue ^= rsValue;
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x0080: // LSL Rd, Rs
                                    if (rsValue != 0) {
                                        if (rsValue < 32) {
                                            cFlag = ((rdOldValue & (1 << (32 - rsValue))) != 0);
                                            rdNewValue <<= rsValue;
                                        } else if (rsValue == 32) {
                                            cFlag = ((rdOldValue & 0x00000001) != 0);
                                            rdNewValue = 0;
                                        } else {
                                            cFlag = false;
                                            rdNewValue = 0;
                                        }
                                        registers[rdIndex] = rdNewValue;
                                    }
                                    break;

                                case 0x00C0: // LSR Rd, Rs
                                    if (rsValue != 0) {
                                        if (rsValue < 32) {
                                            cFlag = ((rdOldValue & (1 << (rsValue - 1))) != 0);
                                            rdNewValue >>>= rsValue;
                                        } else if (rsValue == 32) {
                                            cFlag = ((rdOldValue & 0x80000000) != 0);
                                            rdNewValue = 0;
                                        } else {
                                            cFlag = false;
                                            rdNewValue = 0;
                                        }
                                        registers[rdIndex] = rdNewValue;
                                    }
                                    break;

                                case 0x0100: // ASR Rd, Rs
                                    if (rsValue != 0) {
                                        if (rsValue < 32) {
                                            cFlag = ((rdOldValue & (1 << (rsValue - 1))) != 0);
                                            rdNewValue >>= rsValue;
                                        } else {
                                            cFlag = ((rdOldValue & 0x80000000) != 0);
                                            rdNewValue >>= 31;
                                        }
                                        registers[rdIndex] = rdNewValue;
                                    }
                                    break;

                                case 0x0140: // ADC Rd, Rs
                                    rdNewValue = rdOldValue + rsValue + (cFlag ? 1 : 0);
                                    op1 = (rdOldValue < 0);
                                    op2 = (rsValue < 0);
                                    res = (rdNewValue < 0);

                                    vFlag = (op1 && op2 && !res) || (!op1 && !op2 && res);
                                    cFlag = (op1 && op2) || (op1 && !res) || (op2 && !res);
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x0180: // SBC Rd, Rs
                                    rdNewValue = rdOldValue - rsValue - (cFlag ? 0 : 1);
                                    op1 = (rdOldValue < 0);
                                    op2 = (rsValue < 0);
                                    res = (rdNewValue < 0);

                                    vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                                    cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x01C0: // ROR Rd, Rs
                                    if (rsValue != 0) {
                                        rsValue &= 0x0000001F;
                                        if (rsValue == 0) {
                                            cFlag = ((rdOldValue & 0x80000000) != 0);
                                        } else {
                                            cFlag = ((rdOldValue & (1 << (rsValue - 1))) != 0);
                                            rdNewValue = (rdOldValue << (32 - rsValue)) | (rdOldValue >>> rsValue);
                                            registers[rdIndex] = rdNewValue;
                                        }
                                    }
                                    break;

                                case 0x0200: // TST Rd, Rs
                                    rdNewValue &= rsValue;
                                    break;

                                case 0x0240: // NEG Rd, Rs
                                    rdNewValue = -rsValue;
                                    op1 = false;
                                    op2 = (rsValue < 0);
                                    res = (rdNewValue < 0);

                                    vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                                    cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x0280: // CMP Rd, Rs
                                    rdNewValue -= rsValue;
                                    op1 = (rdOldValue < 0);
                                    op2 = (rsValue < 0);
                                    res = (rdNewValue < 0);

                                    vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                                    cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                                    break;

                                case 0x02C0: // CMN Rd, Rs
                                    rdNewValue += rsValue;
                                    op1 = (rdOldValue < 0);
                                    op2 = (rsValue < 0);
                                    res = (rdNewValue < 0);

                                    vFlag = (op1 && op2 && !res) || (!op1 && !op2 && res);
                                    cFlag = (op1 && op2) || (op1 && !res) || (op2 && !res);
                                    break;

                                case 0x0300: // ORR Rd, Rs
                                    rdNewValue |= rsValue;
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x0340: // MUL Rd, Rs
                                    rdNewValue *= rsValue;
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x0380: // BIC Rd, Rs
                                    rdNewValue &= ~rsValue;
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                case 0x03C0: // MVN Rd, Rs
                                    rdNewValue = ~rsValue;
                                    registers[rdIndex] = rdNewValue;
                                    break;

                                default: // Unknown
                            }

                            zFlag = (rdNewValue == 0);
                            nFlag = (rdNewValue < 0);
                        }
                        break;

                        case 0x05: {
                            int rsIndex = (opcode >>> 3) & 0x000F;
                            int rsValue = registers[rsIndex];
                            int rdIndex, rdValue;

                            switch (opcode & 0x0300) {
                                case 0x0000: // ADD Rd, Rs
                                    rdIndex = (opcode & 0x0007) | ((opcode & 0x0080) >>> 4);
                                    rdValue = registers[rdIndex];
                                    registers[rdIndex] = rdValue + rsValue;
                                    if (rdIndex == 15) {
                                        registers[15] &= 0xFFFFFFFE;
                                        pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                                        registers[15] += 2;
                                        pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                                    }
                                    break;

                                case 0x0100: // CMP Rd, Rs
                                    rdIndex = (opcode & 0x0007) | ((opcode & 0x0080) >>> 4);
                                    rdValue = registers[rdIndex];
                                    int result = rdValue - rsValue;
                                    op1 = (rdValue < 0);
                                    op2 = (rsValue < 0);
                                    res = (result < 0);

                                    vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                                    cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                                    zFlag = (result == 0);
                                    nFlag = (result < 0);
                                    break;

                                case 0x0200: // MOV Rd, Rs
                                    rdIndex = (opcode & 0x0007) | ((opcode & 0x0080) >>> 4);
                                    registers[rdIndex] = rsValue;
                                    if (rdIndex == 15) {
                                        registers[15] &= 0xFFFFFFFE;
                                        pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                                        registers[15] += 2;
                                        pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                                    }
                                    break;

                                case 0x0300: // BX Rs
                                    tFlag = ((rsValue & 0x00000001) != 0);
                                    registers[15] = rsValue;
                                    if (tFlag) {
                                        registers[15] &= 0xFFFFFFFE;
                                        pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                                        registers[15] += 2;
                                        pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                                    } else {
                                        registers[15] &= 0xFFFFFFFC;
                                        pipelineStage1 = getWord(registers[15]);
                                        registers[15] += 4;
                                        pipelineStage2 = getWord(registers[15]);
                                    }
                                    break;

                                default: // Unknown
                            }
                        }
                        break;

                        case 0x06: {
                            // LDR Rd, [PC, #nn]
                            int rdIndex = (opcode >>> 8) & 0x0007;
                            int offset = (registers[15] & 0xFFFFFFFC) + ((opcode & 0x00FF) << 2);
                            registers[rdIndex] = loadWord(offset);
                        }
                        break;

                        case 0x07: {
                            int rdIndex = opcode & 0x0007;
                            int rbIndex = (opcode >>> 3) & 0x0007;
                            int roIndex = (opcode >>> 6) & 0x0007;
                            int offset = registers[rbIndex] + registers[roIndex];

                            switch (opcode & 0x0C00) {
                                case 0x0000: // STR Rd, [Rb, Ro]
                                    storeWord(offset, registers[rdIndex]);
                                    break;

                                case 0x0400: // STRB Rd, [Rb, Ro]
                                    storeByte(offset, (byte) registers[rdIndex]);
                                    break;

                                case 0x0800: // LDR Rd, [Rb, Ro]
                                    registers[rdIndex] = loadWord(offset);
                                    break;

                                case 0x0C00: // LDRB Rd, [Rb, Ro]
                                    registers[rdIndex] = loadByte(offset) & 0x000000FF;
                                    break;

                                default: // Unknown
                            }
                        }
                        break;

                        case 0x08: {
                            int rdIndex = opcode & 0x0007;
                            int rbIndex = (opcode >>> 3) & 0x0007;
                            int roIndex = (opcode >>> 6) & 0x0007;
                            int offset = registers[rbIndex] + registers[roIndex];

                            switch (opcode & 0x0C00) {
                                case 0x0000: // STRH Rd, [Rb, Ro]
                                    storeHalfWord(offset, (short) registers[rdIndex]);
                                    break;

                                case 0x0400: // LDSB Rd, [Rb, Ro]
                                    registers[rdIndex] = loadByte(offset);
                                    break;

                                case 0x0800: // LDRH Rd, [Rb, Ro]
                                    registers[rdIndex] = loadHalfWord(offset) & 0x0000FFFF;
                                    break;

                                case 0x0C00: // LDSH Rd, [Rb, Ro]
                                    registers[rdIndex] = loadHalfWord(offset);
                                    break;

                                default: // Unknown
                            }
                        }
                        break;

                        case 0x09: {
                            int rdIndex = opcode & 0x0007;
                            int rbIndex = (opcode >>> 3) & 0x0007;
                            int rbValue = registers[rbIndex];
                            int offset = (opcode >>> 6) & 0x001F;

                            switch (opcode & 0x1800) {
                                case 0x0000: // STR Rd, [Rb, #nn]
                                    storeWord(rbValue + (offset << 2), registers[rdIndex]);
                                    break;

                                case 0x0800: // LDR Rd, [Rb, #nn]
                                    registers[rdIndex] = loadWord(rbValue + (offset << 2));
                                    break;

                                case 0x1000: // STRB Rd, [Rb, #nn]
                                    storeByte(rbValue + offset, (byte) registers[rdIndex]);
                                    break;

                                case 0x1800: // LDRB Rd, [Rb, #nn]
                                    registers[rdIndex] = loadByte(rbValue + offset) & 0x000000FF;
                                    break;

                                default: // Unknown
                            }
                        }
                        break;

                        case 0x0A: {
                            int rdIndex = opcode & 0x0007;
                            int rbIndex = (opcode >>> 3) & 0x0007;
                            int rbValue = registers[rbIndex];
                            int offset = (opcode >>> 5) & 0x003E;

                            if ((opcode & 0x0800) == 0) { // STRH Rd, [Rb, #nn]
                                storeHalfWord(rbValue + offset, (short) registers[rdIndex]);
                            } else { // LDRH Rd, [Rb, #nn]
                                registers[rdIndex] = loadHalfWord(rbValue + offset) & 0x0000FFFF;
                            }
                        }
                        break;

                        case 0x0B: {
                            int rdIndex = (opcode >>> 8) & 0x0007;
                            int spValue = registers[13];
                            int offset = (opcode & 0x00FF) << 2;

                            if ((opcode & 0x0800) == 0) { // STR Rd, [SP, #nn]
                                storeWord(spValue + offset, registers[rdIndex]);
                            } else { // LDR Rd, [SP, #nn]
                                registers[rdIndex] = loadWord(spValue + offset);
                            }
                        }
                        break;

                        case 0x0C: {
                            int rdIndex = (opcode >>> 8) & 0x0007;
                            int operand2 = ((opcode & 0x0800) == 0) ? (registers[15] & 0xFFFFFFFC) : registers[13];
                            int offset = (opcode & 0x00FF) << 2;

                            registers[rdIndex] = operand2 + offset; // ADD Rd, PC/SP, #nn
                        }
                        break;

                        case 0x0D: {
                            int spValue = registers[13];
                            int offset = (opcode & 0x007F) << 2;

                            if ((opcode & 0x0080) != 0) {
                                offset = -offset;
                            }
                            // ADD SP, #nn
                            registers[13] = spValue + offset;
                        }
                        break;

                        case 0x0E: {
                            int spValue = registers[13] & 0xFFFFFFFC;

                            if ((opcode & 0x0800) == 0) { // PUSH {Rlist}

                                if ((opcode & 0x0100) != 0) { // PUSH LR
                                    spValue -= 4;
                                    storeWord(spValue, registers[14]);
                                }

                                for (int i = 7; i >= 0; i--) {
                                    if ((opcode & (1 << i)) != 0) {
                                        spValue -= 4;
                                        storeWord(spValue, registers[i]);
                                    }
                                }

                            } else { // POP {Rlist}

                                for (int i = 0; i <= 7; i++) {
                                    if ((opcode & (1 << i)) != 0) {
                                        registers[i] = loadWord(spValue);
                                        spValue += 4;
                                    }
                                }

                                if ((opcode & 0x0100) != 0) { // POP PC
                                    registers[15] = loadWord(spValue);
                                    spValue += 4;
                                    registers[15] &= 0xFFFFFFFE;
                                    pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                                    registers[15] += 2;
                                    pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                                }

                            }

                            registers[13] = spValue;
                        }
                        break;

                        case 0x0F: {
                            int rbIndex = (opcode >>> 8) & 0x0007;
                            int rbValue = registers[rbIndex] & 0xFFFFFFFC;

                            if ((opcode & 0x0800) == 0) { // STMIA Rb!, {Rlist}

                                for (int i = 0; i <= 7; i++) {
                                    if ((opcode & (1 << i)) != 0) {
                                        storeWord(rbValue, registers[i]);
                                        rbValue += 4;
                                    }
                                }

                            } else { // LDMIA Rb!, {Rlist}

                                for (int i = 0; i <= 7; i++) {
                                    if ((opcode & (1 << i)) != 0) {
                                        registers[i] = loadWord(rbValue);
                                        rbValue += 4;
                                    }
                                }

                            }

                            registers[rbIndex] = rbValue;
                        }
                        break;

                        case 0x10: {
                            boolean condition;

                            switch (opcode & 0x0F00) {
                                case 0x0000: // BEQ label
                                    condition = zFlag;
                                    break;

                                case 0x0100: // BNE label
                                    condition = !zFlag;
                                    break;

                                case 0x0200: // BCS label
                                    condition = cFlag;
                                    break;

                                case 0x0300: // BCC label
                                    condition = !cFlag;
                                    break;

                                case 0x0400: // BMI label
                                    condition = nFlag;
                                    break;

                                case 0x0500: // BPL label
                                    condition = !nFlag;
                                    break;

                                case 0x0600: // BVS label
                                    condition = vFlag;
                                    break;

                                case 0x0700: // BVC label
                                    condition = !vFlag;
                                    break;

                                case 0x0800: // BHI label
                                    condition = (cFlag && !zFlag);
                                    break;

                                case 0x0900: // BLS label
                                    condition = (!cFlag || zFlag);
                                    break;

                                case 0x0A00: // BGE label
                                    condition = (nFlag == vFlag);
                                    break;

                                case 0x0B00: // BLT label
                                    condition = (nFlag != vFlag);
                                    break;

                                case 0x0C00: // BGT label
                                    condition = (!zFlag && (nFlag == vFlag));
                                    break;

                                case 0x0D00: // BLE label
                                    condition = (zFlag || (nFlag != vFlag));
                                    break;

                                default: // Unknown
                                    condition = false;
                            }

                            if (condition) {
                                int offset = ((byte) (opcode & 0x00FF)) << 1;
                                registers[15] += offset;
                                registers[15] &= 0xFFFFFFFE;
                                pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                                registers[15] += 2;
                                pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                            }
                        }
                        break;

                        case 0x11: {
                            // SWI nn
                            generateInterrupt(0x00000013, 0x00000008, registers[15] - (tFlag ? 2 : 4));
                        }
                        break;

                        case 0x12: {
                            // B label
                            int offset = (opcode & 0x03FF) << 1;
                            if ((opcode & 0x0400) != 0) {
                                offset |= 0xFFFFF800;
                            }

                            registers[15] += offset;
                            registers[15] &= 0xFFFFFFFE;
                            pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                            registers[15] += 2;
                            pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                        }
                        break;

                        case 0x13: {
                            int offset = opcode & 0x07FF;

                            if ((opcode & 0x0800) == 0) {
                                // BLL label
                                if ((opcode & 0x0400) != 0) {
                                    offset |= 0xFFFFF800;
                                }
                                registers[14] = registers[15] + (offset << 12);
                            } else {
                                // BLH label
                                int lrValue = registers[14];
                                int pcValue = registers[15] - (tFlag ? 2 : 4);
                                registers[15] = lrValue + (offset << 1);
                                registers[14] = pcValue | 0x00000001;
                                registers[15] &= 0xFFFFFFFE;
                                pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                                registers[15] += 2;
                                pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                            }
                        }
                        break;

                        case 0x7F: {
                            generateInterrupt(0x0000001B, 0x00000004, registers[15] - (tFlag ? 2 : 4));
                        }
                        break;
                    }
                } else { // ARM state
                    registers[15] += 4;
                    opcode = pipelineStage1;
                    pipelineStage1 = pipelineStage2;
                    pipelineStage2 = getWord(registers[15]);
                    instruction = armInstruction[((opcode >>> 12) & 0x0000FF00) | ((opcode >>> 4) & 0x000000FF)];
                    executeARM(opcode, instruction);
                }
            }
            cycles -= 4;
        }
    }

    private final void executeARM(int opcode, byte instruction) {
        switch (instruction) {
            case 0x03: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int rnIndex = opcode & 0x0000000F;
                int rnValue = registers[rnIndex];

                tFlag = ((rnValue & 0x00000001) != 0);
                registers[15] = rnValue;
                if (tFlag) {
                    registers[15] &= 0xFFFFFFFE;
                    pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                    registers[15] += 2;
                    pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                } else {
                    registers[15] &= 0xFFFFFFFC;
                    pipelineStage1 = getWord(registers[15]);
                    registers[15] += 4;
                    pipelineStage2 = getWord(registers[15]);
                }
            }
            break;

            case 0x04: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int offset = opcode & 0x00FFFFFF;
                if ((offset & 0x00800000) != 0) {
                    offset |= 0xFF000000;
                }
                offset <<= 2;

                if ((opcode & 0x01000000) != 0) {
                    registers[14] = registers[15] - (tFlag ? 2 : 4);
                }
                registers[15] += offset;
                registers[15] &= 0xFFFFFFFC;
                pipelineStage1 = getWord(registers[15]);
                registers[15] += 4;
                pipelineStage2 = getWord(registers[15]);
            }
            break;

            case 0x05: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int rnIndex = (opcode >>> 16) & 0x0000000F;
                int rnValue = registers[rnIndex];
                boolean CFlag = cFlag;

                int operand1 = rnValue;
                int operand2;
                if ((opcode & 0x02000000) != 0) { // operand2 is an immediate value
                    operand2 = opcode & 0x000000FF;
                    // shiftType == ROR
                    int shiftAmount = (opcode & 0x00000F00) >>> 7;
                    if (shiftAmount != 0) {
                        CFlag = ((operand2 & (1 << (shiftAmount - 1))) != 0);
                        operand2 = (operand2 << (32 - shiftAmount)) | (operand2 >>> shiftAmount);
                    }
                } else { // operand2 is a register
                    int rmIndex = opcode & 0x0000000F;
                    int rmValue = registers[rmIndex];
                    int shiftType = opcode & 0x00000060;
                    int shiftAmount;

                    operand2 = rmValue;

                    if ((opcode & 0x00000010) == 0) { // Shift by immediate
                        shiftAmount = (opcode >>> 7) & 0x0000001F;

                        switch (shiftType) {
                            case 0x00000000: // LSL
                                if (shiftAmount != 0) {
                                    CFlag = ((operand2 & (1 << (32 - shiftAmount))) != 0);
                                    operand2 <<= shiftAmount;
                                }
                                break;

                            case 0x00000020: // LSR
                                if (shiftAmount != 0) {
                                    CFlag = ((operand2 & (1 << (shiftAmount - 1))) != 0);
                                    operand2 >>>= shiftAmount;
                                } else {
                                    CFlag = ((operand2 & 0x80000000) != 0);
                                    operand2 = 0;
                                }
                                break;

                            case 0x00000040: // ASR
                                if (shiftAmount != 0) {
                                    CFlag = ((operand2 & (1 << (shiftAmount - 1))) != 0);
                                    operand2 >>= shiftAmount;
                                } else {
                                    CFlag = ((operand2 & 0x80000000) != 0);
                                    operand2 >>= 31; // Fill all bits of operand2 with bit 31
                                }
                                break;

                            case 0x00000060: // ROR
                                if (shiftAmount != 0) {
                                    CFlag = ((operand2 & (1 << (shiftAmount - 1))) != 0);
                                    operand2 = (operand2 << (32 - shiftAmount)) | (operand2 >>> shiftAmount);
                                } else {
                                    CFlag = ((operand2 & 0x00000001) != 0);
                                    operand2 >>>= 1;
                                    if (CFlag) {
                                        operand2 |= 0x80000000;
                                    }
                                }
                                break;
                        }
                    } else { // Shift by register
                        int rsIndex = (opcode >>> 8) & 0x0000000F;
                        int rsValue = registers[rsIndex];

                        shiftAmount = (rsValue + ((rsIndex == 15) ? 4 : 0)) & 0x000000FF;

                        switch (shiftType) {
                            case 0x00000000: // LSL
                                if (shiftAmount != 0) {
                                    if (shiftAmount < 32) {
                                        CFlag = ((operand2 & (1 << (32 - shiftAmount))) != 0);
                                        operand2 <<= shiftAmount;
                                    } else if (shiftAmount == 32) {
                                        CFlag = ((operand2 & 0x00000001) != 0);
                                        operand2 = 0;
                                    } else {
                                        CFlag = false;
                                        operand2 = 0;
                                    }
                                }
                                break;

                            case 0x00000020: // LSR
                                if (shiftAmount != 0) {
                                    if (shiftAmount < 32) {
                                        CFlag = ((operand2 & (1 << (shiftAmount - 1))) != 0);
                                        operand2 >>>= shiftAmount;
                                    } else if (shiftAmount == 32) {
                                        CFlag = ((operand2 & 0x80000000) != 0);
                                        operand2 = 0;
                                    } else {
                                        CFlag = false;
                                        operand2 = 0;
                                    }
                                }
                                break;

                            case 0x00000040: // ASR
                                if (shiftAmount != 0) {
                                    if (shiftAmount < 32) {
                                        CFlag = ((operand2 & (1 << (shiftAmount - 1))) != 0);
                                        operand2 >>= shiftAmount;
                                    } else {
                                        CFlag = ((operand2 & 0x80000000) != 0);
                                        operand2 >>= 31;
                                    }
                                }
                                break;

                            case 0x00000060: // ROR
                                if (shiftAmount != 0) {
                                    shiftAmount &= 0x0000001F;
                                    if (shiftAmount != 0) {
                                        CFlag = ((operand2 & (1 << (shiftAmount - 1))) != 0);
                                        operand2 = (operand2 << (32 - shiftAmount)) | (operand2 >>> shiftAmount);
                                    } else {
                                        CFlag = ((operand2 & 0x80000000) != 0);
                                    }
                                } else {
                                    CFlag = ((operand2 & 0x80000000) != 0);
                                }
                                break;
                        }
                    }
                }

                int rdIndex = (opcode >>> 12) & 0x0000000F;
                int rdValue;
                boolean sBit = ((opcode & 0x00100000) != 0);

                switch (opcode & 0x01E00000) {
                    case 0x00000000: // AND{cond}{S} Rd, Rn, Op2
                        rdValue = operand1 & operand2;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            cFlag = CFlag;
                        }
                        break;

                    case 0x00200000: // EOR{cond}{S} Rd, Rn, Op2
                        rdValue = operand1 ^ operand2;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            cFlag = CFlag;
                        }
                        break;

                    case 0x00400000: // SUB{cond}{S} Rd, Rn, Op2
                        rdValue = operand1 - operand2;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            op1 = (operand1 < 0);
                            op2 = (operand2 < 0);
                            res = (rdValue < 0);

                            vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                            cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                        }
                        break;

                    case 0x00600000: // RSB{cond}{S} Rd, Rn, Op2
                        rdValue = operand2 - operand1;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            op1 = (operand2 < 0);
                            op2 = (operand1 < 0);
                            res = (rdValue < 0);

                            vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                            cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                        }
                        break;

                    case 0x00800000: // ADD{cond}{S} Rd, Rn, Op2
                        rdValue = operand1 + operand2;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            op1 = (operand1 < 0);
                            op2 = (operand2 < 0);
                            res = (rdValue < 0);

                            vFlag = (op1 && op2 && !res) || (!op1 && !op2 && res);
                            cFlag = (op1 && op2) || (op1 && !res) || (op2 && !res);
                        }
                        break;

                    case 0x00A00000: // ADC{cond}{S} Rd, Rn, Op2
                        rdValue = operand1 + operand2 + (CFlag ? 1 : 0);
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            op1 = (operand1 < 0);
                            op2 = (operand2 < 0);
                            res = (rdValue < 0);

                            vFlag = (op1 && op2 && !res) || (!op1 && !op2 && res);
                            cFlag = (op1 && op2) || (op1 && !res) || (op2 && !res);
                        }
                        break;

                    case 0x00C00000: // SBC{cond}{S} Rd, Rn, Op2
                        rdValue = operand1 - operand2 - (CFlag ? 0 : 1);
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            op1 = (operand1 < 0);
                            op2 = (operand2 < 0);
                            res = (rdValue < 0);

                            vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                            cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                        }
                        break;

                    case 0x00E00000: // RSC{cond}{S} Rd, Rn, Op2
                        rdValue = operand2 - operand1 - (CFlag ? 0 : 1);
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            op1 = (operand2 < 0);
                            op2 = (operand1 < 0);
                            res = (rdValue < 0);

                            vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                            cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                        }
                        break;

                    case 0x01000000: // TST{cond} Rn, Op2
                        rdValue = operand1 & operand2;
                        cFlag = CFlag;
                        break;

                    case 0x01200000: // TEQ{cond} Rn, Op2
                        rdValue = operand1 ^ operand2;
                        cFlag = CFlag;
                        break;

                    case 0x01400000: // CMP{cond} Rn, Op2
                        rdValue = operand1 - operand2;
                        op1 = (operand1 < 0);
                        op2 = (operand2 < 0);
                        res = (rdValue < 0);

                        vFlag = (op1 && !op2 && !res) || (!op1 && op2 && res);
                        cFlag = (op1 && !op2) || (op1 && !res) || (!op2 && !res);
                        break;

                    case 0x01600000: // CMN{cond} Rn, Op2
                        rdValue = operand1 + operand2;
                        op1 = (operand1 < 0);
                        op2 = (operand2 < 0);
                        res = (rdValue < 0);

                        vFlag = (op1 && op2 && !res) || (!op1 && !op2 && res);
                        cFlag = (op1 && op2) || (op1 && !res) || (op2 && !res);
                        break;

                    case 0x01800000: // ORR{cond}{S} Rd, Rn, Op2
                        rdValue = operand1 | operand2;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            cFlag = CFlag;
                        }
                        break;

                    case 0x01A00000: // MOV{cond}{S} Rd, Op2
                        rdValue = operand2;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            cFlag = CFlag;
                        }
                        break;

                    case 0x01C00000: // BIC{cond}{S} Rd, Rn, Op2
                        rdValue = operand1 & ~operand2;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            cFlag = CFlag;
                        }
                        break;

                    case 0x01E00000: // MVN{cond}{S} Rd, Op2
                        rdValue = ~operand2;
                        registers[rdIndex] = rdValue;
                        if (sBit) {
                            cFlag = CFlag;
                        }
                        break;

                    default:
                        rdValue = 0;
                }

                if (sBit) {
                    nFlag = (rdValue < 0);
                    zFlag = (rdValue == 0);
                }

                if (rdIndex == 15) {
                    if (sBit) {
                        int value = registers[SPSR];
                        setMode(value & 0x0000001F);
                        tFlag = ((value & 0x00000020) != 0);
                        fFlag = ((value & 0x00000040) != 0);
                        iFlag = ((value & 0x00000080) != 0);
                        vFlag = ((value & 0x10000000) != 0);
                        cFlag = ((value & 0x20000000) != 0);
                        zFlag = ((value & 0x40000000) != 0);
                        nFlag = ((value & 0x80000000) != 0);
                    }
                    if (tFlag) {
                        registers[15] &= 0xFFFFFFFE;
                        pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                        registers[15] += 2;
                        pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                    } else {
                        registers[15] &= 0xFFFFFFFC;
                        pipelineStage1 = getWord(registers[15]);
                        registers[15] += 4;
                        pipelineStage2 = getWord(registers[15]);
                    }
                }
            }
            break;

            case 0x06: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int psrIndex, psrValue;
                if ((opcode & 0x00400000) == 0) {
                    psrIndex = 16;
                    int cpsr = 0x00000000;
                    cpsr |= mFlag;
                    cpsr = tFlag ? (cpsr | 0x00000020) : (cpsr & ~0x00000020);
                    cpsr = fFlag ? (cpsr | 0x00000040) : (cpsr & ~0x00000040);
                    cpsr = iFlag ? (cpsr | 0x00000080) : (cpsr & ~0x00000080);
                    cpsr = vFlag ? (cpsr | 0x10000000) : (cpsr & ~0x10000000);
                    cpsr = cFlag ? (cpsr | 0x20000000) : (cpsr & ~0x20000000);
                    cpsr = zFlag ? (cpsr | 0x40000000) : (cpsr & ~0x40000000);
                    cpsr = nFlag ? (cpsr | 0x80000000) : (cpsr & ~0x80000000);
                    psrValue = cpsr;
                } else {
                    if (SPSR == -1) {
                        return;
                    }
                    psrIndex = SPSR;
                    psrValue = registers[SPSR];
                }

                if ((opcode & 0x00200000) == 0) {
                    // MRS{cond} Rd, Psr

                    int rdIndex = (opcode >>> 12) & 0x0000000F;
                    registers[rdIndex] = psrValue;

                } else {
                    // MSR{cond} Psr{_field}, Op

                    int operand;
                    if ((opcode & 0x02000000) == 0) {
                        int rmIndex = opcode & 0x0000000F;
                        operand = registers[rmIndex];
                    } else {
                        operand = opcode & 0x000000FF;
                        int shiftAmount = (opcode & 0x00000F00) >>> 7;
                        if (shiftAmount != 0) {
                            operand = (operand << (32 - shiftAmount)) | (operand >>> shiftAmount);
                        }
                    }

                    //if (getMode() != 0x00000010) {
                    if (mFlag != 0x00000010) {
                        // We enter here if we are in a privileged mode
                        if ((opcode & 0x00010000) != 0) {
                            psrValue = (psrValue & 0xFFFFFF00) | (operand & 0x000000FF);
                        }
                        if ((opcode & 0x00020000) != 0) {
                            psrValue = (psrValue & 0xFFFF00FF) | (operand & 0x0000FF00);
                        }
                        if ((opcode & 0x00040000) != 0) {
                            psrValue = (psrValue & 0xFF00FFFF) | (operand & 0x00FF0000);
                        }
                    }
                    if ((opcode & 0x00080000) != 0) {
                        psrValue = (psrValue & 0x00FFFFFF) | (operand & 0xFF000000);
                    }
                    psrValue |= 0x00000010;

                    if (psrIndex == 16) {
                        registers[15] -= (tFlag ? 2 : 4);
                        int value = psrValue;
                        setMode(value & 0x0000001F);
                        tFlag = ((value & 0x00000020) != 0);
                        fFlag = ((value & 0x00000040) != 0);
                        iFlag = ((value & 0x00000080) != 0);
                        vFlag = ((value & 0x10000000) != 0);
                        cFlag = ((value & 0x20000000) != 0);
                        zFlag = ((value & 0x40000000) != 0);
                        nFlag = ((value & 0x80000000) != 0);
                        if (tFlag) {
                            registers[15] &= 0xFFFFFFFE;
                            pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                            registers[15] += 2;
                            pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                        } else {
                            registers[15] &= 0xFFFFFFFC;
                            pipelineStage1 = getWord(registers[15]);
                            registers[15] += 4;
                            pipelineStage2 = getWord(registers[15]);
                        }
                    } else {
                        registers[psrIndex] = psrValue;
                    }

                }
            }
            break;

            case 0x07: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int rmIndex = opcode & 0x0000000F;
                int rmValue = registers[rmIndex];
                int rsIndex = (opcode >>> 8) & 0x0000000F;
                int rsValue = registers[rsIndex];
                int rdIndex = (opcode >>> 16) & 0x0000000F;
                int rdValue;

                if ((opcode & 0x00200000) == 0) {
                    rdValue = (rmValue * rsValue);
                } else {
                    int rnIndex = (opcode >>> 12) & 0x0000000F;
                    int rnValue = registers[rnIndex];
                    rdValue = (rmValue * rsValue) + rnValue;
                }

                registers[rdIndex] = rdValue;

                if ((opcode & 0x00100000) != 0) {
                    zFlag = (rdValue == 0);
                    nFlag = (rdValue < 0);
                }
            }
            break;

            case 0x08: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int rdloIndex = (opcode >>> 12) & 0x0000000F;
                int rdloValue;
                int rdhiIndex = (opcode >>> 16) & 0x0000000F;
                int rdhiValue;
                int rmValue = registers[opcode & 0x0000000F];
                int rsValue = registers[(opcode >>> 8) & 0x0000000F];
                long rdValue;

                switch (opcode & 0x00600000) {
                    case 0x00000000: // UMULL{cond}{S} RdLo, RdHi, Rm, Rs
                        rdValue = ((long) rmValue & 0xFFFFFFFFL) * ((long) rsValue & 0xFFFFFFFFL);
                        break;

                    case 0x00200000: // UMLAL{cond}{S} RdLo, RdHi, Rm, Rs
                        rdloValue = registers[rdloIndex];
                        rdhiValue = registers[rdhiIndex];
                        rdValue = ((long) rmValue & 0xFFFFFFFFL) * ((long) rsValue & 0xFFFFFFFFL);
                        rdValue += ((long) rdhiValue << 32) | ((long) rdloValue & 0xFFFFFFFFL);
                        break;

                    case 0x00400000: // SMULL{cond}{S} RdLo, RdHi, Rm, Rs
                        rdValue = rmValue * rsValue;
                        break;

                    case 0x00600000: // SMLAL{cond}{S} RdLo, RdHi, Rm, Rs
                        rdloValue = registers[rdloIndex];
                        rdhiValue = registers[rdhiIndex];
                        rdValue = rmValue * rsValue;
                        rdValue += ((long) rdhiValue << 32) | ((long) rdloValue & 0xFFFFFFFFL);
                        break;

                    default:
                        rdValue = 0;
                        break;
                }

                registers[rdloIndex] = (int) rdValue;
                registers[rdhiIndex] = (int) (rdValue >>> 32);

                if ((opcode & 0x00100000) != 0) {
                    zFlag = (rdValue == 0);
                    nFlag = (rdValue < 0);
                }
            }
            break;

            case 0x09: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int rdIndex = (opcode >>> 12) & 0x0000000F;
                int rnIndex = (opcode >>> 16) & 0x0000000F;
                int rnValue = registers[rnIndex];

                int offset;
                if ((opcode & 0x02000000) == 0) {
                    offset = opcode & 0x00000FFF;
                } else {
                    int rmIndex = opcode & 0x0000000F;
                    int rmValue = registers[rmIndex];
                    int shiftType = opcode & 0x00000060;
                    int shiftAmount = (opcode >>> 7) & 0x0000001F;

                    offset = rmValue;

                    switch (shiftType) {
                        case 0x00000000: // LSL
                            if (shiftAmount != 0) {
                                offset <<= shiftAmount;
                            }
                            break;

                        case 0x00000020: // LSR
                            if (shiftAmount != 0) {
                                offset >>>= shiftAmount;
                            } else {
                                offset = 0;
                            }
                            break;

                        case 0x00000040: // ASR
                            if (shiftAmount != 0) {
                                offset >>= shiftAmount;
                            } else {
                                offset >>= 31;
                            }
                            break;

                        case 0x00000060: // ROR
                            if (shiftAmount != 0) {
                                offset = (offset << (32 - shiftAmount)) | (offset >>> shiftAmount);
                            } else {
                                offset >>>= 1;
                                if (cFlag) {
                                    offset |= 0x80000000;
                                }
                            }
                            break;
                    }
                }

                boolean isPostIndexing = ((opcode & 0x01000000) == 0);
                boolean isDown = ((opcode & 0x00800000) == 0);
                boolean isWordTransfer = ((opcode & 0x00400000) == 0);
                boolean isWriteBack = isPostIndexing || ((opcode & 0x00200000) != 0);

                if (isDown) {
                    offset = -offset;
                }
                if (!isPostIndexing) {
                    rnValue += offset;
                }

                if ((opcode & 0x00100000) == 0) { // STR{cond}{B}{T} Rd, <Address>
                    int rdValue = registers[rdIndex];
                    if (rdIndex == 15) {
                        rdValue += 4;
                    }
                    if (isWordTransfer) {
                        storeWord(rnValue, rdValue);
                    } else {
                        storeByte(rnValue, (byte) rdValue);
                    }
                } else { // LDR{cond}{B}{T} Rd, <Address>
                    if (isWordTransfer) {
                        registers[rdIndex] = loadWord(rnValue);
                    } else {
                        registers[rdIndex] = loadByte(rnValue) & 0x000000FF;
                    }

                    if (rdIndex == 15) {
                        registers[15] &= 0xFFFFFFFC;
                        pipelineStage1 = getWord(registers[15]);
                        registers[15] += 4;
                        pipelineStage2 = getWord(registers[15]);
                    }
                    if (rdIndex == rnIndex) {
                        return; // Don't update Rn if it's equal to Rd
                    }
                }

                if (isWriteBack) { // Update Rn
                    if (isPostIndexing) {
                        rnValue += offset;
                    }
                    registers[rnIndex] = rnValue;
                    if (rnIndex == 15) {
                        registers[15] &= 0xFFFFFFFC;
                        pipelineStage1 = getWord(registers[15]);
                        registers[15] += 4;
                        pipelineStage2 = getWord(registers[15]);
                    }
                }
            }
            break;

            case 0x0A: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int rdIndex = (opcode >>> 12) & 0x0000000F;
                int rnIndex = (opcode >>> 16) & 0x0000000F;
                int rnValue = registers[rnIndex];

                int offset;
                if ((opcode & 0x00400000) == 0) {
                    int rmIndex = opcode & 0x0000000F;
                    int rmValue = registers[rmIndex];
                    offset = rmValue;
                } else {
                    offset = ((opcode & 0x00000F00) >>> 4) | (opcode & 0x0000000F);
                }

                boolean isPostIndexing = ((opcode & 0x01000000) == 0);
                boolean isDown = ((opcode & 0x00800000) == 0);
                boolean isWriteBack = isPostIndexing || ((opcode & 0x00200000) != 0);

                if (isDown) {
                    offset = -offset;
                }
                if (!isPostIndexing) {
                    rnValue += offset;
                }

                int Instruction = (opcode & 0x00000060);

                if ((opcode & 0x00100000) == 0) { // Store to memory
                    if (Instruction == 0x00000020) { // STR{cond}H  Rd, <Address>
                        int rdValue = registers[rdIndex];
                        if (rdIndex == 15) {
                            rdValue += 4;
                        }
                        storeHalfWord(rnValue, (short) rdValue);
                    }
                } else { // Load from memory
                    switch (Instruction) {
                        case 0x00000020: // LDR{cond}H  Rd, <Address>
                            registers[rdIndex] = loadHalfWord(rnValue) & 0x0000FFFF;
                            break;
                        case 0x00000040: // LDR{cond}SB Rd, <Address>
                            registers[rdIndex] = loadByte(rnValue);
                            break;
                        case 0x00000060: // LDR{cond}SH Rd, <Address>
                            registers[rdIndex] = loadHalfWord(rnValue);
                            break;
                        default:
                            return;
                    }

                    if (rdIndex == 15) {
                        registers[15] &= 0xFFFFFFFC;
                        pipelineStage1 = getWord(registers[15]);
                        registers[15] += 4;
                        pipelineStage2 = getWord(registers[15]);
                    }
                    if (rdIndex == rnIndex) {
                        return;
                    }
                }

                if (isWriteBack) {
                    if (isPostIndexing) {
                        rnValue += offset;
                    }
                    registers[rnIndex] = rnValue;
                    if (rnIndex == 15) {
                        registers[15] &= 0xFFFFFFFC;
                        pipelineStage1 = getWord(registers[15]);
                        registers[15] += 4;
                        pipelineStage2 = getWord(registers[15]);
                    }
                }
            }
            break;

            case 0x0B: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int rnIndex = (opcode >>> 16) & 0x0000000F;
                int rnValue = registers[rnIndex];

                int nbRegisters = 0;
                for (int i = 0; i <= 15; i++) {
                    if ((opcode & (1 << i)) != 0) {
                        nbRegisters++;
                    }
                }

                int stackAddress = (rnValue & 0xFFFFFFFC);
                int finalAddress;
                if ((opcode & 0x00800000) != 0) { // Up
                    finalAddress = stackAddress + (nbRegisters << 2);
                    if ((opcode & 0x01000000) != 0) { // Increment Before
                        stackAddress += 4;
                    }
                } else { // Down
                    finalAddress = stackAddress - (nbRegisters << 2);
                    stackAddress = finalAddress;
                    if ((opcode & 0x01000000) == 0) { // Decrement After
                        stackAddress += 4;
                    }
                }

                boolean isPCBitSet = ((opcode & 0x00008000) != 0);
                boolean isPSRBitSet = ((opcode & 0x00400000) != 0);

                if ((opcode & 0x00100000) == 0) { // STM
                    int mode = (isPSRBitSet) ? 0x00000010 : mFlag;
                    //if (mode == 0x00000010) System.out.println("Unsupported opcode (STM in User mode)");

                    int i = 0;
                    while (i < 15) {
                        if ((opcode & (1 << i)) != 0) {
                            storeWord(stackAddress, registers[i]); // TODO: USR mode
                            stackAddress += 4;
                            break;
                        }
                        i++;
                    }

                    if ((opcode & 0x00200000) != 0) { // Write-back
                        registers[rnIndex] = (rnValue & 0x00000003) | finalAddress;
                    }

                    for (i++; i < 15; i++) {
                        if ((opcode & (1 << i)) != 0) {
                            storeWord(stackAddress, registers[i]); // TODO: USR mode
                            stackAddress += 4;
                        }
                    }

                    if (isPCBitSet) {
                        storeWord(stackAddress, registers[15] + 4);
                    }

                } else { // LDM
                    int mode = (isPSRBitSet && !isPCBitSet) ? 0x00000010 : mFlag;
                    //if (mode == 0x00000010) System.out.println("Unsupported opcode (LDM in User mode)");

                    if ((opcode & 0x00200000) != 0) { // Write-back
                        registers[rnIndex] = (rnValue & 0x00000003) | finalAddress;
                    }

                    for (int i = 0; i < 15; i++) {
                        if ((opcode & (1 << i)) != 0) {
                            registers[i] = loadWord(stackAddress); // TODO: USR mode
                            stackAddress += 4;
                        }
                    }

                    if (isPCBitSet) {
                        registers[15] = loadWord(stackAddress);
                        if (isPSRBitSet) {
                            int value = registers[SPSR];
                            setMode(value & 0x0000001F);
                            tFlag = ((value & 0x00000020) != 0);
                            fFlag = ((value & 0x00000040) != 0);
                            iFlag = ((value & 0x00000080) != 0);
                            vFlag = ((value & 0x10000000) != 0);
                            cFlag = ((value & 0x20000000) != 0);
                            zFlag = ((value & 0x40000000) != 0);
                            nFlag = ((value & 0x80000000) != 0);
                        }
                        if (tFlag) {
                            registers[15] &= 0xFFFFFFFE;
                            pipelineStage1 = getHalfWord(registers[15]) & 0xFFFF;
                            registers[15] += 2;
                            pipelineStage2 = getHalfWord(registers[15]) & 0xFFFF;
                        } else {
                            registers[15] &= 0xFFFFFFFC;
                            pipelineStage1 = getWord(registers[15]);
                            registers[15] += 4;
                            pipelineStage2 = getWord(registers[15]);
                        }
                    }

                }
            }
            break;

            case 0x0C: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                int rnIndex = (opcode >>> 16) & 0x0000000F;
                int rnValue = registers[rnIndex];
                int rmIndex = opcode & 0x0000000F;
                int rmValue = registers[rmIndex];
                int rdIndex = (opcode >>> 12) & 0x0000000F;

                int value = loadWord(rnValue);
                if ((opcode & 0x00400000) == 0) { // Swap word quantity
                    registers[rdIndex] = value;
                    storeWord(rnValue, rmValue);
                } else { // Swap byte quantity
                    registers[rdIndex] = value & 0x000000FF;
                    storeByte(rnValue, (byte) rmValue);
                }
            }
            break;

            case 0x0D: {
                if (!isPreconditionSatisfied(opcode)) {
                    return;
                }

                generateInterrupt(0x00000013, 0x00000008, registers[15] - (tFlag ? 2 : 4));
            }
            break;

            case 0x11: {
                generateInterrupt(0x0000001B, 0x00000004, registers[15] - (tFlag ? 2 : 4));
            }
            break;
            /*
            case 0x12:
            {
            }
            break;
             */
        }
    }
    // ----- D�codage des instructions -----
    private static final byte thumbInstruction[] = {
        /*	    0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F */
        /* 0x0 */1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        /* 0x1 */ 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2,
        /* 0x2 */ 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        /* 0x3 */ 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
        /* 0x4 */ 4, 4, 4, 4, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6,
        /* 0x5 */ 7, 7, 8, 8, 7, 7, 8, 8, 7, 7, 8, 8, 7, 7, 8, 8,
        /* 0x6 */ 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
        /* 0x7 */ 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9,
        /* 0x8 */ 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
        /* 0x9 */ 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
        /* 0xA */ 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
        /* 0xB */ 13, 127, 127, 127, 14, 14, 127, 127, 127, 127, 127, 127, 14, 14, 127, 127,
        /* 0xC */ 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
        /* 0xD */ 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 17,
        /* 0xE */ 18, 18, 18, 18, 18, 18, 18, 18, 127, 127, 127, 127, 127, 127, 127, 127,
        /* 0xF */ 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19};
    private static byte[] armInstruction;

    private static final void initARM() {
        armInstruction = new byte[0x10000];
        byte instruction;

        for (int i = 0; i < armInstruction.length; i++) {
            int opcode = ((i & 0x0000FF00) << 12) | ((i & 0x000000FF) << 4); // Se baser sur les bits (20-27) et (4-11)

            switch ((opcode >>> 25) & 0x00000007) {
                case 0x00:
                    if ((opcode & 0x0FC000F0) == 0x00000090) {
                        instruction = 0x07;
                        break;
                    } else if ((opcode & 0x0F8000F0) == 0x00800090) {
                        instruction = 0x08;
                        break;
                    } else if ((opcode & 0x0F0000F0) == 0x01000090) {
                        instruction = 0x0C;
                        break;
                    } else if ((opcode & 0x0E000090) == 0x00000090) {
                        instruction = 0x0A;
                        break;
                    } else if ((opcode & 0x0FF00FF0) == 0x01200F10) {
                        instruction = 0x03;
                        break;
                    }

                case 0x01:
                    if (((opcode & 0x0FB00FF0) == 0x01000000) || // MRS
                            ((opcode & 0x0FB00FF0) == 0x01200000) || // MSR (I=0)
                            ((opcode & 0x0FB00000) == 0x03200000)) { // MSR (I=1)
                        instruction = 0x06;
                    } else {
                        instruction = 0x05;
                    }
                    break;

                case 0x02:
                case 0x03:
                    if ((opcode & 0x0E000010) == 0x06000010) {
                        instruction = 0x11;
                    } else {
                        instruction = 0x09;
                    }
                    break;

                case 0x04:
                    instruction = 0x0B;
                    break;

                case 0x05:
                    instruction = 0x04;
                    break;

                case 0x06:
                case 0x07:
                    if (((opcode & 0x0E000000) == 0x0C000000)
                            || ((opcode & 0x0F000000) == 0x0E000000)) {
                        instruction = 0x12;
                    } else {
                        instruction = 0x0D;
                    }
                    break;

                default:
                    instruction = 0x11;
                    break;
            }

            armInstruction[i] = instruction;
        }
    }

    private final boolean isPreconditionSatisfied(int opcode) {
        boolean condition;

        switch (opcode & 0xF0000000) {
            case 0x00000000: // EQ
                condition = zFlag;
                break;
            case 0x10000000: // NE
                condition = !zFlag;
                break;
            case 0x20000000: // CS
                condition = cFlag;
                break;
            case 0x30000000: // CC
                condition = !cFlag;
                break;
            case 0x40000000: // MI
                condition = nFlag;
                break;
            case 0x50000000: // PL
                condition = !nFlag;
                break;
            case 0x60000000: // VS
                condition = vFlag;
                break;
            case 0x70000000: // VC
                condition = !vFlag;
                break;
            case 0x80000000: // HI
                condition = (cFlag && !zFlag);
                break;
            case 0x90000000: // LS
                condition = (!cFlag || zFlag);
                break;
            case 0xA0000000: // GE
                condition = (nFlag == vFlag);
                break;
            case 0xB0000000: // LT
                condition = (nFlag != vFlag);
                break;
            case 0xC0000000: // GT
                condition = (!zFlag && (nFlag == vFlag));
                break;
            case 0xD0000000: // LE
                condition = (zFlag || (nFlag != vFlag));
                break;
            case 0xE0000000: // AL
                condition = true;
                break;
            default:
                condition = false;
        }

        return condition;
    }

    private byte[] sysrom = new byte[0x00004000];
    private byte[] ewram = new byte[0x00040000];
    private byte[] iwram = new byte[0x00008000];
    private byte[] ioram = new byte[0x00000400];
    private byte[] palram = new byte[0x00000400];
    private byte[] vram = new byte[0x00018000];
    private byte[] oam = new byte[0x00000400];
    private byte[] rom;
    private byte[] sram = new byte[0x00010000];
    private int romMask;
    private DMA dma0, dma1, dma2, dma3;
    private Timer timer0, timer1, timer2, timer3;

    private void Memory_reset() {
        /*for (int i = 0; i < ewram.length; i++) {
            ewram[i] = 0;
        }
        for (int i = 0; i < iwram.length; i++) {
            iwram[i] = 0;
        }
        for (int i = 0; i < ioram.length; i++) {
            ioram[i] = 0;
        }
        for (int i = 0; i < palram.length; i++) {
            palram[i] = 0;
        }
        for (int i = 0; i < vram.length; i++) {
            vram[i] = 0;
        }
        for (int i = 0; i < oam.length; i++) {
            oam[i] = 0;
        }
        for (int i = 0; i < sram.length; i++) {
            sram[i] = 0;
        }*/

        writeIO16(0x0000, (short) 0x0080);
        writeIO16(0x0020, (short) 0x0100);
        writeIO16(0x0026, (short) 0x0100);
        writeIO16(0x0030, (short) 0x0100);
        writeIO16(0x0036, (short) 0x0100);
        writeIO16(0x130, (short) 0x03FF);
    }

    private void connectToDMA(DirectMemoryAccess dma) {
        this.dma0 = dma.getDMA(0);
        this.dma1 = dma.getDMA(1);
        this.dma2 = dma.getDMA(2);
        this.dma3 = dma.getDMA(3);
    }

    private void connectToTime(Time time) {
        timer0 = time.getTimer(0);
        timer1 = time.getTimer(1);
        timer2 = time.getTimer(2);
        timer3 = time.getTimer(3);
    }

    private static short getValue16(boolean isOffsetAligned, short oldValue, byte newValue) {
        return (short) (isOffsetAligned ? ((oldValue & 0xFF00) | (newValue & 0x00FF))
                : ((oldValue & 0x00FF) | (newValue << 8)));
    }

    public void generateInterrupt(int interruptBit) {
        writeIO16(0x0202, (short) (readIO16(0x0202) | (readIO16(0x0200) & interruptBit)));
    }

    private void enterHBlank() {
        int scanline = ioram[0x0006] & 0x000000FF;

        // Draw the line
        //if (draw) drawLine(scanline);

        // Enter HBlank
        dma0.signalHBlank();
        dma1.signalHBlank();
        dma2.signalHBlank();
        dma3.signalHBlank();
        writeIO16(0x0202, (short) (readIO16(0x0202) | (readIO16(0x0200) & 0x0002)));
        byte val = ioram[0x0004];
        val |= 0x02;
        ioram[0x0004] = val;

        // Handle V-Counter Match interrupt
        if (((ioram[0x0004] & 0x20) != 0) && (scanline == (ioram[0x0004 + 1] & 0x000000FF))) {
            writeIO16(0x0202, (short) (readIO16(0x0202) | (readIO16(0x0200) & 0x0004)));
        }
    }

    private int getXSize(int objNumber) {
        int objAttributesAddress = (objNumber << 3);
        switch (((oam[objAttributesAddress + 1] & 0x000000C0) >>> 6)
                | ((oam[objAttributesAddress + 3] & 0x000000C0) >>> 4)) {
            case 0:
            case 2:
            case 6:
                return 8;
            case 1:
            case 4:
            case 10:
                return 16;
            case 5:
            case 8:
            case 9:
            case 14:
                return 32;
            case 12:
            case 13:
                return 64;
            default:
                return 0;
        }
    }

    private int getYSize(int objNumber) {
        int objAttributesAddress = (objNumber << 3);
        switch (((oam[objAttributesAddress + 1] & 0x000000C0) >>> 6)
                | ((oam[objAttributesAddress + 3] & 0x000000C0) >>> 4)) {
            case 0:
            case 1:
            case 5:
                return 8;
            case 2:
            case 4:
            case 9:
                return 16;
            case 6:
            case 8:
            case 10:
            case 13:
                return 32;
            case 12:
            case 14:
                return 64;
            default:
                return 0;
        }
    }

    private short getHalfWord(int adr) {
        switch ((adr >>> 24) & 0xF) {
            case 0x0:
                adr &= 0x00003FFF;
                return (short) ((sysrom[adr] & 0x00FF) | (sysrom[adr + 1] << 8));
            case 0x1:
            case 0xE:
            case 0xF:
                return 0;
            case 0x2:
                adr &= 0x0003FFFF;
                return (short) ((ewram[adr] & 0x00FF) | (ewram[adr + 1] << 8));
            case 0x3:
                adr &= 0x00007FFF;
                return (short) ((iwram[adr] & 0x00FF) | (iwram[adr + 1] << 8));
            case 0x4:
                adr &= 0x000003FF;
                return (short) ((ioram[adr] & 0x00FF) | (ioram[adr + 1] << 8));
            case 0x5:
                adr &= 0x000003FF;
                return (short) ((palram[adr] & 0x00FF) | (palram[adr + 1] << 8));
            case 0x6:
                adr = (adr & 0x00FFFFFF) % vram.length;
                return (short) ((vram[adr] & 0x00FF) | (vram[adr + 1] << 8));
            case 0x7:
                adr &= 0x000003FF;
                return (short) ((oam[adr] & 0x00FF) | (oam[adr + 1] << 8));
            default:
                adr &= romMask;
                return (short) ((rom[adr] & 0x00FF) | (rom[adr + 1] << 8));
        }
    }

    private int getWord(int adr) {
        switch ((adr >>> 24) & 0xF) {
            case 0x0:
                adr &= 0x00003FFF;
                return (((sysrom[adr] & 0x000000FF)) | ((sysrom[adr + 1] & 0x000000FF) << 8)
                        | ((sysrom[adr + 2] & 0x000000FF) << 16) | ((sysrom[adr + 3]) << 24));
            case 0x1:
            case 0xE:
            case 0xF:
                return 0;
            case 0x2:
                adr &= 0x0003FFFF;
                return (((ewram[adr] & 0x000000FF)) | ((ewram[adr + 1] & 0x000000FF) << 8)
                        | ((ewram[adr + 2] & 0x000000FF) << 16) | ((ewram[adr + 3]) << 24));
            case 0x3:
                adr &= 0x00007FFF;
                return (((iwram[adr] & 0x000000FF)) | ((iwram[adr + 1] & 0x000000FF) << 8)
                        | ((iwram[adr + 2] & 0x000000FF) << 16) | ((iwram[adr + 3]) << 24));
            case 0x4:
                adr &= 0x000003FF;
                return (((ioram[adr] & 0x000000FF)) | ((ioram[adr + 1] & 0x000000FF) << 8)
                        | ((ioram[adr + 2] & 0x000000FF) << 16) | ((ioram[adr + 3]) << 24));
            case 0x5:
                adr &= 0x000003FF;
                return (((palram[adr] & 0x000000FF)) | ((palram[adr + 1] & 0x000000FF) << 8)
                        | ((palram[adr + 2] & 0x000000FF) << 16) | ((palram[adr + 3]) << 24));
            case 0x6:
                adr = (adr & 0x00FFFFFF) % vram.length;
                return (((vram[adr] & 0x000000FF)) | ((vram[adr + 1] & 0x000000FF) << 8)
                        | ((vram[adr + 2] & 0x000000FF) << 16) | ((vram[adr + 3]) << 24));
            case 0x7:
                adr &= 0x000003FF;
                return (((oam[adr] & 0x000000FF)) | ((oam[adr + 1] & 0x000000FF) << 8)
                        | ((oam[adr + 2] & 0x000000FF) << 16) | ((oam[adr + 3]) << 24));
            default:
                adr &= romMask;
                return (((rom[adr] & 0x000000FF)) | ((rom[adr + 1] & 0x000000FF) << 8)
                        | ((rom[adr + 2] & 0x000000FF) << 16) | ((rom[adr + 3]) << 24));
        }
    }

    private void setWord(int adr, int val) {
        switch ((adr >>> 24) & 0xF) {
            case 0x2:
                adr &= 0x0003FFFF;
                ewram[adr] = (byte) val;
                ewram[adr + 1] = (byte) (val >>> 8);
                ewram[adr + 2] = (byte) (val >>> 16);
                ewram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x3:
                adr &= 0x00007FFF;
                iwram[adr] = (byte) val;
                iwram[adr + 1] = (byte) (val >>> 8);
                iwram[adr + 2] = (byte) (val >>> 16);
                iwram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x4:
                adr &= 0x000003FF;
                ioram[adr] = (byte) val;
                ioram[adr + 1] = (byte) (val >>> 8);
                ioram[adr + 2] = (byte) (val >>> 16);
                ioram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x5:
                adr &= 0x000003FF;
                palram[adr] = (byte) val;
                palram[adr + 1] = (byte) (val >>> 8);
                palram[adr + 2] = (byte) (val >>> 16);
                palram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x6:
                adr = (adr & 0x00FFFFFF) % vram.length;
                vram[adr] = (byte) val;
                vram[adr + 1] = (byte) (val >>> 8);
                vram[adr + 2] = (byte) (val >>> 16);
                vram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x7:
                adr &= 0x000003FF;
                oam[adr] = (byte) val;
                oam[adr + 1] = (byte) (val >>> 8);
                oam[adr + 2] = (byte) (val >>> 16);
                oam[adr + 3] = (byte) (val >>> 24);
                break;
        }
    }

    private byte loadByte(int adr) {
        switch ((adr >>> 24) & 0xF) {
            case 0x0:
                return sysrom[adr & 0x00003FFF];
            case 0x1:
            case 0xF:
                return 0;
            case 0x2:
                return ewram[adr & 0x0003FFFF];
            case 0x3:
                return iwram[adr & 0x00007FFF];
            case 0x4:
                return in(adr);
            case 0x5:
                return palram[adr & 0x000003FF];
            case 0x6:
                return vram[(adr & 0x00FFFFFF) % vram.length];
            case 0x7:
                return oam[adr & 0x000003FF];
            case 0xE:
                return sram[adr & 0x0000FFFF];
            default:
                return rom[adr & romMask];
        }
    }

    public short loadHalfWord(int adr) {
        switch ((adr >>> 24) & 0xF) {
            case 0x0:
                adr &= 0x00003FFF; // & 0xFFFFFFFE;
                return (short) ((sysrom[adr] & 0x00FF) | (sysrom[adr + 1] << 8));
            case 0x1:
            case 0xE:
            case 0xF:
                return 0;
            case 0x2:
                adr &= 0x0003FFFF; // & 0xFFFFFFFE;
                return (short) ((ewram[adr] & 0x00FF) | (ewram[adr + 1] << 8));
            case 0x3:
                adr &= 0x00007FFF; // & 0xFFFFFFFE;
                return (short) ((iwram[adr] & 0x00FF) | (iwram[adr + 1] << 8));
            case 0x4:
                adr &= 0xFFFFFFFE;
                return (short) ((in(adr) & 0x00FF) | (in(adr + 1) << 8));
            case 0x5:
                adr &= 0x000003FF; // & 0xFFFFFFFE;
                return (short) ((palram[adr] & 0x00FF) | (palram[adr + 1] << 8));
            case 0x6:
                adr = (adr & 0x00FFFFFF) % vram.length; // & 0xFFFFFFFE;
                return (short) ((vram[adr] & 0x00FF) | (vram[adr + 1] << 8));
            case 0x7:
                adr &= 0x000003FF; // & 0xFFFFFFFE;
                return (short) ((oam[adr] & 0x00FF) | (oam[adr + 1] << 8));
            default:
                adr &= romMask; // & 0xFFFFFFFE;
                return (short) ((rom[adr] & 0x00FF) | (rom[adr + 1] << 8));
        }
    }

    public int loadWord(int adr) {
        switch ((adr >>> 24) & 0xF) {
            case 0x0:
                adr &= 0x00003FFF; // & 0xFFFFFFFC;
                return (((sysrom[adr] & 0x000000FF)) | ((sysrom[adr + 1] & 0x000000FF) << 8)
                        | ((sysrom[adr + 2] & 0x000000FF) << 16) | ((sysrom[adr + 3]) << 24));
            case 0x1:
            case 0xE:
            case 0xF:
                return 0;
            case 0x2:
                adr &= 0x0003FFFF; // & 0xFFFFFFFC;
                return (((ewram[adr] & 0x000000FF)) | ((ewram[adr + 1] & 0x000000FF) << 8)
                        | ((ewram[adr + 2] & 0x000000FF) << 16) | ((ewram[adr + 3]) << 24));
            case 0x3:
                adr &= 0x00007FFF; // & 0xFFFFFFFC;
                return (((iwram[adr] & 0x000000FF)) | ((iwram[adr + 1] & 0x000000FF) << 8)
                        | ((iwram[adr + 2] & 0x000000FF) << 16) | ((iwram[adr + 3]) << 24));
            case 0x4:
                adr &= 0xFFFFFFFC;
                return (((in(adr) & 0x000000FF)) | ((in(adr + 1) & 0x000000FF) << 8)
                        | ((in(adr + 2) & 0x000000FF) << 16) | ((in(adr + 3)) << 24));
            case 0x5:
                adr &= 0x000003FF; // & 0xFFFFFFFC;
                return (((palram[adr] & 0x000000FF)) | ((palram[adr + 1] & 0x000000FF) << 8)
                        | ((palram[adr + 2] & 0x000000FF) << 16) | ((palram[adr + 3]) << 24));
            case 0x6:
                adr = (adr & 0x00FFFFFF) % vram.length; // & 0xFFFFFFFC;
                return (((vram[adr] & 0x000000FF)) | ((vram[adr + 1] & 0x000000FF) << 8)
                        | ((vram[adr + 2] & 0x000000FF) << 16) | ((vram[adr + 3]) << 24));
            case 0x7:
                adr &= 0x000003FF; // & 0xFFFFFFFC;
                return (((oam[adr] & 0x000000FF)) | ((oam[adr + 1] & 0x000000FF) << 8)
                        | ((oam[adr + 2] & 0x000000FF) << 16) | ((oam[adr + 3]) << 24));
            default:
                adr &= romMask; // & 0xFFFFFFFC;
                return (((rom[adr] & 0x000000FF)) | ((rom[adr + 1] & 0x000000FF) << 8)
                        | ((rom[adr + 2] & 0x000000FF) << 16) | ((rom[adr + 3]) << 24));
        }
    }

    private void storeByte(int adr, byte val) {
        switch ((adr >>> 24) & 0xF) {
            case 0x2:
                ewram[adr & 0x0003FFFF] = val;
                break;
            case 0x3:
                iwram[adr & 0x00007FFF] = val;
                break;
            case 0x4:
                out(adr, val);
                break;
            case 0xE:
                sram[adr & 0x0000FFFF] = val;
                break;
        }
    }

    public void storeHalfWord(int adr, short val) {
        switch ((adr >>> 24) & 0xF) {
            case 0x2:
                adr &= 0x0003FFFF; // & 0xFFFFFFFE;
                ewram[adr] = (byte) val;
                ewram[adr + 1] = (byte) (val >>> 8);
                break;
            case 0x3:
                adr &= 0x00007FFF; // & 0xFFFFFFFE;
                iwram[adr] = (byte) val;
                iwram[adr + 1] = (byte) (val >>> 8);
                break;
            case 0x4:
                adr &= 0xFFFFFFFE;
                out(adr, (byte) val);
                out(adr + 1, (byte) (val >>> 8));
                break;
            case 0x5:
                adr &= 0x000003FF; // & 0xFFFFFFFE;
                palram[adr] = (byte) val;
                palram[adr + 1] = (byte) (val >>> 8);
                break;
            case 0x6:
                adr = (adr & 0x00FFFFFF) % vram.length; // & 0xFFFFFFFE;
                vram[adr] = (byte) val;
                vram[adr + 1] = (byte) (val >>> 8);
                break;
            case 0x7:
                adr &= 0x000003FF; // & 0xFFFFFFFE;
                oam[adr] = (byte) val;
                oam[adr + 1] = (byte) (val >>> 8);
                break;
        }
    }

    public void storeWord(int adr, int val) {
        switch ((adr >>> 24) & 0xF) {
            case 0x2:
                adr &= 0x0003FFFF; // & 0xFFFFFFFC;
                ewram[adr] = (byte) val;
                ewram[adr + 1] = (byte) (val >>> 8);
                ewram[adr + 2] = (byte) (val >>> 16);
                ewram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x3:
                adr &= 0x00007FFF; // & 0xFFFFFFFC;
                iwram[adr] = (byte) val;
                iwram[adr + 1] = (byte) (val >>> 8);
                iwram[adr + 2] = (byte) (val >>> 16);
                iwram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x4:
                adr &= 0xFFFFFFFC;
                out(adr, (byte) val);
                out(adr + 1, (byte) (val >>> 8));
                out(adr + 2, (byte) (val >>> 16));
                out(adr + 3, (byte) (val >>> 24));
                break;
            case 0x5:
                adr &= 0x000003FF; // & 0xFFFFFFFC;
                palram[adr] = (byte) val;
                palram[adr + 1] = (byte) (val >>> 8);
                palram[adr + 2] = (byte) (val >>> 16);
                palram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x6:
                adr = (adr & 0x00FFFFFF) % vram.length; // & 0xFFFFFFFC;
                vram[adr] = (byte) val;
                vram[adr + 1] = (byte) (val >>> 8);
                vram[adr + 2] = (byte) (val >>> 16);
                vram[adr + 3] = (byte) (val >>> 24);
                break;
            case 0x7:
                adr &= 0x000003FF; // & 0xFFFFFFFC;
                oam[adr] = (byte) val;
                oam[adr + 1] = (byte) (val >>> 8);
                oam[adr + 2] = (byte) (val >>> 16);
                oam[adr + 3] = (byte) (val >>> 24);
                break;
        }
    }

    private byte in(int adr) {
        adr &= 0x000003FF;
        int adr16 = adr & 0xFFFFFFFE; // Halfword aligned offset

        switch (adr16) {
            // DMA0
            case 0x00B0:
                writeIO16(adr16, dma0.getSourceLRegister());
                break;
            case 0x00B0 + 2:
                writeIO16(adr16, dma0.getSourceHRegister());
                break;
            case 0x00B4:
                writeIO16(adr16, dma0.getDestinationLRegister());
                break;
            case 0x00B4 + 2:
                writeIO16(adr16, dma0.getDestinationHRegister());
                break;
            case 0x00B8:
                writeIO16(adr16, dma0.getCountRegister());
                break;
            case 0x00BA:
                writeIO16(adr16, dma0.getControlRegister());
                break;

            // DMA1
            case 0x00BC:
                writeIO16(adr16, dma1.getSourceLRegister());
                break;
            case 0x00BC + 2:
                writeIO16(adr16, dma1.getSourceHRegister());
                break;
            case 0x00C0:
                writeIO16(adr16, dma1.getDestinationLRegister());
                break;
            case 0x00C0 + 2:
                writeIO16(adr16, dma1.getDestinationHRegister());
                break;
            case 0x00C4:
                writeIO16(adr16, dma1.getCountRegister());
                break;
            case 0x00C6:
                writeIO16(adr16, dma1.getControlRegister());
                break;

            // DMA2
            case 0x00C8:
                writeIO16(adr16, dma2.getSourceLRegister());
                break;
            case 0x00C8 + 2:
                writeIO16(adr16, dma2.getSourceHRegister());
                break;
            case 0x00CC:
                writeIO16(adr16, dma2.getDestinationLRegister());
                break;
            case 0x00CC + 2:
                writeIO16(adr16, dma2.getDestinationHRegister());
                break;
            case 0x00D0:
                writeIO16(adr16, dma2.getCountRegister());
                break;
            case 0x00D2:
                writeIO16(adr16, dma2.getControlRegister());
                break;

            // DMA3
            case 0x00D4:
                writeIO16(adr16, dma3.getSourceLRegister());
                break;
            case 0x00D4 + 2:
                writeIO16(adr16, dma3.getSourceHRegister());
                break;
            case 0x00D8:
                writeIO16(adr16, dma3.getDestinationLRegister());
                break;
            case 0x00D8 + 2:
                writeIO16(adr16, dma3.getDestinationHRegister());
                break;
            case 0x00DC:
                writeIO16(adr16, dma3.getCountRegister());
                break;
            case 0x00DE:
                writeIO16(adr16, dma3.getControlRegister());
                break;

            // Timers
            case 0x0100:
                writeIO16(adr16, timer0.getTime());
                break;
            case 0x0104:
                writeIO16(adr16, timer1.getTime());
                break;
            case 0x0108:
                writeIO16(adr16, timer2.getTime());
                break;
            case 0x010C:
                writeIO16(adr16, timer3.getTime());
                break;
        }
        return ioram[adr];
    }

    private void out(int adr, byte val) {
        adr &= 0x000003FF;
        int adr16 = adr & 0xFFFFFFFE;
        boolean isOffsetAligned = ((adr & 0x00000001) == 0);
        short val16 = getValue16(isOffsetAligned, readIO16(adr16), val);

        switch (adr16) {
            // VCOUNT
            case 0x0006:
                return;

            // DMA0
            case 0x00B0:
                dma0.setSourceLRegister(val16);
                break;
            case 0x00B0 + 2:
                dma0.setSourceHRegister(val16);
                break;
            case 0x00B4:
                dma0.setDestinationLRegister(val16);
                break;
            case 0x00B4 + 2:
                dma0.setDestinationHRegister(val16);
                break;
            case 0x00B8:
                dma0.setCountRegister(val16);
                break;
            case 0x00BA:
                dma0.setControlRegister(val16);
                break;

            // DMA1
            case 0x00BC:
                dma1.setSourceLRegister(val16);
                break;
            case 0x00BC + 2:
                dma1.setSourceHRegister(val16);
                break;
            case 0x00C0:
                dma1.setDestinationLRegister(val16);
                break;
            case 0x00C0 + 2:
                dma1.setDestinationHRegister(val16);
                break;
            case 0x00C4:
                dma1.setCountRegister(val16);
                break;
            case 0x00C6:
                dma1.setControlRegister(val16);
                break;

            // DMA2
            case 0x00C8:
                dma2.setSourceLRegister(val16);
                break;
            case 0x00C8 + 2:
                dma2.setSourceHRegister(val16);
                break;
            case 0x00CC:
                dma2.setDestinationLRegister(val16);
                break;
            case 0x00CC + 2:
                dma2.setDestinationHRegister(val16);
                break;
            case 0x00D0:
                dma2.setCountRegister(val16);
                break;
            case 0x00D2:
                dma2.setControlRegister(val16);
                break;

            // DMA3
            case 0x00D4:
                val16 = getValue16(isOffsetAligned, dma3.getSourceLRegister(), val);
                dma3.setSourceLRegister(val16);
                break;
            case 0x00D4 + 2:
                val16 = getValue16(isOffsetAligned, dma3.getSourceHRegister(), val);
                dma3.setSourceHRegister(val16);
                break;
            case 0x00D8:
                val16 = getValue16(isOffsetAligned, dma3.getDestinationLRegister(), val);
                dma3.setDestinationLRegister(val16);
                break;
            case 0x00D8 + 2:
                val16 = getValue16(isOffsetAligned, dma3.getDestinationHRegister(), val);
                dma3.setDestinationHRegister(val16);
                break;
            case 0x00DC:
                val16 = getValue16(isOffsetAligned, dma3.getCountRegister(), val);
                dma3.setCountRegister(val16);
                break;
            case 0x00DE:
                val16 = getValue16(isOffsetAligned, dma3.getControlRegister(), val);
                dma3.setControlRegister(val16);
                break;

            // Timers
            case 0x0100:
                timer0.setTime(val16);
                break;
            case 0x0102:
                timer0.updateState(val16);
                break;
            case 0x0104:
                timer1.setTime(val16);
                break;
            case 0x0106:
                timer1.updateState(val16);
                break;
            case 0x0108:
                timer2.setTime(val16);
                break;
            case 0x010A:
                timer2.updateState(val16);
                break;
            case 0x010C:
                timer3.setTime(val16);
                break;
            case 0x010E:
                timer3.updateState(val16);
                break;

            // Keypad
            case 0x130:
                return;

            // Interrupts
            case 0x0202:
                ioram[adr] &= ~val;
                return;
        }
        ioram[adr] = val;
    }

    private short readIO16(int adr) {
        return (short) ((ioram[adr] & 0x00FF) | (ioram[adr + 1] << 8));
    }

    private void writeIO16(int adr, short val) {
        ioram[adr] = (byte) val;
        ioram[adr + 1] = (byte) (val >>> 8);
    }

    private short readPal16(int adr) {
        adr &= 0x000003FF;
        return (short) ((palram[adr] & 0x00FF) | (palram[adr + 1] << 8));
    }

    private short readVRam16(int adr) {
        adr = (adr & 0x00FFFFFF) % vram.length;
        return (short) ((vram[adr] & 0x00FF) | (vram[adr + 1] << 8));
    }

    public final int loadROM(InputStream is, int size) {
        int i = 0;
        try {
            rom = new byte[size];
            is.read(rom, 0, size);
			is.close();
            /*int len = rom[0xA0];
            StringBuffer sb = new StringBuffer();
            for (int j = 0; j < len; j++) {
                if (rom[0xA0 + j] == 0x00) {
                    break;
                }
                char c = (char) rom[0xA0 + j];
                if (c < ' ') {
                    c = ' ';
                }
                sb.append(c);
            }
            rom_name = sb.toString();
            System.out.println("rom_name    = " + rom_name);*/
			for( romMask=1; romMask <= 0x1FFFFFF; romMask = (romMask<<1) | 1 )
				if( size-1<=romMask ) break;
			
			
            is = getClass().getResourceAsStream("/bios.bin");
            is.read(sysrom, 0, 0x4000);
			is.close();
        } catch (IOException e) {
            i = -1;
        } finally {
            is = null;
        }
        return i;
    }

    private boolean bg0Enabled;
    private boolean bg1Enabled;
    private boolean bg2Enabled;
    private boolean bg3Enabled;
    private boolean objEnabled;
    private boolean isMosaicEnabled;
    private int bg0Priority;
    private int bg1Priority;
    private int bg2Priority;
    private int bg3Priority;
    private int xMosaic;
    private int yMosaic;
    private int drawMode_x;
    private int drawMode_y;

    private void connectToGBACanvas(GBACanvas gui) {
        this.gui = gui;
    }

    private void gfx_reset() {
        for (int i = 0; i < pixels.length; i++) {
            pixels[i] = 0;
        }
    }

	private boolean pf;

    public void drawLine(/*int y*/) {
        //if (y < 160) {
		//pf = !pf;
		//for(int y=pf ? 1 : 0; y<160; y+=2 ) {
		for(int y=0; y<160; y++ ) {
            switch (ioram[0x0000] & 0x07) {
                case 0:
                    bg0Enabled = ((ioram[1] & 0x01) != 0);
                    bg1Enabled = ((ioram[1] & 0x02) != 0);
                    bg2Enabled = ((ioram[1] & 0x04) != 0);
                    bg3Enabled = ((ioram[1] & 0x08) != 0);
                    objEnabled = ((ioram[1] & 0x10) != 0);

                    bg0Priority = ioram[0x0008] & 0x03;
                    bg1Priority = ioram[0x000A] & 0x03;
                    bg2Priority = ioram[0x000C] & 0x03;
                    bg3Priority = ioram[0x000E] & 0x03;

                    drawBGLine(y);

                    for (int p = 3; p >= 0; p--) {
                        if (bg3Enabled && (bg3Priority == p)) {
                            int characterBase = ((ioram[0x000E] & 0x0C) << 12);
                            int screenBase = ((ioram[0x000F] & 0x1F) << 11);

                            int xSize = ((ioram[0x000F] & 0x40) == 0) ? 256 : 512;
                            int ySize = ((ioram[0x000F] & 0x80) == 0) ? 256 : 512;
                            int xMask = xSize - 1;
                            int yMask = ySize - 1;

                            int xOffset = (((short) ((ioram[0x001C] & 0x00FF) | (ioram[0x001D] << 8))) & 0x01FF);
                            int yOffset = (((short) ((ioram[0x001E] & 0x00FF) | (ioram[0x001F] << 8))) & 0x01FF);

                            boolean isTileDataUpsideDown = (xSize != 256);

                            boolean is256ColorPalette = ((ioram[0x000E] & 0x80) != 0);

                            boolean isMosaicEnabled = ((ioram[0x000E] & 0x40) != 0);
                            int xMosaic = ((ioram[0x004C] & 0x0F) + 1);
                            int yMosaic = (((ioram[0x004C] >>> 4) & 0x0F) + 1);

                            int drawBGTextMode_y = (isMosaicEnabled ? (y - (y % yMosaic)) : y);
                            drawBGTextMode_y = (drawBGTextMode_y + yOffset) & yMask;

                            int yOffsetToAdd = 0;
                            if (isTileDataUpsideDown) {
                                if (drawBGTextMode_y >= 256) {
                                    yOffsetToAdd = 0x0800 * 2;
                                }
                                drawBGTextMode_y &= 0xFF;
                            }

                            int yTileDataOffset = (drawBGTextMode_y >>> 3) * 32;

                            int tileY = drawBGTextMode_y & 0x07;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int x = (isMosaicEnabled ? (xScreen - (xScreen % xMosaic)) : xScreen);
                                x = (x + xOffset) & xMask;

                                int xOffsetToAdd = 0;
                                if (isTileDataUpsideDown) {
                                    if (x >= 256) {
                                        xOffsetToAdd = 0x0800;
                                    }
                                    x &= 0xFF;
                                }

                                int xTileDataOffset = (x >>> 3);

                                int tileX = x & 0x07;

                                int tileDataOffset = ((yTileDataOffset + xTileDataOffset) * 2) + (yOffsetToAdd + xOffsetToAdd);

                                short tileData = readVRam16(screenBase + tileDataOffset);

                                int tileNumber = tileData & 0x03FF;
                                if ((tileData & 0x0400) != 0) {
                                    tileX = 7 - tileX; // H-Flip
                                }
                                if ((tileData & 0x0800) != 0) {
                                    tileY = 7 - tileY; // V-Flip
                                }
                                if (is256ColorPalette) {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + (tileX)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) { // Not a transparent color
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                } else {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 4) + (tileY * 4) + (tileX / 2)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if ((tileX & 0x01) != 0) {
                                        colorIndex >>>= 4;
                                    } else {
                                        colorIndex &= 0x0F;
                                    }

                                    if (colorIndex != 0) {
                                        int paletteNumber = (tileData >>> 12) & 0x0F;
                                        short rgb15 = readPal16(((paletteNumber * 16) + colorIndex) * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }
                            }
                        }
                        if (bg2Enabled && (bg2Priority == p)) {
                            int characterBase = ((ioram[0x000C] & 0x0C) << 12);
                            int screenBase = ((ioram[0x000D] & 0x1F) << 11);

                            int xSize = ((ioram[0x000D] & 0x40) == 0) ? 256 : 512;
                            int ySize = ((ioram[0x000D] & 0x80) == 0) ? 256 : 512;
                            int xMask = xSize - 1;
                            int yMask = ySize - 1;

                            int xOffset = (((short) ((ioram[0x0018] & 0x00FF) | (ioram[0x0019] << 8))) & 0x01FF);
                            int yOffset = (((short) ((ioram[0x001A] & 0x00FF) | (ioram[0x001B] << 8))) & 0x01FF);

                            boolean isTileDataUpsideDown = (xSize != 256);

                            boolean is256ColorPalette = ((ioram[0x000C] & 0x80) != 0);

                            boolean isMosaicEnabled = ((ioram[0x000C] & 0x40) != 0);
                            int xMosaic = ((ioram[0x004C] & 0x0F) + 1);
                            int yMosaic = (((ioram[0x004C] >>> 4) & 0x0F) + 1);

                            int drawBGTextMode_y = (isMosaicEnabled ? (y - (y % yMosaic)) : y);
                            drawBGTextMode_y = (drawBGTextMode_y + yOffset) & yMask;

                            int yOffsetToAdd = 0;
                            if (isTileDataUpsideDown) {
                                if (drawBGTextMode_y >= 256) {
                                    yOffsetToAdd = 0x0800 * 2;
                                }
                                drawBGTextMode_y &= 0xFF;
                            }

                            int yTileDataOffset = (drawBGTextMode_y >>> 3) * 32;

                            int tileY = drawBGTextMode_y & 0x07;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int x = (isMosaicEnabled ? (xScreen - (xScreen % xMosaic)) : xScreen);
                                x = (x + xOffset) & xMask;

                                int xOffsetToAdd = 0;
                                if (isTileDataUpsideDown) {
                                    if (x >= 256) {
                                        xOffsetToAdd = 0x0800;
                                    }
                                    x &= 0xFF;
                                }

                                int xTileDataOffset = (x >>> 3);

                                int tileX = x & 0x07;

                                int tileDataOffset = ((yTileDataOffset + xTileDataOffset) * 2) + (yOffsetToAdd + xOffsetToAdd);

                                short tileData = readVRam16(screenBase + tileDataOffset);

                                int tileNumber = tileData & 0x03FF;
                                if ((tileData & 0x0400) != 0) {
                                    tileX = 7 - tileX; // H-Flip
                                }
                                if ((tileData & 0x0800) != 0) {
                                    tileY = 7 - tileY; // V-Flip
                                }
                                if (is256ColorPalette) {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + (tileX)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) { // Not a transparent color
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                } else {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 4) + (tileY * 4) + (tileX / 2)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if ((tileX & 0x01) != 0) {
                                        colorIndex >>>= 4;
                                    } else {
                                        colorIndex &= 0x0F;
                                    }

                                    if (colorIndex != 0) {
                                        int paletteNumber = (tileData >>> 12) & 0x0F;
                                        short rgb15 = readPal16(((paletteNumber * 16) + colorIndex) * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }
                            }
                        }
                        if (bg1Enabled && (bg1Priority == p)) {
                            int characterBase = ((ioram[0x000A] & 0x0C) << 12);
                            int screenBase = ((ioram[0x000B] & 0x1F) << 11);

                            int xSize = ((ioram[0x000B] & 0x40) == 0) ? 256 : 512;
                            int ySize = ((ioram[0x000B] & 0x80) == 0) ? 256 : 512;
                            int xMask = xSize - 1;
                            int yMask = ySize - 1;

                            int xOffset = (((short) ((ioram[0x0014] & 0x00FF) | (ioram[0x0015] << 8))) & 0x01FF);
                            int yOffset = (((short) ((ioram[0x0016] & 0x00FF) | (ioram[0x0017] << 8))) & 0x01FF);

                            boolean isTileDataUpsideDown = (xSize != 256);

                            boolean is256ColorPalette = ((ioram[0x000A] & 0x80) != 0);

                            boolean isMosaicEnabled = ((ioram[0x000A] & 0x40) != 0);
                            int xMosaic = ((ioram[0x004C] & 0x0F) + 1);
                            int yMosaic = (((ioram[0x004C] >>> 4) & 0x0F) + 1);

                            int drawBGTextMode_y = (isMosaicEnabled ? (y - (y % yMosaic)) : y);
                            drawBGTextMode_y = (drawBGTextMode_y + yOffset) & yMask;

                            int yOffsetToAdd = 0;
                            if (isTileDataUpsideDown) {
                                if (drawBGTextMode_y >= 256) {
                                    yOffsetToAdd = 0x0800 * 2;
                                }
                                drawBGTextMode_y &= 0xFF;
                            }

                            int yTileDataOffset = (drawBGTextMode_y >>> 3) * 32;

                            int tileY = drawBGTextMode_y & 0x07;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int x = (isMosaicEnabled ? (xScreen - (xScreen % xMosaic)) : xScreen);
                                x = (x + xOffset) & xMask;

                                int xOffsetToAdd = 0;
                                if (isTileDataUpsideDown) {
                                    if (x >= 256) {
                                        xOffsetToAdd = 0x0800;
                                    }
                                    x &= 0xFF;
                                }

                                int xTileDataOffset = (x >>> 3);

                                int tileX = x & 0x07;

                                int tileDataOffset = ((yTileDataOffset + xTileDataOffset) * 2) + (yOffsetToAdd + xOffsetToAdd);

                                short tileData = readVRam16(screenBase + tileDataOffset);

                                int tileNumber = tileData & 0x03FF;
                                if ((tileData & 0x0400) != 0) {
                                    tileX = 7 - tileX; // H-Flip
                                }
                                if ((tileData & 0x0800) != 0) {
                                    tileY = 7 - tileY; // V-Flip
                                }
                                if (is256ColorPalette) {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + (tileX)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) { // Not a transparent color
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                } else {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 4) + (tileY * 4) + (tileX / 2)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if ((tileX & 0x01) != 0) {
                                        colorIndex >>>= 4;
                                    } else {
                                        colorIndex &= 0x0F;
                                    }

                                    if (colorIndex != 0) {
                                        int paletteNumber = (tileData >>> 12) & 0x0F;
                                        short rgb15 = readPal16(((paletteNumber * 16) + colorIndex) * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }
                            }
                        }
                        if (bg0Enabled && (bg0Priority == p)) {
                            int characterBase = ((ioram[0x0008] & 0x0C) << 12);
                            int screenBase = ((ioram[0x0009] & 0x1F) << 11);

                            int xSize = ((ioram[0x0009] & 0x40) == 0) ? 256 : 512;
                            int ySize = ((ioram[0x0009] & 0x80) == 0) ? 256 : 512;
                            int xMask = xSize - 1;
                            int yMask = ySize - 1;

                            int xOffset = (((short) ((ioram[0x0010] & 0x00FF) | (ioram[0x0011] << 8))) & 0x01FF);
                            int yOffset = (((short) ((ioram[0x0012] & 0x00FF) | (ioram[0x0013] << 8))) & 0x01FF);

                            boolean isTileDataUpsideDown = (xSize != 256);

                            boolean is256ColorPalette = ((ioram[0x0008] & 0x80) != 0);

                            boolean isMosaicEnabled = ((ioram[0x0008] & 0x40) != 0);
                            int xMosaic = ((ioram[0x004C] & 0x0F) + 1);
                            int yMosaic = (((ioram[0x004C] >>> 4) & 0x0F) + 1);

                            int drawBGTextMode_y = (isMosaicEnabled ? (y - (y % yMosaic)) : y);
                            drawBGTextMode_y = (drawBGTextMode_y + yOffset) & yMask;

                            int yOffsetToAdd = 0;
                            if (isTileDataUpsideDown) {
                                if (drawBGTextMode_y >= 256) {
                                    yOffsetToAdd = 0x0800 * 2;
                                }
                                drawBGTextMode_y &= 0xFF;
                            }

                            int yTileDataOffset = (drawBGTextMode_y >>> 3) * 32;

                            int tileY = drawBGTextMode_y & 0x07;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int x = (isMosaicEnabled ? (xScreen - (xScreen % xMosaic)) : xScreen);
                                x = (x + xOffset) & xMask;

                                int xOffsetToAdd = 0;
                                if (isTileDataUpsideDown) {
                                    if (x >= 256) {
                                        xOffsetToAdd = 0x0800;
                                    }
                                    x &= 0xFF;
                                }

                                int xTileDataOffset = (x >>> 3);

                                int tileX = x & 0x07;

                                int tileDataOffset = ((yTileDataOffset + xTileDataOffset) * 2) + (yOffsetToAdd + xOffsetToAdd);

                                short tileData = readVRam16(screenBase + tileDataOffset);

                                int tileNumber = tileData & 0x03FF;
                                if ((tileData & 0x0400) != 0) {
                                    tileX = 7 - tileX; // H-Flip
                                }
                                if ((tileData & 0x0800) != 0) {
                                    tileY = 7 - tileY; // V-Flip
                                }
                                if (is256ColorPalette) {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + (tileX)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) { // Not a transparent color
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                } else {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 4) + (tileY * 4) + (tileX / 2)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if ((tileX & 0x01) != 0) {
                                        colorIndex >>>= 4;
                                    } else {
                                        colorIndex &= 0x0F;
                                    }

                                    if (colorIndex != 0) {
                                        int paletteNumber = (tileData >>> 12) & 0x0F;
                                        short rgb15 = readPal16(((paletteNumber * 16) + colorIndex) * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }
                            }
                        }
                        if (objEnabled) {
                            drawOBJLine(y, p);
                        }
                    }
                    break;
                case 1:
                    bg0Enabled = ((ioram[1] & 0x01) != 0);
                    bg1Enabled = ((ioram[1] & 0x02) != 0);
                    bg2Enabled = ((ioram[1] & 0x04) != 0);
                    objEnabled = ((ioram[1] & 0x10) != 0);

                    bg0Priority = ioram[0x0008] & 0x03;
                    bg1Priority = ioram[0x000A] & 0x03;
                    bg2Priority = ioram[0x000C] & 0x03;

                    drawBGLine(y);

                    for (int p = 3; p >= 0; p--) {
                        if (bg2Enabled & (bg2Priority == p)) {
                            int characterBase = ((ioram[0x000C] & 0x0C) << 12);
                            int screenBase = ((ioram[0x000D] & 0x1F) << 11);

                            int xySize;
                            int xyMask;
                            /*							switch (ioram[0x000D] & 0xC0) {
                            case 0x00:
                            xySize = 128;
                            xyMask = 127;
                            case 0x40:
                            xySize = 256;
                            xyMask = 255;
                            case 0x80:
                            xySize = 512;
                            xyMask = 511;
                            case 0xC0:
                            xySize = 1024;
                            xyMask = 1023;
                            default:
                            xySize = 0;
                            xyMask = -1;
                            }*/
                            switch (ioram[0x000D] & 0xC0) {
                                case 0x00:
                                    xySize = 128;
                                    xyMask = 127;
                                    break;
                                case 0x40:
                                    xySize = 256;
                                    xyMask = 255;
                                    break;
                                case 0x80:
                                    xySize = 512;
                                    xyMask = 511;
                                    break;
                                case 0xC0:
                                    xySize = 1024;
                                    xyMask = 1023;
                                    break;
                                default:
                                    xySize = 0;
                                    xyMask = -1;
                                    break;
                            }

                            int xCoordinate = ((((ioram[0x0028] & 0xFF) | ((ioram[0x0029] & 0xFF) << 8)
                                    | ((ioram[0x002A] & 0xFF) << 16) | ((ioram[0x002B]) << 24)) << 4) >> 4);
                            int yCoordinate = ((((ioram[0x002C] & 0xFF) | ((ioram[0x002D] & 0xFF) << 8)
                                    | ((ioram[0x002E] & 0xFF) << 16) | ((ioram[0x002F]) << 24)) << 4) >> 4);

                            int pa = (short) ((ioram[0x0020] & 0x00FF) | (ioram[0x0021] << 8)); // DX
                            int pb = (short) ((ioram[0x0022] & 0x00FF) | (ioram[0x0023] << 8)); // DMX
                            int pc = (short) ((ioram[0x0024] & 0x00FF) | (ioram[0x0025] << 8)); // DY
                            int pd = (short) ((ioram[0x0026] & 0x00FF) | (ioram[0x0027] << 8)); // DMY

                            boolean wraparoundEnabled = ((ioram[0x000D] & 0x20) != 0);

                            int xCurrentCoordinate = (y * pb) + xCoordinate;
                            int yCurrentCoordinate = (y * pd) + yCoordinate;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int drawBGRotScalMode_x = xCurrentCoordinate >> 8;
                                int drawBGRotScalMode_y = yCurrentCoordinate >> 8;

                                if (wraparoundEnabled) {
                                    drawBGRotScalMode_x &= xyMask;
                                    drawBGRotScalMode_y &= xyMask;
                                }

                                if ((drawBGRotScalMode_x >= 0) && (drawBGRotScalMode_x < xySize) && (drawBGRotScalMode_y >= 0) && (drawBGRotScalMode_y < xySize)) {
                                    int xTile = drawBGRotScalMode_x >>> 3;
                                    int yTile = drawBGRotScalMode_y >>> 3;

                                    int tileX = drawBGRotScalMode_x & 0x07;
                                    int tileY = drawBGRotScalMode_y & 0x07;

                                    int tileNumber = vram[((screenBase + (yTile * (xySize / 8)) + xTile) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + tileX) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) {
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }

                                xCurrentCoordinate += pa;
                                yCurrentCoordinate += pc;
                            }
                        }
                        if (bg1Enabled & (bg1Priority == p)) {
                            int characterBase = ((ioram[0x000A] & 0x0C) << 12);
                            int screenBase = ((ioram[0x000B] & 0x1F) << 11);

                            int xSize = ((ioram[0x000B] & 0x40) == 0) ? 256 : 512;
                            int ySize = ((ioram[0x000B] & 0x80) == 0) ? 256 : 512;
                            int xMask = xSize - 1;
                            int yMask = ySize - 1;

                            int xOffset = (((short) ((ioram[0x0014] & 0x00FF) | (ioram[0x0015] << 8))) & 0x01FF);
                            int yOffset = (((short) ((ioram[0x0016] & 0x00FF) | (ioram[0x0017] << 8))) & 0x01FF);

                            boolean isTileDataUpsideDown = (xSize != 256);

                            boolean is256ColorPalette = ((ioram[0x000A] & 0x80) != 0);

                            boolean isMosaicEnabled = ((ioram[0x000A] & 0x40) != 0);
                            int xMosaic = ((ioram[0x004C] & 0x0F) + 1);
                            int yMosaic = (((ioram[0x004C] >>> 4) & 0x0F) + 1);

                            int drawBGTextMode_y = (isMosaicEnabled ? (y - (y % yMosaic)) : y);
                            drawBGTextMode_y = (drawBGTextMode_y + yOffset) & yMask;

                            int yOffsetToAdd = 0;
                            if (isTileDataUpsideDown) {
                                if (drawBGTextMode_y >= 256) {
                                    yOffsetToAdd = 0x0800 * 2;
                                }
                                drawBGTextMode_y &= 0xFF;
                            }

                            int yTileDataOffset = (drawBGTextMode_y >>> 3) * 32;

                            int tileY = drawBGTextMode_y & 0x07;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int x = (isMosaicEnabled ? (xScreen - (xScreen % xMosaic)) : xScreen);
                                x = (x + xOffset) & xMask;

                                int xOffsetToAdd = 0;
                                if (isTileDataUpsideDown) {
                                    if (x >= 256) {
                                        xOffsetToAdd = 0x0800;
                                    }
                                    x &= 0xFF;
                                }

                                int xTileDataOffset = (x >>> 3);

                                int tileX = x & 0x07;

                                int tileDataOffset = ((yTileDataOffset + xTileDataOffset) * 2) + (yOffsetToAdd + xOffsetToAdd);

                                short tileData = readVRam16(screenBase + tileDataOffset);

                                int tileNumber = tileData & 0x03FF;
                                if ((tileData & 0x0400) != 0) {
                                    tileX = 7 - tileX; // H-Flip
                                }
                                if ((tileData & 0x0800) != 0) {
                                    tileY = 7 - tileY; // V-Flip
                                }
                                if (is256ColorPalette) {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + (tileX)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) { // Not a transparent color
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                } else {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 4) + (tileY * 4) + (tileX / 2)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if ((tileX & 0x01) != 0) {
                                        colorIndex >>>= 4;
                                    } else {
                                        colorIndex &= 0x0F;
                                    }

                                    if (colorIndex != 0) {
                                        int paletteNumber = (tileData >>> 12) & 0x0F;
                                        short rgb15 = readPal16(((paletteNumber * 16) + colorIndex) * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }
                            }
                        }
                        if (bg0Enabled & (bg0Priority == p)) {
                            int characterBase = ((ioram[0x0008] & 0x0C) << 12);
                            int screenBase = ((ioram[0x0009] & 0x1F) << 11);

                            int xSize = ((ioram[0x0009] & 0x40) == 0) ? 256 : 512;
                            int ySize = ((ioram[0x0009] & 0x80) == 0) ? 256 : 512;
                            int xMask = xSize - 1;
                            int yMask = ySize - 1;

                            /*							int xOffset = (((short) ((ioram[0x0014] & 0x00FF) | (ioram[0x0015] << 8))) & 0x01FF);*/
                            int xOffset = (((short) ((ioram[0x0010] & 0x00FF) | (ioram[0x0011] << 8))) & 0x01FF);
                            int yOffset = (((short) ((ioram[0x0012] & 0x00FF) | (ioram[0x0013] << 8))) & 0x01FF);

                            boolean isTileDataUpsideDown = (xSize != 256);

                            boolean is256ColorPalette = ((ioram[0x0008] & 0x80) != 0);

                            boolean isMosaicEnabled = ((ioram[0x0008] & 0x40) != 0);
                            int xMosaic = ((ioram[0x004C] & 0x0F) + 1);
                            int yMosaic = (((ioram[0x004C] >>> 4) & 0x0F) + 1);

                            int drawBGTextMode_y = (isMosaicEnabled ? (y - (y % yMosaic)) : y);
                            drawBGTextMode_y = (drawBGTextMode_y + yOffset) & yMask;

                            int yOffsetToAdd = 0;
                            if (isTileDataUpsideDown) {
                                if (drawBGTextMode_y >= 256) {
                                    yOffsetToAdd = 0x0800 * 2;
                                }
                                drawBGTextMode_y &= 0xFF;
                            }

                            int yTileDataOffset = (drawBGTextMode_y >>> 3) * 32;

                            int tileY = drawBGTextMode_y & 0x07;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int x = (isMosaicEnabled ? (xScreen - (xScreen % xMosaic)) : xScreen);
                                x = (x + xOffset) & xMask;

                                int xOffsetToAdd = 0;
                                if (isTileDataUpsideDown) {
                                    if (x >= 256) {
                                        xOffsetToAdd = 0x0800;
                                    }
                                    x &= 0xFF;
                                }

                                int xTileDataOffset = (x >>> 3);

                                int tileX = x & 0x07;

                                int tileDataOffset = ((yTileDataOffset + xTileDataOffset) * 2) + (yOffsetToAdd + xOffsetToAdd);

                                short tileData = readVRam16(screenBase + tileDataOffset);

                                int tileNumber = tileData & 0x03FF;
                                if ((tileData & 0x0400) != 0) {
                                    tileX = 7 - tileX; // H-Flip
                                }
                                if ((tileData & 0x0800) != 0) {
                                    tileY = 7 - tileY; // V-Flip
                                }
                                if (is256ColorPalette) {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + (tileX)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) { // Not a transparent color
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                } else {
                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 4) + (tileY * 4) + (tileX / 2)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if ((tileX & 0x01) != 0) {
                                        colorIndex >>>= 4;
                                    } else {
                                        colorIndex &= 0x0F;
                                    }

                                    if (colorIndex != 0) {
                                        int paletteNumber = (tileData >>> 12) & 0x0F;
                                        short rgb15 = readPal16(((paletteNumber * 16) + colorIndex) * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }
                            }
                        }
                        if (objEnabled) {
                            drawOBJLine(y, p);
                        }
                    }
                    break;
                case 2:
                    bg2Enabled = ((ioram[1] & 0x04) != 0);
                    bg3Enabled = ((ioram[1] & 0x08) != 0);
                    objEnabled = ((ioram[1] & 0x10) != 0);

                    bg2Priority = ioram[0x000C] & 0x03;
                    bg3Priority = ioram[0x000E] & 0x03;

                    drawBGLine(y);

                    for (int p = 3; p >= 0; p--) {
                        if (bg3Enabled & (bg3Priority == p)) {
                            int characterBase = ((ioram[0x000E] & 0x0C) << 12);
                            int screenBase = ((ioram[0x000F] & 0x1F) << 11);
                            int xySize;
                            int xyMask;
                            /*							switch (ioram[0x000F] & 0xC0) {
                            case 0x00:
                            xySize = 128;
                            xyMask = 127;
                            case 0x40:
                            xySize = 256;
                            xyMask = 255;
                            case 0x80:
                            xySize = 512;
                            xyMask = 511;
                            case 0xC0:
                            xySize = 1024;
                            xyMask = 1023;
                            default:
                            xySize = 0;
                            xyMask = -1;
                            }*/
                            switch (ioram[0x000F] & 0xC0) {
                                case 0x00:
                                    xySize = 128;
                                    xyMask = 127;
                                    break;
                                case 0x40:
                                    xySize = 256;
                                    xyMask = 255;
                                    break;
                                case 0x80:
                                    xySize = 512;
                                    xyMask = 511;
                                    break;
                                case 0xC0:
                                    xySize = 1024;
                                    xyMask = 1023;
                                    break;
                                default:
                                    xySize = 0;
                                    xyMask = -1;
                                    break;
                            }

                            int xCoordinate = ((((ioram[0x0038] & 0xFF) | ((ioram[0x0039] & 0xFF) << 8)
                                    | ((ioram[0x003A] & 0xFF) << 16) | ((ioram[0x003B]) << 24)) << 4) >> 4);
                            int yCoordinate = ((((ioram[0x003C] & 0xFF) | ((ioram[0x003D] & 0xFF) << 8)
                                    | ((ioram[0x003E] & 0xFF) << 16) | ((ioram[0x003F]) << 24)) << 4) >> 4);

                            int pa = (short) ((ioram[0x0030] & 0x00FF) | (ioram[0x0031] << 8)); // DX
                            int pb = (short) ((ioram[0x0032] & 0x00FF) | (ioram[0x0033] << 8)); // DMX
                            int pc = (short) ((ioram[0x0034] & 0x00FF) | (ioram[0x0035] << 8)); // DY
                            int pd = (short) ((ioram[0x0036] & 0x00FF) | (ioram[0x0037] << 8)); // DMY

                            boolean wraparoundEnabled = ((ioram[0x000F] & 0x20) != 0);

                            int xCurrentCoordinate = (y * pb) + xCoordinate;
                            int yCurrentCoordinate = (y * pd) + yCoordinate;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int drawBGRotScalMode_x = xCurrentCoordinate >> 8;
                                int drawBGRotScalMode_y = yCurrentCoordinate >> 8;

                                if (wraparoundEnabled) {
                                    drawBGRotScalMode_x &= xyMask;
                                    drawBGRotScalMode_y &= xyMask;
                                }

                                if ((drawBGRotScalMode_x >= 0) && (drawBGRotScalMode_x < xySize) && (drawBGRotScalMode_y >= 0) && (drawBGRotScalMode_y < xySize)) {
                                    int xTile = drawBGRotScalMode_x >>> 3;
                                    int yTile = drawBGRotScalMode_y >>> 3;

                                    int tileX = drawBGRotScalMode_x & 0x07;
                                    int tileY = drawBGRotScalMode_y & 0x07;

                                    int tileNumber = vram[((screenBase + (yTile * (xySize / 8)) + xTile) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + tileX) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) {
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }

                                xCurrentCoordinate += pa;
                                yCurrentCoordinate += pc;
                            }
                        }
                        if (bg2Enabled & (bg2Priority == p)) {
                            int characterBase = ((ioram[0x000C] & 0x0C) << 12);
                            int screenBase = ((ioram[0x000D] & 0x1F) << 11);

                            int xySize;
                            int xyMask;
                            /*							switch (ioram[0x000D] & 0xC0) {
                            case 0x00:
                            xySize = 128;
                            xyMask = 127;
                            case 0x40:
                            xySize = 256;
                            xyMask = 255;
                            case 0x80:
                            xySize = 512;
                            xyMask = 511;
                            case 0xC0:
                            xySize = 1024;
                            xyMask = 1023;
                            default:
                            xySize = 0;
                            xyMask = -1;
                            }*/
                            switch (ioram[0x000D] & 0xC0) {
                                case 0x00:
                                    xySize = 128;
                                    xyMask = 127;
                                    break;
                                case 0x40:
                                    xySize = 256;
                                    xyMask = 255;
                                    break;
                                case 0x80:
                                    xySize = 512;
                                    xyMask = 511;
                                    break;
                                case 0xC0:
                                    xySize = 1024;
                                    xyMask = 1023;
                                    break;
                                default:
                                    xySize = 0;
                                    xyMask = -1;
                                    break;
                            }

                            int xCoordinate = ((((ioram[0x0028] & 0xFF) | ((ioram[0x0029] & 0xFF) << 8)
                                    | ((ioram[0x002A] & 0xFF) << 16) | ((ioram[0x002B]) << 24)) << 4) >> 4);
                            int yCoordinate = ((((ioram[0x002C] & 0xFF) | ((ioram[0x002D] & 0xFF) << 8)
                                    | ((ioram[0x002E] & 0xFF) << 16) | ((ioram[0x002F]) << 24)) << 4) >> 4);

                            int pa = (short) ((ioram[0x0020] & 0x00FF) | (ioram[0x0021] << 8)); // DX
                            int pb = (short) ((ioram[0x0022] & 0x00FF) | (ioram[0x0023] << 8)); // DMX
                            int pc = (short) ((ioram[0x0024] & 0x00FF) | (ioram[0x0025] << 8)); // DY
                            int pd = (short) ((ioram[0x0026] & 0x00FF) | (ioram[0x0027] << 8)); // DMY

                            boolean wraparoundEnabled = ((ioram[0x000D] & 0x20) != 0);

                            int xCurrentCoordinate = (y * pb) + xCoordinate;
                            int yCurrentCoordinate = (y * pd) + yCoordinate;

                            for (int xScreen = 0; xScreen < 240; xScreen++) {

                                int drawBGRotScalMode_x = xCurrentCoordinate >> 8;
                                int drawBGRotScalMode_y = yCurrentCoordinate >> 8;

                                if (wraparoundEnabled) {
                                    drawBGRotScalMode_x &= xyMask;
                                    drawBGRotScalMode_y &= xyMask;
                                }

                                if ((drawBGRotScalMode_x >= 0) && (drawBGRotScalMode_x < xySize) && (drawBGRotScalMode_y >= 0) && (drawBGRotScalMode_y < xySize)) {
                                    int xTile = drawBGRotScalMode_x >>> 3;
                                    int yTile = drawBGRotScalMode_y >>> 3;

                                    int tileX = drawBGRotScalMode_x & 0x07;
                                    int tileY = drawBGRotScalMode_y & 0x07;

                                    int tileNumber = vram[((screenBase + (yTile * (xySize / 8)) + xTile) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    int colorIndex = vram[((characterBase + (tileNumber * 8 * 8) + (tileY * 8) + tileX) & 0x00FFFFFF) % vram.length] & 0xFF;

                                    if (colorIndex != 0) {
                                        short rgb15 = readPal16(colorIndex * 2);
                                        pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                                    }
                                }

                                xCurrentCoordinate += pa;
                                yCurrentCoordinate += pc;
                            }
                        }
                        if (objEnabled) {
                            drawOBJLine(y, p);
                        }
                    }
                    break;
                case 3:
                    bg2Enabled = ((ioram[1] & 0x04) != 0);
                    if (bg2Enabled) {
                        isMosaicEnabled = ((ioram[0x000C] & 0x40) != 0);
                        xMosaic = ((ioram[0x004C] & 0x0F) + 1);
                        yMosaic = (((ioram[0x004C] >>> 4) & 0x0F) + 1);

                        drawMode_y = (isMosaicEnabled ? (y - (y % yMosaic)) : y);

                        for (int xScreen = 0; xScreen < 240; xScreen++) {
                            drawMode_x = (isMosaicEnabled ? (xScreen - (xScreen % xMosaic)) : xScreen);

                            short rgb15 = readVRam16(((drawMode_y * 240) + drawMode_x) * 2);
                            pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                        }

                        objEnabled = ((ioram[1] & 0x10) != 0);
                        if (objEnabled) {
                            for (int p = 3; p >= 0; p--) {
                                drawOBJLine(y, p);
                            }
                        }
                    }
                    break;
                case 4:
                    bg2Enabled = ((ioram[1] & 0x04) != 0);
                    if (bg2Enabled) {
                        int frameAddress = (((ioram[0x0000] & 0x10) != 0) ? 0xA000 : 0x0000);

                        isMosaicEnabled = ((ioram[0x000C] & 0x40) != 0);
                        xMosaic = ((ioram[0x004C] & 0x0F) + 1);
                        yMosaic = (((ioram[0x004C] >>> 4) & 0x0F) + 1);

                        drawMode_y = (isMosaicEnabled ? (y - (y % yMosaic)) : y);

                        for (int xScreen = 0; xScreen < 240; xScreen++) {
                            drawMode_x = (isMosaicEnabled ? (xScreen - (xScreen % xMosaic)) : xScreen);

                            int colorIndex = vram[((frameAddress + ((drawMode_y * 240) + drawMode_x)) & 0x00FFFFFF) % vram.length] & 0xFF;
                            short rgb15 = readPal16(colorIndex * 2);
                            pixels[(y * 240) + xScreen] = toRGBA(rgb15);
                        }

                        objEnabled = ((ioram[1] & 0x10) != 0);
                        if (objEnabled) {
                            for (int p = 3; p >= 0; p--) {
                                drawOBJLine(y, p);
                            }
                        }
                    }
                    break;
                case 5:
                    System.out.println("Video Mode 5 unsupported");
                    break;
            }
        }// else if (y == 160) gui.repaint();
		//{
        //    System.arraycopy(pixels, 0, gui.fb, 0, (240 * 160));
        //}
    }

    private void drawBGLine(int yScreen) {

        int bgColor = toRGBA((short) ((palram[0] & 0x00FF) | (palram[1] << 8)));

        int begPixel = (yScreen) * 240;
        int endPixel = (yScreen + 1) * 240;

        for (int xScreen = begPixel; xScreen < endPixel; xScreen++) {
            pixels[xScreen] = bgColor;
        }
    }

    private void drawOBJLine(int yScreen, int priority) {

        boolean is1DMapping = ((ioram[0x0000] & 0x40) != 0);

        int xMosaic = ((ioram[0x004D] & 0x0F) + 1);
        int yMosaic = (((ioram[0x004D] >>> 4) & 0x0F) + 1);

        for (int objNumber = 127; objNumber >= 0; objNumber--) {
            int objPriority = ((oam[(objNumber << 3) + 5] >>> 2) & 0x00000003);

            if (objPriority == priority) {
                boolean isRotScalEnabled = ((oam[(objNumber << 3) + 1] & 0x01) != 0);

                int xSize = getXSize(objNumber);
                int ySize = getYSize(objNumber);

                int xTiles = xSize >>> 3;
                int yTiles = ySize >>> 3;

                int xCoordinate = ((oam[(objNumber << 3) + 2] & 0x000000FF) | ((oam[(objNumber << 3) + 3] << 31) >> 23));
                int yCoordinate = (oam[(objNumber << 3)] & 0x000000FF);

                boolean is256ColorPalette = ((oam[(objNumber << 3) + 1] & 0x20) != 0);
                int paletteNumber = ((oam[(objNumber << 3) + 5] & 0xF0) >>> 4);

                int firstTileNumber = ((oam[(objNumber << 3) + 4] & 0x000000FF) | ((oam[(objNumber << 3) + 5] & 0x00000003) << 8));
                int tileNumberIncrement;
                if (is1DMapping) {
                    tileNumberIncrement = (is256ColorPalette ? xTiles * 2 : xTiles);
                } else {
                    tileNumberIncrement = 32;
                    if (is256ColorPalette) {
                        firstTileNumber &= 0xFFFE;
                    }
                }

                boolean isMosaicEnabled = ((oam[(objNumber << 3) + 1] & 0x10) != 0);

                if (!isRotScalEnabled) {
                    boolean isDisplayable = ((oam[(objNumber << 3) + 1] & 0x02) == 0);

                    if (isDisplayable) {
                        boolean isHFlip = ((oam[(objNumber << 3) + 3] & 0x10) != 0);
                        boolean isVFlip = ((oam[(objNumber << 3) + 3] & 0x20) != 0);

                        if (yCoordinate >= 160) {
                            yCoordinate -= 256;
                        }

                        if ((yScreen >= yCoordinate) && (yScreen < yCoordinate + ySize)) {
                            int ySprite = yScreen - yCoordinate;

                            for (int xSprite = 0; xSprite < xSize; xSprite++) {
                                int xScreen = xCoordinate + xSprite;

                                if ((xScreen >= 0) && (xScreen < 240)) {
                                    int x = (isHFlip ? xSize - 1 - xSprite : xSprite);
                                    int y = (isVFlip ? ySize - 1 - ySprite : ySprite);

                                    int xTile = x >>> 3;
                                    int yTile = y >>> 3;

                                    int tileX = x & 0x07;
                                    int tileY = y & 0x07;

                                    if (is256ColorPalette) {
                                        int tileNumber = firstTileNumber + (yTile * tileNumberIncrement) + (xTile * 2);

                                        int colorIndex = vram[((0x00010000 + (tileNumber * 32) + (tileY * 8) + (tileX)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                        if (colorIndex != 0) {
                                            short rgb15 = readPal16(0x00000200 + (colorIndex * 2));
                                            pixels[(yScreen * 240) + xScreen] = toRGBA(rgb15);
                                        }
                                    } else {
                                        int tileNumber = firstTileNumber + (yTile * tileNumberIncrement) + (xTile);

                                        int colorIndex = vram[((0x00010000 + (tileNumber * 32) + (tileY * 4) + (tileX / 2)) & 0x00FFFFFF) % vram.length] & 0xFF;

                                        if ((tileX & 0x01) != 0) {
                                            colorIndex >>>= 4;
                                        } else {
                                            colorIndex &= 0x0F;
                                        }

                                        if (colorIndex != 0) {
                                            short rgb15 = readPal16(0x00000200 + (((paletteNumber * 16) + colorIndex) * 2));
                                            pixels[(yScreen * 240) + xScreen] = toRGBA(rgb15);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    private static int toRGBA(short rgb15) {

        int red = (rgb15 & 0x001F) << 19; // >> 0  << 3 << 16
        int green = (rgb15 & 0x03E0) << 6; // >> 5  << 3 << 8
        int blue = (rgb15 & 0x7C00) >>> 7; // >> 10 << 3 << 0
        int alpha = 0xFF000000;

        return (red | green | blue | alpha);
    }

}
