/*
 * arch/arm/kernel/hw-sa1100.c
 *
 * SA1100-dependent machine specifics
 *
 * Copyright (C) 2000 Nicolas Pitre <nico@cam.org>
 *
 * This will certainly contain more stuff with time... like power management,
 * PCMCIA abstraction for all different implementations, etc.
 *
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/user.h>
#include <asm/arch/pcmcia.h>

#ifdef CONFIG_SA1100_ASSABET

/* 
 * Some detection of the Neponset/SA1111 board need to be done...
 * Currently hardwired to stand-alone Assabet.
 */
unsigned long BCR_value = BCR_DB1110;

EXPORT_SYMBOL(BCR_value);

#endif



/* gpio_setup()
 * ^^^^^^^^^^^^
 * Associates the service routine `handler' with all of the GPIO signals
 * set in `gpio_mask' (configured as inputs). The boolean `enable'
 * determines whether IRQs are requested or freed, and whether edge
 * detects for the GPIO signals are set or cleared. The string
 * `identifier' is passed to request_irq() if `handler' is non-NULL,
 * as is `dev_id' (which, if non-NULL, causes free_irq() to be called
 * on disable).
 *
 * At most one shared IRQ (GPIO pins 11-27) will be registered or
 * freed.
 *
 * Returns: 0 on success, -1 if the requested interrupt was unavailble
 */
static int gpio_setup(unsigned int enable,
		      unsigned long gpio_re_mask,
		      unsigned long gpio_fe_mask,
		      void (*handler)(int irq, void *dev, 
				      struct pt_regs *regs),
		      char *identifier, void *dev_id){
  unsigned long i, flags, gpio_mask;
  
  gpio_mask=gpio_re_mask | gpio_fe_mask;
  GPDR&=~gpio_mask;

  if(enable){

    if(handler!=NULL)
      for(i=GPIO_MIN; i<GPIO_MAX; ++i)
	if(gpio_mask&GPIO_GPIO(i)){
	  if(request_irq(SA1100_GPIO_TO_IRQ(i), handler, SA_INTERRUPT, 
			 identifier, dev_id)){
	    printk(KERN_ERR "Request for IRQ %lu (GPIO %lu) failed\n",
		   SA1100_GPIO_TO_IRQ(i), i);
	    return -1;
	  }
	}

    save_flags_cli(flags);
    GFER=(GFER | gpio_fe_mask);
    GRER=(GRER | gpio_re_mask);
    restore_flags(flags);

  } else {

    if(dev_id!=NULL)
      for(i=GPIO_MIN; i<GPIO_MAX; ++i)
	if(gpio_mask&GPIO_GPIO(i))
	  free_irq(SA1100_GPIO_TO_IRQ(i), dev_id);
    
    save_flags_cli(flags);
    GFER=(GFER & ~gpio_fe_mask);
    GRER=(GRER & ~gpio_re_mask);
    restore_flags(flags);

  }

  return 0;
}


/*
 * Bitsy has extended, write-only memory-mapped GPIO's
 */
#if defined(CONFIG_SA1100_BITSY)
static int bitsy_egpio = EGPIO_BITSY_RS232_ON;
void clr_bitsy_egpio(unsigned long x) 
{
  bitsy_egpio &= ~x;
  *(volatile int *)0xdc000000 = bitsy_egpio;
}
void set_bitsy_egpio(unsigned long x) 
{
  bitsy_egpio |= x;
  *(volatile int *)0xdc000000 = bitsy_egpio;
}
#endif


/* map_setup()
 * ^^^^^^^^^^^
 * Convenience routine for filling in the virtual mapping table for the
 * PCMCIA spaces.
 */
static void map_setup(struct pcmcia_maps *maps){

  /* These values are from arch/arm/mm/mm-sa1100.c: */

  maps->sock[0].io.start=0xe0000000;
  maps->sock[0].io.stop=0xe3ffffff;

  maps->sock[0].attr.start=0xe8000000;
  maps->sock[0].attr.stop=0xebffffff;

  maps->sock[0].mem.start=0xf0000000;
  maps->sock[0].mem.stop=0xf3ffffff;

  maps->sock[1].io.start=0xe4000000;
  maps->sock[1].io.stop=0xe7ffffff;

  maps->sock[1].attr.start=0xec000000;
  maps->sock[1].attr.stop=0xefffffff;

  maps->sock[1].mem.start=0xf4000000;
  maps->sock[1].mem.stop=0xf7ffffff;

}


#ifdef CONFIG_SA1100_ASSABET

static int assabet_pcmcia_init(struct pcmcia_init *init){
  int return_val;

  /* Enable CF bus: */
  BCR_clear(BCR_CF_BUS_OFF);

  return_val=gpio_setup(1, /* enable */
			(GPIO_MBREQ_CF_CD | GPIO_GFX_IRQ_CF_BVD2 |
			 GPIO_SA111_IRQ_CF_BVD1), 
			(GPIO_MBREQ_CF_CD | GPIO_GFX_IRQ_CF_BVD2 |
			 GPIO_SA111_IRQ_CF_BVD1), init->handler, 
			SA1100_PCMCIA_IDENTIFIER, assabet_pcmcia_init);

  /* Caution: the card-ready line is being configured for edge
   *          detection here so that no additional setup is 
   *          required when client drivers request an IRQ. This
   *          assumes that the interrupt facility in the kernel
   *          can efficiently (and safely) ignore pin level
   *          transitions from GPIO sources for which no software
   *          handler has been registered.
   */
  return_val+=gpio_setup(1, 0, GPIO_MBGNT_CF_IRQ, NULL, NULL, NULL);
  
  map_setup(init->maps);

  /* There's only one slot, but it's "Slot 1": */
  return (return_val<0) ? -1 : 2;
}

static int assabet_pcmcia_shutdown(void){
  int return_val;

  /* Disable CF bus: */
  BCR_set(BCR_CF_BUS_OFF);

  return_val=gpio_setup(0, /* disable */
			(GPIO_MBREQ_CF_CD | GPIO_GFX_IRQ_CF_BVD2 |
			 GPIO_SA111_IRQ_CF_BVD1), 
			(GPIO_MBREQ_CF_CD | GPIO_GFX_IRQ_CF_BVD2 |
			 GPIO_SA111_IRQ_CF_BVD1), NULL, NULL,
			assabet_pcmcia_init);

  return_val+=gpio_setup(0, 0, GPIO_MBGNT_CF_IRQ, NULL, NULL, NULL);

  return (return_val<0) ? -1 : 0;
}

static int assabet_pcmcia_socket_state(struct pcmcia_state_array
				       *state_array){
  unsigned long levels;
  int return_val=1;

  if(state_array->size<2) return -1;

  memset(state_array->state, 0, 
	 (state_array->size)*sizeof(struct pcmcia_state));

  levels=GPLR;

  state_array->state[1].detect=((levels & GPIO_MBREQ_CF_CD)==0)?1:0;

  state_array->state[1].ready=(levels & GPIO_MBGNT_CF_IRQ)?1:0;

  state_array->state[1].bvd1=(levels & GPIO_SA111_IRQ_CF_BVD1)?1:0;

  state_array->state[1].bvd2=(levels & GPIO_GFX_IRQ_CF_BVD2)?1:0;

  state_array->state[1].wrprot=0; /* Not available on Assabet. */

  state_array->state[1].vs_3v=1;  /* Can only apply 3.3V on Assabet. */

  state_array->state[1].vs_Xv=0;

  return return_val;
}

static int assabet_pcmcia_get_irq_info(struct pcmcia_irq_info *info){

  if(info->sock>1) return -1;

  if(info->sock==1)
    info->irq=IRQ_MBGNT_CF;

  return 0;
}

static int assabet_pcmcia_configure_socket(const struct pcmcia_configure
					   *configure){

  unsigned long value=BCR_value;

  if(configure->sock>1) return -1;

  if(configure->sock==0) return 0;

  switch(configure->vcc){
  case 0:
    value=(value & ~BCR_CF_PWR);
    break;

  case 50:
    printk(KERN_WARNING "%s(): CS asked for 5V, applying 3.3V...\n",
	   __FUNCTION__);

  case 33:  /* Can only apply 3.3V to the CF slot. */
    value=(value | BCR_CF_PWR);
    break;

  default:
    printk(KERN_ERR "%s(): unrecognized Vcc %u\n", __FUNCTION__,
	   configure->vcc);
    return -1;
  }

  value=(configure->reset)?(value | BCR_CF_RST):(value & ~BCR_CF_RST);

  /* Silently ignore Vpp, output enable, speaker enable. */

  if(value!=BCR_value)
    BCR = BCR_value = value;

  return 0;
}

static struct pcmcia_low_level assabet_pcmcia_ops = { 
  assabet_pcmcia_init,
  assabet_pcmcia_shutdown,
  assabet_pcmcia_socket_state,
  assabet_pcmcia_get_irq_info,
  assabet_pcmcia_configure_socket
};

#endif  /* CONFIG_SA1100_ASSABET */

#ifdef CONFIG_SA1100_BITSY

static int bitsy_pcmcia_init(struct pcmcia_init *init)
{
  int ret;

  printk(__FUNCTION__ ": init=%p\n", init);
  set_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON);

  /* Hm. We want to pass a unique dev_id to request_irq(), so that
   * our handler can be disambiguated from others that share IRQ 11.
   * We don't actually do anything with this pointer, so just pick
   * something unique like the address of this routine:
   */
  ret=gpio_setup(1, /* enable */
		 (GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1),
                 (GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1),
                 init->handler,
		 SA1100_PCMCIA_IDENTIFIER, bitsy_pcmcia_init);

  map_setup(init->maps);

  return (ret<0)?ret:2;
}

static int bitsy_pcmcia_shutdown(void)
{

  printk(__FUNCTION__ "\n");
  clr_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON);
  return gpio_setup(0, /* disable */
		    (GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1), 
		    (GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1), NULL, NULL,
		    bitsy_pcmcia_init);
}

static int bitsy_pcmcia_socket_state(struct pcmcia_state_array
				       *state_array)
{
  int return_val = 0;
  unsigned long levels = GPLR;
  int i;

  if(state_array->size<2) return -1;
  printk(__FUNCTION__ ": state_array=%p\n", state_array);
  memset(state_array->state, 0, 
	 (state_array->size)*sizeof(struct pcmcia_state));

  if (levels & GPIO_BITSY_PCMCIA_CD0)
     state_array->state[0].detect = return_val = 1;
  if (levels & GPIO_BITSY_PCMCIA_IRQ0) /* compact flash mode */
     state_array->state[0].ready = return_val = 1;
  state_array->state[0].bvd1 = 0;   /* @@@@ check availability on Bitsy */
  state_array->state[0].bvd2 = 0;   /* @@@@ check availability on Bitsy */
  state_array->state[0].wrprot = 0; /* Not available on Bitsy. */
  state_array->state[0].vs_3v = 1;  /* Can only apply 3.3V on Bitsy. */
  state_array->state[0].vs_Xv = 0;  /* not available on Bitsy */

  if (levels & GPIO_BITSY_PCMCIA_CD1)
     state_array->state[1].detect = return_val = 1;
  if (levels & GPIO_BITSY_PCMCIA_IRQ1) /* compact flash mode */
     state_array->state[1].ready = return_val = 1;
  state_array->state[1].bvd1 = 0;   /* @@@@ check availability on Bitsy */
  state_array->state[1].bvd2 = 0;   /* @@@@ check availability on Bitsy */
  state_array->state[1].wrprot = 0; /* Not available on Bitsy. */
  state_array->state[1].vs_3v = 1;  /* Can only apply 3.3V on Bitsy. */
  state_array->state[1].vs_Xv = 0;  /* not available on Bitsy */

  /* clear the edge detect bits */
  if (return_val)
     GEDR |= (GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_IRQ0 | GPIO_BITSY_PCMCIA_CD1 | GPIO_BITSY_PCMCIA_IRQ1);

  for (i = 0; i < state_array->size; i++) {
     struct pcmcia_state *state = &state_array->state[i];
     printk(__FUNCTION__ ": sock=%d detect=%d read=%d bvd1=%d bvd1=%d wrprot=%d vs_3v=%d vs_Xv=%d \n", 
            i, 
            state->detect, state->ready,
            state->bvd1, state->bvd2, state->wrprot,
            state->vs_3v, state->vs_Xv);
     
  }

  return return_val;
}

static int bitsy_pcmcia_get_irq_info(struct pcmcia_irq_info *info)
{

  if(info->sock>1) return -1;

  if (info->sock == 0)
    info->irq = SA1100_GPIO_TO_IRQ(GPIO_BITSY_PCMCIA_IRQ0);
  else if (info->sock == 1)
    info->irq = SA1100_GPIO_TO_IRQ(GPIO_BITSY_PCMCIA_IRQ1);
  else
     return -1;

  printk(__FUNCTION__ ": sock=%d irq=%lx\n", info->sock, info->irq);
  return 0;
}

static int bitsy_pcmcia_configure_socket(const struct pcmcia_configure
					   *configure)
{

   if(configure->sock>1) return -1;

   printk(__FUNCTION__ ": sock=%d vcc=%d reset=%d\n", configure->sock, configure->vcc, configure->reset);

   switch(configure->vcc){
   case 0:
      clr_bitsy_egpio(EGPIO_BITSY_OPT_ON);
      break;

   case 50:
      printk(KERN_WARNING "%s(): CS asked for 5V, applying 3.3V...\n",
             __FUNCTION__);

   case 33:  /* Can only apply 3.3V to the CF slot. */
      set_bitsy_egpio(EGPIO_BITSY_OPT_ON);
      break;

   default:
      printk(KERN_ERR "%s(): unrecognized Vcc %u\n", __FUNCTION__,
             configure->vcc);
      return -1;
   }

   if (configure->reset) {
     set_bitsy_egpio(EGPIO_BITSY_CARD_RESET);
   } else {
     clr_bitsy_egpio(EGPIO_BITSY_CARD_RESET);
   }

   /* Silently ignore Vpp, output enable, speaker enable. */

   return 0;
}

static struct pcmcia_low_level bitsy_pcmcia_ops = { 
  bitsy_pcmcia_init,
  bitsy_pcmcia_shutdown,
  bitsy_pcmcia_socket_state,
  bitsy_pcmcia_get_irq_info,
  bitsy_pcmcia_configure_socket
};

#endif  /* CONFIG_SA1100_BITSY */

struct pcmcia_low_level *pcmcia_low_level=NULL;

EXPORT_SYMBOL(pcmcia_low_level);


void __init pcmcia_low_level_init(void){

#ifdef CONFIG_SA1100_ASSABET
  if(machine_is_assabet())
    pcmcia_low_level=&assabet_pcmcia_ops;
#endif
#ifdef CONFIG_SA1100_BITSY
  if(machine_is_bitsy())
    pcmcia_low_level=&bitsy_pcmcia_ops;
#endif

}

module_init(pcmcia_low_level_init);
