/*
 * linux/arch/arm/drivers/char/keyb_brutus.c
 *
 * Keyboard driver for Brutus ARM Linux.
 *
 * Changelog:
 * 22-01-1998	lsb	Created by modifying keyb_rpc.c
 * 19-12-1999	Nicolas Pitre:	Completely revamped (IRQ driven, cleanup...)
 */

#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/kbd_ll.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/kbd_kern.h>

#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/keyboard.h>

#include "kbmap_brutus.h"


static int key_fn_down=0;
static int key_numl_down=0;


static int kbd_putc (unsigned char c)
{
  /*
   * NOTE: on older versions of sa1100 you need to reset the ssp port
   * every time (I assume new versions here). These functions
   * were copied from code that came with early brutus boards.
   */
  
	while (!(Ser4SSSR & SSSR_TNF)); /* check for space */
	Ser4SSDR = (c<<8) | c; /* put char */
	udelay(100);
	while (!(Ser4SSSR & SSSR_RNE)); /* wait to get char back */
	return Ser4SSDR;
}

static void kbd_puts (char *m, int cnt)
{
#if 0  /* is this true?... */
	int i, x;

	GPCR = 1<<23;
	mdelay(1);
	GPSR = 1<<23;
	mdelay(1);

	for (i=0; i<cnt; i++) {
		x = kbd_putc( m[i] );
		udelay(100);
	}
#endif
}

static char lrc (char *m, int l)
{
	int i, crc = 0;
  
	for (i=0; i<l; i++) crc ^= *m++;
	if (crc & 0x80) crc ^= 0xC0;
	return crc;
}


static int Brutus_kbd_scancode(void)
{
	int x, y;
	static int last = -1;
  
	x = kbd_putc(0)&0xff; 	/* get char by sending one */
//printk( "kb: %02X\n", x );
	if (x == last || x == 0 || x == 0xff)
		return -1;

	last = x;

	if (x & 0x80) {    		/* key up */
		x &= 0x7f;
		if (x == 0x21) {	  /* fn key up */
			key_fn_down = 0;
			return -1;
		}else if (key_fn_down) {  /* this is a fn modified key */
			y = kbmapFN[x];
			if (y == KK_NUML) 
				return -1;
		}else if (key_numl_down)  /* this is a numlock modified key */
			y = kbmapNL[x];
		else
			y = kbmap[x];
		return (y | 0x80);
	}else{
		/* key down */
		if (x == 0x21) {	  /* fn key down */
			key_fn_down = 1;
			return -1;
		}else if (key_fn_down) {  /* this is a fn modified key */
			y = kbmapFN[x];
			if (y == KK_NUML) {	/* toggle local numlock */
				key_numl_down ^= 1;
				return -1;
			}
		}else if (key_numl_down)  /* this is a numlock modified key */
			y = kbmapNL[x];
		else
			y = kbmap[x];
	}
	return y;
}


static void Brutus_kbd_processdata(void)
{
	int k, keys = 0;

	while ( !(GPLR & (1<<25)) ) {	/* while there's data available */
		if( (k = Brutus_kbd_scancode()) != -1 ) {
			handle_scancode( (unsigned char)k, (k & 0x80) ? 0 : 1 );
			keys = 1;
		}
	}
	if (keys) tasklet_schedule(&keyboard_tasklet);
}

static void Brutus_kbd_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	
	if( !(GEDR & (1<<25)) )  return;  /* irq isn't for us */

#ifdef CONFIG_VT
	kbd_pt_regs = regs;
#endif

	do {
		Brutus_kbd_processdata();
		GEDR = 1<<25;	/* clear edge detect */
	} while (!(GPLR & (1<<25)));
}

void Brutus_kbd_disable_irq( void )
{
	long flags;
	save_flags_cli( flags );
	GFER &= ~(1<<25);
	restore_flags( flags );
}

void Brutus_kbd_enable_irq( void )
{
	long flags;
	save_flags_cli( flags );
	GFER |= (1<<25);
	/* be sure no key event happened while we weren't listening... */
	Brutus_kbd_processdata();
	restore_flags( flags );
}


int Brutus_kbd_translate(unsigned char scancode, unsigned char *keycode_p, char raw_mode)
{
	/* We already did most of the translation work in handle_rawcode.  Just
	   figure out whether this is an up or down event.  */
	*keycode_p = scancode & 0x7f;
	return 1;
}

void Brutus_kbd_leds(unsigned char leds)
{
	char msg[20];

	msg[0]=0x1B; /* led on */
	msg[1]=0xa6;
	msg[2]=0;
	msg[3]=(char) (leds & 1);
	msg[4]=0;
	msg[5]=0;
	msg[6]=0;
	msg[7]=0;
	msg[8]=lrc(msg,8);
	kbd_puts(msg, 9);
}


static inline void Brutus_kbd_init(void)
{
	char msg[30];
	int  i;

	/* set dir for ssp */
	GPDR |=  (0xD << 10) | (1<<23);
	GPDR &= ~(0x2 << 10) & ~(1 << 25);
	GAFR |=  (0xF << 10);

	GPSR = 1<<23;	/* deassert the kbctl wakup pin */
	PPAR |= 1<<18;	/* set alt function for spi interface  */

	Ser4SSCR0 = 0;
	Ser4SSCR1 = 0;	/* no ints no loopback */
	Ser4SSSR = 0;	/* remove any rcv overrun errors */

	/* turn on SSP */
	Ser4SSCR0 = SSCR0_DataSize(8) + SSCR0_SerClkDiv(8) + 
			SSCR0_SSE + SSCR0_Motorola;

	/* drain any data already there */
	while (Ser4SSSR & SSSR_RNE)  i = Ser4SSDR;
  
	msg[0]=0x1B; /* led on */
	msg[1]=0xa7;
	msg[2]=0;
	msg[3]=3;
	msg[4]=lrc(msg, 4);
	kbd_puts(msg, 5);

	/* clear keyboard buffer */
	while (Brutus_kbd_scancode() != -1);
}


void __init Brutus_kbd_init_hw(void)
{
	unsigned long flags;

	printk (KERN_INFO "Brutus keyboard driver v1.0\n");

	save_flags_cli (flags);
	Brutus_kbd_init();
	if (request_irq (IRQ_GPIO11_27, Brutus_kbd_interrupt, SA_SHIRQ, "keyboard", NULL) != 0)
		printk("Could not allocate keyboard IRQ!\n");
	
	GEDR = 1<<25;
	GFER |= 1<<25;
	GRER &= ~(1<<25);

	restore_flags (flags);
}
