/*
 * SA11x0 + UCB1x00 Touch Screen Driver Version 0.2
 * Put together by Tak-Shing Chan <tchan.rd@idthk.com>
 *
 * 90% derived from the Touch screen driver for Tifon
 * Copyright 1999 Peter Danielsson
 *
 * Codec routines derived from Itsy's Touchscreen driver
 * Copyright (c) Compaq Computer Corporation, 1998, 1999
 * Author: Larry Brakmo.
 *
 * Sample filtering derived from Diagman (ucb1x00Touch.c)
 * Copyright (C) 1999 Intel Corp.
 *
 * IOCTL's and jitter elimination derived from ADS Touchscreen driver
 * Copyright (c) 2000 Century Software, Inc.
 *
 * iPAQ emulation derived from Driver for the h3600 Touch Screen
 * Copyright 2000 Compaq Computer Corporation.
 * Author: Charles Flynn.
 */

/*
 * INSTRUCTIONS
 * To use this driver, simply ``mknod /dev/h3600_ts c 11 0'' and
 * use it as if it were an iPAQ.  TODO: iPAQ-compatible IOCTL's.
 */

#include <linux/types.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/irq.h>
#include <linux/string.h>
#include "sa1100_ts.h"

/*
 * From Compaq's Touch Screen Specification version 0.2 (draft)
 */
typedef struct {
	short	pressure;
	int	x;
	int	y;
	int	millisecs;
} TS_EVENT;

/*
 * Settings and definitions
 */
#define TS_MAJOR		11
#define TS_NAME			"h3600"
static int			raw_max_x, raw_max_y, res_x, res_y,
				raw_min_x, raw_min_y, xyswap;
static char			*dev_id = "h3600";
static wait_queue_head_t	queue;
static struct timer_list	timer;
static int			Reg10;
#define PRESSED			0
#define P_DONE			1
#define X_DONE			2
#define Y_DONE			3
#define RELEASED		4
#define BUFSIZE			128
#define XLIMIT			160
#define YLIMIT			160
static int			state, head, tail, sample;
static TS_EVENT			cur_data, samples[3], buf[BUFSIZE];
static struct fasync_struct	*fasync;
static unsigned long		in_timehandle = 0;

static inline void
ts_clear_codec_int(void)
{
	int	s;

	/*
	 * Clear interrupts at codec
	 */
	if ((s = codec_read(CODEC_REG_INT_STATUS) & (TSPX_INT | ADC_INT))) {
		codec_write(CODEC_REG_INT_STATUS, 0);
		codec_write(CODEC_REG_INT_STATUS, s);
	}
}

static inline void
set_read_x_pos(void)
{
	/*
	 * See Philips' AN809 for an explanation of the pressure mode
	 * switch
	 */
	codec_write(CODEC_REG_TS_CTL, TSPX_POW | TSMX_GND |
		    TSC_MODE_PRESSURE | TSC_BIAS_ENA);
	codec_write(CODEC_REG_ADC_CTL, Reg10 = ADC_INPUT_TSPY |
		    ADC_ENA | ADC_SYNC_ENA);
	codec_write(CODEC_REG_TS_CTL, TSPX_POW | TSMX_GND |
		    TSC_MODE_POSITION | TSC_BIAS_ENA);
	udelay(50);
	codec_write(CODEC_REG_ADC_CTL, Reg10 | ADC_START);
	codec_write(CODEC_REG_IO_DATA,
		    codec_read(CODEC_REG_IO_DATA) | 0x200);
	codec_write(CODEC_REG_ADC_CTL, Reg10);
	codec_write(CODEC_REG_IO_DATA,
		    codec_read(CODEC_REG_IO_DATA) & ~0x200);
}

static inline void
set_read_y_pos(void)
{
	codec_write(CODEC_REG_TS_CTL, TSPY_POW | TSMY_GND |
		    TSC_MODE_PRESSURE | TSC_BIAS_ENA);
	codec_write(CODEC_REG_ADC_CTL, Reg10 = ADC_INPUT_TSPX |
		    ADC_ENA | ADC_SYNC_ENA);
	codec_write(CODEC_REG_TS_CTL, TSPY_POW | TSMY_GND |
		    TSC_MODE_POSITION | TSC_BIAS_ENA);
	udelay(50);
	codec_write(CODEC_REG_ADC_CTL, Reg10 | ADC_START);
	codec_write(CODEC_REG_IO_DATA,
		    codec_read(CODEC_REG_IO_DATA) | 0x200);
	codec_write(CODEC_REG_ADC_CTL, Reg10);
	codec_write(CODEC_REG_IO_DATA,
		    codec_read(CODEC_REG_IO_DATA) & ~0x200);
}

static inline void
set_read_pressure(void)
{
	codec_write(CODEC_REG_TS_CTL, TSPX_POW | TSMX_POW |
		    TSPY_GND | TSMY_GND | TSC_MODE_PRESSURE |
		    TSC_BIAS_ENA);
	codec_write(CODEC_REG_ADC_CTL, Reg10 = ADC_INPUT_TSPX |
		    ADC_ENA | ADC_SYNC_ENA);
	udelay(50);
	codec_write(CODEC_REG_ADC_CTL, Reg10 | ADC_START);
	codec_write(CODEC_REG_IO_DATA,
		    codec_read(CODEC_REG_IO_DATA) | 0x200);
	codec_write(CODEC_REG_ADC_CTL, Reg10);
	codec_write(CODEC_REG_IO_DATA,
		    codec_read(CODEC_REG_IO_DATA) & ~0x200);
}

static int
read_adc(void)
{
	int	v;

	while (!((v = codec_read(CODEC_REG_ADC_DATA)) & ADC_DAT_VAL));
	return GET_ADC_DATA(v);
}

static int
sa1100_ts_ioctl(struct inode *inode, struct file *filp, unsigned int cmd,
		unsigned long arg)
{
	/*
	 * Microwindows style (should change to TS_CAL when the
	 * specification is ready)
	 */
	switch (cmd) {
	case 3:
		raw_max_x = arg;
		break;
	case 4:
		raw_max_y = arg;
		break;
	case 5:
		res_x = arg;
		break;
	case 6:
		res_y = arg;
		break;
	case 10:
		raw_min_x = arg;
		break;
	case 11:
		raw_min_y = arg;
		break;
	case 12:
		/*
		 * New attribute for portrait modes
		 */
		xyswap = arg;
	}
	return 0;
}

static inline int
pen_up(void)
{
	codec_write(CODEC_REG_ADC_CTL, 0);
	codec_write(CODEC_REG_TS_CTL, TSPX_POW | TSMX_POW |
		    TSPY_GND | TSMY_GND);
	return codec_read(CODEC_REG_TS_CTL) & TSPX_LOW;
}

static void
new_data(void)
{
	static TS_EVENT	last_data = { 0, 0, 0, 0 };
	int		diff0, diff1, diff2, diff3;

	if (cur_data.pressure) {
		if (sample < 3) {
			samples[sample].x = cur_data.x;
			samples[sample++].y = cur_data.y;
			return;
		}
		sample = 0;

		/*
		 * Check the variance between X samples (discard if not
		 * similar), then choose the closest pair
		 */
		diff0 = abs(samples[0].x - samples[1].x);
		diff1 = abs(samples[1].x - samples[2].x);
		diff2 = abs(samples[2].x - cur_data.x);
		diff3 = abs(cur_data.x - samples[1].x);

		if (diff0 > XLIMIT || diff1 > XLIMIT || diff2 > XLIMIT ||
		    diff3 > XLIMIT)
			return;

		if (diff1 < diff2) {
			if (diff1 < diff3)
				cur_data.x =
				    (samples[1].x + samples[2].x) / 2;
			else
				cur_data.x =
				    (cur_data.x + samples[1].x) / 2;
		} else {
			if (diff2 < diff3)
				cur_data.x =
				    (samples[2].x + cur_data.x) / 2;
			else
				cur_data.x =
				    (cur_data.x + samples[1].x) / 2;
		}

		/*
		 * Do the same for Y
		 */
		diff0 = abs(samples[0].y - samples[1].y);
		diff1 = abs(samples[1].y - samples[2].y);
		diff2 = abs(samples[2].y - cur_data.y);
		diff3 = abs(cur_data.y - samples[1].y);

		if (diff0 > YLIMIT || diff1 > YLIMIT || diff2 > YLIMIT ||
		    diff3 > YLIMIT)
			return;

		if (diff1 < diff2) {
			if (diff1 < diff3)
				cur_data.y =
				    (samples[1].y + samples[2].y) / 2;
			else
				cur_data.y =
				    (cur_data.y + samples[1].y) / 2;
		} else {
			if (diff2 < diff3)
				cur_data.y =
				    (samples[2].y + cur_data.y) / 2;
			else
				cur_data.y =
				    (cur_data.y + samples[1].y) / 2;
		}
	} else {
		/*
		 * Reset jitter detection on pen release
		 */
		last_data.x = 0;
		last_data.y = 0;
	}

	/*
	 * Jitter elimination
	 */
	if ((last_data.x || last_data.y)
	    && abs(last_data.x - cur_data.x) <= 3
	    && abs(last_data.y - cur_data.y) <= 3)
		return;
	cur_data.millisecs = jiffies;
	last_data = cur_data;

	if (head != tail) {
		int last = head--;
		if (last < 0)
			last = BUFSIZE - 1;
	}
	buf[head] = cur_data;
	if (++head == BUFSIZE)
		head = 0;
	if (head == tail && tail++ == BUFSIZE)
		tail = 0;
	if (fasync)
		kill_fasync(&fasync, SIGIO, POLL_IN);
	wake_up_interruptible(&queue);
}

static TS_EVENT
get_data(void)
{
	int	last = tail;

	if (++tail == BUFSIZE)
		tail = 0;
	return buf[last];
}

static void
wait_for_action(void)
{
	state = PRESSED;
	sample = 0;
	codec_write(CODEC_REG_RISE_INT_ENABLE, 0);
	codec_write(CODEC_REG_FALL_INT_ENABLE, TSPX_INT);
	codec_write(CODEC_REG_ADC_CTL, 0);
	codec_write(CODEC_REG_TS_CTL, TSPX_POW | TSMX_POW |
		    TSPY_GND | TSMY_GND | TSC_MODE_INT);
}

static void
start_chain(void)
{
	state = P_DONE;
	codec_write(CODEC_REG_RISE_INT_ENABLE, ADC_INT);
	set_read_pressure();
}

static unsigned int
sa1100_ts_poll(struct file *filp, struct poll_table_struct *wait)
{
	poll_wait(filp, &queue, wait);
	if (head != tail)
		return POLLIN | POLLRDNORM;
	return 0;
}

static ssize_t
sa1100_ts_read(struct file *filp, char *buf, size_t count, loff_t *l)
{
	DECLARE_WAITQUEUE(wait, current);
	int		i;
	TS_EVENT	t;
	short		out_buf[4];

	if (head == tail) {
		if (filp->f_flags & O_NONBLOCK)
			return -EAGAIN;
		add_wait_queue(&queue, &wait);
		current->state = TASK_INTERRUPTIBLE;
		while (head == tail && !signal_pending(current)) {
			schedule();
			current->state = TASK_INTERRUPTIBLE;
		}
		current->state = TASK_RUNNING;
		remove_wait_queue(&queue, &wait);
	}
	for (i = count; i >= sizeof out_buf;
	     i -= sizeof out_buf, buf += sizeof out_buf) {
		if (head == tail)
			break;
		t = get_data();
		out_buf[0] = t.pressure;

#ifdef CONFIG_SA1100_ASSABET
		if (xyswap) {
			out_buf[1] =
			    (((raw_max_y - t.y)) * res_y) /
			    (raw_max_y - raw_min_y);
			out_buf[2] =
			    (((t.x - raw_min_x)) * res_x) /
			    (raw_max_x - raw_min_x);
		} else {
			out_buf[1] =
			    (((raw_max_x - t.x)) * res_x) /
			    (raw_max_x - raw_min_x);
			out_buf[2] =
			    (((raw_max_y - t.y)) * res_y) /
			    (raw_max_y - raw_min_y);
		}
#else
		if (xyswap) {
			out_buf[1] =
			    (((t.y - raw_min_y)) * res_y) /
			    (raw_max_y - raw_min_y);
			out_buf[2] =
			    (((t.x - raw_min_x)) * res_x) /
			    (raw_max_x - raw_min_x);
		} else {
			out_buf[1] =
			    (((t.x - raw_min_x)) * res_x) /
			    (raw_max_x - raw_min_x);
			out_buf[2] =
			    (((t.y - raw_min_y)) * res_y) /
			    (raw_max_y - raw_min_y);
		}
#endif
		out_buf[3] = t.millisecs;
		copy_to_user(buf, &out_buf, sizeof out_buf);
	}
	return count - i;
}

/*
 * Forward declaration 
 */
static void	sa1100_ts_timer(unsigned long);

static int
sa1100_ts_starttimer(void)
{
	in_timehandle++;
	init_timer(&timer);
	timer.function = sa1100_ts_timer;
	timer.expires = jiffies + HZ / 100;
	add_timer(&timer);
	return 0;
}

static void
sa1100_ts_timer(unsigned long data)
{
	in_timehandle--;
	if (pen_up()) {
		cur_data.pressure = 0;
		new_data();
		wait_for_action();
	} else {
		start_chain();
	}
}

static int
sa1100_ts_fasync(int fd, struct file *filp, int on)
{
	int	retval;

	retval = fasync_helper(fd, filp, on, &fasync);
	if (retval < 0)
		return retval;
	else
		return 0;
}

static int
sa1100_ts_open(struct inode *inode, struct file *filp)
{
	MOD_INC_USE_COUNT;
	return 0;
}

static int
sa1100_ts_release(struct inode *inode, struct file *filp)
{
	sa1100_ts_fasync(-1, filp, 0);
	MOD_DEC_USE_COUNT;
	return 0;
}

static void
sa1100_ts_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	ts_clear_codec_int();
	if (in_timehandle > 0)
		return;
	switch (state) {
	case PRESSED:
		start_chain();
		break;
	case P_DONE:
		cur_data.pressure = read_adc();
		codec_write(CODEC_REG_RISE_INT_ENABLE, ADC_INT);
		codec_write(CODEC_REG_FALL_INT_ENABLE, 0);
		set_read_x_pos();
		state++;
		break;
	case X_DONE:
		cur_data.x = read_adc();
		codec_write(CODEC_REG_RISE_INT_ENABLE, ADC_INT);
		set_read_y_pos();
		state++;
		break;
	case Y_DONE:
		cur_data.y = read_adc();
		codec_write(CODEC_REG_RISE_INT_ENABLE, TSPX_INT);
		codec_write(CODEC_REG_ADC_CTL, 0);
		codec_write(CODEC_REG_TS_CTL, TSPX_POW | TSMX_POW |
			    TSPY_GND | TSMY_GND | TSC_MODE_INT);
		state++;
		new_data();
		sa1100_ts_starttimer();
		break;
	case RELEASED:
		cur_data.pressure = 0;
		new_data();
		wait_for_action();
	}
}

static struct file_operations	sa1100_ts_fops = {
	read:		sa1100_ts_read,
	poll:		sa1100_ts_poll,
	ioctl:		sa1100_ts_ioctl,
	fasync:		sa1100_ts_fasync,
	open:		sa1100_ts_open,
	release:	sa1100_ts_release,
};

int __init
sa1100_ts_init(void)
{
	unsigned int	irq;

#ifdef CONFIG_SA1100_ASSABET
	raw_max_x = 944;
	raw_max_y = 944;
	raw_min_x = 70;
	raw_min_y = 70;
	res_x = 320;
	res_y = 240;
#else
	raw_max_x = 885;
	raw_max_y = 885;
	raw_min_x = 70;
	raw_min_y = 70;
	res_x = 640;
	res_y = 480;
#endif
	xyswap = 0;
	head = 0;
	tail = 0;
	init_waitqueue_head(&queue);

#ifdef CONFIG_SA1100_ASSABET
	if (machine_is_assabet()) {
		BCR_set(BCR_CODEC_RST);
		Ser4MCCR0 = MCCR0_MCE | MCCR0_SmpCnt;
		GPDR &= ~GPIO_UCB1300_IRQ;
		set_GPIO_IRQ_edge(GPIO_UCB1300_IRQ, GPIO_BOTH_EDGES);
	}
#endif
#ifdef CONFIG_SA1100_GRAPHICSCLIENT
	if (machine_is_graphicsclient()) {
		Ser4MCCR0 = MCCR0_MCE;
		set_GPIO_IRQ_edge(GPIO_GPIO22, GPIO_BOTH_EDGES);
	}
#endif

	/*
	 * Initialize the touchscreen controller
	 */
	codec_write(CODEC_REG_ADC_CTL, 0);
	codec_write(CODEC_REG_RISE_INT_ENABLE, 0);
	codec_write(CODEC_REG_FALL_INT_ENABLE, 0);
	codec_write(CODEC_REG_IO_DIRECTION, 0x200);
	ts_clear_codec_int();
	register_chrdev(TS_MAJOR, TS_NAME, &sa1100_ts_fops);

#ifdef CONFIG_SA1100_ASSABET
	if (machine_is_assabet())
		irq = IRQ_GPIO_UCB1300_IRQ;
#endif

#ifdef CONFIG_SA1100_GRAPHICSCLIENT
	if (machine_is_graphicsclient())
		irq = IRQ_GPIO6;
#endif

	if (request_irq(irq, sa1100_ts_interrupt, SA_INTERRUPT |
			SA_SHIRQ, TS_NAME, dev_id))
		printk("sa1100_ts_init: failed to register IRQ\n");

	wait_for_action();
	printk("sa1100 touch screen driver initialized\n");
	return 0;
}

void __exit
sa1100_ts_cleanup(void)
{
	unsigned int	irq;

#ifdef CONFIG_SA1100_ASSABET
	if (machine_is_assabet())
		irq = IRQ_GPIO_UCB1300_IRQ;
#endif

#ifdef CONFIG_SA1100_GRAPHICSCLIENT
	if (machine_is_graphicsclient())
		irq = IRQ_GPIO6;
#endif

	codec_write(CODEC_REG_ADC_CTL, 0);
	codec_write(CODEC_REG_RISE_INT_ENABLE, 0);
	codec_write(CODEC_REG_FALL_INT_ENABLE, 0);
	free_irq(irq, dev_id);
	if (in_timehandle)
		del_timer(&timer);
	unregister_chrdev(TS_MAJOR, TS_NAME);

	printk("sa1100 touch screen driver removed\n");
}

module_init(sa1100_ts_init);
module_exit(sa1100_ts_cleanup);
