/*
 * sa1100_ssp.c: an audio driver for the Digital/Intel SA-1100 CPU
 *
 * Copyright (C) 2000, The Delft University of Technology
 *
 * Author:
 *   Erik Mouw (J.A.K.Mouw@its.tudelft.nl)
 *
 * This software has been developed for the LART computing board
 * (http://www.lart.tudelft.nl/). The development has been sponsored by
 * the Mobile MultiMedia Communications (http://www.mmc.tudelft.nl/)
 * and Ubiquitous Communications (http://www.ubicom.tudelft.nl/)
 * projects.
 *
 * The author can be reached at:
 *
 *  Erik Mouw
 *  Information and Communication Theory Group
 *  Faculty of Information Technology and Systems
 *  Delft University of Technology
 *  P.O. Box 5031
 *  2600 GA Delft
 *  The Netherlands
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>    /* for struct file_operations in linux/sound.h */
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/errno.h>
#include <linux/ioport.h>

#include <asm/io.h>
#include <asm/dma.h>
#include <asm/uaccess.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>

/* FIXME: for debug */
#include <linux/string.h>



//#define MAGIC_CHECK

#define MODULE_NAME "sa1100_ssp"

#define AUDIO_FORMAT_DEFAULT    (AFMT_S16_LE)
#define AUDIO_CHANNELS_DEFAULT  (1)
#define AUDIO_RATE_DEFAULT      (44100)
#define AUDIO_BLOCKSIZE_DEFAULT (8192)

#define AUDIO_DMA_BUFFER_SIZE   (8192)
#define AUDIO_DMA_BUFFERS       (2)

#define AUDIO_IRQ               (IRQ_DMA0)




typedef struct
{
	char *buf;  /* the virtual address */
	char *phys; /* the physical address */
	volatile int num;
	volatile int in_use; /* this buffer is in use by the DMA engine */
} ssp_dma_t;




typedef struct
{
	/* init data */
	int minor_device;
	int refcount;

	/* /dev/dsp state */
	int format;
	int channels;
	int rate;
	int blocksize;
	
	/* dma stuff */
	wait_queue_head_t dmawaitq;
	char *dmabuf;
	char *phys_dmabuf;
	ssp_dma_t dmastate[2];
} ssp_audio_state_t;




static int debug = 0;
static int check = 0;
static int pvalue = 0x7fff;
static int nvalue = 0x8001;
static ssp_audio_state_t ssp_audio_state;

static int bit_counter = 0;
static int saw_tooth = 0;
static unsigned int uval;




static void reset_ssp_audio_state(void)
{
	ssp_audio_state.format = AUDIO_FORMAT_DEFAULT;
	ssp_audio_state.channels = AUDIO_CHANNELS_DEFAULT;
	ssp_audio_state.rate = AUDIO_RATE_DEFAULT;
	ssp_audio_state.blocksize = AUDIO_BLOCKSIZE_DEFAULT;
}




static int init_ssp_audio_dma(void)
{
/*
 	ssp_audio_state.dmabuf = (char *)kmalloc(AUDIO_DMA_BUFFERS * AUDIO_DMA_BUFFER_SIZE,
 						 GFP_ATOMIC | GFP_DMA);
*/

	/* FIXME: don't know if we should use GFP_KERNEL or GFP_ATOMIC */
	ssp_audio_state.dmabuf = (char *)consistent_alloc(GFP_ATOMIC,
							  AUDIO_DMA_BUFFERS * AUDIO_DMA_BUFFER_SIZE,
							  (dma_addr_t *)&ssp_audio_state.phys_dmabuf);

	if(! ssp_audio_state.dmabuf)
		return -ENOMEM;

	/* The SA1100 manual says: cache misses on write never
	 * allocate the cache. Since we will only be writing to this
	 * memory, we flush it once right here.  
	 */
	cpu_flush_cache_area((unsigned long)ssp_audio_state.dmabuf,
			     (unsigned long)(ssp_audio_state.dmabuf + 
					     AUDIO_DMA_BUFFERS * AUDIO_DMA_BUFFER_SIZE),
			     0);

	ssp_audio_state.dmastate[0].buf = ssp_audio_state.dmabuf;
	ssp_audio_state.dmastate[0].phys = ssp_audio_state.phys_dmabuf;
	ssp_audio_state.dmastate[0].num = 0;
	ssp_audio_state.dmastate[0].in_use = 0;

	ssp_audio_state.dmastate[1].buf = ssp_audio_state.dmabuf + AUDIO_DMA_BUFFER_SIZE;
	ssp_audio_state.dmastate[1].phys = ssp_audio_state.phys_dmabuf + AUDIO_DMA_BUFFER_SIZE;
	ssp_audio_state.dmastate[1].num = 0;
	ssp_audio_state.dmastate[1].in_use = 0;

	return 0;
}




static int exit_ssp_audio_dma(void)
{
	ClrDCSR0 = DCSR_RUN | DCSR_IE;

	consistent_free(ssp_audio_state.dmabuf);

	return 0;
}




static void ssp_audio_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	int status = RdDCSR0;
	int clearstatus = 0;
	int buffer_ready = 0;
	
	/* check errors */
	if(status & DCSR_ERROR) {
		printk(KERN_ERR MODULE_NAME ": DMA error!\n");
		clearstatus |= DCSR_ERROR;
	}

	
	/* Buffer A is done */
	if(status & DCSR_DONEA) {
		if(debug)
			printk(KERN_INFO MODULE_NAME ": DMA buffer A done\n");

		ssp_audio_state.dmastate[0].in_use = 0;
		buffer_ready = 1;

		clearstatus |= DCSR_DONEA;
	}

	/* Buffer B is done */
	if(status & DCSR_DONEB) {
		if(debug)
			printk(KERN_INFO MODULE_NAME ": DMA buffer B done\n");

		ssp_audio_state.dmastate[1].in_use = 0;
		buffer_ready = 1;

		clearstatus |= DCSR_DONEB;
	}

	if(clearstatus)
		ClrDCSR0 = clearstatus;

	if(buffer_ready) {
		/* wake up any sleeping write() command */
		wake_up_interruptible(&ssp_audio_state.dmawaitq);
	}
}




static int ssp_audio_write(struct file *file, const char *buffer, 
			   size_t count, loff_t *ppos)
{
	int minor = MINOR(file->f_dentry->d_inode->i_rdev);
	int numwritten = 0;
	int towrite;
	int current_dma;
	int val;

	if(minor != ssp_audio_state.minor_device)
		return -EINVAL;

	while(count > 0) {
		/* see if there are no empty buffers */
		if(ssp_audio_state.dmastate[0].in_use &&
		   ssp_audio_state.dmastate[1].in_use) {
			/* non-blocking should always return */
			if(file->f_flags & O_NONBLOCK) {
				return numwritten ? numwritten : -EAGAIN;
			} else {
				/* wait for a buffer to become free */
				interruptible_sleep_on(&ssp_audio_state.dmawaitq);
				
				if(signal_pending(current)) 
					return numwritten ? numwritten : -EINTR;
			}
		}
			
		/* at this point we can be sure that there is at least
		 * one empty buffer 
		 */
		if(! ssp_audio_state.dmastate[0].in_use)
			current_dma = 0;
		else if(! ssp_audio_state.dmastate[1].in_use)
			current_dma = 1;
		else {
			/* this is an error, should not happen */
			printk(KERN_ERR MODULE_NAME ": Error: no DMA buffer available!\n");
			return -EIO;
		}
		
		/* check how much we should write */
		if(count > AUDIO_DMA_BUFFER_SIZE)
			towrite = AUDIO_DMA_BUFFER_SIZE;
		else
			towrite = (int)count;

		/* copy user data */
		if(check) {
			int i;
			char *ptr = ssp_audio_state.dmastate[current_dma].buf;

			/* block */
			for(i = 0; i < towrite; i+=8) {
				if(check == 1) {
					/* constant */
					*((int *)(&ptr[i    ])) = (pvalue << 16) | pvalue;
					*((int *)(&ptr[i + 4])) = (pvalue << 16) | pvalue;
				} else if (check == 2) {
					/* block */
					if(i&0x20)
						uval = (((unsigned int)pvalue)<< 16) | 
							(((unsigned int)pvalue) & 0xffff);
					else
						uval = (((unsigned int)nvalue)<< 16) | 
							(((unsigned int)nvalue) & 0xffff);

					uval &= 0xffff;

					*((unsigned int *)(&ptr[i    ])) = uval; /* L+R first sample */
					*((unsigned int *)(&ptr[i + 4])) = uval; /* L+R second sample */
				} else if(check == 3) {
					*((int *)(&ptr[i    ])) = (i<<16)+i;
					*((int *)(&ptr[i + 4])) = (i<<16)+i;
				}
				
			}

#if 0			
			/* saw tooth */
			for(i = 0; i < towrite; i++) {
				*ptr++ = saw_tooth>>2;
				
				saw_tooth++;
			}
#endif
		} else {
			copy_from_user_ret(ssp_audio_state.dmastate[current_dma].buf,
					   buffer, towrite, -EFAULT);
			/* FIXME: debug */
#ifdef MAGIC_CHECK
			get_user_ret(val, (int *)buffer, -EFAULT);
			if(val != 0xdeadbeef)
				printk(KERN_INFO MODULE_NAME ": Warning: first word 0x%08x, towrite = %d\n", 
				       val,
				       towrite);
#endif
		}
		  

		ssp_audio_state.dmastate[current_dma].num = towrite;
		buffer += towrite;
		count -= towrite;
		numwritten += towrite;

		if(debug)
			printk(KERN_INFO MODULE_NAME ": writing %d bytes using DMA buffer %s\n",
			       towrite, current_dma ? "B" : "A");

		/* set up and start DMA */
		if(current_dma == 0) {
			/* buffer A */
			ClrDCSR0 = DCSR_DONEA | DCSR_STRTA;
			DBSA0 = ssp_audio_state.dmastate[0].phys;
			DBTA0 = towrite;
			ssp_audio_state.dmastate[0].in_use = 1;
			SetDCSR0 = DCSR_STRTA | DCSR_IE | DCSR_RUN;
		} else {
			/* buffer B */
			ClrDCSR0 = DCSR_DONEB | DCSR_STRTB;
			DBSB0 = ssp_audio_state.dmastate[1].phys;
			DBTB0 = towrite;
			ssp_audio_state.dmastate[1].in_use = 1;
			SetDCSR0 = DCSR_STRTB | DCSR_IE | DCSR_RUN;
		}
	};

	return numwritten;
}




static int ssp_audio_flush(struct file *file)
{
	int minor = MINOR(file->f_dentry->d_inode->i_rdev);

	if(minor != ssp_audio_state.minor_device)
		return -EINVAL;

	while(ssp_audio_state.dmastate[0].in_use || 
	      ssp_audio_state.dmastate[1].in_use) {
		/* wait for any buffer to be flushed */
		if(debug)
			printk(KERN_INFO MODULE_NAME ": waiting for buffers to be flushed\n");

		interruptible_sleep_on(&ssp_audio_state.dmawaitq);

		/* check for signals */
		if(signal_pending(current) &&
		   (ssp_audio_state.dmastate[0].in_use || 
		    ssp_audio_state.dmastate[1].in_use))
			return -EINTR;
		else
			return 0;
	};

	return 0;
}




static int ssp_audio_ioctl(struct inode *inode, struct file *file,
			   uint cmd, ulong arg)
{
	int val;
	int minor = MINOR(file->f_dentry->d_inode->i_rdev);

	if(debug)
		printk(KERN_INFO MODULE_NAME 
		       ": ioctl(): command = %2d, arg = 0x%08x\n",
		       _IOC_NR(cmd), arg ? *(int *)arg : 0);

	if(minor != ssp_audio_state.minor_device)
		return -EINVAL;

	switch(cmd)
	{
	case OSS_GETVERSION:
		return put_user(SOUND_VERSION, (int *)arg);
		break;

	case SNDCTL_DSP_RESET:
		/* FIXME: should stop all output */
	
		/* re-initialize parameters */
		reset_ssp_audio_state();
		return 0;
	
	case SNDCTL_DSP_SYNC:
		return ssp_audio_flush(file);

	case SNDCTL_DSP_SPEED: /* set sample rate */
		get_user_ret(val, (int *)arg, -EFAULT);
		
		if(val != AUDIO_RATE_DEFAULT)
			return -EINVAL;

		ssp_audio_state.rate = val;
		return 0;

	case SNDCTL_DSP_CHANNELS:
	case SNDCTL_DSP_STEREO: /* set stereo or mono channel */
		get_user_ret(val, (int *)arg, -EFAULT);

		if(val != AUDIO_CHANNELS_DEFAULT)
			return -EINVAL;

		ssp_audio_state.channels = val;
		return 0;

	case SNDCTL_DSP_SETFMT:
		get_user_ret(val, (int *)arg, -EFAULT);

		if(val == AFMT_QUERY) {
			return put_user(AUDIO_FORMAT_DEFAULT, (int *)arg);
		} else if(val == AUDIO_FORMAT_DEFAULT) {
			ssp_audio_state.format = val;
			return 0;
		} else {
			return -EINVAL;
		}

	case SNDCTL_DSP_GETBLKSIZE:
		return put_user(ssp_audio_state.blocksize, (int *)arg);

	case SNDCTL_DSP_GETFMTS:
		return put_user(AUDIO_FORMAT_DEFAULT, (int *)arg);

	case SNDCTL_DSP_GETOSPACE:
		/* FIXME: return available space */
		return put_user(AUDIO_BLOCKSIZE_DEFAULT, (int *)arg);

	case SOUND_PCM_READ_RATE:
		return put_user(ssp_audio_state.rate, (int *)arg);
		
	case SOUND_PCM_READ_CHANNELS:
		return put_user(ssp_audio_state.channels, (int *)arg);

	case SOUND_PCM_READ_BITS:
		return put_user(ssp_audio_state.format, (int *)arg);
	 
	default:
		return -EINVAL;
		break;
	}

	/* not reached */
	return -EINVAL;
}




static int ssp_audio_open(struct inode *inode, struct file *file)
{
	int minor = MINOR(inode->i_rdev);
	int rv;

	MOD_INC_USE_COUNT;

	if(debug)
		printk(KERN_INFO MODULE_NAME ": open()\n");

	/* sanity check */
	if(minor != ssp_audio_state.minor_device) {
		MOD_DEC_USE_COUNT;
		return -ENODEV;
	}

	/* we're actually a write-only device */
	if(! (file->f_mode & FMODE_WRITE)) {
		MOD_DEC_USE_COUNT;
		return -EPERM;
	}

	/* if the device already open? */
	if(ssp_audio_state.refcount > 0) {
		MOD_DEC_USE_COUNT;
		return -EBUSY;
	}

	/* initialize parameters */
	rv = init_ssp_audio_dma();
	if(rv < 0) {
		MOD_DEC_USE_COUNT;
		return rv;
	}

	/* we're using it */
	ssp_audio_state.refcount ++;

	/* reset */
	reset_ssp_audio_state();

	return 0;
}




static int ssp_audio_release(struct inode *inode, struct file *file)
{
	int minor = MINOR(file->f_dentry->d_inode->i_rdev);

	if(debug)
		printk(KERN_INFO MODULE_NAME ": release()\n");

	if(minor != ssp_audio_state.minor_device)
		return -EINVAL;

	/* flush all DMA buffers */
	while(ssp_audio_flush(file) != 0);

	/* cleanup DMA buffer */
	exit_ssp_audio_dma();

	ssp_audio_state.refcount --;

	MOD_DEC_USE_COUNT;
	return 0;
}




static struct file_operations ssp_audio_fops =
{
	write:   ssp_audio_write,
	ioctl:   ssp_audio_ioctl,
	open:    ssp_audio_open,
	flush:   ssp_audio_flush,
	release: ssp_audio_release,
};




static int __init init_sa1100_ssp(void)
{
	int rv;

	/* claim interrupt handler */
	rv = request_irq(AUDIO_IRQ, ssp_audio_irq, SA_INTERRUPT, 
			 MODULE_NAME, NULL);
	if(rv < 0) {
		printk(KERN_ERR MODULE_NAME ": couldn't get interrupt\n");
		return rv;
	}

	
	/* initialize waitqueue */
	init_waitqueue_head(&ssp_audio_state.dmawaitq);

	/* setup I/O */
	if(machine_is_lart()) {
		/* LART has the SSP port rewired to GPIO 10-13, 19 */
		/* alternate functions for the GPIOs */
		GAFR |= ( GPIO_SSP_TXD | GPIO_SSP_RXD | GPIO_SSP_SCLK | 
			  GPIO_SSP_SFRM | GPIO_SSP_CLK );
		
		/* Set the direction: 10, 12, 13 output; 11, 19 input */
		GPDR |= ( GPIO_SSP_TXD | GPIO_SSP_SCLK | GPIO_SSP_SFRM );
		GPDR &= ~ ( GPIO_SSP_RXD | GPIO_SSP_CLK );
    
		/* enable SSP pin swap */
		PPAR |= PPAR_SPR;
	} else {
		printk(KERN_ERR MODULE_NAME ": no support for this SA-1100 design!\n");
		printk(KERN_ERR MODULE_NAME ": edit " __FILE__ ":%d\n", __LINE__);
		free_irq(AUDIO_IRQ, NULL);
		return -ENODEV;
	}
	
	/* init SSP port */
	/* FIXME: disable MCP not necessary! */
	Ser4MCCR0 &= ~MCCR0_MCE;

	/* turn on the SSP */
	Ser4SSCR0 = (SSCR0_DataSize(16) | SSCR0_TI | SSCR0_SerClkDiv(2) |
		     SSCR0_SSE);
	Ser4SSCR1 = (SSCR1_SClkIactL | SSCR1_SClk1P | SSCR1_ExtClk);

	/* initialize DMA */
	ClrDCSR0 = (DCSR_DONEA| DCSR_DONEB | DCSR_STRTA | DCSR_STRTB |
		    DCSR_IE | DCSR_ERROR | DCSR_RUN);
	/* according to 11.12.7.3, the DMA must be set to a
	 * burst-of-four half-words, but DDAR_Ser4SSPWr already
	 * includes that.
	 */
	DDAR0 = (DDAR_Ser4SSPWr | DDAR_LtlEnd);

	/* reserve I/O regions */
	request_region((unsigned long)&Ser4SSCR0, 0x18, MODULE_NAME);
	request_region((unsigned long)&DDAR0, 0x40, MODULE_NAME " (dma)");
	
	/* register dsp device */
	ssp_audio_state.minor_device = register_sound_dsp(&ssp_audio_fops, -1);
	
	if(ssp_audio_state.minor_device < 0) {
		printk(KERN_ERR MODULE_NAME 
		       ": couldn't register DSP device!\n");
		release_region((unsigned long)&Ser4SSCR0, 0x18);
		release_region((unsigned long)&DDAR0, 0x40);
		free_irq(AUDIO_IRQ, NULL);
		return -ENODEV;
	}

	if(debug)
		printk(KERN_INFO MODULE_NAME 
		       ": minor number = %d\n", ssp_audio_state.minor_device);

	printk(KERN_INFO MODULE_NAME ": check = %d, pvalue = 0x%08x, nvalue = 0x%08x\n", 
	       check, pvalue, nvalue);
	if(check == 2)
		printk(KERN_INFO MODULE_NAME ": block: sending 0x%08x and 0x%08x\n",
		       ((((unsigned int)pvalue)<< 16) | (((unsigned int)pvalue) & 0xffff)),
		       ((((unsigned int)nvalue)<< 16) | (((unsigned int)nvalue) & 0xffff)));


	return(0);
}




static void __exit cleanup_sa1100_ssp(void)
{
	/* unregister dsp device */
	unregister_sound_dsp(ssp_audio_state.minor_device);

	/* release I/O regions */
	release_region((unsigned long)&Ser4SSCR0, 0x18);
	release_region((unsigned long)&DDAR0, 0x40);

	/* free interrupt line */
	free_irq(AUDIO_IRQ, NULL);
}




module_init(init_sa1100_ssp);
module_exit(cleanup_sa1100_ssp);

MODULE_AUTHOR("Erik Mouw");
MODULE_DESCRIPTION("SA-1100 SSP audio driver");
MODULE_PARM(debug, "i");
MODULE_PARM_DESC(debug, "Enable debug mode");
MODULE_PARM(check, "i");
MODULE_PARM_DESC(check, "Enable check mode (1 = constant, 2 = block, 3 = sawtooth)");
MODULE_PARM(pvalue, "i");
MODULE_PARM_DESC(pvalue, "positive value for check");
MODULE_PARM(nvalue, "i");
MODULE_PARM_DESC(nvalue, "negative value for check");

EXPORT_NO_SYMBOLS;
