/*
 *      usbd_h3600_dma.c  --
 *
 *      Used to wrap the itsy dma functions
 *	for use with the iPAQ USB driver files usbd_h3600_xxx.c  --
 *
 *      Copyright (C) Compaq Computer Corporation, 2000
 *	Author Charles Flynn	August 2000
 *	Uses some code from dma-sa1100.c
 *      Copyright (C) 2000 Nicolas Pitre
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/sched.h>
#include <linux/init.h>

#include <asm/dma.h>
#include <asm/fiq.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/hardware.h>

/*
	TODO
	This module includes /arch/arm/kernel/SA-1000.h whuch defines
	the DMA & UDC bit twiddling stuff.
	Unfortuneately usb_h3600_ctl.h conflicts with SA-1100.h
	so dont include it from here.

*/

/* longword offsets into DMA register list (SA-1100 manual, 11.6.3) */
#define DAR     0
#define CSR_SET 1
#define CSR_CLR 2
#define CSR_GET 3
#define BSA     4
#define BTA     5
#define BSB     6
#define BTB     7

#define MAX_USB_CHANNELS	2

static dma_regs_t * DMA[MAX_USB_CHANNELS];  /* base address of DMA registers */
static int buffer_state[MAX_USB_CHANNELS] = {0,0};

/* DMA- this is where the DMA structure is setup */
int udc_init_dma( dmach_t channel )
{
	dma_regs_t * pDMA;
	if ( channel >= MAX_USB_CHANNELS )
	{
		printk("udc_init_dma:Bad channel\n");
		return -1 ;
	}
	pDMA = (dma_regs_t*)io_p2v(_DDAR(channel));
	DMA[channel]=pDMA;

	/* Clear the status register */
	pDMA->ClrDCSR = (DCSR_DONEA|DCSR_DONEB|DCSR_STRTA|DCSR_STRTB|
		 		DCSR_IE|DCSR_ERROR|DCSR_RUN);

	return 0;
}

int set_dma_mode_sa1100(dmach_t channel, int mode)
{
	dma_regs_t * pDMA;
	unsigned int dar;
	if ( channel >= MAX_USB_CHANNELS )
	{
		printk("set_dma_mode:Bad channel\n");
		return -1;
	}
	pDMA = DMA[channel];

	/*
		Bursts of 8 , each 8 bits ( 1 byte).
		This must be less than the FIFO size
	*/
	dar = ( DDAR_Brst8 | DDAR_8BitDev);

	/* this is completely usb specific...  */

	switch (mode)
	{
	/* case DMA_UDC_RECEIVE: */
	case 0x80000A15	:
		dar |= DDAR_DevRd | DDAR_Ser0UDCRc | DDAR_DevAdd(_Ser0UDCDR);
		break;
	/* case DMA_UDC_TRANSMIT: */
	case 0x80000A04:
		dar |= DDAR_DevWr | DDAR_Ser0UDCTr | DDAR_DevAdd(_Ser0UDCDR);
		break;
	default:
		printk("set_dma_mode_sa1100: Unknown mode\n");
		return -1;
	}

	pDMA->DDAR = dar;

	return 0;
}

/*
	This will return the DMA address for the given channel
	TODO physical addresses should be long
	see /arch/arm/mm/ioremap.c
*/
void * get_dma_address_sa1100(dmach_t channel)
{
	dma_regs_t * pDMA;
	if ( channel >= MAX_USB_CHANNELS )
	{
		printk("get_dma_address:Bad channel\n");
		return NULL;
	}
	pDMA = DMA[channel];

	/* State saved previously from enable_dma() */
	return ( buffer_state[channel] ) ?
		(void *)pDMA->DBSA : (void *)pDMA->DBSB;
}

int _sa1100_start_dma( dma_regs_t *DMA, void *dma_ptr, int size )
{
	int status;
	int use_bufa;

	status = DMA->RdDCSR;

	/* If both DMA channels are started, there's nothing else we can do. */
	if( (status & DCSR_STRTA) && (status & DCSR_STRTB) ){
		printk(" busy\n");
		return -EBUSY;
	}

	use_bufa = ( ((status & DCSR_BIU) && (status & DCSR_STRTB)) ||
		     (!(status & DCSR_BIU) && !(status & DCSR_STRTA)) );

	/* NOTE THE DCSR_IE isn't set - since there is no ISR defined */
	if( use_bufa ){
		DMA->ClrDCSR = DCSR_DONEA|DCSR_STRTA;
		DMA->DBSA = dma_ptr;
		DMA->DBTA = size;
		DMA->SetDCSR = DCSR_STRTA|DCSR_RUN;
	}else{
		DMA->ClrDCSR = DCSR_DONEB|DCSR_STRTB;
		DMA->DBSB = dma_ptr;
		DMA->DBTB = size;
		DMA->SetDCSR = DCSR_STRTB|DCSR_RUN;
	}

	return 0;
}

/*
	This function replaces 
		set_dma_count()
		set_dma_addr()
		enable_dma()
	Enter with DMA registers,
		pointer you a physical contiguous area of memory.
		size of buffer to DMA'd
*/
int enable_dma_sa1100(dmach_t channel,void * ptr, unsigned size)
{
	int status, use_bufa;
	dma_regs_t * pDMA;

	/*
		Did this to ensure the DMA DONEA?B bits set so we
		were able to safely stop DMA otherwise disabling
		DMA may have unpredictable results.
		TODO remove magic number dmachn_rx is zero
	*/

	if( channel == 0 )
		size -= 8;
	
	if ( channel >= MAX_USB_CHANNELS )
	{
		printk("enable_dma:Bad channel\n");
		return -1;
	}

	/* KLUDGE: store the last buffer enabled */
	pDMA = DMA[channel];
	status = pDMA->RdDCSR;
	use_bufa = ( ((status & DCSR_BIU) && (status & DCSR_STRTB)) ||
		     (!(status & DCSR_BIU) && !(status & DCSR_STRTA)) );

	buffer_state[channel]=use_bufa;


	/* TODO TODO 
		Maybe we can call the resident sa1100_start_dma
		however it sets the DCSR_IE bit and we dont want
		DMA completion trying to fire a non-existant IRQ!
	*/
	if( _sa1100_start_dma( pDMA , ptr, size ) != 0 )
	{
		printk("DMA busy\n");
		return -1;
	}

	return 0;
}

/* Replaces disable_dma()
	Clearing a running DMA causes unpredictable results
	so wait for it to stop
	RETURN
	0	- DMA successfully disabled
	1	- Unable to disable DMA
*/
int disable_dma_sa1100 (dmach_t channel)
{
	int status;
	dma_regs_t * pDMA;
	
	if ( channel >= MAX_USB_CHANNELS )
	{
		printk("enable_dma:Bad channel\n");
		return -1;
	}

	pDMA = DMA[channel];
	status = pDMA->RdDCSR;
	if( !(status & (DCSR_DONEA|DCSR_DONEB)) )
	    printk("WARNING : DMA DONEA/B not SET\n");
	/* need to do this for buffers fragments 0 < x < 64 */
	pDMA->ClrDCSR = 0x7f;
	return 0;
}

dma_regs_t * get_dma_regbase( dmach_t channel)
{
	return ( channel < MAX_USB_CHANNELS ) ? DMA[channel] : NULL;
}

int get_dma_residue_sa1100(dmach_t channel)
{
	dma_regs_t * pDMA;
	
	if ( channel >= MAX_USB_CHANNELS )
	{
		printk("dma_get_xfer:Bad channel\n");
		return -1;
	}

	pDMA = DMA[channel];
	return ( buffer_state[channel] ) ?
		(unsigned int)pDMA->DBTA : (unsigned int)pDMA->DBTB;


	return 0;
}
