/****************************************************************************/
/* Copyright 2000 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * bootldr file for Compaq Personal Server Bootloader
 *
 */

/*
 * Maintainer: Jamey Hicks (jamey@crl.dec.com)
 * Original Authors: Edwin Foo, Jamey Hicks, Dave Panariti, Mike Schexnaydre, Chris Joerg
 * June 29, 2000 - save commands and xmodem send added.
 *                             - George France <france@crl.dec.com>
 * July  3, 2000 - add commands to allow the baudrate to be changed.
 *                             - George France <france@crl.dec.com>
 * July  5, 2000 - extended the range of baudrate change.
 *                             - George France <france@crl.dec.com>
 * July  5, 2000 - added PF_LOGICAL.
 *                             - George France <france@crl.dec.com>
 * July  6, 2000 - added display command.
 *                             - George France <france@crl.dec.com>
 * July 31, 2000 - add Architecture field to boot loader.
 *                             - George France <france@crl.dec.com>
 * Aug  04, 2000 - made the cached and uncached flash size the 
 *                 same as the actual flash size.
 *                             - George France <france@crl.dec.com>
 * Aug  04, 2000 - Auto protected the boot sector after the boot
 *                 loader is loaded.
 *                             - George France <france@crl.dec.com> 
*/

static char bootprog_name[] = "Compaq OHH BootLoader";
static char bootprog_date[] = DATE;

static char cvsversion[] = "$Id: bootldr.c,v 1.28 2000/09/12 01:53:02 jamey Exp $";

#define USE_XMODEM
#define BOOT_ALTKERNEL

#include "bootldr.h"
#include "btpci.h"
#include "btflash.h"
#include "btusb.h"
#include "heap.h"
#include "xmodem.h"
#include <asm-arm/setup.h>
#include "sa1100.h"
#include "bsdsum.h"
#include "architecture.h"

#ifdef BZIPPED_KERNELS
#define BZ_NO_STDIO
#include "bzip/bzlib.h"
#endif

#define	DEB(s)	putstr(s) 

typedef unsigned long u_int32_t;
typedef unsigned long pd_entry_t;
long paramoldvalue;
#include "cyclone_boot.h"
		
#define ROUNDUP(v,szpwr2) (((v)+((szpwr2)-1))&~(szpwr2-1))

#if 0
void reboot(void);
#endif
void enableMMU(void);
void flushTLB(void);
void writeBackDcache(unsigned long region);
unsigned long readCPR1(void);
unsigned long readCPR3(void);

/* Boot Loader Function Prototypes */
void print_help_on_commands(struct bootblk_command *commands);
void command_help(int argc, const char **argv);
void command_boot(int argc, const char **argv);
void command_boot_flash(int argc, const char **argv);
void command_boot_nfsroot(int argc, const char **argv);
void command_display(int argc, const char **argv );
void command_load(int argc, const char **argv);
void command_load_flash(int argc, const char **argv);
void command_load_altkernel(int argc, const char **argv);
void command_load_debugger(int argc, const char **argv);
void command_load_ram(int argc, const char **argv);
/* helper function */
static void command_load_flash_region(const char *regionName, unsigned long regionBase, unsigned long regionSize, int flags);
void command_load_bootldr(int argc, const char **argv);
void command_load_usercode(int argc, const char **argv);
void command_load_ramdisk(int argc, const char **argv);
void command_load_params(int argc, const char **argv);
void command_load_kernel(int argc, const char **argv);
void command_save(int argc, const char **argv);
void command_save_all(int argc, const char **argv);
void command_save_altkernel(int argc, const char **argv);
void command_save_bootldr(int argc, const char **argv);
void command_save_debugger(int argc, const char **argv);
void command_save_flash(int argc, const char **argv);
void command_save_kernel(int argc, const char **argv);
void command_save_params(int argc, const char **argv);
void command_save_ramdisk(int argc, const char **argv);
void command_save_usercode(int argc, const char **argv);
void command_save_world(int argc, const char **argv);
void command_boot_altkernel(int argc, const char **argv);
void command_peek(int argc, const char **argv);
void command_peek_ram(int argc, const char **argv);
void command_poke(int argc, const char **argv);
void command_poke_ram(int argc, const char **argv);
void command_breakpoint(int argc, const char **argv);
void command_qflash(int argc, const char **argv);
void command_eflash(int argc, const char **argv);
void command_pflash(int argc, const char **argv);
#if 0
void command_reboot(int argc, const char **argv);
#endif
void command_physaddr(int argc, const char **argv);
void command_call(int argc, const char **argv);
void command_set(int argc, const char **argv);
struct bootblk_param *get_param(const char *name);
int get_param_value(const char *name, void *value_p);
void command_flash_type(int argc, const char **argv);
void command_params(int argc, const char **argv);
void command_params_show(int argc, const char **argv);
void command_params_eval(int argc, const char **argv);
void command_params_save(int argc, const char **argv);
void testCirrusCardbusBridge(int argc, const char **argv);

void parseargs(char *argstr, int *argc_p, char **argv);
void unparseargs(char *argstr, int argc, const char **argv);


byte awaitkey(unsigned long delay, int *error_p);
void bootmenu(void);

void monitorConfigureMMU(void);

#ifdef CONFIG_GZIP      
int gunzip_region(char*   src,    char*   dst,
		  long len, const char *name);
#endif

/* Boot Loader Variables/Constants */

static struct bootblk_command commands[] = {
  { "?",      command_help, "?" },
  { "help",   command_help, "help <command>" },
  { "help",   command_help, "<command> help" },
  { "boot",   command_boot, "boot [flash|nfsroot|altkernel]" },
  { "display",command_display, "display" },
  { "load",   command_load, "load [kernel | ramdisk | bootldr | params | usercode | ram]" },
  { "save",   command_save, "save [all | bootldr | debugger | flash | kernel | ramdisk | params | usercode | world | altkernel]" },
  { "peek",   command_peek, "peek ram|flash|int|short|byte <addr>" },
  { "poke",   command_poke, "poke ram|flash|int|short|byte <addr>" },
#ifdef CONFIG_BITSY
  { "breakpoint", command_breakpoint, "breakpoint" },
#endif
  { "qflash", command_qflash, "qflash [cfi|id] <waddr> -- query flash" },
  { "eflash", command_eflash, "eflash <sectoraddr>|chip -- erase sector or chip" },
  { "pflash", command_pflash, "pflash <addr> <len> 0/1 (1 -> protect, 0 -> unprotect all!) -- protect address range" },
#if 0
  { "reboot", command_reboot, "reboot" },
#endif
  { "physaddr", command_physaddr, "physaddr <vaddr> -- returns <paddr>" },
  /*  { "testusb", testUSB, "testusb" },*/
  { "call",   command_call, "call <addr> args" },
  { "jump",   command_call, "jump <addr>" },
  { "set",    command_set, "set <param> <value>" },
  { "show",    command_params_show, "show [<param>]" },
  { "evalparams", command_params_eval, "evalparams"},
  { "params", command_params, "params [eval|show|save|reset]"},
  { "flash_type", command_flash_type, "flash_type <flashtype>"},
#if 0
  { "cardbus", testCirrusCardbusBridge, "cardbus" },
#endif
  { NULL,     NULL, NULL }
};

static struct bootblk_command boot_commands[] = {
  { "flash",  command_boot_flash, "boot flash [bootargs ...]" },
#ifdef BOOT_ALTKERNEL
  { "altkernel", command_boot_altkernel, "boot altkernel [bootargs ...]" },
#endif
  { "nfsroot",   command_boot_nfsroot, "boot nfsroot [bootargs ...]" },
  { NULL,     NULL, NULL }
};

static struct bootblk_command load_commands[] = {
  { "kernel", command_load_kernel, "load kernel" },
  { "bootldr", command_load_bootldr,"load bootldr" },
#ifdef BOOT_ALTKERNEL
  { "altkernel", command_load_altkernel, "load altkernel (alternate kernel)" },
#endif
  { "usercode", command_load_usercode,"load usercode" },
  { "ramdisk", command_load_ramdisk,"load ramdisk" },
  { "params",  command_load_params,"load params" },
  { "ram",     command_load_ram, "load ram <dstaddr>" },
  { "flash",   command_load_flash, "load flash <dstaddr>" },
  { "debugger", command_load_debugger, "load debugger" },
  { NULL,     NULL, NULL }
};

static struct bootblk_command save_commands[] = {
  { "all",       command_save_all,       "save all" },
  { "altkernel", command_save_altkernel, "save altkernel" },
  { "bootldr",   command_save_bootldr,   "save bootldr" },
  { "debugger",  command_save_debugger,  "save debugger" },
  { "flash",     command_save_flash,     "save flash" },
  { "kernel",    command_save_kernel,    "save kernel" },
  { "params",    command_save_params,    "save params" },
  { "ramdisk",   command_save_ramdisk,   "save ramdisk" },
  { "usercode",  command_save_usercode,  "save usercode" },
  { "world",     command_save_world,     "save world" },
  { NULL,        NULL,                   NULL }
};

static struct bootblk_command peek_commands[] = {
  { "ram",    command_peek_ram, "peek ram <addr> reads 32 bits" },
  { "byte",   command_peek_ram, "peek byte <addr> reads 8 bits" },
  { "short",   command_peek_ram, "peek short <addr> reads 16 bits" },
  { "int",    command_peek_ram,  "peek int <addr> reads 32 bits" },
  { "flash",  command_peek_ram, "peek flash <offset>" },
  { "gpio",   command_peek_ram, "peek gpio <offset>" },
  { NULL,     NULL, NULL }
};

static struct bootblk_command poke_commands[] = {
  { "ram",    command_poke_ram, "poke ram <addr> <dword>" },
  { "byte",   command_poke_ram, "poke byte <addr> <byte> writes 8 bits" },
  { "short",   command_poke_ram, "poke short <addr> <word> writes 16 bits" },
  { "int",   command_poke_ram, "poke int <addr> <dword> writes 32 bits" },
  { "flash",  command_poke_ram, "poke flash <offset> <dword>" },
  { "gpio",  command_poke_ram, "poke gpio <offset> <dword>" },
  { NULL,     NULL, NULL }
};

static struct bootblk_command params_commands[] = {
  { "eval",  command_params_eval, "params eval" },
  { "show",  command_params_show, "params show" },
  { "save",  command_params_save, "params save" },
  { "reset",  command_params,     "params reset" },
  { NULL,     NULL, NULL }
};

static struct bootblk_param bootldr_params[] = {
#define UPDATE_BAUDRATE
#ifdef UPDATE_BAUDRATE
    { "baudrate", PT_INT, PF_DECIMAL, UART_BAUD_RATE, update_baudrate }, 
#endif
    { "os", PT_STRING, PF_STRING, (long)"autoselect", NULL },  /* "linux", "netbsd", "autoselect" */
    { "boot_type", PT_STRING, PF_STRING, (long)"flash", NULL },
    { "download_kernel", PT_INT, PF_HEX, 0, NULL },
    { "kernel_in_ram", PT_INT, PF_HEX, 0, NULL }, /* set this to the address of where kernel is loaded in ram (0xC0008000) */
    { "force_unzip", PT_INT, PF_HEX, 0, NULL }, 
    { "noerase", PT_INT, PF_HEX, 0, NULL }, 
    { "override", PT_INT, PF_HEX, 0, NULL }, 
#ifdef CONFIG_SKIFF
    { "entry", PT_INT, PF_HEX, 0x00000000, NULL }, 
#else
    { "entry", PT_INT, PF_HEX, 0xC0000000, NULL }, 
#endif
    { "copy_ramdisk", PT_INT, PF_HEX, 0, NULL },
    { "dram_size", PT_INT|PT_READONLY, PF_HEX, DRAM_SIZE, NULL },
    { "dcache_enabled", PT_INT, PF_HEX, 0x1, update_dcache_enabled },
    { "icache_enabled", PT_INT, PF_HEX, 0x1, update_icache_enabled },
    { "memc_ctrl_reg", PT_INT, PF_HEX, 0x110c, NULL },
#ifdef CONFIG_PCI
    { "mem_fclk_21285", PT_INT|PT_READONLY, PF_DECIMAL, 48000000, NULL },
    { "maclsbyte", PT_INT, PF_HEX, 0xFF, program_all_eeprom },
#endif
    { "serial_number", PT_INT|PT_READONLY, PF_HEX, 0xFF, set_serial_number },
    { "system_rev", PT_INT|PT_READONLY, PF_HEX, 0x01, set_system_rev },
#ifdef CONFIG_SKIFF
    { "linuxargs", PT_STRING, PF_STRING, (long)" root=/dev/ram initrd ramdisk_size=8192", NULL },
#else
    { "linuxargs", PT_STRING, PF_STRING, (long)" root=/dev/ram initrd ramdisk_size=8192", NULL },
#endif
    { "hostname", PT_STRING, PF_STRING, 0, NULL },
    { "ipaddr", PT_STRING, PF_STRING, 0, NULL },
    { "gateway", PT_STRING, PF_STRING, 0, NULL },
    { "netmask", PT_STRING, PF_STRING, 0, NULL },
    { "nfs_server_address", PT_STRING, PF_STRING, 0, NULL },
    { "nfsroot", PT_STRING, PF_STRING, 0, NULL },
    { "xmodem", PT_INT, PF_HEX, 1, NULL },
    { "verbose", PT_INT, PF_HEX, 0, NULL },
    { NULL, PT_NONE, PF_HEX, 0, NULL }
};

struct bootblk_param *get_param(const char *name)
{
   struct bootblk_param *param = bootldr_params;
   while (param->name != NULL) {
      int namelen = strlen(param->name);
      if (strncmp(name, param->name, namelen) == 0
          && name[namelen] == 0) {
         return param;
      }
      param++;
   }
   putstr("get_param: could not find parameter "); putstr(name); putstr("\r\n");
   return NULL;
}

int get_param_value(const char *name, void *value_p)
{
   struct bootblk_param *param = get_param(name);
   if (param != NULL) {
      if (value_p != NULL) 
         *(long *)value_p = param->value;
      return 0;
   } else {
      return -1;
   }  
}


#ifdef CONFIG_BITSY
/* The EGPIO is a write only control register at physical address 0x49000000
 * See the hardware spec for more details.
 */

static int egpio = EGPIO_BITSY_RS232_ON;
static void set_egpio(int mask)
{
  egpio |= mask;
  *(volatile int *)BITSY_EGPIO = egpio;
}
static void clr_egpio(int mask)
{
  egpio &= ~mask;
  *(volatile int *)BITSY_EGPIO = egpio;
}

#define SET_VPPEN() set_egpio(EGPIO_BITSY_VPEN)
#define CLR_VPPEN() clr_egpio(EGPIO_BITSY_VPEN)

#else

/* nothing to do here on Skiff */
#define SET_VPPEN() 
#define CLR_VPPEN() 

#endif



/*
 * this variable MUST be initialized explicitly to zero to prevent its
 * placement in the BSS.
 */
enum {
    mmue_enabled = 99,
    mmue_notenabled = 101
};

byte mmuEnabled = mmue_notenabled; /* we set this after we call enableMMU() */
#define isMmuEnabled()	(mmuEnabled == mmue_enabled)
#define setMmuEnabled()	(mmuEnabled =  mmue_enabled)

#ifdef NoLibC
/*
 * very simple memcpy to blast bits around from place to place
 */
void *memcpy(char *dst, const char *src, long n)
{
  if (   (((dword)dst)&3) == 0
	 && (((dword)src)&3) == 0
	 && (n&3) == 0 ) {
    unsigned long *ca,*cb;
    for (ca = (unsigned long *)dst, cb = (unsigned long *)src; n > 0;n -= 4)
      *ca++ = *cb++;
  } else {
    byte *ca,*cb;
    for (ca = (byte *)dst, cb = (byte *)src; n-- > 0;)
      *ca++ = *cb++;
  }
  return dst;
}


/*
 * memset routine: fills first n bytes of dst with value c
 */
void memset(void *dst, char c, unsigned long n)
{
  byte *ca;
  for (ca = (byte *)dst; n-- > 0; )
    *ca++ = c;
}

int strlen(const char *s)
{
   int l = 0;
   while (*s++ != 0)
      l++;
   return l;
}

char *strcat(char *dest, const char *src)
{
   char c;
   int len = strlen(dest);
   dest += len;
   while (( *dest++ = *src++) != 0) {
   }
}

int strncmp(const char *s1, const char *s2, int len)
{
   int i = 0;
   int c1, c2;
   if (s1 == NULL || s2 == NULL)
      return -1;
   for (c1 = s1[i], c2 = s2[i]; i < len; c1 = s1[i], c2 = s2[i], i++) {
      if (c1 < c2)
         return -1;
      else if (c1 > c2)
         return 1;
      else
         continue;
   }
   return 0;
}

int strcmp(const char *s1, const char *s2)
{
   int l1 = strlen(s1);
   int l2 = strlen(s2);
   int l = (l1 < l2) ? l1 : l2;
   int result = strncmp(s1, s2, l);
   if (l == 0 && (l1 != l2)) {
      if (l1 < l2)
         return -1;
      else
         return 1;
   } else {
      return result;
   }
}

char *strchr(const char *s, int c)
{
   char sc;
   while (((sc = *s) != c) && sc != 0) {
     s++;
   }
   if (sc == c)
      return (char *)s;
   else
      return NULL;
}
#endif /* NoLibC */

void putc(char c) 
{
  /* wait for space in the TX FIFO */
#ifdef CONFIG_SKIFF
  while (CSR_READ_BYTE(UARTFLG_REG) & UART_TX_FIFO_BUSY);
  CSR_WRITE_BYTE(UARTDR_REG,c);
#elif defined(CONFIG_BITSY)
  while (!((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_TNF)); /* wait until TX FIFO not full */
  *(byte *)SA1100_UART3_UTDR = c;
#else
#error no definition for putc for this architecture
#endif
}

void putstr(const char *str)
{
   extern void *Ser3Base;
   if (str == NULL)
      return;
   while (*str != '\0') {
#ifdef CONFIG_BITSY
         PrintChar(*str, Ser3Base) ;
#else
         putc(*str) ;
#endif
      str++ ;
   }
}

#ifdef ZBSS
/*
 * zero bss (using no bss vars!)
 */
void
zbss(void)
{
    char*	    start;
    unsigned	    len;
    aout_header*    hdrp;
    
    hdrp = (aout_header*)FLASH_BASE;
    start = (char*)hdrp + hdrp->a_text + hdrp->a_data;
    
    len = hdrp->a_bss;
    if (len) {
	memset(start, 0x00, len);
    }
    
}
#endif /* ZBSS */

extern char HEX_TO_ASCII_TABLE[16];

void dwordtodecimal(char *buf, unsigned long x)
{
  int i = 0;
  int j = 0;
  char localbuf[16];

  while (x > 0) {
    unsigned long rem = x % 10;
    localbuf[i++] = HEX_TO_ASCII_TABLE[rem];
    x /= 10;
  }
  /* now reverse the characters into buf */
  while (i > 0) {
    i--;
    buf[j++] = localbuf[i];
  }
}

void binarytohex(char *buf, long x, int nbytes)
{
  int i;
  int s = 4*(2*nbytes - 1);
  if (HEX_TO_ASCII_TABLE[0] != '0')
     putstr("HEX_TO_ASCII_TABLE corrupted\n");
  for (i = 0; i < 2*nbytes; i++) {
    buf[i] = HEX_TO_ASCII_TABLE[(x >> s) & 0xf];
    s -= 4;
  }
  buf[2*nbytes] = 0;
}

byte strtoul_err;
unsigned long strtoul(const char *str, char **endptr, int requestedbase)
{
   unsigned long num = 0;
   int i;
   char c;
   byte digit;
   int base = 10;
   int nchars = 0;
   int leadingZero = 0;

   strtoul_err = 0;

   while ((c = *str) != 0) {
      if (nchars == 0 && c == '0') {
         leadingZero = 1;
         if (0) putLabeledWord("strtoul: leadingZero nchars=", nchars);
         goto step;
      } else if (leadingZero && nchars == 1) {
         if (c == 'x') {
            base = 16;
            if (0) putLabeledWord("strtoul: base16 nchars=", nchars);
            goto step;
         } else if (c = 'o') {
            base = 8;
            if (0) putLabeledWord("strtoul: base8 nchars=", nchars);
            goto step;
         }
      }
      if (0) putLabeledWord("strtoul: c=", c);
      if (c >= '0' && c <= '9') {
         digit = c - '0';
      } else if (c >= 'a' && c <= 'z') {
         digit = c - 'a' + 10;
      } else if (c >= 'A' && c <= 'Z') {
         digit = c - 'A' + 10;
      } else {
         strtoul_err = 3;
         return 0;
      }
      if (digit >= base) {
         strtoul_err = 4;
         return 0;
      }
      num *= base;
      num += digit;
   step:
      str++;
      nchars++;
   }
   return num;
}

void putLabeledWord(const char *msg, unsigned long value)
{
   char buf[9];
   binarytohex(buf,value,4);
   putstr(msg);
   putstr(buf);
   putstr("\r\n");
}

void putLabeledAddrWord(const char *msg, unsigned long *value)
{
   char buf[9];

   putstr(msg);
   binarytohex(buf, (unsigned long) value, 4);
   putstr(" *0x" );
   putstr(buf);
   putstr( " == " );
   binarytohex(buf,*value,4);
   putstr(buf);
   putstr("\r\n");
}


void putHexInt32(unsigned long value)
{
  char buf[9];
  binarytohex(buf,value,4);
  putstr(buf);
}

void putHexInt16(word value)
{
  char buf[9];
  binarytohex(buf,value,2);
  putstr(buf);
}

void putHexInt8(byte value)
{
  char buf[3];
  binarytohex(buf,value,1);
  putstr(buf);
}
  
static struct ebsaboot bootinfo = {
   BT_MAGIC_NUMBER,	/* boot info magic number */
   0x20000000,		/* virtual addr of arg page */
   0x20000000,		/* physical addr of arg page */
   NULL,		/* kernel args string pointer */
   (pd_entry_t *)MMU_TABLE_START,	/* active L1 page table */
   0,		        /* start of physical memory */
   DRAM_SIZE,		/* end of physical memory */
   SZ_2M+SZ_1M,         /* start of avail phys memory */
   48000000,		/* fclk frequency */
   UART_BAUD_RATE,    	/* baudrate of serial console */
   1 			/* n stop bits */
};

void print_banner(void)
{
  long armrev = 0;
  long corelogicrev = 0;
  long cpsr = 0;
  __asm__("mrc p15, 0, %0, c0, c0, 0" : "=r" (armrev));
  __asm__("mrs %0, cpsr" : "=r" (cpsr));
#ifdef CONFIG_SKIFF
  corelogicrev = *(volatile unsigned long *)(DC21285_ARMCSR_BASE + PCI_VENDOR_ID);
#endif

  putstr("\r\n");
  putstr(">> ");
  putstr(bootprog_name);
  putstr(", Rev ");
  putHexInt8(VERSION_MAJOR); putstr("."); putHexInt8(VERSION_MINOR); putstr("."); putHexInt8(VERSION_MICRO);
  putstr("\r\n");

  putstr(">> ");
  putstr(bootprog_date);
  putstr("\r\n");
  putLabeledWord(">>  ARM Processor Rev=", armrev);
#if CONFIG_SKIFF
  putLabeledWord(">>  Corelogic Rev=", corelogicrev);
  if (get_param("mem_fclk_21285") != NULL)
    putLabeledWord(">>  Corelogic fclk=", get_param("mem_fclk_21285")->value);
#endif
  putLabeledWord(">>  CPSR=", cpsr);

  putstr(">> (c) 2000 Compaq Cambridge Research Laboratory\r\n");
  putstr("Press Return to start the OS now, any other key for monitor menu\r\n");
}

/*
 * Reads and returns a character from the serial port
 *  - Times out after delay iterations checking for presence of character
 *  - Sets *error_p to UART error bits or -1 on timeout
 *  - On timeout, sets *error_p to -1 and returns 0
 */
byte awaitkey(unsigned long delay, int *error_p)
{
  unsigned long i = delay;
  byte c;
  int errors = 0;

  while (1) {
    i = delay;
#ifdef CONFIG_SKIFF
    while (((CSR_READ_BYTE(UARTFLG_REG) & UART_RX_FIFO_EMPTY)) && i)
	i--;
#else
    {
       while ((!((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_RNE)) && i)
          i--;
    }
#endif
    
    if (i) {
      errors = 0;
#ifdef CONFIG_SKIFF
      c = CSR_READ_BYTE(UARTDR_REG);
      errors = CSR_READ_BYTE(RXSTAT_REG);
#else
      c = *(volatile long *)SA1100_UART3_UTDR;
      errors = (*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_ERROR_MASK;
#endif
      if (!errors) /* if no receive errors found.. */
	break; /* get out of here */
    } else {
      c = 0; /* no one pressed a key. return a NULL and get out of here */
      errors = -1;
      break;
    }
  }
  if (error_p != NULL)
    *error_p = errors;
  return(c);
}

int getc_verbose_errors = 1;
int getc_errno = 0;
byte getc(void) {
   byte c, rxstat;
   getc_errno = 0; /* reset errno */
#ifdef CONFIG_SKIFF
   while ((CSR_READ_BYTE(UARTFLG_REG) & UART_RX_FIFO_EMPTY));
   /* must read UARTDR_REG before RXSTAT_REG */
   c =  CSR_READ_BYTE(UARTDR_REG);
   rxstat = CSR_READ_BYTE(RXSTAT_REG);
#elif defined(CONFIG_BITSY)
   while ((!((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_RNE)))
     /* spin wait */;
   c = *(volatile byte *)SA1100_UART3_UTDR;
   rxstat = (*(volatile byte *)SA1100_UART3_UTSR1) & SA1100_UTSR1_ERROR_MASK;
#else
#error no architecture defined for getc
#endif
   if (rxstat) {
      getc_errno = rxstat;
      if (getc_verbose_errors) {
         putLabeledWord("RXSTAT error: ", rxstat);
      }
   }
   return(c);
}


#define CMDBUFSIZE 256
char cmdBuffer[CMDBUFSIZE];
void getcmd(void) {
   byte curpos = 0; /* curpos position - index into cmdBuffer */
   byte c;
   byte noreturn = 1;

   /* first clear out the buffer.. */
   memset(cmdBuffer, 0, CMDBUFSIZE);

   /* here we go..*/

   while (noreturn) {
      c = getc();
      switch (c)
         {
	 case 0x08:
         case 0x06:
         case 0x07:
         case 0x7E:
         case 0x7F: /* backspace / delete */
            if (curpos) { /* we're not at the beginning of the line */
               curpos--;
               putc(0x08); /* go backwards.. */
               putc(' ');  /* overwrite the char */
               putc(0x08); /* go back again */
            }
            cmdBuffer[curpos] = '\0';
            break;
         case '\r':
         case '\n':
         case '\0':
            noreturn = 0;
            putc('\r');
            putc('\n');
            break;
         default:
            if (curpos < CMDBUFSIZE) {
               cmdBuffer[curpos] = c;
               /* echo it back out to the screen */
               putc(c);
               curpos++;
            }
            break;
         }
   }
   /*
     putstr("COMMAND: ");
     putstr(cmdBuffer);
     for (c=0;c<CMDBUFSIZE;c++) {
       putHexInt8(cmdBuffer[c]);
       putc(' ');
     }
     putstr("\r\n");
   */

}

int argc;

enum ParseState {
   PS_WHITESPACE,
   PS_TOKEN,
   PS_STRING,
   PS_ESCAPE
};

enum ParseState stackedState;

void parseargs(char *argstr, int *argc_p, char **argv)
{
  const char *whitespace = " \t";
  int argc = 0;
  char c;
  enum ParseState lastState = PS_WHITESPACE;
   
  /* tokenize the argstr */
  while ((c = *argstr) != 0) {
    enum ParseState newState;
      
    if (lastState == PS_ESCAPE) {
      newState = stackedState;
    } else if (lastState == PS_STRING) {
      if (c == '"') {
        newState = PS_WHITESPACE;
        *argstr = 0;
      } else {
        newState = PS_STRING;
      }
    } else if ((c == ' ') || (c == '\t')) {
      /* whitespace character */
      *argstr = 0;
      newState = PS_WHITESPACE;
    } else if (c == '"') {
      newState = PS_STRING;
      *argstr++ = 0;
      argv[argc++] = argstr;
    } else if (c == '\\') {
      stackedState = lastState;
      newState = PS_ESCAPE;
    } else {
      /* token */
      if (lastState == PS_WHITESPACE) {
        argv[argc++] = argstr;
      }      
      newState = PS_TOKEN;
    }

    lastState = newState;
    argstr++;
  }

  if (0) { 
    int i;
    putLabeledWord("parseargs: argc=", argc);
    for (i = 0; i < argc; i++) {
      putstr("   ");
      putstr(argv[i]);
      putstr("\r\n");
    }
  }

  argv[argc] = NULL;
  if (argc_p != NULL)
    *argc_p = argc;
}

void unparseargs(char *argstr, int argc, const char **argv)
{
   int i;
   for (i = 0; i < argc; i++) {
     if (argv[i] != NULL) {
       strcat(argstr, " ");
       strcat(argstr, argv[i]);
     }
   }
}

int parsecmd(struct bootblk_command *cmdlist, int argc, const char **argv)
{
   /* find the command name */
   const char *cmdname = argv[0];
   int cmdnum = 0;

   if (argc < 1)
      return -1;
   /* do a string compare for the first offset characters of cmdstr
      against each member of the cmdlist */
   while (cmdlist[cmdnum].cmdstr != NULL) {
      if (strcmp(cmdlist[cmdnum].cmdstr, cmdname) == 0)
         return(cmdnum);
     cmdnum++;
   }
   return(-1);
}

void execcmd(struct bootblk_command *cmdlist, int argc, const char **argv)
{
   int cmdnum = parsecmd(cmdlist, argc, argv);

   if (0) {
      putstr("execcmd: argv[0]="); putstr(argv[0]); putstr("\r\n");
      putLabeledWord("  argc=", argc);
      putLabeledWord("  cmdnum=", cmdnum);
   }
   if (cmdnum >= 0) {
      (*cmdlist[cmdnum].cmdfunc)(argc, argv);
   } else if (strcmp(argv[1], "help") == 0) {
      print_help_on_commands(cmdlist);
   } else {
#if 1
      print_help_on_commands(cmdlist);
#else
      /* default to the first command and insert it into the head of argv */
      int i;
      for (i = argc; i >= 0; i--) {
         argv[i+1] = argv[i];
      }
      argc++;
      argv[0] = cmdlist[0].cmdstr;
      (*cmdlist[0].cmdfunc)(argc, argv);
#endif
   }
}

void bootmenu(void)
{
  int cmdnum;
  char *cmdpos;
  int argc;
  char *argv[128];

  while (1) {
    putstr("boot> ");
    getcmd();
    if (!cmdBuffer[0])
	continue;		/* empty line, ignore */
    
    memset(argv, 0, sizeof(argv));
    parseargs(cmdBuffer, &argc, argv);
    if (argc > 0) {
       execcmd(commands, argc, (const char **)argv);
    } else {
       print_help_on_commands(commands);
    }
  }
}

#define PARAM_PREFIX ("bootldr:")
/* #define DEBUG_COMMAND_PARAMS_EVAL 1 */

void command_params_eval(int argc, const char **argv)
{
  char* p;
  char* endp;
  char* prefix;
  int   cmdnum;
  char cmdbuf[1024];
  char* cmdp;
  int	just_show = 0;
  int my_argc;
  char *my_argv[128];

  if (argc > 1)
     just_show = (strncmp(argv[1], "-n", 2) == 2);

  putstr("eval param blk\r\n");
  
  p = ((char*)flashword) + flashDescriptor->params.base;
  endp = p + flashDescriptor->params.size;

  /* stops at end of sector or first unwritten byte */
  while (p < endp && *p != 0xff && *p != 0) {
#ifdef DEBUG_COMMAND_PARAMS_EVAL
    putLabeledWord("top of loop, p=", (dword)p);
#endif
    prefix = PARAM_PREFIX;
    while (p < endp && *prefix && *p == *prefix) {
      p++;
      prefix++;
    }
#ifdef DEBUG_COMMAND_PARAMS_EVAL
    putLabeledWord("past prefix check, p=", (dword)p);
#endif

    if (*prefix != '\0') {
#ifdef DEBUG_COMMAND_PARAMS_EVAL
      putstr("no prefix match\r\n");
#endif

      /* skip to end of line */
      while (p < endp && *p != '\n')
	p++;

#ifdef DEBUG_COMMAND_PARAMS_EVAL
      putLabeledWord("skipped line, p=", (dword)p);
      putLabeledWord("char at p=", (dword)(*p) & 0xff);
#endif
      p++;
      continue;
    }

    /* copy line to buffer */
    /* terminate with eof, or eol */
    cmdp = cmdbuf;
    while (p < endp && *p != 0xff) {
      if (*p == '\r' || *p == '\n') {
	p++;
	if (*p == '\r' || *p == '\n') 
	  p++;
	break;
      }
      *cmdp++ = *p++;
    }
    *cmdp = '\0';
    cmdp = cmdbuf;

    putstr("+");
    putstr(cmdbuf);
    putstr("\r\n");

    if (just_show)
	continue;
    
    memset(my_argv, 0, sizeof(my_argv));
    parseargs(cmdbuf, &my_argc, my_argv);
    execcmd(commands, my_argc, (const char **)my_argv);
  }
}

int main()
{
  char c;

  /*
     At this stage, we assume:
        SDRAM is configured.
        UART is configured and ready to go.
        Initial PCI setup is done.
        Interrupts are OFF.
        MMU is DISABLED
  */
  
#ifdef DEBUG
  putstr("\r\nBOOT");
#endif

#ifdef CONFIG_PCI
  /* Configure the PCI devices */
#ifdef DEBUG
  putstr(" PCI");
#endif
  bootConfigurePCI();
#endif

  putLabeledWord("MMU table start=", (dword)mmu_table);
  putLabeledWord("Boot data start=", (dword)BOOT_DATA_START);
  putLabeledWord("Boot data size=", (dword)BOOT_DATA_SIZE);
  putLabeledWord("Stack data base=", (dword)STACK_BASE);
  putLabeledWord("Stack data size=", (dword)STACK_SIZE);
  
  /* 
   * set up MMU table:
   *  DRAM is cacheable
   *  cached and uncached images of Flash
   *  after MMU is enabled, bootldr is running in DRAM
   */
  bootConfigureMMU(); 
  writeBackDcache(CACHE_FLUSH_BASE);
  enableMMU();
  /* flashword logically should be assigned in bootConfigureMMU,
   * but we can't do it until the MMU is enabled
   * because before then it resides in flash 
   * -Jamey 9/4/2000
   */
  flashword = (unsigned long *)UNCACHED_FLASH_BASE;

  /* initialize the heap */
  mmalloc_init((unsigned char *)(HEAP_START), HEAP_SIZE);

  /* figure out what kind of flash we have */
  btflash_init();
  flashConfigureMMU(flash_size);

  {
     struct bootblk_param *system_rev_param = get_param("system_rev");
     struct bootblk_param *dram_size_param = get_param("dram_size");
#ifdef CONFIG_SKIFF
     struct bootblk_param *mem_fclk_21285_param = get_param("mem_fclk_21285");
     struct bootblk_param *maclsbyte_param = get_param("maclsbyte");
#endif
     long system_rev = get_system_rev();
     long dram_size;
#ifdef CONFIG_SKIFF
     dram_size = ((system_rev&SYSTEM_REV_MAJOR_MASK) == SYSTEM_REV_SKIFF_V2) ? DRAM_SIZE : SZ_16M;
#endif
#ifdef CONFIG_BITSY
     /* assumes one bank of DRAM and that DRAM_BASE0 is not in the cache */
     *(long *)(DRAM_BASE0 + SZ_64M) = SZ_64M;
     *(long *)(DRAM_BASE0 + SZ_32M) = SZ_32M;
     *(long *)(DRAM_BASE0 + SZ_16M) = SZ_16M;
     dram_size = *(long *)DRAM_BASE0;
#endif
#ifdef CONFIG_SKIFF
#if   defined(MEMCLK_33MHZ)
#warning 33MHZ override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_33MHZ;
#elif defined(MEMCLK_48MHZ)
#warning 48MHz override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_48MHZ;
#elif defined(MEMCLK_60MHZ)
#warning 60MHZ override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_60MHZ;
#endif
     if (system_rev_param != NULL)
        system_rev_param->value = system_rev;
     if (dram_size_param != NULL)
        dram_size_param->value = dram_size;
     putLabeledWord("system_rev: ", system_rev);
     switch (system_rev & SYSTEM_REV_MEMCLK_MASK) {
     case SYSTEM_REV_MEMCLK_60MHZ: 
        mem_fclk_21285_param->value = 60000000; break;
     case SYSTEM_REV_MEMCLK_48MHZ: 
        mem_fclk_21285_param->value = 48000000; break;
     case SYSTEM_REV_MEMCLK_33MHZ: 
     default:
        mem_fclk_21285_param->value = 33333333; break;
     }
     if (maclsbyte_param)
       maclsbyte_param->value = get_maclsbyte();

     putstr("memclk_hz: ");
     putstr(((system_rev & SYSTEM_REV_MEMCLK_MASK) == SYSTEM_REV_MEMCLK_33MHZ) 
            ? "33MHz\r\n" : "48MHz\r\n");
     /* this string should agree with the MAC in btpci.c */
     if (maclsbyte_param) {
       putstr("MAC=08:00:2b:00:01:"); putHexInt8(maclsbyte_param->value); putstr("\r\n");
     }
#endif
     putLabeledWord("dram_size: ", dram_size);
     
     bootinfo.bt_memend = dram_size;
  }

  /* print the opening banner */
  print_banner();

#define EVAL_PARAMS_BLOCK
#ifdef EVAL_PARAMS_BLOCK
  command_params_eval(0, NULL);
#else
  putstr("not evaluating params\r\n");
#endif

  c = awaitkey(TIMEOUT, NULL);
  if ((c != '\r') && (c != '\n') && (c != '\0')) {
     putstr("type \"?\" or \"help\" for help.\r\n");
     bootmenu(); /* does not return */
  }
    
  while (1) {

     /* default boot */
     int my_argc = 1;
     char *my_argv[128];
     my_argv[0] = "boot";
     my_argv[1] = NULL;
     execcmd(commands, my_argc, (const char **)my_argv);

     /* if default boot fails, drop into the bootmenu */
     bootmenu();
  }
}

void print_help_on_commands(struct bootblk_command *commands)
{
  int i = 0;
  while (commands[i].cmdstr != NULL) {
    putstr(commands[i].helpstr); putstr("\r\n");
    i++;
  }
}

/* no arguments */
void command_help(int argc, const char **argv)
{
  int i = 0;
  putstr("Available Commands:\r\n");
  if (argc == 2 && (strcmp(argv[1], "help") != 0)) {
     /* help <command>
      *   invoke <command> with 'help' as an argument
     */
     argv[0] = argv[1];
     argv[1] = "help";
     argv[2] = NULL;
     execcmd(commands, argc, argv);
     return;
  }
  print_help_on_commands(commands);
}


/* can have arguments or not */
void command_boot(int argc, const char **argv)
{
   if (argv[1] == NULL) {
      char *boot_type;
      get_param_value("boot_type", &boot_type);
      argv[1] = boot_type;
      argc++;
   }
   putstr("booting ");
   putstr(argv[1]);
   putstr("...\r\n");

   execcmd(boot_commands,argc-1,argv+1);
}

/* can have arguments or not */
void command_flash_type(int argc, const char **argv)
{
  if (argc == 1) {
     /* print the available flash types */
     btflash_print_types();
  } else {
    /* manually set the flash type */
     btflash_set_type(argv[1]);
  }
}

/* can have arguments or not */
void command_params(int argc, const char **argv)
{
   if (argc == 2) {
      if (strcmp(argv[1], "reset") == 0) {
         putstr("setting params to default values\r\n");
         memcpy((char*)bootldr_params, (const char*)(((char*)flashword) + (((dword)bootldr_params) - FLASH_BASE)), sizeof(bootldr_params));
         return;
      }
   } else if (argv[1] == NULL) {
      argv[1] = "show";
      argc++;
   }
   execcmd(params_commands,argc-1,argv+1);
}



void setup_linux_params(long bootimg_dest, long initrd_start, long initrd_size, 
			long memc_ctrl_reg, long mem_fclk_21285, long dram_size,
			const char *cmdline, int flashroot)
{
   struct param_struct *params = (struct param_struct *)(bootimg_dest + 0x100);
   int system_rev = get_param("system_rev")->value;
   long copy_ramdisk = 0;
   long first_word_of_ramdisk = *(long *)(FLASH_BASE + flashDescriptor->ramdisk.base);
   int rootdev = 0x00ff;
   int using_ramdisk = flashroot;

   get_param_value("copy_ramdisk", &copy_ramdisk);
   putLabeledWord("first_word_of_ramdisk=", first_word_of_ramdisk);
   if (flashroot && (first_word_of_ramdisk == 0x28cd3d45)) {
     /* cramfs */
     copy_ramdisk = 0;
     rootdev = 0x3c02;
     using_ramdisk = 0;
   }
   if (!flashroot) {
     copy_ramdisk = 0;
   }

   putLabeledWord("Setting up Linux parameters at address=", (unsigned long)params);

   memset(params, 0, sizeof(struct param_struct));
   params->u1.s.page_size = LINUX_PAGE_SIZE;
   params->u1.s.nr_pages = (dram_size >> LINUX_PAGE_SHIFT);
   params->u1.s.ramdisk_size = 0; /* seems to be duplicate of initrd_size */
   params->u1.s.rootdev = rootdev;
   params->u1.s.flags = 0;
   if (using_ramdisk) {
     if (copy_ramdisk) {
       /* virtual address using Linux MMU settings */
       params->u1.s.initrd_start = (long)(initrd_start&0x0FFFFFFF) + 0xC0000000;
     } else {
       /* virtual flash address using Linux MMU settings */
#ifdef CONFIG_SKIFF
       params->u1.s.initrd_start = 0xf8000000 + flashDescriptor->ramdisk.base + 4; 
#else
       params->u1.s.initrd_start = 0xd0000000 + flashDescriptor->ramdisk.base + 4; /* skip past the size word */
#endif
     }

     params->u1.s.initrd_size = initrd_size;
     params->u1.s.rd_start = 0; /* seems to be a duplicate of initrd_start */

   }
   putLabeledWord("using_ramdisk=", using_ramdisk);
   putLabeledWord("initrd_start=", params->u1.s.initrd_start);

   params->u1.s.system_rev = 0x0200; /* skiff-v2 A */
   params->u1.s.system_serial_low = 22;
   params->u1.s.memc_control_reg = memc_ctrl_reg;
#if PARAMS_HAS_FCLK_21285
   params->u1.s.mem_fclk_21285 = mem_fclk_21285;
#endif
   memcpy(params->commandline, cmdline, strlen(cmdline)+1);
   putLabeledWord("dram_size=", dram_size);
   putLabeledWord("nr_pages=", params->u1.s.nr_pages);
   putstr("command line is: ");
   putstr(cmdline);
   putstr("\r\n");
}


/*
 * boot_kernel
 *  root_filesystem_name:
 *  kernel_region_start: virtual address of region holding kernel image
 *  kernel_region_size:  size of region holding kernel image
 *  argc: number of arguments in argv (including argv[0], the command name)
 *  argv: argument vector (including argv[0], the command name)
 */
static void boot_kernel(const char *root_filesystem_name, 
                        vaddr_t kernel_region_start, size_t kernel_region_size, int argc, const char **argv)
{
   unsigned long boot_magic = *(unsigned long *)(kernel_region_start+0);
   unsigned long kernel_image_size = *(unsigned long *)(kernel_region_start+4);
   unsigned long kernel_image_dest = *(unsigned long *)(kernel_region_start+8);
   vaddr_t kernel_image_start = kernel_region_start + 12;
   unsigned long kernel_image_first_word = *(dword*)kernel_image_start;
   unsigned long kernel_image_tenth_word = *(dword*)(kernel_image_start+40);
   unsigned long kernel_image_maxsize = kernel_region_size;
   int unzipped = 0;
   int i;
   char *os;
   long bootingLinux = 0;
   long bootNetBSD = 0;
   long force_unzip_kernels = 0;
   long linuxEntryPoint = 0;
   long unzip_ramdisk = 0;
   long memc_ctrl_reg = 0x110c;
   long mem_fclk_21285;
   long dram_size = 0;
   long download_kernel = 0;
   long kernel_in_ram = 0;
   long copy_ramdisk = 0;

   get_param_value("dram_size", &dram_size);
   get_param_value("os", (long*)&os);
   get_param_value("entry", &linuxEntryPoint);
   get_param_value("download_kernel", &download_kernel);
   get_param_value("kernel_in_ram", &kernel_in_ram);
   get_param_value("copy_ramdisk", &copy_ramdisk);

   if (download_kernel) {
      putstr("Ready to download kernel to memory for booting\r\n");
      kernel_image_start = (vaddr_t)linuxEntryPoint;
      kernel_image_dest = linuxEntryPoint;
      kernel_image_size = modem_receive((char*)kernel_image_start, kernel_region_size);

      boot_magic = KERNELIMG_MAGIC;
      kernel_image_first_word = *(dword*)kernel_image_start;
      kernel_image_tenth_word = *(dword*)(kernel_image_start+40);
      kernel_image_maxsize = kernel_region_size;
   } else if (kernel_in_ram) {
      kernel_image_start = (vaddr_t)kernel_in_ram;
      kernel_image_dest = linuxEntryPoint;
      kernel_image_size = SZ_2M;

      boot_magic = KERNELIMG_MAGIC;
      kernel_image_first_word = *(dword*)kernel_image_start;
      kernel_image_tenth_word = *(dword*)(kernel_image_start+40);
      kernel_image_maxsize = kernel_region_size;
   }

   if (boot_magic == 0xFFFFFFFFL) {
      putstr("no boot image in flash\r\n");
      
      return;
   } 
   if (strncmp(os, "linux", 5) == 0) {
      putstr("os="), putstr(os); putstr("\r\n");
      putstr("expecting to boot a linux kernel\r\n");
      bootingLinux = 1;
   } else if (strncmp(os, "netbsd", 5) == 0) {
      putstr("os="), putstr(os); putstr("\r\n");
      putstr("expecting to boot a netbsd kernel\r\n");
      bootingLinux = 1;
   }


   putLabeledWord("boot_magic=", boot_magic);
   putLabeledWord("kernel_image_first_word=", kernel_image_first_word);

   if (force_unzip_kernels || 
       ( /* NETBSD */ (kernel_image_first_word != 0x0b018f00)
         && /* LINUX */ (kernel_image_first_word != SKIFF_ZIMAGE_MAGIC)
         && /* LINUX */ (kernel_image_first_word != SKIFF_IMAGE_MAGIC)
         && /* CATS-style linux image */ (kernel_image_first_word != CATS_ZIMAGE_MAGIC) )) {
#if defined(CONFIG_GZIP) || defined(CONFIG_BZIP)
      /* initialize the memory allocator */
      mmalloc_init((unsigned char *)(HEAP_START), HEAP_SIZE);
#endif
      
#ifdef CONFIG_GZIP_REGION      
      /* try to unzip kernel */
      if (gunzip_region((char*)kernel_image_start,
                        (char*)kernel_image_dest,
                        (long)kernel_image_size,
                        "kernel") == 0) {
	unzipped = 1;
      }
#endif      

#ifdef CONFIG_BZIP      
      /* attempt to uncompress */
      if (bzBuffToBuffDecompress((char *)kernel_image_dest,
				 &kernel_image_maxsize,
				 (char *)kernel_image_start,
				 kernel_image_size,0,0) == BZ_OK) {
	unzipped = 1;
      }
#endif      
      
      if (unzipped)
	kernel_image_first_word = *(unsigned long *)kernel_image_dest;

   }

   if (kernel_image_first_word == SKIFF_ZIMAGE_MAGIC
       || kernel_image_first_word == SKIFF_IMAGE_MAGIC
       || kernel_image_first_word == CATS_ZIMAGE_MAGIC
       || bootingLinux) {
      unsigned long compressed_rd_start = FLASH_BASE + flashDescriptor->ramdisk.base + 4;
      unsigned long compressed_rd_size = *(unsigned long *)(FLASH_BASE + flashDescriptor->ramdisk.base);
      unsigned long initrd_size = SZ_8M;
      unsigned long initrd_start = STACK_BASE - initrd_size;
      unsigned long kernel_image_offset = 0x0;

     /* CATS-style Linux zImage gets loaded to physaddr 0 */

     if (unzipped) kernel_image_dest = 0x0;
     if (   kernel_image_first_word == SKIFF_ZIMAGE_MAGIC
         || kernel_image_first_word == SKIFF_IMAGE_MAGIC
	 || kernel_image_tenth_word == LINUX_ZIMAGE_MAGIC ) {
         /* standard Linux zImage gets loaded to phyasaddr 0x8000 */
               kernel_image_offset = 0x8000;
	       linuxEntryPoint += 0x8000;
      }

     putstr("Linux ELF flash_imgstart=");
     putHexInt32((long)kernel_image_start);
     putstr(" size=");
     putHexInt32(kernel_image_size);
     putstr(" dest=");
     putHexInt32(kernel_image_dest);
     putstr(" offset=");
     putHexInt32(kernel_image_offset);
     putstr("\r\n");
     
     enable_caches(0, 0); /* also flushes caches */
     putLabeledWord("MMU Control=", 
		    readCPR1());

     if (!unzipped && ((void *)kernel_image_dest != (void *)kernel_image_start)) {
       putstr("copying Linux kernel ... ");
       memcpy((void*)(kernel_image_dest + kernel_image_offset), (void*)kernel_image_start, kernel_image_size);
       putstr("done\r\n");
     }

     putLabeledWord("linuxEntryPoint: ", linuxEntryPoint);
     putLabeledWord("kernel_image_dest:    ", kernel_image_dest);
     putLabeledWord("kernel_image_offset:    ", kernel_image_offset);
     for (i = 0; i < 40; i += 4) {
       /* we still have the MMU on, kernel_image_dest gets us to the right virtual address */
        putHexInt32(kernel_image_dest + kernel_image_offset + i);
        putstr(": ");
        putHexInt32(*(unsigned long *)(kernel_image_dest + kernel_image_offset +i));
        putstr("\r\n");
     }

#ifdef CONFIG_GZIP_REGION      
     if (unzip_ramdisk) {
        /* initialize the memory allocator */
        mmalloc_init((unsigned char *)(HEAP_START), HEAP_SIZE);
        putLabeledWord("compressed ramdisk size is ", compressed_rd_size);
        putstr("unpacking ramdisk from ");
        putHexInt32(compressed_rd_start);
        putstr(" to ");
        putHexInt32(initrd_start);
        putstr(" ... ");
        if (compressed_rd_start != 0) {
           if (gunzip_region((char*)compressed_rd_start,
                             (char*)initrd_start,
                             (long)compressed_rd_size, 
                             "ramdisk") == 0) {
              putstr("done.\r\n");
           } else {
              putstr("failed.\r\n");
              return;
           }
        }
     } else {
#endif /* CONFIG_GZIP */
       if (copy_ramdisk && compressed_rd_size != -1) {
         putstr("Copying compressed ramdisk from ");
         putHexInt32(compressed_rd_start);
         putstr(" to ");
         putHexInt32(initrd_start);
         putstr("...");
         memcpy((char*)initrd_start,(char*)compressed_rd_start,compressed_rd_size); 
         putstr("Done \r\n");
       } else if (compressed_rd_size != -1) {
          initrd_start = compressed_rd_start;
       }
#ifdef CONFIG_GZIP      
     }
#endif /* CONFIG_GZIP */

     if (1) {
        char boot_args[1024];
        char *linuxargs = NULL;
	int flashroot = 0;
	if (strcmp(root_filesystem_name, "ramdisk") == 0) {
	  flashroot = 1;
	}

	putstr("root_filesystem_name="); putstr(root_filesystem_name); putstr("\r\n");
        memset(boot_args, 0, 1024);
        get_param_value("linuxargs", &linuxargs);
        if (linuxargs != NULL)
           strcat(boot_args, linuxargs);
        putLabeledWord("argc=", argc);
        if (argc > 2) {
           unparseargs(boot_args, argc-1, argv+1);
        }

        putstr("args=");
        putstr(boot_args);
        putstr("\r\n");

        putstr("setting boot parameters\r\n");
        setup_linux_params(kernel_image_dest, initrd_start, compressed_rd_size,
                          memc_ctrl_reg, mem_fclk_21285, dram_size, boot_args, flashroot);
     }

     putLabeledWord("linuxEntryPoint=", linuxEntryPoint);
     putstr("Booting Linux image\r\n");
     bootLinux(&bootinfo,
#ifdef CONFIG_SKIFF
               MACH_TYPE_PERSONAL_SERVER,
#else
               MACH_TYPE_BITSY,
#endif
#ifdef CONFIG_SKIFF
               /* hack -- linux entry point virtual address is 0x10008000, physaddr is 0x00008000 */
               /* after we disable the MMU we have to use the physical address */
               linuxEntryPoint&0x00FFFFFF
#else
               linuxEntryPoint
#endif

);

   } else if (boot_magic == KERNELIMG_MAGIC) {
      char boot_args[1024];

      if (!unzipped) {
	putstr("copying NetBSD kernel ... ");
	memcpy((void*)kernel_image_dest, (void*)kernel_image_start, kernel_image_size);
	putstr("done\r\n");
      }

      /*bootinfo.bt_memavail = ROUNDUP(kernel_image_size, SZ_1M);*/
      bootinfo.bt_memavail = SZ_2M+SZ_1M;
      /*putLabeledWord("bt_memavail = ",bootinfo.bt_memavail);*/
      
      boot_args[0] = 0;
      strcat(boot_args, "netbsd ");
      if (argc > 1) {
         unparseargs(boot_args, argc-1, argv+1);
      }
      bootinfo.bt_args = boot_args;
      bootinfo.bt_vargp = (u_int32_t)boot_args & PAGE_MASK;
      bootinfo.bt_pargp = (u_int32_t)boot_args & PAGE_MASK;

      putstr("done!\r\nJumping to 0xF0000020..\r\n");
      flush_caches();
      boot(&bootinfo,0xF0000020);
   } else {
     putstr("Corrupt kernel image\r\n");
     return;
   }
}

/* can have arguments or not */
void command_boot_flash(int argc, const char **argv)
{
   const char *ipaddr = NULL;
   const char *serveraddr = NULL;
   const char *gatewayaddr = NULL;
   const char *netmask = NULL;
   const char *hostname = NULL;
   const char *nfsroot = NULL;
   char bootargs[2048];

   get_param_value("ipaddr", &ipaddr);
   get_param_value("nfs_server_address", &serveraddr);
   get_param_value("gateway", &gatewayaddr);
   get_param_value("netmask", &netmask);
   get_param_value("hostname", &hostname);
   get_param_value("nfsroot", &nfsroot);

   if (nfsroot != NULL) {
     strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
   }
   if ((ipaddr != NULL) || (serveraddr != NULL) || (gatewayaddr != NULL) || (netmask != NULL) || (hostname != NULL)) {
      strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
      strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
      strcat(bootargs, ":eth0 ");
   }
   
   argv[argc++] = bootargs;
   argv[argc] = NULL;

   boot_kernel("ramdisk",
               (vaddr_t)(((unsigned long)flashword) + flashDescriptor->kernel.base), flashDescriptor->kernel.size, argc, argv);
}

#ifdef BOOT_ALTKERNEL
/* can have arguments or not */
void command_boot_altkernel(int argc, const char **argv)
{
   const char *ipaddr = NULL;
   const char *serveraddr = NULL;
   const char *gatewayaddr = NULL;
   const char *netmask = NULL;
   const char *hostname = NULL;
   const char *nfsroot = NULL;
   char bootargs[2048];

   get_param_value("ipaddr", &ipaddr);
   get_param_value("nfs_server_address", &serveraddr);
   get_param_value("gateway", &gatewayaddr);
   get_param_value("netmask", &netmask);
   get_param_value("hostname", &hostname);
   get_param_value("nfsroot", &nfsroot);

   if (nfsroot != NULL) {
     strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
   }
   strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
   strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
   strcat(bootargs, ":eth0 ");
   
   argv[argc++] = bootargs;
   argv[argc] = NULL;

   boot_kernel("ramdisk",
               (vaddr_t)(((unsigned long)flashword) + flashDescriptor->altkernel.base), flashDescriptor->altkernel.size, argc, argv);
}
#endif

/* can have arguments or not */
void command_boot_nfsroot(int argc, const char **argv)
{
   const char *ipaddr = NULL;
   const char *serveraddr = NULL;
   const char *gatewayaddr = NULL;
   const char *netmask = NULL;
   const char *hostname = NULL;
   const char *nfsroot = NULL;
   char bootargs[2048];

   get_param_value("ipaddr", &ipaddr);
   get_param_value("nfs_server_address", &serveraddr);
   get_param_value("gateway", &gatewayaddr);
   get_param_value("netmask", &netmask);
   get_param_value("hostname", &hostname);
   get_param_value("nfsroot", &nfsroot);

   strcat(bootargs, " noinitrd root=/dev/nfs");
   if (nfsroot != NULL) {
     strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
   }
   strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
   strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
   strcat(bootargs, ":eth0 ");
   
   argv[argc++] = bootargs;
   argv[argc] = NULL;

   boot_kernel("nfsroot",
               (vaddr_t)(((unsigned long)flashword) + flashDescriptor->kernel.base), flashDescriptor->kernel.size, argc, argv);
}

void command_display(int argc, const char **argv)
{

  /* char buf[9];
     binarytohex(buf,value,4); */

 #ifdef CONFIG_BITSY
   putstr("\r\nSA1100 Registers:\r\n" );
     putLabeledWord("   SA1100_UART3:  ", (long) SA1100_UART3_UTCR0);
       putLabeledAddrWord( "      UTCR0   0x00 ", (long *)SA1100_UART3_UTCR0);
       putLabeledAddrWord( "      UTCR1   0x04 ", (long *)SA1100_UART3_UTCR1);
       putLabeledAddrWord( "      UTCR2   0x08 ", (long *)SA1100_UART3_UTCR2);
       putLabeledAddrWord( "      UTCR3   0x0c ", (long *)SA1100_UART3_UTCR3);
       putLabeledAddrWord( "      UTDR    0x10 ", (long *)SA1100_UART3_UTDR);
       putLabeledAddrWord( "      UTSR0   0x14 ", (long *)SA1100_UART3_UTSR0);
       putLabeledAddrWord( "      UTSR0+4 0x18 ", (long *)SA1100_UART3_UTSR0+4);
       putLabeledAddrWord( "      UTSR0+8 0x1c ", (long *)SA1100_UART3_UTSR0+8);
       putLabeledAddrWord( "      UTSR1   0x20 ", (long *)SA1100_UART3_UTSR1);
#endif
}


/* requires arguments */
void command_load(int argc, const char **argv)
{
   execcmd(load_commands,argc-1,argv+1);
}

/* requires arguments */
void command_load_flash(int argc, const char **argv)
{
   unsigned long offset;

   unsigned long img_size = 0;
   unsigned long flash_dest = 0;
   unsigned long i;
   int override;

   get_param_value("override", &override);

   /* loading flash is different.. because writing flash is slow, we do
      not load it directly from the serial port. So we receive into
      memory first, and then program the flash... */
  
   if (argc < 2) { /* default to first command */
      putstr("usage: load flash <flashaddr>\r\n");
   } else {
      flash_dest = strtoul(argv[1], NULL, 0);
      if (strtoul_err) {
         putstr("error parsing flash_dest\r\n");
         return;
      }

      if (flash_dest < flashDescriptor->bootldr.size && !override) {
         putstr("That is bootloader space! Use load bootldr.  Operation canceled\r\n");
         return;
      }
      command_load_flash_region("flash", flash_dest, flash_size, 0);
   }
}


void command_load_ram(int argc, const char **argv)
{
  unsigned long offset;

  unsigned long img_size = 0;
  unsigned long img_dest = 0;
  
  if (argc < 3) { /* default to first command */
    putstr("usage: load ram <ramaddr> <length>\r\n");
  } else {
    /* parse the command line to fill img_size and img_dest */

    img_dest = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing img_dest\r\n");
      return;
    }
    img_size = strtoul(argv[2], NULL, 0);
    
    img_size = modem_receive((char*)img_dest, img_size);
    /* in case this is a kernel download, update bt_memavail */
    bootinfo.bt_memavail = ROUNDUP(img_size, SZ_1M);

    putHexInt32(img_size);
    putLabeledWord(" bytes loaded to ",img_dest);
    putstr("\r\n");
  }
}

/**
 * does the work for command_load_bootldr, etc.
 */

static void program_flash_region(const char *regionName, 
                                 unsigned long regionBase, size_t regionSize, 
                                 unsigned long src, size_t img_size, int flags)
{
   unsigned long hdr_size = 0; /* size of header */
   unsigned long i;
   int remaining_bytes;
   long noerase;
   get_param_value("noerase", &noerase);

   if (flags & LFR_PATCH_BOOTLDR) {
     unsigned int bsd_sum;
     /* check here that it's really a bootldr, otherwise print a message and exit */
     if (*(long *)(src + 0x20) != BOOTLDR_MAGIC) {
       putstr("Not loading bootldr into flash\r\n");
       putLabeledWord("Downloaded image does not have BOOTLDR_MAGIC: ", BOOTLDR_MAGIC);
       putLabeledWord("Instead found: ", *(long *)(src + 0x20));
       return;
       }
     /* check here to see if the architecture number matches */
     if (*(long *)(src + 0x2C) != ARCHITECTURE_MAGIC) {
       putstr( "Not loading bootldr into flash\r\n");
       putLabeledWord("Downloaded image does not have ARCHITECTURE_MAGIC=", ARCHITECTURE_MAGIC);
       putLabeledWord("Instead found=", *(long *)(src + 0x2C));
       return;
     }
     /* check here to be certain that the BSD sum value is zero */
     bsd_sum = bsd_sum_memory( src, img_size);
     putLabeledWord("BSD sum value is: ", bsd_sum);
     if (bsd_sum != 0) {
       putstr("BSD sum nonzero -- corrupt bootldr, not programming flash.\r\n");
       return;
     }
   }

   if (flags & LFR_SIZE_PREFIX) {
      hdr_size += 4;
      src -= 4;
      *(long *)src = img_size;
      img_size += hdr_size;
   }

   if (img_size > regionSize) {
      putLabeledWord("img_size is too large for region: ", regionSize);
      return;
   }
    
    
   putstr("\r\nprogramming flash...");

   SET_VPPEN();
   if (flags & LFR_PATCH_BOOTLDR) {
      putstr( "\r\nunlocking boot sector of flash\r\n" );
      protectFlashRange( 0x0, img_size+hdr_size, 0);
   }
   if (!noerase) {
     putstr("erasing ...\r\n");
     if (eraseFlashRange(regionBase,img_size+hdr_size)) {
       putstr("erase error!\r\n");
       goto clear_VPPEN;
     }
   }

   /* the magic number.. */
   putstr("writing flash..\r\n");
    
   i = 0;
   remaining_bytes = img_size;
   while(remaining_bytes > 0) {
      int bytes_programmed = 0;
      if ((i % SZ_64K) == 0) {
         putstr("addr: ");
         putHexInt32(regionBase+i);
         putstr(" data: ");
         putHexInt32(*(unsigned long *)(src+i));
         putstr("\r\n");
      }

      if (((i % 64) == 0) && (remaining_bytes > 64)) {
         /* program a block */
         if (programFlashBlock(regionBase+i, (unsigned long*)(src+i), 64)) {
            putstr("error while copying to flash!\r\n");
            goto clear_VPPEN;
            break;
         }
         bytes_programmed = 64;
      } else {
         if (programFlashWord(regionBase+i, *(unsigned long *)(src+i))) {
            putstr("error while copying to flash!\r\n");
            goto clear_VPPEN;
            break;
         }
         bytes_programmed = 4;
      }
      i += bytes_programmed;
      remaining_bytes -= bytes_programmed;
   }
   putstr("verifying ... ");
   {
      int i;
      int nwords = (img_size >> 2);
      unsigned long *srcwords = (unsigned long *)src;
      unsigned long *dstwords = (unsigned long *)&flashword[regionBase >> 2];
      for (i = 0; i < nwords; i++) {
         if (srcwords[i] != dstwords[i]) {
            putLabeledWord("error programming flash at offset=", i << 2);
            putLabeledWord("  src=", srcwords[i]);
            putLabeledWord("  flash=", dstwords[i]);
            putstr("not checking any more locations\r\n");
            goto clear_VPPEN;
         }
      }
   }
   putstr("done.\r\n");
   if (flags & LFR_PATCH_BOOTLDR) {
      protectFlashRange( 0x0, img_size+hdr_size, 1);
   }

 clear_VPPEN:
   CLR_VPPEN();
}

static void command_load_flash_region(const char *regionName, unsigned long regionBase, unsigned long regionSize, int flags)
{
   unsigned long img_size = 0; /* size of data */
   unsigned long hdr_size = 0; /* size of header */
   unsigned long img_dest = VKERNEL_BASE + 1024; /* just a temporary holding area */
   unsigned long i;
   long xmodem = 0;
  
   putstr("loading flash region "); putstr(regionName); putstr("\r\n");

   get_param_value("xmodem", &xmodem);

#ifdef CONFIG_XYMODEM
   if (xmodem) {
#endif
      putstr("using xmodem\r\n");
      img_size = modem_receive((char*)img_dest, regionSize);
#ifdef CONFIG_XYMODEM
   } else {
      putstr("using xymodem\r\n");
      img_size = xymodem_receive((char*)img_dest, regionSize);
   }
#endif
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   }
    
   putHexInt32(img_size);
   putLabeledWord(" bytes loaded to ",img_dest);
   if ((img_size % 4) != 0) {
      putstr("img_size is not a multiple of 4 -- are we sure that's OK?\r\n");
   }

   program_flash_region(regionName, regionBase, regionSize, img_dest, img_size, flags);
}

void command_load_bootldr(int argc, const char **argv)
{
  command_load_flash_region("bootldr", flashDescriptor->bootldr.base, flashDescriptor->bootldr.size, flashDescriptor->bootldr.flags);
}

void command_load_params(int argc, const char **argv)
{
  command_load_flash_region("params", flashDescriptor->params.base, flashDescriptor->params.size, flashDescriptor->params.flags);
}

void command_load_usercode(int argc, const char **argv)
{
  command_load_flash_region("usercode", flashDescriptor->usercode.base, flashDescriptor->usercode.size, flashDescriptor->usercode.flags);
}

void command_load_ramdisk(int argc, const char **argv)
{
  command_load_flash_region("ramdisk", flashDescriptor->ramdisk.base, flashDescriptor->ramdisk.size, flashDescriptor->ramdisk.flags);
}

static void load_kernel(const char *kernel_region_name, 
                        unsigned long kernel_region_start, unsigned long kernel_region_size)
{
   unsigned long img_size = 0;
   unsigned long img_dest = VKERNEL_BASE;
   unsigned long i;
  
   img_size = modem_receive((char*)img_dest+12, kernel_region_size);

   if (!img_size) {
      putstr("error occured during download. aborting.\r\n");
      return;
   }
    
   putHexInt32(img_size);
   putLabeledWord(" bytes loaded to ",img_dest);
   if ((img_size % 4) != 0) {
      putstr("img_size is not a multiple of 4 -- are we sure that's OK?\r\n");
   }
   if ((img_size+12) > kernel_region_size) {
      putLabeledWord("kernel too large for region, but programming anyway.  Size=", kernel_region_size);
#if FIXME
      return;
#endif
   }

   /* set up the header */
   {
      /* the magic number.. */
      *(unsigned long *)(img_dest + 0x0) = KERNELIMG_MAGIC;
      /* the actual kernel image size */
      *(unsigned long *)(img_dest + 0x4) = img_size;
      /* where to copy the image before boot */
      *(unsigned long *)(img_dest + 0x8) = VKERNEL_BASE;
   }
   /* now program flash */
   program_flash_region(kernel_region_name, kernel_region_start, kernel_region_size,
                        img_dest, img_size, 0);
   putstr("done.\r\n");
}


void command_load_kernel(int argc, const char **argv)
{
   load_kernel("kernel", flashDescriptor->kernel.base, flashDescriptor->kernel.size);
}

#ifdef BOOT_ALTKERNEL
void command_load_altkernel(int argc, const char **argv)
{
   load_kernel("altkernel", flashDescriptor->altkernel.base, flashDescriptor->altkernel.size);
}
#endif

void command_load_debugger(int argc, const char **argv)
{
   command_load_flash_region("debugger", flashDescriptor->debugger.base, flashDescriptor->debugger.size, flashDescriptor->debugger.flags);
}

/* can have arguments or not */
void command_save(int argc, const char **argv)
{
   putstr("Saving flash ");
   putstr(argv[1]);
   putstr("...\r\n");

   execcmd(save_commands,argc-1,argv+1);
}

void command_save_all(int argc, const char **argv) {

   dword img_size;

   img_size = modem_send(0x0, 0x01000000); 
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   } 
}

void command_save_altkernel(int argc, const char **argv) {

  dword img_size;

    sendPacketZero( 0x0, 1, 128 );
  
  /*  img_size = modem_send( flashDescriptor->altkernel.base, flashDescriptor->altkernel.size );
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
      } */
}

void command_save_bootldr(int argc, const char **argv) {

  dword img_size;

  img_size = modem_send( flashDescriptor->bootldr.base, flashDescriptor->bootldr.size );
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   }
}

void command_save_debugger(int argc, const char **argv) {

  dword img_size;

  sendPacketTest( 0x0, 1, 128 );
  /* img_size = modem_send( flashDescriptor->debugger.base, flashDescriptor->debugger.size );
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
      } */

}

void command_save_flash(int argc, const char **argv) {

   dword         img_size;
   unsigned long flash_size = 0;
   unsigned long flash_dest = 0;

   if (argc < 3) { /* default to first command */
      putstr("usage: load flash <flashaddr> <size>\r\n");
   } else {
      flash_dest = strtoul(argv[1], NULL, 0);
      if (strtoul_err) {
         putstr("error parsing <flashaddr>\r\n");
         return;
      }
      flash_size = strtoul(argv[2], NULL, 0);
      if (strtoul_err) {
	putstr("error parsing <size>\r\n");
      }
      putLabeledWord("flash_dest=", flash_dest );
      putLabeledWord("flash_size=", flash_size );
      img_size = modem_send( flash_dest, flash_size );
      if (!img_size) {
         putstr("download error. aborting.\r\n");
         return;
      }
   }
   return;
}

void command_save_kernel(int argc, const char **argv) {

  dword img_size;

  img_size = modem_send( flashDescriptor->kernel.base, flashDescriptor->kernel.size );
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   }
}

void command_save_params(int argc, const char **argv) {

  dword img_size;

  img_size = modem_send( flashDescriptor->params.base, flashDescriptor->params.size );
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   }
}

void command_save_ramdisk(int argc, const char **argv) {

  dword img_size;

  img_size = modem_send( flashDescriptor->ramdisk.base, flashDescriptor->ramdisk.size );
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   }
}

void command_save_usercode(int argc, const char **argv) {

  dword img_size;

  img_size = modem_send(  flashDescriptor->usercode.base, flashDescriptor->usercode.size );
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   }
}

void command_save_world(int argc, const char **argv) {

  putstr( "\r\nI don't know how to save the world.\r\n\r\n" );

}

void command_peek(int argc, const char **argv)
{
   execcmd(peek_commands,argc-1,argv+1);
}

void command_peek_ram(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long result = 0;
  unsigned long offset = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("peek ram requires arguments!\r\n");
  } else {
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    if (strcmp(argv[0], "byte") == 0)
       result = *(unsigned char *)(addr);
    else if (strcmp(argv[0], "short") == 0)
       result = *(unsigned short *)(addr);
    else if (strcmp(argv[0], "int") == 0)
       result = *(int *)(addr);
    else if (strcmp(argv[0], "flash") == 0)
       result = flashword[(addr&flash_address_mask) >> 2];
    else if (strcmp(argv[0], "gpio") == 0)
       /* read from the gpio read port */
       result = *(int *)(0x21800008 + addr);
    else 
       result = *(unsigned long *)(addr);

    putLabeledWord("  addr  = ",addr);
    putLabeledWord("  value = ",result);
  }
}

void command_poke(int argc, const char **argv)
{
   execcmd(poke_commands,argc-1,argv+1);
}

void command_poke_ram(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long value = 0;
  unsigned long offset = 0;
  
  if (argc < 3) { /* default to first command */
    putstr("poke ram requires arguments!\r\n");
  } else {

    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    value = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing value\r\n");
      return;
    }

    putstr("poke ram: \r\n");
    putLabeledWord(" addr=", addr);
    putLabeledWord(" value=", value);

    if (strcmp(argv[0], "byte") == 0) {
       *(unsigned char *)(addr) = value;
    } else if (strcmp(argv[0], "short") == 0) {
       *(unsigned short *)(addr) = value;
    } else if (strcmp(argv[0], "int") == 0) {
       *(int *)addr = value;
    } else if (strcmp(argv[0], "gpio") == 0) {
       /* write to the gpio write port */
       *(int *)(0x21800000 + addr) = value;
    } else if (strcmp(argv[0], "flash") == 0) {
       if (programFlashWord(addr & flash_address_mask, value))
          putstr("flash write failed!\r\n");
    } else {
       *(unsigned long *)(addr) = value;
    }
  }
}


#ifdef CONFIG_BITSY
extern unsigned long delayedBreakpointPC;
void command_breakpoint(int argc, const char **argv)
{
  if (argv[1] != NULL) {
    const char *pcstr = NULL;
    unsigned long pc;
    int delayed = 0;
    if (strcmp(argv[1], "delayed") == 0) {
      delayed = 1;
      pcstr = argv[2];
    } else {
      pcstr = argv[1];
    }
    pc = strtoul(pcstr, NULL, 0);
    pc &= 0xFFFFFFFC;
    pc |= 1;
    if (delayed) {
      putLabeledWord("PC breakpoint will be set after kernel unzip at: ", pc);
      delayedBreakpointPC = pc;
    } else {
      putLabeledWord("Setting hardware PC breakpoint at: ", pc);
      __asm__ ("mcr     p15, 0, %0, c14, c8, 0" : : "r" (pc));
    }
  } else {
    unsigned long pc = 0;
    putstr("Clearing PC breakpoint");
    __asm__ ("mcr     p15, 0, %0, c14, c8, 0" : : "r" (pc));
  }
}
#endif

void command_qflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long result = 0;
  unsigned long offset = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("qflash requires arguments!\r\n");
  } else {
    addr = strtoul(argv[argc-1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    if (strcmp(argv[1], "id") == 0) {
       result = queryFlashID(addr);
    } else if (strcmp(argv[1], "security") == 0) {
       result = queryFlashSecurity(addr);
    } else {
       result = queryFlash(addr);
    }
    putLabeledWord("value = ",result);
  }
}

void command_eflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long result = 0;
  unsigned long offset = 0;
  int override;

  get_param_value("override", &override);
  
  if (argc < 2) { /* default to first command */
    putstr("eflash requires arguments: <addr>|chip!\r\n");
  } else {
    if (strncmp(argv[1], "chip", 4) == 0) {
      if (!override) {
	putstr("Cannot erase whole chip without setting override to 1.\r\n");
	return;
      }
       putstr("erasing flash chip\r\n");
       SET_VPPEN();
       eraseFlashChip();
       CLR_VPPEN();
       return;
    }
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr: "); putstr(argv[1]); putstr("\r\n");
    }
    putLabeledWord("addr=", addr);

    if (addr == 0 && !override) {
      putstr("Cannot erase first sector without setting override to 1.\r\n");
      return;
    }

    SET_VPPEN();
    result = eraseFlashSector(addr);
    CLR_VPPEN();
  }
}

void command_pflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long len = 0;
  unsigned long protect = 9;
  
  if (argc < 4) { /* default to first command */
    putstr("pflash requires arguments: <addr> <len> 0/1  (1 -> protect, 0 -> unprotect all)!\r\n");
  } else {
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr: "); putstr(argv[1]); putstr("\r\n"); return;
    }
    len = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing len: "); putstr(argv[2]); putstr("\r\n"); return;
    }
    protect = strtoul(argv[3], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing protect: "); putstr(argv[3]); putstr("\r\n"); return;
    }
    /* be certain that protect is only 0/1 */
    if (!((protect == 0) || (protect == 1)) ) {
      putstr("error protect value must be 0/1: "); putstr(argv[3]); putstr("\r\n"); return;
    }
    
    if (verifiedFlashRange( addr, len )) {
       putLabeledWord("  addr=", addr);
       putLabeledWord("  len=", len);
       putLabeledWord("  protect=", protect);

       SET_VPPEN();
       protectFlashRange(addr, len, protect);
       CLR_VPPEN();
    } else { 
      putstr("Region specified is out of Range.\r\n" );
      putLabeledWord( " Please use a range less than:", flash_size );      
    }
  }
}

#if 0
void command_reboot(int argc, const char **argv)
{
  if (argc < 1) {
    reboot();
  } else {
    putstr("reboot does not need arguments!\r\n");
  }
}
#endif

void command_call(int argc, const char **argv)
{
  void (*fcn)(long a0, long a1, long a2, long a3) = NULL;
  int offset;
  
  if (argc < 2) {
    putstr("usage: call <addr>\r\n");
  } else {
    long a0 = 0;
    long a1 = 0;
    long a2 = 0;
    long a3 = 0;
    fcn = (void*)strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing vaddr\r\n");
      return;
    }
    if (argc > 2) {
      a0 = strtoul(argv[2], NULL, 0);
    }
    if (argc > 3) {
      a1 = strtoul(argv[3], NULL, 0);
    }
    if (argc > 4) {
      a2 = strtoul(argv[4], NULL, 0);
    }
    if (argc > 5) {
      a3 = strtoul(argv[5], NULL, 0);
    }
    flush_caches();
    putLabeledWord("Calling fcn=", (long)fcn);
    putLabeledWord("  a0=", a0);
    putLabeledWord("  a1=", a1);
    putLabeledWord("  a2=", a2);
    putLabeledWord("  a3=", a3);
    fcn(a0, a1, a2, a3);
  }
}

void command_physaddr(int argc, const char **argv)
{
  unsigned long vaddr = 0;
  int offset;
  
  if (argc < 2) { /* default to first command */
    putstr("physaddr requires vaddr argument!\r\n");
  } else {
    vaddr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing vaddr\r\n");
      return;
    }
    {
      unsigned long mmu_table_entry = mmu_table[vaddr >> 20];
      unsigned long paddr = mmu_table_entry & ~((1L << 20) - 1);
      unsigned long sectionDescriptor = mmu_table_entry & ((1L << 20) - 1);
      putLabeledWord("vaddr=", vaddr);
      putLabeledWord("paddr=", paddr);
      putLabeledWord("sectionDescriptor=", sectionDescriptor);
    }
  }
}

void command_set(int argc, const char **argv)
{
   struct bootblk_param *param = bootldr_params;
   const char *paramValueString = argv[2];
   if (paramValueString == NULL)
      paramValueString = "";
   while (param->name != NULL) {
      int namelen = strlen(param->name);
      if (strncmp(argv[1], param->name, namelen) == 0) {
         long value;
         int paramValueLen = strlen(paramValueString);
         putstr("  setting param=");
         putstr(param->name);
         putstr(" to value=");
         putstr(paramValueString);
         putstr("\r\n");
         switch (param->paramFormat) {
         case PF_STRING: {
            if (paramValueLen == 0) {
               value = 0;
            } else {
               char *newstr = malloc(paramValueLen+1);
               memcpy(newstr, paramValueString, paramValueLen+1);
               value = (long)newstr;
            }
            break;
         }
         case PF_LOGICAL:
	   paramoldvalue = param->value;
           value = strtoul(paramValueString, NULL, 0);
         case PF_DECIMAL:
         case PF_HEX:
         default:
	    paramoldvalue = param->value;
            value = strtoul(paramValueString, NULL, 0);
         }
         param->value = value;
         if (param->update != NULL) {
            (param->update)(param);
         }
      }
      param++;
   }
}

void command_params_show(int argc, const char **argv)
{
   struct bootblk_param *params = bootldr_params;
   while (params->name != NULL) {
      putstr("  ");
      putstr(params->name);
      switch (params->paramFormat) {
      case PF_STRING:
         putstr("="); putstr((char*)params->value); putstr("\r\n");
         break;
      case PF_DECIMAL: {
        char buf[16];
        dwordtodecimal(buf, params->value);
        putstr("="); putstr(buf); putstr("\r\n");
        break;
      }
      case PF_HEX:
      default:
         putLabeledWord("=0x", params->value);
         break;
      }
      params++;
   }
}

#ifdef UPDATE_BAUDRATE
void update_baudrate(struct bootblk_param *param)
{
   volatile unsigned long *csr = (volatile unsigned long *)DC21285_ARMCSR_BASE;
   unsigned long system_rev = get_system_rev();
   unsigned long fclk_hz = 33333333;
   int baudrate = param->value;
   unsigned long ubrlcr;
   unsigned long l_ubrlcr = 0;
   unsigned long m_ubrlcr = 0;
   unsigned long h_ubrlcr = 0;

   switch ( baudrate ) {
#ifdef CONFIG_BITSY
   case 230400:
     break;
   case 115200:
     break;
#endif
   case 57600:
     break;
   case 38400:
     break;
   case 19200:
     break;
   case 9600:
     break;
   case 4800:
     break;
   case 2400:
     break;
   case 1200:
     break;
   case 300:
     break;
   case 110:
     break;
   default:
     param->value = paramoldvalue;
     putstr( "invalid baud rate:\r\n" );
     putstr( "     Please, try:   110,   300,  1200,   2400,  4800,  9600\r\n" );
     putstr( "                  19200, 38400, 57600, 115200 or 230,400\r\n" );
     return; 
   }
   
#ifdef CONFIG_SKIFF
   switch (system_rev&SYSTEM_REV_MEMCLK_MASK) {
   case SYSTEM_REV_MEMCLK_33MHZ:
      fclk_hz = 33333333;
      break;
   case SYSTEM_REV_MEMCLK_48MHZ:
      fclk_hz = 48000000;
      break;
   case SYSTEM_REV_MEMCLK_60MHZ:
      fclk_hz = 60000000;
      break;
   }

  fclk_hz = 48000000;

   ubrlcr = (fclk_hz / 64 / baudrate)-1;
   h_ubrlcr = csr[H_UBRLCR_REG];

#elif defined(CONFIG_BITSY)
   ubrlcr = (3686400 / 16 / baudrate) - 1;
#else
   #error no architecture defined in update_baudrate
#endif

   l_ubrlcr = ubrlcr & 0xFF;
   m_ubrlcr = (ubrlcr >> 8) & 0xFF;
   
   putLabeledWord("update_baudrate:  new baudrate=", baudrate);

   putLabeledWord("  CSRBASE=", (int)csr);
   putLabeledWord("  l_ubrlcr=", l_ubrlcr);
   putLabeledWord("  m_ubrlcr=", m_ubrlcr);
   putLabeledWord("  h_ubrlcr=", h_ubrlcr);

#ifdef CONFIG_SKIFF
   /* wait for the TX FIFO to empty */
   while ((csr[UARTFLG_REG]&UART_TX_BUSY) != 0)
      /* spin */;

   /* disable the uart */
   csr[UARTCON_REG] = 0;

   /* set the new values, in the right order */
   csr[L_UBRLCR_REG] = l_ubrlcr;
   csr[M_UBRLCR_REG] = m_ubrlcr;
   csr[H_UBRLCR_REG] = h_ubrlcr;

   /* reenable the uart */
   csr[UARTCON_REG] = 1;
#else
   /* wait for the TX FIFO to empty */
  while (((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_TBY)); /* wait until TX not busy */
  *(volatile byte*)SA1100_UART3_UTCR3 = 0; /* disable UART */

  *(volatile byte*)SA1100_UART3_UTCR1 = m_ubrlcr; /* MSB of baudrate divisor */
  *(volatile byte*)SA1100_UART3_UTCR2 = l_ubrlcr; /* LSB of baudrate divisor */
  
  *(volatile byte*)SA1100_UART3_UTCR3 = SA1100_UTCR3_RXE|SA1100_UTCR3_TXE; /* enable UART */
#endif

   putLabeledWord(" baudrate changed to 0x", baudrate);

   /* set the value for the kernel too */

   bootinfo.bt_comspeed = baudrate;
}
#endif

void set_serial_number(struct bootblk_param *param)
{
}


long get_system_rev()
{
   long word1 = *(long *)FLASH_BASE;
   long word2 = *(long *)(FLASH_BASE+0x0080000C);
   long revision = SYSTEM_REV_SKIFF_V1;
   if (word1 != word2) {
      /* Skiff-V2 -- GPIO register 0 contains the revision mask */
      revision = (word2 & SYSTEM_REV_MASK);
   }
   return revision;
}

void set_system_rev(struct bootblk_param *param)
{
   param->value = get_system_rev();
}

void flush_caches(void)
{
   writeBackDcache(CACHE_FLUSH_BASE);
   __asm__("  mcr p15, 0, r0, c7, c7, 0x00 /* flush I+D caches */\n"
           "  mcr p15, 0, r0, c7, c10, 4 /* drain the write buffer*/\n");

}

long enable_caches(int dcache_enabled, int icache_enabled)
{
  long mmuc;
  long enabled = (icache_enabled ? 0x1000 : 0x0);
  enabled |= (dcache_enabled ? 0xC : 0);
  flush_caches();
  __asm__("  mrc p15, 0, %0, c1, c0, 0\n"
	  "  bic %0, %0, #0x1000\n"
	  "  bic %0, %0, #0x000c\n"
	  "  orr %0, %0, %1\n"
	  "  mcr p15, 0, %0, c1, c0, 0\n"
	  : "=r" (mmuc) : "r" (enabled));
  flush_caches();
  return mmuc;
}

void update_dcache_enabled(struct bootblk_param *param)
{
   long dcache_enabled = param->value;
   long icache_enabled = 0;
   long mmuc;
   get_param_value("icache_enabled", &icache_enabled);
   mmuc = enable_caches(dcache_enabled, icache_enabled);
   putLabeledWord("MMU Control word=", mmuc);
}

void update_icache_enabled(struct bootblk_param *param)
{
   long dcache_enabled = 0;
   long icache_enabled = param->value;
   long mmuc;
   get_param_value("dcache_enabled", &dcache_enabled);
   mmuc = enable_caches(dcache_enabled, icache_enabled);
   putLabeledWord("MMU Control word=", mmuc);
}

void command_params_save(int argc, const char **argv)
{
   char buf[SZ_16K];
   struct bootblk_param *param = bootldr_params;
   /* default params points into the flash image as opposed to the copy in dram */
   const struct bootblk_param *defaultParam = 
      (const struct bootblk_param *)(((char*)flashword) + (((dword)bootldr_params) - FLASH_BASE));
   buf[0] = '\0';

   /* should pick up the old params from flash in case there are any non-bootldr lines */

   /* construct the parameter setting code */
   while (param->name != NULL) {
      int modifiedFromDefault = (param->value != defaultParam->value);
      if (modifiedFromDefault && !(param->paramType&PT_READONLY)) {
         strcat(buf, "bootldr: set ");
         strcat(buf, param->name);
         strcat(buf, " ");
         switch (param->paramFormat) {
         case PF_STRING:
	    strcat(buf, "\"");
            strcat(buf, (char *)param->value);
	    strcat(buf, "\"");
            break;
         case PF_HEX:
         case PF_DECIMAL:
         default: {
            char num[16];
            strcat(buf, "0x");
            binarytohex(num, param->value,4);
            strcat(buf, num);
         }         
         }
         strcat(buf, "\r\n");
      }
      param++;
      defaultParam++;
   }
   putstr(buf);

   /* now erase and program the params sector */
   program_flash_region("params", flashDescriptor->params.base, flashDescriptor->params.size,
                        (dword)buf, strlen(buf)+1, flashDescriptor->params.flags);
}
