/*
 * arch/arm/kernel/dma-sa1100.c
 *
 * some code cribbed from dma-rpc.c for support of USB function
 * co-exists with sa-1100 specific dma api in mach-sa1100/dma-sa1100.c
 * brad@parker.boston.ma.us
 *
 * Copyright (C) 2000 Nicolas Pitre
 *
 * generic DMA functions to support SA1100 DMA architecture
 */

#include <linux/sched.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/pci.h>

#include <asm/dma.h>
#include <asm/fiq.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <asm/mach/dma.h>

extern void sa1100_init_dma(void);

//#define DEBUG
#ifdef DEBUG
#define DPRINTK( x... )  printk( ##x )
#else
#define DPRINTK( x... )
#endif

/* longword offsets into DMA register list (SA-1100 manual, 11.6.3) */
#define DAR	0
#define CSR_SET	1
#define CSR_CLR	2
#define CSR_GET 3
#define BSA	4
#define BTA	5
#define BSB	6
#define BTB	7

#define state_prog_a	0
#define state_wait_a	1
#define state_wait_b	2

typedef struct scatterlist dmasg_t;

static void arch_get_next_sg(dmasg_t *sg, dma_t *dma)
{
	DPRINTK("arch_get_next_sg(sg=%p, dma=%p)\n", sg, dma);
	DPRINTK("using_sg %d, sg %p, buf.length %d\n",
		dma->using_sg, dma->sg, dma->buf.length);

	if (dma->sg) {
		sg->address = dma->sg->address;
		sg->length = dma->sg->length;

		dma->sg->length = 0;
		dma->sg->address += sg->length;

		DPRINTK("sg->length %ld, address %p\n",
			sg->length, (void *)sg->address);

		if (dma->sgcount > 1) {
			dma->sg++;
			dma->sgcount--;
		} else {
			dma->sg = NULL;
		}
	} else {
		sg->address = 0;
		sg->length = 0;
	}
}

static inline void arch_setup_dma_a(dmasg_t *sg, dma_t *dma)
{
	volatile unsigned int *dma_base = (void *)dma->dma_base;

	dma_base[BSA] = (unsigned int)sg->address;
	dma_base[BTA] = sg->length;
}

static inline void arch_setup_dma_b(dmasg_t *sg, dma_t *dma)
{
	volatile unsigned int *dma_base = (void *)dma->dma_base;

	dma_base[BSB] = (unsigned int)sg->address;
	dma_base[BTB] = sg->length;
}

static void arch_dma_handle(int irq, void *dev_id, struct pt_regs *regs)
{
	dma_t *dma = (dma_t *)dev_id;
	volatile unsigned int *dma_base = (void *)dma->dma_base;
	unsigned int status = 0, no_buffer = dma->sg == NULL;

	DPRINTK("arch_dma_handle(irq=%d,dma=%p) state=%d\n",
		irq, dma, dma->state);

	do {
		switch (dma->state) {
		case state_prog_a:
			arch_get_next_sg(&dma->cur_sg, dma);
			arch_setup_dma_a(&dma->cur_sg, dma);
			dma->state = state_wait_a;

		case state_wait_a:
			status = dma_base[CSR_GET];
			switch (status & (DCSR_ERROR|DCSR_DONEA|DCSR_DONEB)) {
			case DCSR_ERROR|DCSR_DONEA:
				arch_get_next_sg(&dma->cur_sg, dma);
				arch_setup_dma_a(&dma->cur_sg, dma);
				break;

			case DCSR_DONEA:
				arch_get_next_sg(&dma->cur_sg, dma);
				arch_setup_dma_b(&dma->cur_sg, dma);
				dma->state = state_wait_b;
				break;

			case DCSR_ERROR|DCSR_DONEB:
				arch_setup_dma_b(&dma->cur_sg, dma);
				dma->state = state_wait_b;
				break;
			}
			break;

		case state_wait_b:
			status = dma_base[CSR_GET];
			switch (status & (DCSR_ERROR|DCSR_DONEA|DCSR_DONEB)) {
			case DCSR_ERROR|DCSR_DONEB:
				arch_get_next_sg(&dma->cur_sg, dma);
				arch_setup_dma_b(&dma->cur_sg, dma);
				break;

			case DCSR_DONEB:
				arch_get_next_sg(&dma->cur_sg, dma);
				arch_setup_dma_a(&dma->cur_sg, dma);
				dma->state = state_wait_a;
				break;

			case DCSR_ERROR|DCSR_DONEA:
				arch_setup_dma_a(&dma->cur_sg, dma);
				dma->state = state_wait_a;
				break;
			}
			break;
		}
	} while (dma->sg && (status & (DCSR_DONEA|DCSR_DONEB|DCSR_ERROR)));

	if (no_buffer)
		disable_irq(irq);
}

static int arch_request_dma(dmach_t channel, dma_t *dma)
{
	unsigned long flags;
	int ret;

	DPRINTK("arch_request_dma(channel=%d,dma=%p)\n", channel, dma);

	if (channel < SA1100_DMA_CHANNELS) {
		save_flags_cli(flags);
		ret = request_irq(dma->dma_irq, arch_dma_handle,
				  SA_INTERRUPT, dma->device_id, dma);
		if (!ret)
			disable_irq(dma->dma_irq);
		restore_flags(flags);

		printk("arch_request_dma() allocating dma channel %d\n",
		       channel);
	} else {
		ret = -EINVAL;
	}

	return ret;
}

static void arch_free_dma(dmach_t channel, dma_t *dma)
{
	DPRINTK("arch_free_dma(channel=%d,dma=%p)\n", channel, dma);

	if (channel < SA1100_DMA_CHANNELS) {
		free_irq(dma->dma_irq, dma);
	} else {
		printk("arch_free_dma: invalid channel %d\n", channel);
	}
}

static int arch_get_dma_residue(dmach_t channel, dma_t *dma)
{
	int residue = 0;

	if (channel >= SA1100_DMA_CHANNELS) {
		printk("arch_get_dma_residue: invalid channel %d\n", channel);
		return -EINVAL;
	}
	return residue;
}

static void arch_enable_dma(dmach_t channel, dma_t *dma)
{
	volatile unsigned int *dma_base = (void *)dma->dma_base;
	unsigned int csr;

	DPRINTK("arch_enable_dma(channel=%d,dma=%p)\n", channel, dma);

	if (channel < SA1100_DMA_CHANNELS) {
		csr = DCSR_RUN | DCSR_IE;

		if (dma->invalid) {
			dma->invalid = 0;

			dma_base[CSR_CLR] = 0x7f;

			arch_get_next_sg(&dma->cur_sg, dma);

			if (dma_base[CSR_GET] & DCSR_BufB) {
			  arch_setup_dma_b(&dma->cur_sg, dma);
			  dma->state = state_wait_b;
			  csr |= DCSR_STRTB;
			} else {
			  arch_setup_dma_a(&dma->cur_sg, dma);
			  dma->state = state_wait_a;
			  csr |= DCSR_STRTA;
			}

		}
		
		dma_base[CSR_SET] = csr;
		enable_irq(dma->dma_irq);
	}
}

static void arch_disable_dma(dmach_t channel, dma_t *dma)
{
	volatile unsigned int *dma_base = (void *)dma->dma_base;

	DPRINTK("arch_disable_dma(channel=%d,dma=%p)\n", channel, dma);

	if (channel < SA1100_DMA_CHANNELS) {
		disable_irq(dma->dma_irq);

		dma_base[CSR_CLR] = DCSR_RUN | DCSR_ERROR |
		  DCSR_DONEA | DCSR_STRTA | DCSR_DONEB | DCSR_STRTB;
	} else {
		printk("arch_disable_dma: invalid channel %d\n", channel);
	}
}

static struct dma_ops arch_dma_ops = {
	type:		"generic",
	request:	arch_request_dma,
	free:		arch_free_dma,
	enable:		arch_enable_dma,
	disable:	arch_disable_dma,
	residue:	arch_get_dma_residue,
};

void __init arch_dma_init(dma_t *dma)
{
	int i;

	/* init generic channels */
	for (i = 0; i < 6; i++) {
		dma[i].dma_base = io_p2v(_DDAR(i));
		dma[i].dma_irq = IRQ_DMA0+i;
		dma[i].d_ops = &arch_dma_ops;
	}

	/* init sa1100 specific dma code */
	sa1100_init_dma();
}

/* magic needed for usb client's dma */

/* this should really be "set_dma_devaddr_sa1100" and we should change
 * the code which calls this to call "set_dma_mode" to set r/w
 */
void set_dma_mode_sa1100(int channel, int addr)
{
	volatile unsigned int *dma_base = (void *)io_p2v(_DDAR(channel));
	unsigned int dar;

	DPRINTK("set_dma_mode_sa1100(ch=%d, addr=%08x)\n", channel, addr);

	dar = DDAR_LtlEnd | DDAR_Brst8 | DDAR_8BitDev;

	/* this is completely usb specific...  */

	switch (addr) {
	case 0x80000A15:
		set_dma_mode(channel, DMA_MODE_READ);
		dar |= DDAR_DevRd | DDAR_Ser0UDCRc | DDAR_DevAdd(_Ser0UDCDR);
		break;
	case 0x80000A04:
		set_dma_mode(channel, DMA_MODE_WRITE);
		dar |= DDAR_DevWr | DDAR_Ser0UDCTr | DDAR_DevAdd(_Ser0UDCDR);
		break;
	default:
		return;
	}

	dma_base[DAR] = dar;
}

unsigned int get_dma_address_sa1100(int channel)
{
	volatile unsigned int *dma_base = (void *)io_p2v(_DDAR(channel));

	return (unsigned int)dma_base[BSA];
}

EXPORT_SYMBOL(set_dma_mode_sa1100);
EXPORT_SYMBOL(get_dma_address_sa1100);

