/*****************************************************************************/

/*
 *	usb_clt.c -- SA1100 USB controller driver. Provides a the USB 
 *                    hardware control.
 *
 *	Copyright (C)  Compaq Computer Corporation, 1998, 1999
 *
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*****************************************************************************/

#include <linux/proc_fs.h>
#include <linux/tqueue.h>

#include <linux/netdevice.h>
#include <linux/version.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/irq.h>
#include <asm/atomic.h>

#include "usb_ctl.h"
#include "usb_dbg.h"

/* 
 * Prototypes
 */

void udc_int_hndlr(int, void *, struct pt_regs *);
void udc_dma_init(void);
void usbctl_create_descriptors(void);
int udc_stallep(int ep);

/* 
 * Global Definitions
 */
char dev_desc[USB_DEVDESC_LENGTH];
#define CONFIGDESC_LENGTH \
        USB_CONFDESC_LENGTH + USB_IFACEDESC_LENGTH +     \
                                (2 * USB_ENDPDESC_LENGTH)
char config_desc[CONFIGDESC_LENGTH];

#ifdef CONFIG_PROC_FS
int usbctl_read_procmem (char *, char **, off_t, int, int);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
static int usbctl_read_proc(char *page, char **start, off_t off,
			    int count, int *eof, void *data)
{
        int len = usbctl_read_procmem(page, (char **)0, 0, 0, 0);
        if (len <= off+count) *eof = 1;
        *start = page + off;
        len -= off;
        if (len>count) len = count;
        if (len<0) len = 0;
        return len;
}
#else
struct proc_dir_entry usb_proc_entry = {
    0,
    3,
    "usb",
    S_IFREG | S_IRUGO,
    1,
    0, 0,
    0,
    NULL,
    usbctl_read_procmem,
    NULL
};
#endif
#endif /* CONFIG_PROC_FS */

/* these variables are global to allow configuring
 * at module load time. Thus if they these should be in the 
 * module symbol table.
 */
int usbd_tx_pktsize = 256;
int usbd_rx_pktsize = 64;

int 
udc_init(void)
{  
    DEBUG_KPrint(DEBUG_CTL, ("SA1100 USB Controller Loaded\n"));

    /* testing the packet routines to ensure that they 
     * what they are supposed to.
     */
    udc_disable_interrupts(0);
    udc_disable();

    /* now allocate the IRQ. */
    if (request_irq(IRQ_Ser0UDC, 
		    udc_int_hndlr, SA_INTERRUPT, "USB function", NULL) != 0)
    {
        printk("usb_ctl: Couldn't request USB irq\n");
    }
    memset(&usbd_info.usb_stats, 0, sizeof(struct usb_stats_t));
    
#ifdef CONFIG_PROC_FS
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
    create_proc_read_entry ("usb", 0, NULL, usbctl_read_proc, NULL);
#else
    proc_register(&proc_root, &usb_proc_entry);
#endif
#endif
    udc_dma_init();
    return 0;
}

void 
udc_cleanup()
{
    DEBUG_KPrint(DEBUG_CTL, ("Unloading SA1100 USB Controller\n"));
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
    remove_proc_entry ("usb", NULL);
#else
    proc_unregister(proc_root, usb_proc_entry.low_ino);
#endif

    udc_disable();
    disable_dma(0);
    disable_dma(1);
    free_dma(0);
    free_dma(1);
	
    free_irq(IRQ_Ser0UDC, NULL); 
}

void
udc_int_hndlr(int irq, void *dev_id, struct pt_regs *regs)
{
    int idx;
    unsigned int status = *(UDCSR);

    idx=0;
    while ((*(UDCSR) & status) && (idx<100000))
    {
        idx++;
        *(UDCSR) = status;
    }

#if 0
    if (idx>=100000)
    {
        printk("status %x\n", status);
    }
#endif

    if (status & UDCSR_EIR)
    {
        ep0_int_hndlr(status);
    }
    
    //if ((status & UDCSR_EIR) == 0)
    //printk("udc_int_hndlr() status %x\n", status);
 
    if (status & UDCSR_RIR)
    {
        ep1_int_hndlr(status);
    }
    
    if (status & UDCSR_TIR)
    {
        ep2_int_hndlr(status);
    }

    if (status & UDCSR_SUSIR)
    {
        /* in suspend case just switch off the suspend interrupt
         * (otherwise it continues to fire), and wait for a resume.
         */
        udc_enable_interrupts(1);
    }
    
#if 0
    if (status & UDCSR_RESIR)
    {
    }
#endif

    if (status & UDCSR_RSTIR)
    {
//        DEBUG_KPrint(DEBUG_CTL, ("resetting..\n"));
        printk("usb_net_func: resetting\n");
        udc_disable();
        udc_enable();
        ep0_reset();
        ep1_reset();
        ep2_reset();
    }
}

/*
 * Initialises the DMA controller for channel 0
 * to be used for USB receives, device->mem, and channel 
 * 1 to be used for USB transmits (mem ->device).
 *
 * No buffers are set up for DMA, once a function has registered
 * the dma buffers should be set up.
 *
 */
void
udc_dma_init(void)
{
    if (request_dma(0, "USB receive"))
    {
        printk("udc_dma_init : channel 0 taken\n");
    }
    if (request_dma(1, "USB transmit"))
    {
        printk("udc_dma_init : channel 1 taken\n");
    }
    /* program the address register for UDC reads 
     * (device -> mem)
     */
    set_dma_mode_sa1100(0, DMA_UDC_RECEIVE);
    
    /* program the address register for UDC writes 
     * (mem -> device)
     */
    set_dma_mode_sa1100(1, DMA_UDC_TRANSMIT);

    /* disable dma channel 0 and 1 */
    disable_dma(0);
    disable_dma(1);
}

/*
 * Switch off the SA1100 UDC
 */
void
udc_disable(void)
{
    unsigned int udccr;
    int idx = 0;

    GPDR |= GPIO_USB_ENABLE;
    GPSR = GPIO_USB_ENABLE;

    while (*(UDCCR) & UDCCR_UDA)
    {
        idx++;
        if (idx>1000000)
        {
            printk("UDC CTL : Waiting too long to disable\n");
        }
        /* wait for the UDC to become inactive */
    }
    idx=0;
    /* Turn the UDC off.
     * If the USB cable is connected but the circuit
     * for hiding itsy is activated we will not be able to 
     * disable the UDC, so we timeout. (Need to loop
     * because of the UDC synchronisation bug with the SA1100)
     */  
    do
    {
        udccr = *(UDCCR);
        *(UDCCR) = udccr | UDCCR_UDD; 
        idx++;
    } while (!(*(UDCCR) & UDCCR_UDD) && (idx < 100) );
    *(UDCCR) &= ~UDCCR_SRM;
}

/*
 * Switch on the SA1100 UDC
 */
void
udc_enable(void)
{
    int idx=0;

    do
    {
        *(UDCCR) &= ~UDCCR_UDD;
        idx++;
        if (idx>1000)
        {
            printk("UDC CTL : can't enable udc\n");
        }
    } while (*(UDCCR) & UDCCR_UDD);

    GPCR = GPIO_USB_ENABLE;
}

/*
 * Switch on UDC interrupts, the flag indicates whether 
 * the suspend interrupt is enabled or disabled.
 */
void
udc_enable_interrupts(int suspend)
{
    if (suspend)
    {
        udc_write_reg(UDCCR, 0);
    }
    else
    {
        udc_write_reg(UDCCR, UDCCR_SRM);
    }
}

/* 
 * switch off interrupts, the suspend argument indicates 
 * whether it should just turn off the suspend interrupt 
 * every interrupt.
 */
void 
udc_disable_interrupts(int suspend)
{
    if (suspend)
    {
        udc_write_reg(UDCCR, UDCCR_SRM);
    }
    else
    {
        udc_write_reg(UDCCR, UDCCR_EIM | UDCCR_RIM | UDCCR_TIM
                      | UDCCR_SRM | UDCCR_REM);
    }
}

/* 
 * Workaround for synchronization bug in SA-1100 UDC, see Note
 * in section 11.8.2 UDC Register Definitions for more detail 
 * on this bug, it requires us to verify some of the register
 * writes.
 */
void 
udc_write_reg(volatile unsigned int *regaddr, unsigned int value)
{
    int idx=0;
    do 
    {
        *(regaddr) = value;
        idx++;
        if (idx>1000000)
        {
            printk("UDC CTL : write register stalled\n");
        }
    } while (*(regaddr) != value);
}

/* 
 * Clear the char array version of the descriptors. This is called
 * when the usb function is released.
 *
 */
void
usbctl_clear_descriptors()
{
    memset(dev_desc, 0, USB_DEVDESC_LENGTH);
    memset(config_desc, 0, CONFIGDESC_LENGTH);
}
    
/*
 * Create the device descriptor packets ready to be transfered to the 
 * host. Done to speed up the transactions, there would be several NAKs
 * if we had to create them from the device tree so we store them in 
 * memory ready to send. The descriptors are stored in global variables.
 *
 * device    : the device tree, which contains all the information required
 *             to create the device and config descriptors.
 */
void
usbctl_create_descriptors(void)
{
    char * buf;
    
    DEBUG_KPrint(DEBUG_CTL, ("usbd_info.rx_pktsize %d\n",
                             usbd_info.rx_pktsize));
    
    /* first create the device descriptor */
    dev_desc[USB_DEVDESC_BLENGTH] = USB_DEVDESC_LENGTH;
    dev_desc[USB_DEVDESC_BDESCTYPE] = USB_DESC_DEVICE;
    dev_desc[USB_DEVDESC_BCDUSB_HI] = (USB_V100_COMPLIANT & 0xff00) >> 8;
    dev_desc[USB_DEVDESC_BCDUSB_LO] = (USB_V100_COMPLIANT & 0x00ff);
    dev_desc[USB_DEVDESC_BDEVCLASS] = VENDOR_SPECIFIC_CLASS;
    dev_desc[USB_DEVDESC_BDEVSUBCLASS] = 0;
    dev_desc[USB_DEVDESC_BDEVPROT] = 0;
    dev_desc[USB_DEVDESC_BMAXSIZE] = 8;
    dev_desc[USB_DEVDESC_IDVEND_HI] = (USB_VENDOR_COMPAQ & 0xff00) >> 8;
    dev_desc[USB_DEVDESC_IDVEND_LO] = (USB_VENDOR_COMPAQ & 0x00ff);
    dev_desc[USB_DEVDESC_IDPROD_HI] = (COMPAQ_ITSY_ID & 0xff00) >> 8;
    dev_desc[USB_DEVDESC_IDPROD_LO] = (COMPAQ_ITSY_ID & 0x00ff);
    dev_desc[USB_DEVDESC_BCDDEV_HI] = 0;
    dev_desc[USB_DEVDESC_BCDDEV_LO] = 0;
    dev_desc[USB_DEVDESC_IMANU] = 0;
    dev_desc[USB_DEVDESC_IPRODUCT] = 0;
    dev_desc[USB_DEVDESC_ISERNUM] = 0;
    /* only suport 1 configuration */
    dev_desc[USB_DEVDESC_BNUMCONF] = 1;

	config_desc[USB_CONFDESC_BLENGTH] = USB_CONFDESC_LENGTH;
	config_desc[USB_CONFDESC_BDESCTYPE] = USB_DESC_CONFIG;
	config_desc[USB_CONFDESC_WTOTLENGTH_HI] = 0;
	config_desc[USB_CONFDESC_WTOTLENGTH_LO] = USB_CONFDESC_LENGTH + 
	    USB_IFACEDESC_LENGTH + (2 * USB_ENDPDESC_LENGTH);
	config_desc[USB_CONFDESC_BNUMIFACE] = 1;
	config_desc[USB_CONFDESC_BCONFVAL] = 1;
	config_desc[USB_CONFDESC_ICONF] = 0;
	config_desc[USB_CONFDESC_BMATTRIB] = USB_CONFIG_BUSPOWERED;
	config_desc[USB_CONFDESC_MAXPOWER] = USB_POWER(500);
	
    /* align the buffer pointer to point at interface space */
    buf = config_desc + USB_CONFDESC_LENGTH;
    buf[USB_IFACEDESC_BLENGTH] = USB_IFACEDESC_LENGTH;
    buf[USB_IFACEDESC_BDESCTYPE] = USB_DESC_INTERFACE;
    buf[USB_IFACEDESC_BIFACENUM] = 0;
    buf[USB_IFACEDESC_BALTSETTING] = 0; 
    buf[USB_IFACEDESC_BNUMENDP] = 2;
    buf[USB_IFACEDESC_BIFACECLASS] = VENDOR_SPECIFIC_CLASS;
    buf[USB_IFACEDESC_BIFACESUBCLASS] = 0;
    buf[USB_IFACEDESC_BIFACEPROT] = 0;
    buf[USB_IFACEDESC_IIFACE] = 0;

    /* now move on to the endpoints */
    buf += USB_IFACEDESC_LENGTH;

    buf[USB_ENDPDESC_BLENGTH] = USB_ENDPDESC_LENGTH;
    buf[USB_ENDPDESC_BDESCTYPE] = USB_DESC_ENDPOINT;
    buf[USB_ENDPDESC_BADDRESS] = USB_EP_ADDRESS(1, USB_OUT);
    buf[USB_ENDPDESC_BMATTR] = USB_EP_BULK;
    buf[USB_ENDPDESC_WMAXPKTSZE_LO] = (usbd_info.rx_pktsize & 0x00ff);
    buf[USB_ENDPDESC_WMAXPKTSZE_HI] = (usbd_info.rx_pktsize & 0xff00) >> 8;
    buf[USB_ENDPDESC_BINTERVAL] = 0;

    buf += USB_ENDPDESC_LENGTH;

    buf[USB_ENDPDESC_BLENGTH] = USB_ENDPDESC_LENGTH;
    buf[USB_ENDPDESC_BDESCTYPE] = USB_DESC_ENDPOINT;
    buf[USB_ENDPDESC_BADDRESS] = USB_EP_ADDRESS(2, USB_IN);
    buf[USB_ENDPDESC_BMATTR] = USB_EP_INT;
    buf[USB_ENDPDESC_WMAXPKTSZE_LO] = (usbd_info.tx_pktsize & 0x00ff);
    buf[USB_ENDPDESC_WMAXPKTSZE_HI] = (usbd_info.tx_pktsize & 0xff00) >> 8;;
    buf[USB_ENDPDESC_BINTERVAL] = 0x1; /* every 1ms */
}

/*
 * Called by the endpoint 0 routines after the host
 * sends a SET_CONFIGURATION packet, marking the 
 * end of bus enumeration and movement into the
 * configured state. Now we may alert the USB function 
 * that we are configured.
 *
 * ncfg   : the configuration number.
 *
 * XXX: this is presently stubbed into a loopback mode.
 */
void 
udc_configure(void *data)
{
    ep1_init(0);
    ep2_init(1);
}

/*
 * stall the endpoint.
 *
 * returns : 0 if ok, -1 otherwise
 */
int
udc_stallep(int ep)
{
    printk("usb_ctl: stalling endpoint %d\n", ep);
    switch (ep)
    {
        case 1 :
            do
            {
                *(UDCCS1) |= UDCCS1_FST;
            } while (!(*(UDCCS1) & UDCCS1_FST));
            break;
        case 2 :
            do
            {
                *(UDCCS2) |= UDCCS2_FST;
            } while (!(*(UDCCS2) & UDCCS2_FST));
            break;
        default :
            break;
    }
    return 0;
}

#ifdef CONFIG_PROC_FS

#define PROC_LIMIT (PAGE_SIZE-80)

/* 
 * Writes out the USB controller statistics using the standard 
 * read only proc interface, allowing a maximum of 1 page of 
 * data.
 *
 * Only the buf parameter is of use to us, it points to the start
 * of the buffer we can write in.
 *
 * returns : number of bytes written.
 *
 */
extern char *receive_page;
/*extern char *receive_page_ptr;*/
extern int dmachn_rx_inuse;
extern struct tq_struct recv_task;
extern char *send_page;
extern struct skb_buff *send_skb;
int
usbctl_read_procmem(char *buf, char **start, off_t offset,
		    int len, int unused)
{
    len = 0;
    
    len += sprintf(buf, "SA1100 USB Controller: ");

    if (usbd_info.state == USB_STATE_CONFIGURED)
    {
        len += sprintf(buf+len, "Configured\n");
    }
    else
    {
        len += sprintf(buf+len, "Not Configured\n");
    }
    
    /* 
     * USB Statistics
     */
    len += sprintf(buf+len, "USB Controller Stats:\n");
    len += sprintf(buf+len, "Receive (OUT) packets %d errors %d\n",
		   usbd_info.usb_stats.outpkts,
		   usbd_info.usb_stats.outpkt_errs);
    len += sprintf(buf+len, "Transmit (IN) packets %d errors %d\n",
		   usbd_info.usb_stats.inpkts,
		   usbd_info.usb_stats.inpkt_errs);
    len += sprintf(buf+len, "send page %p page %p ptr %p in use %d\n",
                   send_page, receive_page,
		   (void *)0/*receive_page_ptr*/, dmachn_rx_inuse);
    len += sprintf(buf+len, "Control %x\n"
		            "Address %x\n"
		            "OUT     %x\n"
		            "IN      %x\n"
		            "EP0 CTL %x\n"
		            "EP1 CTL %x\n"
		            "EP2 CTL %x\n"
		            "Status  %x\n", *(UDCCR), *(UDCAR),
		   *(UDCOMP), *(UDCIMP), *(UDCCS0), *(UDCCS1),
		   *(UDCCS2), *(UDCSR));
    len +=sprintf(buf+len, "address %x\n"
                  "C/S %x\n"
                  "buffer A %x %x\n"
                  "buffer B %x %x\n",
                  *(DDAR),*(DDCS),*(DDBA),*(DDTA),*(DDBB),*(DDTB));

#if 0
    len += sprintf(buf+len, "tbusy %d send skb  %p\n", usbd_info.dev->tbusy,
                   send_skb);
#else
    len += sprintf(buf+len, "send skb  %p\n", send_skb);
#endif
    return len;
}

#endif


/*
 * Called to request access to the USB. If the USB controller
 * is not already allocated and the parameters are ok then the
 * USB controller is enabled. At this point a connection will be 
 * accepted.
 */
int
usbctl_start(void)
{
   usbd_info.magic = USB_MAGIC;
   usbd_info.state = USB_STATE_NOTATTACHED;
   usbd_info.address = 0;
   usbd_info.tx_pktsize = usbd_tx_pktsize;
   usbd_info.rx_pktsize = usbd_rx_pktsize;

   /* initialise the controller */
   udc_init();

   /* create descriptors for enumeration */
   usbctl_create_descriptors();

   atomic_set(&usbd_info.rx_lock, 0);
   atomic_set(&usbd_info.tx_lock, 0);
    
   udc_enable(); 
   udc_enable_interrupts(0);

   return USB_Ok;
}

/*
 * called if the client function is no longer needed. This will 
 * disable the UCD, and the power system will be alerted of the 
 * change in available power.
 */
int 
usbctl_release(void)
{
    /* 1. disable the UDC */
    udc_disable();

    /* 2. clear the usb_info structure */
    usbd_info.state = USB_STATE_NOTATTACHED;
    usbd_info.address = 0;
    usbd_info.tx_pktsize = 0;
    usbd_info.rx_pktsize = 0;

    /* 3. clear dev_desc and config_desc */
    usbctl_clear_descriptors();

    /* 4. free up any udc resources */
    udc_cleanup();

    return USB_Ok;
}

/*
 * register a power driver with the client USB controller. The power 
 * driver now waits for the provided callback to tell it that it may 
 * start drawing power.
 */
int 
register_usb_power(int power, usb_power_callback_t call)
{
    DEBUG_KPrint(DEBUG_CTL, ("usb_register_power\n"));
    return 0;
}

