/*****************************************************************************/

/*
 *
 *	Copyright (C) Compaq Computer Corporation, 1998, 1999
 *
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*****************************************************************************/

#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/skbuff.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/system.h>

#include "usb_ctl.h"
#include "usb_dbg.h"

#define DMA_IN_COUNT_MAX usbd_info.tx_pktsize
#define MAXIN_PACKET_SZ  usbd_info.tx_pktsize

/*
 * globals
 */
static int dmachn_tx=-1;
static int dmachn_tx_inuse = 0;
static int tx_size=0;

struct sk_buff *send_skb = NULL;
unsigned int send_len=0;
char *send_page=NULL;
char *send_ptr=NULL;

unsigned long send_page_dma;
#define send_page_vtop(v)	(unsigned)(send_page_dma + ((v) - send_page))

extern unsigned int address;

/*
 * prototypes
 */


/*
 * Initialise the DMA system to use the given chn for receiving
 * packets over endpoint 2.
 *
 * chn     : the dma channel to use.
 * returns : 0 if ok, -1 if channel couldn't be used.
 */
int 
ep2_init(int chn)
{
    if (dmachn_tx >= 0)
	return 0;

    dmachn_tx = chn;

    if (send_page == NULL)
    {
#if 0
        send_page = (char *)__get_dma_pages(GFP_KERNEL, 0);
        /* disable caching */
        allocate_vspace((ulong)virt_to_phys((ulong)send_page), PGDIR_SIZE,1,0);
        DEBUG_KPrint(DEBUG_SEND, ("send page %p\n", send_page));
#else
	send_page = consistent_alloc(GFP_KERNEL, PAGE_SIZE,
				     (dma_addr_t *)&send_page_dma);
        DEBUG_KPrint(DEBUG_SEND, ("send page %p (size %ld); dma %p\n",
				  send_page, PAGE_SIZE,
				  (dma_addr_t *)send_page_dma));
#endif
    }
    send_skb = NULL;
    return 0;
}

/*
 * Setup the transmit DMA buffers with the first packet off the 
 * send queue, if the queue is empty, or if a DMA packet has already
 * been setup for send, this will not change the DMA setup.
 *
 */
void 
ep2_stop(void)
{
   disable_dma(dmachn_tx);
}

/*
 * Setup the transmit DMA buffers with the first packet off the 
 * send queue, if the queue is empty, or if a DMA packet has already
 * been setup for send, this will not change the DMA setup.
 *
 */
int 
ep2_start(void)
{
    int idx = 0;
    int retval = -1;

    CHECK_ADDRESS;
    if ((dmachn_tx>=0) && !(dmachn_tx_inuse) && (send_skb))
    {
        /* just check it actually has data to tx first. */
        if (send_len == 0)
        {
	    struct net_device *dev = send_skb->dev;

	    usbd_info.eth_stats.tx_packets++;
	    dev_kfree_skb_any(send_skb);
	    send_skb = NULL;

	    if (netif_queue_stopped(dev))
		netif_wake_queue(dev);
        }
        else
        {
            /* setup DMA so the packet will be sent. */
            if (usbd_info.tx_pktsize <= send_len)
            {
                tx_size = usbd_info.tx_pktsize;
            }
            else
            {
                tx_size = send_len;
            }
            set_dma_addr(dmachn_tx, send_page_vtop(send_ptr));
            send_ptr += tx_size;
            send_len -= tx_size;

            /* must tell both the UDC system and the dma system
             * how much to transfer.
             */
#if 1
            do 
            {
                idx++;
                *(UDCIMP) = (unsigned int) (tx_size-1);
            } while ((*(UDCIMP) != (unsigned int) (tx_size-1)) &&
                     (idx < 1000));
#else
	    *(UDCIMP) = (unsigned int) (tx_size-1);
//	    udc_write_reg(UDCIMP, tx_size-1);
#endif

            set_dma_count(dmachn_tx, tx_size);

            dmachn_tx_inuse = 1;
            enable_dma(dmachn_tx);

            retval = 0;
        }
    }
    CHECK_ADDRESS;
    return retval;
}

/*
 * reset any txpkts, it will be up to the client driver to free up the send
 * queue.
 */
void
ep2_reset(void)
{
    CHECK_ADDRESS;
    if (dmachn_tx >= 0)
    {
        disable_dma(dmachn_tx);
        /* reset dma */
        set_dma_mode_sa1100(dmachn_tx, 0);
        set_dma_mode_sa1100(dmachn_tx, DMA_UDC_TRANSMIT);
    }

    /* stall and unstall */
    do
    {
        *(UDCCS2) |= UDCCS2_FST;
    } while (!(*(UDCCS2) & UDCCS2_FST));
    do
    {
        *(UDCCS2) &= ~UDCCS2_FST;
    } while (*(UDCCS2) & UDCCS2_FST);

    
    /* clear the tx list */
    if (send_skb)
    {
        dev_kfree_skb(send_skb);
        send_skb = NULL;
        send_ptr = send_page;
    }

    dmachn_tx_inuse = 0;
}

void
ep2_int_hndlr(unsigned int status)
{
    unsigned int ep2_status;
    int idx;

    CHECK_ADDRESS;
    if (*(UDCAR) > 100)
    {
        *(UDCAR) = usbd_info.address;
    }
    
    ep2_status = *(UDCCS2);
    if (dmachn_tx>=0)
    {
        if (ep2_status & UDCCS2_TPC)
        {
            disable_dma(dmachn_tx);
            /* must have a completed transfer */ 
        }
        else
        {
            printk("USB Send : irq no TPC %x\n", ep2_status);
        }

        usbd_info.usb_stats.inpkts++;
        idx = 0;
        do
        {
            idx++;
            *(UDCCS2) |= UDCCS2_TPC;
            if (idx>1000)
            {
                printk("USB Send : loop overrun\n");
            }
        } while ((*(UDCCS2) & UDCCS2_TPC) && (idx<1000));
        /* enable DMA if an error occurred. */
        if (ep2_status & (UDCCS2_TPE | UDCCS2_TUR | UDCCS2_SST))
        {
            /* circumventing ep2_start, don't worry about
             * dmachn_tx_inuse
             */
            printk("USB Send : Transmit error %x\n", ep2_status);
            set_dma_addr(dmachn_tx, send_page_vtop(send_ptr-tx_size));
            do 
            {
                idx++;
                *(UDCIMP) = (unsigned int) (tx_size-1);
            } while ((*(UDCIMP) != (unsigned int) (tx_size-1)) &&
                     (idx < 1000));
            set_dma_count(dmachn_tx, tx_size);
            enable_dma(dmachn_tx);
        }
        else
        {
            if (ep2_status & (UDCCS2_TPC))
            {
                /* no errors, schedule the control to free the packet
                 * at a later time.
                 */
                dmachn_tx_inuse = 0;
                ep2_start();
            }
        }
    }
    else
    {
        printk("USB Send : interrupt before initialised %x\n", ep2_status);
    }    
    CHECK_ADDRESS;
}

/*
 * send a block across the bus.
 */
int
usb_transmit_skb(struct sk_buff *skb)
{    
    int status = -1;
    struct net_device *dev = skb->dev;

    CHECK_ADDRESS;

    /* the block must have a buffer, length, and callback */
    if (send_skb == NULL)
    {
	netif_stop_queue(dev);

        send_skb = skb;
        /* copy the buffer into the send page...
         * I can't corrupt the actual skb because
         * it is used after I am finished with it, so I make
         * a copy.
         */
        send_len = send_skb->len-10;
        if (send_len > PAGE_SIZE)
        {
            printk("USB Send : copying more than a page ...\n");
        }
        memcpy(send_page, (send_skb->data+10), send_len);
        send_ptr = send_page;

        send_page[0] = (char) (send_len & 0x00ff);
        send_page[1] = (char) ((send_len & 0xff00) >> 8) ;
flush_cache_all();

        /* if the packet is aligned on a 256 byte boundary then send
         * an extra byte so the host doesn't stall.
         */
        if ((send_len % 256) == 0)
        {
            /* this doesn't work because the skb has been
             * allocated for the exact length. Doesn't matter
             * what I send though..
             */
            DEBUG_KPrint(DEBUG_SEND, ("256 byte aligned send.\n"));
            send_len++;
        }
            
        /* flush the cache of the transmit page. */
/*        processor.u.armv3v4._flush_cache_area((unsigned int) send_page,
          PAGE_SIZE, 0);*/
        status = ep2_start();
    }

    if (status < 0)
    {
        DEBUG_KPrint(DEBUG_SEND, ("USB Send : send_skb != NULL\n"));

	usbd_info.eth_stats.tx_dropped++;
        dev_kfree_skb(skb);

	if (netif_queue_stopped(dev))
	    netif_wake_queue(dev);
    }

    return status;
}



